/**
******************************************************************************
* File Name          : 
* Description        : 
*                      
******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/

#include <string.h>
#include "mbedtls/chacha20.h"
#include "ble_nfc_ext_board_lib/ble_nfc_ext_board_master.h"
#include "crc_lib/crc_lib.h"
#include "debug_port.h"

#define ODEBUG_L                        0//off debug
#define LDEBUG_L                        1//low level (err only)
#define MDEBUG_L                        2//medium level
#define HDEBUG_L                        3//high level

#define BLE_EXT_BOARD_DEBUG_LEVEL       LDEBUG_L

static const char* const dbg_name="ble_serial";

static bool is_init=false;
static uint8_t* pbuff=NULL;
static uint16_t pbuff_size=0;
static interface_txrx_t interface_txrx=NULL;
static rand_func_t rand_func=NULL;
static mbedtls_chacha20_context crypt_ctx;
static const uint8_t* crypt_key=NULL;
static uint8_t stoken[TOKEN_SIZE];

//   input == output
static int chacha20_crypt(mbedtls_chacha20_context* ctx, 
                          const unsigned char key[32],
                          const unsigned char nonce[12],
                          uint32_t counter,
                          size_t data_len,
                          const unsigned char* input,
                          unsigned char* output)
{
  int ret;
  
  mbedtls_chacha20_init(ctx);
  
  ret=mbedtls_chacha20_setkey(ctx, key);
  if(ret != 0)
    goto cleanup;
  
  ret=mbedtls_chacha20_starts(ctx, nonce, counter);
  if(ret != 0)
    goto cleanup;
  
  ret=mbedtls_chacha20_update(ctx, data_len, input, output);
  
cleanup:
  mbedtls_chacha20_free(ctx);
  return(ret);
}

static inline ext_board_mess_crc_t ext_board_get_crc(const uint8_t* data, const uint16_t len)
{
  return (ext_board_mess_crc_t)crc16_fast(data, len);
}

static void ext_board_add_crc(void* const packet)
{
  const ext_board_mess_header_t* const header=(const ext_board_mess_header_t* const)packet;
  uint8_t* const buff=(uint8_t* const)packet;
  
  ext_board_mess_crc_t* const crc=(ext_board_mess_crc_t* const)&buff[BLE_NFC_EXT_BOARD_PACKET_CRC_OFFSET(header->len)];
  crc[0]=ext_board_get_crc(packet, BLE_NFC_EXT_BOARD_PACKET_CRC_CALC_LEN(header->len));
}

static int16_t ext_board_check_packet(const void* const packet, const uint16_t data_size, const char** verbose)
{
  const ext_board_mess_header_t* const header=(const ext_board_mess_header_t* const)packet;
  const uint8_t* const buff=(uint8_t* const)packet;
  
  if(sizeof(ext_board_mess_header_t)>data_size)
  {
    if(data_size)
    {
      if(verbose!=NULL) *verbose="few data";
    }
    else
    {
      if(verbose!=NULL) *verbose="no data";
    }
    return 0;
  }
  
  if(header->sync_byte!=BLE_BOARD_SYNC_BYTE)
  {
    if(verbose!=NULL) *verbose="sync err";
    return -1;
  }
    
  if(header->len>BLE_NFC_EXT_BOARD_PACKET_LEN(data_size))
  {
    if(verbose!=NULL) *verbose="rx buff size err";
    return -2;
  }
  
  const ext_board_mess_crc_t* const crc=(ext_board_mess_crc_t* const)&buff[BLE_NFC_EXT_BOARD_PACKET_CRC_OFFSET(header->len)];
  
  if(crc[0]!=ext_board_get_crc(packet, BLE_NFC_EXT_BOARD_PACKET_CRC_CALC_LEN(header->len)))
  {
    if(verbose!=NULL) *verbose="bad crc";
    return -3;
  }
  else
  {
    if(verbose!=NULL) *verbose="ok";
    return (int16_t)(BLE_NFC_EXT_BOARD_PACKET_LEN(header->len));
  }
}

static uint32_t ext_board_get_counter(bool is_tx)
{
  static uint32_t counter=0;
  
  if(is_tx) {counter++;}

  return (counter-1);
}

int16_t ext_board_init(uint8_t* const heap, const uint16_t heap_size, const interface_txrx_t pinterface_txrx, const rand_func_t prand_func, const uint8_t* key)
{
  if(heap==NULL || heap_size<BLE_NFC_EXT_BOARD_PACKET_LEN(128) || pinterface_txrx==NULL || prand_func==NULL)
  {
    is_init=false;
    
    return -1;
  }
  
  pbuff=heap;
  pbuff_size=heap_size;
  interface_txrx=pinterface_txrx;
  rand_func=prand_func;
  crypt_key=key;
  
  is_init=true;
  
  return 0;
}

const char* ext_board_get_verbose_mess_type(const ext_board_mess_type_t mess_type)
{
  switch(mess_type)
  {
  case CMD_GET_FW_VERS:                 return "CMD_GET_FW_VERS";
  case CMD_ANS_GET_FW_VERS:             return "CMD_ANS_GET_FW_VERS";
  case CMD_GET_FW_VERS_VERBOSE:         return "CMD_GET_FW_VERS_VERBOSE";
  case CMD_ANS_GET_FW_VERS_VERBOSE:     return "CMD_ANS_GET_FW_VERS_VERBOSE";
  case CMD_REBOOT:                      return "CMD_REBOOT";
  case CMD_ANS_REBOOT:                  return "CMD_ANS_REBOOT";
  case CMD_SET_NFC_SCAN_PERIOD:         return "CMD_SET_NFC_SCAN_PERIOD";
  case CMD_ANS_SET_NFC_SCAN_PERIOD:     return "CMD_ANS_SET_NFC_SCAN_PERIOD";
  case CMD_FIRMWARE_FILE:               return "CMD_FIRMWARE_FILE";
  case CMD_ANS_FIRMWARE_FILE:           return "CMD_ANS_FIRMWARE_FILE";
  case CMD_BLE_INIT:                    return "CMD_BLE_INIT";
  case CMD_ANS_BLE_INIT:                return "CMD_ANS_BLE_INIT";
  case CMD_BLE_STATUS:                  return "CMD_BLE_STATUS";
  case CMD_ANS_BLE_STATUS:              return "CMD_ANS_BLE_STATUS";
  case CMD_BLE_CONNECTION_COUNT:        return "CMD_BLE_CONNECTION_COUNT";
  case CMD_ANS_BLE_CONNECTION_COUNT:    return "CMD_ANS_BLE_CONNECTION_COUNT";
  case CMD_BLE_CONNECTION_STATUS:       return "CMD_BLE_CONNECTION_STATUS";
  case CMD_ANS_BLE_CONNECTION_STATUS:   return "CMD_ANS_BLE_CONNECTION_STATUS";
  case CMD_BLE_CONNECT:                 return "CMD_BLE_CONNECT";
  case CMD_ANS_BLE_CONNECT:             return "CMD_ANS_BLE_CONNECT";
  case CMD_BLE_SEND_DATA:               return "CMD_BLE_SEND_DATA";
  case CMD_ANS_BLE_SEND_DATA:           return "CMD_ANS_BLE_SEND_DATA";
  case CMD_BLE_RECEIVE_DATA:            return "CMD_BLE_RECEIVE_DATA";
  case CMD_ANS_BLE_RECEIVE_DATA:        return "CMD_ANS_BLE_RECEIVE_DATA";
  case CMD_BLE_DISCONNECT:              return "CMD_BLE_DISCONNECT";
  case CMD_ANS_BLE_DISCONNECT:          return "CMD_ANS_BLE_DISCONNECT";
  case CMD_BLE_BEACON_VISIBLE:          return "CMD_BLE_BEACON_VISIBLE";
  case CMD_ANS_BLE_BEACON_VISIBLE:      return "CMD_ANS_BLE_BEACON_VISIBLE";
  case CMD_NFC_VISIBLE:                 return "CMD_NFC_VISIBLE";
  case CMD_ANS_NFC_VISIBLE:             return "CMD_ANS_NFC_VISIBLE";
  case CMD_DFU_RESULT:                  return "CMD_DFU_RESULT";
  case CMD_ANS_DFU_RESULT:              return "CMD_ANS_DFU_RESULT";
  case CMD_SET_BLE_NAME:                return "CMD_SET_BLE_NAME";
  case CMD_ANS_SET_BLE_NAME:            return "CMD_ANS_SET_BLE_NAME";
  case CMD_GET_BLE_NAME:                return "CMD_GET_BLE_NAME";
  case CMD_ANS_GET_BLE_NAME:            return "CMD_ANS_GET_BLE_NAME";
  case CMD_SET_PASS_KEY:                return "CMD_SET_PASS_KEY";
  case CMD_ANS_SET_PASS_KEY:            return "CMD_ANS_SET_PASS_KEY";
  case CMD_GET_PASS_KEY:                return "CMD_GET_PASS_KEY";
  case CMD_ANS_GET_PASS_KEY:            return "CMD_ANS_GET_PASS_KEY";
  case CMD_GET_BLE_BOARD_MAC:           return "CMD_GET_BLE_BOARD_MAC";
  case CMD_ANS_GET_BLE_BOARD_MAC:       return "CMD_ANS_GET_BLE_BOARD_MAC";
  case CMD_APPROTECT_ENABLE:            return "CMD_APPROTECT_ENABLE";
  case CMD_ANS_APPROTECT_ENABLE:        return "CMD_ANS_APPROTECT_ENABLE";
  case CMD_STOKEN_REQUEST:              return "CMD_STOKEN_REQUEST";
  case CMD_ANS_STOKEN_REQUEST:          return "CMD_ANS_STOKEN_REQUEST";
  default:                              return "Unknown";
  }
}

//for CMD_FIRMWARE_FILE
int16_t ext_board_get_pointer_for_cmd_firmware_file_payload(uint8_t** const payload, uint16_t* max_payload_size)
{
  if(!is_init)
  {
    payload[0]=NULL;
    max_payload_size[0]=NULL;
    return -1;
  }
  
  payload[0]=&pbuff[BLE_NFC_EXT_BOARD_PACKET_PAYLOAD_OFFSET()+sizeof(ext_board_mess_firmware_upd_cmd_t)];
  max_payload_size[0]=pbuff_size-BLE_NFC_EXT_BOARD_PACKET_LEN(sizeof(ext_board_mess_firmware_upd_cmd_t));
  
  return 0;
}

//for CMD_BLE_SEND_DATA
int16_t ext_board_get_pointer_for_cmd_ble_send_data_payload(uint8_t** const payload, uint16_t* max_payload_size)
{
  if(!is_init)
  {
    payload[0]=NULL;
    max_payload_size[0]=NULL;
    return -1;
  }
  
  payload[0]=&pbuff[BLE_NFC_EXT_BOARD_PACKET_PAYLOAD_OFFSET()+sizeof(ext_board_ble_send_data_t)];
  max_payload_size[0]=pbuff_size-BLE_NFC_EXT_BOARD_PACKET_LEN(sizeof(ext_board_ble_send_data_t));
  
  return 0;
}

uint16_t ext_board_get_ble_max_receive_data_len(void)
{
  return (pbuff_size-BLE_NFC_EXT_BOARD_PACKET_LEN(sizeof(ext_board_ble_ans_receive_data_t)));
}

uint16_t ext_board_get_ble_max_receive_visible_beacons_count(void)
{
  return (pbuff_size-BLE_NFC_EXT_BOARD_PACKET_LEN(sizeof(ext_board_ans_ble_beacon_visible_t)))/sizeof(ext_board_ble_beacon_info_t);
}

//return: rx_args_len
int16_t ext_board_txrx(const ext_board_mess_type_t req_mess_type, const void* const req_arg, const void** ans_arg)
{
  if(ans_arg!=NULL)
  {
    ans_arg[0]=NULL;
  }
  
  if(!is_init)
  {
    if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
    {
      __PRINTF("%s: no init\n", dbg_name);
    }
    
    return -1;
  }
  
  ext_board_mess_header_t* const header=(ext_board_mess_header_t*)&pbuff[0];
  uint8_t* const payload=(uint8_t* const)&pbuff[BLE_NFC_EXT_BOARD_PACKET_PAYLOAD_OFFSET()];
  
  header->type=req_mess_type;
  
  uint16_t txrx_timeout=300;
  
  ext_board_mess_type_t ans_mess_type;
  
  switch(header->type)
  {    
  case CMD_GET_FW_VERS:
    {
      ans_mess_type=CMD_ANS_GET_FW_VERS;
      
      header->len=0;
    }
    break;
    
  case CMD_GET_FW_VERS_VERBOSE:
    {
      ans_mess_type=CMD_ANS_GET_FW_VERS_VERBOSE;
      
      header->len=0;
    }
    break;
    
  case CMD_REBOOT:
    {
      ans_mess_type=CMD_ANS_REBOOT;
      
      header->len=0;
    }
    break;
    
  case CMD_SET_NFC_SCAN_PERIOD:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_SET_NFC_SCAN_PERIOD;
      
      ext_board_nfc_scan_period_t* req=(ext_board_nfc_scan_period_t*)payload;
      header->len=sizeof(*req);
      
      memcpy(req, req_arg, sizeof(*req));
    }
    break;
    
  case CMD_FIRMWARE_FILE:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      txrx_timeout=999;
        
      ans_mess_type=CMD_ANS_FIRMWARE_FILE;
      
      ext_board_mess_firmware_upd_cmd_t* req=(ext_board_mess_firmware_upd_cmd_t*)payload;
      header->len=sizeof(*req);
      const uint16_t max_data_leght=pbuff_size-BLE_NFC_EXT_BOARD_PACKET_LEN(header->len);
            
      const ext_board_mess_firmware_upd_cmd_aux_t* req_aux=(const ext_board_mess_firmware_upd_cmd_aux_t*)req_arg;
      
      req->length=req_aux->length;
      req->offset=req_aux->offset;
      
      if(req->length>max_data_leght)
      {
        return -1;
      }
      
      //        
      if(req_aux->data!=NULL)
      {
        memcpy((uint8_t*)req+sizeof(*req), req_aux->data, req->length);
      }
      
      header->len+=req->length;
    }
    break;
    
  case CMD_BLE_INIT:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_BLE_INIT;
            
      ext_board_ble_init_t* req=(ext_board_ble_init_t*)payload;
      header->len=sizeof(*req);
      
      memcpy(req, req_arg, sizeof(*req));
    }
    break;
    
  case CMD_BLE_CONNECTION_COUNT:
    {
      ans_mess_type=CMD_ANS_BLE_CONNECTION_COUNT;
      
      header->len=0;
    }
    break;
    
  case CMD_BLE_STATUS:
    {
      ans_mess_type=CMD_ANS_BLE_STATUS;
      
      header->len=0;
    }
    break;
    
  case CMD_BLE_CONNECTION_STATUS:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_BLE_CONNECTION_STATUS;
            
      ext_board_ble_connection_id_t* req=(ext_board_ble_connection_id_t*)payload;
      header->len=sizeof(*req);
      
      memcpy(req, req_arg, sizeof(*req));
    }
    break;
    
  case CMD_BLE_CONNECT:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_BLE_CONNECT;
            
      ext_board_ble_connection_id_t* req=(ext_board_ble_connection_id_t*)payload;
      header->len=sizeof(*req);
      
      memcpy(req, req_arg, sizeof(*req));
    }
    break;
    
  case CMD_BLE_SEND_DATA:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
            
      ans_mess_type=CMD_ANS_BLE_SEND_DATA;
            
      ext_board_ble_send_data_t* req=(ext_board_ble_send_data_t*)payload;
      header->len=sizeof(*req);
      const uint16_t max_data_leght=pbuff_size-BLE_NFC_EXT_BOARD_PACKET_LEN(header->len);
      
      const ext_board_ble_send_data_aux_t* req_aux=(const ext_board_ble_send_data_aux_t*)req_arg;
      
      req->conn_id=req_aux->conn_id;
      req->length=req_aux->length;
      
      if(req->length>max_data_leght)
      {
        return -1;
      }
      
      //  ?
      if(req_aux->data!=NULL)
      {
        memcpy((uint8_t*)req+sizeof(*req), req_aux->data, req->length);
      }
      
      header->len+=req->length;
    }
    break;
    
  case CMD_BLE_RECEIVE_DATA:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_BLE_RECEIVE_DATA;
            
      ext_board_ble_receive_data_t* req=(ext_board_ble_receive_data_t*)payload;
      header->len=sizeof(*req);
      
      memcpy(req, req_arg, sizeof(*req));
      
      if(req->length>ext_board_get_ble_max_receive_data_len())
      {
        return -1;
      }
    }
    break;
    
  case CMD_BLE_DISCONNECT:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_BLE_DISCONNECT;
                  
      ext_board_ble_connection_id_t* req=(ext_board_ble_connection_id_t*)payload;
      header->len=sizeof(*req);
      
      req->conn_id=0;
    }
    break;
    
  case CMD_BLE_BEACON_VISIBLE:
    {
      ans_mess_type=CMD_ANS_BLE_BEACON_VISIBLE;
      
      uint16_t* req=(uint16_t*)payload;
      header->len=sizeof(*req);
      
      req[0]=ext_board_get_ble_max_receive_visible_beacons_count();
    }
    break;
    
  case CMD_NFC_VISIBLE:
    {
      ans_mess_type=CMD_ANS_NFC_VISIBLE;
      
      header->len=0;
    }
    break;
    
  case CMD_DFU_RESULT:
    {
      ans_mess_type=CMD_ANS_DFU_RESULT;
      
      header->len=0;
    }
    break;
    
  case CMD_SET_BLE_NAME:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_SET_BLE_NAME;
      
      ext_board_ble_name_t* req=(ext_board_ble_name_t*)payload;
      
      int s_len=snprintf(req->name, sizeof(req->name), "%s", (const char*)req_arg);
      
      if(s_len<0 || s_len>=sizeof(req->name))
      {
        return -1;
      }
      
      header->len=(uint16_t)s_len+1;
    }
    break;
    
  case CMD_GET_BLE_NAME:
    {
      ans_mess_type=CMD_ANS_GET_BLE_NAME;
      
      header->len=0;
    }
    break;
    
  case CMD_SET_PASS_KEY:
    {
      if(req_arg==NULL)
      {
        return -1;
      }
      
      if(strnlen((const char*)req_arg, sizeof(ext_board_ble_passkey_t))!=sizeof(ext_board_ble_passkey_t))
      {
        return -1;
      }
      
      ans_mess_type=CMD_ANS_SET_PASS_KEY;
      
      ext_board_ble_passkey_t* req=(ext_board_ble_passkey_t*)payload;
      header->len=sizeof(*req);
      
      memcpy(req, req_arg, sizeof(*req));
    }
    break;
    
  case CMD_GET_PASS_KEY:
    {
      ans_mess_type=CMD_ANS_GET_PASS_KEY;
      
      header->len=0;
    }
    break;
    
  case CMD_GET_BLE_BOARD_MAC:
    {
      ans_mess_type=CMD_ANS_GET_BLE_BOARD_MAC;
      
      header->len=0;
    }
    break;
    
  case CMD_APPROTECT_ENABLE:
    {
      ans_mess_type=CMD_ANS_APPROTECT_ENABLE;
      
      header->len=0;
    }
    break;
  case CMD_STOKEN_REQUEST:
    {
      ans_mess_type=CMD_ANS_STOKEN_REQUEST;
      
      header->len=0;
    }
    break;
    
  default:
    if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
    {
      __PRINTF("%s: %s: unexpected tx message type: %s(%u)\n", \
        dbg_name, __func__, ext_board_get_verbose_mess_type(header->type), header->type);
    }
    
    return -1;
  }
    
  header->sync_byte=BLE_BOARD_SYNC_BYTE;
  header->counter=ext_board_get_counter(true);
  
  uint8_t mtoken[TOKEN_SIZE];
  
  rand_func(mtoken, sizeof(mtoken));
  
  memcpy(header->mtoken, mtoken, sizeof(header->mtoken));
  memcpy(header->stoken, stoken, sizeof(header->stoken));
   
  rand_func(header->nonce, sizeof(header->nonce));
  
  ext_board_mess_type_t mess_type=header->type;
  
  if(0 != chacha20_crypt(&crypt_ctx, crypt_key,
                         header->nonce,
                         header->counter,
                         header->len+sizeof(header->mtoken)+sizeof(header->stoken)+sizeof(header->type),
                         header->mtoken,
                         header->mtoken))
  {
    return -1;
  }
  
  ext_board_add_crc(header);
  
  if(BLE_EXT_BOARD_DEBUG_LEVEL>=HDEBUG_L)
  {
    __BIN_BUFF_PRINTF(pbuff, BLE_NFC_EXT_BOARD_PACKET_LEN(header->len), "%s: %s(TX):\n", dbg_name, ext_board_get_verbose_mess_type(mess_type));
  }
  else if(BLE_EXT_BOARD_DEBUG_LEVEL>=MDEBUG_L)
  {
    __PRINTF("%s: %s(TX)\n", dbg_name, ext_board_get_verbose_mess_type(mess_type));
  }
    
  if(interface_txrx==NULL)
  {
    return -1;
  }
  
  int16_t rlen=interface_txrx(pbuff, BLE_NFC_EXT_BOARD_PACKET_LEN(header->len), pbuff, pbuff_size, txrx_timeout);
  
  if(rlen<0)
  {
    if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
    {
      __PRINTF("%s: %s: interface error: %hhi\n", dbg_name, __func__, rlen);
    }
    
    return -1;
  }
    
  const char* verbose;
  if(ext_board_check_packet(header, rlen, &verbose)<=0)
  {
    if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
    {
      __PRINTF("%s: rx packet error: %s\n", dbg_name, verbose);
    }
    
    return -1;
  }
  
  if(BLE_EXT_BOARD_DEBUG_LEVEL>=HDEBUG_L)
  {
    __BIN_BUFF_PRINTF(pbuff, rlen, "%s: %s(RX):\n", dbg_name, "Crypted");
  }
  
  if(0 != chacha20_crypt(&crypt_ctx, crypt_key,
                         header->nonce,
                         header->counter,
                         header->len+sizeof(header->mtoken)+sizeof(header->stoken)+sizeof(header->type),
                         header->mtoken,
                         header->mtoken))
  {
    return -1;
  }
  
  if(BLE_EXT_BOARD_DEBUG_LEVEL>=HDEBUG_L)
  {
    __BIN_BUFF_PRINTF(pbuff, rlen, "%s: %s(RX):\n", dbg_name, ext_board_get_verbose_mess_type(header->type));
  }
  else if(BLE_EXT_BOARD_DEBUG_LEVEL>=MDEBUG_L)
  {
    __PRINTF("%s: %s(RX)\n", dbg_name, ext_board_get_verbose_mess_type(header->type));
  }
  
  if(memcmp(header->mtoken, mtoken, sizeof(header->mtoken))!=0)
  {
    if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
    {
      __PRINTF("%s: wrong rx mtoken\n", dbg_name);
    }
    
    return -1;
  }
    
  if(header->type!=ans_mess_type)
  {    
    if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
    {
      __PRINTF("%s: unexpected rx ans message type: %s(%u), we wait: %s(%u)\n", \
        dbg_name, ext_board_get_verbose_mess_type(header->type), header->type, ext_board_get_verbose_mess_type(ans_mess_type), ans_mess_type);
    }
    
    if(header->type==CMD_ANS_STOKEN_REQUEST)
    {
      if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
      {
        __PRINTF("%s: stoken update...\n", dbg_name);
      }
      memcpy(stoken, header->stoken, sizeof(stoken));
    }
    
    return -1;
  }
  
  if((header->counter)!=ext_board_get_counter(false))
  {
    if(BLE_EXT_BOARD_DEBUG_LEVEL>=LDEBUG_L)
    {
      __PRINTF("%s: unexpected rx counter %u, we wait: %u\n", \
        dbg_name, header->counter, ext_board_get_counter(false));
    }
    
    return -1;
  }
  
  memcpy(stoken, header->stoken, sizeof(stoken));
    
  if(ans_arg!=NULL)
  {
    ans_arg[0]=payload;
  }
      
  return header->len;
}

int16_t __cmd_get_fw_vers_verbose(const char** device_marker, const char** bootloader_version, const char** application_version)
{
  const char* ans;
    
  int16_t res=cmd_get_fw_vers_verbose(&ans);
  
  if(device_marker!=NULL)       device_marker[0]="";
  if(bootloader_version!=NULL)  bootloader_version[0]="";
  if(application_version!=NULL) application_version[0]="";
  
  if(res>0)
  {
    if(device_marker!=NULL) device_marker[0]=ans;
    
    for(uint16_t i=0, j=0; i<(uint16_t)res; i++)
    {
      if(ans[i]==0 && i)
      {
        if(j==0)
        {
          if(bootloader_version!=NULL)  bootloader_version[0]=ans+i+1;
        }
        else if(j==1)
        {
          if(application_version!=NULL) application_version[0]=ans+i+1;
          break;
        }
        j++;
      }
    }
  }
  
  return res;
}
