#ifndef LIB_TIME_H
#define LIB_TIME_H

#include <FreeRTOS.h>
#include <task.h>

namespace NTime
{
class TTime
{
public:
    using TTick = TickType_t;

public:
    TTime();
    explicit TTime(TTick ticks);

    TTime(const TTime& other);
    TTime& operator=(const TTime& other);

    static TTime Now();
    static TTime MilliSeconds(TTick value);
    static TTime Seconds(TTick value);
    static TTime Minutes(TTick value);
    static TTime Hours(TTick value);

    TTick GetTicks() const { return this->Ticks; }

    TTime& operator+(const TTime& rhs);
    TTime& operator+(TTick ticks);
    TTime& operator-(const TTime& rhs);
    TTime& operator-(TTick ticks);
    TTime& operator*(const TTime& rhs);
    TTime& operator*(TTick ticks);
    TTime& operator/(const TTime& rhs);
    TTime& operator/(TTick ticks);

private:
    TTick Ticks = 0;
};

inline TTime::TTime()
    : Ticks(0)
{
}

inline TTime::TTime(TTick ticks)
    : Ticks(ticks)
{
}

inline TTime::TTime(const TTime& other)
    : Ticks(other.Ticks)
{
}

inline TTime& TTime::operator=(const TTime& other) {
    this->Ticks = other.Ticks;
    return *this;
}

inline TTime TTime::Now() {
    auto ticks = static_cast<TTick>(xTaskGetTickCount());
    return TTime(ticks);
}

inline TTime TTime::MilliSeconds(TTick value) {
    return TTime(value);
}

inline TTime TTime::Seconds(TTick value) {
    return TTime(MilliSeconds(value) * 1000);
}

inline TTime TTime::Minutes(TTick value) {
    return TTime(Seconds(value) * 60);
}

inline TTime TTime::Hours(TTick value) {
    return TTime(Minutes(value) * 60);
}

/// External time operators
inline bool operator==(const TTime& lhs, const TTime& rhs) {
    return lhs.GetTicks() == rhs.GetTicks();
}

inline bool operator!=(const TTime& lhs, const TTime& rhs) {
    return lhs.GetTicks() != rhs.GetTicks();
}

inline bool operator<(const TTime& lhs, const TTime& rhs) {
    return lhs.GetTicks() < rhs.GetTicks();
}

inline bool operator>(const TTime& lhs, const TTime& rhs) {
    return lhs.GetTicks() > rhs.GetTicks();
}

inline bool operator<=(const TTime& lhs, const TTime& rhs) {
    return lhs.GetTicks() <= rhs.GetTicks();
}

inline bool operator>=(const TTime& lhs, const TTime& rhs) {
    return lhs.GetTicks() >= rhs.GetTicks();
}

/// Internal time operators
inline TTime& TTime::operator+(const TTime& rhs) {
    this->Ticks += rhs.Ticks;
    return *this;
}

inline TTime& TTime::operator+(TTick ticks) {
    this->Ticks += ticks;
    return *this;
}

inline TTime& TTime::operator-(const TTime& rhs) {
    this->Ticks -= rhs.Ticks;
    return *this;
}

inline TTime& TTime::operator-(TTick ticks) {
    this->Ticks -= ticks;
    return *this;
}

inline TTime& TTime::operator*(const TTime& rhs) {
    this->Ticks *= rhs.Ticks;
    return *this;
}

inline TTime& TTime::operator*(TTick ticks) {
    this->Ticks *= ticks;
    return *this;
}

inline TTime& TTime::operator/(const TTime& rhs) {
    this->Ticks /= rhs.Ticks;
    return *this;
}

inline TTime& TTime::operator/(TTick ticks) {
    this->Ticks /= ticks;
    return *this;
}
}

#endif /* LIB_TIME_H */
