/**
  ******************************************************************************
  * File Name          : gps_manager.cpp
  * Description        : Менеджер GPS
  *
  *
  ******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include <math.h>

#include "gps_manager.h"
#include "syscom_manager.h"
#include "debug_port.h"
#include "UBX_protocol/ubx_protocol.h"
#include "Black_box/Black_box.h"
#include "gps_uart.h"
#include "leds_driver.h"
#include "ext_string_lib/ext_string.h"
#include "rtc_driver.h"

#if (!defined(UINT16_MAX))
#define UINT16_MAX __UINT16_T_MAX__
#endif //(!defined(UINT16_MAX))

#if (!defined(INT8_MAX))
#define INT8_MAX __INT8_T_MAX__
#endif //(!defined(INT8_MAX))

#if !defined(GPS_MANAGER_MODEL_2_PRESENT)
#error not difined GPS_MANAGER_MODEL_2_PRESENT
#endif //!defined(GPS_MANAGER_MODEL_2_PRESENT)

#if defined(GPS_USB_FORWARDING)
#include "USB_CDC_STM32/usbd_cdc_interface.h"
#endif //GPS_USB_FORWARDING

//В блоках DEVICE_TYPE_VEGA_MT_32K_LTE, DEVICE_TYPE_VEGA_MT_32K_LTE_V2, DEVICE_TYPE_VEGA_MT_32K_LTE_V3, DEVICE_TYPE_VEGA_MT_25K и DEVICE_TYPE_VEGA_MT_27 при отсутвии зажигания фиксированные настройки генерации точек трека: 180s, 300m, 0.

/* Global variables-----------------------------------------------------------*/
extern CBlack_box Black_box;

const TTrack_param CGPS_manager::ign_off_track_param=
{
  .time_step=3*60,
  .distance_step=300,
  .course_step=0,
};

const psave_settings_t CGPS_manager::psave_settings=
{
  .use_move_det_opt=1,
  .use_can_act=1,
  .use_ign=1,
  .ignore_track_point_counter=20,
  .valid_track_point_counter=5,
  .firts_position_fixed_ignor_timeS=8*60,
  .position_fixed_ignor_timeS=90,
  .first_end_wait_fix_timeS=10*60,
  .end_wait_fix_timeS=5*60,

  .can_inact_valid_timeS=1*60,
  .ign_off_valid_timeS=1*60,

  .pos_fixed_sleep_timeS=30*60,
  .pos_not_fixed_1_sleep_timeS=10*60,
  .pos_not_fixed_2_sleep_timeS=5*60,
};

// GPS менеджер
CGPS_manager GPS_manager(GPS_MANAGER_NAME, &Syscom_manager.input_queue, GPS_MANAGER_INPUT_CUEUE_SIZE, GPS_MANAGER_PRIORITY, Start_gps_manager, GPS_MANAGER_STACK_SIZE);

/* C++ code-------------------------------------------------------------------*/
// Конструктор GPS-менеджера
CGPS_manager::CGPS_manager(const char* mngr_name, QueueHandle_t* output_queue, uint8_t input_queue_size, uint8_t priority, TaskFunction_t task_starter, uint16_t stack_size):
  diag_timer("gps_diag", 0), CManager(mngr_name, output_queue, input_queue_size, priority, task_starter, stack_size) , NMEA(&this->state)
{
  this->name = mngr_name;
  // Обнуление текущего состояния
  memset(&this->state, 0x00, sizeof(this->state));
  // Обнуление gnss_raw_state
  this->Reset_gnss_raw_state(false);
  // Принудительная фиксация остановки либо движения
  this->state.in_move = 0;
  // Включение трансляции трека
  this->translate_track = 1;
  // По умолчанию приемник выключен
  this->state.is_on = 0;
  // По умолчанию диагностика в 0
  this->state.receiver_is_ok = 0;
  // Положение не зафиксировано
  this->position_fixed = 0;
  // Трансляция трека включена
  translate_track = 1;
  // Счетчик остановки в ноль
  stop_counter = 0;
  // Инициализируем счетчик ошибок NMEA
  nmea_err_counter = NMEA_MAX_ERROR_COUNT;

  // Копирование текущего состояния в систему
  memcpy(&System.gnss_state, &this->state, sizeof(System.gnss_state));
  // Обнуление последних координат
  this->last_lat = 0;
  this->last_lon = 0;

  // Инициализируем настройки невалидными. Если они есть файле настройки, то применятся из файла, если нет, то применятся стандартные
  memset(&System.track_settings.nav_systems, 0xFF, sizeof(System.track_settings.nav_systems));
  memset(&System.track_settings.extended_filters, 0xFF, sizeof(System.track_settings.extended_filters));
  System.track_settings.extended_filters.p_acc=100;
  System.track_settings.time_step_with_ign_off=0xFFFF;

  isSetFastGpsMode = false;
}

void CGPS_manager::reset_psave_state(void)
{
  psave.in_psave=0;

  psave.move_det=0;

  psave.can_act=0;
  psave.can_inact_valid_timeS=GetSecondsFromStartup()+psave_settings.can_inact_valid_timeS;

  psave.ign=0;
  psave.ign_off_valid_timeS=GetSecondsFromStartup()+psave_settings.ign_off_valid_timeS;

  psave.led_tim=0;
  psave.end_wait_fix_timeS=GetSecondsFromStartup()+psave_settings.first_end_wait_fix_timeS;
  psave.position_fixed_ignor_end_timeS=GetSecondsFromStartup()+psave_settings.firts_position_fixed_ignor_timeS;

  psave.ignore_track_point_counter=psave_settings.ignore_track_point_counter;
  psave.valid_track_point_counter=psave_settings.valid_track_point_counter;
}

void CGPS_manager::parse_receiver_version(void)
{
  char* ptr;
  uint8_t ver_idx;
  const uint8_t max_ver_size=sizeof(this->state.receiver_version)-1;
  const char* parse_buff;
  int16_t parse_size;

  parse_size=gps_uart_read(this->gps_rx_ctx.rx_buff, sizeof(this->gps_rx_ctx.rx_buff));
  parse_buff=(const char*)this->gps_rx_ctx.rx_buff;

  if(parse_size<16)
  {
    this->state.receiver_version[0]='\0';
    return;
  }

  //смещаемя, т.к. 5м байтом приходит '\0'
  parse_buff+=8;
  parse_size-=8;

  ver_idx=0;
  if(!(parse_buff==NULL || parse_size==0))
  {
    ptr=strnstr(parse_buff, "HW ", parse_size);
    if(ptr!=NULL)
    {
      while(ver_idx<max_ver_size)
      {
        this->state.receiver_version[ver_idx]=*ptr;
        ptr++; ver_idx++;
        if(*ptr=='*') break;
      }
    }

    if(ver_idx<max_ver_size)
    {
      this->state.receiver_version[ver_idx]=' ';
      ver_idx++;
    }

    ptr=strnstr(parse_buff, "ROM CORE ", parse_size);
    if(ptr!=NULL)
    {
      while(ver_idx<max_ver_size)
      {
        this->state.receiver_version[ver_idx]=*ptr;
        ptr++; ver_idx++;
        if(*ptr=='*') break;
      }
    }
  }
  this->state.receiver_version[ver_idx]='\0';
}

void CGPS_manager::SetDefaultExtFiltersSettings(TNav_extended_filters* settings)
{
  // default for Yandex
  settings->cno_thresh=10;
  settings->cno_thresh_num_svs=5;
  settings->min_elev=8;
  settings->static_hold_thresh=100;
  settings->static_hold_max_dist=5;
  settings->p_acc=100;
}

void CGPS_manager::SetDefaultNavSystemSettings(TNav_system_settings* settings, const char** rmc)
{
  // EVA-M8M-0 default
  settings->gps=1;
  settings->glonass=1;
  settings->galileo=0;
  settings->beidou=0;
  settings->sbass=1;
  settings->qzss=1;

  if(rmc!=NULL)
  {
    *rmc="$GNRMC";
  }
}

bool CGPS_manager::CheckExtFiltersSettings(TNav_extended_filters* settings)
{
  if(settings->min_elev>90 || settings->cno_thresh_num_svs>16)
  {
    return false;
  }
  else
  {
    return true;
  }
}

bool CGPS_manager::CheckNavSystemsSettings(TNav_system_settings* settings, const char** rmc)
{
  // Возможные режимы работы приемника
  static const struct
  {
    bool gps:1;
    bool galileo:1;
    bool glonass:1;
    bool beidou:1;
    const char* rmc;
  }support_gnss_mode[]=
  {
    {.gps=1, .galileo=0, .glonass=0, .beidou=0, .rmc="$GPRMC"},
    {.gps=0, .galileo=1, .glonass=0, .beidou=0, .rmc="$GARMC"},
    {.gps=0, .galileo=0, .glonass=1, .beidou=0, .rmc="$GLRMC"},
    {.gps=0, .galileo=0, .glonass=0, .beidou=1, .rmc="$GBRMC"},
    {.gps=1, .galileo=1, .glonass=0, .beidou=0, .rmc="$GNRMC"},
    {.gps=1, .galileo=1, .glonass=1, .beidou=0, .rmc="$GNRMC"},
    {.gps=1, .galileo=1, .glonass=0, .beidou=1, .rmc="$GNRMC"},
    {.gps=1, .galileo=0, .glonass=1, .beidou=0, .rmc="$GNRMC"},
    {.gps=1, .galileo=0, .glonass=0, .beidou=1, .rmc="$GNRMC"},
    {.gps=0, .galileo=1, .glonass=1, .beidou=0, .rmc="$GNRMC"},
    {.gps=0, .galileo=1, .glonass=0, .beidou=1, .rmc="$GNRMC"},
    {.gps=0, .galileo=0, .glonass=1, .beidou=1, .rmc="$GNRMC"},
  };

  for(uint8_t i=0; i<sizeof(support_gnss_mode)/sizeof(support_gnss_mode[0]); i++)
  {
    if(support_gnss_mode[i].gps==settings->gps && support_gnss_mode[i].galileo==settings->galileo \
      && support_gnss_mode[i].glonass==settings->glonass && support_gnss_mode[i].beidou==settings->beidou)
    {
      if(rmc!=NULL)
      {
        *rmc=support_gnss_mode[i].rmc;
      }
      return true;
    }
  }

  return false;
}

// Инициализация менеджера
void CGPS_manager::Init (void)
{
  // Инициализация пинов ON и BACKUP
  GPIO_InitTypeDef GPIO_InitStruct;

  GPIO_InitStruct.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStruct.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStruct.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStruct.GPIO_Pin = GPS_ON_PIN;
  GPIO_Init(GPS_ON_GPIO, &GPIO_InitStruct);

  GPIO_InitStruct.GPIO_Pin = GPS_BACKUP_PIN;
  GPIO_Init(GPS_BACKUP_GPIO, &GPIO_InitStruct);

  // Запуск приемника
  LOG("%s: Start\n", this->name);
  // GNSS ON
  GPIO_WriteBit(GPS_ON_GPIO, GPS_ON_PIN, GPS_ON_STATE);
  // GNSS BACKUP
  GPIO_WriteBit(GPS_BACKUP_GPIO, GPS_BACKUP_PIN, Bit_SET);
  vTaskDelay(20);
  // Инициализация GPS USART интерфейса
    sync_rx_string="$GNRMC";
    gps_uart_init(9600, "RMC");
  // Действия при первом старте прошивки
  if(System.first_start)
  {
#if defined(UBXLOX_NMEA_RECEIVER_PRESENT)
#if !defined(UBLOX_GNSS_EXTERNAL_RTC_QUARTZ_PRESENT)
    vTaskDelay(2000);

    // Включение тактирования часов RTC GPS от внутреннего источника
    gps_uart_write((uint8_t*)ubx_ext_crystal_off_message, sizeof(ubx_ext_crystal_off_message));
    vTaskDelay(1000);

    // GNSS OFF
    GPIO_WriteBit(GPS_ON_GPIO, GPS_ON_PIN, GPS_OFF_STATE);
    // GNSS BACKUP OFF
    GPIO_WriteBit(GPS_BACKUP_GPIO, GPS_BACKUP_PIN, Bit_RESET);

    vTaskDelay(300);

    // GNSS ON
    GPIO_WriteBit(GPS_ON_GPIO, GPS_ON_PIN, GPS_ON_STATE);
    // GNSS BACKUP ON
    GPIO_WriteBit(GPS_BACKUP_GPIO, GPS_BACKUP_PIN, Bit_SET);

    vTaskDelay(1000);
#endif //!defined(UBLOX_GNSS_EXTERNAL_RTC_QUARTZ_PRESENT)
#endif //UBXLOX_NMEA_RECEIVER_PRESENT
  }

  // Загрузка одометра и счетчика поездок
  this->state.trip_counter = System.gnss_state.trip_counter;
  this->state.odometer = System.gnss_state.odometer;

  // Ожидание первых данных от приемника
  this->diag_timer.Start(SEC*5);
  for(;;)
  {
    if(gps_uart_wait_sync(50))
    {
      this->parse_receiver_version();

      gps_uart_write((uint8_t*)ubx_baud_115200_message, sizeof(ubx_baud_115200_message));
      vTaskDelay(2);//ждем пока удёт последний символ
      gps_uart_deinit();
      gps_uart_init(115200, "RMC");

      this->state.receiver_is_ok = 1;

      //ждем чтобы чтобы приемник успел переключиться
      vTaskDelay(500);
      gps_uart_flush_rx();

      //Включение сообщения HW, по которому определяется глушение
      if(true)
      {
        T_ubx_msg* packet=(T_ubx_msg*)&this->gps_rx_ctx.rx_buff[0];
        T_ubx_cfg_rates_payload* rates_payload=(T_ubx_cfg_rates_payload*)&this->gps_rx_ctx.rx_buff[UBX_HEADER_SIZE];

        memset(this->gps_rx_ctx.rx_buff, 0, sizeof(this->gps_rx_ctx.rx_buff));
        packet->header = UBX_HEADER;
        packet->clas = UBX_MSG_CFG;
        packet->message_id = UBX_MSG_CFG_RATE;
        packet->len = UBX_MSG_CFG_RATE_LEN;
        //packet.pPayload = &rates_payload;
        rates_payload->message_class = UBX_MSG_CFG_RATE_MON;
        rates_payload->message_id = UBX_MSG_CFG_RATE_MON_HW;
        rates_payload->message_rate = GPS_UBX_MESSAGE_RATE;
        ubx_add_crc(packet, this->gps_rx_ctx.rx_buff);

        gps_uart_write(this->gps_rx_ctx.rx_buff, UBX_HEADER_SIZE+packet->len+UBX_CRC_SIZE);
        vTaskDelay(10);
        gps_uart_flush_rx();
      }

      //Включение сообщения, по которому определяется подмена сиганла
      if(true)
      {
        T_ubx_msg* packet=(T_ubx_msg*)&this->gps_rx_ctx.rx_buff[0];
        T_ubx_cfg_rates_payload* rates_payload=(T_ubx_cfg_rates_payload*)&this->gps_rx_ctx.rx_buff[UBX_HEADER_SIZE];

        memset(this->gps_rx_ctx.rx_buff, 0, sizeof(this->gps_rx_ctx.rx_buff));
        packet->header = UBX_HEADER;
        packet->clas = UBX_MSG_CFG;
        packet->message_id = UBX_MSG_CFG_RATE;
        packet->len = UBX_MSG_CFG_RATE_LEN;
        //packet.pPayload = &rates_payload;
        rates_payload->message_class = UBX_MSG_CFG_RATE_NAV;
        rates_payload->message_id = UBX_MSG_CFG_RATE_NAV_STATUS;
        rates_payload->message_rate = GPS_UBX_MESSAGE_RATE;
        ubx_add_crc(packet, this->gps_rx_ctx.rx_buff);

        gps_uart_write(this->gps_rx_ctx.rx_buff, UBX_HEADER_SIZE+packet->len+UBX_CRC_SIZE);
        vTaskDelay(10);
        gps_uart_flush_rx();
      }

        //Настройка фильтров
        if(this->use_extended_filters_settings)
        {
          bool is_settings_wrong=false;

          System.Grab();

          if(!this->CheckExtFiltersSettings(&System.track_settings.extended_filters))
          {
            this->SetDefaultExtFiltersSettings(&System.track_settings.extended_filters);
            is_settings_wrong=true;
          }

          memcpy(&this->extended_filters_settings, &System.track_settings.extended_filters, sizeof(this->extended_filters_settings));

          System.Release();

          if(is_settings_wrong)
          {
            LOG("%s: wrong filters configuration, using default setting\n", this->name);

            this->output_message.source = GPS_MANAGER;
            this->output_message.code = MESSAGE_GPS_SAVE_BASE;
            this->Send_message(this->output_queue);
          }

          if(false)
          {
            static const uint8_t nav5_raw[]=
            {0xB5, 0x62, 0x06, 0x24, 0x24, 0x00, 0xFF, 0xFF, 0x04, 0x03, 0x00, 0x00, 0x00, 0x00, 0x10, 0x27, 0x00, 0x00, 0x0A, 0x00, 0xFA,
            0x00, 0xFA, 0x00, 0x32, 0x00, 0x5E, 0x01, 0x8C, 0x3C, 0x05, 0x0C, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0xF7, 0x48};

            static const uint8_t navx5_raw[]=
            {0xB5, 0x62, 0x06, 0x23, 0x28, 0x00, 0x02, 0x00, 0x4C, 0x66, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x20, 0x0C, 0x00, 0x01,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x5C, 0x64};

            static const T_ubx_cfg_nav5_payload* nav5=(const T_ubx_cfg_nav5_payload*)&nav5_raw[UBX_HEADER_SIZE];
            static const T_ubx_cfg_navx5_payload* navx5=(const T_ubx_cfg_navx5_payload*)&navx5_raw[UBX_HEADER_SIZE];

            navx5++; navx5--;
            nav5++; nav5--;
          }

          if(true)
          {
            T_ubx_msg* packet=(T_ubx_msg*)&this->gps_rx_ctx.rx_buff[0];
            T_ubx_cfg_nav5_payload* nav5_payload=(T_ubx_cfg_nav5_payload*)&this->gps_rx_ctx.rx_buff[UBX_HEADER_SIZE];

            memset(this->gps_rx_ctx.rx_buff, 0, sizeof(this->gps_rx_ctx.rx_buff));
            packet->header = UBX_HEADER;
            packet->clas = UBX_MSG_CFG;
            packet->message_id = UBX_MSG_CFG_NAV5;
            packet->len = UBX_MSG_CFG_NAV5_LEN;
            //packet.pPayload = &nav5_payload;
            nav5_payload->bit_mask.u16 = 0;
            nav5_payload->bit_mask.bf.dyn = 1;
            nav5_payload->bit_mask.bf.minEl = 1;
            nav5_payload->bit_mask.bf.posFixMode = 1;
            nav5_payload->bit_mask.bf.drLim = 0;
            nav5_payload->bit_mask.bf.posMask = 1;
            nav5_payload->bit_mask.bf.timeMask = 0;
            nav5_payload->bit_mask.bf.staticHoldMask = 1;
            nav5_payload->bit_mask.bf.dgpsMask = 0;
            nav5_payload->bit_mask.bf.cnoThreshold = 1;
            nav5_payload->bit_mask.bf.reserved1 = 0;
            nav5_payload->bit_mask.bf.utc = 0;
            nav5_payload->bit_mask.bf.reserved2 = 0;

            nav5_payload->dyn_model = UBX_MSG_CFG_NAV5_DYN_MODEL_AUTOMOTIVE;
            nav5_payload->fix_mode = UBX_MSG_CFG_NAV5_FIX_MODE_3DONLY;
            nav5_payload->fixed_alt = 0;
            nav5_payload->fixed_alt_var = 10000;
            nav5_payload->min_elev = (int8_t)this->extended_filters_settings.min_elev;
            nav5_payload->dr_limit = 0;
            nav5_payload->pdop = 250;
            nav5_payload->tdop = 250;
            nav5_payload->pacc = this->extended_filters_settings.p_acc;
            nav5_payload->tacc = 350;
            nav5_payload->static_hld_trshld = this->extended_filters_settings.static_hold_thresh;
            nav5_payload->dgps_tim = 60;
            nav5_payload->cno_thresh_num_svs = this->extended_filters_settings.cno_thresh_num_svs;
            nav5_payload->cno_thresh = this->extended_filters_settings.cno_thresh;
            nav5_payload->static_hold_max_dist = this->extended_filters_settings.static_hold_max_dist;
            nav5_payload->utc_standart = 0;

            ubx_add_crc(packet, this->gps_rx_ctx.rx_buff);

            gps_uart_write(this->gps_rx_ctx.rx_buff, UBX_HEADER_SIZE+packet->len+UBX_CRC_SIZE);
            vTaskDelay(10);
            gps_uart_flush_rx();
          }

          if(false)
          {
            T_ubx_msg* packet=(T_ubx_msg*)&this->gps_rx_ctx.rx_buff[0];
            T_ubx_cfg_navx5_payload* navx5_payload=(T_ubx_cfg_navx5_payload*)&this->gps_rx_ctx.rx_buff[UBX_HEADER_SIZE];

            memset(this->gps_rx_ctx.rx_buff, 0, sizeof(this->gps_rx_ctx.rx_buff));
            packet->header = UBX_HEADER;
            packet->clas = UBX_MSG_CFG;
            packet->message_id = UBX_MSG_CFG_NAVX5;
            packet->len = UBX_MSG_CFG_NAVX5_LEN;
            //packet.pPayload = &navx5_payload;
            navx5_payload->version = 2;

            navx5_payload->mask1.u16 = 0;
            navx5_payload->mask1.bf.reserved1 = 0;
            navx5_payload->mask1.bf.minMax = 1;
            navx5_payload->mask1.bf.minCno = 1;
            navx5_payload->mask1.bf.reserved2 = 0;
            navx5_payload->mask1.bf.initial3dfix = 1;
            navx5_payload->mask1.bf.reserved3 = 0;
            navx5_payload->mask1.bf.wknRoll = 0;
            navx5_payload->mask1.bf.ackAid = 0;
            navx5_payload->mask1.bf.reserved4 = 0;
            navx5_payload->mask1.bf.ppp = 0;
            navx5_payload->mask1.bf.aop = 0;
            navx5_payload->mask1.bf.reserved5 = 0;

            navx5_payload->mask2.u32 = 0;
            navx5_payload->mask2.bf.reserved1 = 0;
            navx5_payload->mask2.bf.adr = 0;
            navx5_payload->mask2.bf.sigAttenComp = 0;
            navx5_payload->mask2.bf.reserved2 = 0;

            memset(navx5_payload->reserved1, 0x00, sizeof(navx5_payload->reserved1));

            navx5_payload->minSVs = this->extended_filters_settings.cno_thresh_num_svs;;
            navx5_payload->maxSVs = 32;
            navx5_payload->minCNO = this->extended_filters_settings.cno_thresh;

            navx5_payload->reserved2 = 0x00;

            navx5_payload->iniFix3D = 1;

            memset(navx5_payload->reserved3, 0x00, sizeof(navx5_payload->reserved3));

            navx5_payload->ackAiding = 0;
            navx5_payload->wknRollover = 0;
            navx5_payload->sigAttenCompMode = 0;

            navx5_payload->reserved4 = 0;
            memset(navx5_payload->reserved5, 0x00, sizeof(navx5_payload->reserved5));
            memset(navx5_payload->reserved6, 0x00, sizeof(navx5_payload->reserved6));

            navx5_payload->usePPP=0;
            navx5_payload->aopCfg.u8=0;

            memset(navx5_payload->reserved7, 0x00, sizeof(navx5_payload->reserved7));

            navx5_payload->aopOrbMaxErr = 100;

            memset(navx5_payload->reserved8, 0x00, sizeof(navx5_payload->reserved8));
            memset(navx5_payload->reserved9, 0x00, sizeof(navx5_payload->reserved9));

            navx5_payload->useAdr = 0;

            memset(navx5_payload->reserved10, 0x00, sizeof(navx5_payload->reserved10));
            memset(navx5_payload->reserved11, 0x00, sizeof(navx5_payload->reserved11));

            ubx_add_crc(packet, this->gps_rx_ctx.rx_buff);

            gps_uart_write(this->gps_rx_ctx.rx_buff, UBX_HEADER_SIZE+packet->len+UBX_CRC_SIZE);
            vTaskDelay(10);
            gps_uart_flush_rx();
          }
        }

      //Настройка используемых сигналов GNSS
      if(this->use_nav_systems_settings)
      {
        bool is_settings_wrong=false;

        System.Grab();

        if(!this->CheckNavSystemsSettings(&System.track_settings.nav_systems, &sync_rx_string))
        {
          this->SetDefaultNavSystemSettings(&System.track_settings.nav_systems, &sync_rx_string);
          is_settings_wrong=true;
        }

        memcpy(&this->nav_system_settings, &System.track_settings.nav_systems, sizeof(this->nav_system_settings));

        System.Release();

        if(is_settings_wrong)
        {
          LOG("%s: wrong navigation system configuration, using default setting\n", this->name);

          this->output_message.source = GPS_MANAGER;
          this->output_message.code = MESSAGE_GPS_SAVE_BASE;
          this->Send_message(this->output_queue);
        }

        T_ubx_msg* packet=(T_ubx_msg*)&this->gps_rx_ctx.rx_buff[0];
        T_ubx_cfg_gnss_payload* cfg_gnss_payload=(T_ubx_cfg_gnss_payload*)&this->gps_rx_ctx.rx_buff[UBX_HEADER_SIZE];

        memset(this->gps_rx_ctx.rx_buff, 0, sizeof(this->gps_rx_ctx.rx_buff));
        packet->header = UBX_HEADER;
        packet->clas = UBX_MSG_CFG;
        packet->message_id = UBX_MSG_CFG_GNSS;
        packet->len = sizeof(T_ubx_cfg_gnss_payload);

        cfg_gnss_payload->msgVer = 0;
        cfg_gnss_payload->numTrkChHw = 0;
        cfg_gnss_payload->numTrkChUse = 0xFF;
        cfg_gnss_payload->numConfigBlocks = sizeof(cfg_gnss_payload->ConfigBlocks)/sizeof(cfg_gnss_payload->ConfigBlocks[0]);

        cfg_gnss_payload->ConfigBlocks[0].gnssId = 0; //GPS
        cfg_gnss_payload->ConfigBlocks[0].resTrkCh = 8;
        cfg_gnss_payload->ConfigBlocks[0].maxTrkCh = 16;
        cfg_gnss_payload->ConfigBlocks[0].reserved1 = 0;
        cfg_gnss_payload->ConfigBlocks[0].flags.u32 = 0;
        if(this->nav_system_settings.gps) {cfg_gnss_payload->ConfigBlocks[0].flags.bf.enable = 1;}
        else                              {cfg_gnss_payload->ConfigBlocks[0].flags.bf.enable = 0;}
        cfg_gnss_payload->ConfigBlocks[0].flags.bf.reserved2 = 0;
        cfg_gnss_payload->ConfigBlocks[0].flags.bf.sigCfgMask = 0x01; //GPS L1C/A
        cfg_gnss_payload->ConfigBlocks[0].flags.bf.reserved3 = 0;

        cfg_gnss_payload->ConfigBlocks[1].gnssId = 1; //SBAS
        cfg_gnss_payload->ConfigBlocks[1].resTrkCh = 1;
        cfg_gnss_payload->ConfigBlocks[1].maxTrkCh = 3;
        cfg_gnss_payload->ConfigBlocks[1].reserved1 = 0;
        cfg_gnss_payload->ConfigBlocks[1].flags.u32 = 0;
        if(this->nav_system_settings.gps && this->nav_system_settings.sbass) {cfg_gnss_payload->ConfigBlocks[1].flags.bf.enable = 1;}
        else                                                                 {cfg_gnss_payload->ConfigBlocks[1].flags.bf.enable = 0;}
        cfg_gnss_payload->ConfigBlocks[1].flags.bf.reserved2 = 0;
        cfg_gnss_payload->ConfigBlocks[1].flags.bf.sigCfgMask =  0x01; //SBAS L1C/A
        cfg_gnss_payload->ConfigBlocks[1].flags.bf.reserved3 = 0;

        cfg_gnss_payload->ConfigBlocks[2].gnssId = 2; //Galileo
        cfg_gnss_payload->ConfigBlocks[2].resTrkCh = 4;
        cfg_gnss_payload->ConfigBlocks[2].maxTrkCh = 8;
        cfg_gnss_payload->ConfigBlocks[2].reserved1 = 0;
        cfg_gnss_payload->ConfigBlocks[2].flags.u32 = 0;
        if(this->nav_system_settings.galileo) {cfg_gnss_payload->ConfigBlocks[2].flags.bf.enable = 1;}
        else                                  {cfg_gnss_payload->ConfigBlocks[2].flags.bf.enable = 0;}
        cfg_gnss_payload->ConfigBlocks[2].flags.bf.reserved2 = 0;
        cfg_gnss_payload->ConfigBlocks[2].flags.bf.sigCfgMask =  0x01; //Galileo E1
        cfg_gnss_payload->ConfigBlocks[2].flags.bf.reserved3 = 0;

        cfg_gnss_payload->ConfigBlocks[3].gnssId = 3; //BeiDou
        cfg_gnss_payload->ConfigBlocks[3].resTrkCh = 8;
        cfg_gnss_payload->ConfigBlocks[3].maxTrkCh = 16;
        cfg_gnss_payload->ConfigBlocks[3].reserved1 = 0;
        cfg_gnss_payload->ConfigBlocks[3].flags.u32 = 0;
        if(this->nav_system_settings.beidou) {cfg_gnss_payload->ConfigBlocks[3].flags.bf.enable = 1;}
        else                                 {cfg_gnss_payload->ConfigBlocks[3].flags.bf.enable = 0;}
        cfg_gnss_payload->ConfigBlocks[3].flags.bf.reserved2 = 0;
        cfg_gnss_payload->ConfigBlocks[3].flags.bf.sigCfgMask =  0x01; //BeiDou B1I
        cfg_gnss_payload->ConfigBlocks[3].flags.bf.reserved3 = 0;

        cfg_gnss_payload->ConfigBlocks[4].gnssId = 4; //IMES
        cfg_gnss_payload->ConfigBlocks[4].resTrkCh= 0;
        cfg_gnss_payload->ConfigBlocks[4].maxTrkCh= 8;
        cfg_gnss_payload->ConfigBlocks[4].reserved1 = 0;
        cfg_gnss_payload->ConfigBlocks[4].flags.u32 = 0;
        cfg_gnss_payload->ConfigBlocks[4].flags.bf.enable = 0;
        cfg_gnss_payload->ConfigBlocks[4].flags.bf.reserved2 = 0;
        cfg_gnss_payload->ConfigBlocks[4].flags.bf.sigCfgMask = 0x01; //IMES L1
        cfg_gnss_payload->ConfigBlocks[4].flags.bf.reserved3 = 0;

        cfg_gnss_payload->ConfigBlocks[5].gnssId = 5; //QZSS
        cfg_gnss_payload->ConfigBlocks[5].resTrkCh = 0;
        cfg_gnss_payload->ConfigBlocks[5].maxTrkCh = 3;
        cfg_gnss_payload->ConfigBlocks[5].reserved1 = 0;
        cfg_gnss_payload->ConfigBlocks[5].flags.u32 = 0;
        if(this->nav_system_settings.gps && this->nav_system_settings.qzss) {cfg_gnss_payload->ConfigBlocks[5].flags.bf.enable = 1;}
        else                                                                {cfg_gnss_payload->ConfigBlocks[5].flags.bf.enable = 0;}
        cfg_gnss_payload->ConfigBlocks[5].flags.bf.reserved2 = 0;
        cfg_gnss_payload->ConfigBlocks[5].flags.bf.sigCfgMask = 0x01; //QZSS L1C/A
        cfg_gnss_payload->ConfigBlocks[5].flags.bf.reserved3 = 0;

        cfg_gnss_payload->ConfigBlocks[6].gnssId = 6; //GLONASS
        cfg_gnss_payload->ConfigBlocks[6].resTrkCh = 8;
        cfg_gnss_payload->ConfigBlocks[6].maxTrkCh = 14;
        cfg_gnss_payload->ConfigBlocks[6].reserved1 = 0;
        if(this->nav_system_settings.glonass) {cfg_gnss_payload->ConfigBlocks[6].flags.bf.enable = 1;}
        else                                  {cfg_gnss_payload->ConfigBlocks[6].flags.bf.enable = 0;}
        cfg_gnss_payload->ConfigBlocks[6].flags.bf.reserved2 = 0;
        cfg_gnss_payload->ConfigBlocks[6].flags.bf.sigCfgMask = 0x01; //GLONASS L1
        cfg_gnss_payload->ConfigBlocks[6].flags.bf.reserved3 = 0;

        ubx_add_crc(packet, this->gps_rx_ctx.rx_buff);

        gps_uart_write(this->gps_rx_ctx.rx_buff, UBX_HEADER_SIZE+packet->len+UBX_CRC_SIZE);
        vTaskDelay(10);
        gps_uart_flush_rx();

        if(this->nav_system_settings.galileo || this->nav_system_settings.beidou)
        {

          gps_uart_write((uint8_t*)ubx_setting_nmea4_1_message, sizeof(ubx_setting_nmea4_1_message));
          gps_uart_flush_rx();
        }
      }

      SetNavigationFrequency(DefaultNavigationFrequency);
      isSetFastGpsMode = false;

      //чтобы просралось
      vTaskDelay(200);

      gps_uart_flush_rx();
      this->gps_rx_ctx.rx_pos=0;
      this->diag_timer.Start(SEC*5);
      break;
    }
    if(this->diag_timer.Is_empty()) break;
  }

  this->jammed_fix_counter=0;
  this->spoof_fix_counter=0;

  this->state.in_move = 0;
  this->state.is_on = 1;
  Save_system_state();
}

// Деинициализация менеджера
void CGPS_manager::Deinit(void)
{
  // Деинициализация USART
  gps_uart_deinit();
  vTaskDelay(50);
  // GNSS OFF
  GPIO_WriteBit(GPS_ON_GPIO, GPS_ON_PIN, GPS_OFF_STATE);
  // Деинициализация пина ON
  GPIO_InitTypeDef GPIO_InitStruct;
  GPIO_InitStruct.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStruct.GPIO_Pin = GPS_ON_PIN;
  GPIO_Init(GPS_ON_GPIO, &GPIO_InitStruct);

  System.gnss_state.in_move = 0;
  // Приемник выключен
  System.gnss_state.is_on = 0;
}

// Разбор данных UBX протокола
void CGPS_manager::parse_ubx_data(const uint8_t* buf, uint16_t bufsize, uint8_t upx_packet_count)
{
  T_ubx_msg packet;
  memset(&packet, 0x00, sizeof(packet));

  for(uint8_t i = 0; i < upx_packet_count; i++)
  {
    packet = ubx_unpack_packet(buf, bufsize, i);

    switch(packet.message_id + packet.clas)
    {
      case (UBX_MSG_MON + UBX_MSG_MON_HW):
      {
        // Пришло сообщение HW (с данными о сигнале)
        uint16_t noise_level;
        uint16_t AGC_monitor;
        uint8_t  CW_jamming_indicator;
        uint8_t jammed;

        memcpy(&noise_level, ((uint8_t*)packet.pPayload + 16), 2);
        memcpy(&AGC_monitor, ((uint8_t*)packet.pPayload + 18), 2);
        memcpy(&CW_jamming_indicator, ((uint8_t*)packet.pPayload + 45), 1);

        if(AGC_monitor < 500 && CW_jamming_indicator < 30 && this->state.fix_type < FIX2D) jammed=1;
        else                                                                               jammed=0;

        if(jammed) this->jammed_fix_counter=20;

        if(this->state.jammed!=jammed)
        {
          if(jammed>0 || this->jammed_fix_counter==0)
          {
            LOG("%s: change jamming state %hhu->%hhu\n", this->name, this->state.jammed, jammed);
            this->state.jammed=jammed;
          }
          else if(jammed==0 && this->jammed_fix_counter>0)
          {
            this->jammed_fix_counter--;
          }
        }

        break;
      }
      case (UBX_MSG_NAV + UBX_MSG_NAV_STATUS):
      {
       __packed struct bitfield_flags2_struct
       {
         uint8_t psmState:2;
         uint8_t reserve:1;
         uint8_t spoofDetState:2;
         uint8_t reserve2:3;
       }bitfield_flags2;

       memcpy(&bitfield_flags2, ((uint8_t*)packet.pPayload + 7), 1);

       if(bitfield_flags2.spoofDetState>1) this->spoof_fix_counter=10;

       if(this->state.spoof!=bitfield_flags2.spoofDetState)
       {
         if(bitfield_flags2.spoofDetState>1 || this->spoof_fix_counter==0)
         {
           LOG("%s: change spoof state %hhu->%hhu\n", this->name, this->state.spoof, bitfield_flags2.spoofDetState);
           this->state.spoof=bitfield_flags2.spoofDetState;
         }
         else if(bitfield_flags2.spoofDetState<=1 && this->spoof_fix_counter>0)
         {
           this->spoof_fix_counter--;
         }
       }

       break;
      }
    }
  }
}

uint8_t CGPS_manager::gps_rx_handler(gps_rx_ctx_t* ctx)
{
  uint8_t is_new_poit_avl=0;

  int16_t recv_len;

#if defined(GNSS_FILTERS_DEBUG)
  recv_len=0;
  if(nmea_ref_pos<nmea_ref_size)
  {
    uint16_t copy_len=sizeof(ctx->rx_buff)-ctx->rx_pos;
    if(copy_len>nmea_ref_size-nmea_ref_pos) copy_len=nmea_ref_size-nmea_ref_pos;
    memcpy(ctx->rx_buff+ctx->rx_pos, &nmea_ref[nmea_ref_pos], copy_len);
    //__PRINTF_2("%.*s", copy_len, &nmea_ref[nmea_ref_pos]);
    nmea_ref_pos+=copy_len;
    recv_len=copy_len;
  }
#else
  recv_len=gps_uart_read(ctx->rx_buff+ctx->rx_pos, sizeof(ctx->rx_buff)-ctx->rx_pos);
#endif

  if(recv_len<0)
  {
    //ошибка при чтении
    const char* err_str;
    if(recv_len==-1) err_str="cbuff full";
    else err_str="ore";

    LOG("%s: gps_uart_read %s err...\n", this->name, err_str);
    //gps_uart_flush_rx();//уже была вызвана в gps_uart_read
    ctx->rx_pos=0;
  }
  else if(recv_len==0)
  {
    //ничего не считали
  }
  else
  {//recv_len>0
    uint16_t filled_len;

    uint16_t processed_len;
    uint16_t parse_len;

    filled_len=recv_len+ctx->rx_pos;

#if defined(GPS_USB_FORWARDING)
    USB_CDC_Send(ctx->rx_buff+ctx->rx_pos, recv_len, 20);
#endif //GPS_USB_FORWARDING

    //__CHAR_BUFF_PRINTF(ctx->rx_buff, ctx->rx_pos, "prev_chunk, prev_len=%hu:\n", ctx->rx_pos);
    //__CHAR_BUFF_PRINTF(ctx->rx_buff+ctx->rx_pos, recv_len, "new_chunk, new_len=%hu, filled=%hu:\n", recv_len, filled_len);

    //__CHAR_BUFF_PRINTF(ctx->rx_buff, filled_len, "update data, new data len=%hu:\n", filled_len);

#if defined(GPS_USB_FORWARDING)
    static uint8_t fwd_buff[256];
    USB_CDC_Send(ctx->rx_buff, filled_len, 200);

    int16_t fwd_len=USB_CDC_Receive(fwd_buff, sizeof(fwd_buff), 5);

    if(fwd_len>0)
    {
      gps_uart_write(fwd_buff, fwd_len);
    }
#endif //GPS_USB_FORWARDING

    const uint8_t* first_marker;
    const uint8_t* second_marker;

    processed_len=0;
    parse_len=0;

    uint8_t rx_sync_string_len=strlen(sync_rx_string);
    if(filled_len>=rx_sync_string_len)//поместится ли GPS_RX_SYNC_STRING_LEN
    {
      first_marker=strfind(ctx->rx_buff, (char*)sync_rx_string, filled_len);
      if(first_marker!=NULL)
      {
        if(filled_len-((first_marker+rx_sync_string_len)-ctx->rx_buff)>=rx_sync_string_len)//поместится ли GPS_RX_SYNC_STRING_LEN
        {
          second_marker=strfind((first_marker+rx_sync_string_len), (char*)sync_rx_string, filled_len-((first_marker+rx_sync_string_len)-ctx->rx_buff));
          if(second_marker!=NULL)
          {
            processed_len=second_marker-ctx->rx_buff;
            parse_len=second_marker-first_marker;

            const uint8_t* nmea_data;
            uint16_t nmea_data_len;
            const uint8_t* ubx_data;
            uint16_t ubx_data_len;

            nmea_data=first_marker;
            nmea_data_len=parse_len;

            ubx_data=strfind(nmea_data, "µb", nmea_data_len);
            if(ubx_data!=NULL)
            {
              ubx_data_len=parse_len-(ubx_data-nmea_data);
              nmea_data_len-=ubx_data_len;
            }
            else
            {
              ubx_data_len=0;
            }

#ifdef GPS_DEBUG_ON
            __CHAR_BUFF_PRINTF(nmea_data, nmea_data_len, "%s: nmea data (%hu bytes):\n", this->name, nmea_data_len);
            __BIN_BUFF_PRINTF(ubx_data, ubx_data_len, "%s: ubx data (%hu bytes, %hhu packets):\n", this->name, ubx_data_len, ubx_get_packets_count(ubx_data, ubx_data_len));
#endif
           //__PRINTF_2("%.*s", nmea_data_len, nmea_data);
#if defined(GNSS_FILTERS_DEBUG)
            //__PRINTF_2("%.*s", nmea_data_len, nmea_data);
#endif

            // Разбор данных NMEA
            uint8_t is_nmea_parse_ok;
            is_nmea_parse_ok=this->NMEA.parse(nmea_data, nmea_data_len);

            // Разбор данных UBX


            uint8_t ubx_packet_count;
            ubx_packet_count=ubx_get_packets_count(ubx_data, ubx_data_len);
            if(ubx_packet_count)
            {
              this->parse_ubx_data(ubx_data, ubx_data_len, ubx_packet_count);
            }

            if(is_nmea_parse_ok)
            {
              is_new_poit_avl=1;
            }
            else
            {
              // Не удалось распарсить пакет, то что-то не так, инкрементируем счетчик ошибок
              nmea_err_counter--;
              LOG("%s: NMEA.parse err...\n", this->name);
            }

            if(is_new_poit_avl)
            {
              // Трансляция данных в USB-порт (для поверки)
              //if(this->translate_nmea) USB_CDC_Send(nmea_data, nmea_data_len, 200);
            }
            else
            {
              gps_uart_flush_rx();
              ctx->rx_pos=0;
              return is_new_poit_avl;
            }
          }
        }
      }
    }
    //если пропарсили без ошибок (в нашем случае всегда)
    filled_len-=processed_len;
    //PRINTF("processed_len=%hu, new filled_len=%hu\n", processed_len, filled_len);

    if(filled_len>=sizeof(ctx->rx_buff))//буфер заполнен
    {
      LOG("%s: rx_buff ovflow...\n", this->name);
      gps_uart_flush_rx();
      ctx->rx_pos=0;
    }
    else if(processed_len>0 && filled_len>0)
    {
      //PRINTF("%s: mem_move_to_head %hu bytes in rx_buff\n", this->name, filled_len);
      //CHAR_BUFF_PRINTF(ctx->rx_buff+processed_len, filled_len, "%s: mem_move_to_head %hu bytes in rx_buff\n", this->name, filled_len);

      mem_move_to_head(ctx->rx_buff, ctx->rx_buff+processed_len, filled_len);
      ctx->rx_pos=filled_len;
    }
    else if(filled_len>0)
    {
      ctx->rx_pos=filled_len;
    }
    else
    {
      ctx->rx_pos=0;
    }
  }

  return is_new_poit_avl;
}

// Основной цикл
void CGPS_manager::Main_cycle(void)
{
  // Ожидание, пока разбудят
  while(1)
  {
    this->Wait_message(portMAX_DELAY);
    if(this->input_message.code == COMMAND_WAKEUP) break;
  }

  // Установка приоритета задаче
  vTaskPrioritySet(this->task_handle, tskIDLE_PRIORITY + this->priority);

  if(System.track_settings.time_step_with_ign_off==0xFFFF)
  {
    //Эта настройка не задана в системе, применяем старую настройку. Сделано для совместимости.
    System.track_settings.time_step_with_ign_off=System.track_settings.track_param.time_step;
  }

  if(psave_settings.use_move_det_opt || psave_settings.use_can_act || psave_settings.use_ign)
  {
    LOG("%s: psave option is on\n", this->name);
  }
  else
  {
    LOG("%s: psave option is off\n", this->name);
  }

  this->Init();
  reset_psave_state();

  this->gps_rx_ctx.rx_pos=0;
  gps_uart_flush_rx();

  for(;;)
  {
    // Получение и разбор команд от центральной задачи
    this->Receive_command(0);
    // Если в течение пяти секунд от приемника не было данных, то диагностику в ноль, рестарт приемника
    if( this->diag_timer.Is_empty() || nmea_err_counter == 0 \
      || (this->use_nav_systems_settings && this->CheckNavSystemsSettings(&System.track_settings.nav_systems, NULL) && memcmp(&this->nav_system_settings, &System.track_settings.nav_systems, sizeof(this->nav_system_settings)) != 0) \
        || (this->use_extended_filters_settings && this->CheckExtFiltersSettings(&System.track_settings.extended_filters) && memcmp(&this->extended_filters_settings, &System.track_settings.extended_filters, sizeof(this->extended_filters_settings)) != 0) )
    {
      if(this->diag_timer.Is_empty())
      {
        LOG("%s: No NMEA data. Reboot receiver\n", this->name);
        this->diag_timer.Start(SEC*5);
      }
      else if(nmea_err_counter == 0)
      {
        LOG("%s: Can`t parse NMEA. Reboot receiver\n", this->name);
        nmea_err_counter = NMEA_MAX_ERROR_COUNT;
      }
      else
      {
        LOG("%s: New settings. Reboot receiver\n", this->name);
      }

      this->state.receiver_is_ok = 0;
      this->Save_system_state();
      this->Reset_gnss_raw_state();
      // Рестарт приемника с выключением BACKUP
      this->Deinit();

      GPIO_WriteBit(GPS_BACKUP_GPIO, GPS_BACKUP_PIN, Bit_RESET);
      vTaskDelay(2000);

      this->Init();
      reset_psave_state();

      this->gps_rx_ctx.rx_pos=0;
      gps_uart_flush_rx();
    }

    if(psave_settings.use_move_det_opt)
    {
      if(System.signal_state.gsensor_move_sensor || System.signal_state.accel_no_init)  {psave.move_det=1;}
      else                                                                              {psave.move_det=0;}
    }

    if(psave_settings.use_can_act)
    {
      if(!System.can_state.sec_flags.can_in_sleep)                                      {psave.can_act=1; psave.can_inact_valid_timeS=GetSecondsFromStartup()+psave_settings.can_inact_valid_timeS;}
      else if(timeAfter(GetSecondsFromStartup(), psave.can_inact_valid_timeS))          {psave.can_act=0;}
    }

    if(psave_settings.use_ign)
    {
      if(System.signal_state.ignition)                                                  {psave.ign=1; psave.ign_off_valid_timeS=GetSecondsFromStartup()+psave_settings.ign_off_valid_timeS;}
      else if(timeAfter(GetSecondsFromStartup(), psave.ign_off_valid_timeS))            {psave.ign=0;}
    }

    if(!psave.in_psave)
    {
      //если включена опция использования psave и есть условия для пробуждения
      if((psave_settings.use_move_det_opt || psave_settings.use_can_act || psave_settings.use_ign) && \
          ((!psave_settings.use_move_det_opt || (psave_settings.use_move_det_opt && !psave.move_det)) && (!psave_settings.use_can_act || (psave_settings.use_can_act && !psave.can_act)) && (!psave_settings.use_ign || (psave_settings.use_ign && !psave.ign))) && \
            ((this->position_fixed && timeAfter(GetSecondsFromStartup(), psave.position_fixed_ignor_end_timeS)) || timeAfter(GetSecondsFromStartup(), psave.end_wait_fix_timeS)))
      {
        uint16_t next_wakeup_timeout;

        if(this->position_fixed)         next_wakeup_timeout=psave_settings.pos_fixed_sleep_timeS;       //позиция найдена
        else if(this->state.sat_inuse<3) next_wakeup_timeout=psave_settings.pos_not_fixed_1_sleep_timeS; //позиции нет, но очень мало используемых спутников
        else                             next_wakeup_timeout=psave_settings.pos_not_fixed_2_sleep_timeS; //позиции нет, но есть используемые спутники

        LOG("%s: enter psave (pfix: %s, s_use: %hhu), wakeup after %hu s\n", this->name, (this->position_fixed)?("true"):("false"), this->state.sat_inuse, next_wakeup_timeout);

        Deinit();
        psave.in_psave=1;
        psave.next_wakeup_timeS=GetSecondsFromStartup()+next_wakeup_timeout;
      }
    }
    else
    {
      const char* wup_event=NULL;

      if(!psave_settings.use_move_det_opt && !psave_settings.use_can_act && !psave_settings.use_ign) wup_event="psave disabled";
      else if(psave_settings.use_move_det_opt && psave.move_det) wup_event="move";
      else if(psave_settings.use_can_act && psave.can_act) wup_event="can";
      else if(psave_settings.use_ign && psave.ign) wup_event="ign";
      else if(timeAfter(GetSecondsFromStartup(), psave.next_wakeup_timeS)) wup_event="wup timer";

      if(wup_event!=NULL)
      {
        LOG("%s: out of psave by %s\n", this->name, wup_event);

        Init();
        psave.ignore_track_point_counter=psave_settings.ignore_track_point_counter;
        psave.valid_track_point_counter=psave_settings.valid_track_point_counter;
        psave.in_psave=0;
        psave.end_wait_fix_timeS=GetSecondsFromStartup()+psave_settings.end_wait_fix_timeS;
        psave.position_fixed_ignor_end_timeS=GetSecondsFromStartup()+psave_settings.position_fixed_ignor_timeS;
      }
    }

    if(psave.in_psave)
    {
      this->diag_timer.Start(SEC*5);
      nmea_err_counter = NMEA_MAX_ERROR_COUNT;

      if(psave.led_tim) psave.led_tim--;
      if(!psave.led_tim)
      {
        LedB_On();
        vTaskDelay(100);
        LedB_Off();
        psave.led_tim=20;
      }
      else
      {
        vTaskDelay(100);
      }
    }
    else
    {
#if defined(GPS_USB_FORWARDING)
      static uint8_t buff[256];
      int16_t r_len=USB_CDC_Receive(buff, sizeof(buff), 1);
      if(r_len>0)
      {
        gps_uart_write(buff, r_len);
      }
#endif //defined(GPS_USB_FORWARDING)

      //ждем GPS_RX_SYNC_STRING
      gps_uart_wait_sync(10);

      //если есть новая точка
      if(this->gps_rx_handler(&this->gps_rx_ctx))
      {
        if(false)
        {
          const uint32_t sat_time=conv_time_to_unixtime(&this->state.utc);
          const uint32_t rtc_time=GetUnixTime();
          const int32_t delta = sat_time-rtc_time;

          __PRINTF("%s: sync delta1 %li (%u)\n", this->name, delta, rtc_time);
        }

        // Рестарт диагностического таймера
        this->diag_timer.Start(SEC*5);

        nmea_err_counter = NMEA_MAX_ERROR_COUNT;
        this->state.receiver_is_ok = 1;

        // Если запущен счётчик пропуска точек
        if(psave.ignore_track_point_counter)
        {
          // Если движемся, то сбрасываем счётчик пропуска точек до 3х
          if( (System.signal_state.gsensor_move_sensor || System.signal_state.accel_no_init) \
            || System.can_state.numeric_data.can_speed > 0 )
          {
            if(psave.ignore_track_point_counter>3) psave.ignore_track_point_counter=3;
          }

          psave.ignore_track_point_counter--;

          if(this->state.coord_status==VALID)
          {
            if(psave.valid_track_point_counter) psave.valid_track_point_counter--;

            // Если подряд пришло psave_settings.valid_track_point_counter вылидных точек, то сбрасываем счётчик пропуска
            if(!psave.valid_track_point_counter) psave.ignore_track_point_counter=0;
          }
          else
          {
            psave.valid_track_point_counter=psave_settings.valid_track_point_counter;
          }
        }

        // Корекция времени
        if(this->state.coord_status==VALID)
        {
          if(false)
          {
            const uint32_t sat_time=conv_time_to_unixtime(&this->state.utc);
            const uint32_t rtc_time=GetUnixTime();
            const int32_t delta = sat_time-rtc_time;

            __PRINTF("%s: sync delta2 %li\n", this->name, delta);
          }

          SetUnixTime(conv_time_to_unixtime(&this->state.utc));
        }

        //
        this->Fill_gnss_raw_state();

        //psave.ignore_track_point_counter не дает нормально функционировать датчикам подмены и глушения!!!
        //при потере местополодения этот таймер взводится и не происходит вызова Save_system_state()!!!
        System.gnss_state.jammed=this->state.jammed;
        System.gnss_state.spoof=this->state.spoof;
        System.gnss_state.is_on=this->state.is_on;

        if(!psave.ignore_track_point_counter)
        {
          // Применение фильтров скачков курса и выбросов координат
          this->Apply_filters();
          // Сохранение текущего состояния в систему
          this->Save_system_state();
          // Генерация событий по обнаружению/потере местоположения
          this->Fix_position();
          // Индикация
          this->Indicate_state();
          // Если положение зафиксировано
          if(this->position_fixed)
          {
            // Фиксация остановки/движения
            this->Fix_stop_or_move();
            // Расчет пробега
            this->Count_odometer();
            // Обработка геозон
            Geozone_handler();
            // Регулярная трансляция трека
            this->Translate_track(REGULAR_TRACK_TRANSLATE);
          }
        }
      }
      else
      {

      }

      CheckNavigationFrequency();
    }
  }
}

// Функция трансляции трека
void CGPS_manager::Translate_track(uint8_t forced)
{
  TTrack_param track_param;
  uint8_t time_to_translate_track;

  System.Grab(portMAX_DELAY);
  if(System.signal_state.ignition)
  {
    memcpy(&track_param, &System.track_settings.track_param, sizeof(track_param));
  }
  else
  {
    memcpy(&track_param, &ign_off_track_param, sizeof(track_param));
    track_param.time_step=System.track_settings.time_step_with_ign_off;
  }
  System.Release();

  // Если параметр трансляции обнулен, не транслировать трек
  if(!translate_track && !forced) return;

  time_to_translate_track=forced;

  if(this->translate_timer) this->translate_timer--;

  // Трансляция в движении по истечению времени, если настройка установлена
  if(this->translate_timer == 0)
  {
    time_to_translate_track++;
    this->output_message.subcode = 0;
  }

  // Трансляция по изменению курса
  if(track_param.course_step && (abs((float)(this->state.direction - this->last_direction)) > track_param.course_step))
  {
    time_to_translate_track++;
    this->output_message.subcode = 1;
  }

  // Трансляция по пройденному расстоянию
  if(track_param.distance_step && (this->Get_distance(this->state.lat, this->state.lon, this->last_lat, this->last_lon) > track_param.distance_step))
  {
    time_to_translate_track++;
    this->output_message.subcode = 2;
  }

  // Если хоть одно из условий сработало, транслировать трек
  if(time_to_translate_track)
  {
    // Рестарт таймера
    this->translate_timer = track_param.time_step;
    this->last_direction = this->state.direction;
    this->last_lat = this->state.lat;
    this->last_lon = this->state.lon;
    // Отправка сообщения с данными очередной точки трека
    this->output_message.source = GPS_MANAGER;
    this->output_message.code = MESSAGE_GPS_TRACK_DATA;
    this->Send_message(this->output_queue);
  }
}

// Получение и разбор команд извне
void CGPS_manager::Receive_command(TickType_t xTicksToWait)
{
  // Проверяем наличие входящих сообщений в очереди
  if(this->Wait_message(xTicksToWait))
  {
    switch(this->input_message.code)
    {
      // Переход в спящий режим
      case COMMAND_SLEEP:
      {
        this->Deinit();
        this->state.is_on = 0;
        this->Save_system_state();
        this->Reset_gnss_raw_state();
        LedB_Off();
        while(1)
        {
          this->Wait_message(portMAX_DELAY);
          if(this->input_message.code == COMMAND_WAKEUP)
          {
            // Инициализация интерфейса
            this->Init();
            reset_psave_state();

            this->gps_rx_ctx.rx_pos=0;
            gps_uart_flush_rx();

            this->state.is_on = 1;
            this->Save_system_state();
            __PRINTF("%s: Wakeup\n", this->name)
            break;
          }
        }
        break;
      }
      case COMMAND_GNSS_HW_COLD_START:
      {
        LOG("%s: cold start cmd recv\n", this->name);

        this->Deinit();
        this->state.is_on = 0;
        this->Save_system_state();
        this->Reset_gnss_raw_state();
        LedB_Off();

        GPIO_WriteBit(GPS_BACKUP_GPIO, GPS_BACKUP_PIN, Bit_RESET);
        vTaskDelay(2000);

        this->Init();
        reset_psave_state();

        this->gps_rx_ctx.rx_pos=0;
        gps_uart_flush_rx();

        this->state.is_on = 1;
        this->Save_system_state();

        break;
      }
      case COMAND_UPDATE_GNSS_FREQ:
      {
        break;
      }
      // Пробуждение из спящего режима
      case COMMAND_WAKEUP:
      {
        LOG("%s: Recv unexpected wakeup command\n", this->name);
        break;
      }
      // Update NMEA frequency
      default:
      {
        __PRINTF("Gps unexpected command: %d\n", this->input_message.code);
        break;
      }
    }
  }

}

// Расчет расстояния между двумя точками
float CGPS_manager::Get_distance(float lat1, float lon1, float lat2, float lon2)
{
  // Удельное расстоние по меридиану (км/градус)
  float urm = 111.194926645;
  // Удельное расстояние по параллели (км/градус)
  float urp = cos( lat2*3.141592/180 )*urm;
  // Расстояние между точками по меридиану (первая сторона прямоугольника)
  float meridian_distance = abs(lat1 - lat2)*urm;
  // Расстояние между точками по параллели (вторая сторона прямоугольника)
  float parallel_distance =  abs(lon1 - lon2)*urp;
  // Расстояние меду точками, в километрах (диагоняль прямоугольника)
  float distance  = sqrt( (meridian_distance*meridian_distance) + (parallel_distance*parallel_distance) );
  // Перевод расстояния в метры
  distance = distance*1000;
  return distance;
}

// Фиксация остановки, либо движения
void CGPS_manager::Fix_stop_or_move(void)
{
  // Сброс счетчика после кратковременых остановок, не фиксируемых как стоянка
  if(this->state.speed > 4) this->stop_counter = 0;
  // Отслеживание остановки
  if(this->state.speed < 4 && this->state.in_move)
  {
    this->stop_counter++;
    if(this->stop_counter > System.sensor_settings.gps_movesensor.stopfix_time)
    {
      // Система перешла в состояние остановки
      this->state.in_move = 0;
      this->stop_counter=0;
      // Формирование и отправка сообщения о новом состоянии
      this->output_message.code = MESSAGE_GPS_STOP_FIXED;
      this->output_message.source = GPS_MANAGER;
      this->Send_message(this->output_queue);
      // Сброс одометра на стоянках
      if(System.track_settings.reset_odometer_on_stop)
      {
        System.Grab();
        System.gnss_state.odometer = 0;
        System.Release();
      }
    }
  }
  // Отслеживание начала движения
  if(!this->state.in_move && this->state.speed > 4)
  {
      // Система перешла в состояние движения
      this->state.in_move = 1;
      // Инкремент счетчика поездок
      if(System.signal_state.gsensor_move_sensor || System.signal_state.accel_no_init)
      {
        System.Grab();
        System.gnss_state.trip_counter++;
        System.Release();
        this->state.trip_counter = System.gnss_state.trip_counter;
        LOG("%s: Moving start fixed\n", this->name);
      }
      // Формирование и отправка сообщения о новом состоянии
      this->output_message.source = GPS_MANAGER;
      this->output_message.code = MESSAGE_GPS_MOVE_FIXED;
      this->Send_message(this->output_queue);
  }
}

// Индикация текущего состояния
void CGPS_manager::Indicate_state(void)
{
  if(this->state.fix_type >= FIX2D) LedB_On();
  else LedB_Toggle();
}

// Генерация сообщений о нахождении или потере местоположения
void CGPS_manager::Fix_position(void)
{
  if(this->position_fixed_filt_counter) this->position_fixed_filt_counter--;

  // Генерация сообщения о нахождении местоположения
  if(!this->position_fixed && this->state.coord_status == VALID)
  {
      this->last_odo_lat = this->state.lat;
      this->last_odo_lon = this->state.lon;

      this->last_lat =     this->state.lat;
      this->last_lon =     this->state.lon;

      this->filt_direction=this->state.direction;
      this->filt_lat =     this->state.lat;
      this->filt_lon =     this->state.lon;

      this->position_fixed = 1;
      this->position_fixed_filt_counter = position_fixed_filt_counter_setting;
      this->output_message.source = GPS_MANAGER;
      this->output_message.code = MESSAGE_GPS_POSITION_FIXED;
      this->Send_message(this->output_queue);
  }
  // Генерация сообщения о потере местоположения
  if(this->position_fixed && this->state.coord_status != VALID)
  {
    this->position_fixed = 0;
    this->output_message.source = GPS_MANAGER;
    this->output_message.code = MESSAGE_GPS_POSITION_LOST;
    this->Send_message(this->output_queue);
  }

  //если позиция не найдена, двигаем таймер
  if(!this->position_fixed)                                psave.position_fixed_ignor_end_timeS=GetSecondsFromStartup()+psave_settings.position_fixed_ignor_timeS;
  //если позиция найдена, но используем мало спутников, двигаем таймер
  if(this->position_fixed && this->state.sat_inuse<8)      psave.position_fixed_ignor_end_timeS=GetSecondsFromStartup()+psave_settings.position_fixed_ignor_timeS;
}

//
void CGPS_manager::Reset_gnss_raw_state(bool is_atomic)
{
  if(is_atomic) System.Grab();
  memset(&System.gnss_raw_state, 0x00, sizeof(System.gnss_raw_state));
  if(is_atomic) System.Release();
}
#define INCREASE_ACCURACY_10MLTP 10.0f
#define INCREASE_ACCURACY_100MLTP 100.0f

//
void CGPS_manager::Fill_gnss_raw_state(bool is_atomic)
{
  if(is_atomic) System.Grab();
  System.gnss_raw_state.unix_time=conv_time_to_unixtime(&this->state.utc);
  System.gnss_raw_state.coord_status=this->state.coord_status;
  System.gnss_raw_state.lat=this->state.lat;
  System.gnss_raw_state.lon=this->state.lon;
  System.gnss_raw_state.alt=(int16_t)(INCREASE_ACCURACY_10MLTP*this->state.alt);
  System.gnss_raw_state.diff=(uint32_t)this->state.diff;
  /*0....6553.5 km/h*/
  System.gnss_raw_state.speed=(uint16_t)(INCREASE_ACCURACY_10MLTP*this->state.speed);//
  System.gnss_raw_state.direction=(uint16_t)(INCREASE_ACCURACY_10MLTP*this->state.direction);
  System.gnss_raw_state.dgps_mode=this->state.dgps_mode;
  System.gnss_raw_state.dgps_age=this->state.dgps_age;
  System.gnss_raw_state.dgps_sid=this->state.dgps_sid;
  System.gnss_raw_state.sig=(uint8_t)this->state.sig;
  System.gnss_raw_state.fix_type=this->state.fix_type;
  System.gnss_raw_state.sat_inuse=this->state.sat_inuse;
  System.gnss_raw_state.PDOP=(uint16_t)(INCREASE_ACCURACY_100MLTP*this->state.PDOP);
  System.gnss_raw_state.HDOP=(uint16_t)(INCREASE_ACCURACY_100MLTP*this->state.HDOP);
  System.gnss_raw_state.VDOP=(uint16_t)(INCREASE_ACCURACY_100MLTP*this->state.VDOP);
  if(is_atomic) System.Release();
}

// Применение фильтров скачков курса и выбросов координат
void CGPS_manager::Apply_filters(void)
{
  // Применение фильтра по HDOP
  if(System.track_settings.filter_track_hdop && this->state.HDOP > System.track_settings.filter_track_hdop)
  {
    this->state.fix_type = INACTIVE;
    this->state.coord_status = INVALID;
  }

  // Фильтрация от скачков курса
  if(this->state.speed < 4)
  {
    this->state.direction = this->last_direction;
  }

  // Фильтрация трека на стоянках и остановках
  if(System.track_settings.filter_track_on_stop)
  {
    if(this->state.coord_status == VALID && this->position_fixed && !this->position_fixed_filt_counter \
      && (System.signal_state.gsensor_move_sensor == 0 && System.signal_state.accel_no_init == 0) \
        && System.can_state.numeric_data.can_speed == 0)
    {
      uint16_t distance;

      if(this->state.HDOP < 2.1f && this->state.sat_inuse > 6)
      {
        distance=(uint16_t)this->Get_distance(this->state.lat, this->state.lon, this->filt_lat, this->filt_lon);
      }
      else
      {
        distance=0;
      }

      if(distance <= max_distance_filt_on_filter_track_on_stop)
      {
        this->state.direction=this->filt_direction;
        this->state.lat = this->filt_lat;
        this->state.lon = this->filt_lon;
      }
    }
  }

  if(this->state.coord_status == VALID && this->position_fixed)
  {
    this->filt_direction=this->state.direction;
    this->filt_lat=this->state.lat;
    this->filt_lon=this->state.lon;
  }

  // Подстановка последней известной координаты, сли не решена навигационная задача и включена соответствующая настройка
  /*
  if(System.track_settings.always_locate == 1 && this->last_odo_lat != 0 && this->state.coord_status == INVALID)
  {
    this->state.lat = this->last_odo_lat;
    this->state.lon = this->last_odo_lon;
    this->state.coord_status = VALID;
  }
  */
  if(System.track_settings.always_locate == 1 && this->last_lat != 0 && this->state.coord_status == INVALID)
  {
    this->state.direction = this->last_direction;
    this->state.lat = this->last_lat;
    this->state.lon = this->last_lon;
    this->state.coord_status = VALID;
  }
}

// Сохранение текущего состояния в систему
void CGPS_manager::Save_system_state(void)
{
  System.Grab();
  // Загрузка одометра и счетчика поездок
  this->state.trip_counter = System.gnss_state.trip_counter;
  this->state.odometer = System.gnss_state.odometer;
  // Сохранение состояния
  memcpy(&System.gnss_state, &this->state, sizeof(System.gnss_state));
  System.Release();
}

// Расчет пробега
void CGPS_manager::Count_odometer(void)
{
  if(this->state.in_move && (System.signal_state.gsensor_move_sensor || System.signal_state.accel_no_init))
  {
    if(System.track_settings.odometer_by_ign == 0 || (System.track_settings.odometer_by_ign && System.signal_state.ignition))
    {
      System.Grab();
      System.gnss_state.odometer += (Get_distance(this->state.lat, this->state.lon, this->last_odo_lat, this->last_odo_lon)/1000);
      System.Release();
      this->state.odometer = System.gnss_state.odometer;
    }
    this->last_odo_lat = this->state.lat;
    this->last_odo_lon = this->state.lon;
  }
}

// Обработчик геозон
void CGPS_manager::Geozone_handler(void)
{
  for(uint16_t i = 0; i < MAX_GEOZONES; i++)
  {
    if(!System.geozone_settings.geozone[i].radius) continue;
    float real_distance = this->Get_distance(this->state.lat, this->state.lon, System.geozone_settings.geozone[i].lat, System.geozone_settings.geozone[i].lon);
    if(real_distance < System.geozone_settings.geozone[i].radius) this->state.geozones[i] = 1;
    else this->state.geozones[i] = 0;
  }
}

/* C code---------------------------------------------------------------------*/
// Функция запуска GPS менеджера
void Start_gps_manager(void* argument)
{
#if defined(GPS_USB_FORWARDING)
  USB_CDC_Init();
#endif //GPS_USB_FORWARDING
  GPS_manager.Main_cycle();
  while(1);
}

void CGPS_manager::SetNavigationFrequency(uint16_t frequency)
{
  auto packet = reinterpret_cast<T_ubx_msg*>(&this->gps_rx_ctx.rx_buff[0]);
  auto ratesPayload = reinterpret_cast<TUbxFrequencyData*>(&this->gps_rx_ctx.rx_buff[UBX_HEADER_SIZE]);

  packet->header = UBX_HEADER;
  packet->clas = UBX_MSG_CFG;
  packet->message_id = UBX_MSG_CFG_FREQ;
  packet->len = sizeof(TUbxFrequencyData);

  if (frequency < MinNavigationFrequency) {
    frequency = MinNavigationFrequency;
  }

  if (frequency > MaxNavigationFrequency) {
    frequency = MaxNavigationFrequency;
  }

  ratesPayload->MeasurementRate = frequency;
  ratesPayload->NavigationRate = DefaultNavigationRate;
  ratesPayload->TimeRef = DefaultNavigationTimeRef;

  ubx_add_crc(packet, this->gps_rx_ctx.rx_buff);

  gps_uart_write(this->gps_rx_ctx.rx_buff, UBX_HEADER_SIZE + packet->len + UBX_CRC_SIZE);
}

void CGPS_manager::CheckNavigationFrequency()
{
  if (!psave.in_psave && System.gnss_state.is_on) {
    if (psave.ign && !isSetFastGpsMode) {
      LOG("Set fast mode gps\n");
      SetNavigationFrequency(FastNavigationFrequency);
      isSetFastGpsMode = true;
    } else if (!psave.ign && isSetFastGpsMode) {
      LOG("Set slow mode gps\n");
      SetNavigationFrequency(DefaultNavigationFrequency);
      isSetFastGpsMode = false;
    }
  }
}
