/**
  ******************************************************************************
  * File Name          : gps_manager.h
  * Description        : Менеджер GPS
  *
  *
  ******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef GPS_MANAGER_H
#define GPS_MANAGER_H

/* Includes ------------------------------------------------------------------*/
#include <stdint.h>
#include "manager.h"
#include "gps_manager_config.h"
#include "FreeRTOS.h"
#include "task.h"
#include "Soft_timers\Soft_timers.h"
#include "System\system.h"
#include "NMEA_protocol\nmea_protocol.h"

#include <time/time.h>
#include <time/timer.h>

/* Defines- ------------------------------------------------------------------*/
//#define GPS_USB_FORWARDING
//#define GPS_DEBUG_ON

// Системные настройки менеджера
#define GPS_MANAGER_NAME              "gps_man" // Имя менеджера
#define GPS_MANAGER_INPUT_CUEUE_SIZE  3         // Количество элементов во входящей очереди сообщений
#define GPS_MANAGER_PRIORITY          1         // Приоритет задачи менеджера
#define GPS_MANAGER_STACK_SIZE        256       // Размер стэка менеджера в словах
#define NMEA_MAX_ERROR_COUNT          5         // Максимальное число ошибок NMEA, после которого произойдет рестарт приемника

// Параметры конфигурации по умолчанию
#define GPS_DEFAULT_BASE_VERSION              1
#define GPS_DEFAULT_DISTANCE_STEP             0   //Метров
#define GPS_DEFAULT_COURSE_STEP               0   //Градусов
#define GPS_DEFAULT_TIME_ONMOVE_STEP          1   //Секунд
#define GPS_DEFAULT_TIME_ONSTOP_STEP          2   //Секунд
#define GPS_DEFAULT_STOP_FIX_TIME             60  //Секунд
#define GPS_DEFAULT_LATLON_FILTER             50  //Метров
#define GPS_DEFAULT_SPEED_FILTER              4   //Км/ч
#define GPS_DEFAULT_CANT_FIX_TIMEOUT          120 //Секунд
#define GPS_DEFAULT_NOSIGNAL_TIMEOUT          240 //Секунд

// Частота выдачи сообщений приемником, сек.
#define GPS_UBX_MESSAGE_RATE              1

// Время, по истечении которого генерируется событие о длительном отсутствии сигналов спутников, в секундах
#define NO_SIGNAL_TIMEOUT    240
#define CANT_FIX_TIMEOUT     120

// Типы трансляции трека, регулярный или принудительный
#define REGULAR_TRACK_TRANSLATE           0
#define FORCED_TRACK_TRANSLATE            1

typedef struct
{
  uint8_t  rx_buff[2048];
  uint16_t rx_pos;
}gps_rx_ctx_t;

typedef struct
{
  uint8_t use_move_det_opt:1;      //использовать датчик движения для перехода/выхода из энергосберегающего режима
  uint8_t use_can_act:1;           //использовать активность can для перехода/выхода из энергосберегающего режима
  uint8_t use_ign:1;               //использовать ign для перехода/выхода из энергосберегающего режима
  uint8_t ignore_track_point_counter;
  uint8_t valid_track_point_counter;
  uint16_t position_fixed_ignor_timeS;
  uint16_t firts_position_fixed_ignor_timeS;
  uint16_t end_wait_fix_timeS;
  uint16_t first_end_wait_fix_timeS;
  uint16_t can_inact_valid_timeS;
  uint16_t ign_off_valid_timeS;

  uint16_t pos_fixed_sleep_timeS;       //время нахождения в psave в случае если позиция найдена
  uint16_t pos_not_fixed_1_sleep_timeS; //время нахождения в psave в случае если позиции нет, но очень мало используемых спутников
  uint16_t pos_not_fixed_2_sleep_timeS; //время нахождения в psave в случае если позиции нет, но есть используемые спутники
}psave_settings_t;

typedef struct
{
  uint8_t in_psave:1;                      //признак нахождения в энергосберегающем режиме

  uint8_t move_det:1;                      //признак движения
  uint8_t can_act:1;                       //признак активности can
  uint8_t ign:1;                           //признак зажигания

  uint8_t led_tim;                         //счетчик индикации
  uint8_t ignore_track_point_counter;      //счетчик пропуска точек трека после включения приёмника
  uint8_t valid_track_point_counter;       //счетчиик валидных координат

  uint32_t ign_off_valid_timeS;            //по истечении этого времени, счиаем что ign отключено
  uint32_t can_inact_valid_timeS;          //по истечении этого времени, счиаем что can уснул

  uint32_t end_wait_fix_timeS;             //по истечению этого времени, перестаем ждать захвата позиции и если есть условия для перехода в psave, уходим в него
  uint32_t next_wakeup_timeS;              //если находимся в psave, это время указываем когда нам проснуться
  uint32_t position_fixed_ignor_end_timeS; //после включения/пробуждения, только по истечению этого времени используем признак fixed в psave
}psave_ctx_t;

// Frequency data
//__attribute__((packed))
__packed struct TUbxFrequencyData
{
  uint16_t MeasurementRate;               // The elapsed time between GNSS measurements
  uint16_t NavigationRate;                // The ratio between the number of measurements and the number of navigation solutions
  uint16_t TimeRef;                       // The time system to which measurements are aligned (0. UTC, 1. GPS, 2. GLONASS, 3. BeiDou, 4. Galileo)
};

/* C++ code-------------------------------------------------------------------*/
class CGPS_manager: public CManager
{
  public:

  // Конструктор, определяется имя менеджера, передается коллбэк на таймер, указатель на исходящую очередь, интерфейс обмена данными с железом
  CGPS_manager(const char* mngr_name, QueueHandle_t* output_queue, uint8_t input_queue_size, uint8_t priority, TaskFunction_t task_starter, uint16_t stack_size);
  // Инициализация менеджера, реализует наследник
  void Init();
  // Деинициализация менеджера, реализует наследник
  void Deinit(void);
  // Главный цикл менеджера
  void Main_cycle(void);

  void SetDefaultExtFiltersSettings(TNav_extended_filters* settings);
  void SetDefaultNavSystemSettings(TNav_system_settings* settings, const char** rmc);
  bool CheckExtFiltersSettings(TNav_extended_filters* settings);
  bool CheckNavSystemsSettings(TNav_system_settings* settings, const char** rmc);

  private:

  static const bool use_extended_filters_settings=true;
  static const bool use_nav_systems_settings=true;

  // Имя менеджера
  const char* name;
  // Текущее состояние GPS-приемника
  TGNSS_state state;
  CNMEA_protocol NMEA;

  CSoft_timer diag_timer;
  uint16_t translate_timer;
  TNav_system_settings nav_system_settings;
  TNav_extended_filters extended_filters_settings;
  uint8_t translate_track;
  uint8_t position_fixed;
  // Этот таймер загружается по position_fixed. По концу его счета начинает работать фильтр на остановках.
  uint8_t position_fixed_filt_counter;
  uint8_t jammed_fix_counter;
  uint8_t spoof_fix_counter;
  uint16_t stop_counter;
  uint32_t trip_counter;
  // Последний запомненный курс, используется для генерации трека по изменении курса
  float last_direction;
  // Последние запомненные координаты, используюся для генерации трека по пройденному расстоянию
  float last_lat;
  float last_lon;
  //
  float filt_direction;
  float filt_lat;
  float filt_lon;
  // Последние запомненные координаты, используюся для подсчета пробега
  float last_odo_lat;
  float last_odo_lon;
  // Счетчик регулярного сохранения показаний одометра
  uint16_t odo_save_counter;
  // Счетчик ошибок NMEA
  uint8_t nmea_err_counter;
  // Контекст парсера разделяющего ubx и NMEA
  gps_rx_ctx_t gps_rx_ctx;
  // Строка синхронизации
  const char* sync_rx_string;
  // Трансляция трека
  void Translate_track(uint8_t forced);
  // Получение и разбор команд извне
  void Receive_command(TickType_t xTicksToWait);
  // Фиксация остановки, либо движения
  void Fix_stop_or_move(void);
  // Индикация текущего состояния
  void Indicate_state(void);
  // Генерация сообщений по обнаружению/потере местоположения
  void Fix_position(void);
  // Применение фильтров скачков курса и выбросов координат
  void Apply_filters(void);

  // Сохранение текущего состояния в систему
  void Save_system_state(void);
  // Функция расчета дистанции между двумя точками
  float Get_distance(float lat1, float lon1, float lat2, float lon2);
  // Разбор данных UBX протокола
  void parse_ubx_data(const uint8_t* buf, uint16_t bufsize, uint8_t upx_packet_count);
  // Расчет пробега
  void Count_odometer(void);
  // Обработка геозон
  void Geozone_handler(void);
  // Считать версию ПО приемника
  void parse_receiver_version(void);
  // Обработчик регулярного перезапуска
  void Regular_restart(void);
  // Парсер разделяющий ubx и NMEA
  uint8_t gps_rx_handler(gps_rx_ctx_t* ctx);
  //
  void Reset_gnss_raw_state(bool is_atomic = true);
  //
  void Fill_gnss_raw_state(bool is_atomic = true);

  //
  void reset_psave_state(void);
  static const uint8_t position_fixed_filt_counter_setting=10;
  static const uint16_t max_distance_filt_on_filter_track_on_stop=200;
  static const TTrack_param ign_off_track_param;
  static const psave_settings_t psave_settings;
  psave_ctx_t psave;

  // Additional parameters
  bool isSetFastGpsMode;

  /// Navigation frequency implementation
private:
  // default navigation frequency = 1 Hz
  static const uint16_t DefaultNavigationFrequency = 1000;
  static const uint16_t FastNavigationFrequency = 100;
  // fast navigation frequency = 10 Hz
  static const uint16_t DefaultNavigationRate = 1;
  static const uint16_t DefaultNavigationTimeRef = 1;
  static const uint16_t MinNavigationFrequency = 25;
  static const uint16_t MaxNavigationFrequency = DefaultNavigationFrequency;

  void SetNavigationFrequency(uint16_t frequency);
  void CheckNavigationFrequency();
};

/* C code---------------------------------------------------------------------*/
// Функция запуска GPS менеджера
extern void Start_gps_manager(void* argument);
// Для обеспечения видимости другими менеджерами
extern CGPS_manager GPS_manager;
#endif