/**
******************************************************************************
* File Name          : signal_manager.cpp
* Description        : Менеджер входных и выходных сигналов
*
*
******************************************************************************
*/

/*
В этой версии signal_manager, по 232 интерфесу работает только камера, в отдельном потоке.
По 485 работают только ДУТ'ы.
*/

/* Includes ------------------------------------------------------------------*/
#include "signal_manager/signal_manager.h"
#include "Soft_timers/Soft_timers.h"
#include "syscom_manager.h"
#include "server_manager/server_manager.h"
#include "subtask_utils_lib/subtask_manage.h"
#include <string.h>

#include "debug_port.h"
#include "rtc_driver.h"
#include "leds_driver.h"
#include "add_tick_timer.h"

#include "usbd_cdc_config.h"

#if defined(RS232_PRESENT)
#include "rs232_driver.h"
#endif //RS232_PRESENT

#if defined(RS485_PRESENT)
#include "rs485_driver.h"
#endif //RS485_PRESENT

#if defined(RS_UART_PRESENT)
#include "rsuart_driver.h"
#endif //RS_UART_PRESENT

#if defined(BR_PRESENT)
#include "br_protocol_lib/br_protocol.h"
#define BR_RX_BUFF_SIZE         (201)
#define BR_TX_BUFF_SIZE         (36)
#else
#define BR_RX_BUFF_SIZE         (0)
#define BR_TX_BUFF_SIZE         (0)
#endif //BR_PRESENT

static uint8_t common_rx_buff[BR_RX_BUFF_SIZE];
static uint8_t common_tx_buff[BR_TX_BUFF_SIZE];

#if defined(DUTS_PRESENT)
//
#endif //DUTS_PRESENT

#if defined(DIGITAL_OUTPUTS_PRESENT)
#include "outputs_driver.h"
#endif //DIGITAL_OUTPUTS_PRESENT

#if defined(CANPRO_PRESENT)
//#include "canpro/canpro.h"
#endif //CANPRO_PRESENT

#if defined(MFI_PRESENT)
#include "mfi_driver.h"
#else
#error
#endif //MFI_PRESENT

#if defined(GSENSOR_PRESENT)
#include "Accel/Accel_sensors.h"
#include "nvic_exti_driver.h"
#endif //GSENSOR_PRESENT

#if defined(TAMPERS_PRESENT)
#if MAX_TAMPERS != 2
#error этот менеджер умеет работать только с двумя тамперами
#endif
#endif //TAMPERS_PRESENT

#if defined(SAVER2021_PRESENT)
#include "saver_lib/saver2021_lib.h"
#endif //defined(SAVER2021_PRESENT)

#if defined(VEGA_CAN_PRESENT)
#include "signal_manager/signalman_can_subtask.h"
#undef CANPRO_PRESENT
__INIT_CONST_SUBTASK_CTX_FIELDS(can_subtask_ctx, "can_stask", configMINIMAL_STACK_SIZE*2, SIGNAL_MANAGER_PRIORITY, true, can_subtask_onetime_init, can_subtask_init, can_subtask_main_cycle, can_subtask_deinit, can_subtask_is_sleep_allowed);
#endif //VEGA_CAN_PRESENT

#if defined(INTERNAL_NRF_PRESENT)
#include "signal_manager/signalman_nrf_subtask.h"
__INIT_CONST_SUBTASK_CTX_FIELDS(nrf_subtask_ctx, "nrf_stask", configMINIMAL_STACK_SIZE+48, SIGNAL_MANAGER_PRIORITY, true, nrf_subtask_onetime_init, nrf_subtask_init, nrf_subtask_main_cycle, nrf_subtask_deinit, NULL);
#endif //INTERNAL_NRF_PRESENT

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
#include "signal_manager/signalman_ble_subtask.h"
__INIT_CONST_SUBTASK_CTX_FIELDS(ble_subtask_ctx, "ble_stask", configMINIMAL_STACK_SIZE*2, SIGNAL_MANAGER_PRIORITY, true, external_ble_subtask_onetime_init, external_ble_subtask_init, external_ble_subtask_main_cycle, external_ble_subtask_deinit, NULL);
#endif //EXTERNAL_BLE_BOARD_PRESENT

#if defined(JQ6500_AUDUO_PRESENT)
#include "signal_manager/audio_manadge.h"
#endif //defined(JQ6500_AUDUO_PRESENT)

/* Defines -------------------------------------------------------------------*/

#define DEBUG_L_O           0//off debug
#define DEBUG_L_L           1//low level (err only)
#define DEBUG_L_M           2//medium level
#define DEBUG_L_H           3//high level
#define SIGNAL_MNG_DEBUG_LEVEL	DEBUG_L_M


/* Global variables-----------------------------------------------------------*/
//- GPS менеджер
CSignal_manager Signal_manager(SIGNAL_MANAGER_NAME, &Syscom_manager.input_queue, SIGNAL_MANAGER_INPUT_QUEUE_SIZE, SIGNAL_MANAGER_PRIORITY, Start_signal_manager, SIGNAL_MANAGER_STACK_SIZE);

static volatile struct adc_vals_struct
{
  uint16_t              adc_vref;
  uint16_t	        adc_inttemp;
  uint16_t	        adc_onboard;

#if defined(INTERNAL_AKB_PRESENT)
  uint16_t	        adc_lipo;
#endif //INTERNAL_AKB_PRESENT

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
  uint16_t adc_mfi_vals[MFI_INPUT_CNT];
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3
}adc_vals;

/* Private variables-----------------------------------------------------------*/
static inline void _delay_loops(uint32_t loops)
{
  asm volatile (
                "1: SUBS %[loops], %[loops], #1 \n"
                  "   BNE 1b \n"
                    : [loops] "+r"(loops)
                      );
}
#define __DELAY_MS(MS, F_CPU) _delay_loops((uint32_t)((double)MS * F_CPU / 3000.0))

#pragma diag_suppress=Pe177
static void lock_state_and_settings(uint8_t is_lock)
#pragma diag_default=Pe177
{
  if(is_lock) System.Grab(portMAX_DELAY);
  else        System.Release();
}

/* C++ code-------------------------------------------------------------------*/
//- Конструктор Signal-менеджера
CSignal_manager::CSignal_manager(const char* mngr_name, QueueHandle_t* output_queue, uint8_t input_queue_size, uint8_t priority, TaskFunction_t task_starter, uint16_t stack_size):
  CManager(mngr_name, output_queue, input_queue_size, priority, task_starter, stack_size)
  {
    this->name = mngr_name;
    this->EXTI_NVIC_Configuration();

    memset((void*)&adc_vals, 0x00, sizeof(adc_vals));
    System.signal_state.power_source=UNKNOWN_POWER;
    System.signal_state.is_on=0;

#if defined(GSENSOR_PRESENT)
    System.signal_state.accel_no_init=1;
#endif

#if defined(TAMPERS_PRESENT)
    for(uint8_t i=0; i<MAX_TAMPERS; i++)
    {
      System.signal_state.tamper[i]=0;
    }
#endif //TAMPERS_PRESENT

#if defined(BR_PRESENT)
    System.signal_state.is_br_ok=2;
#endif //BR_PRESENT

#if defined(VEGA_CAN_PRESENT)
    Signal_manager.can_commands_queue=xQueueCreate(SIGNAL_MANAGER_CAN_CMD_QUEUE_SIZE, sizeof(this->input_message));
#endif //VEGA_CAN_PRESENT
  }

  //- Инициализация менеджера
  void CSignal_manager::Init (void)
  {
    // Установка приоритета задаче
    vTaskPrioritySet(this->task_handle, tskIDLE_PRIORITY + this->priority);

    Leds_Init();

    // Ожидание, пока разбудят
    while(1)
    {
#if IWDG_ENABLE > 0
      IWDG_ReloadCounter();
      if(this->Wait_message(200))
      {
        if(this->input_message.code == COMMAND_WAKEUP) break;
      }
#else
      if(this->Wait_message(portMAX_DELAY))
      {
        if(this->input_message.code == COMMAND_WAKEUP) break;
      }
#endif
    }

    /* Check if the system has resumed from IWDG reset */
#if IWDG_ENABLE > 0
    if (RCC_GetFlagStatus(RCC_FLAG_IWDGRST) != RESET)
    {
      uint16_t reset_count=RTC_ReadBackupRegister(RTC_BKP_DR5);
      reset_count++;
      //если мы сбрасывались по iwdg 8 раз подряд, вероятно из-за ЧЯ
      if(reset_count>=8)
      {
        LOG("!!!!!%s: system started after IWDG reset, bbox clean!!!!!\n", this->name);
        Black_box.Grab();
        Black_box.Clean();
        RCC_ClearFlag();//Clear reset flags
        RTC_WriteBackupRegister(RTC_BKP_DR5, 0);
        NVIC_SystemReset();
      }
      else
      {
        RTC_WriteBackupRegister(RTC_BKP_DR5, reset_count);
        RCC_ClearFlag();//Clear reset flags
        LOG("!!!!!%s: system started after IWDG reset, reset_count=%hu!!!!!\n", this->name, reset_count);
      }
    }
    else
    {
      //сбрасываем
      RTC_WriteBackupRegister(RTC_BKP_DR5, 0);
    }
#else
    LOG("%s: warning, IWDG disabled\n", this->name);
    RTC_WriteBackupRegister(RTC_BKP_DR5, 0);
#endif

    LOG("%s: LPWRRST: %u, SFTRST: %u, PORRST: %u, PINRST: %u, BORRST: %u\n", this->name,
        RCC_GetFlagStatus(RCC_FLAG_LPWRRST), RCC_GetFlagStatus(RCC_FLAG_SFTRST), RCC_GetFlagStatus(RCC_FLAG_PORRST), RCC_GetFlagStatus(RCC_FLAG_PINRST), RCC_GetFlagStatus(RCC_FLAG_BORRST));
    // Clear reset flags
    RCC_ClearFlag();

    if (RTC_ReadBackupRegister(RTC_BKP_DR2) == 0xA5A5)
    {
      LOG("!!!!!%s: system started after %s reset!!!!!\n", this->name, "PVD");
      /* Clear flag */
      RTC_WriteBackupRegister(RTC_BKP_DR2, 0x0000);
    }

    if (RTC_ReadBackupRegister(RTC_BKP_DR4) == 0xA5A5)
    {
      LOG("!!!!!%s: system started after mem dma err reset!!!!!\n", this->name);
      /* Clear flag */
      RTC_WriteBackupRegister(RTC_BKP_DR4, 0x0000);
    }
    else if (RTC_ReadBackupRegister(RTC_BKP_DR4) == 0xA4A4)
    {
      LOG("!!!!!%s: system started after mem dma err reset in bootloader!!!!!\n", this->name);
      /* Clear flag */
      RTC_WriteBackupRegister(RTC_BKP_DR4, 0x0000);
    }

    if (RTC_ReadBackupRegister(RTC_BKP_DR7) == 0x01)
    {
      LOG("!!!!!%s: tofactory by syscom_man!!!!!\n", this->name);
      /* Clear flag */
      RTC_WriteBackupRegister(RTC_BKP_DR7, 0x0000);
    }
    else if (RTC_ReadBackupRegister(RTC_BKP_DR7) == 0x02)
    {
      LOG("!!!!!%s: tofactory by server_man!!!!!\n", this->name);
      /* Clear flag */
      RTC_WriteBackupRegister(RTC_BKP_DR7, 0x0000);
    }

    System.signal_state.is_on=1;
    if(SIGNAL_MNG_DEBUG_LEVEL>=DEBUG_L_L) __PRINTF("%s: manager initialized\n", this->name);
  }

  //- Деинициализация менеджера
  void CSignal_manager::Deinit (void)
  {

  }


#if defined(PROD_TESTING_PRESENT)
#undef PROD_TESTING_PRESENT
  uint8_t start_prod_tests_cmd=0;
#endif //PROD_TESTING_PRESENT

#if (defined(RS485_PRESENT) || defined(RS232_PRESENT) || defined(RS_UART_PRESENT))

#if defined(RS232_PRESENT)
  static volatile uint8_t rs232_mutex=0;
#endif //RS232_PRESENT

#if defined(RS485_PRESENT)
  static volatile uint8_t rs485_mutex=0;
#endif //RS485_PRESENT

#if defined(RS_UART_PRESENT)
  static volatile uint8_t rsuart_mutex=0;
#endif //RS_UART_PRESENT

  #pragma inline=forced_no_body
  static uint8_t interface_mutex_take(volatile uint8_t* const mutex, bool is_wait_taking)
  {
    return bad_mutex_take(mutex, is_wait_taking);
  }

  #pragma inline=forced_no_body
  static void interface_mutex_give(volatile uint8_t* const mutex)
  {
    bad_mutex_give(mutex);
  }

#if defined(RS232_PRESENT)
  void RS232_Grab(void)
  {
    interface_mutex_take(&rs232_mutex, 1);
  }

  void RS232_Release(void)
  {
    interface_mutex_give(&rs232_mutex);
  }
#endif //RS232_PRESENT

#if defined(RS485_PRESENT)
  void RS485_Grab(void)
  {
    interface_mutex_take(&rs485_mutex, 1);
  }

  bool RS485_Trying_Grab(void)
  {
    return interface_mutex_take(&rs485_mutex, 0);
  }

  void RS485_Release(void)
  {
    interface_mutex_give(&rs485_mutex);
  }
#endif //RS232_PRESENT

#if defined(RS_UART_PRESENT)
  void RSUART_Grab(void)
  {
    interface_mutex_take(&rsuart_mutex, 1);
  }

  bool RSUART_Trying_Grab(void)
  {
    return interface_mutex_take(&rsuart_mutex, 0);
  }

  void RSUART_Release(void)
  {
    interface_mutex_give(&rsuart_mutex);
  }
#endif //RS_UART_PRESENT

#if defined(PROD_TESTING_PRESENT)
  static uint8_t start_prod_tests_cmd=0;
#endif //PROD_TESTING_PRESENT

#endif //(defined(RS485_PRESENT) || defined(RS232_PRESENT) || defined(RS_UART_PRESENT))

#if defined(DUTS_PRESENT)
  static uint8_t Dut_Idx = MAX_DUTS; // MAX_DUTS - Макс. количество дутов в конфигурации

  //- DUT crc
  static uint8_t DUT_crc(uint8_t *sbuf, uint8_t len)
  {
#if defined(CANPRO_PRESENT)
    return canpro_crc8_fast(sbuf, len);
#else
    uint8_t crc=0;
    for(;len>0;len--,sbuf++)
    {
      uint8_t i=8, b=*sbuf;
      do {
        if ( (b ^ crc) & 0x01) {
          crc = ( (crc ^ 0x18) >> 1 ) | 0x80;
        } else {
          crc >>= 1;
        }
        b >>= 1;
      } while (--i);
    }
    return crc;
#endif
  }

  //- DUT Опрос
  static void DUT_Polling(uint8_t* is_interface_busy)
  {
    // Проверяем текущий ДУТ
    if(++Dut_Idx >= MAX_DUTS) Dut_Idx=0;

    if(is_interface_busy!=NULL) *is_interface_busy=0;

    TDut_settings settings;

    System.Grab(portMAX_DELAY);
    memcpy(&settings, &System.sensor_settings.dut[Dut_Idx], sizeof(settings));
    System.Release();

    if(settings.type == DUT_RS485)
    {
      if(interface_mutex_take(&rs485_mutex, 0))
      {
        // Отправляем команду и ждем ответ
        uint8_t netaddr = settings.bus_address;// get netaddr
        RS485_SetParam(DUT_SERIAL_BAUDRATE_DEFAULT,DUT_SERIAL_TIMEOUT_DEFAULT,"8N1");
        uint8_t sbuf[4]={DUT_BIN_START_BYTE_CMD, 0xFF, DUT_BIN_CMD_GET_DATA, 0x00};
        sbuf[1]=netaddr; sbuf[3]=DUT_crc(sbuf,3);
        uint8_t rbuf[9];

        uint8_t is_val_ok=0;

        for(;;)
        {
          if( RS485_TxRx(sbuf, sizeof(sbuf), rbuf, sizeof(rbuf)) != sizeof(rbuf)) break;//нет ответа, не все символы пришли
          if(rbuf[0] != DUT_BIN_START_BYTE_ANSW) break;//нет ответа
          if(netaddr!=0xFF && rbuf[1] != netaddr) break;//нет ответа, не наш адрес
          if(rbuf[2] != DUT_BIN_CMD_GET_DATA) break;//ответ не на ту команду
          if( DUT_crc(rbuf,8) != rbuf[8] ) break;//нет ответа
          is_val_ok=1;
          break;
        }

        // Сохраняем значение
        System.Grab(portMAX_DELAY);
        if(is_val_ok)
        {
          System.signal_state.dut[Dut_Idx] = *((uint16_t*)&rbuf[4]);
          System.signal_state.dut_is_ok[Dut_Idx]=1;
        }
        else
        {
          System.signal_state.dut[Dut_Idx]=0;
          System.signal_state.dut_is_ok[Dut_Idx]=0;
        }
        System.Release();

        interface_mutex_give(&rs485_mutex);
      }
      else
      {
        if(Dut_Idx) Dut_Idx--;
        else        Dut_Idx=MAX_DUTS-1;

        if(is_interface_busy!=NULL) *is_interface_busy=1;
      }
    }
    else
    {
      // Сохраняем значение
      System.Grab(portMAX_DELAY);
      System.signal_state.dut[Dut_Idx]=0;
      if(settings.type == DUT_OFF)  System.signal_state.dut_is_ok[Dut_Idx]=1;
      else                          System.signal_state.dut_is_ok[Dut_Idx]=0;
      System.Release();
    }

    return;
  }
#endif //DUTS_PRESENT

  //- Основной цикл --------------------------------------------------------------
  void CSignal_manager::Main_cycle(void)
  {
#if defined(MFI_PRESENT)
    mfi_gpio_init();//мультифункциональные входы инициализируем один раз
#endif //MFI_PRESENT

#if defined(DIGITAL_OUTPUTS_PRESENT)
    outputs_init();
#endif //DIGITAL_OUTPUTS_PRESENT

    for(;;)
    {
      System.signal_state.is_on=1;

      this->Gpio_Init();
      this->Adc_Init();

#if defined(MFI_PRESENT)
      mfi_init();
#endif //MFI_PRESENT

      vTaskDelay(10);

#if defined(GSENSOR_PRESENT)
      this->Accel_Init(0);
#endif

#if (defined(RS485_PRESENT) || defined(RS232_PRESENT))
      __RS232_RS485_PWR_EN();
#endif

#if defined(RS485_PRESENT)
      RS485_Init();
#endif //RS485_PRESENT

#if defined(RS232_PRESENT)
      RS232_Init();
#endif //RS232_PRESENT

#if defined(RS_UART_PRESENT)
      RSUART_Init();
#endif //RS_UART_PRESENT

#if defined(BR_PRESENT)
      this->br_handler(WAKEUP, NULL);
#endif //BR_PRESENT

#if defined(DUTS_PRESENT)
      //RS485_Init();
#endif

#if defined(VEGA_CAN_PRESENT)
      subtask_wakeup(&can_subtask_ctx);
      //очищаем can_commands_queue
      while(xQueueReceive(Signal_manager.can_commands_queue,  &this->input_message, 0));
#endif //VEGA_CAN_PRESENT

#if defined(INTERNAL_NRF_PRESENT)
      subtask_wakeup(&nrf_subtask_ctx);
#endif //INTERNAL_NRF_PRESENT

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
      subtask_wakeup(&ble_subtask_ctx);
#endif //EXTERNAL_BLE_BOARD_PRESENT

      uint32_t slow_timer_1=xTaskGetTickCount();

#if defined(BR_PRESENT)
      uint32_t br_poll_timer=xTaskGetTickCount()+200;//тут опрашивается блок расширения, даем время ему проснуться
#endif //BR_PRESENT

#if defined(DUTS_PRESENT)
      uint32_t duts_poll_timer=xTaskGetTickCount();
#endif //DUTS_PRESENT

      uint8_t sleep_type;

#if defined(ALARM_BUTTON_PRESENT)
      uint32_t fix_alarm_timer_S=GetSecondsFromStartup();
      if(System.signal_state.alarm_button && System.sensor_settings.alarm_button.din_num) fix_alarm_timer_S+=ALARM_BUTTON_FIX_TIME_S;//если после сна состояние тервоги, удерживаем его
#endif

      static struct panic_ctx_struct
      {
        uint8_t panic_event;
        TVega_panic_args panic_args;
        uint16_t delay_time;
        bool  prev_ign;

        uint8_t server_notify_id;
        uint32_t uid;
        uint8_t nonce[8];

        uint32_t panic_timer_1;
        uint32_t panic_timer_2;
      }panic_ctx=
      {
        .panic_event=0,
        .panic_args.type=0,
        .delay_time=500,
      };

      for(;;)
      {
        if(this->Wait_message(20))
        {
          if(this->input_message.code==COMMAND_SLEEP)
          {
            if(this->input_message.subcode==NORMAL_SLEEP)
            {
              __PRINTF("%s: recv sleep cmd\n", this->name);
              sleep_type=NORMAL_SLEEP;
            }
            else if(this->input_message.subcode==BOOT_UPD_QUASI_SLEEP)
            {
              __PRINTF("%s: recv boot upd cmd\n", this->name);
              sleep_type=BOOT_UPD_QUASI_SLEEP;
            }
            else
            {
              __PRINTF("%s: recv emerg sleep cmd\n", this->name);
              sleep_type=EMERGENCY_SLEEP;
            }
            break;
          }

#if defined(VEGA_CAN_PRESENT)
          else if(this->input_message.code==COMMAND_CAN_CONTROL)
          {
            if(this->input_message.subcode==YADRIVE_PANIC_SUBCODE)
            {
              panic_ctx.panic_event=1;

              memcpy(&panic_ctx.panic_args, &this->input_message.ext_args, sizeof(panic_ctx.panic_args));
              panic_ctx.server_notify_id=this->input_message.source_id;
              panic_ctx.uid=this->input_message.command_uid;
              memcpy(panic_ctx.nonce, this->input_message.command_nonce, sizeof(panic_ctx.nonce));
              panic_ctx.delay_time=1000;
              if(panic_ctx.panic_args.type==2) panic_ctx.delay_time+=500;
              panic_ctx.prev_ign=false;
            }
            else
            {
              if(!xQueueSend(Signal_manager.can_commands_queue, &this->input_message, 0))
              {
                uint8_t add_res=add_server_notify(this->input_message.source_id, this->input_message.command_uid, this->input_message.command_nonce, IS_BUSY_NOTIFY_STATE, "command lost (busy)");
                notify_logging(this->input_message.source_id, this->input_message.command_uid, IS_BUSY_NOTIFY_STATE, "command lost (busy)", add_res);
              }
            }
          }
#endif //VEGA_CAN_PRESENT

#if defined(INTERNAL_NRF_PRESENT)
          else if(this->input_message.code==COMMAND_SIGNAL_NRF_ADD_RELAY || this->input_message.code==COMMAND_SIGNAL_NRF_DEL_RELAY || this->input_message.code==COMMAND_SIGNAL_NRF_ADD_MARK || this->input_message.code==COMMAND_SIGNAL_NRF_DEL_MARK)
          {
            uint8_t res=signalman_nrf_subtask_commnad_handle(this->input_message.code, this->input_message.subcode);
            if(res==NRF_SUBTASK_COMMAND_IS_BUSY)
            {
              LOG("%s: nrf cmd lost (busy)\n", this->name);
            }
            else if(res==NRF_SUBTASK_COMMAND_UNKNOWN_ARG)
            {
              LOG("%s: nrf cmd unknown arg\n", this->name);
            }
          }
#endif //INTERNAL_NRF_PRESENT

//#if defined(PROD_TESTING_PRESENT)
          else if(this->input_message.code==COMMAND_SIGNAL_START_PROD_TESTS)
          {
            start_prod_tests_cmd=1;
          }
//#endif //PROD_TESTING_PRESENT
        }

#if defined(VEGA_CAN_PRESENT)
        if(xQueuePeek(Signal_manager.can_commands_queue,  &this->input_message, 0))
        {
          uint8_t res=signalman_can_subtask_commnad_handle(this->input_message.subcode, this->input_message.source_id, this->input_message.command_uid, this->input_message.command_nonce, &this->input_message.ext_args);
          if(res==CAN_SUBTASK_COMMAND_START_PROCCES || res==CAN_SUBTASK_COMMAND_UNKNOWN_ARG)
          {
            xQueueReceive(Signal_manager.can_commands_queue,  &this->input_message, 0);
            if(res==CAN_SUBTASK_COMMAND_UNKNOWN_ARG)
            {
              uint8_t add_res=add_server_notify(this->input_message.source_id, this->input_message.command_uid, this->input_message.command_nonce, UNKNOWN_ARGS_NOTIFY_STATE, "command unknown arg");
              notify_logging(this->input_message.source_id, this->input_message.command_uid, UNKNOWN_ARGS_NOTIFY_STATE, "command unknown arg", add_res);
            }
          }
        }

        //Разблокировка капота, при падении напряжения бортовой сети
#if defined(YANDEX_AUTO_HOOD_UNLOCK_PRESENT)
        auto_hood_unlock_watchdog();
#endif //defined(YANDEX_AUTO_HOOD_UNLOCK_PRESENT)

        // Validation fuel level
        check_fuel_level_watchdog();

        //остановка двигателя в случае открытия дверей итд
        engine_stop_watchdog();

        //отключения функции старт-стоп на AUDI
        startstop_off_watchdog(false);

        //асинхронное закрытие окон
        async_analog_windows_closing_watchdog(false);

        //асинхронное выключение питания ключа
        async_key_power_off_watchdog(false);

        //включение режима ECO при старте
        forced_eco_on_handler();

        //выключение двигателя если он на ручнике и др.
        engine_stop_by_xxx_handler();

        //обновления статуса зарядки
        wakeup_electric_car_on_charge_handler();

        //отложенное дозакрытие окон
        delayed_windows_closing_handler();

        //отработка режима охрана
#if defined(YANDEX_GUARD_MODE_PRESENT)
        guard_handler();
#endif //defined(YANDEX_GUARD_MODE_PRESENT)

#if defined(YANDEX_ANTITHEFT_PRESENT)
#warning
        antitheft_handler();
#endif //defined(YANDEX_ANTITHEFT_PRESENT)

#if defined(YANDEX_ANTITHEFT2_PRESENT)
        antitheft2_handler();
#endif //defined(YANDEX_ANTITHEFT2_PRESENT)

        //отработка режима паники
        if(true)
        {
          if(panic_ctx.panic_event)
          {
            panic_ctx.panic_event=0;
            panic_ctx.panic_timer_1=xTaskGetTickCount()+panic_ctx.panic_args.time*1000;
            panic_ctx.panic_timer_2=xTaskGetTickCount()-1;

            if(panic_ctx.panic_args.type>3 || panic_ctx.panic_args.time>60)
            {
              const char* res_str="panic cmd wrong arg";
              uint8_t add_res=add_server_notify(panic_ctx.server_notify_id, panic_ctx.uid, panic_ctx.nonce, UNKNOWN_ARGS_NOTIFY_STATE, res_str);
              notify_logging(panic_ctx.server_notify_id, panic_ctx.uid, UNKNOWN_ARGS_NOTIFY_STATE, res_str, add_res);
            }
            else if(panic_ctx.panic_args.type==0)
            {
              const char* res_str="stop panic by command";
              uint8_t add_res=add_server_notify(panic_ctx.server_notify_id, panic_ctx.uid, panic_ctx.nonce, IS_LAST_NOTIFY_STATE, res_str);
              notify_logging(panic_ctx.server_notify_id, panic_ctx.uid, IS_LAST_NOTIFY_STATE, res_str, add_res);
            }
            else
            {
              LOG("%s: new panic process, type=%hhu, time=%hhu\n", this->name, panic_ctx.panic_args.type, panic_ctx.panic_args.time);
            }
          }

          bool ign;
          if(System.can_state.sec_flags.ignition || System.signal_state.ignition) ign=true;
          else                                                                    ign=false;

          if(panic_ctx.panic_args.type && panic_ctx.prev_ign && !ign)
          {
            panic_ctx.panic_args.type=0;
            const char* res_str="stop panic by ign off";
            uint8_t add_res=add_server_notify(panic_ctx.server_notify_id, panic_ctx.uid, panic_ctx.nonce, IS_LAST_NOTIFY_STATE, res_str);
            notify_logging(panic_ctx.server_notify_id, panic_ctx.uid, IS_LAST_NOTIFY_STATE, res_str, add_res);
          }
          else if(panic_ctx.panic_args.type && timeAfter(xTaskGetTickCount(), panic_ctx.panic_timer_1))
          {
            panic_ctx.panic_args.type=0;
            const char* res_str="stop panic by timeout";
            uint8_t add_res=add_server_notify(panic_ctx.server_notify_id, panic_ctx.uid, panic_ctx.nonce, IS_LAST_NOTIFY_STATE, res_str);
            notify_logging(panic_ctx.server_notify_id, panic_ctx.uid, IS_LAST_NOTIFY_STATE, res_str, add_res);
          }

          panic_ctx.prev_ign=ign;

          //если работает аларм, то пропускаем
          if(panic_ctx.panic_args.type && !is_horn_alarm_active() && !is_can_horn_alarm_active())
          {
            if(timeAfter(xTaskGetTickCount(), panic_ctx.panic_timer_2))
            {
              if(!uxQueueMessagesWaiting(Signal_manager.can_commands_queue))
              {
                uint8_t cmd=CAN_BLINKER_FLASHING_SUBCODE;

                if(panic_ctx.panic_args.type==1)      cmd=CAN_HORN_SIGNALING_SUBCODE;
                else if(panic_ctx.panic_args.type==2) cmd=CAN_BLINKER_FLASHING_SUBCODE;
                else if(panic_ctx.panic_args.type==3) cmd=CAN_HORN_W_BLINKER_SIGNALIG_SUBCODE;

                uint8_t add_res=signalman_can_subtask_commnad_handle(cmd, 0xff, 0, NULL, NULL);
                if(CAN_SUBTASK_COMMAND_IS_BUSY!=add_res)
                {
                  panic_ctx.panic_timer_2=xTaskGetTickCount()+panic_ctx.delay_time;
                }
              }
            }
          }
        }
#endif //VEGA_CAN_PRESENT

#if defined(JQ6500_AUDUO_PRESENT)
        audio_handler(common_tx_buff, sizeof(common_tx_buff), common_rx_buff, sizeof(common_rx_buff));
#endif //defined(JQ6500_AUDUO_PRESENT)

#if IWDG_ENABLE > 0
        IWDG_ReloadCounter();
#endif
        //System.server_state.usb_connected=USB_DETECT_PIN_STATE();

#if defined(TAMPERS_PRESENT)
        this->Tamper_handler();
#endif //TAMPERS_PRESENT

        if(timeAfter(xTaskGetTickCount(), slow_timer_1))
        {
#if defined(DIGITAL_OUTPUTS_PRESENT)
          outputs_handler();
#endif

#if defined(MFI_PRESENT)
          mfi_handler();
#endif //MFI_PRESENT

          this->Ign_handler();

#if defined(ALARM_BUTTON_PRESENT)
          this->Alarm_handler(&fix_alarm_timer_S);
#endif //ALARM_BUTTON_PRESENT

#if defined(INTERNAL_AKB_PRESENT)
          this->LiPo_handler(false);
#endif //INTERNAL_AKB_PRESENT

          this->Adc_handler();

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
          static const float ext_voltage_off_th=3.0f;
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K)
          static const float ext_voltage_off_th=5.0f;
#endif //DEVICE_TYPE_VEGA_MT_25K

          if((float)System.signal_state.external_voltage >= ext_voltage_off_th)
          {
            System.signal_state.power_source=EXTERNAL_POWER;
            LedR_On();
          }
          else
          {
            System.signal_state.power_source=ACC_POWER;
            LedR_Toggle();
          }

          if(true)
          {
            static uint8_t __is_on=0xff;
            static uint8_t __pwr_bounce=0;
            static uint32_t __last_change_time=0;
            static uint32_t __last_check_time=0;

            if(timeAfter(xTaskGetTickCount(), __last_check_time+500))
            {
              uint8_t is_on=__GET_VBAT_IN_DETECT_STATE();

              if(__is_on!=is_on)
              {
                if(!timeAfter(GetSecondsFromStartup(), __last_change_time+4))
                {
                  if(__pwr_bounce<255) __pwr_bounce++;
                }
                else
                {
                  __pwr_bounce=0;
                }

                if(__pwr_bounce<6)
                {
                  LOG("%s: ext pwr changed to \"%s\"\n", this->name, (is_on)?("on"):("off"));
                }
                else if(__pwr_bounce==6)
                {
                  LOG("%s: ext pwr bounce detect, stop logging!\n", this->name);
                }

                __is_on=is_on;

                __last_change_time=GetSecondsFromStartup();
              }

              __last_check_time=xTaskGetTickCount();
            }
          }

#if defined(GSENSOR_PRESENT)
          if(AccelSensors_Check())
          {
            System.Grab(portMAX_DELAY);

            if(System.sensor_settings.gsensor_movesensor.stopfix_time)
              System.signal_state.gsensor_move_sensor=AccelSensor_MoveState;
            else
              System.signal_state.gsensor_move_sensor=0;

            System.signal_state.evacuation_sensor=AccelSensor_TiltState;
            System.signal_state.gsensor_axis_x = (AccelAxes_Curr.AXIS_X>>6)/256.0f;
            System.signal_state.gsensor_axis_y = (AccelAxes_Curr.AXIS_Y>>6)/256.0f;
            System.signal_state.gsensor_axis_z = (AccelAxes_Curr.AXIS_Z>>6)/256.0f;

            uint32_t stopfix_time=System.sensor_settings.gsensor_movesensor.stopfix_time;

            System.Release();

            if(GetAccelMoveDelayToStop()!=stopfix_time)
            {
              SetAccelMoveDelayToStop(stopfix_time);
              //переинициализируем, чтобы при настройка применилась сразу
              this->Accel_DeInit();
              this->Accel_Init(0);
            }
            else
            {
              uint32_t curr_image=crc32_fast_full((uint8_t*)&AccelAxes_Curr, sizeof(AccelAxes_Curr));
              if(this->accel_wdt_image==curr_image)
              {
                if(timeAfter(xTaskGetTickCount(), this->accel_wdt_timer1_mS))
                {
                  LOG("%s: accel wdt1 triggered, reinit it...\n", this->name);
                  this->Accel_DeInit();
                  this->Accel_Init(0);
                }
              }
              else
              {
                this->accel_wdt_image=curr_image;
                this->accel_wdt_timer1_mS=xTaskGetTickCount()+ACCEL_WDT_TIMEOUT_mS-1;
              }
            }
            this->accel_wdt_timer2_mS=xTaskGetTickCount()+20000-1;
          }
          else if(timeAfter(xTaskGetTickCount(), this->accel_wdt_timer2_mS))
          {
            //иногда попадаем сюда, по всей видимсоти из-за кривого драйвера или фронт и пропускаем прерывание
            LOG("%s: accel wdt2 triggered...\n", this->name);
            this->Accel_DeInit();
            this->Accel_Init(0);
          }
#endif

          slow_timer_1=xTaskGetTickCount()+100-1;
        }

#if defined(BR_PRESENT)
        if(timeAfter(xTaskGetTickCount(), br_poll_timer))
        {
          uint8_t is_interface_busy;
          uint8_t result;

          result=this->br_handler(POLL_DATA, &is_interface_busy);
          if(!is_interface_busy) System.signal_state.is_br_ok=result;

          //если интерфейс интерфес оказялся занят, то форсирсируем опрос
          if(is_interface_busy) br_poll_timer=xTaskGetTickCount();
          else
          {
            if(0){}
#if defined(PROD_TESTING_PRESENT)
            else if(start_prod_tests_cmd) {br_poll_timer=xTaskGetTickCount()+20;}//если идет тест
#endif //PROD_TESTING_PRESENT
            else {br_poll_timer=xTaskGetTickCount()+453;}
          }
        }
#endif //BR_PRESENT

#if defined(SAVER2021_PRESENT)
#if defined(RS232_PRESENT)
        //place at ROM
        static const saver2021_itf_driver_t saver2021_rs232 =
        {
          .set_param = RS232_SetParam,
          .txrx = RS232_TxRx,
          .lock = RS232_Grab,
          .unlock = RS232_Release,
        };
#endif //RS232_PRESENT

#if defined(RS_UART_PRESENT)
        //place at ROM
        static const saver2021_itf_driver_t saver2021_rsuart =
        {
          .set_param = RSUART_SetParam,
          .txrx = RSUART_TxRx,
          .lock = RSUART_Grab,
          .unlock = RSUART_Release,
        };
#endif //RS_UART_PRESENT

        //place at ROM
        static const saver2021_pconst_t saver2021_pconst =
        {
          .rx_buff = common_rx_buff,
          .rx_buff_size = sizeof(common_rx_buff),
          .state = &System.saver2021_state,
          .state_lock_unlock = lock_state_and_settings,
        };

        static saver2021_ctx_t saver2021_ctx =
        {
          .poll_timer = 0,
          .flags = 0,

          .pconst = &saver2021_pconst,
          .itf = NULL,
        };

        //todo
        System.saver_settings.type = 0;           //SAVER2021
        System.saver_settings.interface_type = 3; //UART

        if(false){}
#if defined(RS232_PRESENT)
        else if(System.saver_settings.interface_type == 1) {saver2021_ctx.itf = &saver2021_rs232;}
#endif //RS232_PRESENT
#if defined(RS_UART_PRESENT)
        else if(System.saver_settings.interface_type == 3) {saver2021_ctx.itf = &saver2021_rsuart;}
#endif //RS_UART_PRESENT
        else                                               {saver2021_ctx.itf = NULL;}

        saver2121_handler(&saver2021_ctx);
#endif //defined(SAVER2021_PRESENT)

#if defined(DUTS_PRESENT)
        if(timeAfter(xTaskGetTickCount(), duts_poll_timer))
        {
          uint8_t is_interface_busy;
          DUT_Polling(&is_interface_busy);

          //если интерфейс один из интерфесов оказялся занят, то форсирсируем опрос
          if(is_interface_busy) duts_poll_timer=xTaskGetTickCount();
          else                  duts_poll_timer=xTaskGetTickCount()+900;
        }
#endif //DUTS_PRESENT
      }

      LedR_Off(); LedG_Off(); LedB_Off();
#if defined(INTERNAL_AKB_PRESENT)
      if(sleep_type==EMERGENCY_SLEEP) {__CHARGE_LIPO_DIS();}
#endif //INTERNAL_AKB_PRESENT
      this->Adc_DeInit();
      this->Gpio_DeInit();

#if defined(GSENSOR_PRESENT)
      this->Accel_DeInit();
#endif //GSENSOR_PRESENT

#if defined(VEGA_CAN_PRESENT)
    //engine_stop_watchdog();
    startstop_off_watchdog(true);
    async_analog_windows_closing_watchdog(true);
    async_key_power_off_watchdog(true);
#endif //VEGA_CAN_PRESENT

#if defined(MFI_PRESENT)
      mfi_deinit();
#endif //MFI_PRESENT

#if defined(DIGITAL_OUTPUTS_PRESENT)
      //outputs_deinit(); //не нужно деинициализировть. Если настроен как частотный выход, то при уходе в сон вывод будет в случайном состоянии
#endif //DIGITAL_OUTPUTS_PRESENT

#if defined(BR_PRESENT)
      //делаем 4 попытки уложить в сон блок расширения
      for(uint8_t i=0; i<4; i++)
      {
        uint8_t is_interface_busy;
        System.signal_state.is_br_ok=this->br_handler(GOTO_SLEEP, &is_interface_busy);
        if(System.signal_state.is_br_ok && !is_interface_busy) break;
        if(i!=(4-1)) vTaskDelay(500);
      }
      if(!System.signal_state.is_br_ok) LOG("%s: br sleep err\n", this->name);
#endif //BR_PRESENT

#if defined(DUTS_PRESENT)
      //RS485_DeInit();
#endif

#if defined(VEGA_CAN_PRESENT)
      subtask_goto_sleep(&can_subtask_ctx);
#endif //VEGA_CAN_PRESENT

#if defined(INTERNAL_NRF_PRESENT)
      subtask_goto_sleep(&nrf_subtask_ctx);
#endif //INTERNAL_NRF_PRESENT

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
      subtask_goto_sleep(&ble_subtask_ctx);
#endif //EXTERNAL_BLE_BOARD_PRESENT

#if defined(RS485_PRESENT)
      RS485_DeInit();
#endif //RS485_PRESENT

#if defined(RS232_PRESENT)
      RS232_DeInit();
#endif //RS232_PRESENT

#if defined(RS_UART_PRESENT)
      RSUART_DeInit();
#endif //RS_UART_PRESENT

#if defined(GSENSOR_PRESENT)
      if(sleep_type==NORMAL_SLEEP && System.power_settings.wakeup_by_move)
      {
        this->Accel_Init(1);//инициализируем работу акселерометра для спящего режима
      }
#endif

      System.signal_state.power_source=UNKNOWN_POWER;
      System.signal_state.is_on=0;

      if(sleep_type==NORMAL_SLEEP)
      {
        LOG("%s: goto sleep...\n", this->name);
      }
      else if(sleep_type==EMERGENCY_SLEEP)
      {
        LOG("%s: goto emerg sleep...\n", this->name);
      }
      else
      {
        //none
      }

#if (defined(RS485_PRESENT) || defined(RS232_PRESENT))
      __RS232_RS485_PWR_DIS();
#endif
      uint8_t wakeup_source;
      uint8_t is_rcc_ok=0;
      if(sleep_type==BOOT_UPD_QUASI_SLEEP)
      {
#if (defined(USE_UART_DEBUG_PORT))
        __RS232_RS485_PWR_EN();
#endif
        this->Boot_upd_handler();
        wakeup_source=SUBCODE_WAKEUP_BY_BOOT_UPD_FAIL;
#if (defined(USE_UART_DEBUG_PORT))
        __RS232_RS485_PWR_DIS();
#endif
      }
      else //sleep_type==NORMAL_SLEEP || sleep_type==EMERGENCY_SLEEP
      {
        AddTickTimerStop();

        portENTER_CRITICAL();
#if configUSE_TICKLESS_IDLE == 1
#error не может использоваться с configUSE_TICKLESS_IDLE == 1
#endif
        SysTick->CTRL &=~ SysTick_CTRL_ENABLE_Msk;
        portEXIT_CRITICAL();
        PWR_PVDCmd(DISABLE);

#if (defined(USE_UART_DEBUG_PORT))
        __RS232_RS485_PWR_DIS();
#endif
        slow_timer_1=0;
        uint32_t wakeup_timeS=GetUnixTime()+60*System.power_settings.wakeup_by_time;

        RTC_WakeUpCmd(DISABLE);
        RTC_ITConfig(RTC_IT_WUT, ENABLE);
        RTC_WakeUpClockConfig(RTC_WakeUpClock_RTCCLK_Div16);
        RTC_SetWakeUpCounter(4096);
        RTC_ClearITPendingBit(RTC_IT_WUT);
        EXTI_ClearITPendingBit(EXTI_Line22);

        for(;;)
        {
          //->
          // Enable Wakeup Counter
          RTC_WakeUpCmd(ENABLE);
          //<-
          PWR_EnterSTOPMode(PWR_Regulator_LowPower, PWR_STOPEntry_WFI);

          /*
          --->sleep--->
          --------------
          <---wakeup<---
          */

          // Disable Wakeup Counter
          RTC_WakeUpCmd(DISABLE);

#if IWDG_ENABLE > 0
          IWDG_ReloadCounter();
#endif

          if(sleep_type==NORMAL_SLEEP)
          {
            if(!System.power_settings.wakeup_by_ignition && !System.power_settings.wakeup_by_time

#if defined(ALARM_BUTTON_PRESENT)
               && !System.sensor_settings.alarm_button.din_num
#endif //ALARM_BUTTON_PRESENT

#if defined(GSENSOR_PRESENT)
                 && !System.power_settings.wakeup_by_move
#endif //GSENSOR_PRESENT
                     )
            {
              wakeup_source=SUBCODE_INVALID_WAKEUP_CONFIG;
              break;
            }

#if defined(TAMPERS_PRESENT)
            this->Tamper_handler();
            if(System.signal_state.tamper[0] || System.signal_state.tamper[1])
            {
              wakeup_source=SUBCODE_WAKEUP_BY_TAMPER;
              break;
            }
#endif //TAMPERS_PRESENT

#if defined(MFI_PRESENT)
            mfi_handler_in_sleep();
#endif //MFI_PRESENT

#if defined(ALARM_BUTTON_PRESENT)
            if(System.sensor_settings.alarm_button.din_num)
            {
              this->Alarm_handler(NULL);
              if(System.signal_state.alarm_button)
              {
                wakeup_source=SUBCODE_WAKEUP_BY_ALARM_BUTTON;
                break;
              }
            }
#endif //ALARM_BUTTON_PRESENT

            this->Ign_handler();

            if(System.power_settings.wakeup_by_ignition)
            {
              if(System.signal_state.ignition)
              {
                wakeup_source=SUBCODE_WAKEUP_BY_IGN;
                break;
              }
            }

            //т.к. pull-up резистор не отключается в suspend, на входе непонятно ~1.5 В, можем ложно проснуться
            if(USB_DETECT_PIN_STATE())
            {
              wakeup_source=SUBCODE_WAKEUP_BY_USB;
              break;
            }

            if(System.power_settings.wakeup_by_move)
            {
#if defined(GSENSOR_PRESENT)
              if(System.signal_state.accel_no_init || AccelSensors_Check())
              {
                wakeup_source=SUBCODE_WAKEUP_BY_MOVE;
                break;
              }
#endif //GSENSOR_PRESENT
            }

            if(System.power_settings.wakeup_by_time)
            {
              if(timeAfter(GetUnixTime(), wakeup_timeS))
              {
                wakeup_source=SUBCODE_WAKEUP_BY_TIME;
                break;
              }
            }

#if defined(INTERNAL_AKB_PRESENT)
            this->LiPo_handler(true);
#endif //INTERNAL_AKB_PRESENT

            if(slow_timer_1>=1)
            {
              DA_LedR_On();
              __DELAY_MS(20, TARGET_CPU_FREQ_IN_SLEEP_HZ);
              DA_LedR_Off();
              slow_timer_1=0;
            }
            else
            {
              slow_timer_1++;
            }
          }
          else//sleep_type==EMERGENCY_SLEEP
          {
            if(__GET_VBAT_IN_DETECT_STATE())
            {
#if defined(INTERNAL_AKB_ON_OFF_CONTROL_PRESENT)
              __LIPO_ON_EN();
#endif //INTERNAL_AKB_ON_OFF_CONTROL_PRESENT
              wakeup_source=SUBCODE_WAKEUP_BY_EXT_POWER;
              break;
            }
#if defined(INTERNAL_AKB_ON_OFF_CONTROL_PRESENT)
            else
            {
              __LIPO_ON_DIS();
              //далее система должна выключиться
            }
#endif //INTERNAL_AKB_ON_OFF_CONTROL_PRESENT
          }

        }
        RCC_HSEConfig(RCC_HSE_ON);
        if(SUCCESS == RCC_WaitForHSEStartUp())
        {
          RCC_PLLCmd(ENABLE);
          while(RCC_GetFlagStatus(RCC_FLAG_PLLRDY) == RESET);
          RCC_SYSCLKConfig(RCC_SYSCLKSource_PLLCLK);
          while(RCC_GetSYSCLKSource() != 0x08);
          is_rcc_ok=1;
        }

        PWR_PVDCmd(ENABLE);
        if(PWR_GetFlagStatus(PWR_FLAG_PVDO))  EXTI_GenerateSWInterrupt(EXTI_Line16);
        portENTER_CRITICAL();
#if configUSE_TICKLESS_IDLE == 1
#error не может использоваться с configUSE_TICKLESS_IDLE == 1
#endif
        SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
        portEXIT_CRITICAL();

        AddTickTimerStart();

#if (defined(USE_UART_DEBUG_PORT))
        __RS232_RS485_PWR_EN();
#endif
      }//sleep_type==NORMAL_SLEEP || sleep_type==EMERGENCY_SLEEP

      if((sleep_type==NORMAL_SLEEP || sleep_type==EMERGENCY_SLEEP) && !is_rcc_ok)
      {
        LOG("%s: HSE or PLL init err\n", this->name);
      }

      const char* wakeup_source_str;

      if(wakeup_source==SUBCODE_WAKEUP_BY_IGN)
      {
        wakeup_source_str="ign";
        System.signal_state.ignition=1;
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_USB)
      {
        wakeup_source_str="usb";
        //System.server_state.usb_connected=1;
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_MOVE)
      {
        wakeup_source_str="move";
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_TIME)
      {
        wakeup_source_str="time";
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_EXT_POWER)
      {
        wakeup_source_str="ext_power";
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_ALARM_BUTTON)
      {
        wakeup_source_str="alarm_button";
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_TAMPER)
      {
        wakeup_source_str="tamper";
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_BOOT_UPD_FAIL)
      {
        wakeup_source_str="boot upd fail";
      }
      else if(wakeup_source==SUBCODE_WAKEUP_BY_CAN_ACTIVITY)
      {
        wakeup_source_str="can activity";
      }
      else
      {
        wakeup_source_str="invalid wakeup config";
      }

      LOG("%s: wakeup by %s...\n", this->name, wakeup_source_str);

      this->output_message.code = COMMAND_WAKEUP;
      this->output_message.subcode = wakeup_source;
      this->Send_message(this->output_queue);

#if defined(GSENSOR_PRESENT)
      if(!System.signal_state.accel_no_init)
      {
        this->Accel_DeInit();
      }
#endif
    }
  }

  void CSignal_manager::Adc_Init(void)
  {
    DMA_InitTypeDef DMA_InitStructure;
    ADC_InitTypeDef ADC_InitStructure;

    RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_DMA2, ENABLE);
    RCC_APB2PeriphClockCmd(RCC_APB2Periph_ADC1, ENABLE);

    //memset(&this->adc_vals, 0xff, sizeof(this->adc_vals));

    /* DMA configuration ----------------------------------------------*/
    DMA_DeInit(DMA2_Stream4);
    while(DMA_GetCmdStatus(DMA2_Stream4) != DISABLE);
    DMA_InitStructure.DMA_Channel=DMA_Channel_0;
    DMA_InitStructure.DMA_PeripheralBaseAddr = (uint32_t)&ADC1->DR;
    DMA_InitStructure.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
    DMA_InitStructure.DMA_Memory0BaseAddr = (uint32_t)&adc_vals;
    DMA_InitStructure.DMA_DIR = DMA_DIR_PeripheralToMemory;
    DMA_InitStructure.DMA_BufferSize = sizeof(adc_vals)/sizeof(uint16_t);
    DMA_InitStructure.DMA_MemoryInc = DMA_MemoryInc_Enable;
    DMA_InitStructure.DMA_PeripheralDataSize = DMA_PeripheralDataSize_HalfWord;
    DMA_InitStructure.DMA_MemoryDataSize = DMA_MemoryDataSize_HalfWord;
    DMA_InitStructure.DMA_Mode = DMA_Mode_Circular;
    DMA_InitStructure.DMA_Priority = DMA_Priority_Low;
    DMA_InitStructure.DMA_FIFOMode=DMA_FIFOMode_Disable;
    DMA_InitStructure.DMA_FIFOThreshold=DMA_FIFOThreshold_Full;
    DMA_InitStructure.DMA_MemoryBurst=DMA_MemoryBurst_Single;
    DMA_InitStructure.DMA_PeripheralBurst=DMA_PeripheralBurst_Single;
    DMA_Init(DMA2_Stream4, &DMA_InitStructure);

    /* Enable DMA1 Channel1 */
    DMA_Cmd(DMA2_Stream4, ENABLE);

    /* ADC1 configuration ------------------------------------------------------*/
    ADC_InitStructure.ADC_Resolution = ADC_Resolution_12b;
    ADC_InitStructure.ADC_ScanConvMode = ENABLE;
    ADC_InitStructure.ADC_ContinuousConvMode = ENABLE;
    ADC_InitStructure.ADC_ExternalTrigConvEdge = ADC_ExternalTrigConvEdge_None;
    ADC_InitStructure.ADC_ExternalTrigConv = ADC_ExternalTrigConv_T1_CC1;
    ADC_InitStructure.ADC_DataAlign = ADC_DataAlign_Right;
    ADC_InitStructure.ADC_NbrOfConversion = sizeof(adc_vals)/sizeof(uint16_t);
    ADC_Init(ADC1, &ADC_InitStructure);

    /* Temperature sensor and Vrefint enable */
    ADC_TempSensorVrefintCmd(ENABLE);

    /* ADC1 regular channels configuration */
    uint8_t chan=1;
    ADC_RegularChannelConfig(ADC1, ADC_Channel_Vrefint,         chan++, ADC_SampleTime_480Cycles);
    ADC_RegularChannelConfig(ADC1, ADC_Channel_TempSensor,      chan++, ADC_SampleTime_480Cycles);
    ADC_RegularChannelConfig(ADC1, ADC_ONBOARD_CHANEL,          chan++, ADC_SampleTime_480Cycles);
#if defined(INTERNAL_AKB_PRESENT)
    ADC_RegularChannelConfig(ADC1, ADC_LIPO_CHANEL,             chan++, ADC_SampleTime_480Cycles);
#endif //INTERNAL_AKB_PRESENT

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
    for(uint8_t i=0; i<MFI_INPUT_CNT; i++)
    {
      ADC_RegularChannelConfig(ADC1, mfi_get_an_in_adc_chanel(i), chan++, ADC_SampleTime_144Cycles);
    }
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

    /* Enable ADC1 DMA */
    ADC_DMACmd(ADC1, ENABLE);

    /* Enable DMA request after last transfer (Single-ADC mode) */
    ADC_DMARequestAfterLastTransferCmd(ADC1, ENABLE);

    /* Enable ADC1 */
    ADC_Cmd(ADC1, ENABLE);

    ADC_SoftwareStartConv(ADC1);
  }

  void CSignal_manager::Adc_DeInit(void)
  {
    /* Disable ADC1 DMA */
    ADC_DMACmd(ADC1, DISABLE);

    /* Disable Vrefint and temperature sensor */
    ADC_TempSensorVrefintCmd(DISABLE);

    /* Disable ADC1 */
    ADC_Cmd(ADC1, DISABLE);
  }

  void CSignal_manager::Gpio_Init(void)
  {
    GPIO_InitTypeDef GPIO_InitStructure;

    //
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

    GPIO_InitStructure.GPIO_Pin = ADC_ONBOARD_PIN;
    GPIO_Init(ADC_ONBOARD_PORT, &GPIO_InitStructure);

#if defined(INTERNAL_AKB_PRESENT)
    GPIO_InitStructure.GPIO_Pin = ADC_LIPO_PIN;
    GPIO_Init(ADC_LIPO_PORT, &GPIO_InitStructure);
#endif //defined(INTERNAL_AKB_PRESENT)

    //
#if defined(INTERNAL_AKB_PRESENT)
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
    GPIO_InitStructure.GPIO_OType = GPIO_OType_OD;
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

    GPIO_InitStructure.GPIO_Pin = LIPO_CHARGE_PIN;
    GPIO_Init(LIPO_CHARGE_PORT, &GPIO_InitStructure);
    __CHARGE_LIPO_DIS();
#endif //defined(INTERNAL_AKB_PRESENT)

    //
    USB_DETECT_INIT(GPIO_InitStructure);

    //
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN;
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

    GPIO_InitStructure.GPIO_Pin = VBAT_IN_DETECT_PIN;
    GPIO_Init(VBAT_IN_DETECT_PORT, &GPIO_InitStructure);

#if defined(IGNITION_PRESENT)
    GPIO_InitStructure.GPIO_Pin = IGN_PIN;
    GPIO_Init(IGN_PORT, &GPIO_InitStructure);
#endif //defined(IGNITION_PRESENT)

#if defined(TAMPERS_PRESENT)
    GPIO_InitStructure.GPIO_Pin = TAMPER1_PIN;
    GPIO_Init(TAMPER1_PORT, &GPIO_InitStructure);

    GPIO_InitStructure.GPIO_Pin = TAMPER2_PIN;
    GPIO_Init(TAMPER2_PORT, &GPIO_InitStructure);
#endif //defined(TAMPERS_PRESENT)

#if (defined(RS485_PRESENT) || defined(RS232_PRESENT))
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
    GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
    GPIO_InitStructure.GPIO_Pin = RS232_RS485_PWR_PIN;
    GPIO_Init(RS232_RS485_PWR_PORT, &GPIO_InitStructure);
    __RS232_RS485_PWR_DIS();
#endif //(defined(RS485_PRESENT) || defined(RS232_PRESENT))

#if defined(EXT_BT_PWR_PRESENT)
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
    GPIO_InitStructure.GPIO_OType = EXT_BT_PWR_PORT_MODE;
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
    GPIO_InitStructure.GPIO_Pin = EXT_BT_PWR_PIN;
    GPIO_Init(EXT_BT_PWR_PORT, &GPIO_InitStructure);
    __EXT_BT_PWR_DIS();
#endif //defined(EXT_BT_PWR_PRESENT)
  }

  void CSignal_manager::Gpio_DeInit(void)
  {
    GPIO_InitTypeDef GPIO_InitStructure;

    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

    GPIO_InitStructure.GPIO_Pin = ADC_ONBOARD_PIN;
    GPIO_Init(ADC_ONBOARD_PORT, &GPIO_InitStructure);

#if defined(INTERNAL_AKB_PRESENT)
    GPIO_InitStructure.GPIO_Pin = ADC_LIPO_PIN;
    GPIO_Init(ADC_LIPO_PORT, &GPIO_InitStructure);
#endif //defined(INTERNAL_AKB_PRESENT)

  }

  void CSignal_manager::Ign_handler(void)
  {
    if(0){}
#if (defined(CANPRO_PRESENT) || defined(VEGA_CAN_PRESENT))
    else if(System.sensor_settings.use_can_ignition)
    {
      if(System.can_state.sec_flags.ignition || System.can_state.sec_flags.dynamic_ign || System.can_state.sec_flags.engine_is_on) System.signal_state.ignition=1;
      else                                                                                                                         System.signal_state.ignition=0;
    }
#endif //(defined(CANPRO_PRESENT) || defined(VEGA_CAN_PRESENT))
    else
    {
#if defined(IGNITION_PRESENT)
      System.signal_state.ignition=__GET_IGN_STATE();
#elif defined(MFI_PRESENT)
      uint8_t ign_din_num=System.sensor_settings.ignition.din_num;

      if(ign_din_num>0 && ign_din_num<=MFI_INPUT_CNT)
      {
        if(System.sensor_settings.mfi[ign_din_num-1].input_type==DIG_IN)
        {
          System.signal_state.ignition=System.signal_state.digital_input[ign_din_num-1];
        }
        else
        {
          System.signal_state.ignition=__GET_VBAT_IN_DETECT_STATE();
        }
      }
      else
      {
        System.signal_state.ignition=__GET_VBAT_IN_DETECT_STATE();
      }
#else
      System.signal_state.ignition=__GET_VBAT_IN_DETECT_STATE();
#endif
    }
  }

#if defined(INTERNAL_AKB_PRESENT)
  void CSignal_manager::LiPo_handler(bool is_sleep)
  {
    if((DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)  && !is_sleep)
    {
      if(timeAfter(GetSecondsFromStartup(), 15))
      {
        static float charge_treshold=11.0f;
        static bool is_first=true;

        if(is_first)
        {
          LOG("%s: ext: %.2f, LiPo: %.2f, temp: %.2f\n", this->name, System.signal_state.external_voltage, System.signal_state.internal_acc_voltage, System.signal_state.internal_temp);
          is_first=false;
        }

        if(System.signal_state.external_voltage>charge_treshold)
        {
          __CHARGE_LIPO_EN();
          charge_treshold=11.0f;
        }
        else
        {
          __CHARGE_LIPO_DIS();
          charge_treshold=12.1f;
        }
      }
    }
    else
    {
      if(System.signal_state.ignition)
      {
        __CHARGE_LIPO_EN();
      }
      else
      {
        __CHARGE_LIPO_DIS();
      }
    }
  }
#endif //defined(INTERNAL_AKB_PRESENT)

  void CSignal_manager::Adc_handler(void)
  {
    uint16_t adc_vref=adc_vals.adc_vref;
    float onboard, inttemp;
#if defined(INTERNAL_AKB_PRESENT)
    float lipo;
#endif //defined(INTERNAL_AKB_PRESENT)
    if(adc_vref>=(uint16_t)(1.11*4095.0f/3.3f) && adc_vref<=(uint16_t)(1.31*4095.0f/3.3f))
    {
      if(DEVICE_TYPE == DEVICE_TYPE_CAN_WAY_N || DEVICE_TYPE == DEVICE_TYPE_ROAD_3G)
      {
        onboard=adc_vals.adc_onboard*((3.3f*((390.0f+33.0f)/33.0f))/4095.0f);
      }
      else
      {
        onboard=adc_vals.adc_onboard*((3.3f*((400.0f+33.0f)/33.0f))/4095.0f);
      }

      if(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_PLATON_COMMERCIAL || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      {
        inttemp = (adc_vals.adc_inttemp*(3.3f/4095.0f) - 0.76f)/(2.5f/1000.0f) + 25.0f;
      }
      else
      {
        inttemp = (1.43f - adc_vals.adc_inttemp*(3.3f/4095.0f))/(4.3f/1000.0f) + 25.0f;
      }

#if defined(INTERNAL_AKB_PRESENT)
      lipo=adc_vals.adc_lipo*((3.3f*((33.0f+33.0f)/33.0f))/4095.0f);
#endif //defined(INTERNAL_AKB_PRESENT)
    }
    else
    {
      static uint32_t valid_vref_fault_timeS=0;
      //делаем запсь в лог не чаще 10 секунд
      if(timeAfter(GetSecondsFromStartup(), valid_vref_fault_timeS))
      {
        LOG("%s: internal vref fault! %.3f\n", this->name, (float)adc_vref*(3.3f/4095.0f));
        valid_vref_fault_timeS=(10-1)+GetSecondsFromStartup(); ;
      }
    }

    System.Grab(portMAX_DELAY);
    System.signal_state.external_voltage=onboard;
    System.signal_state.internal_temp=inttemp;
#if defined(INTERNAL_AKB_PRESENT)
    System.signal_state.internal_acc_voltage=lipo;
#endif //defined(INTERNAL_AKB_PRESENT)
    System.Release();
  }

#if defined(ALARM_BUTTON_PRESENT)
  void CSignal_manager::Alarm_handler(uint32_t* fix_alarm_timer_S)
  {
    uint8_t alarm_din_num=System.sensor_settings.alarm_button.din_num;
    if(alarm_din_num>0 && alarm_din_num<=MFI_INPUT_CNT)
    {
      if(System.sensor_settings.mfi[alarm_din_num-1].input_type==DIG_IN)
      {
        //если в тревоге, опрашиваем состояние не чаще ALARM_BUTTON_FIX_TIME_S
        if(System.signal_state.alarm_button==0 || fix_alarm_timer_S==NULL || timeAfter(GetSecondsFromStartup(), *fix_alarm_timer_S))
        {
          System.signal_state.alarm_button=System.signal_state.digital_input[alarm_din_num-1];
          if(fix_alarm_timer_S!=NULL) *fix_alarm_timer_S=ALARM_BUTTON_FIX_TIME_S+GetSecondsFromStartup();
        }
        else
        {
          //none
        }
      }
      else
      {
        System.signal_state.alarm_button=0;
      }
    }
    else
    {
      System.signal_state.alarm_button=0;
    }
  }
#endif //defined(ALARM_BUTTON_PRESENT)

#if defined(TAMPERS_PRESENT)
  void CSignal_manager::Tamper_handler(void)
  {
    System.signal_state.tamper[0]=__GET_TAMPER1_STATE();
    System.signal_state.tamper[1]=__GET_TAMPER2_STATE();
  }
#endif //defined(TAMPERS_PRESENT)

#if defined(GSENSOR_PRESENT)
  uint8_t CSignal_manager::Accel_Init(uint8_t is_sleep_mode)
  {
    this->accel_wdt_timer1_mS=xTaskGetTickCount()+ACCEL_WDT_TIMEOUT_mS-1;
    this->accel_wdt_timer2_mS=xTaskGetTickCount()+ACCEL_WDT_TIMEOUT_mS-1;
    this->accel_wdt_image=0;

    LIS_IRQ_Init(); 		        // Настройка, запрещение, сброс EXTI
    __LIS_IRQ_NVIC_Init(); 	        // Настройка и запрещение
    NVIC_DisableIRQ(LIS_NVIC_IRQn);     //

    if(AccelSensors_Init(is_sleep_mode))
    {
      System.signal_state.accel_no_init=0;
      NVIC_EnableIRQ(LIS_NVIC_IRQn); 	// и его групповые прерывания
      return 1;
    }
    else
    {
      System.signal_state.accel_no_init=1;
      LOG("%s: accel init fail (mode: %s)\n", this->name, is_sleep_mode?"sleep":"normal");
      return 0;
    }
  }

  void CSignal_manager::Accel_DeInit(void)
  {
    System.signal_state.accel_no_init=0;
    NVIC_DisableIRQ(LIS_NVIC_IRQn);
    AccelSensors_DeInit();
  }
#endif //defined(GSENSOR_PRESENT)

  void CSignal_manager::EXTI_NVIC_Configuration(void)
  {
    EXTI_InitTypeDef EXTI_InitStructure;
    NVIC_InitTypeDef NVIC_InitStructure;

    /* Connect EXTI_Line22 to the RTC Wakeup event */
    EXTI_ClearITPendingBit(EXTI_Line22);
    EXTI_InitStructure.EXTI_Line = EXTI_Line22;
    EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
    EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Rising;
    EXTI_InitStructure.EXTI_LineCmd = ENABLE;
    EXTI_Init(&EXTI_InitStructure);

    /* Enable the RTC Wakeup Interrupt */
    NVIC_InitStructure.NVIC_IRQChannel = RTC_WKUP_IRQn;
    NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
    NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
    NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
    NVIC_Init(&NVIC_InitStructure);
  }

  //если вышли из этой фунуции - загрузчик не трогали
  void CSignal_manager::Boot_upd_handler(void)
  {
    //не обновляем бутлоадер
  }

  boot_status CSignal_manager::Bootloader_update(const char* const f_name, uint8_t* is_flashed)
  {
    *is_flashed=0;
    return RES_IMAGE_ERR;
  }

  //DataLength in worlds
  uint8_t CSignal_manager::Flash_write(uint32_t* flash_addr, uint32_t* data ,uint32_t len)
  {
    return 1;
  }

#if defined(BR_PRESENT)
  uint8_t CSignal_manager::br_handler(br_action_t action, uint8_t* is_interface_busy)
  {
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
    volatile uint8_t* interface_mutex;
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT) || (DEVICE_TYPE==DEVICE_TYPE_VEGA_MT_32K_LTE_V2) || (DEVICE_TYPE==DEVICE_TYPE_VEGA_MT_32K_LTE_V3))
    void(*SetParam)(int, int, const char *) = NULL;
    uint16_t(*TxRx)(const uint8_t*, uint16_t, uint8_t*, uint16_t) = NULL;
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))

    System.Grab(portMAX_DELAY);
    const TBr_settings settings = System.sensor_settings.br_settings;
    System.Release();

    if(is_interface_busy!=NULL) *is_interface_busy=0;

    if(settings.interface==BR_OFF) return 2; //блок расширения отключен

    else if(!(false
#if defined(RS232_PRESENT)
              || settings.interface==BR_RS232
#endif //RS232_PRESENT
#if defined(RS485_PRESENT)
                || settings.interface==BR_RS485
#endif //RS485_PRESENT
                  )) return 0; //неизвестный интерфейс блока расширения

#pragma diag_suppress=Pe111
    if(false){}
#pragma diag_default=Pe111
#if defined(RS232_PRESENT)
    else if(settings.interface==BR_RS232)
    {
      interface_mutex=&rs232_mutex;
      SetParam=RS232_SetParam;
      TxRx=RS232_TxRx;
    }
#endif //RS232_PRESENT
#if defined(RS485_PRESENT)
    else if(settings.interface==BR_RS485)
    {
      interface_mutex=&rs485_mutex;
      SetParam=RS485_SetParam;
      TxRx=RS485_TxRx;
    }
#endif //RS485_PRESENT
    else
    {
      return 0; //неизвестный интерфейс блока расширения
    }
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
    if(!interface_mutex_take(interface_mutex, 0)) {if(is_interface_busy!=NULL){*is_interface_busy=1;} return 1;}
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
    SetParam(115200, 150, "8N1");

    uint16_t* out_crc;
    uint16_t rx_len;
    br_header_t* out_header=(br_header_t*)&common_tx_buff[0];
    br_header_t* in_header=(br_header_t*)&common_rx_buff[0];
    uint8_t* tx_payload;
    uint8_t* rx_payload;

    if(action==GOTO_SLEEP)
    {
      //GOTOSLEEP_BR_MESS_TYPE
      out_header->sync_byte=SYNC_BYTE;
      out_header->len=0;
      out_header->type=GOTOSLEEP_BR_MESS_TYPE;
      out_crc=(uint16_t*)&common_tx_buff[BR_PACKET_CRC_OFFSET(out_header->len)];
      *out_crc=crc16_fast(&common_tx_buff[0], BR_PACKET_CRC_CALC_LEN(out_header->len));

      rx_len=TxRx(common_tx_buff, BR_PACKET_LEN(out_header->len), common_rx_buff, GOTOSLEEP_BR_ACK_MESS_TYPE_LEN);

      if(rx_len==GOTOSLEEP_BR_ACK_MESS_TYPE_LEN && rx_len==BR_PACKET_LEN(in_header->len) && in_header->sync_byte==SYNC_BYTE && in_header->type==GOTOSLEEP_BR_ACK_MESS_TYPE \
        && \
          *(uint16_t*)&common_rx_buff[BR_PACKET_CRC_CALC_LEN(in_header->len)]==crc16_fast(&common_rx_buff[0], BR_PACKET_CRC_CALC_LEN(in_header->len)))
      {
        //next
      }
      else
      {
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
        interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
        return 0;
      }
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
        interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
      return 1;
    }
    else if(action==POLL_DATA)
    {
      //GET_DEV_TYPE_BR_MESS_TYPE
      out_header->sync_byte=SYNC_BYTE;
      out_header->len=0;
      out_header->type=GET_DEV_TYPE_BR_MESS_TYPE;
      out_crc=(uint16_t*)&common_tx_buff[BR_PACKET_CRC_OFFSET(out_header->len)];
      *out_crc=crc16_fast(&common_tx_buff[0], BR_PACKET_CRC_CALC_LEN(out_header->len));

      rx_len=TxRx(common_tx_buff, BR_PACKET_LEN(out_header->len), common_rx_buff, GET_DEV_TYPE_BR_ACK_MESS_LEN);

      if(rx_len==GET_DEV_TYPE_BR_ACK_MESS_LEN && rx_len==BR_PACKET_LEN(in_header->len) && in_header->sync_byte==SYNC_BYTE && in_header->type==GET_DEV_TYPE_BR_ACK_MESS_TYPE \
        && \
          *(uint16_t*)&common_rx_buff[BR_PACKET_CRC_CALC_LEN(in_header->len)]==crc16_fast(&common_rx_buff[0], BR_PACKET_CRC_CALC_LEN(in_header->len)))
      {
        rx_payload=&common_rx_buff[BR_PACKET_PAYLOAD_OFFSET()];

        if(*rx_payload==DEVICE_TYPE_VEGA_BR_1)
        {
          //next
        }
        else
        {
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
        interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
          return 0;
        }
      }
      else
      {
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
        interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
        return 0;
      }

      //SET_EXT_MFI_SETTINGS_BR_MESS_TYPE
      out_header->sync_byte=SYNC_BYTE;
      out_header->len=0;
      tx_payload=&common_tx_buff[BR_PACKET_PAYLOAD_OFFSET()];

      System.Grab(portMAX_DELAY);
      for(uint8_t i=0; i<MAX_EXT_MF_INPUTS; i++)
      {
        memcpy(tx_payload+out_header->len, &System.sensor_settings.ext_mfi[i], sizeof(System.sensor_settings.ext_mfi[i]));
        out_header->len+=sizeof(System.sensor_settings.ext_mfi[i]);
      }
      System.Release();

      out_header->type=SET_EXT_MFI_SETTINGS_BR_MESS_TYPE;
      *(uint16_t*)&common_tx_buff[BR_PACKET_CRC_OFFSET(out_header->len)]=crc16_fast(&common_tx_buff[0], BR_PACKET_CRC_CALC_LEN(out_header->len));

      rx_len=TxRx(common_tx_buff, BR_PACKET_LEN(out_header->len), common_rx_buff, SET_EXT_MFI_SETTINGS_BR_ACK_MESS_LEN);

      if(rx_len==SET_EXT_MFI_SETTINGS_BR_ACK_MESS_LEN && rx_len==BR_PACKET_LEN(in_header->len) && in_header->sync_byte==SYNC_BYTE && in_header->type==SET_EXT_MFI_SETTINGS_BR_ACK_MESS_TYPE \
        && \
          *(uint16_t*)&common_rx_buff[BR_PACKET_CRC_CALC_LEN(in_header->len)]==crc16_fast(&common_rx_buff[0], BR_PACKET_CRC_CALC_LEN(in_header->len)))
      {
        //next
      }
      else
      {
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
        interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
        return 0;
      }

      //SET_EXT_OUTPUTS_STATE_BR_MESS_TYPE
      out_header->sync_byte=SYNC_BYTE;
      out_header->len=0;
      out_header->type=SET_EXT_OUTPUTS_STATE_BR_MESS_TYPE;
      tx_payload=&common_tx_buff[BR_PACKET_PAYLOAD_OFFSET()];

      System.Grab(portMAX_DELAY);
      for(uint8_t i=0; i<MAX_EXT_DIGITAL_OUTPUTS; i++)
      {
        memcpy(tx_payload+out_header->len, &System.signal_state.ext_digital_output[i], sizeof(System.signal_state.ext_digital_output[i]));
        out_header->len+=sizeof(System.signal_state.ext_digital_output[i]);
      }
      System.Release();

      *(uint16_t*)&common_tx_buff[BR_PACKET_CRC_OFFSET(out_header->len)]=crc16_fast(&common_tx_buff[0], BR_PACKET_CRC_CALC_LEN(out_header->len));

      rx_len=TxRx(common_tx_buff, BR_PACKET_LEN(out_header->len), common_rx_buff, SET_EXT_MFI_SETTINGS_BR_ACK_MESS_LEN);

      if(rx_len==SET_EXT_OUTPUTS_STATE_BR_ACK_MESS_LEN && rx_len==BR_PACKET_LEN(in_header->len) && in_header->sync_byte==SYNC_BYTE && in_header->type==SET_EXT_OUTPUTS_STATE_BR_ACK_MESS_TYPE \
        && \
          *(uint16_t*)&common_rx_buff[BR_PACKET_CRC_CALC_LEN(in_header->len)]==crc16_fast(&common_rx_buff[0], BR_PACKET_CRC_CALC_LEN(in_header->len)))
      {
        //next
      }
      else
      {
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
        interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
        return 0;
      }

      //GET_EXT_MFI_FULL_STATE_BR_MESS_TYPE
      out_header->sync_byte=SYNC_BYTE;
      out_header->len=0;
      out_header->type=GET_EXT_MFI_FULL_STATE_BR_MESS_TYPE;
      *(uint16_t*)&common_tx_buff[BR_PACKET_CRC_OFFSET(out_header->len)]=crc16_fast(&common_tx_buff[0], BR_PACKET_CRC_CALC_LEN(out_header->len));

      rx_len=TxRx(common_tx_buff, BR_PACKET_LEN(out_header->len), common_rx_buff, GET_EXT_MFI_FULL_STATE_BR_ACK_MESS_LEN);

      if(rx_len==GET_EXT_MFI_FULL_STATE_BR_ACK_MESS_LEN && rx_len==BR_PACKET_LEN(in_header->len) && in_header->sync_byte==SYNC_BYTE && in_header->type==GET_EXT_MFI_FULL_STATE_BR_ACK_MESS_TYPE \
        && \
          *(uint16_t*)&common_rx_buff[BR_PACKET_CRC_CALC_LEN(in_header->len)]==crc16_fast(&common_rx_buff[0], BR_PACKET_CRC_CALC_LEN(in_header->len)))
      {
        rx_payload=&common_rx_buff[BR_PACKET_PAYLOAD_OFFSET()];

        System.Grab(portMAX_DELAY);
        for(uint8_t i=0; i<MAX_EXT_MF_INPUTS; i++)
        {
          memcpy(&System.signal_state.ext_digital_input[i], rx_payload, sizeof(System.signal_state.ext_digital_input[i]));
          rx_payload+=sizeof(System.signal_state.ext_digital_input[i]);
          memcpy(&System.signal_state.ext_analog_input[i], rx_payload, sizeof(System.signal_state.ext_analog_input[i]));
          rx_payload+=sizeof(System.signal_state.ext_analog_input[i]);
          memcpy(&System.signal_state.ext_freq_input[i], rx_payload, sizeof(System.signal_state.ext_freq_input[i]));
          rx_payload+=sizeof(System.signal_state.ext_freq_input[i]);
          //memcpy(&System.signal_state.ext_pulse_input[i], rx_payload, sizeof(System.signal_state.ext_pulse_input[i]));
          System.signal_state.ext_pulse_input[i]+=*(uint32_t*)rx_payload;
          rx_payload+=sizeof(System.signal_state.ext_pulse_input[i]);
        }
        System.Release();
        //next
      }
      else
      {
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
        interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
        return 0;
      }

      //poll ok
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
      interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
      return 1;
    }
    else if(action==WAKEUP)
    {
      common_tx_buff[0]=0xAA;

      rx_len=TxRx(common_tx_buff, 1, common_rx_buff, 0);

      //vTaskDelay(200);
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
      interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
      return 1;
    }
    else
    {
      //unknown action
#if (defined(RS232_PRESENT) || defined(RS485_PRESENT))
      interface_mutex_give(interface_mutex);
#endif //(defined(RS232_PRESENT) || defined(RS485_PRESENT))
      return 0;
    }
  }
#endif //BR_PRESENT

  /* C code---------------------------------------------------------------------*/
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
  uint16_t get_mfi_adc_rav_val(uint8_t idx)
  {
    return adc_vals.adc_mfi_vals[idx];
  }
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

  //- Функция запуска Signal менеджера
  void Start_signal_manager(void* argument)
  {
#if (defined(CAMERA_PRESENT) || defined(FRIDGE_PRESENT) || defined(PROD_TESTING_PRESENT))
#error
#endif //(defined(CAMERA_PRESENT) || defined(FRIDGE_PRESENT) || defined(PROD_TESTING_PRESENT))

#if defined(VEGA_CAN_PRESENT)
    subtask_init(&can_subtask_ctx);
#endif //VEGA_CAN_PRESENT

#if defined(INTERNAL_NRF_PRESENT)
    subtask_init(&nrf_subtask_ctx);
#endif //INTERNAL_NRF_PRESENT

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
    subtask_init(&ble_subtask_ctx);
#endif //EXTERNAL_BLE_BOARD_PRESENT

    Signal_manager.Init();
    Signal_manager.Main_cycle();
    Signal_manager.Deinit();
    while(1);
  }


#ifdef __cplusplus
  extern "C" {
#endif
  void RTC_WKUP_IRQHandler(void)
  {
    if(RTC_GetITStatus(RTC_IT_WUT) != RESET)
    {
      RTC_ClearITPendingBit(RTC_IT_WUT);
      EXTI_ClearITPendingBit(EXTI_Line22);
    }
  }
#ifdef __cplusplus
  }
#endif
