/**
******************************************************************************
* File Name          : signal_manager.h
* Description        : Менеджер входных и выходных сигналов
*                      
*                      
******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef SIGNAL_MANAGER_H
#define SIGNAL_MANAGER_H

/* Includes ------------------------------------------------------------------*/
#include <stdint.h>
#include "signal_manager_config.h"
#include "manager.h"
#include "Black_box/Black_box.h"
#include "System\system.h"
#include "boot_config.h"

/* Defines ------------------------------------------------------------------*/
//- Системные настройки менеджера
#define SIGNAL_MANAGER_NAME      "signal_man" // Имя менеджера
#define SIGNAL_MANAGER_INPUT_QUEUE_SIZE   3 // Количество элементов во входящей очереди сообщений
#define SIGNAL_MANAGER_PRIORITY           1 // Приоритет задачи менеджера
#define SIGNAL_MANAGER_STACK_SIZE       256 // Размер стэка менеджера в словах

#define SIGNAL_MANAGER_CAN_CMD_QUEUE_SIZE 4

#define BOOT_FILE_HEADER_LEN            32 //iv+crc32+fw_len+fw_ver

#if defined(DUTS_PRESENT) 
#define DUT_SERIAL_BAUDRATE_DEFAULT	19200	// скорость передачи по умолчанию
#define DUT_SERIAL_TIMEOUT_DEFAULT	50 	// ожидание ответа по умолчанию
#define DUT_BIN_START_BYTE_CMD	        0x31
#define DUT_BIN_START_BYTE_ANSW	        0x3E
#define DUT_BIN_CMD_GET_DATA	        0x06
#endif //DUTS_PRESENT

#define ACCEL_WDT_TIMEOUT_mS            4000//
#define ALARM_BUTTON_FIX_TIME_S         5

typedef enum
{
  GOTO_SLEEP=0,
  POLL_DATA,
  WAKEUP
}br_action_t;

/* C++ code-------------------------------------------------------------------*/
class CSignal_manager: public CManager
{
public:
  // Конструктор, определяется имя менеджера, передается коллбэк на таймер, указатель на исходящую очередь, интерфейс обмена данными с железом
  CSignal_manager(const char* mngr_name, QueueHandle_t* output_queue, uint8_t input_queue_size, uint8_t priority, TaskFunction_t task_starter, uint16_t stack_size);  
  // Инициализация менеджера, реализует наследник
  void Init();
  // Деинициализация менеджера, реализует наследник
  void Deinit(void); 
  // Главный цикл менеджера
  void Main_cycle(void);
    
#if defined(VEGA_CAN_PRESENT)
  QueueHandle_t can_commands_queue;
#endif //VEGA_CAN_PRESENT
  
  private:
  // Имя менеджера
  const char* name;
  void EXTI_NVIC_Configuration(void);
  void Adc_Init(void);
  void Adc_DeInit(void);
  void Gpio_Init(void);
  void Gpio_DeInit(void);
  
#if defined(GSENSOR_PRESENT)
  uint8_t Accel_Init(uint8_t is_sleep_mode);
  void Accel_DeInit(void);
  uint32_t accel_wdt_image;
  uint32_t accel_wdt_timer1_mS;
  uint32_t accel_wdt_timer2_mS;
#endif //GSENSOR_PRESENT
  
#if defined(ALARM_BUTTON_PRESENT)
  void Alarm_handler(uint32_t* alarm_timer_S);
#endif //ALARM_BUTTON_PRESENT
  
#if defined(TAMPERS_PRESENT)
  void Tamper_handler(void);
#endif //TAMPERS_PRESENT
  
  void Ign_handler(void);
#if defined(INTERNAL_AKB_PRESENT)
  void LiPo_handler(bool is_sleep);
#endif //INTERNAL_AKB_PRESENT
  void Adc_handler(void);
  
#if defined(BR_PRESENT)
  uint8_t br_handler(br_action_t action, uint8_t* is_interface_busy);
#endif //BR_PRESENT
  
  // Обновление загрузчика
  void Boot_upd_handler(void);
  boot_status Bootloader_update(const char* const f_name, uint8_t* is_flashed);
  uint8_t Flash_write(uint32_t* flash_addr, uint32_t* data ,uint32_t len);
};

/* C code---------------------------------------------------------------------*/
// Функция создания Signal менеджера
extern void Create_signal_manager(void);
// Функция запуска Signal менеджера
extern void Start_signal_manager(void* argument);
// Для обеспечения видимости другими менеджерами
extern CSignal_manager Signal_manager;
//
extern CBlack_box Black_box; 
#endif //SIGNAL_MANAGER_H
