/**
******************************************************************************
* File Name          : external_ble_subtask.cpp
* Description        :
*
*
******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "signal_manager/signal_manager.h"
#include "signal_manager/signalman_ble_subtask.h"
#include "ble_gpio_driver.h"
#include "rng_driver.h"
#include "external_uart.h"

#include "debug_port.h"
#include "leds_driver.h"
#include "ble_nfc_ext_board_lib/ble_nfc_ext_board_master.h"
#include "server_manager/server_manager.h"

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
/* C code---------------------------------------------------------------------*/
static const char* const dbg_name="ble_stask";

//#define USE_SAME_HEAP_MEM_AND_TX_MEM //режим, при котором при вызове socket данные уже сформированы в tx_mem и готовы для отправки
                                       //при промежуточном шифровании не может быть использовано

#if defined(USE_SAME_HEAP_MEM_AND_TX_MEM) && defined(VEGA_ESECURE_PRESENT)
#error Не поддержано
#endif //defined(USE_SAME_HEAP_MEM_AND_TX_MEM) && defined(VEGA_ESECURE_PRESENT)

static conn_ctx_t ble_ctx;
//static uint8_t ble_heap_mem[EXTERNAL_UART_RX_MEM_SIZE-1];
static uint8_t ble_heap_mem[2048];
static uint8_t ble_rx_mem[512];
#if !defined(USE_SAME_HEAP_MEM_AND_TX_MEM)
static uint8_t ble_tx_mem[512];
#endif //!defined(USE_SAME_HEAP_MEM_AND_TX_MEM)

//static_assert(!(sizeof(ble_heap_mem)>=EXTERNAL_UART_RX_MEM_SIZE), "ble_heap_mem size err");

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
static const uint8_t crypt_key[32] =
{
  0xfd, 0x73, 0x70, 0xd1, 0x69, 0xfd, 0xfc, 0x16, 0x14, 0x01, 0xe7, 0x7f, 0x1f, 0x1b, 0x23, 0xee,
  0x1d, 0x42, 0xac, 0xe2, 0x06, 0xac, 0x83, 0x2e, 0x2b, 0xc8, 0xff, 0x77, 0xab, 0x4b, 0xe0, 0xee
};
#else
#error
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)

void external_ble_subtask_onetime_init(void)
{
  ble_gpio_init();
}

#ifdef __cplusplus
extern "C" {
#endif

static volatile struct
{
  uint8_t* ptr;
  uint16_t offset;
  uint16_t max_size;
}rx_it={.ptr=NULL};

static bool rx_it_callback(cbuff_t* const cbuff)
{
  if(true)
  {
    if(rx_it.ptr==NULL)
    {
      read_byte_from_cbuff(cbuff);
      return false;
    }

    if(rx_it.offset>=rx_it.max_size)
    {
      read_byte_from_cbuff(cbuff);
      return true;
    }

    rx_it.ptr[rx_it.offset++]=read_byte_from_cbuff(cbuff);

    if(rx_it.offset>=BLE_NFC_EXT_BOARD_PACKET_LEN(0))
    {
      const ext_board_mess_header_t* const header=(const ext_board_mess_header_t*)rx_it.ptr;

      if(BLE_NFC_EXT_BOARD_PACKET_LEN(header->len)==rx_it.offset && header->sync_byte==BLE_BOARD_SYNC_BYTE)
      {
        rx_it.ptr=NULL;
        return true;
      }
    }
  }
  else
  {
    static ext_board_mess_header_t header;
    const uint16_t len=filled_cbuff_len(cbuff);

    if(len>=BLE_NFC_EXT_BOARD_PACKET_LEN(0))
    {
      if(len==BLE_NFC_EXT_BOARD_PACKET_LEN(0))
      {
        peek_from_cbuff(cbuff, (uint8_t* const)&header, sizeof(header));
      }

      if(BLE_NFC_EXT_BOARD_PACKET_LEN(header.len)==len && header.sync_byte==BLE_BOARD_SYNC_BYTE)
      {
        return true;
      }
    }
  }

  return false;
}

static int16_t interface_txrx(const void* const tx, const uint16_t tx_size, void* const rx, const uint16_t rx_size, const uint32_t timeout)
{
  if(true)
  {
    external_uart_flush_rx(EXTERNAL_UART_CTX1);

    rx_it.ptr=NULL;
    rx_it.offset=0;
    rx_it.max_size=rx_size;
    rx_it.ptr=(uint8_t*)rx;

    external_uart_write(EXTERNAL_UART_CTX1, tx, tx_size);

    external_uart_wait_sync(EXTERNAL_UART_CTX1, timeout);

    rx_it.ptr=NULL;

    return (int16_t)rx_it.offset;
  }
  else
  {
    external_uart_flush_rx(EXTERNAL_UART_CTX1);

    external_uart_write(EXTERNAL_UART_CTX1, tx, tx_size);

    external_uart_wait_sync(EXTERNAL_UART_CTX1, timeout);

    return external_uart_read(EXTERNAL_UART_CTX1, rx, rx_size);
  }
}

#ifdef __cplusplus
}
#endif

static const char* got_firmware_file=NULL;

void external_ble_fw_update_task(const char* const fname)
{
  if(got_firmware_file==NULL)
  {
    got_firmware_file=fname;
  }
}

static const uint8_t BLE_FIRS_CONN_ID=SERVER_BLE_CONN_ID;

static int16_t _socket_write(uint8_t conn_id, uint8_t* buff, uint16_t len)
{
  (void)buff;

  if(conn_id<BLE_FIRS_CONN_ID) return -1;

  if(!len) return (int16_t)len;

  const ext_board_ble_send_data_aux_t send_data=
  {
    .conn_id=conn_id-BLE_FIRS_CONN_ID,
    .length=len,
#if defined(USE_SAME_HEAP_MEM_AND_TX_MEM)
    .data=NULL,//данные уже скопированы
#else
    .data=buff,
#endif //defined(USE_SAME_HEAP_MEM_AND_TX_MEM)
  };

  const ext_board_ble_ans_send_data_t* ans_send_data;

  if(cmd_ble_send_data(send_data, &ans_send_data)>0 \
    && send_data.conn_id==ans_send_data->conn_id && send_data.length==ans_send_data->length \
      && ans_send_data->result==CMD_SUCCESS)
  {
    return (int16_t)ans_send_data->length;
  }

  return -1;
}

static int16_t _socket_read(uint8_t conn_id, uint8_t* buff, uint16_t len)
{
  if(conn_id<BLE_FIRS_CONN_ID) return -1;

  if(!len) return (int16_t)len;

  if(len>ext_board_get_ble_max_receive_data_len()) len=ext_board_get_ble_max_receive_data_len();

  const ext_board_ble_receive_data_t receive_data=
  {
    .conn_id=conn_id-BLE_FIRS_CONN_ID,
    .length=len,
  };

  const ext_board_ble_ans_receive_data_aux_t* ans_receive_data;

  if(cmd_ble_receive_data(receive_data, &ans_receive_data)>0 \
    && receive_data.conn_id==ans_receive_data->conn_id && receive_data.length>=ans_receive_data->length \
      && ans_receive_data->result==CMD_SUCCESS)
  {
    memcpy(buff, ans_receive_data->data, ans_receive_data->length);
    return (int16_t)ans_receive_data->length;
  }

  return -1;
}

static int16_t _socket_open(uint8_t conn_id, const char* addr)
{
  (void)addr;

  if(conn_id<BLE_FIRS_CONN_ID) return -1;

  const ext_board_ble_connection_id_t req =
  {
    .conn_id=conn_id-BLE_FIRS_CONN_ID,
  };

  const ext_board_ble_ans_conn_cmd_t* ans;

  if(cmd_ble_connect(req, &ans)>0 && req.conn_id==ans->conn_id)
  {
    switch(ans->result)
    {
    case CMD_ERROR:             return -1;
    case CMD_INPROGRESS:        return 0;
    case CMD_SUCCESS:           return 0;
    default:                    return -1;
    }
  }

  return -1;
}

static cstate_t _socket_state(uint8_t conn_id)
{
  if(conn_id<BLE_FIRS_CONN_ID) return INTERNAL_ERR_CSTATE;

  const ext_board_ble_connection_id_t req =
  {
    .conn_id=conn_id-BLE_FIRS_CONN_ID,
  };

  const ext_board_ble_info_t* ans;

  if(cmd_ble_connection_status(req, &ans)>0 && req.conn_id==ans->conn_id)
  {
    switch(ans->state)
    {
    case BLE_STATE_DISCONNECTED:                        return DOWN_CSTATE;
    case BLE_STATE_SHUTDOWN:                            return DOWN_CSTATE;
    case BLE_STATE_CONNECTING:                          return WAIT_CONNECTED_CSTATE;

    case BLE_STATE_CONNECTED_NO_ENC:                    if(System.ext_board_ble_settings.ble_min_security_conn_level<=0) return CONNECTED_CSTATE;
                                                        else                                                             return WAIT_CONNECTED_CSTATE;

    case BLE_STATE_CONNECTED_ENC_NO_MITM:               if(System.ext_board_ble_settings.ble_min_security_conn_level<=1) return CONNECTED_CSTATE;
                                                        else                                                             return WAIT_CONNECTED_CSTATE;

    case BLE_STATE_CONNECTED_ENC_W_MITM:                if(System.ext_board_ble_settings.ble_min_security_conn_level<=2) return CONNECTED_CSTATE;
                                                        else                                                             return WAIT_CONNECTED_CSTATE;

    case BLE_STATE_CONNECTED_LESC_ENC_WITH_MITM:        return CONNECTED_CSTATE;

    case BLE_STATE_STATE_ERROR:                         return INTERNAL_ERR_CSTATE;
    default:                                            return INTERNAL_ERR_CSTATE;
    }
  }

  return INTERNAL_ERR_CSTATE;
}

static int16_t _socket_close(uint8_t conn_id)
{
  if(conn_id<BLE_FIRS_CONN_ID) return -1;

  const ext_board_ble_connection_id_t req =
  {
    .conn_id=conn_id-BLE_FIRS_CONN_ID,
  };

  const ext_board_ble_ans_conn_cmd_t* ans;

  if(cmd_ble_disconnect(req, &ans)>0 && req.conn_id==ans->conn_id)
  {
    switch(ans->result)
    {
    case CMD_SUCCESS:   return 0;
    default:            return -1;
    }
  }

  return -1;
}

static int16_t _socket_get_nack_len(uint8_t conn_id, uint32_t* naclen)
{
  (void)conn_id;
  *naclen=0;

  return 0;
}

static const common_prot_const_t protocols_common_const =
{
  .get_device_imei=get_protocols_common_const_pointer()->get_device_imei,
  .get_unix_time=get_protocols_common_const_pointer()->get_unix_time,
  .socket_write=_socket_write,
  .socket_read=_socket_read,
  .socket_open=_socket_open,
  .socket_state=_socket_state,
  .socket_close=_socket_close,
  .socket_get_nack_len=_socket_get_nack_len,
  .grab_mess=get_protocols_common_const_pointer()->grab_mess,
  .release_mess=get_protocols_common_const_pointer()->release_mess,
  .get_mess_count=get_protocols_common_const_pointer()->get_mess_count,
  .read_message=get_protocols_common_const_pointer()->read_message,
  .delete_messages=get_protocols_common_const_pointer()->delete_messages,
  .get_message_param_name_type=get_protocols_common_const_pointer()->get_message_param_name_type,
  //.find_ad_sens_in_bbox_mess=get_protocols_common_const_pointer()->find_ad_sens_in_bbox_mess,
  .photo_inf=get_protocols_common_const_pointer()->photo_inf,
  .photo_read_chunk=get_protocols_common_const_pointer()->photo_read_chunk,
  .photo_delete=get_protocols_common_const_pointer()->photo_delete,
  .file_write_chunk=get_protocols_common_const_pointer()->file_write_chunk,
  .file_downloaded_signal=get_protocols_common_const_pointer()->file_downloaded_signal,
  .file_clean=get_protocols_common_const_pointer()->file_clean,
  .file_read_chunk=get_protocols_common_const_pointer()->file_read_chunk,
  .common_cmd_callback=get_protocols_common_const_pointer()->common_cmd_callback,
#if defined(ITF_TRANSPORT_PRESENT)
  .itf_tx_rx_callback=get_protocols_common_const_pointer()->itf_tx_rx_callback,
#endif //defined(ITF_TRANSPORT_PRESENT)
#if defined(CAN_VEGA_FWD_PRESENT)
  .is_can_fwd_enabled_callback=get_protocols_common_const_pointer()->is_can_fwd_enabled_callback,
#endif //defined(CAN_VEGA_FWD_PRESENT)
#if defined(FASTDATA_PRESENT)
  .is_fastdata_fwd_enabled_callback=get_protocols_common_const_pointer()->is_fastdata_fwd_enabled_callback,
#endif //defined(FASTDATA_PRESENT)
#if defined(CHANGE_SIM_BY_NO_VALID_VEGA_PACKET_PRESENT)
  .got_valid_vega_packet=get_protocols_common_const_pointer()->got_valid_vega_packet,
#endif //defined(CHANGE_SIM_BY_NO_VALID_VEGA_PACKET_PRESENT)
};

void external_ble_subtask_init(void)
{
  rng_init();
  ble_power(true);

  static uint8_t __rx_heap[4];
  external_uart_init(EXTERNAL_UART_CTX1, 115200, __rx_heap, sizeof(__rx_heap), rx_it_callback);

  ext_board_init(ble_heap_mem, sizeof(ble_heap_mem), interface_txrx, randmemset, crypt_key);

  ble_ctx.protocol_type=VEGA_PROTOCOL_TYPE;
  ble_ctx.rx_buff=ble_rx_mem;
  ble_ctx.rx_buff_size=sizeof(ble_rx_mem);
  ble_ctx.server_addr="ble";
  ble_ctx.server_ctx_name="conn_ctx_ble";
  ble_ctx.keepalive_timeoutS=60;//???
  ble_ctx.connection_periodM=0;
  ble_ctx.server_ctx_state=DO_CONNECTING;
  ble_ctx.conn_ctx_id=BLE_FIRS_CONN_ID;
  ble_ctx.conn_fail_counter=0;
  ble_ctx.common_const=&protocols_common_const;
}

void external_ble_subtask_deinit(void)
{
  rng_deinit();
  ble_power(false);
  external_uart_deinit(EXTERNAL_UART_CTX1);
}

static bool ext_board_firmware_file_transfer(const char* fname, const char** verbose)
{
  const char ref_file_marker[]="NRF_BLE";

  uint8_t* buff;
  uint16_t buff_chunk_size;

  ext_board_get_pointer_for_cmd_firmware_file_payload(&buff, &buff_chunk_size);

  if(buff_chunk_size>(512-BLE_NFC_EXT_BOARD_PACKET_LEN(sizeof(ext_board_mess_firmware_upd_cmd_t)))) buff_chunk_size=512-BLE_NFC_EXT_BOARD_PACKET_LEN(sizeof(ext_board_mess_firmware_upd_cmd_t)); //ограничено размером RX буфера платы

  Black_box.Grab(portMAX_DELAY);

  uint8_t is_opened;

  if(Black_box.fopened(fname, &is_opened) != BB_OPERATION_OK)
  {
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="fopened err";
    return false;
  }

  if(is_opened)
  {
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="file already opened err";
    return false;
  }

  uint32_t fsize;

  if(Black_box.fsize(fname, &fsize) != BB_OPERATION_OK)
  {
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="fsize err";
    return false;
  }

  if(fsize<1024)
  {
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="fsize err";
    return false;
  }

  if(Black_box.fopen(fname) != BB_OPERATION_OK)
  {
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="fopen err";
    return false;
  }

  if(Black_box.fread(fname, 0, buff, 16) != BB_OPERATION_OK)
  {
    Black_box.fclose(fname);
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="fread err";
    return false;
  }

  if(memcmp(&buff[0], ref_file_marker, sizeof(ref_file_marker))!=0)
  {
    Black_box.fclose(fname);
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="file marker err";
    return false;
  }

  uint32_t fsize_in_file;
  memcpy(&fsize_in_file, &buff[8], sizeof(fsize_in_file));

  if(fsize_in_file!=fsize-16)
  {
    Black_box.fclose(fname);
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="wrong file size err";
    return false;
  }

  Black_box.Release();

  const ext_board_file_load_result_t* ans;
  ext_board_mess_firmware_upd_cmd_aux_t req;

  for(uint32_t offset=16; offset<fsize; offset+=buff_chunk_size)
  {
    uint16_t rlen;

    if((offset+buff_chunk_size)>fsize) rlen=fsize-offset;
    else                               rlen=buff_chunk_size;

    Black_box.Grab(portMAX_DELAY);
    if(Black_box.fread(fname, offset, buff, rlen) != BB_OPERATION_OK)
    {
      Black_box.fclose(fname);
      Black_box.Release();
      if(verbose!=NULL) verbose[0]="fread err";
      return false;
    }
    Black_box.Release();

    req.offset=offset-16;
    req.length=rlen;
    req.data=NULL;//данные уже скопированы

    if(cmd_firmware_file(req, &ans)<=0 || ans[0]!=FILE_LOAD_IN_PROGRESS)
    {
      Black_box.Grab(portMAX_DELAY);
      Black_box.fclose(fname);
      Black_box.Release();
      if(verbose!=NULL) verbose[0]="file transfer err";
      return false;
    }

    System.ext_board_ble_state.firmware_download_pct=(uint8_t)(((offset+rlen)*100)/fsize);
    __PRINTF("%s: file transfer progress: %hhu%%\n", dbg_name, System.ext_board_ble_state.firmware_download_pct);
  }

  req.offset=0;
  req.length=0;
  req.data=NULL;

  if(cmd_firmware_file(req, &ans)<=0 || ans[0]!=FILE_LOAD_DONE)
  {
    Black_box.Grab(portMAX_DELAY);
    Black_box.fclose(fname);
    Black_box.Release();
    if(verbose!=NULL) verbose[0]="file transfer err";
    return false;
  }

  Black_box.Grab(portMAX_DELAY);
  Black_box.fclose(fname);
  Black_box.Release();

  if(verbose!=NULL) verbose[0]="firmware transfer done";

  return true;
}

static ext_board_dfu_result_t ext_board_wait_update_result(uint32_t timeout)
{
  const ext_board_dfu_result_t* ans;

  timeout+=xTaskGetTickCount();

  for(;;)
  {
    if(cmd_get_dfu_result(&ans)>0)
    {
      if(ans[0]!=DFU_RESULT_NONE && ans[0]!=DFU_RESULT_GOTO_BOOTLOADER_FOR_FLASHING)
      {
        return ans[0];
      }
    }

    if(timeAfter(xTaskGetTickCount(), timeout))
    {
      return DFU_RESULT_TIMEOUT_ERROR;
    }

    vTaskDelay(200);
  }
}

void external_ble_subtask_main_cycle(void)
{
  static uint32_t slow_timer=xTaskGetTickCount()-1;

  if(timeAfter(xTaskGetTickCount(), slow_timer))
  {
    if(false)
    {
      const ext_board_fw_vers_t* ans;

      if(cmd_get_fw_vers(&ans)>0)
      {
      }
    }

    if(true)
    {
      const char* ans1;
      const char* ans2;
      if(__cmd_get_fw_vers_verbose(NULL, &ans1, &ans2)>0)
      {
        System.ext_board_ble_state.inteface_state=1;

        if(strncmp(System.ext_board_ble_state.boot_vers, ans1, sizeof(System.ext_board_ble_state.boot_vers))!=0 \
          || strncmp(System.ext_board_ble_state.app_vers, ans2, sizeof(System.ext_board_ble_state.app_vers))!=0)
        {
          System.Grab();
          snprintf(System.ext_board_ble_state.boot_vers, sizeof(System.ext_board_ble_state.boot_vers), ans1);
          snprintf(System.ext_board_ble_state.app_vers, sizeof(System.ext_board_ble_state.app_vers), ans2);
          System.Release();

          LOG("%s: new fw info: %s/%s\n", dbg_name, ans1, ans2);
        }
      }
      else
      {
        System.ext_board_ble_state.inteface_state=0;
      }
    }

    if(true)
    {
      const ext_board_ble_init_t* ans;

      if(get_ble_status(&ans)>0)
      {
        if(ans[0]!=BLE_INIT)
        {
          cmd_ble_init(BLE_INIT, NULL);
        }
      }
    }

    if(true)
    {
      const ext_board_ble_connection_count_t* ans;

      if(cmd_ble_connection_count(&ans)>0)
      {

      }
    }

    if(true)
    {
      const ext_board_ble_info_t* ans;

      ext_board_ble_connection_id_t req=
      {
        .conn_id=1,
      };

      if(cmd_ble_connection_status(req, &ans)>0)
      {
        if(memcmp(&System.ext_board_ble_state.ble_conn_status, ans, sizeof(System.ext_board_ble_state.ble_conn_status))!=0)
        {
          System.Grab();
          memcpy(&System.ext_board_ble_state.ble_conn_status, ans, sizeof(System.ext_board_ble_state.ble_conn_status));
          System.Release();
        }
      }
    }

    if(true)
    {
      const ext_board_ble_mac_t* ans;

      if(cmd_get_ble_board_mac(&ans)>0)
      {
        if(memcmp(System.ext_board_ble_state.board_mac.addr, ans->addr, sizeof(System.ext_board_ble_state.board_mac.addr))!=0)
        {
          System.Grab();
          memcpy(System.ext_board_ble_state.board_mac.addr, ans->addr, sizeof(System.ext_board_ble_state.board_mac.addr));
          System.Release();

          LOG("%s: dev mac: %02hhX:%02hhX:%02hhX:%02hhX:%02hhX:%02hhX\n", dbg_name, ans->addr[5], ans->addr[4], ans->addr[3], ans->addr[2], ans->addr[1], ans->addr[0]);
        }
      }
    }

#if (MAX_EXTERNAL_BOARD_BLE_BEACONS_INFO_COUNT > 0)
    static_assert(!(sizeof(System.ext_board_ble_state.beacons_info[0])>=MAX_SENSOR_R_SIZE),"beacons_info size err");

    if(true)
    {
      const ext_board_ans_ble_beacon_visible_aux_t* ans;

      if(cmd_ble_beacon_visible(&ans)>0)
      {
        System.Grab();
        memset(&System.ext_board_ble_state.beacons_info, 0, sizeof(System.ext_board_ble_state.beacons_info));

        const uint16_t total_beacon_inf_count=sizeof(System.ext_board_ble_state.beacons_info)/sizeof(System.ext_board_ble_state.beacons_info[0][0]);
        const uint8_t beacon_inf_cell_count=sizeof(System.ext_board_ble_state.beacons_info)/sizeof(System.ext_board_ble_state.beacons_info[0]);
        const uint8_t beacon_inf_count_in_cell=sizeof(System.ext_board_ble_state.beacons_info[0])/sizeof(System.ext_board_ble_state.beacons_info[0][0]);

        for(uint16_t total_counter=0, cell_counter=0; total_counter<ans->count; cell_counter++)
        {
          if(cell_counter>=beacon_inf_cell_count) goto exit;

          for(uint8_t i=0; i<beacon_inf_count_in_cell; i++)
          {
            if(total_counter>=ans->count || total_counter>=total_beacon_inf_count) goto exit;
            memcpy(&System.ext_board_ble_state.beacons_info[cell_counter][i], &ans->info[total_counter], sizeof(System.ext_board_ble_state.beacons_info[cell_counter][i]));
            total_counter++;
          }
        }
      exit:
        System.Release();
      }
    }
#endif //(MAX_EXTERNAL_BOARD_BLE_BEACONS_INFO_COUNT > 0)

    if(true)
    {
      const ext_board_ble_name_t* getname_ans;

      static_assert(!(sizeof(System.ext_board_ble_settings.ble_name)!=sizeof(getname_ans->name)), "wrong System.ext_board_ble_settings.ble_name size");

      bool is_valid=true;

      if(System.ext_board_ble_settings.ble_name[0]=='\0') is_valid=false;

      if(is_valid && cmd_get_ble_name(&getname_ans)>0)
      {
        if(strncmp(getname_ans->name, System.ext_board_ble_settings.ble_name, sizeof(getname_ans->name))!=0)
        {
          bool is_update_setting=false;

          ext_board_ble_name_t ble_name;

          System.Grab();
          if(strncmp(getname_ans->name, System.ext_board_ble_settings.ble_name, sizeof(getname_ans->name))!=0)
          {
            snprintf(ble_name.name, sizeof(ble_name.name), System.ext_board_ble_settings.ble_name);
            is_update_setting=true;
          }
          System.Release();

          if(ble_name.name[0]=='\0') is_update_setting=false;

          if(is_update_setting)
          {
            const ext_board_cmd_result_t* setname_ans;
            const char* const req = ble_name.name;

            cmd_set_ble_name(req, &setname_ans);
          }
        }
      }
    }

    if(true)
    {
      const ext_board_ble_passkey_t* getkey_ans;

      bool is_valid=true;

      //только 6 символов и символы 0...9
      for(uint8_t i=0; i<sizeof(System.ext_board_ble_settings.ble_passkey)-1; i++)
      {
        if(System.ext_board_ble_settings.ble_passkey[i]<'0' || System.ext_board_ble_settings.ble_passkey[i]>'9') {is_valid=false; break;}
      }

      if(is_valid && cmd_get_ble_passkey(&getkey_ans)>0)
      {
        if(strncmp(getkey_ans->passkey, System.ext_board_ble_settings.ble_passkey, sizeof(getkey_ans->passkey))!=0)
        {
          bool is_update_setting=false;
          ext_board_ble_passkey_t ble_passkey;

          System.Grab();
          if(strncmp(getkey_ans->passkey, System.ext_board_ble_settings.ble_passkey, sizeof(getkey_ans->passkey))!=0)
          {
            memcpy(ble_passkey.passkey, System.ext_board_ble_settings.ble_passkey, sizeof(ble_passkey.passkey));
            is_update_setting=true;
          }
          System.Release();


          for(uint8_t i=0; i<sizeof(ble_passkey.passkey); i++)
          {
            if(ble_passkey.passkey[i]<'0' || ble_passkey.passkey[i]>'9') {is_update_setting=false; break;}
          }

          if(is_update_setting)
          {
            const ext_board_cmd_result_t* setkey_ans;
            const char* const req = ble_passkey.passkey;

            cmd_set_ble_pass_key(req, &setkey_ans);
          }
        }
      }
    }

    if(false)
    {
      const ext_board_cmd_result_t* ans;

      const char* req = "123987";

      cmd_set_ble_pass_key(req, &ans);
    }

    if(false)
    {
      static const uint8_t test_data[]="Hui1Hui2Hui3Hui4Hui5Hui6Hui7Hui8Hui9Hui10Hui11Hui12Hui13";

      const ext_board_ble_send_data_aux_t send_data=
      {
        .conn_id=0,
        .length=sizeof(test_data)-1,
        .data=test_data,
      };

      const ext_board_ble_ans_send_data_t* ans_send_data;

      cmd_ble_send_data(send_data, &ans_send_data);
    }

    if(false)
    {
      const ext_board_ble_ans_receive_data_aux_t* ans_receive_data;

      const ext_board_ble_receive_data_t receive_data=
      {
        .conn_id=0,
        .length=10,
      };

      cmd_ble_receive_data(receive_data, &ans_receive_data);
    }

    if(got_firmware_file!=NULL)
    {
      const char* verbose="";

      bool file_transfer_result=ext_board_firmware_file_transfer(got_firmware_file, &verbose);

      LOG("%s: firmware file transfer result: %s\n", dbg_name, verbose);

      if(file_transfer_result)
      {
        ext_board_dfu_result_t update_result=ext_board_wait_update_result(10000);

        if(update_result==DFU_RESULT_UPDATE_DONE)        verbose="done";
        else if(update_result==DFU_RESULT_TIMEOUT_ERROR) verbose="timeout err";
        else                                             verbose="fail";

        LOG("%s: firmware update result: %s(%hhu)\n", dbg_name, verbose, update_result);
      }

      got_firmware_file=NULL;
    }

    static volatile bool reboot=false;
    if(reboot)
    {
      reboot=false;
      cmd_reboot(NULL);
    }

    slow_timer=xTaskGetTickCount()+2000;
  }

  static bool is_init=false;

  if(!is_init)
  {
#if defined(VEGA_SECURE_PRESENT)
    static vega_secure_ctx_t ble_secure=
    {
      .chacha20_key=System.ext_board_ble_state.session_key,
    };

    ble_ctx.secure=&ble_secure;
#endif //defined(VEGA_SECURE_PRESENT)

#if defined(VEGA_ESECURE_PRESENT)
    static vega_esecure_ctx_t ble_esecure=
    {
      .chachapoly_key=System.ext_board_ble_state.session_key,
    };

    ble_ctx.esecure=&ble_esecure;
#endif //defined(VEGA_ESECURE_PRESENT)

    static const uint8_t use_pin=false;

    ble_ctx.protocol.vega.use_passwd_opt=&use_pin;
    ble_ctx.protocol.vega.passwd=System.security_settings.pin;
    ble_ctx.protocol.vega.bbox_receiver=0;
#if defined(USE_SAME_HEAP_MEM_AND_TX_MEM)
    ext_board_get_pointer_for_cmd_ble_send_data_payload(&ble_ctx.protocol.vega.tx_mem, &ble_ctx.protocol.vega.tx_mem_size);
#else
    ble_ctx.protocol.vega.tx_mem=ble_tx_mem;
    ble_ctx.protocol.vega.tx_mem_size=sizeof(ble_tx_mem);
#endif //defined(USE_SAME_HEAP_MEM_AND_TX_MEM)
    ble_ctx.protocol.vega.dev_type=DEVICE_TYPE;
    ble_ctx.protocol.vega.got_vega_packet_callback=get_got_vega_packet_callback_pointer();
    is_init=true;
  }

  server_conn_handler(&ble_ctx, dbg_name);

  vTaskDelay(20);
}

#endif //defined(EXTERNAL_BLE_BOARD_PRESENT)