/**
******************************************************************************
* File Name          : signalman_can_subtask.cpp
* Description        :
*
*
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "signal_manager/signal_manager.h"
#include "server_manager/server_manager.h"
#include "signal_manager/signalman_can_subtask.h"
#include "Soft_timers/Soft_timers.h"
#include "signal_manager/outputs_defines.h"
#include "signal_manager/prod_tests.h"
#include "utils/data_utils.h"
#include "cbuff/sfifo.h"
#include "mfi_driver.h"
#include "time/timer.h"
#include "rtc_driver.h"

#if defined(JQ6500_AUDUO_PRESENT)
#include "signal_manager/audio_manadge.h"
#endif //defined(JQ6500_AUDUO_PRESENT)

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2) || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3
#include "lin_uart.h"
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

#if defined(CAN_VEGA_FWD_PRESENT)
#include "serv_protocols_lib/can_vega_fwd.h"
#endif //defined(CAN_VEGA_FWD_PRESENT)

#if defined(FASTDATA_PRESENT)
#warning FASTDATA_PRESENT demo mode only!
#include "serv_protocols_lib/fast_data_upload.h"
#include "Accel/Accel_sensors.h"
#endif //defined(FASTDATA_PRESENT)

#include "debug_port.h"
#include "outputs_driver.h"
#include "leds_driver.h"

#define LEASE_STATE_BKP_DR                                      (RTC_BKP_DR9)
#define BLOCKS_STATE_BKP_DR                                     (RTC_BKP_DR6)
#define WIRED_BLOCK_BKP_MASK                                    (1<<0)
#define WIRELESS_BLOCK_BKP_MASK                                 (1<<1)
#define KEY_POWER_BKP_MASK                                      (1<<2)

#define CAN1_MAIN_RX_TIMER_ID                                   0
#define CAN2_MAIN_RX_TIMER_ID                                   1
#define CAN3_MAIN_RX_TIMER_ID                                   2
#define AUX_CAN_TIMER_1                                         3
#define AUX_CAN_TIMER_2                                         4
#define AUX_CAN_TIMER_3                                         5
#define AUX_CAN_TIMER_4                                         6
#define UDS_1_TIMER_ID                                          7
#define OBD_1_TIMER_ID                                          8
#define OBD_2_TIMER_ID                                          9
#define OBD_3_TIMER_ID                                          10
#define OBD_4_TIMER_ID                                          11
#define DRIFT_EVENT_TIMER_ID                                    12
#define LAUNCH_EVENT_TIMER_ID                                   13
#define DRIVER_DOOR_IGNORE_TIMER_ID                             14
#define UNAUTHORIZED_START_EVENT_TIMER_ID                       15
#define ANALOG_WINDOWS_CLOSING_TIMER_ID                         16
#define KEY_POWER_OFF_TIMER_ID                                  17
#define UDS_ANS_TIMER_ID                                        18
#define UDS_2_TIMER_ID                                          19
#define SETTINGS_CHECK_TIMER_ID                                 20
#define CAN_RX_FWD_TIMER_ID                                     21
#define OBD_ENG_ON_TIMER                                        22
#define ECO_ON_ENG_ON_TIMER                                     23
#define ENG_STOP_TIMER                                          24
#define FUEL_FILT_TIMER                                         25
#define DELAYED_WINDOWS_CLOSING_TIMER                           26
#define FASTDATA_TIMER                                          27
#define EMERG_LIGHT_FIX_TIMER                                   28
#define HOOD_UNLOCK_TIMER                                       29
#define LAST_CAN_TIMER_ID                                       HOOD_UNLOCK_TIMER

#define MAX_WARMING_DURATION_MINUTES                            240

/* C code---------------------------------------------------------------------*/

/* Global variables-----------------------------------------------------------*/
static const bool IS_CAN_RX_DEBUG_TRACE_ENABLE=false;
static const bool IS_CAN_TX_DEBUG_TRACE_ENABLE=false;

static SemaphoreHandle_t can_rx_bin_sem=NULL;

static uint32_t last_warming_timeS=0;
static uint32_t warming_timer=0;
static bool is_engine_started_by_terminal=false;
static bool if_forced_stop_warming=false;
static uint32_t is_engine_started_by_terminal_uid;
static uint8_t  is_engine_started_by_terminal_nonce[8];
static uint8_t is_engine_started_by_terminal_server_notify_id;
static bool is_wired_block_relay_enable=false;

static sfifo_t can_rx_sfifo[HW_CAN_COUNT];
static can_rx_frame_t can_rx_queue[HW_CAN_COUNT][CAN_RX_QUEUE_LEN];
static can_mode_t can_mode_config[HW_CAN_COUNT];

static uint8_t lin_cmd_step;
static cls_state_t _cls_state=UNKNOWN_CLS_STATE; //статус ЦЗ. Пока A3, Octavia и Rapid
static bool light_switch_position_in_auto_or_off=false;
static vehicle_can_bus_t can_bus_type[HW_CAN_COUNT];
static bool head_control_output_state=false; //состояние выхода управления мультимедиа
static uint32_t head_control_output_off_timer=0;
static bool is_genesis_w_electric_hand_break=false;
static bool passenger_present=false;
static bool windscreen_wiper=false;
static uint8_t hood_lock_event=0;

static volatile uint16_t can_timers[LAST_CAN_TIMER_ID+1];

static lease_state_t lease_state=UNKNOWN_LEASE_STATE;
static bool is_ignore_end_lease_state=false;
static void save_key_power_state(bool is_on);
static void save_wired_block_relay_state(bool is_lock);
static void save_wireless_block_relay_state(bool is_lock);
static void save_lease_state(void);
static void restore_lease_state(void);
static uint32_t get_lin_bus_speed(void);
static bool is_horn_output_present(uint8_t* output_id);
static bool is_windows_output_present(uint8_t* output_id);
static bool is_ignition_output_present(uint8_t* output_id);
static bool is_starter_output_present(uint8_t* output_id);
static bool is_pedal_break_output_present(uint8_t* output_id);
static uint16_t get_max_engine_on_wait_time_ms(void);
static bool is_start_stop_button_output_present(uint8_t* output_id);
static bool is_wired_block_present(uint8_t* output_id);
static bool is_wireless_block_present(uint8_t* output_id);
static bool is_key_power_present(uint8_t* output_id);
static bool is_ext_key_power_present(uint8_t* output_id);
static bool is_analog_emerg_lights_present(uint8_t* output_id);
static bool is_auto_doors_present(void);
static bool is_ignore_hood(void);
static uint16_t get_async_key_power_off_time_ms(void);
static bool is_key_power_off_no_blocking(void);
static bool is_horn_alarm_w_can_present(void);
static uint16_t get_timeout_in_horn_can_alarm_handler(void);
static void set_head_control_state(const bool is_on, uint16_t timer_m);
static void blocks_handler(void);
static void taxi_blocks_handler(void);
static void restart_sleep_and_rx_timers(uint8_t can_id);
static bool can_sleep_indicate_handler(void);
#if defined(CAN_VEGA_FWD_PRESENT)
static void rx_fwd_hadler(const uint8_t can_id, const can_rx_frame_t* rx);
#endif //defined(CAN_VEGA_FWD_PRESENT)
#if defined(YANDEX_OBD_PRESENT)
static void obd_rx_handler(const uint8_t can_id, const can_rx_frame_t* rx);
#endif //defined(YANDEX_OBD_PRESENT)
static void can_hw_diag_handler(const uint8_t can_id, bool is_reset_state);

static struct can_sleep_struct
{
  const uint16_t can_inactive_timeout;
  uint16_t active_timer[HW_CAN_COUNT];
  bool wakeup_event[HW_CAN_COUNT];
}can_sleep={.can_inactive_timeout=5000};

static struct
{
  uint16_t on_counter;
  uint16_t off_counter;
  uint16_t repeat_counter;

  uint8_t horn_output_id;
  uint16_t on;
  uint16_t off;
}horn_script=
{
  .on_counter=0,
  .off_counter=0,
  .repeat_counter=0,
  .horn_output_id=0xff,
};

static uint8_t horn_can_script_counter=0;

#if defined(YANDEX_OBD_PRESENT)
#include "iso_tp_lib/isotp.h"
#include "eobd_lib/eobd.h"
static IsoTpLink obd_link[2];
static uint8_t obd_isotpRecvBuf[2][128];
static uint8_t obd_isotpSendBuf[2][32];
static struct
{
  uint8_t is_force_cmd;
  uint8_t engine_is_on:1;
  uint8_t step:7;
  uint8_t can_id;
  uint16_t ecu_id[2];
  bool is_broadcast;
}obd_ctx =
{
  .is_force_cmd=0,
  .engine_is_on=0,
  .step=0,
  .can_id=CAN_ID_UNKNOWN,
  .ecu_id[0]=0,
  .ecu_id[1]=0,
};
#endif //defined(YANDEX_OBD_PRESENT)

#if defined(CAN_VEGA_FWD_PRESENT)
static bool update_can_hw_fwd_filter(const uint8_t can_id, const can_fwd_settings_t* const fwd_can_setting);
static can_fwd_settings_t fwd_can_setting[MAX_CAN_INTERFACES]={0};
#endif //defined(CAN_VEGA_FWD_PRESENT)

static inline void force_expired_can_timer(const uint8_t timer_id)
{
  if(timer_id>=sizeof(can_timers)/sizeof(can_timers[0])) return;

  can_timers[timer_id]=0;
  can_timers[timer_id]=0;
}

static inline void restart_can_timer(const uint8_t timer_id, const uint16_t time_ms)
{
  if(timer_id>=sizeof(can_timers)/sizeof(can_timers[0])) return;

  can_timers[timer_id]=time_ms;
  can_timers[timer_id]=time_ms;
}

static inline bool is_can_timer_expired(const uint8_t timer_id)
{
  if(timer_id>=sizeof(can_timers)/sizeof(can_timers[0])) return false;

  if(!can_timers[timer_id]) return true;
  else                      return false;
}

static void horn_alarm_stop(void)
{
  horn_script.repeat_counter=0;

  if(horn_script.horn_output_id!=0xFF)
  {
    System.signal_state.digital_output[horn_script.horn_output_id]=0;
  }
}

bool is_horn_alarm_active(void)
{
  if(horn_script.repeat_counter) {return true;}
  else                           {return false;}
}

static void horn_alarm_start(uint16_t on_time, uint16_t off_time, uint16_t repeats)
{
  horn_script.repeat_counter=0;

  if(!on_time || !off_time || !repeats) return;

  is_horn_output_present(&horn_script.horn_output_id);

  horn_script.on=on_time;
  horn_script.off=off_time;
  horn_script.on_counter=horn_script.on;
  horn_script.off_counter=horn_script.off;
  horn_script.repeat_counter=repeats;
}

bool is_can_horn_alarm_active(void)
{
  if(horn_can_script_counter) {return true;}
  else                        {return false;}
}

static void horn_can_alarm_start(uint8_t repeats)
{
  if(is_horn_alarm_w_can_present())
  {
    horn_can_script_counter=repeats;
  }
}

static void horn_can_alarm_stop(void)
{
  horn_can_script_counter=0;
}

static void horn_can_alarm_handler(void)
{
  if(!is_horn_alarm_w_can_present()) return;

  static uint32_t timer=xTaskGetTickCount();

  if(horn_can_script_counter)
  {
    if(timeAfter(xTaskGetTickCount(), timer+get_timeout_in_horn_can_alarm_handler()))
    {
      uint8_t cmd=CAN_HORN_W_BLINKER_FLASHING_IN_ALARM_SUBCODE;

      uint8_t add_res=signalman_can_subtask_commnad_handle(cmd, 0xFF, 0, NULL, NULL);

      if(CAN_SUBTASK_COMMAND_IS_BUSY!=add_res)
      {
        timer=xTaskGetTickCount();
        horn_can_script_counter--;
      }
    }
  }
  else
  {
    timer=xTaskGetTickCount();
  }
}

#pragma optimize=speed
static inline void horn_alarm_it_handler(void)
{
  if(horn_script.repeat_counter)
  {
    if(horn_script.on_counter)
    {
      if(horn_script.on_counter==horn_script.on)
      {
        if(horn_script.horn_output_id!=0xFF)
        {
          System.signal_state.digital_output[horn_script.horn_output_id]=1;
          outputs_state_force_apply();
        }
      }

      horn_script.on_counter--;
    }
    else if(horn_script.off_counter)
    {
      if(horn_script.off_counter==horn_script.off)
      {
        if(horn_script.horn_output_id!=0xFF)
        {
          System.signal_state.digital_output[horn_script.horn_output_id]=0;
          outputs_state_force_apply();
        }
      }

      horn_script.off_counter--;

      if(!horn_script.off_counter)
      {
        horn_script.repeat_counter--;
        horn_script.on_counter=horn_script.on;
        horn_script.off_counter=horn_script.off;
      }
    }
    else
    {
      if(horn_script.horn_output_id!=0xFF)
      {
        System.signal_state.digital_output[horn_script.horn_output_id]=0;
        outputs_state_force_apply();
      }

      horn_script.repeat_counter=0;
    }
  }
}

#pragma optimize=speed
static inline void head_control_it_handler(void)
{
  //todo: не тестировали
  if(head_control_output_off_timer && head_control_output_off_timer!=0xFFFFFFFF)
  {
    head_control_output_off_timer--;

    if(!head_control_output_off_timer)
    {
      head_control_output_state=false;
    }
  }
}

static inline CAN_TypeDef* get_can_periph_from_id(uint8_t can_id)
{
  if(can_id==CAN1_ID)      {return CAN1;}

#if (HW_CAN_COUNT > 1)
  else if(can_id==CAN2_ID) {return CAN2;}
#endif //(HW_CAN_COUNT > 1)

#if (HW_CAN_COUNT > 2)
  else if(can_id==CAN3_ID) {return CAN3;}
#endif //(HW_CAN_COUNT > 2)

  else               {return NULL;}
}

static inline uint8_t get_can_from_id_from_periph(CAN_TypeDef* CANx)
{
  if(CANx==CAN1)      {return CAN1_ID;}

#if (HW_CAN_COUNT > 1)
  else if(CANx==CAN2) {return CAN2_ID;}
#endif //(HW_CAN_COUNT > 1)

#if (HW_CAN_COUNT > 2)
  else if(CANx==CAN3) {return CAN3_ID;}
#endif //(HW_CAN_COUNT > 2)

  else                {return CAN_ID_UNKNOWN;}
}

static inline const char* get_name_from_can_id(uint8_t can_id)
{
  if(can_id==CAN1_ID)      {return "CAN1";}

#if (HW_CAN_COUNT > 1)
  else if(can_id==CAN2_ID) {return "CAN2";}
#endif //(HW_CAN_COUNT > 1)

#if (HW_CAN_COUNT > 2)
  else if(can_id==CAN3_ID) {return "CAN3";}
#endif //(HW_CAN_COUNT > 2)

  else                     {return "CANX";}
}

#ifdef __cplusplus
extern "C" {
#endif
#pragma optimize=speed
//fixme: вызов этого обработчика желательно убрать из vApplicationTickHook.
//При добавлении дополнителшьных таймеров, выполнение этого обработчика может занимать значительное время. Т.к. vApplicationTickHook выполняется с высоким приоритетом, это может повлять на работу прерываний RX UART, работающих с более низким приоритетом.
//Поэтому, для этого обработчика следует использовать отдельный аппаратный таймер с низшим приоритетом 15.
void can_timers_tick_callback(void)
{
  if(can_timers[KEY_POWER_OFF_TIMER_ID]==1)
  {
    if(is_key_power_off_no_blocking())
    {
      uint8_t key_power_output_id;
      if(is_key_power_present(&key_power_output_id))
      {
        System.signal_state.digital_output[key_power_output_id]=0;
      }
    }
  }

  static const uint8_t CAN_TIMERS_COUNT=sizeof(can_timers)/sizeof(can_timers[0]);

  for(uint8_t timer_id=0; timer_id<CAN_TIMERS_COUNT; timer_id++)
  {
    if(can_timers[timer_id])
    {
      can_timers[timer_id]--;
    }
  }

  for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
  {
    CAN_TypeDef* CANx;

    CANx=get_can_periph_from_id(can_id);
    if(CANx==NULL) break;

    //если есть активность приемника или передатчика или вход rx==0, то перезагружаем таймер
    if((CANx->MSR & CAN_MSR_RXM) || !(CANx->MSR & CAN_MSR_RX) || (CANx->MSR & CAN_MSR_TXM)) {can_sleep.active_timer[can_id]=can_sleep.can_inactive_timeout;}
    else if(can_sleep.active_timer[can_id])                                                 {can_sleep.active_timer[can_id]--;}
  }

  horn_alarm_it_handler();

  head_control_it_handler();
}
#ifdef __cplusplus
}
#endif

typedef enum
{
  KIA_RIO_2017_PROGRAM = 0,
  RENAULT_KAPTUR_2016_PROGRAM,
  HYUNDAI_GENESIS_2017_PROGRAM,
  AUDI_A3_2018_PROGRAM,
  AUDI_Q3_2018_PROGRAM,
  PORSCHE_911_2018_PROGRAM,
  PORSCHE_MACAN_2018_PROGRAM,
  SKODA_OCTAVIA_2018_PROGRAM,
  SKODA_RAPID_2018_PROGRAM,
  VW_POLO_2018_PROGRAM,
  VW_CADDY_2018_PROGRAM,
  BMW_520I_2018_PROGRAM,
  MB_E200_2018_PROGRAM,
  TOYOTA_RAV4_2019_PROGRAM,
  HYUNDAI_SOLARIS_2017_PROGRAM,
  FORD_TRANSIT_2017_PROGRAM,
  CITROEN_JUMPY_2018_PROGRAM,
  RENAULT_DOKKER_2018_PROGRAM,
  LADA_LARGUS_2017_PROGRAM,
  VOLVO_XC60_2018_PROGRAM,
  VOLVO_XC40_2018_PROGRAM,
  RANGE_ROVER_VELAR_2018_PROGRAM,
  NISSAN_QASHQAI_2018_PROGRAM,
  VW_TRANSPORTER_2018_PROGRAM,
  HYUNDAI_CRETA_2019_PROGRAM,
  BMW_X1_2019_PROGRAM,
  FORD_MUSTANG_1965_PROGRAM,
  FORD_MUSTANG_1969_PROGRAM,
  NISSAN_LEAF_2019_PROGRAM,
  PORSCHE_PANAMERA_2019_PROGRAM,
  HYUNDAI_SONATA_SD_2020_PROGRAM,
  RENAULT_LOGAN_2013_PROGRAM,
  RENAULT_DUSTER_2021_PROGRAM,
  JEEP_WRANGLER_2018_PROGRAM,
  SKODA_RAPID_2013_PROGRAM,
  AUDI_A6_2018_PROGRAM,
  RENAULT_ARKANA_2018_PROGRAM,
  LADA_LARGUS_2020_PROGRAM,
}can_program_t;

static const can_program_t can_program=
#if defined(IS_RIO_2017_BUILD)
KIA_RIO_2017_PROGRAM;
#elif defined(IS_KAPTUR_2016_BUILD)
RENAULT_KAPTUR_2016_PROGRAM;
#elif defined(IS_GENESIS_2017_BUILD)
HYUNDAI_GENESIS_2017_PROGRAM;
#elif defined(IS_AUDI_A3_2018_BUILD)
AUDI_A3_2018_PROGRAM;
#elif defined(IS_AUDI_Q3_2018_BUILD)
AUDI_Q3_2018_PROGRAM;
#elif defined(IS_PORSCHE_911_2018_BUILD)
PORSCHE_911_2018_PROGRAM;
#elif defined(IS_PORSCHE_MACAN_2018_BUILD)
PORSCHE_MACAN_2018_PROGRAM;
#elif defined(IS_SKODA_OCTAVIA_2018_BUILD)
SKODA_OCTAVIA_2018_PROGRAM;
#elif defined(IS_SKODA_RAPID_2018_BUILD)
SKODA_RAPID_2018_PROGRAM;
#elif defined(IS_VW_POLO_2018_BUILD)
VW_POLO_2018_PROGRAM;
#elif defined(IS_VW_CADDY_2018_BUILD)
VW_CADDY_2018_PROGRAM;
#elif defined(IS_BMW_520I_2018_BUILD)
BMW_520I_2018_PROGRAM;
#elif defined(IS_MB_E200_2018_BUILD)
MB_E200_2018_PROGRAM;
#elif defined(IS_TOYOTA_RAV4_2019_BUILD)
TOYOTA_RAV4_2019_PROGRAM;
#elif defined(IS_FORD_TRANSIT_2017_BUILD)
FORD_TRANSIT_2017_PROGRAM;
#elif defined(IS_CITROEN_JUMPY_2018_BUILD)
CITROEN_JUMPY_2018_PROGRAM;
#elif defined(IS_RENAULT_DOKKER_2018_BUILD)
RENAULT_DOKKER_2018_PROGRAM;
#elif defined(IS_LADA_LARGUS_2017_BUILD)
LADA_LARGUS_2017_PROGRAM;
#elif defined(IS_HYUNDAI_SOLARIS_2017_TAXI_BUILD)
HYUNDAI_SOLARIS_2017_PROGRAM;
#elif defined(IS_SKODA_OCTAVIA_2018_TAXI_BUILD)
SKODA_OCTAVIA_2018_PROGRAM;
#elif defined(IS_VW_POLO_2018_TAXI_BUILD)
VW_POLO_2018_PROGRAM;
#elif defined(IS_RENAULT_LOGAN_2018_TAXI_BUILD)
RENAULT_DOKKER_2018_PROGRAM;
#elif defined(IS_VOLVO_XC60_2018_BUILD)
VOLVO_XC60_2018_PROGRAM;
#elif defined(IS_VOLVO_XC40_2018_BUILD)
VOLVO_XC40_2018_PROGRAM;
#elif defined(IS_RANGE_ROVER_VELAR_2018_BUILD)
RANGE_ROVER_VELAR_2018_PROGRAM;
#elif defined(IS_NISSAN_QASHQAI_2018_BUILD)
NISSAN_QASHQAI_2018_PROGRAM;
#elif defined(IS_VW_TRANSPORTER_2018_BUILD)
VW_TRANSPORTER_2018_PROGRAM;
#elif defined(IS_HYUNDAI_CRETA_2019_BUILD)
HYUNDAI_CRETA_2019_PROGRAM;
#elif defined(IS_BMW_X1_2019_BUILD)
BMW_X1_2019_PROGRAM;
#elif defined(IS_FORD_MUSTANG_1965_BUILD)
FORD_MUSTANG_1965_PROGRAM;
#elif defined(IS_FORD_MUSTANG_1969_BUILD)
FORD_MUSTANG_1969_PROGRAM;
#elif defined(IS_NISSAN_LEAF_2019_BUILD)
NISSAN_LEAF_2019_PROGRAM;
#elif defined(IS_PORSCHE_PANAMERA_2019_BUILD)
PORSCHE_PANAMERA_2019_PROGRAM;
#elif defined(IS_HYUNDAI_SONATA_SD_2020_BUILD)
HYUNDAI_SONATA_SD_2020_PROGRAM;
#elif defined(IS_RENAULT_LOGAN_2013_BUILD)
RENAULT_LOGAN_2013_PROGRAM;
#elif defined(IS_LADA_LARGUS_2020_BUILD)
LADA_LARGUS_2020_PROGRAM;
#elif defined(IS_RENAULT_DUSTER_2021_BUILD)
RENAULT_DUSTER_2021_PROGRAM;
#elif defined(IS_JEEP_WRANGLER_2018_BUILD)
JEEP_WRANGLER_2018_PROGRAM;
#elif defined(IS_SKODA_RAPID_2013_BUILD)
SKODA_RAPID_2013_PROGRAM;
#elif defined(IS_AUDI_A6_2018_BUILD)
AUDI_A6_2018_PROGRAM;
#elif defined(IS_RENAULT_ARKANA_2018_BUILD)
RENAULT_ARKANA_2018_PROGRAM;
#else
#error unknown program
#endif

#if defined(IS_NISSAN_QASHQAI_2018_BUILD)
#warning добавили эмуляцию двери, теперь не ясно закрываются ли окна
#endif //defined(IS_NISSAN_QASHQAI_2018_BUILD)

static bool is_taxi(void)
{
#if defined(IS_TAXI)
  return true;
#else
  return false;
#endif
}

typedef enum
{
  EMTY_CAN_CMD_TYPE = 0,

  DELAY_1,
  DELAY_2,
  DELAY_3,
  DELAY_4,
  CAN_WAKE_UP_CMD_TYPE,
  LEAF_WAKEUP_CMD_TYPE,
  DIAG_SESSION_CMD_TYPE,
  CENTRAL_LOCK_OPEN_CMD_TYPE,
  CENTRAL_LOCK_CLOSE_CMD_TYPE,
  CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE,
  CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE,
  TRUNK_OPEN_CMD_TYPE,
  PARKING_LIGHT_ON_CMD_TYPE,
  PARKING_LIGHT_OFF_CMD_TYPE,
  START_ENGINE_CMD_TYPE,
  STOP_ENGINE_CMD_TYPE,
  DRIVER_DOOR_EMULATION_CMD_TYPE,
  HORN_CMD_TYPE,
  SILENT_HORN_CMD_TYPE, //если тихо гудеть не получается, то молчим
  WINDOWS_CLOSING_CMD_TYPE,
  WINDOWS_CLOSING_NEW_A3_CMD_TYPE,
  EXT_WINDOWS_CLOSING_CMD_TYPE_GENESIS, //дополнительная команды закрыть окна для GENESIS (по CAN), закрывает все кроме водительского (водительское закрывается по аналогу)
  WINDOWS_OPENING_CMD_TYPE,
  ROOF_CLOSING_CMD_TYPE,

  HOOD_LOCK_CMD_TYPE,
  HOOD_LOCK_NOTIFY_CMD_TYPE,
  HOOD_UNLOCK_CMD_TYPE,
  HOOD_UNLOCK_NOTIFY_CMD_TYPE,

  WIRELESS_BLOCK_RELAY_CMD_TYPE,
  WIRELESS_UNBLOCK_RELAY_CMD_TYPE,
  WIRED_BLOCK_RELAY_CMD_TYPE,
  WIRED_UNBLOCK_RELAY_CMD_TYPE,

  KEY_POWER_ON_CMD_TYPE,
  KEY_POWER_OFF_CMD_TYPE,

  IGN_ON_CMD_TYPE,
  IGN_OFF_CMD_TYPE,

  FORD_TRANSIT_IGN_ON_CMD_TYPE,  //используется только перед открытием ЦЗ
  FORD_TRANSIT_IGN_OFF_CMD_TYPE, //используется только перед открытием ЦЗ

  CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE,
  CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE,
  CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE,          //Проверяем закрытие всех дверей и выключение зажигания. Если одна из дверей или багажник или капот открыты или вставлен ключ, то очищаем все последующие команды.
  CHECK_DOORS_CLOSED_CMD_TYPE,
  CHECK_PARKING_STATE_CMD_TYPE,                                  //Проверяем положение коробки в паркинге. Если не паркинг, то очищаем все последующие команды.
  CHECK_PARKING_LIGHT_STATE_CMD_TYPE,                            //Проверяем отлючение света и положения коробки. Если не свет горит или трансмиссия не в паркинге, то очищаем все последующие команды. Работает только при вкл зажигании
  CHECK_PARKING_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE,       //Тоже самое, с отключением проводной блокировки на вермя проверки (когда блокировка отключает зажигание)
  CHECK_PARKING_LIGHT_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE, //Тоже самое, с отключением проводной блокировки на вермя проверки (когда блокировка отключает зажигание)
  CHECH_KEY_ACC_MODE_CMD_TYPE,                                   //Prohibit end of leasing at ACC key mode
  
  START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE,                          //Отправить уведомлнение об успешной начале аренды
  END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE,                            //Отправить уведомлнение об успешном завершении аренды
  FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE,                     //Отправить уведомлнение об успешном завершении аренды форсированно
  NOT_AVAILABLE_NOTTIFY_CMD_TYPE,                                //Отправить уведомлнение о не доступности операции
  ADD_EMTY_LAST_NOTIFY_CMD_TYPE,                                 //Отправить пустое IS_LAST_NOTIFY
  CHECK_CLS_AND_WINDOWS_CMD_TYPE,                                //
  CHECK_CLS_CMD_TYPE,                                            //
  CHECK_WINDOWS_CMD_TYPE,                                        //
  CHECK_ROOF_CMD_TYPE,                                           //
  CHECK_BMW_CLS_CMD_TYPE,                                        //
  WAIT_WINDOWS_CLOSING_CMD_TYPE,
  STATRTSTOP_BUTTON_PUSH_CMD_TYPE,

  CHARGE_UNLOCK_CMD_TYPE,
  CHARGE_LOCK_CMD_TYPE,
  ECO_ON_CMD_TYPE,
  ECO_OFF_CMD_TYPE,
  FUEL_CAP_UNLOCK_CMD_TYPE,
  FUEL_CAP_LOCK_CMD_TYPE,
  TRUNK_UNLOCK_CMD_TYPE,
  TRUNK_LOCK_CMD_TYPE,
  FUEL_CAP_AND_TRUNK_UNLOCK_CMD_TYPE,
  FUEL_CAP_AND_TRUNK_LOCK_CMD_TYPE,

  CRETA_CAN_PARKING_LIGHT_ON_CMD_TYPE,
  KIA_BCM_BEGIN_CONFIG_1_CMD_TYPE,
  KIA_BCM_DISABLE_DOORS_BLOCK_CMD_TYPE,
  KIA_BCM_DISABLE_DOORS_UNBLOCK_CMD_TYPE,
  KIA_BCM_END_CONFIG_2_CMD_TYPE,

  CLEAR_BCM_ERRS_CMD_TYPE,    //стирание ошибок BCM
  CLEAR_ENGINE_ERRS_CMD_TYPE, //стирание ошибок двигателя

  PANAMERA_WINDOWS_CLOSING1_1_CMD_TYPE,
  PANAMERA_WINDOWS_CLOSING1_2_CMD_TYPE,
  PANAMERA_WINDOWS_CLOSING2_1_CMD_TYPE,
  PANAMERA_WINDOWS_CLOSING2_2_CMD_TYPE,
  PANAMERA_WINDOWS_CLOSING3_1_CMD_TYPE,
  PANAMERA_WINDOWS_CLOSING3_2_CMD_TYPE,
  PANAMERA_WINDOWS_CLOSING4_1_CMD_TYPE,
  PANAMERA_WINDOWS_CLOSING4_2_CMD_TYPE,

  TOYOTA_RAV4_WINDOWS_CLOSING_CMD_TYPE,
  DRIVER_WINDOW_CLOSING_CMD_TYPE,
  PASS_WINDOW_CLOSING_CMD_TYPE,
  REAR_L_WINDOW_CLOSING_CMD_TYPE,
  REAR_R_WINDOW_CLOSING_CMD_TYPE,

  VOLVO_XC40_ROOF_CLOSING_1_CMD_TYPE,
  VOLVO_XC40_ROOF_CLOSING_2_CMD_TYPE,
  VOLVO_XC40_CAN_CENTRAL_LOCK_CLOSE_CMD_TYPE,
  VOLVO_XC40_CAN_CENTRAL_LOCK_OPEN_CMD_TYPE,

  SKIP_TWO_CMD_IF_CLS_OPENED_CMD_TYPE, //пропустить две команды, если ЦЗ открыт
  SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE, //пропустить две команды, если ЦЗ закрыт

  LOGAN_2013_PRECAN_CMD_TYPE, //отправляется перед перед CAN командами

  PRESS_TRC_BUTTON_CMD_TYPE, //нажатие кнопки TRC/ESP на Wrangler

  OFF_AUTO_OPEN_DOORS_CMD_TYPE, // Off auto open doors while warming complete (Only for Creta)
  FIX_FUEL_VIEW_LOGAN_2013,

  OBD_DTC_CLEAR_TYPE,  //reset DTC, only for DTC_CLEAR
  OBD_DTC_UPDATE_TYPE, //read DTC, only for DTC_CLEAR
}can_cmd_type_t;

typedef __packed struct
{
  can_cmd_type_t cmd_type;
  CAN_TypeDef* CANx;
  id_type_t id_type;
  uint32_t id_1;
  uint8_t repeats_1;
  uint8_t mess_1[8];
  uint8_t len_1;
  uint16_t delay_1;
  uint32_t id_2;
  uint8_t repeats_2;
  uint8_t mess_2[8];
  uint8_t len_2;
  uint16_t delay_2;
  uint8_t  seq_repeats;
}can_commands_table_t;

//Команды управления. Устанавливаюются из потокока signalman, обрабатываются в signalman_can_subtask
static struct can_cmd_proc_struct
{
  can_cmd_type_t cmd_q[42];
  uint16_t       step;
  uint32_t       timer;
  uint32_t       ext_param;
  uint32_t       uid; //идентификатор для последовательности команд
  uint8_t        nonce[8];
  uint16_t       delay[4];
  uint8_t        server_notify_id;
}can_cmd_proc;

typedef enum
{
  EMTY_CAN_REQ_TYPE,
  GET_UDS_FUEL_LEVEL_CMD_TYPE,
  GET_UDS_WASHER_LIQUID_RAV4_CMD_TYPE,
  GET_UDS_DASH_BOARD_STATE_CMD_TYPE,
  GET_OBD_FUEL_TANK_CMD_TYPE,
  GET_OBD_FUEL_LITERS_DOKKER_CMD_TYPE,
  GET_OBD_FUEL_LITERS_QASHQAI_CMD_TYPE,
  GET_OBD_ENGINE_TEMP_CMD_TYPE,
  GET_OBD_FUEL_FLOW_CMD_TYPE,
  GET_OBD_KAPTUR_ODO_CMD_TYPE,
  GET_OBD_RAPID_ODO_CMD_TYPE,
  GET_OBD_FUEL_TANK_XC60_CMD_TYPE,
  GET_OBD_ENGINE_TEMP_XC60_CMD_TYPE,
  GET_OBD_FUEL_TANK_VELAR_CMD_TYPE,
  GET_OBD_ENGINE_TEMP_VELAR_CMD_TYPE,
}can_req_type_t;

//Запросы obd, uds. Устанавливаюются из потока signalman_can_subtask, в нем же и обрабатываются
static struct can_req_proc_struct
{
  can_req_type_t req;
  uint8_t        step;
  uint32_t       timer;
  CAN_TypeDef *  CANx;
}can_req_proc;

static bool is_can_cmd_proc_q_emty(void)
{
  for(uint8_t i=0; i<(sizeof(can_cmd_proc.cmd_q)/sizeof(can_cmd_proc.cmd_q[0])); i++)
  {
    if(can_cmd_proc.cmd_q[i]!=EMTY_CAN_CMD_TYPE) return false;
  }
  return true;
}

static void move_next_command_to_head(void)
{
  uint8_t next_cmd_id;
  for(next_cmd_id=1; next_cmd_id<(sizeof(can_cmd_proc.cmd_q)/sizeof(can_cmd_proc.cmd_q[0])); next_cmd_id++)
  {
    if(can_cmd_proc.cmd_q[next_cmd_id]!=EMTY_CAN_CMD_TYPE)
    {
      can_cmd_proc.cmd_q[0]=can_cmd_proc.cmd_q[next_cmd_id];//в начало
      can_cmd_proc.cmd_q[next_cmd_id]=EMTY_CAN_CMD_TYPE; //стираем
      break;
    }
  }
  if(next_cmd_id==(sizeof(can_cmd_proc.cmd_q)/sizeof(can_cmd_proc.cmd_q[0]))) can_cmd_proc.cmd_q[0]=EMTY_CAN_CMD_TYPE; //если последующих команд нет, то просто стираем текущую
}

static void clear_next_commands(void)
{
  do
  {
    move_next_command_to_head();
    can_cmd_proc.step=0;
  }while(is_can_cmd_proc_q_emty()==false);
}

static void can_rx_debug_trace(can_rx_frame_t* rx, uint8_t can_id)
{
  if(IS_CAN_RX_DEBUG_TRACE_ENABLE)
  {
    __PRINTF("|%s_RX|%5.5lu|%8.8lX(%s%s)|%01hhu|%02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX|\n", \
      get_name_from_can_id(can_id), rx->time, rx->id, (rx->id_type==EXT_ID_TYPE)?("EFF"):("SFF"), (rx->rtr)?(",RTR"):(""), \
      rx->dlen, rx->data[0], rx->data[1], rx->data[2], rx->data[3], rx->data[4], rx->data[5], rx->data[6], rx->data[7]);
  }
}

static bool is_can_in_sleep(CAN_TypeDef* CANx)
{
  if(CANx==CAN1)      {}
#if (HW_CAN_COUNT > 1)
  else if(CANx==CAN2) {}
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
  else if(CANx==CAN3) {}
#endif //(HW_CAN_COUNT > 2)
  else                {return true;}

  if(((CANx->MSR & (CAN_MSR_SLAK|CAN_MSR_INAK)) == CAN_MSR_SLAK)) return true;
  else                                                            return false;
}

static uint8_t CAN_Transmit_Blocked(CAN_TypeDef* CANx, CanTxMsg* TxMessage, uint32_t TimeWait)
{
  uint8_t can_id;

  if(CANx==CAN1)      {can_id=0;}
#if (HW_CAN_COUNT > 1)
  else if(CANx==CAN2) {can_id=1;}
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
  else if(CANx==CAN3) {can_id=2;}
#endif //(HW_CAN_COUNT > 2)
  else                {return 0;}

  if(!(can_mode_config[can_id]==SILENT_CAN_MODE || can_mode_config[can_id]==NORMAL_CAN_MODE || can_mode_config[can_id]==LOOPBACK_CAN_MODE || can_mode_config[can_id]==SILENT_LOOPBACK_CAN_MODE)) return false;

  if(is_can_in_sleep(CANx))
  {
    if(CANx==CAN1)      {__CAN1_NORMAL_MODE();}
#if (HW_CAN_COUNT > 1)
    else if(CANx==CAN2) {__CAN2_NORMAL_MODE();}
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
    else if(CANx==CAN3) {__CAN3_NORMAL_MODE();}
#endif //(HW_CAN_COUNT > 2)
    else                {return 0;}

    can_sleep.active_timer[can_id]=can_sleep.can_inactive_timeout;

    if(CAN_WakeUp_Ok==CAN_WakeUp(CANx))
    {
      LOG("%s: wakeup request %s\n",  get_name_from_can_id(can_id), "ok");
    }
    else
    {
      LOG("%s: wakeup request %s\n",  get_name_from_can_id(can_id), "err");
    }
  }

  uint8_t mailbox=CAN_Transmit(CANx, TxMessage);
  TimeWait+=xTaskGetTickCount();

  uint8_t is_tx_done=1;
  for(;;)
  {
    if(CAN_TxStatus_Ok==CAN_TransmitStatus(CANx, mailbox)) break;
    if(timeAfter(xTaskGetTickCount(), TimeWait))
    {
      CAN_CancelTransmit(CANx, mailbox);
      is_tx_done=0;
      break;
    }
    vTaskDelay(1);
  }

  if(IS_CAN_TX_DEBUG_TRACE_ENABLE)
  {
    __PRINTF("|%s_TX|%5.5lu|%8.8lX(%s%s)|%01hhu|%02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX|%s\n", \
      get_name_from_can_id(can_id), xTaskGetTickCount(), (TxMessage->IDE==CAN_Id_Extended)?(TxMessage->ExtId):(TxMessage->StdId), (TxMessage->IDE==CAN_Id_Extended)?("EFF"):("SFF"), (TxMessage->RTR==CAN_RTR_Remote)?(",RTR"):(""), \
        TxMessage->DLC, TxMessage->Data[0], TxMessage->Data[1], TxMessage->Data[2], TxMessage->Data[3], TxMessage->Data[4], TxMessage->Data[5], TxMessage->Data[6], TxMessage->Data[7], (is_tx_done)?(""):("tx fail|"));
  }

  return is_tx_done;
}

static bool is_can_rx_in_idle(uint8_t can_id)
{
  bool rx_in_idle=true;

  if(can_id<HW_CAN_COUNT)
  {
    if(can_id==CAN1_ID)
    {
      if(can_timers[CAN1_MAIN_RX_TIMER_ID]) rx_in_idle=false;
    }

#if (HW_CAN_COUNT > 1)
    else if(can_id==CAN2_ID)
    {
      if(can_timers[CAN2_MAIN_RX_TIMER_ID]) rx_in_idle=false;
    }
#endif //(HW_CAN_COUNT > 1)

#if (HW_CAN_COUNT > 2)
    else if(can_id==CAN3_ID)
    {
      if(can_timers[CAN3_MAIN_RX_TIMER_ID]) rx_in_idle=false;
    }
#endif //(HW_CAN_COUNT > 2)
  }

  return rx_in_idle;
}

static void can_hw_onetime_init(void)
{
  GPIO_InitTypeDef       GPIO_InitStructure;
  NVIC_InitTypeDef       NVIC_InitStructure;

  LedCAN_Off();

  if(can_rx_bin_sem==NULL)
  {
    vSemaphoreCreateBinary(can_rx_bin_sem);
  }
  xSemaphoreTake(can_rx_bin_sem, 0);


  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_OType = CAN_DRIVERS_PWR_OTYPE;
  GPIO_InitStructure.GPIO_Pin = CAN_DRIVERS_PWR_PIN;
  GPIO_Init(CAN_DRIVERS_PWR_PORT, &GPIO_InitStructure);
  __CAN_DRIVERS_PWR_EN();

  //CAN1
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_CAN1, ENABLE);

  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_OType = CAN1_RXE_OTYPE;
  GPIO_InitStructure.GPIO_Pin = CAN1_RXE_PIN;
  GPIO_Init(CAN1_RXE_PORT, &GPIO_InitStructure);
  __CAN1_STANDBY_MODE();

  __CAN1_REMAP();

  // Configure CAN RX and TX pins
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_UP;

  GPIO_InitStructure.GPIO_Pin = CAN1_RX_PIN;
  GPIO_Init(CAN1_RX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = CAN1_TX_PIN;
  GPIO_Init(CAN1_TX_PORT, &GPIO_InitStructure);

  NVIC_InitStructure.NVIC_IRQChannel = CAN1_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = CAN1_RX_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  NVIC_InitStructure.NVIC_IRQChannel = CAN1_SCE_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = CAN1_SCE_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  //CAN2
#if (HW_CAN_COUNT > 1)
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_CAN2, ENABLE);

  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_OType = CAN2_RXE_OTYPE;
  GPIO_InitStructure.GPIO_Pin = CAN2_RXE_PIN;
  GPIO_Init(CAN2_RXE_PORT, &GPIO_InitStructure);
  __CAN2_STANDBY_MODE();

  __CAN2_REMAP();

  // Configure CAN RX and TX pins
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_UP;

  GPIO_InitStructure.GPIO_Pin = CAN2_RX_PIN;
  GPIO_Init(CAN2_RX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = CAN2_TX_PIN;
  GPIO_Init(CAN2_TX_PORT, &GPIO_InitStructure);

  NVIC_InitStructure.NVIC_IRQChannel = CAN2_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = CAN2_RX_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  NVIC_InitStructure.NVIC_IRQChannel = CAN2_SCE_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = CAN2_SCE_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
#endif //(HW_CAN_COUNT > 1)

  //CAN3
#if (HW_CAN_COUNT > 2)
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_CAN3, ENABLE);

  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_OType = CAN3_RXE_OTYPE;
  GPIO_InitStructure.GPIO_Pin = CAN3_RXE_PIN;
  GPIO_Init(CAN3_RXE_PORT, &GPIO_InitStructure);
  __CAN3_STANDBY_MODE();

  __CAN3_REMAP();

  // Configure CAN RX and TX pins
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_UP;

  GPIO_InitStructure.GPIO_Pin = CAN3_RX_PIN;
  GPIO_Init(CAN3_RX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = CAN3_TX_PIN;
  GPIO_Init(CAN3_TX_PORT, &GPIO_InitStructure);

  NVIC_InitStructure.NVIC_IRQChannel = CAN3_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = CAN3_RX_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  NVIC_InitStructure.NVIC_IRQChannel = CAN3_SCE_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = CAN3_SCE_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
#endif //(HW_CAN_COUNT > 2)

  for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
  {
    can_bus_type[can_id]=UNKNOWN_CAN_BUS_TYPE;
    can_mode_config[can_id]=OFF_CAN_MODE;
  }
}

#include "can_filter_and_comm_tables.c"
#include "can_rx_handlers.c"

static uint8_t can_hw_init(uint16_t can1_baudrate, const can_filter_t* can1_filter_table, uint16_t can1_filter_table_size,
                           uint16_t can2_baudrate, const can_filter_t* can2_filter_table, uint16_t can2_filter_table_size,
                           uint16_t can3_baudrate, const can_filter_t* can3_filter_table, uint16_t can3_filter_table_size)
{
  LedCAN_Off();

  static const __packed struct can_baudrate_table_struct
  {
    uint16_t baudrate;
    uint8_t  swj;
    uint8_t  bs1;
    uint8_t  bs2;
    uint16_t prescaler;
  }can_baudrate_table[]=
  {
#if (TARGET_CPU_FREQ_HZ == 96000000)
    //at 96Mhz PCLK1
    {.baudrate=100, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_11tq, .bs2=CAN_BS2_4tq, .prescaler=60},
    {.baudrate=125, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_11tq, .bs2=CAN_BS2_4tq, .prescaler=48},
    {.baudrate=120, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_11tq, .bs2=CAN_BS2_4tq, .prescaler=50},
    {.baudrate=250, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_11tq, .bs2=CAN_BS2_4tq, .prescaler=24},
    {.baudrate=500, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_11tq, .bs2=CAN_BS2_4tq, .prescaler=12},
#elif (TARGET_CPU_FREQ_HZ == 60000000)
    //at 30Mhz PCLK1
    {.baudrate=100, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_14tq, .bs2=CAN_BS2_5tq, .prescaler=15},
    {.baudrate=125, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_11tq, .bs2=CAN_BS2_4tq, .prescaler=15},
    {.baudrate=120, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_7tq,  .bs2=CAN_BS2_2tq, .prescaler=25},
    {.baudrate=250, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_14tq, .bs2=CAN_BS2_5tq, .prescaler=6},
    {.baudrate=500, .swj=CAN_SJW_1tq, .bs1=CAN_BS1_14tq, .bs2=CAN_BS2_5tq, .prescaler=3},
#else
#error
#endif
  };
  static const uint8_t can_baudrate_table_elements_count=sizeof(can_baudrate_table)/sizeof(can_baudrate_table[0]);

  for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
  {
    CAN_TypeDef* CANx;
    uint16_t     can_baudrate;
    uint8_t      can_baud_tindex[HW_CAN_COUNT];

    if(can_id==CAN1_ID)
    {
      CANx=CAN1;
      can_baudrate=can1_baudrate;
    }
#if (HW_CAN_COUNT > 1)
    else if(can_id==CAN2_ID)
    {
      CANx=CAN2;
      can_baudrate=can2_baudrate;
    }
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
    else if(can_id==CAN3_ID)
    {
      CANx=CAN3;
      can_baudrate=can3_baudrate;
    }
#endif //(HW_CAN_COUNT > 2)
    else
    {
      break;
    }

    if(can_baudrate)
    {
      for(can_baud_tindex[can_id]=0; can_baud_tindex[can_id]<can_baudrate_table_elements_count; can_baud_tindex[can_id]++)
      {
        if(can_baudrate_table[can_baud_tindex[can_id]].baudrate==can_baudrate) break;
      }
      if(can_baud_tindex[can_id]>=can_baudrate_table_elements_count)
      {
        can_baudrate=0;
        if(CANx==CAN1)
        {
          can1_baudrate=can_baudrate;
        }
#if (HW_CAN_COUNT > 1)
        else if(CANx==CAN2)
        {
          can2_baudrate=can_baudrate;
        }
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
        else if(CANx==CAN3)
        {
          can3_baudrate=can_baudrate;
        }
#endif //(HW_CAN_COUNT > 2)
      }
    }

    CAN_DeInit(CANx);
    if(can_baudrate)
    {
      CAN_InitTypeDef CAN_InitStructure;
      CAN_StructInit(&CAN_InitStructure);

      // CAN cell init
      CAN_InitStructure.CAN_TTCM = DISABLE;
      CAN_InitStructure.CAN_ABOM = ENABLE;
      CAN_InitStructure.CAN_AWUM = ENABLE;
      CAN_InitStructure.CAN_NART = DISABLE;
      CAN_InitStructure.CAN_RFLM = DISABLE;
      CAN_InitStructure.CAN_TXFP = DISABLE;
      CAN_InitStructure.CAN_Mode = CAN_Mode_Normal;

      CAN_InitStructure.CAN_SJW = can_baudrate_table[can_baud_tindex[can_id]].swj;
      CAN_InitStructure.CAN_BS1 = can_baudrate_table[can_baud_tindex[can_id]].bs1;
      CAN_InitStructure.CAN_BS2 = can_baudrate_table[can_baud_tindex[can_id]].bs2;
      CAN_InitStructure.CAN_Prescaler = can_baudrate_table[can_baud_tindex[can_id]].prescaler;
      CAN_Init(CANx, &CAN_InitStructure);
    }

    for(uint8_t i=0; i<MAX_HW_FILTER_BANK_PER_CAN; i++)
    {
      CAN_FilterInitTypeDef CAN_FilterInitStructure;
      const can_filter_t* can_filter_bank;
      uint8_t can_hw_fifo;
      uint8_t filter_number;

      if(can_id==CAN1_ID)
      {
        can_hw_fifo=CAN1_FIFO;
        filter_number=i+0;
        if(can1_filter_table==NULL || i>=(can1_filter_table_size/sizeof(*can1_filter_table))) can_filter_bank=NULL;
        else                                                                                  can_filter_bank=&can1_filter_table[i];
      }
#if (HW_CAN_COUNT > 1)
      else if(can_id==CAN2_ID)
      {
        can_hw_fifo=CAN2_FIFO;
        filter_number=i+MAX_HW_FILTER_BANK_PER_CAN;//для CAN1 и CAN2 фильтр один
        if(can2_filter_table==NULL || i>=can2_filter_table_size/sizeof(*can2_filter_table)) can_filter_bank=NULL;
        else                                                                                can_filter_bank=&can2_filter_table[i];
      }
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
      else if(can_id==CAN3_ID)
      {
        can_hw_fifo=CAN3_FIFO;
        filter_number=i+0;
        if(can3_filter_table==NULL || i>=can3_filter_table_size/sizeof(*can3_filter_table)) can_filter_bank=NULL;
        else                                                                                can_filter_bank=&can3_filter_table[i];
      }
#endif //(HW_CAN_COUNT > 2)
      else
      {
        break;
      }

      if(can_filter_bank!=NULL && can_filter_bank->bank_type==STD_ID_FILTER_BANK_TYPE)
      {
        CAN_FilterInitStructure.CAN_FilterActivation = ENABLE;
        CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdList;
        CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_16bit;
        CAN_FilterInitStructure.CAN_FilterIdHigh =     (uint16_t)((can_filter_bank->bank.std.value1)<<5);
        CAN_FilterInitStructure.CAN_FilterIdLow =      (uint16_t)((can_filter_bank->bank.std.value2)<<5);
        CAN_FilterInitStructure.CAN_FilterMaskIdHigh = (uint16_t)((can_filter_bank->bank.std.value3)<<5);
        CAN_FilterInitStructure.CAN_FilterMaskIdLow =  (uint16_t)((can_filter_bank->bank.std.value4)<<5);
      }
      else if(can_filter_bank!=NULL && can_filter_bank->bank_type==EXT_ID_FILTER_BANK_TYPE)
      {
        CAN_FilterInitStructure.CAN_FilterActivation = ENABLE;
        CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdList;
        CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_32bit;
        CAN_FilterInitStructure.CAN_FilterIdHigh =     (uint16_t)((can_filter_bank->bank.ext.value1)>>13);
        CAN_FilterInitStructure.CAN_FilterIdLow =      (uint16_t)(((can_filter_bank->bank.ext.value1)<<3)|(1<<2));
        CAN_FilterInitStructure.CAN_FilterMaskIdHigh = (uint16_t)((can_filter_bank->bank.ext.value2)>>13);
        CAN_FilterInitStructure.CAN_FilterMaskIdLow =  (uint16_t)(((can_filter_bank->bank.ext.value2)<<3)|(1<<2));
      }
      else if(can_filter_bank!=NULL && can_filter_bank->bank_type==STD_ID_W_MASK_FILTER_BANK_TYPE)
      {
        CAN_FilterInitStructure.CAN_FilterActivation = ENABLE;
        CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdMask;
        CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_16bit;
        CAN_FilterInitStructure.CAN_FilterIdHigh =     (uint16_t)((can_filter_bank->bank.std_w_mask.value1)<<5);
        CAN_FilterInitStructure.CAN_FilterIdLow =      (uint16_t)((can_filter_bank->bank.std_w_mask.value2)<<5);
        CAN_FilterInitStructure.CAN_FilterMaskIdHigh = (uint16_t)((can_filter_bank->bank.std_w_mask.mask1)<<5);
        CAN_FilterInitStructure.CAN_FilterMaskIdLow =  (uint16_t)((can_filter_bank->bank.std_w_mask.mask2)<<5);
      }
      else if(can_filter_bank!=NULL && can_filter_bank->bank_type==EXT_ID_W_MASK_FILTER_BANK_TYPE)
      {
        CAN_FilterInitStructure.CAN_FilterActivation = ENABLE;
        CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdMask;
        CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_32bit;
        CAN_FilterInitStructure.CAN_FilterIdHigh =     (uint16_t)((can_filter_bank->bank.ext_w_mask.value1)>>13);
        CAN_FilterInitStructure.CAN_FilterIdLow =      (uint16_t)(((can_filter_bank->bank.ext_w_mask.value1)<<3)|(1<<2));
        CAN_FilterInitStructure.CAN_FilterMaskIdHigh = (uint16_t)((can_filter_bank->bank.ext_w_mask.mask1)>>13);
        CAN_FilterInitStructure.CAN_FilterMaskIdLow =  (uint16_t)(((can_filter_bank->bank.ext_w_mask.mask1)<<3)|(1<<2));
      }
      else //(can_filter_bank==NULL || can_filter_bank->bank_type==EMPTY_FILTER_BANK_TYPE)
      {
        CAN_FilterInitStructure.CAN_FilterActivation = DISABLE;
        CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdList;
        CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_32bit;
        CAN_FilterInitStructure.CAN_FilterIdHigh =     0;
        CAN_FilterInitStructure.CAN_FilterIdLow =      0;
        CAN_FilterInitStructure.CAN_FilterMaskIdHigh = 0;
        CAN_FilterInitStructure.CAN_FilterMaskIdLow =  0;
      }

      CAN_FilterInitStructure.CAN_FilterFIFOAssignment = can_hw_fifo;
      CAN_FilterInitStructure.CAN_FilterNumber =         filter_number;

#if defined(STM32F413_423xx)
      if(CANx==CAN2) CANx=CAN1;//для CAN1 и CAN2 фильтр один
      CAN_FilterInit(CANx, &CAN_FilterInitStructure);
#else
      CAN_FilterInit(&CAN_FilterInitStructure);
#endif
    }
  }

  for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
  {
    sfifo_init(&can_rx_sfifo[can_id], can_rx_queue[can_id], sizeof(can_rx_queue[can_id]), sizeof(can_rx_queue[can_id][0]));
  }

  xSemaphoreTake(can_rx_bin_sem, 0);

  CAN_ClearFlag(CAN1, CAN_IT_TME|CAN_IT_FF0|CAN_IT_FOV0|CAN_IT_FF1|CAN_IT_FOV1|CAN_IT_WKU|CAN_IT_SLK|CAN_IT_EWG|CAN_IT_EPV|CAN_IT_BOF|CAN_IT_LEC|CAN_IT_ERR);

  can_sleep.wakeup_event[0]=false;
  can_sleep.active_timer[0]=can_sleep.can_inactive_timeout;

  if(can1_baudrate)
  {
    can_mode_config[CAN1_ID]=NORMAL_CAN_MODE;
    CAN_ITConfig(CAN1, CAN1_IT_FMP | CAN_IT_WKU | CAN_IT_SLK, ENABLE);
    __CAN1_NORMAL_MODE();
  }
  else
  {
    can_mode_config[CAN1_ID]=OFF_CAN_MODE;
    __CAN1_STANDBY_MODE();
  }

#if (HW_CAN_COUNT > 1)
  CAN_ClearFlag(CAN2, CAN_IT_TME|CAN_IT_FF0|CAN_IT_FOV0|CAN_IT_FF1|CAN_IT_FOV1|CAN_IT_WKU|CAN_IT_SLK|CAN_IT_EWG|CAN_IT_EPV|CAN_IT_BOF|CAN_IT_LEC|CAN_IT_ERR);

  can_sleep.wakeup_event[1]=false;
  can_sleep.active_timer[1]=can_sleep.can_inactive_timeout;

  if(can2_baudrate)
  {
    can_mode_config[CAN2_ID]=NORMAL_CAN_MODE;
    CAN_ITConfig(CAN2, CAN2_IT_FMP | CAN_IT_WKU | CAN_IT_SLK, ENABLE);
    __CAN2_NORMAL_MODE();
  }
  else
  {
    can_mode_config[CAN2_ID]=OFF_CAN_MODE;
    __CAN2_STANDBY_MODE();
  }
#endif //(HW_CAN_COUNT > 1)

#if (HW_CAN_COUNT > 2)
  CAN_ClearFlag(CAN3, CAN_IT_TME|CAN_IT_FF0|CAN_IT_FOV0|CAN_IT_FF1|CAN_IT_FOV1|CAN_IT_WKU|CAN_IT_SLK|CAN_IT_EWG|CAN_IT_EPV|CAN_IT_BOF|CAN_IT_LEC|CAN_IT_ERR);

  can_sleep.wakeup_event[2]=false;
  can_sleep.active_timer[2]=can_sleep.can_inactive_timeout;

  if(can3_baudrate)
  {
    can_mode_config[CAN3_ID]=NORMAL_CAN_MODE;
    CAN_ITConfig(CAN3, CAN3_IT_FMP | CAN_IT_WKU | CAN_IT_SLK, ENABLE);
    __CAN3_NORMAL_MODE();
  }
  else
  {
    can_mode_config[CAN3_ID]=OFF_CAN_MODE;
    __CAN3_STANDBY_MODE();
  }
#endif //(HW_CAN_COUNT > 2)

  return 1;
}

static void can_hw_deinit(void)
{
  can_mode_config[CAN1_ID]=OFF_CAN_MODE;
  __CAN1_STANDBY_MODE();
  CAN_ITConfig(CAN1, CAN1_IT_FMP | CAN_IT_WKU | CAN_IT_SLK, DISABLE);
  CAN_DeInit(CAN1);
  can_sleep.wakeup_event[0]=false;

#if (HW_CAN_COUNT > 1)
  can_mode_config[CAN2_ID]=OFF_CAN_MODE;
  __CAN2_STANDBY_MODE();
  CAN_ITConfig(CAN2, CAN2_IT_FMP | CAN_IT_WKU | CAN_IT_SLK, DISABLE);
  CAN_DeInit(CAN2);
  can_sleep.wakeup_event[1]=false;
#endif //(HW_CAN_COUNT > 1)

#if (HW_CAN_COUNT > 2)
  can_mode_config[CAN3_ID]=OFF_CAN_MODE;
  __CAN3_STANDBY_MODE();
  CAN_ITConfig(CAN3, CAN3_IT_FMP | CAN_IT_WKU | CAN_IT_SLK, DISABLE);
  CAN_DeInit(CAN3);
  can_sleep.wakeup_event[2]=false;
#endif //(HW_CAN_COUNT > 2)

  LedCAN_Off();
}

static void can_sleep_wakeup_hadler(const uint8_t can_id)
{
  CAN_TypeDef* CANx=get_can_periph_from_id(can_id);
  if(CANx==NULL) return;

  if(can_sleep.wakeup_event[can_id])
  {
    can_sleep.active_timer[can_id]=can_sleep.can_inactive_timeout;
    LOG("%s: wakeup\n", get_name_from_can_id(can_id));
    can_sleep.wakeup_event[can_id]=false;
  }

  if(!can_sleep.active_timer[can_id] && !is_can_in_sleep(CANx))
  {
    if(CAN_Sleep_Ok==CAN_Sleep(CANx))
    {
      LOG("%s: enter sleep %s\n", get_name_from_can_id(can_id), "done");
    }
    else
    {
      LOG("%s: enter sleep %s\n", get_name_from_can_id(can_id), "err");
    }
  }
 
  if(!is_can_in_sleep(CANx))
  {
    if(CANx == CAN1) { System.syscom_state.can1_last_activity = GetUnixTime(); }
    if(CANx == CAN2) { System.syscom_state.can2_last_activity = GetUnixTime(); }
    if(CANx == CAN3) { System.syscom_state.can3_last_activity = GetUnixTime(); }
  }
}

static void can_hw_diag_handler(const uint8_t can_id, bool is_reset_state)
{
  static uint32_t log_timerS[HW_CAN_COUNT];
  static can_hw_state_t can_diag[HW_CAN_COUNT];

  CAN_TypeDef* CANx;

  CANx=get_can_periph_from_id(can_id);
  if(CANx==NULL) return;

  if(can_mode_config[can_id] != OFF_CAN_MODE)
  {
    uint32_t esr=CANx->ESR;

    System.can_hw_state[can_id].err_flags.byte=((esr)&(CAN_ESR_EWGF|CAN_ESR_EPVF|CAN_ESR_BOFF));

    System.can_hw_state[can_id].rec=(uint8_t)(((esr)&CAN_ESR_REC)>>24);

    System.can_hw_state[can_id].tec=(uint8_t)(((esr)&CAN_ESR_TEC)>>16);

    System.can_hw_state[can_id].last_err_code=(CAN_GetLastErrorCode(CANx))>>4;

    if(is_can_in_sleep(get_can_periph_from_id(can_id))) System.can_hw_state[can_id].state=2;
    else                                                System.can_hw_state[can_id].state=1;

    System.can_hw_state[can_id].rx_drop_counter=can_rx_sfifo[can_id].drop_counter;
  }
  else
  {
    System.can_hw_state[can_id].state=0;
    memset(&System.can_hw_state[can_id], 0, sizeof(System.can_hw_state[can_id]));
  }

  if(is_reset_state)
  {
    log_timerS[can_id]=GetSecondsFromStartup();
    can_diag[can_id].rec=0;
    can_diag[can_id].tec=0;
    can_diag[can_id].err_flags.byte=0;
    can_diag[can_id].last_err_code=0x0F;
  }
  else
  {
    if(timeAfter(GetSecondsFromStartup(), log_timerS[can_id]))
    {
      if(can_diag[can_id].err_flags.byte!=System.can_hw_state[can_id].err_flags.byte)
      {
        LOG("%s: err flags |warning:%hhu|passive:%hhu|bus-off:%hhu|->|warning:%hhu|passive:%hhu|bus-off:%hhu|\n", get_name_from_can_id(can_id), \
          can_diag[can_id].err_flags.bf.warning, can_diag[can_id].err_flags.bf.passive, can_diag[can_id].err_flags.bf.bus_off, \
            System.can_hw_state[can_id].err_flags.bf.warning, System.can_hw_state[can_id].err_flags.bf.passive, System.can_hw_state[can_id].err_flags.bf.bus_off);

        can_diag[can_id].err_flags.byte=System.can_hw_state[can_id].err_flags.byte;
      }

      if(can_diag[can_id].rec!=System.can_hw_state[can_id].rec)
      {
        LOG("%s: REC %hhu->%hhu\n", get_name_from_can_id(can_id), can_diag[can_id].rec, System.can_hw_state[can_id].rec);

        can_diag[can_id].rec=System.can_hw_state[can_id].rec;
      }

      if(can_diag[can_id].tec!=System.can_hw_state[can_id].tec)
      {
        LOG("%s: TEC %hhu->%hhu\n", get_name_from_can_id(can_id), can_diag[can_id].tec, System.can_hw_state[can_id].tec);

        can_diag[can_id].tec=System.can_hw_state[can_id].tec;
      }

      if(can_diag[can_id].last_err_code!=System.can_hw_state[can_id].last_err_code)
      {
        LOG("%s err: %s (%hhu)\n", get_name_from_can_id(can_id), get_verbose_last_err_code(System.can_hw_state[can_id].last_err_code), System.can_hw_state[can_id].last_err_code);

        can_diag[can_id].last_err_code=System.can_hw_state[can_id].last_err_code;
      }

      log_timerS[can_id]=GetSecondsFromStartup()+15;
    }
  }
}

#if defined(YANDEX_OBD_PRESENT)
static void get_obd_conf(uint8_t* const can_id, bool* const is_broadcast_req, uint16_t ecu_id[2])
{
  switch(can_program)
  {
  case KIA_RIO_2017_PROGRAM:
  case HYUNDAI_SOLARIS_2017_PROGRAM:
    {
      can_id[0]=CAN2_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case RENAULT_KAPTUR_2016_PROGRAM://VIN + DTC ?
  case RENAULT_DOKKER_2018_PROGRAM:
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case HYUNDAI_GENESIS_2017_PROGRAM: //VIN + DTC +
    {
      //через моторную шину не работает, нужно подключать к OBD.
      can_id[0]=CAN3_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;//Ответ на запрос 0x7DF приходит от 0x7E8 и 0x7E9
      break;
    }
  case AUDI_A3_2018_PROGRAM:
  case SKODA_OCTAVIA_2018_PROGRAM: //VIN + ECM DTC + TCM DTC +
  case SKODA_RAPID_2018_PROGRAM:
    {
      can_id[0]=CAN2_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case AUDI_Q3_2018_PROGRAM: //VIN + DTC ?
  case VW_POLO_2018_PROGRAM: //VIN + ECM DTC + TCM DTC +
  case VW_CADDY_2018_PROGRAM:
  case VW_TRANSPORTER_2018_PROGRAM: //VIN + DTC +
  case SKODA_RAPID_2013_PROGRAM:
    {
      can_id[0]=CAN2_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case PORSCHE_MACAN_2018_PROGRAM:
  case PORSCHE_911_2018_PROGRAM: //VIN + DTC ?
  case PORSCHE_PANAMERA_2019_PROGRAM: //VIN + DTC ?
    {
      //CAN3 подключена только на машинах с 2019 года
      can_id[0]=CAN3_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case MB_E200_2018_PROGRAM: //VIN + DTC +, проверено на C180, на E200 не работает (возможно нужено слать 7DF). На E200 2020 года зажигается сообщение о подключении диаг. оорудования.
    {
      if(IS_MB_C180())
      {
        can_id[0]=CAN2_ID;
        ecu_id[0]=0x7E0;
        ecu_id[1]=0x7E1;
        is_broadcast_req[0]=false;
      }
      else
      {
        can_id[0]=CAN_ID_UNKNOWN;
        ecu_id[0]=0;
        ecu_id[1]=0;
      }
      break;
    }
  case TOYOTA_RAV4_2019_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case BMW_520I_2018_PROGRAM: //VIN + DTC + (+0x7E9)
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case FORD_TRANSIT_2017_PROGRAM:
    {
      can_id[0]=CAN2_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case CITROEN_JUMPY_2018_PROGRAM:
    {
      can_id[0]=CAN2_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case LADA_LARGUS_2017_PROGRAM:
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case VOLVO_XC60_2018_PROGRAM: //VIN ? DTC ?
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case VOLVO_XC40_2018_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN3_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case NISSAN_QASHQAI_2018_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case RANGE_ROVER_VELAR_2018_PROGRAM:
    {
      can_id[0]=CAN3_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case HYUNDAI_CRETA_2019_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN2_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case BMW_X1_2019_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case NISSAN_LEAF_2019_PROGRAM:
    {
      can_id[0]=CAN_ID_UNKNOWN;
      ecu_id[0]=0;
      ecu_id[1]=0;
      is_broadcast_req[0]=true;
      break;
    }
  case HYUNDAI_SONATA_SD_2020_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case RENAULT_LOGAN_2013_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case LADA_LARGUS_2020_PROGRAM:
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case RENAULT_DUSTER_2021_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case JEEP_WRANGLER_2018_PROGRAM: //VIN + DTC +
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case AUDI_A6_2018_PROGRAM: //VIN ? DTC ?
    {
      can_id[0]=CAN2_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  case RENAULT_ARKANA_2018_PROGRAM: //VIN ? DTC ?
    {
      can_id[0]=CAN1_ID;
      ecu_id[0]=0x7E0;
      ecu_id[1]=0x7E1;
      is_broadcast_req[0]=true;
      break;
    }
  default:
    {
      can_id[0]=CAN_ID_UNKNOWN;
      ecu_id[0]=0;
      ecu_id[1]=0;
      is_broadcast_req[0]=true;
      break;
    }
  }
}

static void obd_rx_handler(const uint8_t can_id, const can_rx_frame_t* rx)
{
  static char log_buf[256]={0};
  static int log_buf_offset=0;

  if(!can_timers[OBD_ENG_ON_TIMER])
  {
    if(log_buf_offset>0)
    {
      LOG("%s", log_buf);
      log_buf_offset=0;
    }

    return;
  }

  for(uint8_t link_idx=0; link_idx<sizeof(obd_link)/sizeof(obd_link[0]); link_idx++)
  {
    if(!obd_ctx.ecu_id[link_idx])
    {
      continue;
    }

    if(can_id==obd_ctx.can_id && rx->id==(obd_ctx.ecu_id[link_idx]+8))
    {
      isotp_on_can_message(&obd_link[link_idx], rx->data, rx->dlen);
    }

    /* Poll link to handle multiple frame transmition */
    isotp_poll(&obd_link[link_idx]);

    const uint8_t* payload; uint16_t payload_size;

    if(ISOTP_RET_OK==isotp_receive_ptr(&obd_link[link_idx], &payload, &payload_size))
    {
      //if Negative Response or SAE J1979 Response
      if(payload[0]==0x7F || (payload[0]>=0x40 && payload[0]<=0x4F))
      {
        if(payload_size==3 && payload[0]==0x7F)
        {
          if(payload[1]==REQUEST_VEHICLE_INFORMATION)
          {
            if(link_idx==0)
            {
              System.Grab(portMAX_DELAY);
              snprintf(System.signal_state.vin_sensor, sizeof(System.signal_state.vin_sensor), "negResp: 0x%02hhX", payload[2]);
              System.Release();
            }
          }
          else if(payload[1]==SHOW_STORED_DIAGNOSTIC_TROUBLE_CODES)
          {
            System.Grab(portMAX_DELAY);
            snprintf(System.signal_state.dtc_list[link_idx], sizeof(System.signal_state.dtc_list[link_idx]), "negResp: 0x%02hhX", payload[2]);
            System.Release();
          }
          else if(payload[1]==CLEAR_DIAGNOSTIC_TROUBLE_CODES_AND_STORED_VALUES)
          {
            System.Grab(portMAX_DELAY);
            snprintf(System.signal_state.dtc_list[link_idx], sizeof(System.signal_state.dtc_list[link_idx]), "negResp: 0x%02hhX", payload[2]);
            System.Release();
          }

          //LOG("%s(0x%lX): Negative response: %s(0x%hhX) for SID=0x%hhX\n", get_name_from_can_id(can_id), rx->id, get_verbose_obd_negative_code(payload[2]), payload[2], payload[1]);
          if(sizeof(log_buf)>log_buf_offset)
          {
            log_buf_offset=snprintf(&log_buf[log_buf_offset], sizeof(log_buf)-log_buf_offset, "%s(0x%lX): Negative response: %s(0x%hhX) for SID=0x%hhX\n", get_name_from_can_id(can_id), rx->id, get_verbose_obd_negative_code(payload[2]), payload[2], payload[1]);
            if(log_buf_offset<0) log_buf_offset=0;
          }
        }
        else if(payload_size>3 && payload[0]==get_responce_for_mode(REQUEST_VEHICLE_INFORMATION) && payload[1]==0x02 && payload[2]==0x01)
        {
          payload_size-=3;
          payload+=3;

          if(link_idx==0)
          {
            System.Grab(portMAX_DELAY);
            snprintf(System.signal_state.vin_sensor, sizeof(System.signal_state.vin_sensor), "%.*s", payload_size, (const char*)payload);
            System.Release();
          }

          //LOG("%s(0x%lX): VIN: %.*s\n", get_name_from_can_id(can_id), rx->id, payload_size, (const char*)payload);

          if(sizeof(log_buf)>log_buf_offset)
          {
            log_buf_offset=snprintf(&log_buf[log_buf_offset], sizeof(log_buf)-log_buf_offset, "%s(0x%lX): VIN: %.*s\n", get_name_from_can_id(can_id), rx->id, payload_size, (const char*)payload);
            if(log_buf_offset<0) log_buf_offset=0;
          }
        }
        else if(payload_size>1 && payload[0]==get_responce_for_mode(SHOW_STORED_DIAGNOSTIC_TROUBLE_CODES))
        {
          payload_size-=1;
          payload+=1;

          System.Grab(portMAX_DELAY);
          int16_t dtc_count=obd_dtcs_decode(payload, payload_size, System.signal_state.dtc_list[link_idx], sizeof(System.signal_state.dtc_list[link_idx]), ' ');

          if(dtc_count>=0)
          {
            if(dtc_count==0) sprintf(System.signal_state.dtc_list[link_idx], "null");
            System.Release();

            //LOG("%s(0x%lX): DTCs Response: %s\n", get_name_from_can_id(can_id), rx->id, System.signal_state.dtc_list[link_idx]);
            if(sizeof(log_buf)>log_buf_offset)
            {
              log_buf_offset=snprintf(&log_buf[log_buf_offset], sizeof(log_buf)-log_buf_offset, "%s(0x%lX): DTCs Response: %s\n", get_name_from_can_id(can_id), rx->id, System.signal_state.dtc_list[link_idx]);
              if(log_buf_offset<0) log_buf_offset=0;
            }
          }
          else
          {
            System.Release();
          }
        }
        else if(payload_size==1 && payload[0]==get_responce_for_mode(CLEAR_DIAGNOSTIC_TROUBLE_CODES_AND_STORED_VALUES))
        {
          payload_size-=1;
          payload+=1;

          System.Grab(portMAX_DELAY);
          sprintf(System.signal_state.dtc_list[link_idx], "cleared");
          System.Release();

          //LOG("%s(0x%lX): clear DTCs Response\n", get_name_from_can_id(can_id), rx->id);
          if(sizeof(log_buf)>log_buf_offset)
          {
            log_buf_offset=snprintf(&log_buf[log_buf_offset], sizeof(log_buf)-log_buf_offset, "%s(0x%lX): clear DTCs Response\n", get_name_from_can_id(can_id), rx->id);
            if(log_buf_offset<0) log_buf_offset=0;
          }
        }
      }
    }
  }
}

bool obd_cmd_callback(const uint8_t* const args, const uint16_t args_len, bool* const is_busy, const char** verbose)
{
  is_busy[0]=false;

  if(args_len!=sizeof(uint8_t))
  {
    verbose[0] = "OBD cmd err, wrong arg len";
    return false;
  }

  if(obd_ctx.can_id==CAN_ID_UNKNOWN)
  {
    verbose[0] = "OBD cmd not support";
    return false;
  }

  if(obd_ctx.is_force_cmd)
  {
    is_busy[0]=true;
    return true;
  }

  switch(args[0])
  {
    case 1: obd_ctx.is_force_cmd=1; verbose[0] = "OBD update VIN"; break;
    case 2: obd_ctx.is_force_cmd=2; verbose[0] = "OBD update DTC"; break;
    case 3: obd_ctx.is_force_cmd=3; verbose[0] = "OBD clear DTC"; break;
    default: verbose[0] = "OBD cmd err, wrong args"; return false;
  }

  return true;
}

static void obd_tx_handler(void)
{
  if(obd_ctx.can_id>=HW_CAN_COUNT || obd_ctx.ecu_id[0]==0) return;

  uint8_t engine_is_on=!!System.can_state.sec_flags.engine_is_on;

  if(engine_is_on!=obd_ctx.engine_is_on)
  {
    if(engine_is_on)
    {
      restart_can_timer(OBD_ENG_ON_TIMER, 5000);
      obd_ctx.step=1;
    }

    obd_ctx.engine_is_on=engine_is_on;
  }

  uint8_t is_force_cmd=obd_ctx.is_force_cmd;

  if((engine_is_on && is_can_timer_expired(OBD_ENG_ON_TIMER) && obd_ctx.step)
     || is_force_cmd)
  {
    if(obd_ctx.step==1 || is_force_cmd==1)
    {
      static const uint8_t payload[]={REQUEST_VEHICLE_INFORMATION, 0x02};//get VIN
      static const uint8_t payload_size=sizeof(payload);

      isotp_init_link(&obd_link[0], obd_ctx.can_id, 0, obd_ctx.ecu_id[0],
                      obd_isotpSendBuf[0], sizeof(obd_isotpSendBuf[0]),
                      obd_isotpRecvBuf[0], sizeof(obd_isotpRecvBuf[0]));

      if(obd_ctx.ecu_id[1])
      {
        isotp_init_link(&obd_link[1], obd_ctx.can_id, 0, obd_ctx.ecu_id[1],
                        obd_isotpSendBuf[1], sizeof(obd_isotpSendBuf[1]),
                        obd_isotpRecvBuf[1], sizeof(obd_isotpRecvBuf[1]));
      }

      if(obd_ctx.is_broadcast) isotp_send_with_id(&obd_link[0], 0x7DF, payload, payload_size);
      else                     isotp_send(&obd_link[0], payload, payload_size);

      if(!is_force_cmd) {obd_ctx.step++;}
      else              {obd_ctx.is_force_cmd=0; obd_ctx.step=0;}
      restart_can_timer(OBD_ENG_ON_TIMER, 1000);
    }
    else if(obd_ctx.step==2 || is_force_cmd==2)
    {
      static const uint8_t payload[]={SHOW_STORED_DIAGNOSTIC_TROUBLE_CODES};//get DTC
      static const uint8_t payload_size=sizeof(payload);

      isotp_init_link(&obd_link[0], obd_ctx.can_id, 0, obd_ctx.ecu_id[0],
                      obd_isotpSendBuf[0], sizeof(obd_isotpSendBuf[0]),
                      obd_isotpRecvBuf[0], sizeof(obd_isotpRecvBuf[0]));

      if(obd_ctx.ecu_id[1])
      {
        isotp_init_link(&obd_link[1], obd_ctx.can_id, 0, obd_ctx.ecu_id[1],
                        obd_isotpSendBuf[1], sizeof(obd_isotpSendBuf[1]),
                        obd_isotpRecvBuf[1], sizeof(obd_isotpRecvBuf[1]));
      }

      if(obd_ctx.is_broadcast) isotp_send_with_id(&obd_link[0], 0x7DF, payload, payload_size);
      else                     isotp_send(&obd_link[0], payload, payload_size);

      if(!is_force_cmd) {obd_ctx.step++;}
      else              {obd_ctx.is_force_cmd=0; obd_ctx.step=0;}
      restart_can_timer(OBD_ENG_ON_TIMER, 1000);
    }
    else if(/*obd_ctx.step==3 ||*/ is_force_cmd==3)
    {
      static const uint8_t payload[]={CLEAR_DIAGNOSTIC_TROUBLE_CODES_AND_STORED_VALUES};//clear DTC
      static const uint8_t payload_size=sizeof(payload);

      isotp_init_link(&obd_link[0], obd_ctx.can_id, 0, obd_ctx.ecu_id[0],
                      obd_isotpSendBuf[0], sizeof(obd_isotpSendBuf[0]),
                      obd_isotpRecvBuf[0], sizeof(obd_isotpRecvBuf[0]));

      if(obd_ctx.ecu_id[1])
      {
        isotp_init_link(&obd_link[1], obd_ctx.can_id, 0, obd_ctx.ecu_id[1],
                        obd_isotpSendBuf[1], sizeof(obd_isotpSendBuf[1]),
                        obd_isotpRecvBuf[1], sizeof(obd_isotpRecvBuf[1]));
      }

      if(obd_ctx.is_broadcast) isotp_send_with_id(&obd_link[0], 0x7DF, payload, payload_size);
      else                     isotp_send(&obd_link[0], payload, payload_size);

      obd_ctx.is_force_cmd=0; obd_ctx.step=0;
      restart_can_timer(OBD_ENG_ON_TIMER, 1000);
    }
    else
    {
      obd_ctx.step=0;
    }
  }
  else if(!engine_is_on)
  {
    obd_ctx.step=0;
  }

}
#endif //defined(YANDEX_OBD_PRESENT)

void forced_eco_on_handler(void)
{
  if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    static struct
    {
      uint8_t engine_is_on:1;
      uint8_t step:7;
    }eco_on_ctx =
    {
      .engine_is_on=0,
      .step=0,
    };

    uint8_t engine_is_on=!!System.can_state.sec_flags.engine_is_on;

    if(engine_is_on!=eco_on_ctx.engine_is_on)
    {
      if(engine_is_on)
      {
        restart_can_timer(ECO_ON_ENG_ON_TIMER, 5000);
        eco_on_ctx.step=1;
      }

      eco_on_ctx.engine_is_on=engine_is_on;
    }

    if(engine_is_on && !is_can_timer_expired(ECO_ON_ENG_ON_TIMER) && eco_on_ctx.step)
    {
      if(System.signal_state.eco_mode)
      {
        eco_on_ctx.step=0;
        LOG("ECO mode is allready on\n");
      }
      else if(can_cmd_proc.cmd_q[0]==EMTY_CAN_CMD_TYPE)
      {
        can_cmd_proc.cmd_q[0]=ECO_ON_CMD_TYPE;
        eco_on_ctx.step=0;
        LOG("ECO mode is forced on\n");
      }
    }
    else if(!engine_is_on)
    {
      eco_on_ctx.step=0;
    }
  }
}

void engine_stop_by_xxx_handler(void)
{
  if(can_program==NISSAN_LEAF_2019_PROGRAM || (can_program==HYUNDAI_GENESIS_2017_PROGRAM && !is_genesis_w_electric_hand_break))
  {
    //Если в машине есть прогрев, то этота логика будет кофнликтовать и глушить машину во время прогрева
    if(System.can_state.sec_flags.engine_is_on && System.can_state.sec_flags.hand_break \
      && !System.can_state.sec_flags.parking && System.can_state.numeric_data.can_speed<5 \
        && is_can_timer_expired(ENG_STOP_TIMER))
    {
      if(is_can_cmd_proc_q_emty())
      {
        can_cmd_proc.cmd_q[0]=STATRTSTOP_BUTTON_PUSH_CMD_TYPE;

        System.signal_state.drive_on_handbrake=1;
        LOG("stop engine by handbrake\n");

        restart_can_timer(ENG_STOP_TIMER, 3000);
      }
    }

    if(is_can_timer_expired(ENG_STOP_TIMER))
    {
      System.signal_state.drive_on_handbrake=0;
    }
  }
  else if(can_program==MB_E200_2018_PROGRAM && !IS_MB_C180())
  {
    //todo: Если в машине есть прогрев, то этота логика будет кофнликтовать и глушить машину во время прогрева
    if((lease_state==IN_END_LEASE_STATE || lease_state==IN_FORCED_END_LEASE_STATE) &&
       System.can_state.sec_flags.engine_is_on && System.can_state.numeric_data.can_speed<5 \
        && is_can_timer_expired(ENG_STOP_TIMER))
    {
      if(!is_engine_started_by_terminal && is_can_cmd_proc_q_emty())
      {
        can_cmd_proc.cmd_q[1]=DELAY_1; can_cmd_proc.delay[0]=500;
        can_cmd_proc.cmd_q[2]=STATRTSTOP_BUTTON_PUSH_CMD_TYPE;
        can_cmd_proc.cmd_q[3]=KEY_POWER_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=KEY_POWER_ON_CMD_TYPE;

        LOG("stop engine by unauthorized start\n");

        restart_can_timer(ENG_STOP_TIMER, 5000);
      }
    }
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    //todo: Если в машине есть прогрев, то этота логика будет кофнликтовать и глушить машину во время прогрева
    if((lease_state==IN_END_LEASE_STATE || lease_state==IN_FORCED_END_LEASE_STATE) &&
       System.can_state.sec_flags.engine_is_on && System.can_state.numeric_data.can_speed<5 \
        && is_can_timer_expired(ENG_STOP_TIMER))
    {
      if(!is_engine_started_by_terminal && is_can_cmd_proc_q_emty())
      {
        if(IS_BMW_320D())
        {
          //тройное нажатие
          can_cmd_proc.cmd_q[1]=STATRTSTOP_BUTTON_PUSH_CMD_TYPE;
          can_cmd_proc.cmd_q[2]=STATRTSTOP_BUTTON_PUSH_CMD_TYPE;
        }
        can_cmd_proc.cmd_q[0]=STATRTSTOP_BUTTON_PUSH_CMD_TYPE;

        LOG("stop engine by unauthorized start\n");

        restart_can_timer(ENG_STOP_TIMER, 5000);
      }
    }
  }
}

void wakeup_electric_car_on_charge_handler(void)
{
  static uint32_t timerS=GetSecondsFromStartup();

  if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    //будим шину не чаще чем раз в 5 минут
    if(System.can_state.sec_flags.charge_cable_present && is_can_rx_in_idle(CAN1_ID) && \
      timeAfter(GetSecondsFromStartup(), timerS))
    {
      if(is_can_cmd_proc_q_emty())
      {
        if(lease_state==IN_END_LEASE_STATE || lease_state==IN_FORCED_END_LEASE_STATE)
        {
          can_cmd_proc.cmd_q[0]=STATRTSTOP_BUTTON_PUSH_CMD_TYPE;

          /*
          uint8_t i=1;
          can_cmd_proc.cmd_q[i++]=LEAF_WAKEUP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1000;
          can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=KEY_POWER_ON_CMD_TYPE;
          */
        }
        else
        {
          can_cmd_proc.cmd_q[0]=LEAF_WAKEUP_CMD_TYPE;
        }

        timerS=GetSecondsFromStartup()+5*60;
      }
    }
  }
}

static uint8_t can_hw_programm_init(void)
{
  if(can_program==KIA_RIO_2017_PROGRAM)                                                         {return can_hw_init(KIA_RIO_CAN1_SPEED, kia_rio_2017_can1, sizeof(kia_rio_2017_can1), KIA_RIO_CAN2_SPEED, kia_rio_hyundai_solaris_2017_can2, sizeof(kia_rio_hyundai_solaris_2017_can2), 0, NULL, 0);}
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM) {return can_hw_init(RENAULT_KAPTUR_CAN1_SPEED, renault_kaptur_2016_can1, sizeof(renault_kaptur_2016_can1), 0, NULL, 0, 0, NULL, 0);}
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)                                            {return can_hw_init(HYUNDAI_GENESIS_CAN1_SPEED, hyundai_genesis_2017_can1, sizeof(hyundai_genesis_2017_can1), HYUNDAI_GENESIS_CAN2_SPEED, hyundai_genesis_2017_can2, sizeof(hyundai_genesis_2017_can2), HYUNDAI_GENESIS_CAN3_SPEED, hyundai_genesis_2017_can3, sizeof(hyundai_genesis_2017_can3));}
  else if(can_program==AUDI_A3_2018_PROGRAM  || can_program==SKODA_OCTAVIA_2018_PROGRAM || \
    can_program==SKODA_RAPID_2018_PROGRAM)                                                      {return can_hw_init(AUDI_A3_CAN1_SPEED, audi_a3_2018_can1, sizeof(audi_a3_2018_can1), AUDI_A3_CAN2_SPEED, audi_a3_2018_can2, sizeof(audi_a3_2018_can2), 0, NULL, 0);}
  else if(can_program==AUDI_Q3_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM  || \
    can_program==VW_CADDY_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)                {return can_hw_init(AUDI_Q3_CAN1_SPEED, audi_q3_2018_can1, sizeof(audi_q3_2018_can1), AUDI_Q3_CAN2_SPEED, audi_q3_2018_can2, sizeof(audi_q3_2018_can2), 0, NULL, 0);}
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)     {return can_hw_init(PORSCHE_CAN1_SPEED, porsche_2018_can1, sizeof(porsche_2018_can1), PORSCHE_CAN2_SPEED, porsche_2018_can2, sizeof(porsche_2018_can2), PORSCHE_CAN3_SPEED, porsche_2018_can3, sizeof(porsche_2018_can3));}
  else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)                                           {return can_hw_init(PORSCHE_PANAMERA_CAN1_SPEED, porsche_panamera_2019_can1, sizeof(porsche_panamera_2019_can1), 0, NULL, 0, PORSCHE_PANAMERA_CAN3_SPEED, porsche_panamera_2019_can3, sizeof(porsche_panamera_2019_can3));}
  else if(can_program==MB_E200_2018_PROGRAM)                                                    {return can_hw_init(MB_E200_CAN1_SPEED, mb_e200_2018_can1, sizeof(mb_e200_2018_can1), MB_E200_CAN2_SPEED, mb_e200_2018_can2, sizeof(mb_e200_2018_can2), 0, NULL, 0);}
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)                                                {return can_hw_init(TOYOTA_RAV4_2019_CAN1_SPEED, toyota_rav4_2019_can1, sizeof(toyota_rav4_2019_can1), TOYOTA_RAV4_2019_CAN2_SPEED, toyota_rav4_2019_can2, sizeof(toyota_rav4_2019_can2), TOYOTA_RAV4_2019_CAN3_SPEED, toyota_rav4_2019_can3, sizeof(toyota_rav4_2019_can3));}
  else if(can_program==BMW_520I_2018_PROGRAM)                                                   {return can_hw_init(BMW_520I_CAN1_SPEED, bmw_520i_2018_can1, sizeof(bmw_520i_2018_can1), BMW_520I_CAN2_SPEED, bmw_520i_2018_can2, sizeof(bmw_520i_2018_can2), 0, NULL, 0);}
  else if(can_program==HYUNDAI_SOLARIS_2017_PROGRAM)                                            {return can_hw_init(HYUNDAI_SOLARIS_CAN1_SPEED, hyundai_solaris_2017_can1, sizeof(hyundai_solaris_2017_can1), HYUNDAI_SOLARIS_CAN2_SPEED, kia_rio_hyundai_solaris_2017_can2, sizeof(kia_rio_hyundai_solaris_2017_can2), 0, NULL, 0);}
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)                                               {return can_hw_init(FORD_TRANSIT_CAN1_SPEED, ford_transit_2017_can1, sizeof(ford_transit_2017_can1), FORD_TRANSIT_CAN2_SPEED, ford_transit_2017_can2, sizeof(ford_transit_2017_can2), 0, NULL, 0);}
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)                                              {return can_hw_init(CITROEN_JUMPY_CAN1_SPEED, citroen_jumpy_2018_can1, sizeof(citroen_jumpy_2018_can1), CITROEN_JUMPY_CAN2_SPEED, citroen_jumpy_2018_can2, sizeof(citroen_jumpy_2018_can2), 0, NULL, 0);}
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)                                                {return can_hw_init(NISSAN_LEAF_CAN1_SPEED, nissan_leaf_2019_can1, sizeof(nissan_leaf_2019_can1), NISSAN_LEAF_CAN2_SPEED, nissan_leaf_2019_can2, sizeof(nissan_leaf_2019_can2), 0, NULL, 0);}
  else if(can_program==LADA_LARGUS_2017_PROGRAM)                                                {return can_hw_init(LADA_LARGUS_CAN1_SPEED, lada_largus_2017_can1, sizeof(lada_largus_2017_can1), 0, NULL, 0, 0, NULL, 0);}
  else if(can_program==VOLVO_XC60_2018_PROGRAM)                                                 {return can_hw_init(VOLVO_XC60_CAN1_SPEED, volvo_xc60_2018_can1, sizeof(volvo_xc60_2018_can1), VOLVO_XC60_CAN2_SPEED, volvo_xc60_2018_can2, sizeof(volvo_xc60_2018_can2), VOLVO_XC60_CAN3_SPEED, volvo_xc60_2018_can3, sizeof(volvo_xc60_2018_can3));}
  else if(can_program==VOLVO_XC40_2018_PROGRAM)                                                 {return can_hw_init(VOLVO_XC40_CAN1_SPEED, volvo_xc40_2018_can1, sizeof(volvo_xc40_2018_can1), VOLVO_XC40_CAN2_SPEED, volvo_xc40_2018_can2, sizeof(volvo_xc40_2018_can2), VOLVO_XC40_CAN3_SPEED, volvo_xc40_2018_can3, sizeof(volvo_xc40_2018_can3));}
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)                                             {return can_hw_init(NISSAN_QASHQAI_CAN1_SPEED, nissan_qashqai_2018_can1, sizeof(nissan_qashqai_2018_can1), 0, NULL, 0, 0, NULL, 0);}
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)                                          {return can_hw_init(RANGE_ROVER_VELAR_CAN1_SPEED, range_rover_velar_2018_can1, sizeof(range_rover_velar_2018_can1), RANGE_ROVER_VELAR_CAN2_SPEED, range_rover_velar_2018_can2, sizeof(range_rover_velar_2018_can2), RANGE_ROVER_VELAR_CAN3_SPEED, range_rover_velar_2018_can3, sizeof(range_rover_velar_2018_can3));}
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)                                             {return can_hw_init(VW_TRANSPORTER_CAN1_SPEED, vw_transporter_2018_can1, sizeof(vw_transporter_2018_can1), VW_TRANSPORTER_CAN2_SPEED, vw_transporter_2018_can2, sizeof(vw_transporter_2018_can2), 0, NULL, 0);}
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)                                              {return can_hw_init(HYUNDAI_CRETA_CAN1_SPEED, hyundai_creta_2019_can1, sizeof(hyundai_creta_2019_can1), HYUNDAI_CRETA_CAN2_SPEED, hyundai_creta_2019_can2, sizeof(hyundai_creta_2019_can2), 0, NULL, 0);}
  else if(can_program==BMW_X1_2019_PROGRAM)                                                     {return can_hw_init(BMW_X1_CAN1_SPEED, bmw_x1_2019_can1, sizeof(bmw_x1_2019_can1), BMW_X1_CAN2_SPEED, bmw_x1_2019_can2, sizeof(bmw_x1_2019_can2), 0, NULL, 0);}
  else if(can_program==HYUNDAI_SONATA_SD_2020_PROGRAM)                                          {return can_hw_init(HYUNDAY_SONATA_SD_CAN1_SPEED, hyunday_sonata_sd_2020_can1, sizeof(hyunday_sonata_sd_2020_can1), HYUNDAY_SONATA_SD_CAN2_SPEED, hyunday_sonata_sd_2020_can2, sizeof(hyunday_sonata_sd_2020_can2), 0, NULL, 0);}
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM ||
          can_program==RENAULT_DUSTER_2021_PROGRAM ||
          can_program==RENAULT_ARKANA_2018_PROGRAM ||
          can_program==LADA_LARGUS_2020_PROGRAM)                                                {return can_hw_init(RENAULT_LOGAN_CAN1_SPEED, renault_logan_2013_can1, sizeof(renault_logan_2013_can1), 0, NULL, 0, 0, NULL, 0);}
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)                                              {return can_hw_init(JEEP_WRANGLER_2018_CAN1_SPEED, jeep_wrangler_2018_can1, sizeof(jeep_wrangler_2018_can1), 0, NULL, 0, 0, NULL, 0);}
  else if(can_program==AUDI_A6_2018_PROGRAM)                                                    {return can_hw_init(AUDI_A6_2018_CAN1_SPEED, audi_a6_2018_can1, sizeof(audi_a6_2018_can1), AUDI_A6_2018_CAN1_SPEED, audi_a6_2018_can2, sizeof(audi_a6_2018_can2), 0, NULL, 0);}
  else                                                                                          {return 0;}
}

namespace
{
bool do_off_auto_open_doors_for_creta()
{
  static TickType_t wait_timeout = 2 * 1000;
  static uint32_t wait_id = 0x7A8;
  static CanTxMsg wakeup_can_bus_command =      {.StdId=0x7E0, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};
  static CanTxMsg auth_request_command =        {.StdId=0x7A0, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x27, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00}};
  static CanTxMsg off_auto_open_doors_command = {.StdId=0x7A0, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x05, 0x2E, 0x12, 0xA0, 0x0A, 0x00, 0x00, 0x00}};

  if (is_can_in_sleep(CAN2)) {
    CAN_Transmit_Blocked(CAN2, &wakeup_can_bus_command, 50);
    vTaskDelay(100);
  }

  CAN_Transmit_Blocked(CAN2, &auth_request_command, 50);

  TickType_t start = xTaskGetTickCount();
  while (xTaskGetTickCount() - start < wait_timeout) {
    xSemaphoreTake(can_rx_bin_sem, static_cast<TickType_t>(10));

    while (filled_count_in_sfifo(&can_rx_sfifo[CAN2_ID]) > 0) {
      can_rx_frame_t rx = {0};
      read_from_sfifo(&can_rx_sfifo[CAN2_ID], &rx);
      can_rx_debug_trace(&rx, CAN2_ID);

      if (rx.id == wait_id) {
        if (rx.data[0] == 0x04 && rx.data[1] == 0x67 && rx.data[2] == 0x01) {
          CAN_Transmit_Blocked(CAN2, &off_auto_open_doors_command, 50);
          return true;
        } else if (rx.data[0] == 0x06 && rx.data[1] == 0x67 && rx.data[2] == 0x01) {
          rx.data[3] ^= 0x7C;
          rx.data[4] ^= 0x08;
          rx.data[5] ^= 0x7C;
          rx.data[6] ^= 0x08;

          CanTxMsg auth_answer_command = {.StdId=0x7A0, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x06, 0x27, 0x02, rx.data[3], rx.data[4], rx.data[5], rx.data[6], 0x00}};
          CAN_Transmit_Blocked(CAN2, &auth_answer_command, 50);
          vTaskDelay(1000);
          CAN_Transmit_Blocked(CAN2, &off_auto_open_doors_command, 50);
          return true;
        }
      }
    }
  }

  return false;
}

bool do_off_auto_open_doors_for_kia_rio()
{
  static CanTxMsg commands[] = {
    {.StdId=0x7A0, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x3E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}}, // wakeup can bus
    {.StdId=0x7A0, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x3E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}}, // wakeup can bus
    {.StdId=0x7A0, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x05, 0x2E, 0x12, 0xA0, 0x0A, 0x00, 0x00, 0x00}}, // set off auto open doors ([5] byte)
  };

  constexpr size_t commands_count = sizeof(commands) / sizeof(CanTxMsg);

  for (size_t i = 0; i < commands_count; ++i) {
    CAN_Transmit_Blocked(CAN2, &commands[i], 50);

    if (i < (commands_count - 1)) {
      vTaskDelay(1000);
    }
  }

  return true;
}

bool do_fix_fuel_view_for_logan_2013()
{
  uint32_t message_id = 0x743;

  static CanTxMsg wakeup_can_bus_command = {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};

  static CanTxMsg commands[] = {
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x10, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x3E, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x1D, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x10, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x06, 0x2E, 0x03, 0x1D, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x1D, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x10, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x05, 0x2E, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x05, 0x2E, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x05, 0x2E, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x05, 0x2E, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x06, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x07, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x06, 0x2E, 0x03, 0x06, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x06, 0x2E, 0x03, 0x07, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x06, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x07, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x07, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x04, 0x2E, 0x03, 0x05, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x03, 0x22, 0x03, 0x07, 0x00, 0x00, 0x00, 0x00}},
    {.StdId=message_id, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x02, 0x11, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00}},
  };

  constexpr size_t commands_count = sizeof(commands) / sizeof(CanTxMsg);

  if (is_can_in_sleep(CAN1)) {
    CAN_Transmit_Blocked(CAN1, &wakeup_can_bus_command, 50);
    vTaskDelay(100);
  }

  for (size_t i = 0; i < commands_count; ++i) {
    CAN_Transmit_Blocked(CAN1, &commands[i], 50);

    if (i < (commands_count - 1)) {
      vTaskDelay(100);
    }
  }

  return true;
}
}

static void fix_fuel_view()
{
  if (can_program == RENAULT_LOGAN_2013_PROGRAM ||
      can_program == LADA_LARGUS_2020_PROGRAM) {
    if (!do_fix_fuel_view_for_logan_2013()) {
      LOG("Error fix view for logan\n");
    }
  }

  move_next_command_to_head();
}

static void off_auto_open_doors()
{
  if (can_program == HYUNDAI_CRETA_2019_PROGRAM) {
    if (!do_off_auto_open_doors_for_creta()) {
      LOG("Error off auto open doors for Creta\n");
    } else {
      LOG("OK off auto open doors for Creta\n");
    }
  } else if (can_program == KIA_RIO_2017_PROGRAM) {
    do_off_auto_open_doors_for_kia_rio();
  }

  move_next_command_to_head();
}

static void start_engine_handler(void)
{
  const char* res_str=NULL;
  notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;

  if(can_program==KIA_RIO_2017_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    uint8_t ignition_output_id;
    uint8_t starter_output_id;

    if(!is_ignition_output_present(&ignition_output_id) || !is_starter_output_present(&starter_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    const uint8_t ign = (can_program==KIA_RIO_2017_PROGRAM)?(System.can_state.sec_flags.ignition):(System.signal_state.ignition); //у дастер CAN зажиагние работает только на заведенную

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        if(System.can_state.sec_flags.engine_is_on)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, engine is allredy run";
          break;
        }

        System.Grab(portMAX_DELAY);
        if(System.signal_state.external_voltage<10.5f)
        {
          System.Release();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, low battery";
          break;
        }
        System.Release();

        if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, one of the doors is open";
          break;
        }

        if((!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, hood or trunk is open";
          break;
        }

        if(ign)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, ignition is already on";
          break;
        }

        //      else if(!System.can_state.sec_flags.key_in_ign)
        //      {
        //        res_str="break, key not detect";
        //        break;
        //      }

        __PRINTF("switch on ignition...\n");

        System.signal_state.digital_output[ignition_output_id]=1;
        System.signal_state.digital_output[starter_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=3000+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        if(!ign)
        {
          System.signal_state.digital_output[ignition_output_id]=0;
          outputs_state_force_apply();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, ignition did not appeared...";
          break;
        }

        if(!System.can_state.sec_flags.parking)
        {
          System.signal_state.digital_output[ignition_output_id]=0;
          outputs_state_force_apply();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, transmission is not at parking";
          break;
        }

        __PRINTF("switch on starter...\n");

        System.signal_state.digital_output[starter_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=RIO_MAX_STARTER_SWITCH_ON_TIME_MS+xTaskGetTickCount();

        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(System.can_state.sec_flags.engine_is_on || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          System.signal_state.digital_output[starter_output_id]=0;
          outputs_state_force_apply();

          if(System.can_state.sec_flags.engine_is_on)
          {
            res_str="engine is on";
            is_engine_started_by_terminal_uid=can_cmd_proc.uid;
            memcpy(is_engine_started_by_terminal_nonce, can_cmd_proc.nonce, sizeof(is_engine_started_by_terminal_nonce));
            is_engine_started_by_terminal_server_notify_id=can_cmd_proc.server_notify_id;
            is_engine_started_by_terminal=true;

            if(can_cmd_proc.ext_param) {set_head_control_state(true, can_cmd_proc.ext_param/60000);}

            //если это прогрев, устанавливаем таймер работы двигателя
            if(can_cmd_proc.ext_param) warming_timer=can_cmd_proc.ext_param+xTaskGetTickCount();
            else                       warming_timer=0;
          }
          else
          {
            System.signal_state.digital_output[ignition_output_id]=0;
            outputs_state_force_apply();

            //если это была попытка прогрева, востанавливаем состоние проводной блокировки
            if(can_cmd_proc.ext_param)
            {
              is_wired_block_relay_enable=true;
              save_wired_block_relay_state(true);
              notify_state=IS_LAST_NOTIFY_STATE;
            }

            res_str="engine run fail";
          }

          break;
        }
      }
      else
      {
        //не должны сюда попадать
        System.signal_state.digital_output[ignition_output_id]=0;
        System.signal_state.digital_output[starter_output_id]=0;
        outputs_state_force_apply();

        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;
        }

        res_str="wrong can_cmd_proc.step...";
        break;
      }
      return; //continue command procces
    }
    //end command procces

    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    uint8_t start_stop_output_id;

    if(!is_start_stop_button_output_present(&start_stop_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        if(System.can_state.sec_flags.engine_is_on)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, engine is allredy run";
          break;
        }

        System.Grab(portMAX_DELAY);
        if(System.signal_state.external_voltage<10.5f)
        {
          System.Release();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, low battery";
          break;
        }
        System.Release();

        if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, one of the doors is open";
          break;
        }

        if((!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, hood or trunk is open";
          break;
        }

        if(System.can_state.sec_flags.ignition)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, ignition is already on";
          break;
        }

        //      else if(!System.can_state.sec_flags.key_in_ign)
        //      {
        //        res_str="key not detect";
        //        break;
        //      }

        __PRINTF("switch on ignition...\n");

        System.signal_state.digital_output[start_stop_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        //мы не видим коробку до тех пор, пока не будет включенно зажигание, а включить зажигание без старта двигателя не можем
//        if(!System.can_state.sec_flags.parking)
//        {
//          res_str="break, transmission is not at parking";
//          break;
//        }

        System.signal_state.digital_output[start_stop_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=get_max_engine_on_wait_time_ms()+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(System.can_state.sec_flags.engine_is_on || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          if(System.can_state.sec_flags.engine_is_on)
          {
            res_str="engine is on";
            is_engine_started_by_terminal_uid=can_cmd_proc.uid;
            memcpy(is_engine_started_by_terminal_nonce, can_cmd_proc.nonce, sizeof(is_engine_started_by_terminal_nonce));
            is_engine_started_by_terminal_server_notify_id=can_cmd_proc.server_notify_id;
            is_engine_started_by_terminal=true;

            if(can_cmd_proc.ext_param) {set_head_control_state(true, can_cmd_proc.ext_param/60000);}

            //если это прогрев, устанавливаем таймер работы двигателя
            if(can_cmd_proc.ext_param) warming_timer=can_cmd_proc.ext_param+xTaskGetTickCount();
            else                       warming_timer=0;
          }
          else
          {
            //если это была попытка прогрева, востанавливаем состоние проводной блокировки
            if(can_cmd_proc.ext_param)
            {
              is_wired_block_relay_enable=true;
              save_wired_block_relay_state(true);
              notify_state=IS_LAST_NOTIFY_STATE;
            }

            res_str="engine run fail";
          }

          break;
        }
      }
      else
      {
        //не должны сюда попадать
        System.signal_state.digital_output[start_stop_output_id]=0;
        outputs_state_force_apply();

        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;
        }

        res_str="wrong can_cmd_proc.step...";
        break;
      }
      return; //continue command procces
    }
    //end command procces

    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    uint8_t start_stop_output_id;
    uint8_t starter_output_id;

    if(!is_start_stop_button_output_present(&start_stop_output_id) || !is_starter_output_present(&starter_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        if(System.can_state.sec_flags.engine_is_on)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, engine is allredy run";
          break;
        }

        System.Grab(portMAX_DELAY);
        if(System.signal_state.external_voltage<10.5f)
        {
          System.Release();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, low battery";
          break;
        }
        System.Release();

        if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, one of the doors is open";
          break;
        }

        if((!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, hood or trunk is open";
          break;
        }

        if(System.can_state.sec_flags.ignition)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, ignition is already on";
          break;
        }

        __PRINTF("switch on ignition...\n");

        System.signal_state.digital_output[start_stop_output_id]=1;
        System.signal_state.digital_output[starter_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=200+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[starter_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=3000+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        if(!System.can_state.sec_flags.ignition)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, ignition did not appeared...";
          break;
        }

        if(!System.can_state.sec_flags.parking)
        {
          if(true)
          {
            //todo: надо дернуть зажиганием, чтобы выключить
            System.signal_state.digital_output[start_stop_output_id]=1;
            outputs_state_force_apply();
            vTaskDelay(200);
            System.signal_state.digital_output[start_stop_output_id]=0;
            outputs_state_force_apply();
          }
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          res_str="break, transmission is not at parking";
          break;
        }

        __PRINTF("switch on starter...\n");

        System.signal_state.digital_output[starter_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=get_max_engine_on_wait_time_ms()+xTaskGetTickCount();

        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(System.can_state.sec_flags.engine_is_on || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          System.signal_state.digital_output[starter_output_id]=0;
          outputs_state_force_apply();

          if(System.can_state.sec_flags.engine_is_on)
          {
            res_str="engine is on";
            is_engine_started_by_terminal_uid=can_cmd_proc.uid;
            memcpy(is_engine_started_by_terminal_nonce, can_cmd_proc.nonce, sizeof(is_engine_started_by_terminal_nonce));
            is_engine_started_by_terminal_server_notify_id=can_cmd_proc.server_notify_id;
            is_engine_started_by_terminal=true;

            if(can_cmd_proc.ext_param) {set_head_control_state(true, can_cmd_proc.ext_param/60000);}

            //если это прогрев, устанавливаем таймер работы двигателя
            if(can_cmd_proc.ext_param) warming_timer=can_cmd_proc.ext_param+xTaskGetTickCount();
            else                       warming_timer=0;
          }
          else
          {
            if(System.can_state.sec_flags.ignition)
            {
              //todo: надо дернуть зажиганием, чтобы выключить
              System.signal_state.digital_output[start_stop_output_id]=1;
              outputs_state_force_apply();
              vTaskDelay(200);
              System.signal_state.digital_output[start_stop_output_id]=0;
              outputs_state_force_apply();
            }

            //если это была попытка прогрева, востанавливаем состоние проводной блокировки
            if(can_cmd_proc.ext_param)
            {
              is_wired_block_relay_enable=true;
              save_wired_block_relay_state(true);
              notify_state=IS_LAST_NOTIFY_STATE;
            }

            res_str="engine run fail";
          }

          break;
        }
      }
      else
      {
        //не должны сюда попадать
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[starter_output_id]=0;
        outputs_state_force_apply();

        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;
        }

        res_str="wrong can_cmd_proc.step...";
        break;
      }
      return; //continue command procces
    }
    //end command procces

    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    //плохо, нужно переделать
    uint8_t start_stop_output_id;
    uint8_t pedal_break_output_id;

    if(!is_start_stop_button_output_present(&start_stop_output_id) || !is_pedal_break_output_present(&pedal_break_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        if(System.can_state.sec_flags.engine_is_on)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, engine is allredy run";
          break;
        }

        System.Grab(portMAX_DELAY);
        if(System.signal_state.external_voltage<10.5f)
        {
          System.Release();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, low battery";
          break;
        }
        System.Release();

        if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, one of the doors is open";
          break;
        }

        if((!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, hood or trunk is open";
          break;
        }

        if(System.signal_state.ignition)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, ignition is already on";
          break;
        }

        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }

      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[start_stop_output_id]=1;
        System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();
        can_cmd_proc.timer=5500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();
        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==6)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=1;
        outputs_state_force_apply();
        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==7)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==8)
      {
        System.signal_state.digital_output[start_stop_output_id]=1;
        System.signal_state.digital_output[pedal_break_output_id]=1;
        outputs_state_force_apply();
        can_cmd_proc.timer=300+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==9)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==10)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=1;
        outputs_state_force_apply();
        can_cmd_proc.timer=get_max_engine_on_wait_time_ms()+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==11)
      {
        if(System.can_state.sec_flags.engine_is_on || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          if(System.can_state.sec_flags.engine_is_on)
          {
            res_str="engine is on";
            is_engine_started_by_terminal_uid=can_cmd_proc.uid;
            memcpy(is_engine_started_by_terminal_nonce, can_cmd_proc.nonce, sizeof(is_engine_started_by_terminal_nonce));
            is_engine_started_by_terminal_server_notify_id=can_cmd_proc.server_notify_id;
            is_engine_started_by_terminal=true;

            if(can_cmd_proc.ext_param) {set_head_control_state(true, can_cmd_proc.ext_param/60000);}

            //если это прогрев, устанавливаем таймер работы двигателя
            if(can_cmd_proc.ext_param) warming_timer=can_cmd_proc.ext_param+xTaskGetTickCount();
            else                       warming_timer=0;

            System.signal_state.digital_output[start_stop_output_id]=0;
            System.signal_state.digital_output[pedal_break_output_id]=0;
            outputs_state_force_apply();

            break;
          }
          else
          {
            System.signal_state.digital_output[start_stop_output_id]=0;
            System.signal_state.digital_output[pedal_break_output_id]=0;
            outputs_state_force_apply();

            if(!System.can_state.sec_flags.ignition)
            {
              can_cmd_proc.step=16;//к завершению
            }
            else
            {
              //зажигание осталось включенным, проверим еще раз через 500 мс
              can_cmd_proc.timer=500+xTaskGetTickCount();
              can_cmd_proc.step++;
            }
          }
        }
      }
      else if(can_cmd_proc.step==12)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==13)
      {
        if(!System.can_state.sec_flags.ignition)
        {
          can_cmd_proc.step=16;//к завершению
        }
        else
        {
          //зажигание осталось включенным, будем пытаться выключить
          System.signal_state.digital_output[start_stop_output_id]=1;
          outputs_state_force_apply();

          can_cmd_proc.timer=250+xTaskGetTickCount();
          can_cmd_proc.step++;
        }
      }
      else if(can_cmd_proc.step==14)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==15)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        outputs_state_force_apply();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==16)
      {
        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;

          //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
          uint8_t key_power_output_id;
          if(is_key_power_present(&key_power_output_id))
          {
            if(is_key_power_off_no_blocking())
            {
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
            }
            else
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
            save_key_power_state(false);
          }

          if(is_ext_key_power_present(&key_power_output_id))
          {
            System.signal_state.digital_output[key_power_output_id]=0;
          }
        }

        if(System.can_state.sec_flags.ignition)
        {
          res_str="engine run fail, ignition left on";
        }
        else
        {
          res_str="engine run fail";
        }

        break;
      }
      else
      {
        //не должны сюда попадать
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;

          //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
          uint8_t key_power_output_id;
          if(is_key_power_present(&key_power_output_id))
          {
            if(is_key_power_off_no_blocking())
            {
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
            }
            else
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
            save_key_power_state(false);
          }

          if(is_ext_key_power_present(&key_power_output_id))
          {
            System.signal_state.digital_output[key_power_output_id]=0;
          }
        }

        res_str="wrong can_cmd_proc.step...";
        break;
      }
      return; //continue command procces
    }
    //end command procces

    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==BMW_520I_2018_PROGRAM \
    || can_program==VOLVO_XC60_2018_PROGRAM /*|| can_program==VOLVO_XC40_2018_PROGRAM*/ /*|| can_program==RANGE_ROVER_VELAR_2018_PROGRAM*/
      || can_program==BMW_X1_2019_PROGRAM || can_program==HYUNDAI_SONATA_SD_2020_PROGRAM || can_program==AUDI_A6_2018_PROGRAM
        || can_program==RENAULT_ARKANA_2018_PROGRAM /*?????*/)
  {
    //TODO: проверить на HYUNDAI_GENESIS_2017_PROGRAM, BMW_520I_2018_PROGRAM, VOLVO_XC60_2018_PROGRAM

    //получилось очень плохо, надо переделать
    uint8_t start_stop_output_id;
    uint8_t pedal_break_output_id;

    if(!is_start_stop_button_output_present(&start_stop_output_id) || !is_pedal_break_output_present(&pedal_break_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        if(System.can_state.sec_flags.engine_is_on)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, engine is allredy run";
          break;
        }

        System.Grab(portMAX_DELAY);
        if(System.signal_state.external_voltage<10.5f)
        {
          System.Release();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, low battery";
          break;
        }
        System.Release();

        if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, one of the doors is open";
          break;
        }

        if((!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, hood or trunk is open";
          break;
        }

        if(System.signal_state.ignition)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, ignition is already on";
          break;
        }

        //else if(!System.can_state.sec_flags.key_in_ign)
        //{
        //  res_str="key not detect";
        //  break;
        //}

        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        //паркинг не проверяем
        System.signal_state.digital_output[start_stop_output_id]=1;
        System.signal_state.digital_output[pedal_break_output_id]=1;
        outputs_state_force_apply();
        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=1;//продолжаем держать
        outputs_state_force_apply();

        can_cmd_proc.timer=get_max_engine_on_wait_time_ms()+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(System.can_state.sec_flags.engine_is_on || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          if(System.can_state.sec_flags.engine_is_on)
          {
            res_str="engine is on";
            is_engine_started_by_terminal_uid=can_cmd_proc.uid;
            memcpy(is_engine_started_by_terminal_nonce, can_cmd_proc.nonce, sizeof(is_engine_started_by_terminal_nonce));
            is_engine_started_by_terminal_server_notify_id=can_cmd_proc.server_notify_id;
            is_engine_started_by_terminal=true;

            if(can_cmd_proc.ext_param) {set_head_control_state(true, can_cmd_proc.ext_param/60000);}

            //если это прогрев, устанавливаем таймер работы двигателя
            if(can_cmd_proc.ext_param) warming_timer=can_cmd_proc.ext_param+xTaskGetTickCount();
            else                       warming_timer=0;

            System.signal_state.digital_output[start_stop_output_id]=0;
            System.signal_state.digital_output[pedal_break_output_id]=0;
            outputs_state_force_apply();

            break;
          }
          else
          {
            System.signal_state.digital_output[start_stop_output_id]=0;
            System.signal_state.digital_output[pedal_break_output_id]=0;
            outputs_state_force_apply();

            if(!System.can_state.sec_flags.ignition)
            {
              can_cmd_proc.step=10;//к завершению
            }
            else
            {
              //зажигание осталось включенным, проверим еще раз через 500 мс
              can_cmd_proc.timer=500+xTaskGetTickCount();
              can_cmd_proc.step++;
            }
          }
        }
      }
      else if(can_cmd_proc.step==6)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==7)
      {
        if(!System.can_state.sec_flags.ignition)
        {
          can_cmd_proc.step=10;//к завершению
        }
        else
        {
          //зажигание осталось включенным, будем пытаться выключить
          System.signal_state.digital_output[start_stop_output_id]=1;
          outputs_state_force_apply();

          can_cmd_proc.timer=250+xTaskGetTickCount();
          can_cmd_proc.step++;
        }
      }
      else if(can_cmd_proc.step==8)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==9)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        outputs_state_force_apply();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==10)
      {
        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;

          //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
          uint8_t key_power_output_id;
          if(is_key_power_present(&key_power_output_id))
          {
            if(is_key_power_off_no_blocking())
            {
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
            }
            else
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
            save_key_power_state(false);
          }

          if(is_ext_key_power_present(&key_power_output_id))
          {
            System.signal_state.digital_output[key_power_output_id]=0;
          }
        }

        if(System.can_state.sec_flags.ignition)
        {
          res_str="engine run fail, ignition left on";
        }
        else
        {
          res_str="engine run fail";
        }

        break;
      }
      else
      {
        //не должны сюда попадать
        System.signal_state.digital_output[start_stop_output_id]=0;
        System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;

          //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
          uint8_t key_power_output_id;
          if(is_key_power_present(&key_power_output_id))
          {
            if(is_key_power_off_no_blocking())
            {
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
            }
            else
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
            save_key_power_state(false);
          }

          if(is_ext_key_power_present(&key_power_output_id))
          {
            System.signal_state.digital_output[key_power_output_id]=0;
          }
        }

        res_str="wrong can_cmd_proc.step...";
        break;
      }
      return; //continue command procces
    }
    //end command procces

    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==VW_POLO_2018_PROGRAM  || can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM \
      || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM \
        || can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM\
          || can_program==CITROEN_JUMPY_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM \
             || can_program==HYUNDAI_CRETA_2019_PROGRAM )
  {
    for(;;)
    {
      uint8_t ignition_output_id;
      uint8_t starter_output_id;
      uint8_t pedal_break_output_id;
      uint16_t delay_after_ignition_off_ms=0;

      if(!is_ignition_output_present(&ignition_output_id) || !is_starter_output_present(&starter_output_id))
      {
        //проверяем, есть ли еще что-то в cmd_q
        move_next_command_to_head();
        can_cmd_proc.step=0;
        return;
      }

      if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
      {
        //todo: при неудачном старте в прогреве, будет отключена моторная шина вместе с зажиганием, надо делать отложенное отключение без блокировки
        //delay_after_ignition_off_ms=500;
      }

      if(can_cmd_proc.step==0)
      {
        if(System.can_state.sec_flags.engine_is_on)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, engine is allredy run";
          break;
        }

        System.Grab(portMAX_DELAY);
        if(System.signal_state.external_voltage<10.5f)
        {
          System.Release();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, low battery";
          break;
        }
        System.Release();

        if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, one of the doors is open";
          break;
        }

        if((!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, hood or trunk is open";
          break;
        }

        if(System.signal_state.ignition)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, ignition is already on";
          break;
        }

        __PRINTF("switch on ignition...\n");

        System.signal_state.digital_output[ignition_output_id]=1;
        System.signal_state.digital_output[starter_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=2000+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        if(!System.signal_state.ignition)
        {
          System.signal_state.digital_output[ignition_output_id]=0;
          if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
          outputs_state_force_apply();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            if(delay_after_ignition_off_ms)
            {
              vTaskDelay(delay_after_ignition_off_ms);
            }

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, ignition did not appeared...";
          break;
        }

        if(!System.can_state.sec_flags.parking)
        {
          System.signal_state.digital_output[ignition_output_id]=0;
          if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
          outputs_state_force_apply();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            if(delay_after_ignition_off_ms)
            {
              vTaskDelay(delay_after_ignition_off_ms);
            }

            //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }

            if(is_ext_key_power_present(&key_power_output_id))
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
          }

          res_str="break, transmission is not at parking";
          break;
        }

        __PRINTF("switch on starter...\n");

        System.signal_state.digital_output[starter_output_id]=1;
        outputs_state_force_apply();

        //держим минимум 700 мс
        can_cmd_proc.timer=700+xTaskGetTickCount();

        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          can_cmd_proc.timer=get_max_engine_on_wait_time_ms()+xTaskGetTickCount();
          can_cmd_proc.step++;
        }
      }
      else if(can_cmd_proc.step==4)
      {
        if(System.can_state.sec_flags.engine_is_on || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          System.signal_state.digital_output[starter_output_id]=0;
	  if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
          outputs_state_force_apply();

          if(System.can_state.sec_flags.engine_is_on)
          {
            res_str="engine is on";
            is_engine_started_by_terminal_uid=can_cmd_proc.uid;
            memcpy(is_engine_started_by_terminal_nonce, can_cmd_proc.nonce, sizeof(is_engine_started_by_terminal_nonce));
            is_engine_started_by_terminal_server_notify_id=can_cmd_proc.server_notify_id;
            is_engine_started_by_terminal=true;

            if(can_cmd_proc.ext_param) {set_head_control_state(true, can_cmd_proc.ext_param/60000);}

            //если это прогрев, устанавливаем таймер работы двигателя
            if(can_cmd_proc.ext_param) warming_timer=can_cmd_proc.ext_param+xTaskGetTickCount();
            else                       warming_timer=0;
          }
          else
          {
            System.signal_state.digital_output[ignition_output_id]=0;
	    if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
            outputs_state_force_apply();

            //если это была попытка прогрева, востанавливаем состоние проводной блокировки
            if(can_cmd_proc.ext_param)
            {
              is_wired_block_relay_enable=true;
              save_wired_block_relay_state(true);
              notify_state=IS_LAST_NOTIFY_STATE;

              if(delay_after_ignition_off_ms)
              {
                vTaskDelay(delay_after_ignition_off_ms);
              }

              //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
              uint8_t key_power_output_id;
              if(is_key_power_present(&key_power_output_id))
              {
                if(is_key_power_off_no_blocking())
                {
                  can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                  can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                }
                else
                {
                  System.signal_state.digital_output[key_power_output_id]=0;
                }
                save_key_power_state(false);
              }

              if(is_ext_key_power_present(&key_power_output_id))
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
            }

            res_str="engine run fail";
          }

          break;
        }
      }
      else
      {
        //не должны сюда попадать
        System.signal_state.digital_output[ignition_output_id]=0;
        System.signal_state.digital_output[starter_output_id]=0;
	if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;

          //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
          uint8_t key_power_output_id;
          if(is_key_power_present(&key_power_output_id))
          {
            if(is_key_power_off_no_blocking())
            {
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
            }
            else
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
            save_key_power_state(false);
          }

          if(is_ext_key_power_present(&key_power_output_id))
          {
            System.signal_state.digital_output[key_power_output_id]=0;
          }
        }

        res_str="wrong can_cmd_proc.step...";
        break;
      }
      return; //continue command procces
    }
    //end command procces

    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    uint8_t key_output_id = JEEP_WRANGLER_2018_REMOTE_KEY_OUTPUT_ID;

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        if(System.can_state.sec_flags.engine_is_on)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }
          }

          res_str="break, engine is allredy run";
          break;
        }

        System.Grab(portMAX_DELAY);
        if(System.signal_state.external_voltage<10.5f)
        {
          System.Release();

          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }
          }

          res_str="break, low battery";
          break;
        }
        System.Release();

        if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;
          }

          //при начале включения прогрева должно было быть включено питание ключа, отключаем его
          uint8_t key_power_output_id;
          if(is_key_power_present(&key_power_output_id))
          {
            if(is_key_power_off_no_blocking())
            {
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
            }
            else
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
            save_key_power_state(false);
          }

          res_str="break, one of the doors is open";
          break;
        }

        if(System.can_state.sec_flags.hood || System.can_state.sec_flags.trunk)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }
          }

          res_str="break, hood or trunk is open";
          break;
        }

        if(System.can_state.sec_flags.ignition)
        {
          //если это была попытка прогрева, востанавливаем состоние проводной блокировки
          if(can_cmd_proc.ext_param)
          {
            is_wired_block_relay_enable=true;
            save_wired_block_relay_state(true);
            notify_state=IS_LAST_NOTIFY_STATE;

            //при начале включения прогрева должно было быть включено питание ключа, отключаем его
            uint8_t key_power_output_id;
            if(is_key_power_present(&key_power_output_id))
            {
              if(is_key_power_off_no_blocking())
              {
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              }
              else
              {
                System.signal_state.digital_output[key_power_output_id]=0;
              }
              save_key_power_state(false);
            }
          }

          res_str="break, ignition is already on";
          break;
        }

        System.signal_state.digital_output[key_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[key_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=200+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[key_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=200+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==6)
      {
        System.signal_state.digital_output[key_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=200+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==7)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==8)
      {
        System.signal_state.digital_output[key_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=get_max_engine_on_wait_time_ms()+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==9)
      {
        if(System.can_state.sec_flags.engine_is_on || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          if(System.can_state.sec_flags.engine_is_on)
          {
            res_str="engine is on";
            is_engine_started_by_terminal_uid=can_cmd_proc.uid;
            memcpy(is_engine_started_by_terminal_nonce, can_cmd_proc.nonce, sizeof(is_engine_started_by_terminal_nonce));
            is_engine_started_by_terminal_server_notify_id=can_cmd_proc.server_notify_id;
            is_engine_started_by_terminal=true;

            if(can_cmd_proc.ext_param) {set_head_control_state(true, can_cmd_proc.ext_param/60000);}

            //если это прогрев, устанавливаем таймер работы двигателя
            if(can_cmd_proc.ext_param) warming_timer=can_cmd_proc.ext_param+xTaskGetTickCount();
            else                       warming_timer=0;
          }
          else
          {
            //если это была попытка прогрева, востанавливаем состоние проводной блокировки
            if(can_cmd_proc.ext_param)
            {
              is_wired_block_relay_enable=true;
              save_wired_block_relay_state(true);
              notify_state=IS_LAST_NOTIFY_STATE;

              //при начале включения прогрева должно было быть включено питание ключа, отключаем его
              uint8_t key_power_output_id;
              if(is_key_power_present(&key_power_output_id))
              {
                if(is_key_power_off_no_blocking())
                {
                  can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                  can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
                }
                else
                {
                  System.signal_state.digital_output[key_power_output_id]=0;
                }
                save_key_power_state(false);
              }
            }

            res_str="engine run fail";
          }

          break;
        }
      }
      else
      {
        //не должны сюда попадать
        System.signal_state.digital_output[key_output_id]=0;
        outputs_state_force_apply();

        //если это была попытка прогрева, востанавливаем состоние проводной блокировки
        if(can_cmd_proc.ext_param)
        {
          is_wired_block_relay_enable=true;
          save_wired_block_relay_state(true);
          notify_state=IS_LAST_NOTIFY_STATE;

          //при начале включения прогрева должно было быть включено питание ключа, отключаем его
          uint8_t key_power_output_id;
          if(is_key_power_present(&key_power_output_id))
          {
            if(is_key_power_off_no_blocking())
            {
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
              can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
            }
            else
            {
              System.signal_state.digital_output[key_power_output_id]=0;
            }
            save_key_power_state(false);
          }
        }

        res_str="wrong can_cmd_proc.step...";
        break;
      }
      return; //continue command procces
    }
    //end command procces

    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
}

static void stop_engine_handler(void)
{
  const char* res_str=NULL;

  static bool is_triple_push=false; //глушение тройным нажатием

  if(can_cmd_proc.step==0)
  {
    if(can_program==BMW_520I_2018_PROGRAM && IS_BMW_320D())
    {
      is_triple_push=true;
    }
    else
    {
      is_triple_push=false;
    }
  }

  if(can_program==KIA_RIO_2017_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    uint8_t ignition_output_id;
    uint8_t starter_output_id;

    if(!is_ignition_output_present(&ignition_output_id) || !is_starter_output_present(&starter_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        uint8_t already_stopped=0;
        if(!System.can_state.sec_flags.engine_is_on)
        {
          already_stopped=1;
        }

        System.signal_state.digital_output[ignition_output_id]=0;
        System.signal_state.digital_output[starter_output_id]=0;
        outputs_state_force_apply();

        if(already_stopped)
        {
          res_str="engine already stopped";
          break;
        }

        can_cmd_proc.timer=2000+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else
      {

        if(!System.can_state.sec_flags.engine_is_on) res_str="engine stop done";
        else                                         res_str="engine stop fail";

        break;
      }

      return; //continue command procces
    }
    //end command procces

    notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;

    is_engine_started_by_terminal=false;
    if_forced_stop_warming=false;

    //если был прогрев
    if(warming_timer!=0)
    {
      warming_timer=0;
      is_wired_block_relay_enable=true;
      save_wired_block_relay_state(true);
      LOG("wired block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
    }
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    uint8_t start_stop_output_id;

    if(!is_start_stop_button_output_present(&start_stop_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        uint8_t already_stopped=0;
        if(!System.can_state.sec_flags.engine_is_on)
        {
          already_stopped=1;
        }

        System.signal_state.digital_output[start_stop_output_id]=0;
        outputs_state_force_apply();

        if(already_stopped)
        {
          res_str="engine already stopped";
          break;
        }

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[start_stop_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=1500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else
      {
        if(!System.can_state.sec_flags.engine_is_on) res_str="engine stop done";
        else                                         res_str="engine stop fail";

        break;
      }

      return; //continue command procces
    }
    //end command procces

    notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;

    is_engine_started_by_terminal=false;
    if_forced_stop_warming=false;

    //если был прогрев
    if(warming_timer!=0)
    {
      warming_timer=0;
      is_wired_block_relay_enable=true;
      save_wired_block_relay_state(true);
      LOG("hood block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
    }
  }
  else if(can_program==BMW_520I_2018_PROGRAM && is_triple_push)
  {
    //получилось очень плохо, надо переделать
    //это хуйня, нужно видеть состояние асс и ign по кан, в первом случае при неудаче дергать зажиганием один раз, во втором два.
    uint8_t start_stop_output_id;
    uint8_t pedal_break_output_id;

    if(!is_start_stop_button_output_present(&start_stop_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        uint8_t already_stopped=0;
        if(!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition)
        {
          already_stopped=1;
        }

        System.signal_state.digital_output[start_stop_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        if(already_stopped)
        {
          res_str="engine already stopped";
          break;
        }

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[start_stop_output_id]=1;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==6)
      {
        System.signal_state.digital_output[start_stop_output_id]=1;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==7)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==8)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==9)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==10)
      {
        System.signal_state.digital_output[start_stop_output_id]=1;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==11)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==12)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        //ждем 2 секунды
        can_cmd_proc.timer=2000+xTaskGetTickCount();//у Genesis очень долго пропадант зажигание, у Velar еще дольше
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==13)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          if(!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition)
          {
            //заглушился, пропускаем следующий шаг
            can_cmd_proc.step++;
          }
          else
          {
            //ждем еще 5 секунд
            can_cmd_proc.timer=5000+xTaskGetTickCount();
          }
          can_cmd_proc.step++;
        }
      }
      else if(can_cmd_proc.step==14)
      {
        if((!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition) \
          || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          can_cmd_proc.step++;
        }
      }
      else
      {
        if(!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition)        res_str="engine stop done";
        else if(!System.can_state.sec_flags.ignition)                                        res_str="ignition off fail";
        else                                                                                 res_str="engine stop fail";

        break;
      }

      return; //continue command procces
    }
    //end command procces

    notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;

    is_engine_started_by_terminal=false;
    if_forced_stop_warming=false;

    //если был прогрев
    if(warming_timer!=0)
    {
      warming_timer=0;

      //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
      uint8_t key_power_output_id;
      if(is_key_power_present(&key_power_output_id))
      {
        if(is_key_power_off_no_blocking())
        {
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
        }
        else
        {
          System.signal_state.digital_output[key_power_output_id]=0;
        }
        save_key_power_state(false);
      }

      if(is_ext_key_power_present(&key_power_output_id))
      {
        System.signal_state.digital_output[key_power_output_id]=0;
      }

      is_wired_block_relay_enable=true;
      save_wired_block_relay_state(true);
      LOG("hood block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
    }
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==BMW_520I_2018_PROGRAM \
    || can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM \
      || can_program==RANGE_ROVER_VELAR_2018_PROGRAM \
        || can_program==BMW_X1_2019_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM || can_program==HYUNDAI_SONATA_SD_2020_PROGRAM || can_program==AUDI_A6_2018_PROGRAM
          || can_program==RENAULT_ARKANA_2018_PROGRAM /*???*/)
  {
    //получилось очень плохо, надо переделать
    //это хуйня, нужно видеть состояние асс и ign по кан, в первом случае при неудаче дергать зажиганием один раз, во втором два.
    uint8_t start_stop_output_id;
    uint8_t pedal_break_output_id;

    if(!is_start_stop_button_output_present(&start_stop_output_id))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        uint8_t already_stopped=0;
        if(!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition)
        {
          already_stopped=1;
        }

        System.signal_state.digital_output[start_stop_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        if(already_stopped)
        {
          res_str="engine already stopped";
          break;
        }

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[start_stop_output_id]=1;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[start_stop_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        //ждем 2 секунды
        can_cmd_proc.timer=2000+xTaskGetTickCount();//у Genesis очень долго пропадант зажигание, у Velar еще дольше
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          if(!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition)
          {
            //заглушился, пропускаем следующий шаг
            can_cmd_proc.step++;
          }
          else
          {
            //ждем еще 5 секунд
            can_cmd_proc.timer=5000+xTaskGetTickCount();
          }
          can_cmd_proc.step++;
        }
      }
      else if(can_cmd_proc.step==6)
      {
        if((!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition) \
          || timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
        {
          can_cmd_proc.step++;
        }
      }
      else
      {
        if(!System.can_state.sec_flags.engine_is_on && !System.signal_state.ignition)        res_str="engine stop done";
        else if(!System.can_state.sec_flags.ignition)                                        res_str="ignition off fail";
        else                                                                                 res_str="engine stop fail";

        break;
      }

      return; //continue command procces
    }
    //end command procces

    notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;

    is_engine_started_by_terminal=false;
    if_forced_stop_warming=false;

    //если был прогрев
    if(warming_timer!=0)
    {
      warming_timer=0;

      //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
      uint8_t key_power_output_id;
      if(is_key_power_present(&key_power_output_id))
      {
        if(is_key_power_off_no_blocking())
        {
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
        }
        else
        {
          System.signal_state.digital_output[key_power_output_id]=0;
        }
        save_key_power_state(false);
      }

      if(is_ext_key_power_present(&key_power_output_id))
      {
        System.signal_state.digital_output[key_power_output_id]=0;
      }

      is_wired_block_relay_enable=true;
      save_wired_block_relay_state(true);
      LOG("hood block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
    }
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM \
      || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM  || can_program==SKODA_RAPID_2013_PROGRAM \
        || can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM \
          || can_program==CITROEN_JUMPY_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM \
             || can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
     uint8_t ignition_output_id;
     uint8_t starter_output_id;
     uint8_t pedal_break_output_id;
     uint16_t delay_after_ignition_off_ms=2000;

     if(!is_ignition_output_present(&ignition_output_id) || !is_starter_output_present(&starter_output_id))
     {
       //проверяем, есть ли еще что-то в cmd_q
       move_next_command_to_head();
       can_cmd_proc.step=0;
       return;
     }

     if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
     {
       //if(warming_timer!=0) delay_after_ignition_off_ms=3500;//если это прогрев, то будет произведено отключение мотороной CAN шины, делаем задежку больше
     }

     for(;;)
     {
      if(can_cmd_proc.step==0)
      {
        uint8_t already_stopped=0;
        if(!System.can_state.sec_flags.engine_is_on)
        {
          already_stopped=1;
        }

        System.signal_state.digital_output[ignition_output_id]=0;
        System.signal_state.digital_output[starter_output_id]=0;
        if(is_pedal_break_output_present(&pedal_break_output_id)) System.signal_state.digital_output[pedal_break_output_id]=0;
        outputs_state_force_apply();

        if(already_stopped)
        {
          res_str="engine already stopped";
          break;
        }

        can_cmd_proc.timer=delay_after_ignition_off_ms+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else
      {
        if(!System.can_state.sec_flags.engine_is_on) res_str="engine stop done";
        else                                         res_str="engine stop fail";

        break;
      }

      return; //continue command procces
    }
    //end command procces

    notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;

    is_engine_started_by_terminal=false;
    if_forced_stop_warming=false;

    //если был прогрев
    if(warming_timer!=0)
    {
      warming_timer=0;

      //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
      uint8_t key_power_output_id;
      if(is_key_power_present(&key_power_output_id))
      {
        if(is_key_power_off_no_blocking())
        {
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
        }
        else
        {
          System.signal_state.digital_output[key_power_output_id]=0;
        }
        save_key_power_state(false);
      }

      if(is_ext_key_power_present(&key_power_output_id))
      {
        System.signal_state.digital_output[key_power_output_id]=0;
      }

      is_wired_block_relay_enable=true;
      save_wired_block_relay_state(true);
      LOG("wired block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
    }
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    uint8_t key_output_id = JEEP_WRANGLER_2018_REMOTE_KEY_OUTPUT_ID;

    for(;;)
    {
      if(can_cmd_proc.step==0)
      {
        uint8_t already_stopped=0;
        if(!System.can_state.sec_flags.engine_is_on)
        {
          already_stopped=1;
        }

        System.signal_state.digital_output[key_output_id]=0;
        outputs_state_force_apply();

        if(already_stopped)
        {
          res_str="engine already stopped";
          break;
        }

        can_cmd_proc.timer=500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[key_output_id]=1;
        outputs_state_force_apply();

        can_cmd_proc.timer=200+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        System.signal_state.digital_output[key_output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=4000+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else
      {
        if(!System.can_state.sec_flags.engine_is_on) res_str="engine stop done";
        else                                         res_str="engine stop fail";

        break;
      }

      return; //continue command procces
    }
    //end command procces

    notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    move_next_command_to_head();
    can_cmd_proc.step=0;

    is_engine_started_by_terminal=false;
    if_forced_stop_warming=false;

    //если был прогрев
    if(warming_timer!=0)
    {
      warming_timer=0;

      //при начале включения прогшрева должно было быть включено питание ключа, отключаем его
      uint8_t key_power_output_id;
      if(is_key_power_present(&key_power_output_id))
      {
        if(is_key_power_off_no_blocking())
        {
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
          can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
        }
        else
        {
          System.signal_state.digital_output[key_power_output_id]=0;
        }
        save_key_power_state(false);
      }

      is_wired_block_relay_enable=true;
      save_wired_block_relay_state(true);
      LOG("hood block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
    }
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;

    is_engine_started_by_terminal=false;
    if_forced_stop_warming=false;

    //если был прогрев
    if(warming_timer!=0)
    {
      warming_timer=0;
      is_wired_block_relay_enable=true;
      save_wired_block_relay_state(true);
      LOG("hood block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
    }

    return;
  }
}

static void driver_door_emulation_handler(void)
{
  uint8_t output_id;

  restart_can_timer(DRIVER_DOOR_IGNORE_TIMER_ID, 10000);

  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    output_id=RIO_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    output_id=KAPTUR_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    output_id=GENESIS_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  /*
  //нет эмуляции двери
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    output_id=AUDI_A3_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
  {
    output_id=AUDI_A3_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  */
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    output_id=BMW_520I_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    output_id=VOLVO_XC60_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    output_id=VOLVO_XC40_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  /*
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    output_id=RANGE_ROVER_VELAR_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  */
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    output_id=NISSAN_QASHQAI_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  /*
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    output_id=HYUNDAI_CRETA_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  */
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    output_id=BMW_X1_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  /*
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM)
  {
    output_id=RENAULT_LOGAN_2013_DRIVER_DOOR_EMULATION_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    output_id=;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    output_id=;
  }
  */
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=800+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void horn_handler(uint16_t beep_time_ms)
{
  uint8_t output_id;

  if(!is_horn_output_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=beep_time_ms+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void lin_windows_closing_handler(void)
{
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
  uint8_t output_id;

  if(!is_windows_output_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(!(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door))
  {
    //если все окна закрыты, перекращаем подачу команд
    if(can_cmd_proc.step==0)
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }
    else
    {
      //lin и выход быди инициализированы, задаем шаг для перехода к деинициализации
      can_cmd_proc.step=0xFFFF;
    }
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      lin_cmd_step=0;
      lin_uart_init(get_lin_bus_speed());

      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=1000+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
      {
        lin_cmd_step++;
        can_cmd_proc.timer=7500+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
    }
    else if(can_cmd_proc.step==2)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      lin_uart_deinit();
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      break;
    }

    return; //continue command procces
  }
  //end command procces
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static bool is_analog_windows_closing_no_blocking(void)
{
  if(can_program==FORD_TRANSIT_2017_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM) return true;
  else                                                                                   return false;
}

static bool is_key_power_off_no_blocking(void)
{
  if(can_program==NISSAN_QASHQAI_2018_PROGRAM) return true;
  else                                         return false;
}

static uint16_t get_analog_windows_closing_time_ms(void)
{
  if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)        return 4500;
  //else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM) return 7000;
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)      return 22000;
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)    return 30000;
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)    return 8000;
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM)     return 500; //импульс
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)    return 4000;
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)    return 20000;
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)     return 500; //импульс
  else if(can_program==LADA_LARGUS_2020_PROGRAM)       return 500;
  else                                                 return 6000;
}

static bool is_can_windows_closing(void)
{
 if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM ||
     can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM ||
     can_program==MB_E200_2018_PROGRAM || can_program==BMW_520I_2018_PROGRAM ||
     can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM /* || can_program==RANGE_ROVER_VELAR_2018_PROGRAM*/ ||
     can_program==BMW_X1_2019_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM || can_program==AUDI_A6_2018_PROGRAM)
  {
    return true;
  }
  else
  {
    return false;
  }
}

static void analog_windows_closing_handler(void)
{
  uint8_t output_id;

  if(!is_windows_output_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(is_analog_windows_closing_no_blocking())
  {
    //обработается асинхронно
    can_timers[ANALOG_WINDOWS_CLOSING_TIMER_ID]=get_analog_windows_closing_time_ms();
    can_timers[ANALOG_WINDOWS_CLOSING_TIMER_ID]=get_analog_windows_closing_time_ms();

    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=get_analog_windows_closing_time_ms()+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static bool is_central_lock_system_with_analog_control(void)
{
  if(can_program==VW_POLO_2018_PROGRAM /*|| can_program==VW_CADDY_2018_PROGRAM*/ || can_program==VW_TRANSPORTER_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || \
     can_program==SKODA_RAPID_2018_PROGRAM || can_program==CITROEN_JUMPY_2018_PROGRAM || \
     can_program==LADA_LARGUS_2017_PROGRAM || can_program==FORD_TRANSIT_2017_PROGRAM || \
     can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM || can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    return true;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)//не используем управление по CAN
  {
    return true;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)//не используем управление по CAN
  {
    return true;
  }
  else
  {
    return false;
  }
}

static void central_lock_system_handler(bool is_lock)
{
  struct
  {
    bool dual_lock_pulse:1;
    bool dual_unlock_pulse:1;
    bool is_cls_state_present:1;
  }ext_param=
  {
    .dual_lock_pulse=false,
    .dual_lock_pulse=false,
    .is_cls_state_present=false,
  };

  uint8_t lock_output_id;
  uint8_t unlock_output_id;

  if(can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM)
  {
    lock_output_id=VW_POLO_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=VW_POLO_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    lock_output_id=VW_TRANSPORTER_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=VW_TRANSPORTER_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==SKODA_RAPID_2018_PROGRAM)
  {
    ext_param.is_cls_state_present=true;
    lock_output_id=SKODA_RAPID_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=SKODA_RAPID_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==SKODA_RAPID_2013_PROGRAM)
  {
    ext_param.is_cls_state_present=true;//статус по аналогу SKODA_RAPID_2013_CLS_STATE_INPUT_ID
    lock_output_id=SKODA_RAPID_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=SKODA_RAPID_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    ext_param.is_cls_state_present=true;
    lock_output_id=CITROEN_JUMPY_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=CITROEN_JUMPY_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==LADA_LARGUS_2017_PROGRAM)
  {
    lock_output_id=LADA_LARGUS_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=LADA_LARGUS_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)//temporary
  {
    ext_param.dual_lock_pulse=false;
    ext_param.dual_unlock_pulse=true;
    lock_output_id=FORD_TRANSIT_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=FORD_TRANSIT_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    ext_param.is_cls_state_present=false;//статус замка есть, но импульс подаем в любом случае
    lock_output_id=RANGE_ROVER_VELAR_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=RANGE_ROVER_VELAR_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    ext_param.is_cls_state_present=false;
    lock_output_id=FORD_MUSTANG65_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=FORD_MUSTANG65_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    ext_param.is_cls_state_present=false;
    lock_output_id=FORD_MUSTANG69_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=FORD_MUSTANG69_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    ext_param.is_cls_state_present=false;
    lock_output_id=NISSAN_LEAF_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=NISSAN_LEAF_CLS_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    ext_param.is_cls_state_present=true;
    lock_output_id=VOLVO_XC40_CLS_LOCK_OUTPUT_ID;
    unlock_output_id=VOLVO_XC40_CLS_UNLOCK_OUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if((ext_param.is_cls_state_present && can_cmd_proc.step==0) && \
    (_cls_state==UNKNOWN_CLS_STATE || (_cls_state==UNLOCKED_CLS_STATE && !is_lock) || (_cls_state==LOCKED_CLS_STATE && is_lock)))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0 || can_cmd_proc.step==4)
    {
      System.signal_state.digital_output[lock_output_id]=0;
      System.signal_state.digital_output[unlock_output_id]=0;
      outputs_state_force_apply();

      if(is_lock)
      {
        //не менять местами
        System.signal_state.digital_output[unlock_output_id]=0;
        System.signal_state.digital_output[lock_output_id]=1;
      }
      else
      {
        //не менять местами
        System.signal_state.digital_output[lock_output_id]=0;
        System.signal_state.digital_output[unlock_output_id]=1;
      }
      outputs_state_force_apply();

      can_cmd_proc.timer=500+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1 || can_cmd_proc.step==3 || can_cmd_proc.step==5)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[lock_output_id]=0;
      System.signal_state.digital_output[unlock_output_id]=0;
      outputs_state_force_apply();

      if(can_cmd_proc.step==2 && ((ext_param.dual_lock_pulse && is_lock) || (ext_param.dual_unlock_pulse && !is_lock)))
      {
        can_cmd_proc.timer=999+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else
      {
        break;
      }
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void charge_lock_unlock_handler(bool is_lock)
{
  uint8_t lock_output_id;
  uint8_t unlock_output_id;
  bool is_state_present=false;

  if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    //статус по аналогу
    is_state_present=true;
    lock_output_id=NISSAN_LEAF_CHARGE_LOCK_OUTPUT_ID;
    unlock_output_id=NISSAN_LEAF_CHARGE_UNLOCK_OUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if((is_state_present && can_cmd_proc.step==0) && \
    ((System.signal_state.tank_flap && !is_lock) || (!System.signal_state.tank_flap && is_lock)))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[lock_output_id]=0;
      System.signal_state.digital_output[unlock_output_id]=0;
      outputs_state_force_apply();

      if(is_lock)
      {
        System.signal_state.digital_output[unlock_output_id]=0;
        System.signal_state.digital_output[lock_output_id]=1;
      }
      else
      {
        //не менять местами
        System.signal_state.digital_output[lock_output_id]=0;
        System.signal_state.digital_output[unlock_output_id]=1;
      }
      outputs_state_force_apply();

      can_cmd_proc.timer=300+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[lock_output_id]=0;
      System.signal_state.digital_output[unlock_output_id]=0;
      outputs_state_force_apply();

      if(is_lock) {hood_lock_event = 1;}

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void eco_on_off_handler(bool is_on)
{
  uint8_t output_id=0xff;

  if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    output_id=NISSAN_LEAF_ECO_KEY_OUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(can_cmd_proc.step==0 && ((output_id==0xff) || (is_on && System.signal_state.eco_mode) || (!is_on && !System.signal_state.eco_mode)))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=500+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void mustang_windows_closing_handler(void)
{
  uint8_t br_output_id;

  if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    br_output_id=FORD_MUSTANG69_WINDOWS_BROUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.ext_digital_output[br_output_id]=1;

      can_cmd_proc.timer=get_analog_windows_closing_time_ms()+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.ext_digital_output[br_output_id]=0;

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void mustang_headlights_flashing_handler(void)
{
  uint8_t br_output_id;

  if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    br_output_id=FORD_MUSTANG65_HEAD_LIGHTS_BROUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    br_output_id=FORD_MUSTANG69_HEAD_LIGHTS_BROUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.ext_digital_output[br_output_id]=1;

      can_cmd_proc.timer=900+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.ext_digital_output[br_output_id]=0;

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void emerg_flashing_handler(uint16_t flash_time_ms)
{
  uint8_t output_id;

  if(!is_analog_emerg_lights_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=200+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==2)
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      //if(flash_time_ms<1000) flash_time_ms=1000;
      can_cmd_proc.timer=flash_time_ms+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==3)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==4)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=200+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==5)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void hood_lock_unlock_handler(bool is_lock)
{
  uint8_t lock_output_id;
  uint8_t unlock_output_id;

  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    lock_output_id=RIO_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=RIO_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM)
  {
    lock_output_id=KAPTUR_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=KAPTUR_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    //На Dokker нет упрваления замком капота
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    lock_output_id=GENESIS_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=GENESIS_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
      || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    lock_output_id=AUDI_A3_Q3_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=AUDI_A3_Q3_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    lock_output_id=VW_TRANSPORTER_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=VW_TRANSPORTER_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    lock_output_id=PORSCHE_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=PORSCHE_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    lock_output_id=BMW_520I_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=BMW_520I_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    lock_output_id=MB_E200_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=MB_E200_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    lock_output_id=TOYOTA_RAV4_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=TOYOTA_RAV4_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    lock_output_id=CITROEN_JUMPY_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=CITROEN_JUMPY_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    lock_output_id=VOLVO_XC60_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=VOLVO_XC60_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    lock_output_id=VOLVO_XC40_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=VOLVO_XC40_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    lock_output_id=RANGE_ROVER_VELAR_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=RANGE_ROVER_VELAR_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    lock_output_id=FORD_TRANSIT_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=FORD_TRANSIT_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    lock_output_id=NISSAN_QASHQAI_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=NISSAN_QASHQAI_HOOD_ULOCK_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    lock_output_id=HYUNDAI_CRETA_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=HYUNDAI_CRETA_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    lock_output_id=BMW_X1_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=BMW_X1_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    lock_output_id=FORD_MUSTANG65_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=FORD_MUSTANG65_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    lock_output_id=FORD_MUSTANG69_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=FORD_MUSTANG69_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    lock_output_id=NISSAN_LEAF_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=NISSAN_LEAF_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    lock_output_id=RENAULT_LOGAN_2013_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=RENAULT_LOGAN_2013_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    lock_output_id=RENAULT_DUSTER_2021_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=RENAULT_DUSTER_2021_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    lock_output_id=RENAULT_ARKANA_2018_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=RENAULT_ARKANA_2018_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    lock_output_id=JEEP_WRANGLER_2018_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=JEEP_WRANGLER_2018_HOOD_UNLOCK_OUTPUT_ID;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    lock_output_id=AUDI_A6_2018_HOOD_LOCK_OUTPUT_ID;
    unlock_output_id=AUDI_A6_2018_HOOD_UNLOCK_OUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(is_lock)
      {
        System.signal_state.digital_output[lock_output_id]=1;
        System.signal_state.digital_output[unlock_output_id]=0;
        outputs_state_force_apply();
      }
      else
      {
        System.signal_state.digital_output[lock_output_id]=0;
        System.signal_state.digital_output[unlock_output_id]=1;
        outputs_state_force_apply();
      }

      can_cmd_proc.timer=150+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      System.signal_state.digital_output[lock_output_id]=0;
      System.signal_state.digital_output[unlock_output_id]=0;
      outputs_state_force_apply();

      if(is_lock) {hood_lock_event = 1;}

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void trunk_fuel_cap_lock_unlock_handler(bool trunk, bool fuel_cap, bool is_lock)
{
  uint8_t fuel_cap_lock_broutput_id;
  uint8_t fuel_cap_unlock_broutput_id;
  uint8_t trunk_lock_broutput_id;
  uint8_t trunk_unlock_broutput_id;

  if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    fuel_cap_lock_broutput_id=FORD_MUSTANG65_FUEL_CAP_LOCK_BROUTPUT_ID;
    fuel_cap_unlock_broutput_id=FORD_MUSTANG65_FUEL_CAP_UNLOCK_BROUTPUT_ID;
    trunk_lock_broutput_id=FORD_MUSTANG65_TRUNK_LOCK_BROUTPUT_ID;
    trunk_unlock_broutput_id=FORD_MUSTANG65_TRUNK_UNLOCK_BROUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    fuel_cap_lock_broutput_id=FORD_MUSTANG69_FUEL_CAP_LOCK_BROUTPUT_ID;
    fuel_cap_unlock_broutput_id=FORD_MUSTANG69_FUEL_CAP_UNLOCK_BROUTPUT_ID;
    trunk_lock_broutput_id=FORD_MUSTANG69_TRUNK_LOCK_BROUTPUT_ID;
    trunk_unlock_broutput_id=FORD_MUSTANG69_TRUNK_UNLOCK_BROUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(is_lock)
      {
        if(fuel_cap && fuel_cap_lock_broutput_id!=0xFF && fuel_cap_unlock_broutput_id!=0xFF)
        {
          System.signal_state.ext_digital_output[fuel_cap_lock_broutput_id]=1;
          System.signal_state.ext_digital_output[fuel_cap_unlock_broutput_id]=0;
        }
        if(trunk && trunk_lock_broutput_id!=0xFF && trunk_unlock_broutput_id!=0xFF)
        {
          System.signal_state.ext_digital_output[trunk_lock_broutput_id]=1;
          System.signal_state.ext_digital_output[trunk_unlock_broutput_id]=0;
        }
      }
      else
      {
        if(fuel_cap && fuel_cap_lock_broutput_id!=0xFF && fuel_cap_unlock_broutput_id!=0xFF)
        {
          System.signal_state.ext_digital_output[fuel_cap_lock_broutput_id]=0;
          System.signal_state.ext_digital_output[fuel_cap_unlock_broutput_id]=1;
        }
        if(trunk && trunk_lock_broutput_id!=0xFF && trunk_unlock_broutput_id!=0xFF)
        {
          System.signal_state.ext_digital_output[trunk_lock_broutput_id]=0;
          System.signal_state.ext_digital_output[trunk_unlock_broutput_id]=1;
        }
      }

      can_cmd_proc.timer=700+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      if(fuel_cap && fuel_cap_lock_broutput_id!=0xFF && fuel_cap_unlock_broutput_id!=0xFF)
      {
        System.signal_state.ext_digital_output[fuel_cap_lock_broutput_id]=0;
        System.signal_state.ext_digital_output[fuel_cap_unlock_broutput_id]=0;
      }
      if(trunk && trunk_lock_broutput_id!=0xFF && trunk_unlock_broutput_id!=0xFF)
      {
        System.signal_state.ext_digital_output[trunk_lock_broutput_id]=0;
        System.signal_state.ext_digital_output[trunk_unlock_broutput_id]=0;
      }
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void wired_block_relay_handler(bool is_lock)
{
  if(is_wired_block_present(NULL))
  {
    if(is_lock)
    {
      is_wired_block_relay_enable=true;
    }
    else
    {
      is_wired_block_relay_enable=false;
    }

    save_wired_block_relay_state(is_lock);
  }

  //проверяем, есть ли еще что-то в cmd_q
  move_next_command_to_head();
  can_cmd_proc.step=0;
  return;
}

static void wireless_block_realy_handler(bool is_lock)
{
  uint8_t output_id;

  if(!is_wireless_block_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(is_lock)
      {
        if(output_id!=0xff)
        {
          System.signal_state.digital_output[output_id]=1;
          outputs_state_force_apply();
        }
#if defined(INTERNAL_NRF_PRESENT)
        System.signal_state.nrf_state.desired_relay_state=1;
#endif //defined(INTERNAL_NRF_PRESENT)
      }
      else
      {
        if(output_id!=0xff)
        {
          System.signal_state.digital_output[output_id]=0;
          outputs_state_force_apply();
        }
#if defined(INTERNAL_NRF_PRESENT)
        System.signal_state.nrf_state.desired_relay_state=0;
#endif //defined(INTERNAL_NRF_PRESENT)
      }

      save_wireless_block_relay_state(is_lock);

      can_cmd_proc.timer=150+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static bool is_horn_output_present(uint8_t* output_id)
{
  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RIO_HORN_OUTPUT_ID;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=KAPTUR_HORN_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=GENESIS_HORN_OUTPUT_ID;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=MB_E200_HORN_OUTPUT_ID; //на E200 не смонтирован, только на C180
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=TOYOTA_RAV4_HORN_OUTPUT_ID;
  }
  else if(can_program==SKODA_RAPID_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=SKODA_RAPID_HORN_OUTPUT_ID;
  }
  else if(can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_POLO_HORN_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_HORN_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_HORN_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC60_HORN_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC40_HORN_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_HORN_OUTPUT_ID;
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_TRANSIT_HORN_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_HORN_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_HORN_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_MUSTANG65_HORN_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_MUSTANG69_HORN_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_LEAF_HORN_OUTPUT_ID;
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_LOGAN_2013_HORN_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_DUSTER_2021_HORN_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_ARKANA_2018_HORN_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=JEEP_WRANGLER_2018_HORN_OUTPUT_ID;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A6_2018_HORN_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static bool is_windows_output_present(uint8_t* output_id)
{
  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RIO_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=KAPTUR_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=GENESIS_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=SKODA_OCTAVIA_RAPID_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_POLO_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_WINDOWS_OUTPUT_ID;//используется для подключения к LIN
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_TRANSIT_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_LEAF_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_LOGAN_2013_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_DUSTER_2021_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_ARKANA_2018_WINDOWS_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=JEEP_WRANGLER_2018_WINDOWS_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static bool is_ignition_output_present(uint8_t* output_id)
{
  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RIO_IGNITION_OUTPUT_ID;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM \
       || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A3_Q3_IGNITION_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_IGNITION_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=PORSCHE_IGNITION_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_IGNITION_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_IGNITION_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_IGNITION_OUTPUT_ID;
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_LOGAN_2013_IGNITION_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_DUSTER_2021_IGNITION_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

bool is_starter_output_present(uint8_t* output_id)
{
  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RIO_STARTER_OUTPUT_ID;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
      || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A3_Q3_STARTER_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_STARTER_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_STARTER_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=PORSCHE_STARTER_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_STARTER_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_STARTER_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_STARTER_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_DUSTER_2021_STARTER_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static bool is_pedal_break_output_present(uint8_t* output_id)
{
  if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=GENESIS_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_520I_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=TOYOTA_RAV4_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC60_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC40_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=PORSCHE_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_X1_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_SONATA_SD_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_SONATA_SD_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A6_2018_PEADAL_BREAK_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_ARKANA_2018_PEADAL_BREAK_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static uint16_t get_max_engine_on_wait_time_ms(void)
{
  if(can_program==JEEP_WRANGLER_2018_PROGRAM) {return 6000;}
  else                                        {return 4000;}
}

static bool is_start_stop_button_output_present(uint8_t* output_id)
{
  if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=KAPTUR_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=GENESIS_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_520I_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=MB_E200_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=TOYOTA_RAV4_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC60_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC40_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_X1_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_LEAF_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_SONATA_SD_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_SONATA_SD_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=JEEP_WRANGLER_2018_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A6_2018_START_STOP_BUTTON_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_ARKANA_2018_START_STOP_BUTTON_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

//автозапуск
static bool is_engine_start_stop_present(void)
{
  if(can_program==KIA_RIO_2017_PROGRAM || can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM ||
     can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM ||
     can_program==VW_POLO_2018_PROGRAM /*|| can_program==VW_CADDY_2018_PROGRAM*/ || can_program==SKODA_RAPID_2013_PROGRAM || can_program==BMW_520I_2018_PROGRAM ||
     (can_program==VW_TRANSPORTER_2018_PROGRAM && !IS_VW_TRANSPORTER_WITH_MANUAL_GEARBOX()) || (can_program==CITROEN_JUMPY_2018_PROGRAM && !IS_CITROEN_JUMPY_WITH_MANUAL_GEARBOX()) ||
     can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM ||
     can_program==HYUNDAI_CRETA_2019_PROGRAM || can_program==BMW_X1_2019_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM || can_program==HYUNDAI_SONATA_SD_2020_PROGRAM ||
     can_program==JEEP_WRANGLER_2018_PROGRAM || can_program==AUDI_A6_2018_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    return true;
  }
  else
  {
    return false;
  }
}

static bool is_key_power_present(uint8_t* output_id)
{
  if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
    || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A3_Q3_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=PORSCHE_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=GENESIS_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=MB_E200_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=TOYOTA_RAV4_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_520I_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC60_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC40_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_ENGINE_CAN_RELAY_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_ACC_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_LIGHTS_POWER_OUTPUT_ID;
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_X1_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_LEAF_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_SONATA_SD_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_SONATA_SD_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=JEEP_WRANGLER_2018_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A6_2018_KEY_POWER_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_ARKANA_2018_KEY_POWER_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static bool is_ext_key_power_present(uint8_t* output_id)
{
  if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_KEY2_POWER_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static bool is_wireless_block_present(uint8_t* output_id)
{
  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RIO_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_SOLARIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=SOLARIS_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=KAPTUR_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=GENESIS_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A3_Q3_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=SKODA_OCTAVIA_RAPID_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_POLO_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=PORSCHE_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=MB_E200_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=TOYOTA_RAV4_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_520I_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==LADA_LARGUS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=LADA_LARGUS_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC60_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC40_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_TRANSIT_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=BMW_X1_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_MUSTANG65_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_MUSTANG69_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_LEAF_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_LOGAN_2013_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_DUSTER_2021_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_ARKANA_2018_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=JEEP_WRANGLER_2018_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A6_2018_WIRELESS_BLOCKING_RELAY_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static bool is_analog_emerg_lights_present(uint8_t* output_id)
{
  if(can_program==AUDI_A3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A3_SKODA_OCTAVIA_RAPID_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_POLO_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC60_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC40_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_TRANSIT_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_LEAF_EMERG_LIGHT_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=JEEP_WRANGLER_2018_EMERG_LIGHT_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static bool is_wired_block_present(uint8_t* output_id)
{
  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RIO_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_SOLARIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=SOLARIS_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=KAPTUR_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=GENESIS_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM \
      || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A3_Q3_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VW_TRANSPORTER_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=PORSCHE_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=MB_E200_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=TOYOTA_RAV4_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=CITROEN_JUMPY_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC60_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=VOLVO_XC40_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RANGE_ROVER_VELAR_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_TRANSIT_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  ///*
  //сжигают стартер. крутят его на завершенной аренде. у некоторых машин после нажатия кнопки крутит пока не сгорит стартер.
  //26.10.21 Артем попросил вернуть
  //сжигают стартер. крутят его на завершенной аренде
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=NISSAN_QASHQAI_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  //*/
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_MUSTANG65_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    if(output_id!=NULL) *output_id=FORD_MUSTANG69_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_LOGAN_2013_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
#warning RENAULT_DUSTER_2021_WIRED_BLOCKING_RELAY_OUTPUT_ID
    if(output_id!=NULL) *output_id=RENAULT_DUSTER_2021_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=RENAULT_ARKANA_2018_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=JEEP_WRANGLER_2018_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    if(output_id!=NULL) *output_id=AUDI_A6_2018_WIRED_BLOCKING_RELAY_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

static uint16_t get_async_key_power_off_time_ms(void)
{
  return 65000;
}

static void key_power_handler(bool is_on)
{
  uint8_t output_id;

  if(!is_key_power_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(is_key_power_off_no_blocking())
  {
    if(!is_on)
    {
      //обработается асинхронно
      can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();
      can_timers[KEY_POWER_OFF_TIMER_ID]=get_async_key_power_off_time_ms();

      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }
    else
    {
      if(can_cmd_proc.step==0)
      {
        can_timers[KEY_POWER_OFF_TIMER_ID]=0;
        can_timers[KEY_POWER_OFF_TIMER_ID]=0;
      }
    }
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(is_on) System.signal_state.digital_output[output_id]=1;
      else      System.signal_state.digital_output[output_id]=0;

      if(is_ext_key_power_present(&output_id))
      {
        if(is_on) System.signal_state.digital_output[output_id]=1;
        else      System.signal_state.digital_output[output_id]=0;
      }

      outputs_state_force_apply();

      save_key_power_state(is_on);

      if(can_program==NISSAN_QASHQAI_2018_PROGRAM && is_on) can_cmd_proc.timer=1200+xTaskGetTickCount();
      else if(is_on)                                        can_cmd_proc.timer=500+xTaskGetTickCount();
      else                                                  can_cmd_proc.timer=100+xTaskGetTickCount();

      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static bool is_head_control_present(uint8_t* output_id)
{
  if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    if(output_id!=NULL) *output_id=HYUNDAI_CRETA_HEAD_CONTROL_OUTPUT_ID;
  }
  else
  {
    if(output_id!=NULL) *output_id=0xff;
    return false;
  }

  return true;
}

//timer_m==0, включить на всегда
static void set_head_control_state(const bool is_on, uint16_t timer_m)
{
  head_control_output_state=is_on;

  if(is_on)
  {
    if(!timer_m) head_control_output_off_timer=0xFFFFFFFF;
    else         head_control_output_off_timer=timer_m*60000;
  }
}

static void head_control_handler(void)
{
  uint8_t output_id;

  if(!is_head_control_present(&output_id))
  {
    return;
  }

  if(head_control_output_state) {System.signal_state.digital_output[output_id]=1;}
  else                          {System.signal_state.digital_output[output_id]=0;}
}

static uint16_t get_delay_after_ign_on_off(void)
{
  //для Octavia делаем задежку больше, т.к. при маленькой задержке включении зажигания и подачи импульса на закрытия на окна, ничего не происходит
  if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM) return 850;
  else                                                                                 return 200;
}

static void check_key_acc_mode_handler()
{ 
  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      can_cmd_proc.timer=xTaskGetTickCount()+400; // wait 400ms for data read
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      
      uint8_t break_event=0;

      if(System.can_state.sec_flags.key_in_acc==1)
      {
        break_event=1;
      }

      if(!break_event)
      {
        add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, IN_PROCESS_NOTIFY_STATE, "check key in ACC ok");
        break;
      }

      const char* res_str=NULL;
      if(break_event==1) res_str="break, key in ACC position";
      
      notify_state_t notify_state=IS_LAST_NOTIFY_STATE;
      uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
      notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);
      
      clear_next_commands(); //clear all commands in scenario
      break;
    }
    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void ign_on_off_handler(bool is_on)
{
  uint8_t output_id;

  if(!is_ignition_output_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(is_on) {System.signal_state.digital_output[output_id]=1;}
      else      {System.signal_state.digital_output[output_id]=0;}
      outputs_state_force_apply();

      can_cmd_proc.timer=get_delay_after_ign_on_off()+xTaskGetTickCount();

      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

//только для передергивания зажиганием перед открытием ЦЗ
static void ford_transit_ign_on_off_handler(bool is_on)
{
  uint8_t output_id;

  if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    output_id=FORD_TRANSIT_TEMP_IGNITION_OUTPUT_ID;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(is_on) {System.signal_state.digital_output[output_id]=1;}
      else      {System.signal_state.digital_output[output_id]=0;}
      outputs_state_force_apply();

      can_cmd_proc.timer=get_delay_after_ign_on_off()+xTaskGetTickCount();

      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void ign_on_off_w_key_handler(bool is_on)
{
  uint8_t output_id;
  uint8_t delay_mul=1;

  if(!is_start_stop_button_output_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    delay_mul=3;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    delay_mul=2;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    delay_mul=2;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(can_cmd_proc.step==0)
  {
    if((is_on && (System.signal_state.key_position=='I' || System.signal_state.key_position=='S')) || \
      (!is_on && (System.signal_state.key_position=='O' || System.signal_state.key_position=='U')))
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=250*delay_mul+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==2)
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      can_cmd_proc.timer=600*delay_mul+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==3)
    {
      if((is_on && (System.signal_state.key_position=='I' || System.signal_state.key_position=='S')) || \
        (!is_on && (System.signal_state.key_position=='O' || System.signal_state.key_position=='U')))
      {
        can_cmd_proc.timer=500*delay_mul+xTaskGetTickCount();
        can_cmd_proc.step=8;
      }
      else
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
    }
    else if(can_cmd_proc.step==4)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=250*delay_mul+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==5)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==6)
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      can_cmd_proc.timer=600*delay_mul+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==7)
    {
      if((is_on && (System.signal_state.key_position=='I' || System.signal_state.key_position=='S')) || \
        (!is_on && (System.signal_state.key_position=='O' || System.signal_state.key_position=='U')))
      {
        can_cmd_proc.timer=500*delay_mul+xTaskGetTickCount();
        can_cmd_proc.step=8;
      }
      else
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
    }
    else if(can_cmd_proc.step==8)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void startstop_button_push_handler(void)
{
  uint8_t output_id;

  if(!is_start_stop_button_output_present(&output_id))
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  uint16_t push_time=250;

  if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    push_time=1050;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      System.signal_state.digital_output[output_id]=1;
      outputs_state_force_apply();

      can_cmd_proc.timer=push_time+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==2)
    {
      System.signal_state.digital_output[output_id]=0;
      outputs_state_force_apply();

      can_cmd_proc.timer=600+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==3)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void delay_handler(void)
{
  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(can_cmd_proc.cmd_q[0]==DELAY_1)      can_cmd_proc.timer=can_cmd_proc.delay[0]+xTaskGetTickCount();
      else if(can_cmd_proc.cmd_q[0]==DELAY_2) can_cmd_proc.timer=can_cmd_proc.delay[1]+xTaskGetTickCount();
      else if(can_cmd_proc.cmd_q[0]==DELAY_3) can_cmd_proc.timer=can_cmd_proc.delay[2]+xTaskGetTickCount();
      else                                    can_cmd_proc.timer=can_cmd_proc.delay[3]+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void check_doors_closed(void)
{
  const char* res_str=NULL;

  if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || \
     System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door || \
     (!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk \
     )
  {
    if(System.can_state.sec_flags.driver_door)                      res_str="break, driver door is open";
    else if(System.can_state.sec_flags.pass_door)                   res_str="break, pass door is open";
    else if(System.can_state.sec_flags.l_rear_door)                 res_str="break, left rear door is open";
    else if(System.can_state.sec_flags.r_rear_door)                 res_str="break, right rear door is open";
    else if((!is_ignore_hood() && System.can_state.sec_flags.hood)) res_str="break, hood door is open";
    else if(System.can_state.sec_flags.trunk)                       res_str="break, trunk door is open";
    else                                                            res_str="break, unknown";

    notify_state_t notify_state=IS_LAST_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    clear_next_commands();//очищаем все последующие команды

    return;
  }

  //проверяем, есть ли еще что-то в cmd_q
  move_next_command_to_head();
  can_cmd_proc.step=0;
  return;
}

static void check_doors_closed_and_ignition_off(void)
{
  const char* res_str=NULL;

  struct
  {
    uint8_t ignition:1;
    uint8_t key_in_ign:1;
  }is_check=
  {
    .ignition=true,
    .key_in_ign=false,
  };

  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    is_check.key_in_ign=true;
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    //is_check.key_in_ign=true;

    //Требуется убрать проверку "Ключ в неправильной позиции" при попытке закончить аренду.
    //Проверку на "Двигатель не заглушен" ОСТАВИТЬ.
    is_check.key_in_ign=false;
    is_check.ignition=false;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    is_check.key_in_ign=true;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
      || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    //is_check.key_in_ign=true;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
  }
  else if(can_program==LADA_LARGUS_2017_PROGRAM)
  {
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    const uint8_t ignore_ignition_check_time_after_warming=10;

    if(!timeAfter(GetSecondsFromStartup(), last_warming_timeS+ignore_ignition_check_time_after_warming))
    {
      is_check.ignition=false;
    }
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    is_check.key_in_ign=true;
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM)
  {
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(  System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door \
    || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door \
    || (!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk \
    || (can_program==MB_E200_2018_PROGRAM && !light_switch_position_in_auto_or_off) \
    || ((can_program==BMW_520I_2018_PROGRAM || can_program==MB_E200_2018_PROGRAM) && !System.can_state.sec_flags.parking) \
    || (can_program==PORSCHE_MACAN_2018_PROGRAM && (System.can_state.cont_flags.marker_lights || System.can_state.cont_flags.dipped_beam || System.can_state.cont_flags.high_beam)) \
    || (can_program==JEEP_WRANGLER_2018_PROGRAM && (!System.signal_state.digital_input[JEEP_WRANGLER_2018_ROOF_INPUT_ID] || System.can_state.cont_flags.fap)) \
    //|| (can_program==JEEP_WRANGLER_2018_PROGRAM && (System.can_state.numeric_data.can_axle_1 != 0x0A && System.can_state.numeric_data.can_axle_1 != 0x0C))
    || (can_program==JEEP_WRANGLER_2018_PROGRAM && (System.can_state.numeric_data.harvest_time != 0x00004C)) \
    || (is_check.key_in_ign && System.can_state.sec_flags.key_in_ign) \
    || (is_check.ignition && (System.can_state.sec_flags.ignition || System.signal_state.ignition)) \
    || System.can_state.sec_flags.engine_is_on)
  {
    if(System.can_state.sec_flags.driver_door)        res_str="break, driver door is open";
    else if(System.can_state.sec_flags.pass_door)     res_str="break, pass door is open";
    else if(System.can_state.sec_flags.l_rear_door)   res_str="break, left rear door is open";
    else if(System.can_state.sec_flags.r_rear_door)   res_str="break, right rear door is open";
    else if((!is_ignore_hood() && System.can_state.sec_flags.hood)) res_str="break, hood door is open";
    else if(System.can_state.sec_flags.trunk)         res_str="break, trunk door is open";
    else if(System.can_state.sec_flags.engine_is_on)  res_str="break, engine is on";
    else if(can_program==MB_E200_2018_PROGRAM && !light_switch_position_in_auto_or_off) res_str="break, wrong light switch position";
    else if((can_program==BMW_520I_2018_PROGRAM || can_program==MB_E200_2018_PROGRAM) && !System.can_state.sec_flags.parking) res_str="break, wrong selector position";
    else if(can_program==PORSCHE_MACAN_2018_PROGRAM && (System.can_state.cont_flags.marker_lights || System.can_state.cont_flags.dipped_beam || System.can_state.cont_flags.high_beam)) res_str="break, wrong light switch position";
    else if(can_program==JEEP_WRANGLER_2018_PROGRAM && (!System.signal_state.digital_input[JEEP_WRANGLER_2018_ROOF_INPUT_ID])) res_str="break, roof is open (sensor 1)";
    else if(can_program==JEEP_WRANGLER_2018_PROGRAM && (System.can_state.cont_flags.fap))                                      res_str="break, roof is open (sensor 2)";
    //else if(can_program==JEEP_WRANGLER_2018_PROGRAM && (System.can_state.numeric_data.can_axle_1 != 0x0A && System.can_state.numeric_data.can_axle_1 != 0x0C)) res_str="break, wrong position of the transfer case selector";
    else if(can_program==JEEP_WRANGLER_2018_PROGRAM && (System.can_state.numeric_data.harvest_time != 0x00004C)) res_str="break, wrong position of the transfer case selector";
    else                                              res_str="break, wrong key position";

    notify_state_t notify_state=IS_LAST_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    clear_next_commands();//очищаем все последующие команды

    return;
  }

  //проверяем, есть ли еще что-то в cmd_q
  move_next_command_to_head();
  can_cmd_proc.step=0;
  return;
}

static void wait_windows_closing(void)
{
  uint16_t wait_time_ms=0;

  if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    wait_time_ms=9000;
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    wait_time_ms=8000;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    wait_time_ms=9000;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM)
  {
    wait_time_ms=9000;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    wait_time_ms=8000;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      can_cmd_proc.timer=wait_time_ms+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(!System.signal_state.windows_position.driver_door &&
         !System.signal_state.windows_position.pass_door &&
         !System.signal_state.windows_position.l_rear_door &&
         !System.signal_state.windows_position.r_rear_door)
      {
        break;
      }
      else if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
      {
        can_cmd_proc.step++;
      }
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void toyota_rav4_windows_closing_handler(void)
{
  uint16_t wait_time_ms=0;
  static uint32_t local_timer;
  static uint8_t window_index;

  if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    wait_time_ms=9000;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  //шлем в течнии wait_time_ms команды на закрытие каждого окнв поочередно
  //прекращаем, если окна закрылись раньше
  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      window_index=0;
      local_timer=wait_time_ms+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(!System.signal_state.windows_position.driver_door &&
         !System.signal_state.windows_position.pass_door &&
         !System.signal_state.windows_position.r_rear_door &&
         !System.signal_state.windows_position.l_rear_door
        )
      {
        break;
      }

      if(timeAfter(xTaskGetTickCount(), local_timer))
      {
        break;
      }

      static const CanTxMsg window_can_cmd[]  =
      {
        {.StdId=0x750, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x90,0x04,0x30,0x01,0x01,0x80,0x00,0x00}},
        {.StdId=0x750, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x91,0x04,0x30,0x01,0x01,0x80,0x00,0x00}},
        {.StdId=0x750, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x92,0x04,0x30,0x01,0x01,0x80,0x00,0x00}},
        {.StdId=0x750, .IDE=CAN_ID_STD, .RTR=CAN_RTR_Data, .DLC=8, .Data={0x93,0x04,0x30,0x01,0x01,0x80,0x00,0x00}},
      };

      bool is_tx_enable=false;

      if(window_index==0)
      {
        if(System.signal_state.windows_position.driver_door)
        {
          is_tx_enable=true;
        }
      }
      else if(window_index==1)
      {
        if(System.signal_state.windows_position.pass_door)
        {
          is_tx_enable=true;
        }
      }
      else if(window_index==2)
      {
        if(System.signal_state.windows_position.r_rear_door)
        {
         is_tx_enable=true;
        }
      }
      else if(window_index==3)
      {
        if(System.signal_state.windows_position.l_rear_door)
        {
          is_tx_enable=true;
        }
      }

      if(is_tx_enable)
      {
        CanTxMsg tx;
        memcpy(&tx, &window_can_cmd[window_index], sizeof(tx));
        CAN_Transmit_Blocked(CAN1, &tx, 20);
      }

      window_index++;
      if(window_index>=4) window_index=0;
      can_cmd_proc.timer=xTaskGetTickCount()+50;
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==2)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer))
      {
        can_cmd_proc.step--;
      }
    }
    else
    {
      //done
      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void check_cls_and_windows_state(bool cls_check, bool windows_check)
{
  const char* res_str=NULL;

  if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    //у MB E200 не читаем, поэтому обнуляем
    System.signal_state.cls_state.driver_door=0;
    System.signal_state.cls_state.pass_door=0;
    System.signal_state.cls_state.l_rear_door=0;
    System.signal_state.cls_state.r_rear_door=0;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    //у XC60 хз как считано, поэтому обнуляем
    System.signal_state.windows_position.driver_door=0;
    System.signal_state.windows_position.pass_door=0;
    System.signal_state.windows_position.l_rear_door=0;
    System.signal_state.windows_position.r_rear_door=0;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    //у XC40 хз как считано, поэтому обнуляем
    //System.signal_state.windows_position.driver_door=0;
    //System.signal_state.windows_position.pass_door=0;
    //System.signal_state.windows_position.l_rear_door=0;
    //System.signal_state.windows_position.r_rear_door=0;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    ////у Velar хз как считано, поэтому обнуляем
    //System.signal_state.windows_position.driver_door=0;
    //System.signal_state.windows_position.pass_door=0;
    //System.signal_state.windows_position.l_rear_door=0;
    //System.signal_state.windows_position.r_rear_door=0;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    //у Qashqai не читаем, поэтому обнуляем
    System.signal_state.windows_position.driver_door=0;
    System.signal_state.windows_position.pass_door=0;
    System.signal_state.windows_position.l_rear_door=0;
    System.signal_state.windows_position.r_rear_door=0;
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    //то что не читаем обнуляем
    System.signal_state.cls_state.driver_door=0;
    System.signal_state.cls_state.pass_door=0;
    System.signal_state.cls_state.l_rear_door=0;
    System.signal_state.cls_state.r_rear_door=0;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    //то что не читаем обнуляем
    System.signal_state.cls_state.driver_door=0;
    System.signal_state.cls_state.pass_door=0;
    System.signal_state.cls_state.l_rear_door=0;
    System.signal_state.cls_state.r_rear_door=0;
    System.signal_state.windows_position.driver_door=0;
    System.signal_state.windows_position.pass_door=0;
    System.signal_state.windows_position.l_rear_door=0;
    System.signal_state.windows_position.r_rear_door=0;
  }
  else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM)
  {
  }
  else if(can_program==SKODA_OCTAVIA_2018_PROGRAM)
  {
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(((can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) && System.signal_state.is_br_ok!=1) || \
    (cls_check && (System.signal_state.cls_state.driver_door || System.signal_state.cls_state.pass_door || System.signal_state.cls_state.l_rear_door || System.signal_state.cls_state.r_rear_door)) || \
      (windows_check && (System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)) || \
        (can_program==FORD_MUSTANG_1965_PROGRAM && System.signal_state.ext_digital_input[FORD_MUSTANG65_FUEL_CAP_BRINPUT_ID]) || \
          (can_program==FORD_MUSTANG_1969_PROGRAM && System.signal_state.ext_digital_input[FORD_MUSTANG69_FUEL_CAP_BRINPUT_ID]) || \
            (can_program==FORD_MUSTANG_1969_PROGRAM && (!System.signal_state.ext_digital_input[FORD_MUSTANG69_ROOF_BRINPUT_ID] || \
                                                        !System.signal_state.ext_digital_input[FORD_MUSTANG69_LEFT_ROOF_LOCK_BRINPUT_ID] || \
                                                        !System.signal_state.ext_digital_input[FORD_MUSTANG69_RIGHT_ROOF_LOCK_BRINPUT_ID]))
              )
  {
    if((can_program==FORD_MUSTANG_1969_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM ) && System.signal_state.is_br_ok!=1)                    res_str="break, external board err";
    else if(cls_check && (System.signal_state.cls_state.driver_door || System.signal_state.cls_state.pass_door || System.signal_state.cls_state.l_rear_door || System.signal_state.cls_state.r_rear_door))                        res_str="break, doors not locked";
    else if(windows_check && (System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door))
    {
      if(can_program==FORD_MUSTANG_1965_PROGRAM)
      {
        if(System.signal_state.windows_position.driver_door)      res_str="break, driver's window not closed";
        else if(System.signal_state.windows_position.pass_door)   res_str="break, passenger's window not closed";
        else if(System.signal_state.windows_position.l_rear_door) res_str="break, driver's window leaf not closed";
        else if(System.signal_state.windows_position.r_rear_door) res_str="break, passenger's window leaf not closed";
        else                                                      res_str="break, unknown";
      }
      else
      {
        if(System.signal_state.windows_position.driver_door)      res_str="break, driver's window not closed";
        else if(System.signal_state.windows_position.pass_door)   res_str="break, passenger's window not closed";
        else if(System.signal_state.windows_position.l_rear_door) res_str="break, left rear window not closed";
        else if(System.signal_state.windows_position.r_rear_door) res_str="break, right rear window not closed";
        else                                                      res_str="break, unknown";
        //res_str="break, windows not closed";
      }
    }
    else if((can_program==FORD_MUSTANG_1965_PROGRAM && !!System.signal_state.ext_digital_input[FORD_MUSTANG65_FUEL_CAP_BRINPUT_ID]) || \
      (can_program==FORD_MUSTANG_1969_PROGRAM && System.signal_state.ext_digital_input[FORD_MUSTANG69_FUEL_CAP_BRINPUT_ID]))                           res_str="break, no fuel cap";
    else if(can_program==FORD_MUSTANG_1969_PROGRAM && !System.signal_state.ext_digital_input[FORD_MUSTANG69_ROOF_BRINPUT_ID])                          res_str="break, roof is open";
    else if(can_program==FORD_MUSTANG_1969_PROGRAM && !System.signal_state.ext_digital_input[FORD_MUSTANG69_LEFT_ROOF_LOCK_BRINPUT_ID])                res_str="break, left roof lock is open";
    else if(can_program==FORD_MUSTANG_1969_PROGRAM && !System.signal_state.ext_digital_input[FORD_MUSTANG69_RIGHT_ROOF_LOCK_BRINPUT_ID])               res_str="break, right roof lock is open";
    else                                                                                                                                               res_str="break, unknown";

    notify_state_t notify_state=IS_LAST_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    clear_next_commands();//очищаем все последующие команды

    return;
  }

  //проверяем, есть ли еще что-то в cmd_q
  move_next_command_to_head();
  can_cmd_proc.step=0;
  return;
}

static void check_roof_state(void)
{
  const char* res_str=NULL;

  if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  if(System.signal_state.roof.sunroof)
  {
    if(System.signal_state.roof.sunroof) res_str="break, sunroof is open";
    else                                 res_str="break, unknown";

    notify_state_t notify_state=IS_LAST_NOTIFY_STATE;
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

    clear_next_commands();//очищаем все последующие команды

    return;
  }

  //проверяем, есть ли еще что-то в cmd_q
  move_next_command_to_head();
  can_cmd_proc.step=0;
  return;
}

static void skip_two_cmd_if_cls_opened_closed(const bool is_closed_check)
{
  if(!is_closed_check && (!System.signal_state.cls_state.driver_door || !System.signal_state.cls_state.pass_door || !System.signal_state.cls_state.l_rear_door || !System.signal_state.cls_state.r_rear_door))
  {
    //двери закрыты
  }
  else if(is_closed_check && (System.signal_state.cls_state.driver_door && System.signal_state.cls_state.pass_door && System.signal_state.cls_state.l_rear_door && System.signal_state.cls_state.r_rear_door))
  {
    //двери открыты
  }
  else
  {
    //двери открыты, пропускам две команды
    move_next_command_to_head();
    can_cmd_proc.step=0;

    move_next_command_to_head();
    can_cmd_proc.step=0;
  }

  //проверяем, есть ли еще что-то в cmd_q
  move_next_command_to_head();
  can_cmd_proc.step=0;

  return;
}

static void check_bmw_cls_state(void)
{
  if(can_program==BMW_520I_2018_PROGRAM)
  {
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  //если замок стал открытым (81) то выставляем признак необходимости проверки настроек головы
  //после прогрвева должен быть в неизвестно (83)
  if(_cls_state==UNKNOWN_CLS_STATE) System.can_state.sec_flags.alarm_act=1;
  else                              System.can_state.sec_flags.alarm_act=0;

  //проверяем, есть ли еще что-то в cmd_q
  move_next_command_to_head();
  can_cmd_proc.step=0;
  return;
}

static void check_parking_light_state(bool is_only_parking_check, bool when_checked_disable_wired_block)
{
  static bool wired_block_state_when_check_initiated=false;

  const char* res_str=NULL;
  uint8_t ignition_output_id;

  uint16_t ign_on_time_ms=2000;

  struct
  {
    uint8_t parking:1;
    uint8_t handbrake:1;
    uint8_t light:2; // 1 - проверка ближний и дальний, 2 - проверка габароиты, ближний, дальний
    uint8_t windscreen_wiper:1;
  }is_check=
  {
    .parking=1,
    .handbrake=0,
    .light=1,
    .windscreen_wiper=0,
  };

  if(is_only_parking_check) is_check.light=0;

  if(can_program==KIA_RIO_2017_PROGRAM)
  {
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    //todo:  ХЗ как проверить паркинг для kaptur, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    //todo: паркинг на genesis не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
    if(IS_VW_TRANSPORTER_WITH_MANUAL_GEARBOX())
    {
      is_check.parking=0;
      is_check.handbrake=1;
      ign_on_time_ms=3700;//для того чтобы на приборной панели успели погаснуть идикаторы
    }
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
    //паркинг на Transit не проверяем
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
      || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM \
        || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
    if(can_program==VW_POLO_2018_PROGRAM)
    {
      //is_check.windscreen_wiper=1; //Датчик дворников работает не корректно
    }
  }
  else if(can_program==VW_CADDY_2018_PROGRAM)
  {
    //паркинг на Caddy не проверяем, т.к. механика
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
    if(IS_CITROEN_JUMPY_WITH_MANUAL_GEARBOX())
    {
      is_check.parking=0;
      is_check.handbrake=1;
      ign_on_time_ms=3700;//для того чтобы на приборной панели успели погаснуть идикаторы
    }
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    //todo: паркинг на E200 не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    //todo: паркинг на 520i не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
  }
  else if(can_program==LADA_LARGUS_2017_PROGRAM)
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    //todo: паркинг на XC60 не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    //todo: паркинг на XC40 не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    //todo: паркинг на XC60 не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    //если подергаем зажиганием, то двери не будет закрываться по CAN примерно в течении 5 секунд
    /*
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
    */
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    //todo: паркинг на X1 не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)
  {
    is_check.parking=1;
    is_check.handbrake=1;
  }
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)
  {
    is_check.parking=1;
    is_check.handbrake=1;
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    is_check.parking=0;
    is_check.light=2; //+габариты
    is_check.handbrake=1;
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
    is_check.parking=1;
    is_check.light=2; //+габариты
    is_check.handbrake=0;
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
    is_check.parking=1;
    is_check.light=2; //+габариты
    is_check.handbrake=0;
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    //todo: паркинг на WRANGLER не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    //todo: паркинг на AUDI_A6_2018 не проверяем, считаем что все ok
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      if(when_checked_disable_wired_block)
      {
        wired_block_state_when_check_initiated=is_wired_block_relay_enable;
        is_wired_block_relay_enable=false;
      }

      if(is_ignition_output_present(&ignition_output_id))
      {
        System.signal_state.digital_output[ignition_output_id]=1;
        outputs_state_force_apply();
      }

      can_cmd_proc.timer=xTaskGetTickCount()+ign_on_time_ms;
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else
    {
      uint8_t break_event=0;

      if(is_check.parking && !System.can_state.sec_flags.parking)                                                   {break_event=1;}
      else if(is_check.light==2 && (System.can_state.cont_flags.marker_lights ||
                                    System.can_state.cont_flags.dipped_beam ||
                                    System.can_state.cont_flags.high_beam))                                         {break_event=2;}
      else if(is_check.light && (System.can_state.cont_flags.dipped_beam || System.can_state.cont_flags.high_beam)) {break_event=2;}
      else if(is_check.handbrake && !System.can_state.sec_flags.hand_break)                                         {break_event=3;}
      else if(is_check.windscreen_wiper && windscreen_wiper)                                                        {break_event=4;}

      if(is_ignition_output_present(&ignition_output_id))
      {
        System.signal_state.digital_output[ignition_output_id]=0;
        outputs_state_force_apply();
      }

      if(!break_event)
      {
        add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, IN_PROCESS_NOTIFY_STATE, "check parking and light ok");

        if(when_checked_disable_wired_block)
        {
          is_wired_block_relay_enable=wired_block_state_when_check_initiated;
        }

        break;
      }

      if(break_event==1) res_str="break, wrong selector position";
      else if(break_event==2) res_str="break, light is on";
      else if(break_event==3) res_str="break, handbreak is't applied";
      else if(break_event==4) res_str="break, windscreen wiper is on";

      notify_state_t notify_state=IS_LAST_NOTIFY_STATE;
      uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
      notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);

      clear_next_commands();//очищаем все последующие команды

      if(when_checked_disable_wired_block)
      {
        is_wired_block_relay_enable=wired_block_state_when_check_initiated;
      }

      break;
    }

    return; //continue command procces
  }
  //end command procces

  move_next_command_to_head();
  can_cmd_proc.step=0;
}

static void add_server_notify_from_cmd(void)
{
  static uint8_t cmd_attempt_counter=1;
  const char* res_str=NULL;
  notify_state_t notify_state=IS_LAST_NOTIFY_STATE;

  if(can_program==KIA_RIO_2017_PROGRAM)
  {
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM \
    || can_program==VW_POLO_2018_PROGRAM  || can_program==VW_CADDY_2018_PROGRAM \
      || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM)
  {
  }
  else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
  {
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
  }
  else if(can_program==HYUNDAI_SOLARIS_2017_PROGRAM)
  {
  }
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
  {
  }
  else if(can_program==LADA_LARGUS_2017_PROGRAM)
  {
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
  }
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)
  {
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
  }
  else if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM)
  {
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
  }
  else if(can_program==HYUNDAI_SONATA_SD_2020_PROGRAM)
  {
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
  }
  else if(can_program==RENAULT_DUSTER_2021_PROGRAM)
  {
  }
  else if(can_program==RENAULT_ARKANA_2018_PROGRAM)
  {
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  bool is_notify_enable=true;

  if(can_cmd_proc.cmd_q[0]==START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE)
  {
    if(is_taxi())
    {
      lease_state=IN_LEASE_STATE;
      res_str="taxi unlock done";
    }
    else
    {
      if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
      {
        if(!System.signal_state.cls_state.driver_door || !System.signal_state.cls_state.pass_door || !System.signal_state.cls_state.l_rear_door || !System.signal_state.cls_state.r_rear_door)
        {
          if(cmd_attempt_counter<3)
          {
            uint8_t i=1;
            can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=500; //ждем перед новой попыткой
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
            can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

            notify_state=IN_PROCESS_NOTIFY_STATE;
            cmd_attempt_counter++;
          }
          else
          {
            cmd_attempt_counter=1;
          }

          res_str="start of lease err, doors unlock fail";
        }
        else
        {
          cmd_attempt_counter=1;
          res_str="start of lease done";
        }
      }
      else if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM ||
              can_program==PORSCHE_PANAMERA_2019_PROGRAM || can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM)
      {
        if(!System.signal_state.cls_state.driver_door || !System.signal_state.cls_state.pass_door || !System.signal_state.cls_state.l_rear_door || !System.signal_state.cls_state.r_rear_door)
        {
          //!!! блокировки остались выключенными !!!
          res_str="start of lease err, doors unlock fail";
        }
        else
        {
          res_str="start of lease done";
        }
      }
      else
      {
        res_str="start of lease done";
      }

      if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM)
      {
        if(can_program==FORD_MUSTANG_1965_PROGRAM)
        {
          System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_EN_BROUTPUT_ID]=1;
          System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_DIS_BROUTPUT_ID]=0;
          vTaskDelay(750);
          System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_EN_BROUTPUT_ID]=0;
          System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_DIS_BROUTPUT_ID]=0;
        }
        else
        {
          System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_EN_BROUTPUT_ID]=1;
          System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_DIS_BROUTPUT_ID]=0;
          vTaskDelay(750);
          System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_EN_BROUTPUT_ID]=0;
          System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_DIS_BROUTPUT_ID]=0;
        }

#if defined(JQ6500_AUDUO_PRESENT)
        yandex_audio_cmd_callback("\x00\x01\x00", 3, NULL, 0xFF, 0, 0, NULL);
#endif //defined(JQ6500_AUDUO_PRESENT)
      }
    }
    //lease_state=IN_LEASE_STATE;
    set_head_control_state(true, 0);
    save_lease_state();
  }
  else if(can_cmd_proc.cmd_q[0]==END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE)
  {
    if(is_taxi())
    {
      res_str="taxi soft lock done";
    }
    else
    {
      res_str="end of lease done";
    }

    if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM)
    {
      if(can_program==FORD_MUSTANG_1965_PROGRAM)
      {
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_DIS_BROUTPUT_ID]=1;
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_EN_BROUTPUT_ID]=0;
        vTaskDelay(750);
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_DIS_BROUTPUT_ID]=0;
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_EN_BROUTPUT_ID]=0;
      }
      else
      {
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_DIS_BROUTPUT_ID]=1;
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_EN_BROUTPUT_ID]=0;
        vTaskDelay(750);
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_DIS_BROUTPUT_ID]=0;
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_EN_BROUTPUT_ID]=0;
      }
    }

    set_head_control_state(false, 0);
    lease_state=IN_END_LEASE_STATE;
    save_lease_state();
    is_ignore_end_lease_state=false;
  }
  else if(can_cmd_proc.cmd_q[0]==FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE)
  {
    if(is_taxi())
    {
      res_str="taxi hard lock done";
    }
    else
    {
      res_str="forced end of lease done";
    }

    if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM)
    {
      if(can_program==FORD_MUSTANG_1965_PROGRAM)
      {
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_DIS_BROUTPUT_ID]=1;
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_EN_BROUTPUT_ID]=0;
        vTaskDelay(750);
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_DIS_BROUTPUT_ID]=0;
        System.signal_state.ext_digital_output[FORD_MUSTANG65_POWER_EN_BROUTPUT_ID]=0;
      }
      else
      {
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_DIS_BROUTPUT_ID]=1;
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_EN_BROUTPUT_ID]=0;
        vTaskDelay(750);
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_DIS_BROUTPUT_ID]=0;
        System.signal_state.ext_digital_output[FORD_MUSTANG69_POWER_EN_BROUTPUT_ID]=0;
      }
    }

    set_head_control_state(false, 0);
    lease_state=IN_FORCED_END_LEASE_STATE;
    save_lease_state();
    is_ignore_end_lease_state=false;
  }
  else if(can_cmd_proc.cmd_q[0]==NOT_AVAILABLE_NOTTIFY_CMD_TYPE)
  {
    res_str="not available";
  }
  else if(can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE)
  {
    if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
    {
      if(System.signal_state.cls_state.driver_door || System.signal_state.cls_state.pass_door || System.signal_state.cls_state.l_rear_door || System.signal_state.cls_state.r_rear_door)
      {
        if(cmd_attempt_counter<3)
        {
          uint8_t i=1;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=500; //ждем перед новой попыткой
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;

          notify_state=IN_PROCESS_NOTIFY_STATE;
          cmd_attempt_counter++;
        }
        else
        {
          cmd_attempt_counter=1;
        }

        res_str="doors lock fail";
      }
      else
      {
        cmd_attempt_counter=1;
        res_str="doors lock done";
      }
    }
    else if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM ||
            can_program==PORSCHE_PANAMERA_2019_PROGRAM || can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM)
    {
      if(System.signal_state.cls_state.driver_door || System.signal_state.cls_state.pass_door || System.signal_state.cls_state.l_rear_door || System.signal_state.cls_state.r_rear_door)
      {
        res_str="doors lock fail";
      }
      else
      {
        res_str="doors lock done";
      }
    }
    else
    {
      res_str="doors lock done";
    }
    is_ignore_end_lease_state=false;
  }
  else if(can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE)
  {
    if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
    {
      if(!System.signal_state.cls_state.driver_door || !System.signal_state.cls_state.pass_door || !System.signal_state.cls_state.l_rear_door || !System.signal_state.cls_state.r_rear_door)
      {
        if(cmd_attempt_counter<3)
        {
          uint8_t i=1;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=500; //ждем перед новой попыткой
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

          notify_state=IN_PROCESS_NOTIFY_STATE;
          cmd_attempt_counter++;
        }
        else
        {
          cmd_attempt_counter=1;
        }

        res_str="doors unlock fail";
      }
      else
      {
        cmd_attempt_counter=1;
        res_str="doors unlock done";
      }
    }
    else if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM ||
            can_program==PORSCHE_PANAMERA_2019_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM)
    {
      if(!System.signal_state.cls_state.driver_door || !System.signal_state.cls_state.pass_door || !System.signal_state.cls_state.l_rear_door || !System.signal_state.cls_state.r_rear_door)
      {
        res_str="doors unlock fail";
      }
      else
      {
        res_str="doors unlock done";
      }
    }
    else
    {
      res_str="doors unlock done";
    }
    //is_ignore_end_lease_state=false;
  }
  else if(can_cmd_proc.cmd_q[0]==HOOD_LOCK_NOTIFY_CMD_TYPE)
  {
    res_str="hood lock done";
  }
  else if(can_cmd_proc.cmd_q[0]==HOOD_UNLOCK_NOTIFY_CMD_TYPE)
  {
    res_str="hood unlock done";
  }
  else if(can_cmd_proc.cmd_q[0]==ADD_EMTY_LAST_NOTIFY_CMD_TYPE)
  {
    res_str=NULL;//только uid
  }
  else
  {
    is_notify_enable=false;
  }

  if(is_notify_enable)
  {
    uint8_t add_res=add_server_notify(can_cmd_proc.server_notify_id, can_cmd_proc.uid, can_cmd_proc.nonce, notify_state, res_str);
    notify_logging(can_cmd_proc.server_notify_id, can_cmd_proc.uid, notify_state, res_str, add_res);
  }

  //end command procces
  move_next_command_to_head();
  can_cmd_proc.step=0;
}

//static const uint8_t wake_trace_max_idx = 12;
//static const uint8_t wake_trace_max_len = 48;
//static char wake_trace_buff[wake_trace_max_idx][wake_trace_max_len];
//static uint8_t wake_trace_idx;
//static uint32_t wake_trace_timer;

static void reset_wake_trace_pos(void)
{
//  wake_trace_idx = 0;
//  wake_trace_timer = xTaskGetTickCount();
}

static void wake_trace(void)
{
//  if(wake_trace_idx >= wake_trace_max_idx) return;
//
//  snprintf(wake_trace_buff[wake_trace_idx], wake_trace_max_len,
//           "%02hhu,%u,%04u,%hhu,%hhu,%hhu,%hhu,%hhu,%hhu\n",
//           can_cmd_proc.step, xTaskGetTickCount(), xTaskGetTickCount() - wake_trace_timer,
//           System.signal_state.digital_output[BMW_X1_START_STOP_BUTTON_OUTPUT_ID], System.signal_state.digital_output[BMW_X1_DRIVER_DOOR_EMULATION_OUTPUT_ID], System.signal_state.digital_output[BMW_X1_PEADAL_BREAK_OUTPUT_ID], System.signal_state.digital_output[BMW_X1_KEY_POWER_OUTPUT_ID],
//           System.can_state.sec_flags.ignition, System.signal_state.ignition);
//  wake_trace_idx++;
//
//  wake_trace_timer = xTaskGetTickCount();
}

static void print_wake_trace(void)
{
//  LOG("step,tim,dtim,ststb,demul,pedal,kpow,aign,cign\n");
//
//  for(int i = 0; i < wake_trace_idx; i++)
//  {
//    LOG("%s", wake_trace_buff[i]);
//  }
}

static void can_wakeup_handler(void)
{
  if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    for(;;)
    {
      //на новых комплектациях машина пробуждается эмуляцией водительской двери
      const uint8_t output_id=KAPTUR_DRIVER_DOOR_EMULATION_OUTPUT_ID;

      if(can_cmd_proc.step==0)
      {
        //если есть активность на шине, то пропускаем эмуляцию водительской двери
        if(can_program==RENAULT_DOKKER_2018_PROGRAM || !is_can_rx_in_idle(CAN1_ID))
        {
          can_cmd_proc.step=4;
        }
        else
        {
          restart_can_timer(DRIVER_DOOR_IGNORE_TIMER_ID, 5000);

          System.signal_state.digital_output[output_id]=1;
          outputs_state_force_apply();

          can_cmd_proc.timer=500+xTaskGetTickCount();
          can_cmd_proc.step++;
        }
      }
      else if(can_cmd_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==2)
      {
        System.signal_state.digital_output[output_id]=0;
        outputs_state_force_apply();

        can_cmd_proc.timer=1000+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==3)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==4)
      {
        static const CanTxMsg wake_up_mess[] =
        {
          {0x090, 0, CAN_ID_STD, CAN_RTR_Data, 4, {0x00,0xFF,0xf0,0x00,0x00,0x00,0x00,0x00}},
          {0x682, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}}
        };

        for(uint8_t i=0; i<(sizeof(wake_up_mess)/sizeof(wake_up_mess[0])); i++)
        {
          CanTxMsg TxMessage;
          memcpy(&TxMessage, &wake_up_mess[i], sizeof(TxMessage));
          CAN_Transmit_Blocked(CAN1, &TxMessage, 25);
        }

        can_cmd_proc.timer=50+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==5)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==6)
      {
        static const CanTxMsg wake_up_mess[] =
        {
          {0x090, 0, CAN_ID_STD, CAN_RTR_Data, 4, {0x00,0xFF,0xf0,0x00,0x00,0x00,0x00,0x00}},
          {0x35D, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00}}
        };

        for(uint8_t i=0; i<(sizeof(wake_up_mess)/sizeof(wake_up_mess[0])); i++)
        {
          CanTxMsg TxMessage;
          memcpy(&TxMessage, &wake_up_mess[i], sizeof(TxMessage));
          CAN_Transmit_Blocked(CAN1, &TxMessage, 25);
        }

        can_cmd_proc.timer=50+xTaskGetTickCount();
        can_cmd_proc.step++;
      }
      else if(can_cmd_proc.step==7)
      {
        if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
      }
      else
      {
        //end command procces
        break;
      }

      return; //continue command procces
    }
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    //передергиваем зажигание
    //передергиваем эмуляцию двери
    CAN_TypeDef* const CANx=CAN2;//управление по CAN1, но для контроля используем CAN2

    uint8_t start_stop_output_id;
    const uint8_t door_emulation_output_id=BMW_X1_DRIVER_DOOR_EMULATION_OUTPUT_ID;

    if(!is_start_stop_button_output_present(&start_stop_output_id) || door_emulation_output_id==0xFF)
    {
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
      return;
    }

    if((!can_cmd_proc.step && !(System.can_state.sec_flags.pedal_break || System.signal_state.ignition) && is_can_rx_in_idle(get_can_from_id_from_periph(CANx))) \
      || can_cmd_proc.step)
    {
      for(;;)
      {
        if(can_cmd_proc.step==0) {reset_wake_trace_pos(); wake_trace();}

        if(can_cmd_proc.step==0 || can_cmd_proc.step==4)
        {
          System.signal_state.digital_output[start_stop_output_id]=1;
          outputs_state_force_apply();

          can_cmd_proc.timer=300+xTaskGetTickCount();

          wake_trace();
          can_cmd_proc.step++;
        }
        else if(can_cmd_proc.step==1 || can_cmd_proc.step==3 || can_cmd_proc.step==5 || can_cmd_proc.step==7 || can_cmd_proc.step==9 || can_cmd_proc.step==11 || can_cmd_proc.step==13 || can_cmd_proc.step==15)
        {
          if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) {can_cmd_proc.step++;}
        }
        else if(can_cmd_proc.step==2 || can_cmd_proc.step==6)
        {
          System.signal_state.digital_output[start_stop_output_id]=0;
          outputs_state_force_apply();

          if(can_cmd_proc.step==6) can_cmd_proc.timer=1000+xTaskGetTickCount();
          else                     can_cmd_proc.timer=500+xTaskGetTickCount();

          wake_trace();
          can_cmd_proc.step++;
        }
        else if(can_cmd_proc.step==8)
        {
          if(System.can_state.sec_flags.ignition || System.signal_state.ignition)
          {
            System.signal_state.digital_output[start_stop_output_id]=1;
            outputs_state_force_apply();

            can_cmd_proc.timer=300+xTaskGetTickCount();
            wake_trace();
            can_cmd_proc.step++;
          }
          else
          {
            wake_trace();
            can_cmd_proc.step=12;
          }
        }
        else if(can_cmd_proc.step==10)
        {
          System.signal_state.digital_output[start_stop_output_id]=0;
          outputs_state_force_apply();

          can_cmd_proc.timer=500+xTaskGetTickCount();
          wake_trace();
          can_cmd_proc.step++;
        }
        else if(can_cmd_proc.step==12)
        {
          System.signal_state.digital_output[door_emulation_output_id]=1;
          outputs_state_force_apply();

          can_cmd_proc.timer=500+xTaskGetTickCount();
          wake_trace();
          can_cmd_proc.step++;
        }
        else if(can_cmd_proc.step==14)
        {
          System.signal_state.digital_output[door_emulation_output_id]=0;
          outputs_state_force_apply();

          can_cmd_proc.timer=500+xTaskGetTickCount();
          wake_trace();
          can_cmd_proc.step++;
        }
        else
        {
          wake_trace();
          print_wake_trace();
          //end command procces
          break;
        }

        return; //continue command procces
      }
    }

    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else if(can_program==BMW_520I_2018_PROGRAM || can_program==MB_E200_2018_PROGRAM ||
          can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM ||
          can_program==NISSAN_QASHQAI_2018_PROGRAM ||
          can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM ||
          /*can_program==BMW_X1_2019_PROGRAM || */
          can_program==RENAULT_LOGAN_2013_PROGRAM ||
          can_program==RENAULT_DUSTER_2021_PROGRAM ||
          can_program==RENAULT_ARKANA_2018_PROGRAM ||
          can_program==LADA_LARGUS_2020_PROGRAM
          )
  {
    CAN_TypeDef* CANx;
    CanTxMsg wake_message;

    uint16_t count = 120;
    uint8_t cycle_time_ms = 25;
    uint8_t skip_idle_check = 0;

    if(can_program==BMW_520I_2018_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN2;//команды по CAN1, но пробуждается по CAN2!
    }
    else if(can_program==MB_E200_2018_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;//нужно проверить, возможно пробуждение по CAN2?
    }
    else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;
    }
    else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
    {
      static const CanTxMsg message={0x7FF, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;
    }
    else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;
    }
    else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;
      count=120;//Pandora шлет несколько раз по 200
      cycle_time_ms=25;//Pandora шлет с периодом 10 мс
    }
    else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN2;
      count=266;//Pandora шлет 400 раз
      cycle_time_ms=15;//Pandora шлет с периодом 10 мс
    }
    else if(can_program==BMW_X1_2019_PROGRAM)
    {
      static const CanTxMsg message={0x130, 0, CAN_ID_STD, CAN_RTR_Data, 5, {0x00,0xF0,0xFC,0xFF,0xFF,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;
      count=20;
      cycle_time_ms=25;
    }
    else if(can_program==VOLVO_XC40_2018_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;
      count=120;
      cycle_time_ms=25;
    }
    else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
    {
      static const CanTxMsg message={0x7F8, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx=CAN1;
      count=50;
      cycle_time_ms=10;
    }
    else if(can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM)
    {
      static const CanTxMsg message={0x7FF, 0, CAN_ID_STD, CAN_RTR_Data, 1, {0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
      memcpy(&wake_message, &message, sizeof(wake_message));
      CANx = CAN1;
      count = 50;
      cycle_time_ms = 10;
      skip_idle_check = 1; //шлем всегда, не зависимо от активности шины
    }

    if(get_can_from_id_from_periph(CANx)<HW_CAN_COUNT && (skip_idle_check || is_can_rx_in_idle(get_can_from_id_from_periph(CANx))))
    {
      for(;;)
      {
        if(can_cmd_proc.step>=2*count)//повторяем count раз, с интервалом cycle_time_ms
        {
          break; //end command procces
        }
        else if(!(can_cmd_proc.step%2))
        {
          CAN_Transmit_Blocked(CANx, &wake_message, 15);

          can_cmd_proc.timer=cycle_time_ms+xTaskGetTickCount();
          can_cmd_proc.step++;
        }
        else
        {
          if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
        }

        return; //continue command procces
      }
    }

    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }
}

static void can_tx_commands_handler(void)
{
  const can_commands_table_t* can_commands_table;
  uint16_t can_commands_table_elements_count;
  //static command_repeat;

  if(can_program==KIA_RIO_2017_PROGRAM)
  {
    can_commands_table=kia_rio_2017_can_commands_table;
    can_commands_table_elements_count=sizeof(kia_rio_2017_can_commands_table)/sizeof(kia_rio_2017_can_commands_table[0]);
  }
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)
  {
    can_commands_table=renault_kaptur_can_commands_table;
    can_commands_table_elements_count=sizeof(renault_kaptur_can_commands_table)/sizeof(renault_kaptur_can_commands_table[0]);
  }
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
  {
    can_commands_table=hyundai_genesis_can_commands_table;
    can_commands_table_elements_count=sizeof(hyundai_genesis_can_commands_table)/sizeof(hyundai_genesis_can_commands_table[0]);
  }
  else if(can_program==AUDI_A3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM)
  {
#if defined(IS_HW_VERSION_2)
    //для новых A3 меняем команду на закрытие окон
    //эти блоки ставятся только на V2
    if(can_program==AUDI_A3_2018_PROGRAM && can_cmd_proc.cmd_q[0]==WINDOWS_CLOSING_CMD_TYPE)
    {
      can_cmd_proc.cmd_q[0]=WINDOWS_CLOSING_NEW_A3_CMD_TYPE;
    }
#endif //defined(IS_HW_VERSION_2)
    can_commands_table=audi_a3_can_commands_table;
    can_commands_table_elements_count=sizeof(audi_a3_can_commands_table)/sizeof(audi_a3_can_commands_table[0]);
  }
  else if(can_program==AUDI_Q3_2018_PROGRAM)
  {
    can_commands_table=audi_q3_can_commands_table;
    can_commands_table_elements_count=sizeof(audi_q3_can_commands_table)/sizeof(audi_q3_can_commands_table[0]);
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
  {
    can_commands_table=porsche_can_commands_table;
    can_commands_table_elements_count=sizeof(porsche_can_commands_table)/sizeof(porsche_can_commands_table[0]);
  }
  else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    can_commands_table=porsche_panamera_2019_can_commands_table;
    can_commands_table_elements_count=sizeof(porsche_panamera_2019_can_commands_table)/sizeof(porsche_panamera_2019_can_commands_table[0]);
  }
  else if(can_program==MB_E200_2018_PROGRAM)
  {
    can_commands_table=mb_e200_2018_can_commands_table;
    can_commands_table_elements_count=sizeof(mb_e200_2018_can_commands_table)/sizeof(mb_e200_2018_can_commands_table[0]);
  }
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
  {
    can_commands_table=toyota_rav4_2019_can_commands_table;
    can_commands_table_elements_count=sizeof(toyota_rav4_2019_can_commands_table)/sizeof(toyota_rav4_2019_can_commands_table[0]);
  }
  else if(can_program==BMW_520I_2018_PROGRAM)
  {
    can_commands_table=bmw_520i_can_commands_table;
    can_commands_table_elements_count=sizeof(bmw_520i_can_commands_table)/sizeof(bmw_520i_can_commands_table[0]);
  }
  else if(can_program==HYUNDAI_SOLARIS_2017_PROGRAM)
  {
    can_commands_table=hyundai_solaris_can_commands_table;
    can_commands_table_elements_count=sizeof(hyundai_solaris_can_commands_table)/sizeof(hyundai_solaris_can_commands_table[0]);
  }
  /*
  else if(can_program==VW_POLO_2018_PROGRAM)
  {
  can_commands_table=vw_polo_can_commands_table;
  can_commands_table_elements_count=sizeof(vw_polo_can_commands_table)/sizeof(vw_polo_can_commands_table[0]);
  }
  */
  else if(can_program==VW_CADDY_2018_PROGRAM)
  {
    can_commands_table=vw_caddy_can_commands_table;
    can_commands_table_elements_count=sizeof(vw_caddy_can_commands_table)/sizeof(vw_caddy_can_commands_table[0]);
  }
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)
  {
    can_commands_table=nissan_leaf_can_commands_table;
    can_commands_table_elements_count=sizeof(nissan_leaf_can_commands_table)/sizeof(nissan_leaf_can_commands_table[0]);
  }
  else if(can_program==VOLVO_XC60_2018_PROGRAM)
  {
    can_commands_table=volvo_xc60_2018_can_commands_table;
    can_commands_table_elements_count=sizeof(volvo_xc60_2018_can_commands_table)/sizeof(volvo_xc60_2018_can_commands_table[0]);
  }
  else if(can_program==VOLVO_XC40_2018_PROGRAM)
  {
    can_commands_table=volvo_xc40_2018_can_commands_table;
    can_commands_table_elements_count=sizeof(volvo_xc40_2018_can_commands_table)/sizeof(volvo_xc40_2018_can_commands_table[0]);
  }
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    can_commands_table=range_rover_velar_commands_table;
    can_commands_table_elements_count=sizeof(range_rover_velar_commands_table)/sizeof(range_rover_velar_commands_table[0]);
  }
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
  {
    can_commands_table=nissan_qashqai_can_commands_table;
    can_commands_table_elements_count=sizeof(nissan_qashqai_can_commands_table)/sizeof(nissan_qashqai_can_commands_table[0]);
  }
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    can_commands_table=hyundai_creta_can_commands_table;
    can_commands_table_elements_count=sizeof(hyundai_creta_can_commands_table)/sizeof(hyundai_creta_can_commands_table[0]);
  }
  else if(can_program==BMW_X1_2019_PROGRAM)
  {
    can_commands_table=bmw_x1_2019_can_commands_table;
    can_commands_table_elements_count=sizeof(bmw_x1_2019_can_commands_table)/sizeof(bmw_x1_2019_can_commands_table[0]);
  }
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
  {
    can_commands_table=renault_logan_2013_can_commands_table;
    can_commands_table_elements_count=sizeof(renault_logan_2013_can_commands_table)/sizeof(renault_logan_2013_can_commands_table[0]);
  }
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    can_commands_table=jeep_wrangler_2018_can_commands_table;
    can_commands_table_elements_count=sizeof(jeep_wrangler_2018_can_commands_table)/sizeof(jeep_wrangler_2018_can_commands_table[0]);
  }
  else if(can_program==AUDI_A6_2018_PROGRAM)
  {
    can_commands_table=audi_a6_2018_can_commands_table;
    can_commands_table_elements_count=sizeof(audi_a6_2018_can_commands_table)/sizeof(audi_a6_2018_can_commands_table[0]);
  }
  else
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  uint16_t can_cmd_index;
  for(can_cmd_index=0; can_cmd_index<can_commands_table_elements_count; can_cmd_index++)
  {
    if(can_commands_table[can_cmd_index].cmd_type==can_cmd_proc.cmd_q[0]) break;
  }

  if(can_cmd_index>=can_commands_table_elements_count)
  {
    //проверяем, есть ли еще что-то в cmd_q
    move_next_command_to_head();
    can_cmd_proc.step=0;
    return;
  }

  /*
  if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
  {
    //если по статусу видим, что открылась/закрылась, прекращаем слать
    if(can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_CLOSE_CMD_TYPE)
    {
      if(cls_state.driver_door && cls_state.pass_door && cls_state.l_rear_door && cls_state.r_rear_door)
      {
        //последняя, проверяем, есть ли еще что-то в cmd_q
        move_next_command_to_head();
        can_cmd_proc.step=0;
      }
    }
    else if(can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_OPEN_CMD_TYPE)
    {
      if(!cls_state.driver_door && !cls_state.pass_door && !cls_state.l_rear_door && !cls_state.r_rear_door)
      {
        //последняя, проверяем, есть ли еще что-то в cmd_q
        move_next_command_to_head();
        can_cmd_proc.step=0;
      }
    }
  }
  */

  CanTxMsg TxMessage;
  TxMessage.RTR=CAN_RTR_Data;

  if(can_commands_table[can_cmd_index].id_type==STD_ID_TYPE)
  {
    TxMessage.IDE=CAN_ID_STD;
    TxMessage.ExtId=0;
  }
  else
  {
    TxMessage.IDE=CAN_ID_EXT;
    TxMessage.StdId=0;
  }

  if(true)
  {
    static uint8_t seq_repeats_counter;
    static bool    is_seq_inprogress=false;

    if(can_program==AUDI_A6_2018_PROGRAM)
    {
      // Если все окна закрыты, преращаем отправку
      #warning ПРОВЕРИТЬ!!!
      if(can_cmd_proc.cmd_q[0]==WINDOWS_CLOSING_CMD_TYPE &&
         (!System.signal_state.windows_position.driver_door &&
          !System.signal_state.windows_position.pass_door &&
          !System.signal_state.windows_position.r_rear_door &&
          !System.signal_state.windows_position.l_rear_door))
      {
        seq_repeats_counter=0;
        is_seq_inprogress=false;
        move_next_command_to_head();
        can_cmd_proc.step=0;
        return;
      }
    }

    if(!is_seq_inprogress/* && can_cmd_proc.step==0*/) {seq_repeats_counter=can_commands_table[can_cmd_index].seq_repeats; is_seq_inprogress=true;}

    if(can_commands_table[can_cmd_index].repeats_1 && !(can_cmd_proc.step%2) && can_cmd_proc.step<2*can_commands_table[can_cmd_index].repeats_1)
    {
      if(can_commands_table[can_cmd_index].id_type==STD_ID_TYPE) {TxMessage.StdId=can_commands_table[can_cmd_index].id_1;}
      else                                                       {TxMessage.ExtId=can_commands_table[can_cmd_index].id_1;}
      TxMessage.DLC=can_commands_table[can_cmd_index].len_1;
      if(TxMessage.DLC>8) TxMessage.DLC=8;
      memcpy(TxMessage.Data, can_commands_table[can_cmd_index].mess_1, TxMessage.DLC);
      CAN_Transmit_Blocked(can_commands_table[can_cmd_index].CANx, &TxMessage, 25);
      can_cmd_proc.timer=can_commands_table[can_cmd_index].delay_1+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else if((can_commands_table[can_cmd_index].repeats_1 && can_cmd_proc.step%2 && can_cmd_proc.step<2*can_commands_table[can_cmd_index].repeats_1) \
      ||(can_commands_table[can_cmd_index].repeats_2 && can_cmd_proc.step%2 && can_cmd_proc.step<2*(can_commands_table[can_cmd_index].repeats_1+can_commands_table[can_cmd_index].repeats_2)))
    {
      if(timeAfter(xTaskGetTickCount(), can_cmd_proc.timer)) can_cmd_proc.step++;
    }
    else if(can_commands_table[can_cmd_index].repeats_2 && !(can_cmd_proc.step%2) && can_cmd_proc.step<2*(can_commands_table[can_cmd_index].repeats_1+can_commands_table[can_cmd_index].repeats_2))
    {
      if(can_commands_table[can_cmd_index].id_type==STD_ID_TYPE) {TxMessage.StdId=can_commands_table[can_cmd_index].id_2;}
      else                                                       {TxMessage.ExtId=can_commands_table[can_cmd_index].id_2;}
      TxMessage.DLC=can_commands_table[can_cmd_index].len_2;
      if(TxMessage.DLC>8) TxMessage.DLC=8;
      memcpy(TxMessage.Data, can_commands_table[can_cmd_index].mess_2, TxMessage.DLC);
      CAN_Transmit_Blocked(can_commands_table[can_cmd_index].CANx, &TxMessage, 25);
      can_cmd_proc.timer=can_commands_table[can_cmd_index].delay_2+xTaskGetTickCount();
      can_cmd_proc.step++;
    }
    else
    {
      if(seq_repeats_counter) seq_repeats_counter--;
      if(seq_repeats_counter==0)
      {
        is_seq_inprogress=false;
        move_next_command_to_head();
      }
      can_cmd_proc.step=0;
    }
  }
}


static bool is_auto_hood_unlock_present(void)
{
  return true;
}

//запускать из процесса signal_man
void auto_hood_unlock_watchdog(void)
{
  //разблокировка замка капота, если напряжение стало менее 9В в течении 1 минуты
  static const float VOLTAGE_H_TH = 12.0f; //при достежении этого напряжения, а затем падения до VOLTAGE_L_TH замок разблокируется
  static const float VOLTAGE_L_TH = 9.0f;
  static const uint16_t TIMEOUT = 60*1000;

  static uint8_t need_unlock_event = 2;

  if(!is_auto_hood_unlock_present()) {return;}

  if(hood_lock_event)
  {
    //если было закрытие командой
    need_unlock_event = 2;
    hood_lock_event = 0;
  }

  //исходное состояние
  if(need_unlock_event == 2)
  {
    restart_can_timer(HOOD_UNLOCK_TIMER, TIMEOUT);
    need_unlock_event = 1;
  }

  if(System.signal_state.external_voltage > VOLTAGE_H_TH)
  {
    need_unlock_event = 1;
  }

  if(!need_unlock_event || System.signal_state.external_voltage > VOLTAGE_L_TH)
  {
    restart_can_timer(HOOD_UNLOCK_TIMER, TIMEOUT);
  }

  if(need_unlock_event && is_can_timer_expired(HOOD_UNLOCK_TIMER))
  {
    if(CAN_SUBTASK_COMMAND_IS_BUSY != signalman_can_subtask_commnad_handle(YADRIVE_HOOD_UNLOCK_SUBCODE, 0xFF, 0, NULL, NULL))
    {
      need_unlock_event = 0;

      LOG("hood auto unlock by low voltage\n");
    }
  }
}

void check_fuel_level_watchdog()
{
  // timeout 2 mins for check can source fuel
  static uint32_t timeout = 2 * 60 * 1000;

  bool timeout_expired = (xTaskGetTickCount() - System.get_last_time_set_fuel_level_p()) >= timeout;

  if (!System.signal_state.ignition && timeout_expired) {
    // set invalid fuel level
    System.set_fuel_level_p(INVALID_FUEL_LEVEL_PERCENT);
  }
}

//запускать из процесса signal_man
void engine_stop_watchdog(void)
{
  static bool is_doors_locked_by_speed=false;
  static bool is_enable_doors_unlock_by_engine_stop=false;

  if(is_engine_started_by_terminal)
  {
    uint8_t stop_event;
    uint32_t uid=is_engine_started_by_terminal_uid;
    const uint8_t* nonce=is_engine_started_by_terminal_nonce;
    uint8_t server_notify_id=is_engine_started_by_terminal_server_notify_id;

    if(System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door \
       || (!is_ignore_hood() && System.can_state.sec_flags.hood) || System.can_state.sec_flags.trunk) {stop_event=1;}
    else if(if_forced_stop_warming)                                                                   {stop_event=2;}
    else if(!System.can_state.sec_flags.parking)                                                      {stop_event=3;}
    else if((warming_timer!=0 && timeAfter(xTaskGetTickCount(), warming_timer)))                      {stop_event=4;}
    else if(!System.can_state.sec_flags.engine_is_on)                                                 {stop_event=5;}
    else                                                                                              {stop_event=0;}

    if(stop_event)
    {
      //шлем команду пока она не поступит на обработку
      uint8_t command_subcode;
      if(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==BMW_520I_2018_PROGRAM || can_program==BMW_X1_2019_PROGRAM) command_subcode=CAN_ENGINE_STOPING_W_CLS_CLOSE_SUBCODE;
      else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)                        command_subcode=CAN_ENGINE_STOPING_W_WIRELESS_BLOCK_ON_SUBCODE; //при прогреве радио реле было деактивировано, включаем его обратно
      else                                                                                                                    command_subcode=CAN_ENGINE_STOPING_SUBCODE;
      if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(command_subcode, server_notify_id, uid, nonce, NULL))
      {
        const char* res_str=NULL;

        if(stop_event==1)      res_str="engine stop event by door/hood/trunk";
        else if(stop_event==2) res_str="engine stop event by stop warming cmd";
        else if(stop_event==3) res_str="engine stop event by transmission";
        else if(stop_event==4) res_str="engine stop event by timer";
        else if(stop_event==5) res_str="engine stop event by itself";

        last_warming_timeS=GetSecondsFromStartup();

        //!!!IS_LAST_NOTIFY_STATE тут не коректно, т.к. будет производится еще остановка двигателя
        uint8_t add_res=add_server_notify(server_notify_id, uid, nonce, IN_PROCESS_NOTIFY_STATE, res_str);
        notify_logging(server_notify_id, uid, IN_PROCESS_NOTIFY_STATE, res_str, add_res);

        set_head_control_state(false, 0);

        //эта обработка перенесена в CAN_ENGINE_STOPING_SUBCODE, в STOP_ENGINE_CMD_TYPE
        /*
        //если был прогрев по таймеру
        if(warming_timer!=0)
        {
          is_wired_block_relay_enable=true;//is_wired_block_relay_state_before_warming;
          LOG("hood block relay state: %s\n", (is_wired_block_relay_enable)?("on"):("off"));
        }

        warming_timer=0;
        if_forced_stop_warming=false;
        */
      }
    }
  }

  //открытие и закрытие дверей оставляем только на бомжах
  if(is_auto_doors_present())
  {
    if(!is_doors_locked_by_speed)
    {
      System.Grab(portMAX_DELAY);
      if(System.can_state.sec_flags.engine_is_on \
        && !System.can_state.sec_flags.driver_door && !System.can_state.sec_flags.pass_door && !System.can_state.sec_flags.l_rear_door && !System.can_state.sec_flags.r_rear_door \
          && !System.can_state.sec_flags.parking && System.can_state.numeric_data.can_speed>10)
      {
        System.Release();

        if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(CAN_ALL_DOOR_CLOSING_V2_SUBCODE, 0xff, 0, NULL, NULL))
        {
          is_doors_locked_by_speed=true;
          is_enable_doors_unlock_by_engine_stop=true;
          LOG("sent doors lock cmd by speed (%u km/h)\n", System.can_state.numeric_data.can_speed);
        }

      }
      else
      {
        System.Release();
      }
    }
    else
    {
      System.Grab(portMAX_DELAY);
      if(System.can_state.numeric_data.can_speed==0 && (System.can_state.sec_flags.driver_door || System.can_state.sec_flags.pass_door || System.can_state.sec_flags.l_rear_door || System.can_state.sec_flags.r_rear_door))
      {
        is_doors_locked_by_speed=false;
        is_enable_doors_unlock_by_engine_stop=false;
      }
      System.Release();

      if(!is_doors_locked_by_speed)
      {
        LOG("wait over speed for doors locking...\n");
      }
    }

    if(is_enable_doors_unlock_by_engine_stop && !System.can_state.sec_flags.engine_is_on)
    {
      if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(CAN_ALL_DOOR_OPENING_V2_SUBCODE, 0xff, 0, NULL, NULL))
      {
        is_enable_doors_unlock_by_engine_stop=false;
        LOG("sent doors unlock cmd by engine stop\n");
      }
    }
  }

  if(System.signal_state.drift_sensor && !can_timers[DRIFT_EVENT_TIMER_ID])
  {
    LOG("drift detect...\n");
    can_timers[DRIFT_EVENT_TIMER_ID]=10000;
    System.signal_state.drift_sensor=0;
  }

  if(System.signal_state.launch_sensor && !can_timers[LAUNCH_EVENT_TIMER_ID])
  {
    LOG("launch detect...\n");
    can_timers[LAUNCH_EVENT_TIMER_ID]=10000;
    System.signal_state.launch_sensor=0;
  }


  static uint8_t last_unauthorized_start_sensor=System.signal_state.unauthorized_start_sensor;
  uint8_t unauthorized_start_sensor=System.signal_state.unauthorized_start_sensor;

  if(unauthorized_start_sensor!=last_unauthorized_start_sensor)
  {
    if(unauthorized_start_sensor) {LOG("unauthorized engine start detect...\n");}

    if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM)
    {
      if(unauthorized_start_sensor)
      {
#if defined(JQ6500_AUDUO_PRESENT)
        yandex_audio_cmd_callback("\x00\x02\x00", 3, NULL, 0xFF, 0, 0, NULL);
#endif //defined(JQ6500_AUDUO_PRESENT)
      }
    }

    last_unauthorized_start_sensor=unauthorized_start_sensor;
  }

  head_control_handler();
}

//запускать из процесса signal_man
void async_analog_windows_closing_watchdog(bool is_force_stop_and_deinit)
{
  uint8_t output_id;

  if(!is_analog_windows_closing_no_blocking())
  {
    return;
  }

  if(!is_windows_output_present(&output_id))
  {
    return;
  }

  if(is_force_stop_and_deinit)
  {
    can_timers[ANALOG_WINDOWS_CLOSING_TIMER_ID]=0;
    can_timers[ANALOG_WINDOWS_CLOSING_TIMER_ID]=0;
    System.signal_state.digital_output[output_id]=0;
    outputs_state_force_apply();

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2) || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3
    if(can_program==NISSAN_QASHQAI_2018_PROGRAM && get_lin_bus_speed())
    {
      while(is_lin_uart_write_busy()) {vTaskDelay(1);}
      lin_uart_deinit();
    }
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

    return;
  }

  uint8_t curr_state=System.signal_state.digital_output[output_id];

  if(can_timers[ANALOG_WINDOWS_CLOSING_TIMER_ID]) System.signal_state.digital_output[output_id]=1;
  else                                            System.signal_state.digital_output[output_id]=0;

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
  if(can_program==NISSAN_QASHQAI_2018_PROGRAM && get_lin_bus_speed())
  {
    static uint32_t tx_timeout=xTaskGetTickCount();

    if(System.signal_state.digital_output[output_id] && !is_lin_uart_write_busy())
    {
      if(timeAfter(xTaskGetTickCount(), tx_timeout))
      {
        static const uint8_t qashqai_windows_close[]={0x55/*sync*/,0x6F/*2F PID*/,0x8F,0xB6,0x08,0x00,0xB1/*classic crc*/};

        lin_uart_send_break();
        lin_uart_write_2(qashqai_windows_close, sizeof(qashqai_windows_close));

        tx_timeout=xTaskGetTickCount()+400;
      }
    }
    else
    {
      tx_timeout=xTaskGetTickCount();
    }
  }
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

  if(curr_state!=System.signal_state.digital_output[output_id]) outputs_state_force_apply();
}


static bool delayed_windows_closing_present(void)
{
  if(can_program==SKODA_OCTAVIA_2018_PROGRAM) return true;
  else                                        return false;
}

//запускать из процесса signal_man
void delayed_windows_closing_handler(void)
{
  if(delayed_windows_closing_present())
  {
    static uint8_t is_enable_delayed_windows_closing=0;
    static lease_state_t lease_state_prev=UNKNOWN_LEASE_STATE;

    lease_state_t _lease_state=lease_state;

    if(_lease_state!=lease_state_prev)
    {
      if(lease_state_prev!=UNKNOWN_LEASE_STATE)
      {
        if(_lease_state==IN_END_LEASE_STATE || _lease_state==IN_FORCED_END_LEASE_STATE)
        {
          is_enable_delayed_windows_closing=1;
          restart_can_timer(DELAYED_WINDOWS_CLOSING_TIMER, 60000);
        }
        else if(_lease_state==IN_LEASE_STATE)
        {
          is_enable_delayed_windows_closing=0;
        }
      }

      lease_state_prev=_lease_state;
    }

    if(is_enable_delayed_windows_closing && is_can_timer_expired(DELAYED_WINDOWS_CLOSING_TIMER))
    {
      bool is_next_step=false;

      if(!System.signal_state.windows_position.driver_door && !System.signal_state.windows_position.pass_door && !System.signal_state.windows_position.l_rear_door && !System.signal_state.windows_position.r_rear_door)
      {
        LOG("windows check %u ok...\n", is_enable_delayed_windows_closing);
        is_next_step=true;
      }
      else if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(CAN_WINDOWS_CLOSING_1S_SUBCODE, 0xFF, 0, NULL, NULL))
      {
        LOG("delayed windows closing %u...\n", is_enable_delayed_windows_closing);
        is_next_step=true;
      }

      if(is_next_step)
      {
        is_enable_delayed_windows_closing++;

        if(is_enable_delayed_windows_closing>=3)
        {
          is_enable_delayed_windows_closing=0;
        }
        else
        {
          restart_can_timer(DELAYED_WINDOWS_CLOSING_TIMER, 60000);
        }
      }
    }
  }
}

//запускать из процесса signal_man
void async_key_power_off_watchdog(bool is_force_off_and_deinit)
{
  uint8_t output_id;

  if(!is_key_power_off_no_blocking())
  {
    return;
  }

  if(!is_key_power_present(&output_id))
  {
    return;
  }

  if(is_force_off_and_deinit)
  {
    can_timers[KEY_POWER_OFF_TIMER_ID]=0;
    can_timers[KEY_POWER_OFF_TIMER_ID]=0;
    System.signal_state.digital_output[output_id]=0;
    outputs_state_force_apply();
    return;
  }
}

//запускать из процесса signal_man
void startstop_off_watchdog(bool is_force_stop_and_deinit)
{
  //отключение функции старт-стоп
  if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM)
  {
    static struct
    {
      uint32_t timerMS;
      uint8_t activate_output_step:3;
      uint8_t is_wait_engine_on_timer:1;
      uint8_t is_engine_on:1;
      uint8_t is_ignore_engine_on:1;
      uint8_t is_engine_on_previous:1;
    }start_stop =
    {
      .is_wait_engine_on_timer=0,
      .activate_output_step=0,
      .is_ignore_engine_on=0,
      .is_engine_on_previous=0,
    };

    if(is_force_stop_and_deinit)
    {
      memset(&start_stop, 0, sizeof(start_stop));
      return;
    }

    if(System.can_state.sec_flags.engine_is_on) start_stop.is_engine_on=1;
    else                                        start_stop.is_engine_on=0;
    if(is_engine_started_by_terminal)           start_stop.is_ignore_engine_on=1;
    else                                        start_stop.is_ignore_engine_on=0;

    if(start_stop.activate_output_step==0)
    {
      if(start_stop.is_engine_on!=start_stop.is_engine_on_previous)
      {
        start_stop.is_engine_on_previous=start_stop.is_engine_on;
        if(start_stop.is_engine_on)
        {
          start_stop.is_wait_engine_on_timer=1;
          start_stop.timerMS=xTaskGetTickCount();
        }
      }

      if(!start_stop.is_ignore_engine_on && start_stop.is_engine_on && start_stop.is_wait_engine_on_timer && timeAfter(xTaskGetTickCount(), start_stop.timerMS+5000))
      {
        start_stop.activate_output_step=1;
        start_stop.is_wait_engine_on_timer=0;
      }
    }
    else
    {
      if(!start_stop.is_engine_on) start_stop.activate_output_step=3;

      if(start_stop.activate_output_step==1)
      {
        LOG("deactivation start-stop system...\n");

        System.signal_state.digital_output[AUDI_START_STOP_OFF_OUTPUT_ID]=1;
        outputs_state_force_apply();
        start_stop.timerMS=250+xTaskGetTickCount();
        start_stop.activate_output_step=2;
      }
      else if(start_stop.activate_output_step==2)
      {
        if(timeAfter(xTaskGetTickCount(), start_stop.timerMS)) start_stop.activate_output_step=3;
      }
      else
      {
        System.signal_state.digital_output[AUDI_START_STOP_OFF_OUTPUT_ID]=0;
        outputs_state_force_apply();
        start_stop.activate_output_step=0;
      }
    }
  }

  //включение TRC, если его отключили
  if(can_program==JEEP_WRANGLER_2018_PROGRAM)
  {
    static uint32_t timerS = GetSecondsFromStartup();
    static uint8_t engine_is_on_prev = 0;

    const uint8_t engine_is_on = System.can_state.sec_flags.engine_is_on;

    if(engine_is_on_prev != engine_is_on)
    {
      if(engine_is_on)
      {
        timerS = GetSecondsFromStartup() + 10;
      }

      engine_is_on_prev = engine_is_on;
    }

    if(engine_is_on && System.can_state.cont_flags.esp)
    {
      if(timeAfter(GetSecondsFromStartup(), timerS))
      {
        if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(CAN_PRESS_TRC_BUTTON_SUBCODE, 0xff, 0, NULL, NULL))
        {
          LOG("forced activation of TRC....\n");
        }

        timerS = GetSecondsFromStartup() + 10;
      }
    }
  }
}

void obd_cmd_handler(const uint8_t cmd)
{
  static NTime::TTimer timer;
  const NTime::TTime timeout = NTime::TTime::Seconds(1);

  for(;;)
  {
    if(can_cmd_proc.step==0)
    {
      timer.Start();
      if(cmd == 1 || cmd == 2 || cmd == 3) {
        obd_ctx.is_force_cmd = cmd;
      } else {
        LOG("obd_cmd_habdler: error command\n");
        // TODO: handle error
        break;
      }
      can_cmd_proc.step++;
    }
    else if(can_cmd_proc.step==1)
    {
      if(timer.HasExpired(timeout)) {
        can_cmd_proc.step++;
      }
    }
    else
    {
      //done
      break;
    }

    obd_tx_handler(); // need for correct OBD work 
    return; // continue command procces
  }
  //end command procces

  timer.Stop();
  move_next_command_to_head();
  can_cmd_proc.step=0;
}

uint8_t signalman_can_subtask_commnad_handle(const uint8_t cmd, const uint8_t server_notify_id, const uint32_t uid, const uint8_t nonce[8], const void* ext_args)
{
  if(cmd==YADRIVE_STOP_WARMING_SUBCODE)
  {
    const char* res_str=NULL;
    notify_state_t notify_state=IN_PROCESS_NOTIFY_STATE;

    if(!is_engine_start_stop_present())
    {
      res_str="not available";
      notify_state=IS_LAST_NOTIFY_STATE;
    }
    else if(if_forced_stop_warming)
    {
      res_str="stop warming already in progress";
    }
    else if(warming_timer && is_engine_started_by_terminal)
    {
      if_forced_stop_warming=true;
      is_engine_started_by_terminal_uid=uid;//меняем uuid
      memcpy(is_engine_started_by_terminal_nonce, nonce, sizeof(is_engine_started_by_terminal_nonce));
      res_str="stop warming in progress";
    }
    else
    {
      res_str="no active warming";
      notify_state=IS_LAST_NOTIFY_STATE;
    }

    uint8_t add_res=add_server_notify(server_notify_id, uid, nonce, notify_state, res_str);
    notify_logging(server_notify_id, uid, notify_state, res_str, add_res);

    return CAN_SUBTASK_COMMAND_START_PROCCES;
  }
  else if(is_can_cmd_proc_q_emty() && can_cmd_proc.step==0)
  {
    //can_cmd_proc.cmd_q[0] заполняем последеним, т.к. signalman_can_subtask проверяет наличие команд по can_cmd_proc.cmd_q[0]!=EMTY_CAN_CMD_TYPE
    uint8_t i=1;
    can_cmd_proc.server_notify_id=server_notify_id;
    can_cmd_proc.uid=uid;

    if(nonce!=NULL) {memcpy(can_cmd_proc.nonce, nonce, sizeof(can_cmd_proc.nonce));}
    else            {memset(can_cmd_proc.nonce, 0, sizeof(can_cmd_proc.nonce));}

    if(cmd==CAN_BLINKER_FLASHING_SUBCODE)
    {
      if(can_program==CITROEN_JUMPY_2018_PROGRAM)
      {
        bool block_relay_state=is_wired_block_relay_enable;
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1800;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else
      {
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(can_program==AUDI_Q3_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        if(can_program==AUDI_Q3_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    else if(cmd==CAN_BLINKER_FLASHING_V2_SUBCODE)
    {
      if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=LOGAN_2013_PRECAN_CMD_TYPE;
      }
      else
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=PARKING_LIGHT_ON_CMD_TYPE;
      }
    }
    //>---------temporary-debug--------->
    else if(can_program==NISSAN_LEAF_2019_PROGRAM && cmd==CAN_WINDOWS_OPENING_15S_SUBCODE)//temporary debug
    {
      can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=CHARGE_UNLOCK_CMD_TYPE;
    }
    else if(can_program==NISSAN_LEAF_2019_PROGRAM && cmd==CAN_WINDOWS_OPENING_19S_SUBCODE)//temporary debug
    {
      can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=CHARGE_LOCK_CMD_TYPE;
    }
    else if(can_program==NISSAN_LEAF_2019_PROGRAM && cmd==CAN_WINDOWS_OPENING_23S_SUBCODE)//temporary debug
    {
      can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=ECO_ON_CMD_TYPE;
    }
    else if(can_program==NISSAN_LEAF_2019_PROGRAM && cmd==CAN_WINDOWS_OPENING_29S_SUBCODE)//temporary debug
    {
      can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=ECO_OFF_CMD_TYPE;
    }
    //<---------temporary-debug----------<
    else if(cmd==CAN_ALL_DOOR_CLOSING_SUBCODE)
    {
      if(can_program==CITROEN_JUMPY_2018_PROGRAM)
      {
        bool block_relay_state=is_wired_block_relay_enable;
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1800;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

        if(System.signal_state.roof.sunroof)
        {
          can_cmd_proc.cmd_q[i++]=ROOF_CLOSING_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=8000;
        }

        can_cmd_proc.cmd_q[i++]=CHECK_ROOF_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==SKODA_OCTAVIA_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==VOLVO_XC40_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
        /*
        can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=5000;
        can_cmd_proc.cmd_q[i++]=VOLVO_XC40_CAN_CENTRAL_LOCK_CLOSE_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=800;
        */
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_1_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_2_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==PORSCHE_MACAN_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

        can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;

        can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
        can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else
      {
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
        }
        else
        {
          if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        }
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        if(can_program==BMW_520I_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        }
        if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
        }
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE;
        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    else if(cmd==CAN_ALL_DOOR_CLOSING_V2_SUBCODE)
    {
      if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
      {
        //can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
      }
      else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=LOGAN_2013_PRECAN_CMD_TYPE;
      }
      else
      {
        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
      }
    }
    else if(cmd==CAN_ALL_DOOR_OPENING_V2_SUBCODE)
    {
      if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=LOGAN_2013_PRECAN_CMD_TYPE;
      }
      else
      {
        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
      }
    }
    else if(can_program==VOLVO_XC40_2018_PROGRAM && cmd==CAN_DRIVER_DOOR_OPENING_SUBCODE)
    {
      //только для VOLVO_XC40_2018_PROGRAM
      can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
      //у XC60 открывается эмуляцией двери водителя, но если в меню автомобиля "закрывание" выставлена настройка "закрывать только водительскую дверь", двери не отпираются
      //альтернативная команда
      can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=600;
      //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

      can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

      can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
    }
    else if(cmd==CAN_ALL_DOOR_OPENING_SUBCODE)
    {
      is_ignore_end_lease_state=true;
      if(can_program==CITROEN_JUMPY_2018_PROGRAM)
      {
        bool block_relay_state=is_wired_block_relay_enable;
         //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1800;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==SKODA_OCTAVIA_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==VOLVO_XC40_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
        /*
        can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_OPENED_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=VOLVO_XC40_CAN_CENTRAL_LOCK_OPEN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=500;
        */
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==PORSCHE_MACAN_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;//
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

        can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_OPENED_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        bool block_relay_state=is_wired_block_relay_enable;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
        //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//проверяется в CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
        can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else
      {
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;

        if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
        {
          //это костыль, нужно убирать
          uint8_t key_power_output_id;

          if(is_key_power_present(&key_power_output_id))
          {
            if(!System.signal_state.digital_output[key_power_output_id])
            {
              can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
            }
          }
        }

        if(can_program==MB_E200_2018_PROGRAM)
        {
          if(System.signal_state.key_position=='O' || System.signal_state.key_position=='U')
          {
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1500;
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1500;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1500;
          }
        }

        if(can_program==SKODA_RAPID_2018_PROGRAM)
        {
          static uint8_t only_once = 1; 
          if (only_once) {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1000;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            only_once = 0;
          }
        }

        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        uint8_t ford_ign_pulse_en = 0;
        if(can_program==FORD_TRANSIT_2017_PROGRAM && !(System.can_state.sec_flags.ignition || System.signal_state.ignition)) {ford_ign_pulse_en=1;}
        if(ford_ign_pulse_en)
        {
          can_cmd_proc.cmd_q[i++]=FORD_TRANSIT_IGN_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;
        }
        if(!(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM)) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        if(ford_ign_pulse_en)
        {
          can_cmd_proc.cmd_q[i++]=FORD_TRANSIT_IGN_OFF_CMD_TYPE;
        }
        if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        if(can_program==BMW_520I_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
        }
        if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//у XC60 открывается эмуляцией двери водителя
        if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
        }
        can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE;
        if(can_program==HYUNDAI_GENESIS_2017_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//чтобы небыло перепостановки на охрану
        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    else if(cmd==CAN_TRUNK_DOOR_OPENING_SUBCODE)
    {
      //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=TRUNK_OPEN_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
      if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
    }
    else if(cmd==CAN_ENGINE_STARTING_SUBCODE)
    {
      if(!is_engine_start_stop_present())
      {
        can_cmd_proc.cmd_q[0]=NOT_AVAILABLE_NOTTIFY_CMD_TYPE;
      }
      else
      {
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        //can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//По просьбе Шевцова, делаем команды атомарными
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=START_ENGINE_CMD_TYPE; can_cmd_proc.ext_param=0;
        //can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE; //По просьбе Шевцова, делаем команды атомарными
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    else if(cmd==CAN_ENGINE_STOPING_SUBCODE || cmd==CAN_ENGINE_STOPING_W_CLS_CLOSE_SUBCODE || cmd==CAN_ENGINE_STOPING_W_WIRELESS_BLOCK_ON_SUBCODE)
    {
      if(!is_engine_start_stop_present())
      {
        can_cmd_proc.cmd_q[0]=NOT_AVAILABLE_NOTTIFY_CMD_TYPE;
      }
      else
      {
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        //can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//По просьбе Шевцова, делаем команды атомарными
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;

        if(cmd==CAN_ENGINE_STOPING_W_CLS_CLOSE_SUBCODE)
        {
          if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          }
          else if(can_program==BMW_520I_2018_PROGRAM)
          {
            //у BMW_520I_2018_PROGRAM эмуляция только здесь
            can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//для отключения фар
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
            can_cmd_proc.cmd_q[i++]=CHECK_BMW_CLS_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1000;//если вслед пришла команда завершения аренды, то статус концевика не успеет обновиться после эмуляции открытия двери
          }
          else if(can_program==BMW_X1_2019_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=2000;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1000;
          }
          else
          {
            if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
            if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
            if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
          }
        }

        if(cmd==CAN_ENGINE_STOPING_W_WIRELESS_BLOCK_ON_SUBCODE)
        {
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;
        }

        //can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;//про просьбе Алеся не закрываем окна
        if(can_program==KIA_RIO_2017_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//для отключения фар

        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//для отключения габаритов

        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[0]=KIA_BCM_BEGIN_CONFIG_1_CMD_TYPE; //дергается зажигание итд, шлем неиспользуемую команду
        else                                 can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    else if(cmd==CAN_EMULATION_DRIVER_DOOR_OPENING_SUBCODE)
    {
      can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=DRIVER_DOOR_EMULATION_CMD_TYPE;
    }
    else if(cmd==CAN_HORN_SIGNALING_SUBCODE)
    {
      if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM
         || can_program==PORSCHE_MACAN_2018_PROGRAM)
      {
        //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=IGN_ON_CMD_TYPE;
      }
      else if(can_program==CITROEN_JUMPY_2018_PROGRAM)
      {
        bool block_relay_state=is_wired_block_relay_enable;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1800;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.delay[1]=0;
        can_cmd_proc.cmd_q[0]=DELAY_2;
      }
      else if(can_program==BMW_520I_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=HORN_CMD_TYPE;
      }
    }
    else if(cmd==CAN_HORN_W_BLINKER_SIGNALIG_SUBCODE)
    {
      if(can_program==CITROEN_JUMPY_2018_PROGRAM)
      {
        bool block_relay_state=is_wired_block_relay_enable;
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1800;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==BMW_520I_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else
      {
        //can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==PORSCHE_MACAN_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        if(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==PORSCHE_MACAN_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    else if(cmd==CAN_HORN_W_BLINKER_FLASHING_IN_ALARM_SUBCODE)
    {
      if(can_program==PORSCHE_MACAN_2018_PROGRAM ||
         can_program==PORSCHE_PANAMERA_2019_PROGRAM ||
         can_program==BMW_520I_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        //if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==BMW_X1_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        //if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==SKODA_OCTAVIA_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        //can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        //can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        //if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        //if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==AUDI_Q3_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=IGN_ON_CMD_TYPE;
        //can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        //can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        //if(server_notify_id!=0xFF) can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        //can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==AUDI_A3_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=HORN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=IGN_ON_CMD_TYPE;
      }
      else if(can_program==FORD_TRANSIT_2017_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=HORN_CMD_TYPE;
      }
    }
    else if(can_program==JEEP_WRANGLER_2018_PROGRAM && cmd==CAN_PRESS_TRC_BUTTON_SUBCODE)
    {
      can_cmd_proc.cmd_q[0]=PRESS_TRC_BUTTON_CMD_TYPE;
    }
    else if(cmd==CAN_WINDOWS_CLOSING_1S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_3S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_5S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_7S_SUBCODE \
      || cmd==CAN_WINDOWS_CLOSING_9S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_11S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_13S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_15S_SUBCODE \
        || cmd==CAN_WINDOWS_CLOSING_17S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_19S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_21S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_23S_SUBCODE \
          || cmd==CAN_WINDOWS_CLOSING_25S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_27S_SUBCODE || cmd==CAN_WINDOWS_CLOSING_29S_SUBCODE)
    {
      if(can_program==AUDI_Q3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM \
        || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
          || can_program==CITROEN_JUMPY_2018_PROGRAM)
      {
        //Для Audi Q3 во время подачи команды закрытия окон нужно включить зажигание (для Audi A3 не нужно)
        //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=IGN_ON_CMD_TYPE;
      }
      else if(can_program==AUDI_A6_2018_PROGRAM)
      {
        //Не трогаем зажигание, если оно уже включено. Для других авто нужно так-же?
        uint8_t ign = System.can_state.sec_flags.ignition;

        if(!ign) can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
        if(!ign) can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.delay[0]=1;
        can_cmd_proc.cmd_q[0]=DELAY_1;
      }
      else if(can_program==VW_TRANSPORTER_2018_PROGRAM)
      {
        //хз, не проверяли
        bool block_relay_state=is_wired_block_relay_enable;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=999;
        can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1500;
        can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;//не используется
      }
      else if(can_program==BMW_520I_2018_PROGRAM || can_program==BMW_X1_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=WINDOWS_CLOSING_CMD_TYPE;
      }
      else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
      {
        //нужно питание ключа
        can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=EXT_WINDOWS_CLOSING_CMD_TYPE_GENESIS;
        if((System.signal_state.key_position=='O' || System.signal_state.key_position=='U')) can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=IGN_ON_CMD_TYPE;
      }
      else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=100;
        if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
        {
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING1_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING1_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING2_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING2_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING3_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING3_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING4_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING4_2_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
        }

        if(System.signal_state.roof.sunroof)
        {
          can_cmd_proc.cmd_q[i++]=ROOF_CLOSING_CMD_TYPE;
        }

        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==VOLVO_XC40_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=100;

        if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
        {
          //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
          //can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
        }

        can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_1_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_2_CMD_TYPE;

        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==MB_E200_2018_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=100;

        if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
        {
          //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
          //can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
        }

        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=100;

        can_cmd_proc.cmd_q[i++]=TOYOTA_RAV4_WINDOWS_CLOSING_CMD_TYPE;//все обрабатывается в отдельном хэндлере

        /*
        if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
        {
          //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
          for(uint8_t p=0; p<4; p++)
          {
            if(System.signal_state.windows_position.driver_door) can_cmd_proc.cmd_q[i++]=DRIVER_WINDOW_CLOSING_CMD_TYPE;
            if(System.signal_state.windows_position.pass_door) can_cmd_proc.cmd_q[i++]=PASS_WINDOW_CLOSING_CMD_TYPE;
            if(System.signal_state.windows_position.l_rear_door) can_cmd_proc.cmd_q[i++]=REAR_L_WINDOW_CLOSING_CMD_TYPE;
            if(System.signal_state.windows_position.r_rear_door) can_cmd_proc.cmd_q[i++]=REAR_R_WINDOW_CLOSING_CMD_TYPE;
          }
          //can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
        }
        */

        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;

        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
      else
      {
        can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    else if(cmd==CAN_WINDOWS_OPENING_1S_SUBCODE || cmd==CAN_WINDOWS_OPENING_3S_SUBCODE || cmd==CAN_WINDOWS_OPENING_5S_SUBCODE || cmd==CAN_WINDOWS_OPENING_7S_SUBCODE || \
      cmd==CAN_WINDOWS_OPENING_9S_SUBCODE || cmd==CAN_WINDOWS_OPENING_11S_SUBCODE || cmd==CAN_WINDOWS_OPENING_13S_SUBCODE || cmd==CAN_WINDOWS_OPENING_15S_SUBCODE || \
        cmd==CAN_WINDOWS_OPENING_17S_SUBCODE || cmd==CAN_WINDOWS_OPENING_19S_SUBCODE || cmd==CAN_WINDOWS_OPENING_21S_SUBCODE || cmd==CAN_WINDOWS_OPENING_23S_SUBCODE || \
          cmd==CAN_WINDOWS_OPENING_25S_SUBCODE || cmd==CAN_WINDOWS_OPENING_27S_SUBCODE || cmd==CAN_WINDOWS_OPENING_29S_SUBCODE)
    {
      if(can_program==BMW_520I_2018_PROGRAM || can_program==BMW_X1_2019_PROGRAM)
      {
        can_cmd_proc.cmd_q[i++]=WINDOWS_OPENING_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
      }
    }
    //else if(cmd==) {can_cmd_proc.cmd_q[0]=HOOD_LOCK_CMD_TYPE;}
    //else if(cmd==) {can_cmd_proc.cmd_q[0]=HOOD_UNLOCK_CMD_TYPE;}
    else if(cmd==YADRIVE_WARMING_SUBCODE)
    {
      if(!is_engine_start_stop_present())
      {
        can_cmd_proc.cmd_q[0]=NOT_AVAILABLE_NOTTIFY_CMD_TYPE;
      }
      else
      {
        uint8_t* warming_time=(uint8_t*)ext_args;

        LOG("warming time: %u minutes\n", warming_time[0]);

        if(warming_time[0]==0 || warming_time[0]>MAX_WARMING_DURATION_MINUTES) return CAN_SUBTASK_COMMAND_UNKNOWN_ARG;

        //can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        if(can_program==CITROEN_JUMPY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM) {can_cmd_proc.cmd_q[i++]=CHECK_PARKING_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE;}
        else                                                                                    {can_cmd_proc.cmd_q[i++]=CHECK_PARKING_STATE_CMD_TYPE;}
        can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
        if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;
        /*
        //запоминаем состояние подкапотной блокировки, до запуска разблокировки и запуска двигателя
        //если команда пришла во время работающего прогрева, состояние блокировки уже запомненно
        if(!(warming_timer && is_engine_started_by_terminal)) is_wired_block_relay_state_before_warming=is_wired_block_relay_enable;
        */
        can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
        if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
        if(can_program==TOYOTA_RAV4_2019_PROGRAM)
        {
          //задержка, чтобы машина проснулась. предположительно из-за этого прогрев не срабатывал после стоянки
          //can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=2000;
        }
        if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
        }
        else
        {
          if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
          if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
        }
        if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
        can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

        if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)
        {
          //Velar так тарахтит при прогреве, что срабатывает радио реле блокировка
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
        }

        can_cmd_proc.cmd_q[i++]=START_ENGINE_CMD_TYPE; can_cmd_proc.ext_param=warming_time[0]*60*1000;
        if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
        }
        else
        {
          if(!(can_program==NISSAN_QASHQAI_2018_PROGRAM)) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
        }
        can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
      }
    }
    else if(cmd==YADRIVE_START_OF_LEASE_SUBCODE)
    {
      restart_can_timer(DRIVER_DOOR_IGNORE_TIMER_ID, 10000); //чтобы небыло сработки, если дернут дверь сразу после начала аренды

      if(is_taxi())
      {
        can_cmd_proc.cmd_q[0]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
      }
      else
      {
        if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
        {
          lease_state=IN_LEASE_STATE;

          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;

          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;

          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==NISSAN_LEAF_2019_PROGRAM)
        {
          lease_state=IN_LEASE_STATE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL))      can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
        {
          lease_state=IN_LEASE_STATE;
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//чтобы небыло перепостановки на охрану
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==SKODA_OCTAVIA_2018_PROGRAM)
        {
          lease_state=IN_LEASE_STATE;
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==VOLVO_XC40_2018_PROGRAM)
        {
          lease_state=IN_LEASE_STATE;
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
          /*
          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_OPENED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=VOLVO_XC40_CAN_CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=500;
          */
          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==PORSCHE_MACAN_2018_PROGRAM)
        {
          lease_state=IN_LEASE_STATE;
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз

          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_OPENED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
        {
          lease_state=IN_LEASE_STATE;
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          //tmp ->
          can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;
          //<-tmp
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          //tmp ->
          can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
          //<-tmp
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else
        {
          lease_state=IN_LEASE_STATE;
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) can_cmd_proc.cmd_q[i++]=FUEL_CAP_AND_TRUNK_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_UNBLOCK_RELAY_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          if(can_program==CITROEN_JUMPY_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1800;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
          }

          if(can_program==MB_E200_2018_PROGRAM)
          {
            if(System.signal_state.key_position=='O' || System.signal_state.key_position=='U')
            {
              can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1500;
              can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1500;
              can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1500;
            }
          }

          if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
          uint8_t ford_ign_pulse_en = 0;
          if(can_program==FORD_TRANSIT_2017_PROGRAM && !(System.can_state.sec_flags.ignition || System.signal_state.ignition)) {ford_ign_pulse_en=1;}
          if(ford_ign_pulse_en)
          {
            can_cmd_proc.cmd_q[i++]=FORD_TRANSIT_IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;
          }
          if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
          if(!(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM)) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_OPEN_CMD_TYPE;
          if(ford_ign_pulse_en)
          {
            can_cmd_proc.cmd_q[i++]=FORD_TRANSIT_IGN_OFF_CMD_TYPE;
          }
          if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          if(can_program==AUDI_Q3_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
          if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          if(can_program==AUDI_Q3_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
          if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//у XC60 открывается эмуляцией двери водителя
          if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
          }
          can_cmd_proc.cmd_q[i++]=START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          if(can_program==HYUNDAI_GENESIS_2017_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//чтобы небыло перепостановки на охрану
          if(can_program==BMW_X1_2019_PROGRAM) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
      }
    }
    else if(cmd==YADRIVE_END_OF_LEASE_SUBCODE)
    {
      if(is_taxi())
      {
        can_cmd_proc.cmd_q[0]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
      }
      else
      {
        if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING1_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING1_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING2_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING2_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING3_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING3_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING4_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING4_2_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;
          }

          if(System.signal_state.roof.sunroof)
          {
            can_cmd_proc.cmd_q[i++]=ROOF_CLOSING_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=8000;
          }

          can_cmd_proc.cmd_q[i++]=CHECK_ROOF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;

          can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
        {
          //can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE; //включаем, если было выключенно

          doors_bf_t __windows_position=System.signal_state.windows_position;
          if(__windows_position.driver_door)
          {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
          }
          if(__windows_position.pass_door || __windows_position.l_rear_door || __windows_position.r_rear_door) can_cmd_proc.cmd_q[i++]=EXT_WINDOWS_CLOSING_CMD_TYPE_GENESIS;
          if(__windows_position.driver_door)
          {
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
          }
          //if(__windows_position.driver_door) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//если закрываем это окно, значит включали зажигание и стекла разложились, шлем команду для их складывания
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          if(__windows_position.driver_door || __windows_position.pass_door || __windows_position.l_rear_door || __windows_position.r_rear_door)
          {
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          }

          can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          if(__windows_position.driver_door) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//если закрываем это окно, значит включали зажигание и стекла разложились, шлем команду для их складывания

          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==AUDI_A6_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

            //can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE; //пропускаем, т.к. на некоторых авто что-то с обучением, показывает открыто, а по факту закрыто
          }

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else if(can_program==VOLVO_XC40_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            //can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;
          }

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          /*
          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=5000;
          can_cmd_proc.cmd_q[i++]=VOLVO_XC40_CAN_CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          */
          can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_2_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else if(can_program==MB_E200_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;
          }

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//сатусы не врут

          can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else if(can_program==PORSCHE_MACAN_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            //can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;
          }

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          //ждем в любом случае
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;

          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;

          //включаем после проверки селектора, если селектор оказался не в P, то зажигание нельзя выключить и будет работать блокировка
          //tmp ->
          can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
          //<-tmp
          can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;
          can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;
          //tmp ->
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          //<-tmp

          //can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=TOYOTA_RAV4_WINDOWS_CLOSING_CMD_TYPE;//все обрабатывается в отдельном хэндлере
          can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_3; can_cmd_proc.delay[2]=200;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else if (can_program==LADA_LARGUS_2020_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CHECH_KEY_ACC_MODE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
        else
        {
          //can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
          if(can_program==CITROEN_JUMPY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM) {can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE;}
          else                                                                                    {can_cmd_proc.cmd_q[i++]=CHECK_PARKING_LIGHT_STATE_CMD_TYPE;}

          if(can_program==FORD_MUSTANG_1965_PROGRAM  || can_program==FORD_MUSTANG_1969_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=CHECK_CLS_AND_WINDOWS_CMD_TYPE;
          }

          can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;

          if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=5000;
          }

          if(can_program==BMW_520I_2018_PROGRAM)
          {
            if(_cls_state==UNLOCKED_CLS_STATE) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          }

          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) can_cmd_proc.cmd_q[i++]=FUEL_CAP_AND_TRUNK_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          if(!(can_program==AUDI_Q3_2018_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM))
          {
            if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
            {
              can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
            }
            else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
            {
              //none
            }
            else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
            {
              //none
            }
            else
            {
              if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
            }
            can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
            if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          }
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;
          if(can_program==AUDI_Q3_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;//у Q3 закрваем окна и все что свзянно с зажиганием до закрытия дверей, иначе машина их открывает вместе с зажиганием
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          }
          if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
            if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE; //включаем, если было выключенно
            if(System.signal_state.windows_position.driver_door)
            {
              can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            }
            if(System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door) can_cmd_proc.cmd_q[i++]=EXT_WINDOWS_CLOSING_CMD_TYPE_GENESIS;
            if(System.signal_state.windows_position.driver_door)
            {
              can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            }
            //if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE; будет отключенно далее
            if(System.signal_state.windows_position.driver_door) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//если закрываем это окно, значит включали зажигание и стекла разложились, шлем команду для их складывания
          }
          if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
          {
            //для того чтобы поехали окна передергиваем зажигание (без зажигания после открытия двери окна не работают)
            if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE; //включаем, если было выключенно
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            //if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE; будет отключенно далее
          }
          if(!(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM))
          {
            can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          }
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;
          if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          }
          else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
          {
            //none
          }
          else
          {
            if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
            if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
          }
          if(!(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==BMW_520I_2018_PROGRAM || \
            can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//у Genesis, XC60 открывает двери, у Velar нет эмуляции
          if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
          {
            #if defined(IS_GENESIS_2017_BUILD)
            #warning проверить, то что аренда завершается быстро и то что правильная последовательность уведомлений сервера
            #endif //defined(IS_GENESIS_2017_BUILD)
            can_cmd_proc.cmd_q[i++]=CHECK_CLS_AND_WINDOWS_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          }
          if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE; //если окна закрыты, то этот этап проходит быстро
            can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=200; //для того, чтобы обновился статус окон
            can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
            can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          }
          if(can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500; //для того, чтобы обновился статус цз
            can_cmd_proc.cmd_q[i++]=CHECK_CLS_AND_WINDOWS_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CHECK_DOORS_CLOSED_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          }
          if(can_program==VW_TRANSPORTER_2018_PROGRAM)
          {
            //должно быть передернуто зажигание, у CITROEN_JUMPY_2018_PROGRAM похоже так-же. Если подать ипульс на окна вместе с зажиганим, окна не поедут.
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
          }
          else
          {
            if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
               || can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            if(!(can_program==AUDI_Q3_2018_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM))
            {
              can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;//закрываем окна после информирования сервера
              if(can_program==NISSAN_QASHQAI_2018_PROGRAM) {can_cmd_proc.cmd_q[i++]=DELAY_3; can_cmd_proc.delay[2]=7000;} //окна управление выходом происходит асинхронно, но на коплектации с LIN водительское едет только с зажиганием
            }
            if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
              || can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
            {
              can_cmd_proc.cmd_q[i++]=DELAY_4; can_cmd_proc.delay[3]=500;
              can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
            }
          }
          can_cmd_proc.cmd_q[0]=CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE;
        }
      }
    }
    else if(cmd==YADRIVE_FORCED_END_OF_LEASE_SUBCODE)//безусловная завершение аренды (блокировка)
    {
      if(is_taxi())
      {
        can_cmd_proc.cmd_q[0]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
      }
      else
      {
        if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING1_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING1_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING2_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING2_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING3_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING3_2_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING4_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PANAMERA_WINDOWS_CLOSING4_2_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
            //can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=8000;
          }

          if(System.signal_state.roof.sunroof)
          {
            can_cmd_proc.cmd_q[i++]=ROOF_CLOSING_CMD_TYPE;
            //can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=8000;
          }

          //can_cmd_proc.cmd_q[i++]=CHECK_ROOF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
          //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
        {
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE; //включаем, если было выключенно

          doors_bf_t __windows_position=System.signal_state.windows_position;

          if(__windows_position.driver_door)
          {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
          }
          if(__windows_position.pass_door || __windows_position.l_rear_door || __windows_position.r_rear_door) can_cmd_proc.cmd_q[i++]=EXT_WINDOWS_CLOSING_CMD_TYPE_GENESIS;
          if(__windows_position.driver_door)
          {
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
          }
          //if(__windows_position.driver_door) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//если закрываем это окно, значит включали зажигание и стекла разложились, шлем команду для их складывания

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          if(__windows_position.driver_door) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//если закрываем это окно, значит включали зажигание и стекла разложились, шлем команду для их складывания

          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==AUDI_A6_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=400;
          //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

            //can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=8000;
          }

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==VOLVO_XC40_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          /*
          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=5000;
          can_cmd_proc.cmd_q[i++]=VOLVO_XC40_CAN_CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          */
          //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            //can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
          }

          can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=VOLVO_XC40_ROOF_CLOSING_2_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==MB_E200_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;

            //can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;
          }

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;//сатусы не врут

          can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==TOYOTA_RAV4_2019_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=TOYOTA_RAV4_WINDOWS_CLOSING_CMD_TYPE;//все обрабатывается в отдельном хэндлере
          can_cmd_proc.cmd_q[i++]=DELAY_3; can_cmd_proc.delay[2]=200;
          //can_cmd_proc.cmd_q[i++]=CHECK_WINDOWS_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else if(can_program==PORSCHE_MACAN_2018_PROGRAM)
        {
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=800;

          can_cmd_proc.cmd_q[i++]=SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;

          //can_cmd_proc.cmd_q[i++]=CHECK_CLS_CMD_TYPE;

          if(System.signal_state.windows_position.driver_door || System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door)
          {
            //can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            //can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=WAIT_WINDOWS_CLOSING_CMD_TYPE;
          }

          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;

          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;

          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
        else
        {
          //can_cmd_proc.cmd_q[i++]=STOP_ENGINE_CMD_TYPE;
          if(can_program==CITROEN_JUMPY_2018_PROGRAM)
          {
            bool block_relay_state=is_wired_block_relay_enable;
            if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_UNBLOCK_RELAY_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=1800;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            if(block_relay_state) can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          }
          if(can_program==BMW_520I_2018_PROGRAM)
          {
            if(_cls_state==UNLOCKED_CLS_STATE) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=500;
          }
          can_cmd_proc.cmd_q[i++]=HOOD_LOCK_CMD_TYPE;
          if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) can_cmd_proc.cmd_q[i++]=FUEL_CAP_AND_TRUNK_LOCK_CMD_TYPE;
          can_cmd_proc.cmd_q[i++]=CAN_WAKE_UP_CMD_TYPE;
          if(!(can_program==AUDI_Q3_2018_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM))
          {
            if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
            {
              can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
            }
            else
            {
              if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
              if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
              if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=DIAG_SESSION_CMD_TYPE;
            }

            //if(can_program==PORSCHE_MACAN_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;

            can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
            if(can_program==RENAULT_LOGAN_2013_PROGRAM || can_program==RENAULT_DUSTER_2021_PROGRAM || can_program==RENAULT_ARKANA_2018_PROGRAM || can_program==LADA_LARGUS_2020_PROGRAM) can_cmd_proc.cmd_q[i++]=LOGAN_2013_PRECAN_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;

            //if(can_program==PORSCHE_MACAN_2018_PROGRAM) can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
          }
          can_cmd_proc.cmd_q[i++]=WIRED_BLOCK_RELAY_CMD_TYPE;
          if(is_wireless_block_present(NULL)) can_cmd_proc.cmd_q[i++]=WIRELESS_BLOCK_RELAY_CMD_TYPE;
          if(can_program==AUDI_Q3_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;//у Q3 закрваем окна и все что свзянно с зажиганием до закрытия дверей, иначе машина их открывает вместе с зажиганием
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          }
          if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=SILENT_HORN_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=PARKING_LIGHT_OFF_CMD_TYPE;
            if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE; //включаем, если было выключенно
            if(System.signal_state.windows_position.driver_door)
            {
              can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
              can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
            }
            if(System.signal_state.windows_position.pass_door || System.signal_state.windows_position.l_rear_door || System.signal_state.windows_position.r_rear_door) can_cmd_proc.cmd_q[i++]=EXT_WINDOWS_CLOSING_CMD_TYPE_GENESIS;
            if(System.signal_state.windows_position.driver_door)
            {
              can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            }
            //if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE; будет отключенно далее
            if(System.signal_state.windows_position.driver_door) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;//если закрываем это окно, значит включали зажигание и стекла разложились, шлем команду для их складывания
          }
          if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
          {
            //для того чтобы поехали окна передергиваем зажигание (без зажигания после открытия двери окна не работают)
            if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_ON_CMD_TYPE; //включаем, если было выключенно
            can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
            //if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE; будет отключенно далее
          }
          if(is_key_power_present(NULL)) can_cmd_proc.cmd_q[i++]=KEY_POWER_OFF_CMD_TYPE;
          if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)
          {
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_1_CMD_TYPE;
            can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_PORSCHE_2_CMD_TYPE;
          }
          else
          {
            if(!(can_program==NISSAN_QASHQAI_2018_PROGRAM)) can_cmd_proc.cmd_q[i++]=CENTRAL_LOCK_CLOSE_CMD_TYPE;
          }

          if(can_program==KIA_RIO_2017_PROGRAM)
          {
            if(!System.can_state.sec_flags.key_in_ign) {can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;}//не шлем эмуляцию, если ключ повернут, т.к. это приведет к открытию цз
          }
          else
          {
            if(!(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==BMW_520I_2018_PROGRAM || \
              can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM)) can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;//у Genesis, XC60 открывает двери, у Velar нет эмуляции
          }
          if(can_program==VW_TRANSPORTER_2018_PROGRAM)
          {
            //должно быть передернуто зажигание, у CITROEN_JUMPY_2018_PROGRAM похоже так-же. Если подать ипульс на окна вместе с зажиганим, окна не поедут.
            can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
          }
          else
          {
            if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
              || can_program==NISSAN_QASHQAI_2018_PROGRAM) {can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;}
            if(!(can_program==AUDI_Q3_2018_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM))
            {
              can_cmd_proc.cmd_q[i++]=WINDOWS_CLOSING_CMD_TYPE;
              if(can_program==NISSAN_QASHQAI_2018_PROGRAM) {can_cmd_proc.cmd_q[i++]=DELAY_3; can_cmd_proc.delay[2]=7000;} //окна управление выходом происходит асинхронно, но на коплектации с LIN водительское едет только с зажиганием
            }
            if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || can_program==VW_CADDY_2018_PROGRAM \
              || can_program==NISSAN_QASHQAI_2018_PROGRAM) {can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;}
            if(can_program==NISSAN_QASHQAI_2018_PROGRAM)
            {
              can_cmd_proc.cmd_q[i++]=DELAY_4; can_cmd_proc.delay[3]=500;
              can_cmd_proc.cmd_q[i++]=DRIVER_DOOR_EMULATION_CMD_TYPE;
            }
          }
          can_cmd_proc.cmd_q[i++]=FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE;
          can_cmd_proc.cmd_q[0]=STOP_ENGINE_CMD_TYPE;
        }
      }
    }
    else if(cmd==YADRIVE_HOOD_UNLOCK_SUBCODE)
    {
      can_cmd_proc.cmd_q[i++]=HOOD_UNLOCK_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=HOOD_UNLOCK_CMD_TYPE;
    }
    else if(cmd==YADRIVE_HOOD_LOCK_SUBCODE)
    {
      can_cmd_proc.cmd_q[i++]=HOOD_LOCK_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=HOOD_LOCK_CMD_TYPE;
    }
    else if(cmd==YADRIVE_DTC_CLEAR_SUBCODE)
    {
      can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=OBD_DTC_CLEAR_TYPE;
      can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE; //
      can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;  // probably useless
      can_cmd_proc.cmd_q[i++]=OBD_DTC_UPDATE_TYPE;
      can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
    }
    else if(cmd==AUXILIARY_CAR_COMMANDS_SUBCODE)
    {
      const uint8_t* arg=(const uint8_t*)ext_args;

      if(arg[0]==2)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CHARGE_LOCK_CMD_TYPE;
      }
      else if(arg[0]==3)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=CHARGE_UNLOCK_CMD_TYPE;
      }
      else if(arg[0]==4)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=ECO_ON_CMD_TYPE;
      }
      else if(arg[0]==5)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=ECO_OFF_CMD_TYPE;
      }
      else if(arg[0]==6)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=FUEL_CAP_LOCK_CMD_TYPE;
      }
      else if(arg[0]==7)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=FUEL_CAP_UNLOCK_CMD_TYPE;
      }
      else if(arg[0]==8)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=TRUNK_LOCK_CMD_TYPE;
      }
      else if(arg[0]==9)
      {
        can_cmd_proc.cmd_q[i++]=ADD_EMTY_LAST_NOTIFY_CMD_TYPE;
        can_cmd_proc.cmd_q[0]=TRUNK_UNLOCK_CMD_TYPE;
      }
      else if(arg[0]==10)
      {
        const char* res_str="";
        notify_state_t notify_state;

        if(lease_state==IN_END_LEASE_STATE || lease_state==IN_FORCED_END_LEASE_STATE)
        {
          //выключить
          if(arg[1]==0)
          {
            res_str="";
            notify_state=IS_LAST_NOTIFY_STATE;

            set_head_control_state(false, 0);
          }
          //включить на время, мин
          else if(arg[1]==1)
          {
            set_head_control_state(true, *(uint16_t*)(&arg[2]));

            res_str="";
            notify_state=IS_LAST_NOTIFY_STATE;
          }
          else
          {
            res_str="";
            notify_state=UNKNOWN_ARGS_NOTIFY_STATE;
          }
        }
        else
        {
          res_str="not available in active lease";
          notify_state=ERROR_NOTIFY_STATE;
        }

        uint8_t add_res=add_server_notify(server_notify_id, uid, nonce, notify_state, res_str);
        notify_logging(server_notify_id, uid, notify_state, res_str, add_res);
      }
      else
      {
        return CAN_SUBTASK_COMMAND_UNKNOWN_ARG;
      }
    }
    else if(can_program==HYUNDAI_CRETA_2019_PROGRAM && cmd==CAN_WEBASTO_STARTING_SUBCODE)
    {
      can_cmd_proc.cmd_q[i++]=IGN_ON_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=CRETA_CAN_PARKING_LIGHT_ON_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=DELAY_1; can_cmd_proc.delay[0]=2000;
      can_cmd_proc.cmd_q[i++]=KIA_BCM_BEGIN_CONFIG_1_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=KIA_BCM_DISABLE_DOORS_BLOCK_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=KIA_BCM_END_CONFIG_2_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=DELAY_2; can_cmd_proc.delay[1]=1000;
      can_cmd_proc.cmd_q[i++]=KIA_BCM_BEGIN_CONFIG_1_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=KIA_BCM_DISABLE_DOORS_UNBLOCK_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=KIA_BCM_END_CONFIG_2_CMD_TYPE;
      can_cmd_proc.cmd_q[i++]=IGN_OFF_CMD_TYPE;
      can_cmd_proc.cmd_q[0]=CAN_WAKE_UP_CMD_TYPE;
    }
    else if(can_program==NISSAN_QASHQAI_2018_PROGRAM && cmd==CAN_WEBASTO_STARTING_SUBCODE)
    {
      can_cmd_proc.cmd_q[0]=CLEAR_BCM_ERRS_CMD_TYPE;
    }
    else if(can_program==NISSAN_QASHQAI_2018_PROGRAM && cmd==CAN_WEBASTO_STOPPING_SUBCODE)
    {
      can_cmd_proc.cmd_q[0]=CLEAR_ENGINE_ERRS_CMD_TYPE;
    }
    else if (cmd == CAN_OFF_AUTO_OPEN_DOORS)
    {
      can_cmd_proc.cmd_q[0] = OFF_AUTO_OPEN_DOORS_CMD_TYPE;
    }
    else if (cmd == CAN_FIX_FUEL_VIEW)
    {
      can_cmd_proc.cmd_q[0] = FIX_FUEL_VIEW_LOGAN_2013;
    }
    else
    {
      return CAN_SUBTASK_COMMAND_UNKNOWN_ARG;
    }

    return CAN_SUBTASK_COMMAND_START_PROCCES;
  }
  else
  {
    //потеряли команду
    return CAN_SUBTASK_COMMAND_IS_BUSY;
  }
}

#if defined(PROD_TESTING_PRESENT)
extern uint8_t start_prod_tests_cmd;
#endif //PROD_TESTING_PRESENT

static bool is_ignore_hood(void)
{
  if(can_program==HYUNDAI_CRETA_2019_PROGRAM)
  {
    return true;
  }
  else
  {
    return false;
  }
}

static bool is_antidrift_antilaunch_present(void)
{
  if(can_program==KIA_RIO_2017_PROGRAM || \
     can_program==RENAULT_KAPTUR_2016_PROGRAM || \
     can_program==NISSAN_QASHQAI_2018_PROGRAM || //проводная блокировка безонасос
     can_program==SKODA_OCTAVIA_2018_PROGRAM || \
     can_program==SKODA_RAPID_2018_PROGRAM || \
     can_program==HYUNDAI_SOLARIS_2017_PROGRAM || \
     //can_program==HYUNDAI_CRETA_2019_PROGRAM || //отъебывает катализатор
     can_program==VW_POLO_2018_PROGRAM || \
     can_program==SKODA_RAPID_2013_PROGRAM)
  {
    return true;
  }
  else
  {
    return false;
  }
}

static bool is_auto_doors_present(void)
{
  //SKODA_OCTAVIA_2018_PROGRAM машина сама блокирует двери, разблокиировки нет. Открывается если просто потянуть за ручку.
  //открытие и закрытие дверей только на "бомжах"
  if(can_program==KIA_RIO_2017_PROGRAM || can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM || \
     can_program==CITROEN_JUMPY_2018_PROGRAM || can_program==HYUNDAI_CRETA_2019_PROGRAM || \
     can_program==NISSAN_QASHQAI_2018_PROGRAM || \
     can_program==VW_POLO_2018_PROGRAM || \
     can_program==VW_TRANSPORTER_2018_PROGRAM || \
     can_program==SKODA_RAPID_2018_PROGRAM || \
     can_program==SKODA_RAPID_2013_PROGRAM)
  {
    return true;
  }
  else
  {
    return false;
  }
}

static void blocks_handler(void)
{
  uint8_t output_id;

  struct
  {
    bool drift_detect:1;
    bool launch_detect:1;
    bool is_force_output_apply:1;
  }vars=
  {
    .drift_detect=false,
    .launch_detect=false,
    .is_force_output_apply=false,
  };

  if(System.can_state.sec_flags.engine_is_on && !System.can_state.sec_flags.parking && System.can_state.sec_flags.hand_break && System.can_state.numeric_data.eng_rpm>=1900)
  {
    vars.drift_detect=true;
    System.signal_state.drift_sensor=1;
  }
  else if(System.can_state.sec_flags.engine_is_on && System.signal_state.selector_position=='N' && System.can_state.numeric_data.can_speed<10 && System.can_state.numeric_data.eng_rpm>=1900)
  {
    vars.launch_detect=true;
    System.signal_state.launch_sensor=1;
  }

  if(is_wired_block_present(&output_id))
  {
    //если активированна блокировка, то включаем реле при появлении зажигания
    if(is_wired_block_relay_enable && (System.can_state.sec_flags.ignition || System.signal_state.ignition))
    {
      System.signal_state.digital_output[output_id]=1;
      //outputs_state_force_apply();
    }
    else if(is_antidrift_antilaunch_present() && vars.drift_detect)
    {
      if(!System.signal_state.digital_output[output_id]) vars.is_force_output_apply=true;
      System.signal_state.digital_output[output_id]=1;
      //System.signal_state.drift_sensor=1;
    }
    else if(is_antidrift_antilaunch_present() && vars.launch_detect)
    {
      if(!System.signal_state.digital_output[output_id]) vars.is_force_output_apply=true;
      System.signal_state.digital_output[output_id]=1;
      //System.signal_state.launch_sensor=1;
    }
    else
    {
      if(is_antidrift_antilaunch_present() && (System.signal_state.launch_sensor || System.signal_state.drift_sensor) && System.can_state.numeric_data.eng_rpm>0)
      {
        //none
      }
      else
      {
        System.signal_state.digital_output[output_id]=0;
      }
    }

    if(vars.is_force_output_apply) outputs_state_force_apply();
  }

  //todo: есть вопросы по ложному срабатыванию при прогреве
  if(is_can_cmd_proc_q_emty() && can_cmd_proc.step==0 &&
     is_engine_started_by_terminal==0 &&
     (lease_state==IN_END_LEASE_STATE || lease_state==IN_FORCED_END_LEASE_STATE) &&
     System.can_state.numeric_data.eng_rpm>100)
  {
    can_timers[UNAUTHORIZED_START_EVENT_TIMER_ID]=3000;
    System.signal_state.unauthorized_start_sensor=1;
    System.can_state.cont_flags.warning=1;//temporary
  }
  else if(!can_timers[UNAUTHORIZED_START_EVENT_TIMER_ID])
  {
    System.signal_state.unauthorized_start_sensor=0;
    System.can_state.cont_flags.warning=0;//temporary
  }
}

static void taxi_blocks_handler(void)
{
  if(is_taxi())
  {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
    struct
    {
      lease_state_t _lease_state;
      bool is_wired_block_changed:1;
      bool is_wireless_block_changed:1;
    }taxi_blocks=
    {
      ._lease_state=lease_state,
      .is_wired_block_changed=false,
      .is_wireless_block_changed=false,
    };

    if(taxi_blocks._lease_state==IN_LEASE_STATE)
    {
      if(is_wired_block_present(NULL) && is_wired_block_relay_enable)
      {
        is_wired_block_relay_enable=false;
        taxi_blocks.is_wired_block_changed=true;
      }

      if(is_wireless_block_present(NULL) && System.signal_state.nrf_state.desired_relay_state)
      {
        System.signal_state.nrf_state.desired_relay_state=0;
        taxi_blocks.is_wireless_block_changed=true;
      }
    }
    else if(taxi_blocks._lease_state==IN_END_LEASE_STATE && System.can_state.sec_flags.engine_is_on==0 && System.can_state.numeric_data.can_speed==0)
    {
      if(is_wired_block_present(NULL) && !is_wired_block_relay_enable)
      {
        is_wired_block_relay_enable=true;
        taxi_blocks.is_wired_block_changed=true;
      }

      if(is_wireless_block_present(NULL) && !System.signal_state.nrf_state.desired_relay_state)
      {
        System.signal_state.nrf_state.desired_relay_state=1;
        taxi_blocks.is_wireless_block_changed=true;
      }
    }
    else if(taxi_blocks._lease_state==IN_FORCED_END_LEASE_STATE)
    {
      if(is_wired_block_present(NULL) && !is_wired_block_relay_enable)
      {
        is_wired_block_relay_enable=true;
        taxi_blocks.is_wired_block_changed=true;
      }

      if(is_wireless_block_present(NULL) && !System.signal_state.nrf_state.desired_relay_state)
      {
        System.signal_state.nrf_state.desired_relay_state=1;
        taxi_blocks.is_wireless_block_changed=true;
      }
    }

    if(is_wired_block_present(NULL) && taxi_blocks.is_wired_block_changed)
    {
      LOG("change wired relay state to %hhu\n", is_wired_block_relay_enable);
    }

    if(is_wireless_block_present(NULL) && taxi_blocks.is_wireless_block_changed)
    {
      LOG("change wireless relay state to %hhu\n", System.signal_state.nrf_state.desired_relay_state);
    }
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3
  }
}

static uint16_t get_timeout_in_horn_can_alarm_handler(void)
{
  if(can_program==AUDI_Q3_2018_PROGRAM)      {return 2000;} //гудит с зажиганием
  else if(can_program==AUDI_A3_2018_PROGRAM) {return 2000;} //гудит с зажиганием
  else                                       {return 1000;}
}

static bool is_horn_alarm_w_can_present(void)
{
  //аларм по CAN
  if(can_program==BMW_520I_2018_PROGRAM ||
     can_program==BMW_X1_2019_PROGRAM ||
     can_program==PORSCHE_MACAN_2018_PROGRAM || //с зажиганием
     //can_program==PORSCHE_PANAMERA_2019_PROGRAM || //зажигание не работет без ключа, без зажигания нет сигнала
     can_program==SKODA_OCTAVIA_2018_PROGRAM ||
     can_program==AUDI_Q3_2018_PROGRAM ||
     can_program==AUDI_A3_2018_PROGRAM ||
     //can_program==TOYOTA_RAV4_2019_PROGRAM
     can_program==FORD_TRANSIT_2017_PROGRAM //||
     )
  {
    return true;
  }
  else
  {
    return false;
  }
}

//Внимание, если есть прогрев и нужно зажигание для сигнала, то нужно рабераться отдельно!!!
static inline bool is_guard_present(bool* is_alarm_with_ign_on)
{
  if(can_program==KIA_RIO_2017_PROGRAM ||
     can_program==HYUNDAI_CRETA_2019_PROGRAM ||
     can_program==RENAULT_KAPTUR_2016_PROGRAM ||
     can_program==MB_E200_2018_PROGRAM ||
     can_program==BMW_520I_2018_PROGRAM ||
     can_program==TOYOTA_RAV4_2019_PROGRAM ||
     can_program==BMW_X1_2019_PROGRAM ||
     can_program==VOLVO_XC60_2018_PROGRAM ||
     can_program==VOLVO_XC40_2018_PROGRAM ||
     can_program==VW_POLO_2018_PROGRAM ||
     can_program==SKODA_RAPID_2013_PROGRAM ||
     can_program==SKODA_RAPID_2018_PROGRAM ||
     can_program==SKODA_OCTAVIA_2018_PROGRAM ||
     can_program==VW_TRANSPORTER_2018_PROGRAM ||
     can_program==NISSAN_QASHQAI_2018_PROGRAM ||
     can_program==AUDI_Q3_2018_PROGRAM ||
     can_program==AUDI_A3_2018_PROGRAM ||
     can_program==HYUNDAI_GENESIS_2017_PROGRAM ||
     can_program==FORD_TRANSIT_2017_PROGRAM ||
     can_program==CITROEN_JUMPY_2018_PROGRAM ||
     can_program==FORD_MUSTANG_1965_PROGRAM ||
     can_program==FORD_MUSTANG_1969_PROGRAM ||
     can_program==RENAULT_LOGAN_2013_PROGRAM ||
     can_program==RENAULT_DUSTER_2021_PROGRAM ||
     can_program==RENAULT_ARKANA_2018_PROGRAM ||
     can_program==JEEP_WRANGLER_2018_PROGRAM ||
     can_program==AUDI_A6_2018_PROGRAM ||
     can_program==LADA_LARGUS_2020_PROGRAM)
  {
    if(is_alarm_with_ign_on!=NULL) *is_alarm_with_ign_on=false;
    return true;
  }
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM
          //can_program==PORSCHE_PANAMERA_2019_PROGRAM //зажигание не работет без ключа, без зажигания нет сигнала
          )
  {
    if(is_alarm_with_ign_on!=NULL) *is_alarm_with_ign_on=true;
    return true;
  }

  if(is_alarm_with_ign_on!=NULL) *is_alarm_with_ign_on=false;
  return false;
}

void guard_handler(void)
{
  if(!is_guard_present(NULL))
  {
    return;
  }

  typedef struct
  {
    uint8_t on_guard:1;
    uint8_t driver_door:1;
    uint8_t pass_door:1;
    uint8_t l_rear_door:1;
    uint8_t r_rear_door:1;
    uint8_t hood:1;
    uint8_t trunk:1;
  }guard_state_t;

  static guard_state_t prev_guard_state =
  {
    .on_guard=0,
    .driver_door=1,
    .pass_door=1,
    .l_rear_door=1,
    .r_rear_door=1,
    .hood=1,
    .trunk=1,
  };

  static uint32_t flashlight_timer=xTaskGetTickCount();

  const char* event_str=NULL;
  uint8_t new_alarm_event=0;
  guard_state_t curr_state;

  if(!is_ignore_end_lease_state && (lease_state==IN_END_LEASE_STATE || lease_state==IN_FORCED_END_LEASE_STATE)) curr_state.on_guard=1;
  else                                                                                                          curr_state.on_guard=0;

  if(!can_timers[DRIVER_DOOR_IGNORE_TIMER_ID]) curr_state.driver_door=!!(System.can_state.sec_flags.driver_door);
  else                                         curr_state.driver_door=prev_guard_state.driver_door;
  curr_state.pass_door=!!(System.can_state.sec_flags.pass_door);
  curr_state.l_rear_door=!!(System.can_state.sec_flags.l_rear_door);
  curr_state.r_rear_door=!!(System.can_state.sec_flags.r_rear_door);
  curr_state.hood=!!((!is_ignore_hood() && System.can_state.sec_flags.hood));
  curr_state.trunk=!!(System.can_state.sec_flags.trunk);

  if(curr_state.on_guard)
  {
    if(curr_state.driver_door && curr_state.driver_door!=prev_guard_state.driver_door)
    {
      event_str="driver door";
      new_alarm_event=1;
    }
    else if(curr_state.pass_door && curr_state.pass_door!=prev_guard_state.pass_door)
    {
      event_str="pass door";
      new_alarm_event=2;
    }
    else if(curr_state.l_rear_door && curr_state.l_rear_door!=prev_guard_state.l_rear_door)
    {
      event_str="l rear door";
      new_alarm_event=3;
    }
    else if(curr_state.r_rear_door && curr_state.r_rear_door!=prev_guard_state.r_rear_door)
    {
      event_str="r rear door";
      new_alarm_event=4;
    }
    else if(curr_state.hood && curr_state.hood!=prev_guard_state.hood)
    {
      event_str="hood";
      new_alarm_event=5;
    }
    else if(curr_state.trunk && curr_state.trunk!=prev_guard_state.trunk)
    {
      event_str="trunk";
      new_alarm_event=6;
    }
  }

  if(new_alarm_event)
  {
    if(!is_horn_alarm_active() || !is_can_horn_alarm_active())
    {
      bool is_ign_enable;
      is_guard_present(&is_ign_enable);

      if(is_ign_enable)
      {
        uint8_t output_id;
        if(is_ignition_output_present(&output_id)) {System.signal_state.digital_output[output_id]=1; outputs_state_force_apply();}
      }
    }

    if(!is_horn_alarm_active())
    {
      horn_alarm_start(500, 500, 15);
    }

    if(!is_can_horn_alarm_active())
    {
      horn_can_alarm_start(15);
    }

    System.signal_state.alarm=new_alarm_event;

    LOG("new alarm event by %s\n", event_str);
  }
  else if(!curr_state.on_guard && curr_state.on_guard!=prev_guard_state.on_guard)
  {
    horn_can_alarm_stop();

    horn_alarm_stop();
    LOG("stop alarm by guard disabled\n");

    bool is_ign_enable;
    is_guard_present(&is_ign_enable);

    if(is_ign_enable)
    {
      uint8_t output_id;
      if(is_ignition_output_present(&output_id)) {System.signal_state.digital_output[output_id]=0;}
    }

    System.signal_state.alarm=0;
  }
  /*
  else if(curr_state.on_guard && is_can_cmd_proc_q_emty() && can_cmd_proc.step==0 && is_engine_started_by_terminal==0)
  {
    if(!is_horn_alarm_active() && !is_can_horn_alarm_active())
    {
      bool is_ign_enable;
      is_guard_present(&is_ign_enable);

      if(is_ign_enable)
      {
        uint8_t output_id;
        if(is_ignition_output_present(&output_id)) {System.signal_state.digital_output[output_id]=0;}
      }
    }
  }
  */

  //todo: проверить
  if(!is_horn_alarm_active() && !is_can_horn_alarm_active())
  {
    System.signal_state.alarm=0;
  }

  System.can_state.sec_flags.alarm_act=System.signal_state.alarm; //temporary

  /*
  if(is_horn_alarm_active())
  {
    if(timeAfter(xTaskGetTickCount(), flashlight_timer))
    {
      if(!uxQueueMessagesWaiting(Signal_manager.can_commands_queue))
      {
        if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(CAN_BLINKER_FLASHING_V2_SUBCODE, 0xff, 0, NULL))
        {
          flashlight_timer=xTaskGetTickCount()+1000;
        }
      }
    }
  }
  else
  {
    flashlight_timer=xTaskGetTickCount();
  }
  */

  horn_can_alarm_handler();

  prev_guard_state=curr_state;
}

//warning: static vars
void antitheft_handler(void)
{
  static struct
  {
    uint32_t emerg_timer_mS;
    uint8_t is_triggered;
  }antitheft=
  {
    .emerg_timer_mS=0,
    .is_triggered=0,
  };

  bool is_horn_and_light_enable=false;

  if(timeAfter(GetSecondsFromStartup(), SERVER_manager.yandex_force_block_timerS+5*60))
  {
    if(!antitheft.is_triggered && System.can_state.numeric_data.can_speed<4)
    {
      if(!uxQueueMessagesWaiting(Signal_manager.can_commands_queue))
      {
        if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(YADRIVE_FORCED_END_OF_LEASE_SUBCODE, 0xff, 0, NULL, NULL))
        {
          LOG("WARNING!!! antitheft triggered!!!\n");
          antitheft.is_triggered=1;
        }
      }
    }

    if(System.signal_state.gsensor_move_sensor || System.can_state.numeric_data.can_speed>=4)
    {
      is_horn_and_light_enable=true;

      if(timeAfter(xTaskGetTickCount(), antitheft.emerg_timer_mS))
      {
        if(!uxQueueMessagesWaiting(Signal_manager.can_commands_queue))
        {
          if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(CAN_HORN_W_BLINKER_SIGNALIG_SUBCODE, 0xff, 0, NULL, NULL))
          {
            antitheft.emerg_timer_mS=xTaskGetTickCount()+5000;
          }
        }
      }
    }
    else
    {
      antitheft.emerg_timer_mS=xTaskGetTickCount();
    }
  }
  else
  {
    antitheft.is_triggered=0;
  }

  static uint32_t debug_timer=xTaskGetTickCount();

  if(timeAfter(xTaskGetTickCount(), debug_timer))
  {
    LOG("antitheft debug: curr_tim: %lu, last_data_tim: %lu, antitheft_trig: %hhu, horn_light: %hhu\n", GetSecondsFromStartup(), SERVER_manager.yandex_force_block_timerS, antitheft.is_triggered, is_horn_and_light_enable);

    debug_timer=xTaskGetTickCount()+10*60*1000;
  }
}

//warning: static vars
void antitheft2_handler(void)
{
  static uint8_t antitheft_trig=0;
  static uint32_t antitheft_odo_start;

  if(System.gnss_state.coord_status != VALID && timeAfter(GetSecondsFromStartup(), SERVER_manager.yandex_force_block_timerS+1*60))
  {
    if(!antitheft_trig)
    {
      antitheft_trig=1;

      System.Grab(portMAX_DELAY);
      antitheft_odo_start=(uint32_t)System.can_state.numeric_data.can_odo_km;
      System.Release();
    }

    bool enable_lock=false;

    System.Grab(portMAX_DELAY);
    if((uint32_t)System.can_state.numeric_data.can_odo_km>antitheft_odo_start)
    {
      System.can_state.numeric_data.aft_srvc_odo=(uint32_t)System.can_state.numeric_data.can_odo_km-antitheft_odo_start;

      if(lease_state!=IN_FORCED_END_LEASE_STATE && System.can_state.numeric_data.can_speed==0 && System.can_state.numeric_data.aft_srvc_odo>=10)
      {
        enable_lock=true;
      }
    }
    else
    {
      System.can_state.numeric_data.aft_srvc_odo=0;
    }
    System.Release();

    if(enable_lock)
    {
      if(!uxQueueMessagesWaiting(Signal_manager.can_commands_queue))
      {
        if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(YADRIVE_FORCED_END_OF_LEASE_SUBCODE, 0xff, 0, NULL, NULL))
        {
          LOG("WARNING!!! antitheft triggered!!! odo: %u, trip: %u\n", System.can_state.numeric_data.can_odo_km, System.can_state.numeric_data.aft_srvc_odo);
        }
      }
    }
  }
  else
  {
    antitheft_trig=0;

    if(System.can_state.numeric_data.aft_srvc_odo)
    {
      System.Grab(portMAX_DELAY);
      System.can_state.numeric_data.aft_srvc_odo=0;
      System.Release();
    }
  }
}

static bool can_sleep_indicate_handler(void)
{
  if(is_can_rx_in_idle(get_can_from_id_from_periph(CAN1)))
  {
    System.can_state.sec_flags.ignition=0;
  }

  bool is_all_can_in_sleep=true;

  for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
  {
    if(!is_can_in_sleep(get_can_periph_from_id(can_id)))
    {
      is_all_can_in_sleep=false;
      break;
    }
  }

  return is_all_can_in_sleep;
}

static void restart_sleep_and_rx_timers(uint8_t can_id)
{
  if(can_id<HW_CAN_COUNT)
  {
    if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM && can_id==CAN3_ID)
    {
      //У Velar на CAN3 только диагностика, опрашиваем редко
      can_sleep.active_timer[can_id]=15000;
    }
    else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM && can_id==CAN3_ID)
    {
      //на CAN3 только диагностика, опрашиваем редко
      can_sleep.active_timer[can_id]=15000;
    }
    else if(can_program==VOLVO_XC40_2018_PROGRAM && can_id==CAN3_ID)
    {
      //на CAN3 почти ничего нет
      can_sleep.active_timer[can_id]=15000;
    }
    else if(can_program==MB_E200_2018_PROGRAM && can_id==CAN3_ID)
    {
      //не подключено
      can_sleep.active_timer[can_id]=15000;
    }
    else if(can_program==TOYOTA_RAV4_2019_PROGRAM && can_id==CAN3_ID)
    {
      //на CAN3 OBD
      can_sleep.active_timer[can_id]=15000;
    }
    //else if(can_program==BMW_520I_2018_PROGRAM && can_id==CAN1_ID)//раскоментировать после проверки
    //{
    //  //диагностика
    //  can_sleep.active_timer[can_id]=can_sleep.can_inactive_timeout*2;
    //}
    else if(can_program==BMW_X1_2019_PROGRAM && can_id==CAN1_ID)
    {
      //диагностика
      can_sleep.active_timer[can_id]=can_sleep.can_inactive_timeout*2;
    }
    else
    {
      can_sleep.active_timer[can_id]=can_sleep.can_inactive_timeout;
    }

    if(can_id==CAN1_ID)
    {
      can_timers[CAN1_MAIN_RX_TIMER_ID]=500;
    }
    else if(can_id==CAN2_ID)
    {
      can_timers[CAN2_MAIN_RX_TIMER_ID]=500;
    }
    else if(can_id==CAN3_ID)
    {
      can_timers[CAN3_MAIN_RX_TIMER_ID]=500;
    }
  }
}

#if defined(CAN_VEGA_FWD_PRESENT)
static void rx_fwd_hadler(const uint8_t can_id, const can_rx_frame_t* rx)
{
  if(can_id>=HW_CAN_COUNT) return;

  if(!is_can_timer_expired(CAN_RX_FWD_TIMER_ID))
  {
    if(fwd_can_setting[can_id].is_enable &&
       rx->id_type==fwd_can_setting[can_id].id_type &&
         ((rx->id&fwd_can_setting[can_id].mask)==(fwd_can_setting[can_id].value&fwd_can_setting[can_id].mask)))
    {
      write_to_rx_can_fwd_sfifo(can_id, rx);
    }
  }
}
#endif //defined(CAN_VEGA_FWD_PRESENT)

static void save_lease_state(void)
{
  RTC_WriteBackupRegister(LEASE_STATE_BKP_DR, lease_state);
}

static void save_key_power_state(bool is_on)
{
  //не используется, восстанавливается по lease_state
  if(true)
  {
    return;
  }
  else
  {
    uint32_t bkp_value=RTC_ReadBackupRegister(BLOCKS_STATE_BKP_DR);
    if(is_on) bkp_value|=(KEY_POWER_BKP_MASK);
    else      bkp_value&=~(KEY_POWER_BKP_MASK);
    RTC_WriteBackupRegister(BLOCKS_STATE_BKP_DR, bkp_value);
  }
}

static void save_wired_block_relay_state(bool is_lock)
{
  //не используется, восстанавливается по lease_state
  if(true)
  {
    return;
  }
  else
  {
    uint32_t bkp_value=RTC_ReadBackupRegister(BLOCKS_STATE_BKP_DR);
    if(is_lock) bkp_value|=(WIRED_BLOCK_BKP_MASK);
    else        bkp_value&=~(WIRED_BLOCK_BKP_MASK);
    RTC_WriteBackupRegister(BLOCKS_STATE_BKP_DR, bkp_value);
  }
}

static void save_wireless_block_relay_state(bool is_lock)
{
  //не используется, восстанавливается по lease_state
  if(true)
  {
    return;
  }
  else
  {
    uint32_t bkp_value=RTC_ReadBackupRegister(BLOCKS_STATE_BKP_DR);
    if(is_lock) bkp_value|=(WIRELESS_BLOCK_BKP_MASK);
    else        bkp_value&=~(WIRELESS_BLOCK_BKP_MASK);
    RTC_WriteBackupRegister(BLOCKS_STATE_BKP_DR, bkp_value);
  }
}

static void restore_lease_state(void)
{
  const char* res_str=NULL;

  lease_state=(lease_state_t)RTC_ReadBackupRegister(LEASE_STATE_BKP_DR);

  if(lease_state==IN_LEASE_STATE)
  {
    if(is_taxi()) res_str="unlock";
    else          res_str="in lease";
  }
  else if(lease_state==IN_END_LEASE_STATE)
  {
    if(is_taxi()) res_str="soft lock";
    else          res_str="end of lease";
  }
  else if(lease_state==IN_FORCED_END_LEASE_STATE)
  {
    if(is_taxi()) res_str="hard lock";
    else          res_str="forced end of lease";
  }
  else
  {
    res_str="unknown";
    lease_state=UNKNOWN_LEASE_STATE;
    save_lease_state();
  }

  if(res_str!=NULL)
  {
    LOG("restore %s state to \"%s\"\n", (is_taxi())?("taxi"):("lease"), res_str);
  }
}

static void restore_blocks_state(void)
{
  uint8_t output_id;

  if(is_taxi())
  {
    //восставнавливается по lease_state далее
    return;
  }
  else
  {
    if(is_wired_block_present(NULL))
    {
      if(lease_state==IN_LEASE_STATE) is_wired_block_relay_enable=false;
      else                            is_wired_block_relay_enable=true;
    }

    if(is_wireless_block_present(&output_id))
    {
      if(output_id!=0xff)
      {
        if(lease_state==IN_LEASE_STATE) System.signal_state.digital_output[output_id]=false;
        else                            System.signal_state.digital_output[output_id]=true;
      }

#if defined(INTERNAL_NRF_PRESENT)
      if(lease_state==IN_LEASE_STATE) System.signal_state.nrf_state.desired_relay_state=false;
      else                            System.signal_state.nrf_state.desired_relay_state=true;
#endif //defined(INTERNAL_NRF_PRESENT)
    }

    if(is_key_power_present(&output_id))
    {
      if(lease_state==IN_LEASE_STATE) System.signal_state.digital_output[output_id]=true;
      else                            System.signal_state.digital_output[output_id]=false;
    }

    if(is_ext_key_power_present(&output_id))
    {
      if(lease_state==IN_LEASE_STATE) System.signal_state.digital_output[output_id]=true;
      else                            System.signal_state.digital_output[output_id]=false;
    }
  }
}

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
static uint32_t get_lin_bus_speed(void)
{
  if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)   return 9600;
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM) return 9600;
  else                                              return 0;
}

void lin_rx_callback(const uint8_t* buff, uint8_t len)
{
  if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)
  {
    static const bool velar_checksum_enhanced=false;
    static const uint8_t velar_windows_ident=0x20;
    static const uint8_t velar_windows_key_quiet[]={0x00,0x00,0x08,0x00};
    static const uint8_t velar_windows_key_up[]={0x88,0x09,0x09,0x00};

    if(len==2 && buff[0]==0x55 && buff[1]==lin_get_pid_from_ident(velar_windows_ident))
    {
      uint8_t slave_responce[9];
      uint8_t responce_len;

      if(lin_cmd_step==0)
      {
        responce_len=sizeof(velar_windows_key_quiet);
        memcpy(slave_responce, velar_windows_key_quiet, responce_len);
      }
      else
      {
        responce_len=sizeof(velar_windows_key_up);
        memcpy(slave_responce, velar_windows_key_up, responce_len);
      }

      slave_responce[responce_len]=lin_calc_checksum(velar_windows_ident, velar_checksum_enhanced, slave_responce, responce_len);
      responce_len++;

      lin_uart_write(slave_responce, responce_len);
    }
  }
}
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

#if defined(FASTDATA_PRESENT)
static void fastdata_subtask(void* argument)
{
  static TVega_fast_data data={.struct_vers=1, .drop_counter=0,};

  fast_data_sfifo_init();

  TickType_t xLastWakeTime=xTaskGetTickCount();

  for(;;)
  {
    TickType_t xFrequency = System.fastdata_fwd_step*10;

    if(!xFrequency)          xFrequency=100;
    else if(xFrequency>1000) xFrequency=1000;

    vTaskDelayUntil(&xLastWakeTime, xFrequency);

    if(System.fastdata_fwd_step && !is_can_timer_expired(FASTDATA_TIMER))
    {
      System.Grab(portMAX_DELAY);

      data.tick_counter=xTaskGetTickCount();

      data.accel.x=AccelRaw.AXIS_X;
      data.accel.y=AccelRaw.AXIS_Y;
      data.accel.z=AccelRaw.AXIS_Z;

      data.gnss.time=System.gnss_raw_state.unix_time;
      data.gnss.lon=System.gnss_raw_state.lon;
      data.gnss.lat=System.gnss_raw_state.lat;
      data.gnss.valid=System.gnss_raw_state.coord_status;
      /*0....255 km/h */
      data.gnss.speed = uint16_to_uint8_limiter( System.gnss_state.speed );
      data.gnss.dir=System.gnss_raw_state.direction;
      data.gnss.is_on=System.gnss_state.is_on;

      data.can.speed=System.can_state.numeric_data.can_speed;
      data.can.accelerator=System.can_state.numeric_data.accelerator;
      data.can.brake=System.can_state.numeric_data.brake;
      data.can.steering_wheel=(int16_t)System.can_state.numeric_data.battery_charge_level;
      data.can.rpm=System.can_state.numeric_data.eng_rpm;
      data.can.steering_acceleration=System.can_state.numeric_data.thresher_rpm;

      System.Release();

      if(!write_to_fast_data_sfifo(&data))
      {
        data.drop_counter++;
      }
    }
  }
}
#endif //defined(FASTDATA_PRESENT)

void can_subtask_onetime_init(void)
{
  can_hw_onetime_init();

  memset(&System.signal_state.cls_state, 0, sizeof(System.signal_state.cls_state));
  System.signal_state.cls_state.pass_door=1; System.signal_state.cls_state.r_rear_door=1;//инициализируем в "разнобой", чтобы состояние было неопределено

  memset(&System.signal_state.windows_position, 0, sizeof(System.signal_state.windows_position));
  System.signal_state.windows_position.pass_door=1; System.signal_state.windows_position.r_rear_door=1;//инициализируем в "разнобой", чтобы состояние было неопределено
  if(can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM)
  {
    //задних стелоподьемники могут быть на "вёслах"
    System.signal_state.windows_position.r_rear_door=0; System.signal_state.windows_position.l_rear_door=0;
  }

  memset(&System.signal_state.roof, 0, sizeof(System.signal_state.roof));
  System.signal_state.roof.sunroof=1;//инициализируем в "разнобой"

  System.signal_state.selector_position='U';
  System.signal_state.key_position='U';
  System.signal_state.launch_sensor=0;
  System.signal_state.drift_sensor=0;
  System.signal_state.drive_on_handbrake=0;
  System.signal_state.eco_mode=0;
  System.signal_state.unauthorized_start_sensor=0;
  System.signal_state.tank_flap=0;
  System.signal_state.washer_liquid=0;
  System.signal_state.emergency_lights=0;
  System.signal_state.current_audio_file=0;
  System.signal_state.current_audio_script=0;
  memset(System.signal_state.vin_sensor, 0, sizeof(System.signal_state.vin_sensor));
  memset(System.signal_state.dtc_list, 0, sizeof(System.signal_state.dtc_list));

  if(can_program==BMW_520I_2018_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM ||  can_program==NISSAN_LEAF_2019_PROGRAM || can_program==HYUNDAI_SONATA_SD_2020_PROGRAM)
  {
    System.can_state.sec_flags.parking=1; //чтобы работал прогрев после рестарта блока
  }

#if defined(CAN_VEGA_FWD_PRESENT)
  can_tx_fwd_sfifo_init();
  can_rx_fwd_sfifo_init();
  can_timers[CAN_RX_FWD_TIMER_ID]=0; can_timers[CAN_RX_FWD_TIMER_ID]=0;

  System.Grab(portMAX_DELAY);
  System.fwd_can_server_id=0xFF;
  memset(&System.fwd_can_setting, 0, sizeof(System.fwd_can_setting));
  System.Release();
  memset(&fwd_can_setting, 0, sizeof(fwd_can_setting));
#endif //defined(CAN_VEGA_FWD_PRESENT)

#if defined(FASTDATA_PRESENT)
  System.fastdata_fwd_can_server_id=0xFF;
  force_expired_can_timer(FASTDATA_TIMER);
#endif //defined(FASTDATA_PRESENT)

#if defined(YANDEX_OBD_PRESENT)
  get_obd_conf(&obd_ctx.can_id, &obd_ctx.is_broadcast, obd_ctx.ecu_id);
#endif //defined(YANDEX_OBD_PRESENT)

  restore_lease_state();
  restore_blocks_state();

  set_head_control_state(((lease_state==IN_LEASE_STATE)?(true):(false)), 0);

  for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
  {
    can_hw_diag_handler(can_id, true);
  }

#if defined(FASTDATA_PRESENT)
  xTaskCreate(fastdata_subtask, "fastdata_subtask", configMINIMAL_STACK_SIZE+8, NULL, configMAX_PRIORITIES-1, NULL);
#endif //defined(FASTDATA_PRESENT)

  System.set_fuel_level_p(INVALID_FUEL_LEVEL_PERCENT);
}

void can_subtask_init(void)
{
  can_hw_programm_init();

#if defined(CAN_VEGA_FWD_PRESENT)
  for(uint8_t can_id=CAN1_ID; can_id<MAX_CAN_INTERFACES; can_id++)
  {
    update_can_hw_fwd_filter(can_id, &fwd_can_setting[can_id]);
  }
#endif //defined(CAN_VEGA_FWD_PRESENT)

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
  if(can_program==NISSAN_QASHQAI_2018_PROGRAM && get_lin_bus_speed())
  {
    lin_uart_init(get_lin_bus_speed());
  }
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3

  for(uint8_t i=0; i<(sizeof(can_cmd_proc.cmd_q)/sizeof(can_cmd_proc.cmd_q[0])); i++)
  {
    can_cmd_proc.cmd_q[i]=EMTY_CAN_CMD_TYPE;
  }
  can_cmd_proc.step=0;

  can_req_proc.req=EMTY_CAN_REQ_TYPE;
  can_req_proc.step=0;
}

void can_subtask_main_cycle(void)
{
  //ждем получения данных по CAN1, CAN2 или CAN3
  uint8_t ticks_wait;
  if(is_can_cmd_proc_q_emty()) ticks_wait=100U;
  else                         ticks_wait=1;

#if defined(CAN_VEGA_FWD_PRESENT)
  if(filled_count_in_tx_can_fwd_sfifo())
  {
    //ticks_wait=0;
  }
#endif //defined(CAN_VEGA_FWD_PRESENT)

  if((can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) &&  ticks_wait>50)
  {
    ticks_wait=50;
  }

  xSemaphoreTake(can_rx_bin_sem, ticks_wait);

#if defined(PROD_TESTING_PRESENT)
  if(start_prod_tests_cmd)
  {
    prod_test_handler();
    can_hw_deinit();
    for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
    {
      can_hw_diag_handler(can_id, true);
    }
    can_hw_programm_init();
    start_prod_tests_cmd=0;
  }
#endif //PROD_TESTING_PRESENT

  if(can_program==KIA_RIO_2017_PROGRAM)                                                             {can_rx_kia_rio_hyundai_solaris_2017_handler();}
  else if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM)     {can_rx_renault_kaptur_2016_handler();}
  else if(can_program==HYUNDAI_GENESIS_2017_PROGRAM)                                                {can_rx_hyundai_genesis_2017_handler();}
  else if(can_program==AUDI_A3_2018_PROGRAM  || can_program==SKODA_OCTAVIA_2018_PROGRAM || \
    can_program==SKODA_RAPID_2018_PROGRAM)                                                          {can_rx_audi_a3_2018_handler();}
  else if(can_program==AUDI_Q3_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || \
    can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM)                 {can_rx_audi_q3_2018_handler();}
  else if(can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM)         {can_rx_porsche_2018_handler();}
  else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)                                               {can_rx_porsche_panamera_2019_handler();}
  else if(can_program==MB_E200_2018_PROGRAM)                                                        {can_rx_mb_e200_2018_handler();}
  else if(can_program==TOYOTA_RAV4_2019_PROGRAM)                                                    {can_rx_toyota_rav4_2019_handler();}
  else if(can_program==BMW_520I_2018_PROGRAM)                                                       {can_rx_bmw_520i_2018_handler();}
  else if(can_program==HYUNDAI_SOLARIS_2017_PROGRAM)                                                {can_rx_kia_rio_hyundai_solaris_2017_handler();}
  else if(can_program==FORD_TRANSIT_2017_PROGRAM)                                                   {can_rx_ford_transit_2017_handler();}
  else if(can_program==CITROEN_JUMPY_2018_PROGRAM)                                                  {can_rx_citroen_jumpy_2018_handler();}
  else if(can_program==NISSAN_LEAF_2019_PROGRAM)                                                    {can_rx_nissan_leaf_2019_handler();}
  else if(can_program==LADA_LARGUS_2017_PROGRAM)                                                    {can_rx_lada_largus_2017_handler();}
  else if(can_program==VOLVO_XC60_2018_PROGRAM)                                                     {can_rx_volvo_xc60_2018_handler();}
  else if(can_program==VOLVO_XC40_2018_PROGRAM)                                                     {can_rx_volvo_xc40_2018_handler();}
  else if(can_program==RANGE_ROVER_VELAR_2018_PROGRAM)                                              {can_rx_range_rover_velar_2018_handler();}
  else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)                                                 {can_rx_nissan_qashqai_2018_handler();}
  else if(can_program==HYUNDAI_CRETA_2019_PROGRAM)                                                  {can_rx_hyundai_creta_2019_handler();}
  else if(can_program==BMW_X1_2019_PROGRAM)                                                         {can_rx_bmw_x1_2019_handler();}
  else if(can_program==FORD_MUSTANG_1965_PROGRAM)                                                   {ford_mustang_1965_handler();}
  else if(can_program==FORD_MUSTANG_1969_PROGRAM)                                                   {ford_mustang_1969_handler();}
  else if(can_program==HYUNDAI_SONATA_SD_2020_PROGRAM)                                              {can_rx_hyundai_sonata_sd_2020_handler();}
  else if(can_program==RENAULT_LOGAN_2013_PROGRAM ||
          can_program==RENAULT_DUSTER_2021_PROGRAM ||
          can_program==RENAULT_ARKANA_2018_PROGRAM ||
          can_program==LADA_LARGUS_2020_PROGRAM)                                                    {can_rx_renault_logan_2013_handler();}
  else if(can_program==JEEP_WRANGLER_2018_PROGRAM)                                                  {can_rx_jeep_wrangler_2018_handler();}
  else if(can_program==AUDI_A6_2018_PROGRAM)                                                        {can_rx_audi_a6_2018_handler();}

#if defined(CAN_VEGA_FWD_PRESENT)
  if(true)
  {
    uint16_t mess_count=filled_count_in_tx_can_fwd_sfifo();

    for(uint16_t j=0; j<mess_count; j++)
    {
      uint8_t can_id;
      can_tx_frame_t tx;

      read_from_tx_can_fwd_sfifo(&can_id, &tx);

      CAN_TypeDef* CANx=get_can_periph_from_id(can_id);
      if(CANx==NULL) continue;

      CanTxMsg can_tx;

      if(tx.id_type==STD_ID_TYPE)
      {
        can_tx.StdId=tx.id;
        can_tx.IDE=CAN_ID_STD;
      }
      else if(EXT_ID_TYPE)
      {
        can_tx.ExtId=tx.id;
        can_tx.IDE=CAN_ID_EXT;
      }
      else
      {
        continue;
      }

      if(!tx.rtr)
      {
        can_tx.RTR=CAN_RTR_DATA;
        can_tx.DLC=tx.dlen;
        memcpy(can_tx.Data, tx.data, sizeof(can_tx.Data));
      }
      else
      {
        can_tx.RTR=CAN_RTR_REMOTE;
      }

      CAN_Transmit_Blocked(CANx, &can_tx, 50);
    }
  }

  if(!can_timers[SETTINGS_CHECK_TIMER_ID])
  {
    for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
    {
      if(!can_timers[CAN_RX_FWD_TIMER_ID])
      {
        if(System.fwd_can_setting[can_id].is_enable)
        {
          System.Grab(portMAX_DELAY);
          memset(&System.fwd_can_setting[can_id], 0, sizeof(System.fwd_can_setting[can_id]));
          System.Release();
        }
      }

      if(memcmp(&fwd_can_setting[can_id], &System.fwd_can_setting[can_id], sizeof(fwd_can_setting[can_id]))!=0)
      {
        System.Grab(portMAX_DELAY);
        memcpy(&fwd_can_setting[can_id], &System.fwd_can_setting[can_id], sizeof(fwd_can_setting[can_id]));
        System.Release();

        update_can_hw_fwd_filter(can_id, &fwd_can_setting[can_id]);
      }
    }

    can_timers[SETTINGS_CHECK_TIMER_ID]=500; can_timers[SETTINGS_CHECK_TIMER_ID]=500;
  }

#endif //defined(CAN_VEGA_FWD_PRESENT)

  if(can_cmd_proc.cmd_q[0]!=EMTY_CAN_CMD_TYPE)
  {
    if(can_cmd_proc.cmd_q[0]==START_ENGINE_CMD_TYPE)
    {
      start_engine_handler();
    }
    else if(can_cmd_proc.cmd_q[0]==STOP_ENGINE_CMD_TYPE)
    {
      stop_engine_handler();
    }
    else if(can_cmd_proc.cmd_q[0]==DRIVER_DOOR_EMULATION_CMD_TYPE)
    {
      driver_door_emulation_handler();
    }
    else if(!(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==BMW_520I_2018_PROGRAM \
           || can_program==BMW_X1_2019_PROGRAM || can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_PANAMERA_2019_PROGRAM \
           /*|| can_program==TOYOTA_RAV4_2019_PROGRAM*/) && can_cmd_proc.cmd_q[0]==HORN_CMD_TYPE)
    {
      //AUDI A3, AUDI Q3 и Skoda Octavia управляется по CAN
      if(can_program==SKODA_RAPID_2018_PROGRAM || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM \
        || can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM) horn_handler(150);
      else                                                                               horn_handler(40);
    }
    else if(!(can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM || can_program==BMW_520I_2018_PROGRAM \
            || can_program==VW_POLO_2018_PROGRAM || can_program==SKODA_RAPID_2013_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM /*|| can_program==PORSCHE_MACAN_2018_PROGRAM*/ \
            /*|| can_program==TOYOTA_RAV4_2019_PROGRAM*/) && can_cmd_proc.cmd_q[0]==SILENT_HORN_CMD_TYPE)
    {
      //Все кроме AUDI A3, AUDI Q3 и Skoda Octavia управляется по аналогу
      if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM) horn_handler(6);
      else if(can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM)                                             horn_handler(15);
      else                                                                                                                                  horn_handler(10);
    }
    else if(is_central_lock_system_with_analog_control() && can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_OPEN_CMD_TYPE)
    {
      central_lock_system_handler(false);
    }
    else if(is_central_lock_system_with_analog_control() && can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_CLOSE_CMD_TYPE)
    {
      central_lock_system_handler(true);
    }
    else if(can_program==BMW_520I_2018_PROGRAM && ((can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_OPEN_CMD_TYPE && _cls_state==UNLOCKED_CLS_STATE) || (can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_CLOSE_CMD_TYPE && _cls_state==LOCKED_CLS_STATE)))
    {
      //если цз уже в нужном состоянии, то пропускаем
      //проверяем, есть ли еще что-то в cmd_q
      move_next_command_to_head();
      can_cmd_proc.step=0;
    }
    else if((can_program==RANGE_ROVER_VELAR_2018_PROGRAM) && can_cmd_proc.cmd_q[0]==WINDOWS_CLOSING_CMD_TYPE)
    {
      lin_windows_closing_handler();
    }
    else if(can_program==FORD_MUSTANG_1969_PROGRAM && can_cmd_proc.cmd_q[0]==WINDOWS_CLOSING_CMD_TYPE)
    {
      mustang_windows_closing_handler();
    }
    else if(!is_can_windows_closing() && can_cmd_proc.cmd_q[0]==WINDOWS_CLOSING_CMD_TYPE)
    {
      //Все кроме AUDI A3, AUDI Q3, Porsche Macan, Porsche 911, MB E200, BMW 520i и VW PPOLO управляется по аналогу
      //В GENESIS водительское закрывает аналогом, остальные по CAN
      //В Velar водительское закрывает аналогом, остальные по CAN (аналоговое управление производится после can_tx_commands_handler)
      analog_windows_closing_handler();
    }
    else if(can_cmd_proc.cmd_q[0]==TOYOTA_RAV4_WINDOWS_CLOSING_CMD_TYPE)
    {
      toyota_rav4_windows_closing_handler();
    }
    else if((can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) && can_cmd_proc.cmd_q[0]==PARKING_LIGHT_ON_CMD_TYPE)
    {
      mustang_headlights_flashing_handler();
    }
    else if(is_analog_emerg_lights_present(NULL) && can_cmd_proc.cmd_q[0]==PARKING_LIGHT_ON_CMD_TYPE)
    {
      emerg_flashing_handler(400);
    }
    else if(can_cmd_proc.cmd_q[0]==HOOD_LOCK_CMD_TYPE)
    {
      hood_lock_unlock_handler(true);
    }
    else if(can_cmd_proc.cmd_q[0]==HOOD_UNLOCK_CMD_TYPE)
    {
      hood_lock_unlock_handler(false);
    }
    else if(can_cmd_proc.cmd_q[0]==WIRED_BLOCK_RELAY_CMD_TYPE)
    {
      wired_block_relay_handler(true);
    }
    else if(can_cmd_proc.cmd_q[0]==WIRED_UNBLOCK_RELAY_CMD_TYPE)
    {
      wired_block_relay_handler(false);
    }
    else if(can_cmd_proc.cmd_q[0]==WIRELESS_BLOCK_RELAY_CMD_TYPE)
    {
      wireless_block_realy_handler(true);
    }
    else if(can_cmd_proc.cmd_q[0]==WIRELESS_UNBLOCK_RELAY_CMD_TYPE)
    {
      wireless_block_realy_handler(false);
    }
    else if(can_cmd_proc.cmd_q[0]==KEY_POWER_ON_CMD_TYPE)
    {
      key_power_handler(true);
    }
    else if(can_cmd_proc.cmd_q[0]==KEY_POWER_OFF_CMD_TYPE)
    {
      key_power_handler(false);
    }
    else if(can_cmd_proc.cmd_q[0]==IGN_ON_CMD_TYPE)
    {
      if(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM || can_program==MB_E200_2018_PROGRAM || can_program==AUDI_A6_2018_PROGRAM)
      {
        ign_on_off_w_key_handler(true);
      }
      else
      {
        ign_on_off_handler(true);
      }
    }
    else if(can_cmd_proc.cmd_q[0]==IGN_OFF_CMD_TYPE)
    {
      if(can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==RANGE_ROVER_VELAR_2018_PROGRAM || can_program==TOYOTA_RAV4_2019_PROGRAM || can_program==MB_E200_2018_PROGRAM || can_program==AUDI_A6_2018_PROGRAM)
      {
        ign_on_off_w_key_handler(false);
      }
      else
      {
        ign_on_off_handler(false);
      }
    }
    else if(can_cmd_proc.cmd_q[0]==FORD_TRANSIT_IGN_ON_CMD_TYPE)
    {
      ford_transit_ign_on_off_handler(true);
    }
    else if(can_cmd_proc.cmd_q[0]==FORD_TRANSIT_IGN_OFF_CMD_TYPE)
    {
      ford_transit_ign_on_off_handler(false);
    }
    else if(can_cmd_proc.cmd_q[0]==STATRTSTOP_BUTTON_PUSH_CMD_TYPE)
    {
      startstop_button_push_handler();
    }
    else if(can_cmd_proc.cmd_q[0]==DELAY_1 || can_cmd_proc.cmd_q[0]==DELAY_2 || can_cmd_proc.cmd_q[0]==DELAY_3 || can_cmd_proc.cmd_q[0]==DELAY_4)
    {
      delay_handler();
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_DOORS_CLOSED_AND_IGNITION_AND_KEY_OFF_CMD_TYPE)
    {
      check_doors_closed_and_ignition_off();
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_DOORS_CLOSED_CMD_TYPE)
    {
      check_doors_closed();
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_PARKING_STATE_CMD_TYPE || can_cmd_proc.cmd_q[0]==CHECK_PARKING_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE)
    {
      bool is_temporary_disable_wired_block=false;
      if(can_cmd_proc.cmd_q[0]==CHECK_PARKING_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE) is_temporary_disable_wired_block=true;

      check_parking_light_state(true, is_temporary_disable_wired_block);
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_PARKING_LIGHT_STATE_CMD_TYPE || can_cmd_proc.cmd_q[0]==CHECK_PARKING_LIGHT_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE)
    {
      bool is_temporary_disable_wired_block=false;
      if(can_cmd_proc.cmd_q[0]==CHECK_PARKING_LIGHT_STATE_W_TMP_DISABLED_WIRED_BLOCK_CMD_TYPE) is_temporary_disable_wired_block=true;

      if(can_program==KIA_RIO_2017_PROGRAM ||
         can_program==AUDI_A3_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM ||
         can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM ||
         can_program==SKODA_RAPID_2013_PROGRAM ||
         can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM ||
         can_program==PORSCHE_MACAN_2018_PROGRAM || can_program==PORSCHE_911_2018_PROGRAM  || can_program==PORSCHE_PANAMERA_2019_PROGRAM ||
         can_program==CITROEN_JUMPY_2018_PROGRAM || can_program==NISSAN_QASHQAI_2018_PROGRAM ||\
         can_program==HYUNDAI_CRETA_2019_PROGRAM /*|| can_program==RENAULT_LOGAN_2013_PROGRAM*/ /*|| can_program==RENAULT_DUSTER_2021_PROGRAM */ /*|| can_program==RENAULT_ARKANA_2018_PROGRAM*/)
      {
        //по просьбе от 16.03.18, не проверяем фары у RIO, A3, Q3, Octavia, Macan и 911
        check_parking_light_state(true, is_temporary_disable_wired_block);
      }
      else
      {
        check_parking_light_state(false, is_temporary_disable_wired_block);
      }
    }
    else if(can_cmd_proc.cmd_q[0]==WAIT_WINDOWS_CLOSING_CMD_TYPE)
    {
      wait_windows_closing();
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_CLS_AND_WINDOWS_CMD_TYPE)
    {
      check_cls_and_windows_state(true, true);
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_CLS_CMD_TYPE)
    {
      check_cls_and_windows_state(true, false);
    }
    else if(can_cmd_proc.cmd_q[0]==SKIP_TWO_CMD_IF_CLS_OPENED_CMD_TYPE)
    {
      skip_two_cmd_if_cls_opened_closed(false);
    }
    else if(can_cmd_proc.cmd_q[0]==SKIP_TWO_CMD_IF_CLS_CLOSED_CMD_TYPE)
    {
      skip_two_cmd_if_cls_opened_closed(true);
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_WINDOWS_CMD_TYPE)
    {
      check_cls_and_windows_state(false, true);
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_ROOF_CMD_TYPE)
    {
      check_roof_state();
    }
    else if(can_cmd_proc.cmd_q[0]==CHECK_BMW_CLS_CMD_TYPE)
    {
      check_bmw_cls_state();
    }
    else if(can_cmd_proc.cmd_q[0]==CHARGE_LOCK_CMD_TYPE)
    {
      charge_lock_unlock_handler(true);
    }
    else if(can_cmd_proc.cmd_q[0]==CHARGE_UNLOCK_CMD_TYPE)
    {
      charge_lock_unlock_handler(false);
    }
    else if(can_cmd_proc.cmd_q[0]==ECO_ON_CMD_TYPE)
    {
      eco_on_off_handler(true);
    }
    else if(can_cmd_proc.cmd_q[0]==ECO_OFF_CMD_TYPE)
    {
      eco_on_off_handler(false);
    }
    else if(can_program==FORD_MUSTANG_1965_PROGRAM && can_cmd_proc.cmd_q[0]==FUEL_CAP_LOCK_CMD_TYPE)
    {
      trunk_fuel_cap_lock_unlock_handler(false, true, true);
    }
    else if(can_program==FORD_MUSTANG_1965_PROGRAM && can_cmd_proc.cmd_q[0]==FUEL_CAP_UNLOCK_CMD_TYPE)
    {
      trunk_fuel_cap_lock_unlock_handler(false, true, false);
    }
    else if((can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) && can_cmd_proc.cmd_q[0]==TRUNK_LOCK_CMD_TYPE)
    {
      trunk_fuel_cap_lock_unlock_handler(true, false, true);
    }
    else if((can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) && can_cmd_proc.cmd_q[0]==TRUNK_UNLOCK_CMD_TYPE)
    {
      trunk_fuel_cap_lock_unlock_handler(true, false, false);
    }
    else if((can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) && can_cmd_proc.cmd_q[0]==FUEL_CAP_AND_TRUNK_UNLOCK_CMD_TYPE)
    {
      trunk_fuel_cap_lock_unlock_handler(true, true, false);
    }
    else if((can_program==FORD_MUSTANG_1965_PROGRAM || can_program==FORD_MUSTANG_1969_PROGRAM) && can_cmd_proc.cmd_q[0]==FUEL_CAP_AND_TRUNK_LOCK_CMD_TYPE)
    {
      trunk_fuel_cap_lock_unlock_handler(true, true, true);
    }
    else if(can_cmd_proc.cmd_q[0]==START_OF_LEASE_DONE_NOTTIFY_CMD_TYPE || can_cmd_proc.cmd_q[0]==END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE || can_cmd_proc.cmd_q[0]==FORCED_END_OF_LEASE_DONE_NOTTIFY_CMD_TYPE \
      || can_cmd_proc.cmd_q[0]==ADD_EMTY_LAST_NOTIFY_CMD_TYPE \
        || can_cmd_proc.cmd_q[0]==NOT_AVAILABLE_NOTTIFY_CMD_TYPE \
          || can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE ||  can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE \
            || can_cmd_proc.cmd_q[0]==HOOD_LOCK_NOTIFY_CMD_TYPE ||  can_cmd_proc.cmd_q[0]==HOOD_UNLOCK_NOTIFY_CMD_TYPE \
                )
    {
      add_server_notify_from_cmd();
    }
    /*
    else if(can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_CLOSE_NOTIFY_CMD_TYPE)
    {
    central_lock_open_close_notify_handler(true);
  }
    else if(can_cmd_proc.cmd_q[0]==CENTRAL_LOCK_OPEN_NOTIFY_CMD_TYPE)
    {
    central_lock_open_close_notify_handler(false);
  }
    else if(can_cmd_proc.cmd_q[0]==HOOD_LOCK_NOTIFY_CMD_TYPE)
    {
    hood_lock_unlock_notify_handler(true);
  }
    else if(can_cmd_proc.cmd_q[0]==HOOD_UNLOCK_NOTIFY_CMD_TYPE)
    {
    hood_lock_unlock_notify_handler(false);
  }
    */
    else if(can_cmd_proc.cmd_q[0]==CAN_WAKE_UP_CMD_TYPE)
    {
      can_wakeup_handler();
    }
    else if (can_cmd_proc.cmd_q[0] == OFF_AUTO_OPEN_DOORS_CMD_TYPE)
    {
      off_auto_open_doors();
    }
    else if (can_cmd_proc.cmd_q[0] == OBD_DTC_UPDATE_TYPE)
    {
      obd_cmd_handler(2 /* DTC UPDATE */ );
    }
    else if (can_cmd_proc.cmd_q[0] == OBD_DTC_CLEAR_TYPE)
    {
      obd_cmd_handler(3 /* DTC CLEAR */ );
    }
    else if (can_cmd_proc.cmd_q[0] == FIX_FUEL_VIEW_LOGAN_2013)
    {
      fix_fuel_view();
    }
    else if (can_cmd_proc.cmd_q[0] == CHECH_KEY_ACC_MODE_CMD_TYPE)
    {
      check_key_acc_mode_handler();
    }
    else
    {
      can_tx_commands_handler();
    }
  }

  //двигатель заведен, нет выполняющихся команд, нет выполняющихся запросов
  if(System.can_state.sec_flags.engine_is_on && can_cmd_proc.cmd_q[0]==EMTY_CAN_CMD_TYPE && can_req_proc.req==EMTY_CAN_REQ_TYPE)
  {
    if((can_program==KIA_RIO_2017_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==HYUNDAI_SOLARIS_2017_PROGRAM || can_program==HYUNDAI_CRETA_2019_PROGRAM) && !can_timers[UDS_1_TIMER_ID])
    {
      can_req_proc.req=GET_UDS_FUEL_LEVEL_CMD_TYPE;
      can_req_proc.CANx=CAN2;
      can_timers[UDS_1_TIMER_ID]=10000;
    }
    else if((can_program==TOYOTA_RAV4_2019_PROGRAM) && !can_timers[UDS_1_TIMER_ID])
    {
      can_req_proc.req=GET_UDS_WASHER_LIQUID_RAV4_CMD_TYPE;
      can_req_proc.CANx=CAN1;
      can_timers[UDS_1_TIMER_ID]=10000;
    }
    else if((can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==HYUNDAI_CRETA_2019_PROGRAM) && !can_timers[UDS_2_TIMER_ID])
    {
      can_req_proc.req=GET_UDS_DASH_BOARD_STATE_CMD_TYPE;
      can_req_proc.CANx=CAN2;
      can_timers[UDS_2_TIMER_ID]=10000;
    }
    else if((can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM /*|| can_program==AUDI_Q3_2018_PROGRAM*/ ||
             can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM /* || can_program==VW_TRANSPORTER_2018_PROGRAM*/ ||
             can_program==AUDI_A3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM ||
             can_program==SKODA_RAPID_2013_PROGRAM ||
             can_program==NISSAN_QASHQAI_2018_PROGRAM ||
             can_program==PORSCHE_PANAMERA_2019_PROGRAM || can_program==JEEP_WRANGLER_2018_PROGRAM) && !can_timers[OBD_1_TIMER_ID])
    {
      if(can_program==RENAULT_DOKKER_2018_PROGRAM)      can_req_proc.req=GET_OBD_FUEL_LITERS_DOKKER_CMD_TYPE;
      else if(can_program==NISSAN_QASHQAI_2018_PROGRAM) can_req_proc.req=GET_OBD_FUEL_LITERS_QASHQAI_CMD_TYPE;
      else                                              can_req_proc.req=GET_OBD_FUEL_TANK_CMD_TYPE;

      if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM) can_req_proc.CANx=CAN1;
      else if(can_program==NISSAN_QASHQAI_2018_PROGRAM)                                        can_req_proc.CANx=CAN1;
      else if(can_program==PORSCHE_PANAMERA_2019_PROGRAM)                                      can_req_proc.CANx=CAN3;
      else if(can_program==JEEP_WRANGLER_2018_PROGRAM )                                        can_req_proc.CANx=CAN1;
      else                                                                                     can_req_proc.CANx=CAN2;
      can_timers[OBD_1_TIMER_ID]=10000;
    }
    else if((can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM) && !can_timers[OBD_1_TIMER_ID])
    {
      can_req_proc.CANx=CAN1;
      can_req_proc.req=GET_OBD_FUEL_TANK_XC60_CMD_TYPE;
      can_timers[OBD_1_TIMER_ID]=10000;
    }
    else if((can_program==RANGE_ROVER_VELAR_2018_PROGRAM) && !can_timers[OBD_1_TIMER_ID])
    {
      can_req_proc.CANx=CAN3;
      can_req_proc.req=GET_OBD_FUEL_TANK_VELAR_CMD_TYPE;
      can_timers[OBD_1_TIMER_ID]=10000;
    }
    else if((can_program==KIA_RIO_2017_PROGRAM || can_program==HYUNDAI_GENESIS_2017_PROGRAM || can_program==HYUNDAI_SOLARIS_2017_PROGRAM ||
             can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM || can_program==AUDI_Q3_2018_PROGRAM ||
             can_program==VW_POLO_2018_PROGRAM || can_program==VW_CADDY_2018_PROGRAM || can_program==VW_TRANSPORTER_2018_PROGRAM ||
             can_program==SKODA_RAPID_2013_PROGRAM ||
             can_program==AUDI_A3_2018_PROGRAM || can_program==SKODA_OCTAVIA_2018_PROGRAM || can_program==SKODA_RAPID_2018_PROGRAM ||
             can_program==FORD_TRANSIT_2017_PROGRAM) && !can_timers[OBD_2_TIMER_ID])
    {
      can_req_proc.req=GET_OBD_ENGINE_TEMP_CMD_TYPE;
      if(can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM) can_req_proc.CANx=CAN1;
      else                                                                                     can_req_proc.CANx=CAN2;
      can_timers[OBD_2_TIMER_ID]=10200;
    }
    else if((can_program==RANGE_ROVER_VELAR_2018_PROGRAM) && !can_timers[OBD_2_TIMER_ID])
    {
      can_req_proc.CANx=CAN3;
      can_req_proc.req=GET_OBD_ENGINE_TEMP_VELAR_CMD_TYPE;
      can_timers[OBD_2_TIMER_ID]=10200;
    }
    else if((can_program==VOLVO_XC60_2018_PROGRAM || can_program==VOLVO_XC40_2018_PROGRAM) && !can_timers[OBD_2_TIMER_ID])
    {
      can_req_proc.CANx=CAN1;
      can_req_proc.req=GET_OBD_ENGINE_TEMP_XC60_CMD_TYPE;
      can_timers[OBD_2_TIMER_ID]=10200;
    }
    else if((can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM) && !can_timers[OBD_3_TIMER_ID])
    {
      can_req_proc.req=GET_OBD_KAPTUR_ODO_CMD_TYPE;
      can_req_proc.CANx=CAN1;
      can_timers[OBD_3_TIMER_ID]=11450;
    }
    else if(can_program==SKODA_RAPID_2018_PROGRAM && can_bus_type[1]!=MOTOR_CAN_BUS_TYPE && !can_timers[OBD_3_TIMER_ID])
    {
      can_req_proc.req=GET_OBD_RAPID_ODO_CMD_TYPE;
      can_req_proc.CANx=CAN2;
      can_timers[OBD_3_TIMER_ID]=11450;
    }
    /*
    else if(can_program==PORSCHE_MACAN_2018_PROGRAM && !can_timers[OBD_3_TIMER_ID])
    {
    can_req_proc.req=GET_OBD_PORSCHE_ODO_CMD_TYPE;
    can_req_proc.CANx=CAN2;
    can_timers[OBD_3_TIMER_ID]=11450;
  }
    */
    /*
    else if((can_program==RENAULT_KAPTUR_2016_PROGRAM || can_program==RENAULT_DOKKER_2018_PROGRAM) &&  !can_timers[OBD_4_TIMER_ID])
    {
    can_req_proc.req=GET_OBD_FUEL_FLOW_CMD_TYPE;
    can_req_proc.CANx=CAN1;
    can_timers[OBD_4_TIMER_ID]=2000;
  }
    */
  }

  if(can_req_proc.req!=EMTY_CAN_REQ_TYPE)
  {
    if(can_req_proc.req==GET_UDS_FUEL_LEVEL_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7C6, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0xB0,0x02,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=150+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else if(can_req_proc.step==2)
      {
        CanTxMsg can_tx = {0x7C6, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x30,0x08,0x02,0x00,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_UDS_WASHER_LIQUID_RAV4_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7C0, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0x11,0x01,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=150+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else if(can_req_proc.step==2)
      {
        CanTxMsg can_tx = {0x7C0, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_UDS_DASH_BOARD_STATE_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7C6, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0xb0,0x03,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=150+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else if(can_req_proc.step==2)
      {
        CanTxMsg can_tx = {0x7C6, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x30,0x00,0x02,0x00,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_FUEL_TANK_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7DF, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x02,0x01,0x2F,0x00,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_FUEL_LITERS_DOKKER_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x743, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0x22,0x03,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_FUEL_LITERS_QASHQAI_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x743, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x02,0x21,0x01,0xFF,0xFF,0xFF,0xFF,0xFF}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=150+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else if(can_req_proc.step==2)
      {
        CanTxMsg can_tx = {0x743, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x30,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0xFF}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_ENGINE_TEMP_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7DF, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x02,0x01,0x05,0x00,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_FUEL_FLOW_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7DF, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x02,0x01,0x5E,0x00,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_KAPTUR_ODO_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x743, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0x02,0x07,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_RAPID_ODO_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7E0, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0x10,0xE0,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_ENGINE_TEMP_XC60_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x730, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0xF4,0x05,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_FUEL_TANK_XC60_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x730, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0xF4,0x2F,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_ENGINE_TEMP_VELAR_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        //CanTxMsg can_tx = {0x7E0, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0xF4,0x67,0x00,0x00,0x00,0x00}};//Температура ОЖ, не работает
        CanTxMsg can_tx = {0x7E0, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0x03,0xA3,0x00,0x00,0x00,0x00}};//Температура масла
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    else if(can_req_proc.req==GET_OBD_FUEL_TANK_VELAR_CMD_TYPE)
    {
      if(can_req_proc.step==0)
      {
        CanTxMsg can_tx = {0x7E0, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0xF4,0x2F,0x00,0x00,0x00,0x00}};
        CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
        can_req_proc.timer=100+xTaskGetTickCount();
        can_req_proc.step++;
      }
      else if(can_req_proc.step==1)
      {
        if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
      }
      else
      {
        can_req_proc.req=EMTY_CAN_REQ_TYPE;
        can_req_proc.step=0;
      }
    }
    /*
    else if(can_req_proc.req==GET_OBD_VIN_CMD_TYPE)
    {
    if(can_req_proc.step==0)
    {
    CanTxMsg can_tx = {0x7DF, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x02,0x09,0x02,0x00,0x00,0x00,0x00,0x00}};
    CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
    can_req_proc.timer=150+xTaskGetTickCount();
    can_req_proc.step++;
  }
    else if(can_req_proc.step==1)
    {
    if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
  }
    else if(can_req_proc.step==2)
    {
    CanTxMsg can_tx = {0x7DF-8, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00}};
    CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
    can_req_proc.timer=500+xTaskGetTickCount();
    can_req_proc.step++;
  }
    else
    {
    can_req_proc.req=EMTY_CAN_REQ_TYPE;
    can_req_proc.step=0;
  }
  }
    */
    /*
    else if (can_req_proc.req==GET_OBD_PORSCHE_ODO_CMD_TYPE)
    {
    if(can_req_proc.step==0)
    {
    CanTxMsg can_tx = {0x7E0, 0, CAN_ID_STD, CAN_RTR_Data, 8, {0x03,0x22,0x10,0x1e,0x00,0x00,0x00,0x00}};
    CAN_Transmit_Blocked(can_req_proc.CANx, &can_tx, 100);
    can_req_proc.timer=100+xTaskGetTickCount();
    can_req_proc.step++;
  }
    else if(can_req_proc.step==1)
    {
    if(timeAfter(xTaskGetTickCount(), can_req_proc.timer)) can_req_proc.step++;
  }
    else
    {
    can_req_proc.req=EMTY_CAN_REQ_TYPE;
    can_req_proc.step=0;
  }
  }
    */
    else
    {
      can_req_proc.req=EMTY_CAN_REQ_TYPE;
      can_req_proc.step=0;
    }
  }

#if defined(YANDEX_OBD_PRESENT)
  if(can_cmd_proc.cmd_q[0]==EMTY_CAN_CMD_TYPE && can_req_proc.req==EMTY_CAN_REQ_TYPE)
  {
    obd_tx_handler();
  }
#endif //defined(YANDEX_OBD_PRESENT)

  if(can_cmd_proc.cmd_q[0]==EMTY_CAN_CMD_TYPE && can_req_proc.req==EMTY_CAN_REQ_TYPE)
  {
    for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
    {
      can_hw_diag_handler(can_id, false);
    }
  }

  for(uint8_t can_id=CAN1_ID; can_id<HW_CAN_COUNT; can_id++)
  {
    can_sleep_wakeup_hadler(can_id);
  }

  LedCAN_Off();
}

void can_subtask_deinit(void)
{
  can_hw_deinit();
}

bool can_subtask_is_sleep_allowed(void)
{
  return is_can_cmd_proc_q_emty();
}

#if defined(CAN_VEGA_FWD_PRESENT)

static bool update_can_hw_fwd_filter(const uint8_t can_id, const can_fwd_settings_t* const fwd_can_setting)
{
  CAN_TypeDef* CANx;
  uint8_t can_hw_fifo;
  uint8_t filter_number;

  if(can_id==CAN1_ID)
  {
    CANx=CAN1;
    can_hw_fifo=CAN1_FIFO;
    filter_number=MAX_HW_FILTER_BANK_PER_CAN-1;
  }
#if (HW_CAN_COUNT > 1)
  else if(can_id==CAN2_ID)
  {
    CANx=CAN2;
    can_hw_fifo=CAN2_FIFO;
    filter_number=2*MAX_HW_FILTER_BANK_PER_CAN-1;//для CAN1 и CAN2 фильтр один
  }
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
  else if(can_id==CAN3_ID)
  {
    CANx=CAN3;
    can_hw_fifo=CAN3_FIFO;
    filter_number=MAX_HW_FILTER_BANK_PER_CAN-1;
  }
#endif //(HW_CAN_COUNT > 2)
  else
  {
    return false;
  }

  CAN_FilterInitTypeDef CAN_FilterInitStructure;

  if(fwd_can_setting->is_enable && fwd_can_setting->id_type==STD_ID_TYPE)
  {
    CAN_FilterInitStructure.CAN_FilterActivation = ENABLE;
    CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdMask;
    CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_16bit;
    CAN_FilterInitStructure.CAN_FilterIdHigh =     (uint16_t)((fwd_can_setting->value)<<5);
    CAN_FilterInitStructure.CAN_FilterIdLow =      (uint16_t)((fwd_can_setting->value)<<5);
    CAN_FilterInitStructure.CAN_FilterMaskIdHigh = (uint16_t)((fwd_can_setting->mask)<<5);
    CAN_FilterInitStructure.CAN_FilterMaskIdLow =  (uint16_t)((fwd_can_setting->mask)<<5);
  }
  else if(fwd_can_setting->is_enable && fwd_can_setting->id_type==EXT_ID_TYPE)
  {
    CAN_FilterInitStructure.CAN_FilterActivation = ENABLE;
    CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdMask;
    CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_32bit;
    CAN_FilterInitStructure.CAN_FilterIdHigh =     (uint16_t)((fwd_can_setting->value)>>13);
    CAN_FilterInitStructure.CAN_FilterIdLow =      (uint16_t)(((fwd_can_setting->value)<<3)|(1<<2));
    CAN_FilterInitStructure.CAN_FilterMaskIdHigh = (uint16_t)((fwd_can_setting->mask)>>13);
    CAN_FilterInitStructure.CAN_FilterMaskIdLow =  (uint16_t)(((fwd_can_setting->mask)<<3)|(1<<2));
  }
  else
  {
    CAN_FilterInitStructure.CAN_FilterActivation = DISABLE;
    CAN_FilterInitStructure.CAN_FilterMode =       CAN_FilterMode_IdList;
    CAN_FilterInitStructure.CAN_FilterScale =      CAN_FilterScale_32bit;
    CAN_FilterInitStructure.CAN_FilterIdHigh =     0;
    CAN_FilterInitStructure.CAN_FilterIdLow =      0;
    CAN_FilterInitStructure.CAN_FilterMaskIdHigh = 0;
    CAN_FilterInitStructure.CAN_FilterMaskIdLow =  0;
  }

  CAN_FilterInitStructure.CAN_FilterFIFOAssignment = can_hw_fifo;
  CAN_FilterInitStructure.CAN_FilterNumber =         filter_number;

#if defined(STM32F413_423xx)
  if(CANx==CAN2) {CANx=CAN1;}//для CAN1 и CAN2 фильтр один
  CAN_FilterInit(CANx, &CAN_FilterInitStructure);
#else
  if(CANx==CAN2) {}
  CAN_FilterInit(&CAN_FilterInitStructure);
#endif

  return true;
}

bool can_fwd_cmd_callback(const uint8_t* const args, const uint16_t args_len, const uint8_t server_id, const char** verbose)
{
  if(args_len!=sizeof(uint8_t)+sizeof(uint8_t)+MAX_CAN_INTERFACES*sizeof(can_fwd_settings_t)+sizeof(uint8_t))
  {
    verbose[0] = "vega can fwd config err, wrong arg len";
    return false;
  }

  if(args[0]==0 && args[1]==MAX_CAN_INTERFACES)
  {
    System.Grab(portMAX_DELAY);
    for(uint8_t can_id=CAN1_ID; can_id<MAX_CAN_INTERFACES; can_id++)
    {
      System.fwd_can_setting[can_id].is_enable=0;
    }

    memset(&System.fwd_can_setting, 0, sizeof(System.fwd_can_setting));
    System.fwd_can_server_id=server_id;
    memcpy(&System.fwd_can_setting, &args[sizeof(uint8_t)+sizeof(uint8_t)], sizeof(System.fwd_can_setting));
    System.Release();

    force_expired_can_timer(SETTINGS_CHECK_TIMER_ID);

    if(is_can_timer_expired(CAN_RX_FWD_TIMER_ID)) verbose[0] = "vega can fwd config";
    else                                          verbose[0] = NULL;

    restart_can_timer(CAN_RX_FWD_TIMER_ID, 1000*args[sizeof(uint8_t)+sizeof(uint8_t)+MAX_CAN_INTERFACES*sizeof(can_fwd_settings_t)]);

    return true;
  }

  verbose[0] = "vega can fwd config err, wrong args";

  return false;
}

void can_fwd_new_data_event(const bool is_tx)
{
  if(can_rx_bin_sem!=NULL)
  {
    xSemaphoreGive(can_rx_bin_sem);
  }
}
#endif //defined(CAN_VEGA_FWD_PRESENT)

#if defined(FASTDATA_PRESENT)
#define MAX_CAN_TIMER_VAL_S 65U
bool fastdata_fwd_cmd_callback(const uint8_t* const args, const uint16_t args_len, const uint8_t server_id, const char** verbose)
{
  if(args_len!=sizeof(uint8_t)+sizeof(uint8_t)+sizeof(uint8_t))
  {
    verbose[0] = "fast data config err, wrong arg len";
    return false;
  }

  if(args[0]==0)
  {
    System.fastdata_fwd_can_server_id=server_id;

    if(args[1]>100)
    {
      System.fastdata_fwd_step=0;
      force_expired_can_timer(FASTDATA_TIMER);
      verbose[0] = "fast data config err, wrong args";
      return false;
    }

    System.fastdata_fwd_step=args[1];

    if(is_can_timer_expired(FASTDATA_TIMER)) verbose[0] = "fast data config";
    else                                     verbose[0] = NULL;

    if(System.fastdata_fwd_step)
    {
      if ( args[2] <= MAX_CAN_TIMER_VAL_S) {
        restart_can_timer (FASTDATA_TIMER, 1000U*args[2]);
      } else {
        verbose[0] = "too big period for can timer";
    	return false;
      }
    }
    else
    {
      force_expired_can_timer(FASTDATA_TIMER);
    }

    return true;
  }

  verbose[0] = "fast data config err, wrong args";

  return false;
}
#endif //defined(FASTDATA_PRESENT)

static void can_receive(CAN_TypeDef* CANx, const uint8_t fifo, can_rx_frame_t* rx)
{
  if(((uint8_t)0x04 & CANx->sFIFOMailBox[fifo].RIR) == CAN_Id_Standard)
  {
    rx->id_type = STD_ID_TYPE;
    rx->id = (uint32_t)0x000007FF & (CANx->sFIFOMailBox[fifo].RIR >> 21);
  }
  else
  {
    rx->id_type = EXT_ID_TYPE;
    rx->id = (uint32_t)0x1FFFFFFF & (CANx->sFIFOMailBox[fifo].RIR >> 3);
  }

  if(((uint8_t)0x02 & CANx->sFIFOMailBox[fifo].RIR) == CAN_RTR_Remote)
  {
    rx->rtr=1;
  }
  else
  {
    rx->rtr=0;
  }

  rx->dlen=(uint8_t)0x0F & CANx->sFIFOMailBox[fifo].RDTR;

  /* Get the data field */
  rx->data[0] = (uint8_t)0xFF & CANx->sFIFOMailBox[fifo].RDLR;
  rx->data[1] = (uint8_t)0xFF & (CANx->sFIFOMailBox[fifo].RDLR >> 8);
  rx->data[2] = (uint8_t)0xFF & (CANx->sFIFOMailBox[fifo].RDLR >> 16);
  rx->data[3] = (uint8_t)0xFF & (CANx->sFIFOMailBox[fifo].RDLR >> 24);
  rx->data[4] = (uint8_t)0xFF & CANx->sFIFOMailBox[fifo].RDHR;
  rx->data[5] = (uint8_t)0xFF & (CANx->sFIFOMailBox[fifo].RDHR >> 8);
  rx->data[6] = (uint8_t)0xFF & (CANx->sFIFOMailBox[fifo].RDHR >> 16);
  rx->data[7] = (uint8_t)0xFF & (CANx->sFIFOMailBox[fifo].RDHR >> 24);

  /* Release the FIFO */
  if(fifo == CAN_FIFO0)
  {
    CANx->RF0R |= CAN_RF0R_RFOM0;
  }
  else /* FIFONumber == CAN_FIFO1 */
  {
    CANx->RF1R |= CAN_RF1R_RFOM1;
  }
}

#ifdef __cplusplus
extern "C" {
#endif
  /* user implemented, send can message */
  int isotp_user_send_can(const uint8_t interface_index, const uint8_t arbitration_id_type, const uint32_t arbitration_id, const uint8_t* data, const uint8_t size)
  {
    CanTxMsg tx;

    CAN_TypeDef* CANx=get_can_periph_from_id(interface_index);
    if(CANx==NULL) return -1;

    if(!arbitration_id_type)
    {
      tx.StdId=arbitration_id;
      tx.IDE=CAN_ID_STD;
    }
    else
    {
      tx.ExtId=arbitration_id;
      tx.IDE=CAN_ID_EXT;
    }

    tx.RTR=CAN_RTR_DATA;
    tx.DLC=size;
    memcpy(tx.Data, data, sizeof(tx.Data));

    if(CAN_Transmit_Blocked(CANx, &tx, 50)) {return 0;}
    else                                    {return -1;}
  }

  /* user implemented, get millisecond */
  uint32_t isotp_user_get_ms(void)
  {
    return xTaskGetTickCount();
  }

#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif

#if defined(VEGA_CAN_PRESENT)

#pragma optimize=speed
  static void can_sce_irq_handler(CAN_TypeDef* CANx)
  {
    if(CAN_GetITStatus(CANx, CAN_IT_WKU) != RESET)
    {
      CAN_ClearITPendingBit(CANx, CAN_IT_WKU);

      if(CANx==CAN1)      {__CAN1_NORMAL_MODE(); can_sleep.wakeup_event[CAN1_ID]=true;}
#if (HW_CAN_COUNT > 1)
      else if(CANx==CAN2) {__CAN2_NORMAL_MODE(); can_sleep.wakeup_event[CAN2_ID]=true;}
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
      else if(CANx==CAN3) {__CAN3_NORMAL_MODE(); can_sleep.wakeup_event[CAN3_ID]=true;}
#endif //(HW_CAN_COUNT > 2)
    }

    if(CAN_GetITStatus(CANx, CAN_IT_SLK) != RESET)
    {
      CAN_ClearITPendingBit(CANx, CAN_IT_SLK);

      if(CANx==CAN1)      {__CAN1_STANDBY_MODE();}
#if (HW_CAN_COUNT > 1)
      else if(CANx==CAN2) {__CAN2_STANDBY_MODE();}
#endif //(HW_CAN_COUNT > 1)
#if (HW_CAN_COUNT > 2)
      else if(CANx==CAN3) {__CAN3_STANDBY_MODE();}
#endif //(HW_CAN_COUNT > 2)
    }
  }

#pragma optimize=speed
  void CAN1_RX_IRQHandler(void)
  {
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;
    can_rx_frame_t rx;

    rx.time=xTaskGetTickCountFromISR();
    can_receive(CAN1, CAN1_FIFO, &rx);

    write_to_sfifo(&can_rx_sfifo[CAN1_ID], &rx);

    LedCAN_Toggle();

    xSemaphoreGiveFromISR(can_rx_bin_sem, &xHigherPriorityTaskWoken);
    if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
  }

  #pragma optimize=speed
  void CAN1_SCE_IRQHandler(void)
  {
    can_sce_irq_handler(CAN1);
  }

#if (HW_CAN_COUNT > 1)
#pragma optimize=speed
  void CAN2_RX_IRQHandler(void)
  {
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;
    can_rx_frame_t rx;

    rx.time=xTaskGetTickCountFromISR();
    can_receive(CAN2, CAN2_FIFO, &rx);

    write_to_sfifo(&can_rx_sfifo[CAN2_ID], &rx);

    LedCAN_Toggle();

    xSemaphoreGiveFromISR(can_rx_bin_sem, &xHigherPriorityTaskWoken);
    if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
  }

#pragma optimize=speed
  void CAN2_SCE_IRQHandler(void)
  {
    can_sce_irq_handler(CAN2);
  }
#endif //(HW_CAN_COUNT > 1)

#if (HW_CAN_COUNT > 2)
#pragma optimize=speed
  void CAN3_RX_IRQHandler(void)
  {
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;
    can_rx_frame_t rx;

    rx.time=xTaskGetTickCountFromISR();
    can_receive(CAN3, CAN3_FIFO, &rx);

    write_to_sfifo(&can_rx_sfifo[CAN3_ID], &rx);

    LedCAN_Toggle();

    xSemaphoreGiveFromISR(can_rx_bin_sem, &xHigherPriorityTaskWoken);
    if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
  }

#pragma optimize=speed
  void CAN3_SCE_IRQHandler(void)
  {
    can_sce_irq_handler(CAN3);
  }
#endif //(HW_CAN_COUNT > 2)

#endif //VEGA_CAN_PRESENT

#ifdef __cplusplus
}
#endif

#include "signal_manager/prod_tests.c"