/**
******************************************************************************
* File Name          : signalman_can_subtask.cpp
* Description        : 
*
*
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "signal_manager/signal_manager.h"
#include "signal_manager/signalman_nrf_subtask.h"
#include "Soft_timers/Soft_timers.h"

#include "debug_port.h"
#include "outputs_driver.h"
#include "leds_driver.h"
#include "math.h"

#include "nrf_bsp_driver.h"
#include "nrf_common.h"
#include "nrf_GlobalService.h"
#include "nrf_subroutines.h"
#include "y_eeprom.h"

/* C code---------------------------------------------------------------------*/

/* Global variables-----------------------------------------------------------*/
static SemaphoreHandle_t nrf_it_bin_sem=NULL;

#define NRF_RELAY_TIMER_ID              0
#define NRF_MARKS_SCAN_TIMER_ID         1
#define NRF_LAST_TIMER_ID               NRF_MARKS_SCAN_TIMER_ID

static volatile uint16_t nrf_timers[NRF_LAST_TIMER_ID+1]={0}; 

typedef enum
{
  NRF_NO_COMMAND=0,
  NRF_ADD_RELAY_COMMAND,
  NRF_DEL_RELAY_COMMAND,
  NRF_ADD_MARK_COMMAND,
  NRF_DEL_MARK_COMMAND,
}nrf_cmd_type_t;

static struct nrf_cmd_proc_struct
{
  nrf_cmd_type_t command;
  uint8_t mark_id;
}nrf_cmd_proc;

uint8_t signalman_nrf_subtask_commnad_handle(uint8_t cmd, uint8_t arg)
{
  if(NRF_NO_COMMAND==nrf_cmd_proc.command)
  {
    if(cmd==COMMAND_SIGNAL_NRF_ADD_RELAY)
    {
      nrf_cmd_proc.command=NRF_ADD_RELAY_COMMAND;
    }
    else if(cmd==COMMAND_SIGNAL_NRF_DEL_RELAY)
    {
      nrf_cmd_proc.command=NRF_DEL_RELAY_COMMAND;
    }
    else if(cmd==COMMAND_SIGNAL_NRF_ADD_MARK)
    {
      nrf_cmd_proc.mark_id=arg;
      nrf_cmd_proc.command=NRF_ADD_MARK_COMMAND;
    }
    else if(cmd==COMMAND_SIGNAL_NRF_DEL_MARK)
    {
      nrf_cmd_proc.mark_id=arg;
      nrf_cmd_proc.command=NRF_DEL_MARK_COMMAND;
    }
    else
    {
      return NRF_SUBTASK_COMMAND_UNKNOWN_ARG;
    }
    
    System.signal_state.nrf_state.ready_for_command=0;
    return NRF_SUBTASK_COMMAND_START_PROCCES;
  }
  else
  {
    // 
    return NRF_SUBTASK_COMMAND_IS_BUSY;
  }
}

static void sync_nrf_with_system(void)
{
  System.Grab(portMAX_DELAY);
  memcpy(&System.signal_state.nrf_state.relay_id, &SNrl.E32, sizeof(System.signal_state.nrf_state.relay_id));
  for(uint8_t i=0; i<MAX_INTERNAL_NRF_MARKS_COUNT; i++)
  {
    memcpy(&System.signal_state.nrf_state.marks_id[i], &SNnrf[i].E32, sizeof(System.signal_state.nrf_state.marks_id[i]));
  }
  
  System.signal_state.nrf_state.visible_marks_bf=NRFCtrl.mark_online_status;
  System.signal_state.nrf_state.battlow_marks_bf=NRFCtrl.mark_battery_status;
  
  System.Release();
}

void nrf_subtask_onetime_init(void)
{
  if(nrf_it_bin_sem==NULL)
  {
    vSemaphoreCreateBinary(nrf_it_bin_sem);
  }
  xSemaphoreTake(nrf_it_bin_sem, 0);
  
  static_assert(!(MAX_INTERNAL_NRF_MARKS_COUNT!=MAX_MARK_NUMBER), "wrong MAX_INTERNAL_NRF_MARKS_COUNT");
  eeprom_RestoreAllBuffers();
  sync_nrf_with_system();
  
  System.signal_state.nrf_state.ready_for_command=0;
  System.signal_state.nrf_state.actual_relay_state=0xff;
}

static void nrf_irq(void) 
{
  BaseType_t xHigherPriorityTaskWoken = pdFALSE;
   
  nrf_SetNordicInterrupt(SET);
  
  xSemaphoreGiveFromISR(nrf_it_bin_sem, &xHigherPriorityTaskWoken);
  if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
}

void nrf_subtask_init(void)
{
  vTaskPrioritySet(NULL, uxTaskPriorityGet(NULL)+2);
  
  nrf_cmd_proc.command=NRF_NO_COMMAND;
  System.signal_state.nrf_state.ready_for_command=1;
    
  nrf_bsp_init(nrf_irq);
}

void nrf_subtask_deinit(void)
{
  vTaskPrioritySet(NULL, uxTaskPriorityGet(NULL)-2);
  
  System.signal_state.nrf_state.ready_for_command=0;
  
  nrf_bsp_deinit();
}

#ifdef __cplusplus
extern "C" {
#endif

#pragma optimize=speed
void nrf_timers_callback(void)
{
  WaitTimeOut_Count(); //       
  nrf_CountModeTimeOut(); //     
  
  for(uint8_t i=0; i<sizeof(nrf_timers)/sizeof(nrf_timers[0]); i++)
  {
    if(nrf_timers[i]) nrf_timers[i]--;
  }
}

void bb_lock_unlock(uint8_t is_lock)
{
  if(is_lock) Black_box.Grab(portMAX_DELAY);
  else        Black_box.Release();
}

void nrf_wait_irq_event(void)
{
  // 1 
  xSemaphoreTake(nrf_it_bin_sem, 1);
}

#ifdef __cplusplus
}
#endif

static uint8_t nrf_sync_poll_counter;

static void nrf_sync_reset(void)
{
  nrf_sync_poll_counter=0;
  xSemaphoreTake(nrf_it_bin_sem, 0);
}

static void nrf_sync_wait_events(void)
{
  if(xSemaphoreTake(nrf_it_bin_sem, 0))
  {
    nrf_sync_poll_counter=3;
  }
  else if(!nrf_sync_poll_counter)
  {
    if(xSemaphoreTake(nrf_it_bin_sem, 200)) nrf_sync_poll_counter=6;
  }
  else
  {
    nrf_sync_poll_counter--;
  }
}

void nrf_subtask_main_cycle(void)
{
  bool is_reley_present=false;
  if(System.signal_state.nrf_state.relay_id) is_reley_present=true;
  
  if(is_reley_present && !nrf_timers[NRF_RELAY_TIMER_ID] && System.signal_state.nrf_state.actual_relay_state!=System.signal_state.nrf_state.desired_relay_state)
  {
    uint8_t relay_state=System.signal_state.nrf_state.desired_relay_state;
    bool is_ok=false;
    
    if(relay_state) GC_Launch(mm_RelayLock);
    else            GC_Launch(mm_RelayUnlock);
    
    nrf_sync_reset();
    nRF_GlobalService();     

    if(TotalResult.flag == SET) //  
    {
      TotalResult.flag = RESET;
      if(TotalResult.code == relay_ConnectSuccess) //    
      {
        //   
        System.signal_state.nrf_state.actual_relay_state=relay_state;
        is_ok=true;
      }
    }
    
    if(is_ok)
    {
      nrf_timers[NRF_RELAY_TIMER_ID]=1000;
    }
    else
    {
      System.signal_state.nrf_state.actual_relay_state=0xff;
      //  ,       
      if(System.signal_state.ignition) nrf_timers[NRF_RELAY_TIMER_ID]=1200;
      else                             nrf_timers[NRF_RELAY_TIMER_ID]=60000;
    }
  }
  else if(System.signal_state.ignition && (nrf_timers[NRF_RELAY_TIMER_ID])>1200)
  {
    //         ,  
    nrf_timers[NRF_RELAY_TIMER_ID]=1200; nrf_timers[NRF_RELAY_TIMER_ID]=1200;
  }
  
  bool is_marks_present=false;
  for(uint8_t i=0; i<MAX_INTERNAL_NRF_MARKS_COUNT; i++)
  {
    if(System.signal_state.nrf_state.marks_id[i]) {is_marks_present=true; break;}
  }
  
  if(!is_marks_present) 
  {
    System.signal_state.nrf_state.visible_marks_bf=0;
    System.signal_state.nrf_state.battlow_marks_bf=0;
  }
  else if(is_marks_present && !nrf_timers[NRF_MARKS_SCAN_TIMER_ID])
  {
    GC_Launch(mm_ScanNorm);

    nrf_sync_reset();
    nRF_GlobalService();     
    
    for(;;)
    {
      nrf_sync_wait_events();
      nRF_GlobalService();
      
      if(TotalResult.flag == SET) //  
      {
        TotalResult.flag = RESET;
        
        if(TotalResult.code == scan_ConnectSuccess || TotalResult.code == scan_SearchTimeOut)
        {
          sync_nrf_with_system();
          break;
        }
      }
    }
    
    nrf_timers[NRF_MARKS_SCAN_TIMER_ID]=6000;
  }
  
  if(nrf_cmd_proc.command==NRF_ADD_RELAY_COMMAND || nrf_cmd_proc.command==NRF_ADD_MARK_COMMAND)
  {
    if(nrf_cmd_proc.command==NRF_ADD_RELAY_COMMAND) {GC_Launch(mm_ScanProgRelay);}
    else                                            {GC_Launch(mm_ScanProgMark); GC_SetCell(nrf_cmd_proc.mark_id);}
    
    for(;;)
    {
      nrf_sync_wait_events();
      nRF_GlobalService();
      
      if(TotalResult.flag == SET) //  
      {
        TotalResult.flag = RESET;
        
        if(nrf_cmd_proc.command==NRF_ADD_RELAY_COMMAND)
        {
          if(TotalResult.code == scan_GraftSuccessRelay) //   
          {
            LOG("nrf relay add done: %lX\n", SNrl.E32);
          }
          
          if(TotalResult.code == scan_SearchTimeOut) //      / 
          {
            LOG("nrf reley add timeout\n");
          }
        }
        else
        {
          if(TotalResult.code == scan_GraftSuccess) //    
          {
            LOG("nrf mark add done: %lX (%hhu)\n", SNnrf[TotalResult.data].E32, nrf_cmd_proc.mark_id);
          }
          
          if(TotalResult.code == scan_SearchTimeOut) //       / 
          {
            LOG("nrf mark add timeout\n");     
          }
        }
        
        sync_nrf_with_system();
        nrf_cmd_proc.command=NRF_NO_COMMAND;
        System.signal_state.nrf_state.ready_for_command=1;
        break;
      }
    }
  }
  else if(nrf_cmd_proc.command==NRF_DEL_RELAY_COMMAND)
  {
    cph_DeleteRelay();
    sync_nrf_with_system();
    LOG("nrf relay deleted\n");
    nrf_cmd_proc.command=NRF_NO_COMMAND;
    System.signal_state.nrf_state.ready_for_command=1;
  }
  else if(nrf_cmd_proc.command==NRF_DEL_MARK_COMMAND)
  {
    cph_DeleteMark(nrf_cmd_proc.mark_id);
    sync_nrf_with_system();
    LOG("nrf %hhu mark deleted\n", nrf_cmd_proc.mark_id);
    nrf_cmd_proc.command=NRF_NO_COMMAND;
    System.signal_state.nrf_state.ready_for_command=1;
  }
  else
  {
    if(NRF_NO_COMMAND!=nrf_cmd_proc.command) 
    {
      nrf_cmd_proc.command=NRF_NO_COMMAND; 
      System.signal_state.nrf_state.ready_for_command=1;
    }
    vTaskDelay(100);
  }
}
