/**
  ******************************************************************************
  * File Name          : syscom_manager.cpp
  * Description        : Осуществляет внутрисистемную коммуникацию между
  *                      менеджерами, определяет логику работы устройства,
  *                      осуществляет конфигурирование устройства
  *
  ******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "syscom_manager.h"
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include "leds_driver.h"
#include "gps_manager/gps_manager.h"
#include "server_manager/server_manager.h"
#include "signal_manager/signal_manager.h"
#include "signal_manager/signalman_can_subtask.h"
#include "system_messages.h"
#include "crc_lib/crc_lib.h"
#include "debug_port.h"
#include "rtc_driver.h"
#include "memory_spi.h"
#include "MX25L6435E/MX25L6435E.h"
#include "NOR_FTL/NOR_FTL.h"
#include "serv_protocols_lib/vega/vega_protocol.h"
#include "mbedtls/poly1305.h"
#include "tea_crypt/tea.h"
#include "time/timer.h"
//#include "fast_data_manager/fast_data_manager.h"

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
#include "signal_manager/signalman_ble_subtask.h"
#endif //EXTERNAL_BLE_BOARD_PRESENT

#if defined(JQ6500_AUDUO_PRESENT)
#include "signal_manager/audio_manadge.h"
#endif //defined(JQ6500_AUDUO_PRESENT)

#include "setup/setup_after_upload.h"

/* Global variables-----------------------------------------------------------*/
// Менеджер
static void Update_flash_condition(TickType_t SE_max_time);
CSyscom_manager Syscom_manager(SYSCOM_MANAGER_NAME, 0, SYSCOM_MANAGER_INPUT_CUEUE_SIZE, SYSCOM_MANAGER_PRIORITY, Start_syscom_manager, SYSCOM_MANAGER_STACK_SIZE);
// Драйвер SPI
CMemory_spi Memory_spi;
// Функции для сязки драйвера SPI и драйвера микросхемы памяти macronix
static void SPI_csup(void) {Memory_spi.CS_up();}
static void SPI_csdown(void) {Memory_spi.CS_down();}
static void SPI_senddata(const uint8_t* data, uint16_t len) {Memory_spi.Send_data(data, len);}
static void SPI_sendbyte(uint8_t byte) {Memory_spi.Send_byte(byte);}
static void SPI_getdata(uint8_t* data, uint16_t len) {Memory_spi.Get_data(data, len);}
static uint8_t SPI_getbyte(void) {return Memory_spi.Get_byte();}
static void SPI_init(void) {Memory_spi.Init();}
static void SPI_deinit(void) {Memory_spi.Deinit();}
// Драйвер памяти macronix
CMX25L64 MX25_memory(&SPI_csup, &SPI_csdown, &SPI_senddata, &SPI_sendbyte, &SPI_getdata, &SPI_getbyte, &SPI_init, &SPI_deinit);
// Функции для связки драйвера памяти macronix с FTL
static uint8_t MX25_memory_init(void){return MX25_memory.Init();}
static void MX25_memory_deinit(void){MX25_memory.Deinit();}
static void MX25_memory_write(uint16_t sector, uint16_t address, const uint8_t* data, uint16_t size){MX25_memory.Write_data(sector, address, data, size);}
static void MX25_memory_read(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size){MX25_memory.Read_data(sector, address, data, size);}
static void MX25_memory_erase_sector(uint16_t sector){MX25_memory.Erase_sector(sector); Update_flash_condition(MX25_memory.Get_SE_max_time());}
static void MX25_memory_erase_chip(void){MX25_memory.Erase_chip();}
// NOR flash translation layer (библиотека для выравнивания износа памяти)
CNOR_ftl NOR_ftl(&MX25_memory_init, &MX25_memory_deinit, &MX25_memory_write, &MX25_memory_read, &MX25_memory_erase_sector, &MX25_memory_erase_chip);
// Функции для связки FTL с черным ящиком
static uint8_t mem_init(void){return (uint8_t)NOR_ftl.Init();}
static void    mem_deinit(void){NOR_ftl.Deinit();}
static uint8_t mem_format(void){return (uint8_t)NOR_ftl.Format_memory();}
static uint8_t mem_write(uint32_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Write_data(address, data, size);}
static uint8_t mem_read(uint32_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Read_data(address, data, size);}
static uint8_t mem_direct_write(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Direct_write_data(sector, address, data, size);}
static uint8_t mem_direct_read(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Direct_read_data(sector, address, data, size);}
static uint8_t mem_direct_erase_sector(uint16_t sector){return (uint8_t)NOR_ftl.Direct_erase_sector(sector);}
// Черный ящик
CBlack_box Black_box(mem_init, mem_deinit, mem_format, mem_write, mem_read, mem_direct_write, mem_direct_read, mem_direct_erase_sector, NOR_FTL_DIRECT_SECTOR_COUNT, NOR_FTL_SECTOR_SIZE, NOR_ftl.Get_memory_size());
// Временный буфер для логирования
static uint8_t LOG_BUF[256*2];

__weak void Shadow_settings_manage(const TSetting_header* header, const uint8_t* payload)
{
  return;
}

/* C++ code-------------------------------------------------------------------*/
CSyscom_manager::CSyscom_manager(const char* mngr_name, QueueHandle_t* output_queue, uint8_t input_queue_size, uint8_t priority, TaskFunction_t task_starter, uint16_t stack_size) :
  bb_timer("bb_timer", 0),
  sensor_translation_timer("sens_tr", 0),
  save_settings_timer("set_save", 0),
  delayed_sleep_timer("del_sleep", 0),
  save_counters_timer("cnt_save", 0),
  CManager(mngr_name, output_queue, input_queue_size, priority, task_starter, stack_size)
{
  // Связывание с входящими очередями всех менеджеров
  this->gps_queue = &GPS_manager.input_queue;
  this->server_queue = &SERVER_manager.input_queue;
  this->signal_queue = &Signal_manager.input_queue;
  this->server_answer_queue = &SERVER_manager.answer_queue;
  this->name = mngr_name;
  this->need_restart = 0;
  this->need_tofactory = 0;
  this->need_bb_clean = 0;
  this->need_save_settings = 0;
  this->need_emergency_stop = 0;
  // Очистка структуры слежения за сценариями
  memset(&this->scenario_state, 0x00, sizeof(this->scenario_state));
  // Очистка структуры слежения за таймерными выходами
  memset(&this->scenario_timer_outputs, 0x00, sizeof(this->scenario_timer_outputs));
}

void CSyscom_manager::Init (void)
{
  // Защита
  if(System.Total_sensor_count > MAX_SENSOR_COUNT) while(1);

  // Сбрасываем в значение по умолчанию
  System.blackbox_settings.disable_priority_messages = 0;

  // Загрузка настроек и инициализация черного ящика
  Load_settings();

  // Запрет приоритетной выгрузки не зависимо от настройки
  if(true || System.blackbox_settings.disable_priority_messages == 1)
  {
    Black_box.Disable_priority_messages();
  }

  // Инициализация защитных таймеров для сценариев
  scenario_last_photo = GetUnixTime() - 10;
  scenario_last_sms = scenario_last_photo;
  scenario_last_call = scenario_last_sms;

  // Установка приоритета задаче
  vTaskPrioritySet(this->task_handle, tskIDLE_PRIORITY + this->priority);

  sleep_by_ignition_timer = Soft_timers.seconds_from_startup;
  sleep_by_stop_timer = Soft_timers.seconds_from_startup;
  sleep_by_wakeup_timer = Soft_timers.seconds_from_startup;
}

// Деинициализация
void CSyscom_manager::Deinit (void)
{

}

void CSyscom_manager::NOR_FTL_diag(void)
{
  uint32_t timer;
  uint16_t addr_dublication, not_found_addr;
  uint8_t range_ok;
  union
  {
    BBresult bb_res;
    bool check_res;
  };

  const lat_recovery_inf_t* const lat_recovery_inf=NOR_ftl.Get_Recovery_Inf();

  if(lat_recovery_inf->event_flag)
  {
    LOG("%s: LAT recovery status: %u, corrupt: %u, sector: %u\n", this->name, lat_recovery_inf->status, lat_recovery_inf->with_corrupt, lat_recovery_inf->sector);
  }

  Black_box.Grab();
  timer=xTaskGetTickCount();
  check_res=NOR_ftl.Check_LAT(&addr_dublication, &not_found_addr, &range_ok);
  timer=xTaskGetTickCount()-timer;
  Black_box.Release();
  LOG("%s: LAT test: %s, addr_dublication: %u, not_found_addr: %u, range: %s (%u ms)\n", this->name, (check_res)?("pass"):("fail"), addr_dublication, not_found_addr, (range_ok)?("ok"):("err"), timer);

  return;
}

static void Update_flash_condition(TickType_t SE_max_time)
{
  System.syscom_state.flash_se_max_time = SE_max_time;
  System.syscom_state.wear_condition = (float)SE_max_time / SectorEraseCycleTime; // + System.syscom_state.battery_capacity / max_capacity;
}

// Главный цикл задачи системного менеджера
void CSyscom_manager::Main_cycle(void)
{
  NOR_FTL_diag();

  // Вывод лога в отладочный порт при старте, использовать только при отладке, иначе хана девайсу
  //Logout();
  memset((uint8_t*)&output_message, 0x00, sizeof(output_message));
  output_message.code = COMMAND_WAKEUP;
  this->Send_message(this->gps_queue);
  this->Send_message(this->server_queue);
  this->Send_message(this->signal_queue);
  // Если обнаружен первый старт, даем задержку, чтобы менеджеры его увидели
  if(System.first_start)
  {
    vTaskDelay(1000);
    System.first_start = 0;
  }
  // Задержка, чтобы сигнал гарантированно успел
  vTaskDelay(200);
  for(;;)
  {
    // Получение сообщений от менеджеров
    Receive_message();
    // Трансляция показаний датчиков
    Translate_sensor_data();
    // Обработчик сценариев
    Handle_scenaries();
    // Если изменились настройки и их пора сохранять
    Handle_save_settings();
    // Заполнение текущего состояния системы
    Fill_system_state();
    // Отработка спящего режима
    Handle_sleep_mode();
    // Регулярное сохранение счетчиков (пробег, траффик, счетчик поездок и т.д.)
    Save_counters();
    // Проверка таймера моргания поворотниками
    Check_blink();
    // Функции регулярного статического выравнивания износа памяти
    if(bb_timer.Is_empty())
    {
      bb_timer.Start(SEC * 10);
      Black_box.Grab();
      // Статическое выравнивание износа
      NOR_ftl.Static_wear_leveller();
      Black_box.Release();
    }
    // Рестарт по команде
    if(this->need_restart)
    {
      vTaskDelay(2000);
      Black_box.Grab();
      NVIC_SystemReset();
    }
    // Сброс к заводским настройкам по команде
    if(this->need_tofactory)
    {
      vTaskDelay(2000);
      Black_box.Grab();
      RTC_WriteBackupRegister(RTC_BKP_DR7, 0x01);
      NOR_ftl.Format_memory();
      NVIC_SystemReset();
    }
    // Очистка ЧЯ по команде
    if(this->need_bb_clean)
    {
      vTaskDelay(2000);
      Black_box.Grab();
      Black_box.Clean();
      NVIC_SystemReset();
    }
    // Остановка машины по команде с сервера
    if(this->need_emergency_stop)
    {
      static NTime::TTimer timer;
      const NTime::TTime timeout = NTime::TTime::Seconds(30);

      if(timer.IsStarted())
      {
        if(System.can_state.numeric_data.can_speed > 0)
        {
          timer.Stop();
        }
        else if(timer.HasExpired(timeout))
        {
          if(!uxQueueMessagesWaiting(Signal_manager.can_commands_queue))
          {
            if(CAN_SUBTASK_COMMAND_START_PROCCES==signalman_can_subtask_commnad_handle(YADRIVE_FORCED_END_OF_LEASE_SUBCODE, 0xff, 0, NULL, NULL))
            {
              LOG("WARNING!!! emergency stop!!!\n");
              this->need_emergency_stop = false;
              timer.Stop();
            }
          }
        }
      }
      else
      {
        if(System.can_state.numeric_data.can_speed == 0)
        {
          timer.Start();
        }
      }
    }

    // Контроль черного ящика на наличие ошибок памяти
    if(Black_box.get_mem_read_err_counter() > 6)
    {
      LOG("%s: Black_box read error count > 6, cleaning black box\n", this->name);
      LOG("%s: Restart due to black box clean\n", this->name);
      Black_box.Grab();
      Black_box.Clean();
      NVIC_SystemReset();
    }
    if(Black_box.get_mem_write_err_counter() > 6)
    {
      LOG("%s: Black_box write error count > 6, cleaning black box\n", this->name);
      LOG("%s: Restart due to black box clean\n", this->name);
      Black_box.Grab();
      Black_box.Clean();
      NVIC_SystemReset();
    }

    if (was_update_firmware) {
      static uint32_t start = xTaskGetTickCount();
      if (xTaskGetTickCount() - start >= 20000) {
        do_setup_after_upload(*this);

        was_update_firmware = false;
      }
    }
  }
}

bool CSyscom_manager::Start_blink()
{
  if (need_blink) {
    __PRINTF("WARNING: Blink is process\n");
    return true;
  }

  __PRINTF("Start blink\n");

  need_blink = true;
  blink_time_timer = xTaskGetTickCount();
  blink_sampling_timer = xTaskGetTickCount();

  // blink first time
  return Do_blink();
}

void CSyscom_manager::Check_blink()
{
  if (!need_blink) {
    return;
  }

  if (xTaskGetTickCount() - blink_time_timer > blink_time_timeout) {
    Clear_blink();
    return;
  }

  if (xTaskGetTickCount() - blink_sampling_timer > blink_sampling_timeout) {
    Do_blink();
    blink_sampling_timer = xTaskGetTickCount();
  }
}

void CSyscom_manager::Clear_blink()
{
  need_blink = false;
  blink_time_timer = xTaskGetTickCount();
  blink_sampling_timer = xTaskGetTickCount();
  __PRINTF("Complete blink\n");
}

bool CSyscom_manager::Do_blink()
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = CAN_BLINKER_FLASHING_SUBCODE;
  __PRINTF("Do blink\n");
  return this->Send_message(this->signal_queue, 2000);
}

// Заполнение текущего состояния системы
void CSyscom_manager::Fill_system_state(void)
{
  Black_box.Grab();
  System.syscom_state.message_count_in_bb[0] = Black_box.get_message_count(1);
  System.syscom_state.message_count_in_bb[1] = Black_box.get_message_count(2);
  System.syscom_state.message_count_in_bb[2] = Black_box.get_message_count(3);
  System.syscom_state.message_count_in_bb[3] = Black_box.get_message_count(4);
  // Подсчет моточасов
  if(!(System.signal_state.motohours == System.signal_state.motohours)) System.signal_state.motohours = 0;
  if(System.signal_state.ignition)
  {
    float motohours_delta = ((float)(Soft_timers.seconds_from_startup - System.syscom_state.uptime))/3600;
    System.signal_state.motohours += motohours_delta;
  }
  System.syscom_state.uptime = Soft_timers.seconds_from_startup;
  System.syscom_state.current_time = GetUnixTime();
  Black_box.Release();
}

// Создать файл с настройками
void CSyscom_manager::Create_base_file(void)
{
  TSetting_header header;
  uint16_t i = 0;
  uint8_t *setting_ptr;
  uint16_t buff_ptr = 0;
  Black_box.Grab();
  Black_box.fopen("BASE");
  Black_box.fflush("BASE");
  Black_box.Release();
  while(header.size = System.Get_setting(i, &header.id, &header.subid, &setting_ptr))
  {
    if(header.size + sizeof(header) > sizeof(logout_buff) - buff_ptr)
    {
      Black_box.Grab();
      Black_box.fwrite("BASE", logout_buff, buff_ptr);
      Black_box.Release();
      buff_ptr = 0;
    }
    memcpy(logout_buff + buff_ptr, (uint8_t*)&header, sizeof(header));
    buff_ptr += sizeof(header);
    memcpy(logout_buff + buff_ptr, setting_ptr, header.size);
    buff_ptr += header.size;
    i++;
    // Если на следующем круге настройки кончатся, дописываем что есть
    if(!System.Get_setting(i, &header.id, &header.subid, &setting_ptr))
    {
      Black_box.Grab();
      Black_box.fwrite("BASE", logout_buff, buff_ptr);
      Black_box.Release();
      buff_ptr = 0;
    }
  }
  Black_box.Grab();
  Black_box.fsave();
  Black_box.fclose("BASE");
  Black_box.Release();
}

// Обработка сохранения настроек
void CSyscom_manager::Handle_save_settings(void)
{
  // Если изменились настройки и их пора сохранять
  if(this->need_save_settings && save_settings_timer.Is_empty())
  {
    Create_base_file();
    RTC_WriteBackupRegister(RTC_BKP_DR8, 0x00);
    need_update_backup_settings_file();
    this->need_save_settings = 0;
    this->output_message.code = COMMAND_UPDATE_SETTINGS;
    this->Send_message(this->signal_queue, 2000);
    this->Send_message(this->gps_queue, 2000);
  }
}

void CSyscom_manager::Handle_sleep_mode(void)
{
  // Проверка напряжения питания
  #ifdef INTERNAL_AKB_PRESENT
  System.Grab();
  if(System.signal_state.internal_acc_voltage < INTERNAL_ACC_MINIMAL_VOLTAGE && System.signal_state.is_on && System.signal_state.power_source == ACC_POWER)
  {
    System.Release();
    LOG("%s: Critical ACC discharge, emergency sleep\n", this->name);
    go_to_sleep(EMERGENCY_SLEEP);
    return;
  }
  System.Release();
  #endif
  if(!System.power_settings.sleep_by_ignition && !System.power_settings.sleep_by_stop && !System.power_settings.sleep_by_wakeup)
  {
    System.syscom_state.operation_mode = 1;
    return;
  }
  // Загрузка таймеров, если есть зажигание или движение
  if(System.power_settings.sleep_by_ignition && System.signal_state.ignition) sleep_by_ignition_timer = Soft_timers.seconds_from_startup;
  if(System.power_settings.sleep_by_stop && (System.gnss_state.in_move || System.signal_state.gsensor_move_sensor)) sleep_by_stop_timer = Soft_timers.seconds_from_startup;
  // Если проблемы с акселерометром, то не засыпать по остановке
  if(System.signal_state.accel_no_init) sleep_by_stop_timer = Soft_timers.seconds_from_startup;

  bool ready_to_sleep_by_ign = ((Soft_timers.seconds_from_startup - sleep_by_ignition_timer) >= 60*System.power_settings.sleep_by_ignition);
  bool ready_to_sleep_by_stop = ((Soft_timers.seconds_from_startup - sleep_by_stop_timer) >= 60*System.power_settings.sleep_by_stop);
  bool ready_to_sleep_by_wakeup = ((Soft_timers.seconds_from_startup - sleep_by_wakeup_timer) >= 60*System.power_settings.sleep_by_wakeup);
  bool ready_to_sleep = false;
  if( System.power_settings.slep_and_case && (ready_to_sleep_by_ign && ready_to_sleep_by_stop && ready_to_sleep_by_wakeup) ) ready_to_sleep = true;
  if(!System.power_settings.slep_and_case && (\
     (System.power_settings.sleep_by_ignition && ready_to_sleep_by_ign)   ||\
     (System.power_settings.sleep_by_stop && ready_to_sleep_by_stop)      ||\
     (System.power_settings.sleep_by_wakeup && ready_to_sleep_by_wakeup)  )) ready_to_sleep = true;

  if(System.server_state.usb_connected) ready_to_sleep = false;

  #if defined(TAMPERS_PRESENT)
  // Не спим, если сработал один из датчиков вскрытия
  for(uint8_t i=0; i<MAX_TAMPERS; i++)
  {
    if(System.signal_state.tamper[i])
    {
      ready_to_sleep = false;
      break;
    }
  }
  #endif //TAMPERS_PRESENT

#if defined(CANPRO_PRESENT)
  // Не засыпаем если стоит пробуждение по активности can и can не спит
   if(System.wakeup_by_can_activity && !System.can_state.sec_flags.can_in_sleep) ready_to_sleep = false;
#endif //CANPRO_PRESENT

  if(!ready_to_sleep)
  {
    delayed_sleep_timer.Start(SEC*30);
    System.syscom_state.operation_mode = 1;
  }

  if(delayed_sleep_timer.Get_time_left(SEC) < 29)
  {
    System.syscom_state.operation_mode = 0;
  }
  // Засыпаем только если есть возможность проснуться
  if(this->delayed_sleep_timer.Is_empty() && (System.power_settings.wakeup_by_ignition || (System.power_settings.wakeup_by_move && !System.signal_state.accel_no_init) || System.power_settings.wakeup_by_time
#if defined(CANPRO_PRESENT)
     || System.wakeup_by_can_activity
#endif //CANPRO_PRESENT
       )) go_to_sleep(NORMAL_SLEEP);

}

// Переход в спящий режим
void CSyscom_manager::go_to_sleep(uint8_t sleep_type)
{
  output_message.code = COMMAND_SLEEP;
  output_message.subcode = sleep_type;
  this->Send_message(this->gps_queue);
  this->Send_message(this->server_queue);
  // Ожидание, пока менеджеры уснут
  while(System.gnss_state.is_on || System.server_state.is_on)
  {
    Receive_message();
  }
  LOG("%s: Enter sleep mode\n", this->name);
  output_message.code = COMMAND_SLEEP;
  output_message.subcode = sleep_type;
  this->Send_message(this->signal_queue);
  while(1)
  {
    // Ожидание, пока нас не разбудит сигнальный менеджер
    this->Wait_message(portMAX_DELAY);
    if(this->input_message.code == COMMAND_WAKEUP) break;
  }
  float odometer_bkp=System.gnss_state.odometer; uint32_t trip_counter_bkp=System.gnss_state.trip_counter;
  memset(&System.gnss_state, 0x00, sizeof(System.gnss_state));
  System.gnss_state.odometer=odometer_bkp; System.gnss_state.trip_counter=trip_counter_bkp;
  LOG("%s: Wakeup\n", this->name);
  // Дадим проснуться и проинициализироваться
  sleep_by_ignition_timer = Soft_timers.seconds_from_startup;
  sleep_by_stop_timer = Soft_timers.seconds_from_startup;
  sleep_by_wakeup_timer = Soft_timers.seconds_from_startup;
  output_message.code = COMMAND_WAKEUP;
  this->Send_message(this->gps_queue);
  this->Send_message(this->server_queue);
  vTaskDelay(500);
  System.syscom_state.operation_mode = 1;
}

// Получение и разбор сообщений извне
void CSyscom_manager::Receive_message(void)
{
  // Проверяем наличие входящих сообщений в очереди
  if(this->Wait_message(50))
  {
    switch(this->input_message.source)
    {
      case GPS_MANAGER: this->Handle_gps_messages(); break;
      case SERVER_MANAGER:  this->Handle_server_messages(); break;
      case SIGNAL_MANAGER:  this->Handle_signal_messages(); break;
      default: break;
    }
  }
}

// Обработка сообщений от менеджера сигналов
void CSyscom_manager::Handle_signal_messages(void)
{

}

// Обработка сообщений от сервера
void CSyscom_manager::Handle_server_messages(void)
{
  switch (this->input_message.code)
  {
    // Получены данные для обработки и ответа
    case MESSAGE_SERVER_RECV_DATA:
    {
      this->output_message.source_id = this->input_message.source_id;
      uint16_t answer_size = this->Handle_vega_messages((uint8_t*)this->input_message.pPayload, (uint8_t*)this->input_message.pAnswer, this->input_message.answer_size);
      if(answer_size)
      {
        this->output_message.code = COMMAND_SERVER_DATA_ANSWER_OK;
        this->output_message.answer_size = answer_size;
      }
      else
      {
        this->output_message.code = COMMAND_SERVER_DATA_ANSWER_ERR;
        this->output_message.answer_size = 0;
      }
      this->Send_message(this->server_answer_queue);
      break;
    }
    // Обнаружена неисправность
    case MESSAGE_SERVER_CHECK:
    {
      break;
    }
    // Сброшена неисправность
    case MESSAGE_SERVER_RESTORE:
    {
      break;
    }
    case MESSAGE_SERVER_FILE_DOWNLOADED:
    {
      const char* filename = (char*)this->input_message.pPayload;
      if(strncmp(filename, "FIRMWARE", strlen("FIRMWARE")) == 0)
      {
        uint32_t fsize;
        Black_box.Grab();
        Black_box.fsize("FIRMWARE", &fsize);
        Black_box.Release();
        if(fsize <= BOOT_FILE_HEADER_LEN+(APP_START_ADDRESS-FLASH_BASE))
        {
          LOG("%s: New bootloader received (maybe)\n", this->name);
          vTaskDelay(5000);
          go_to_sleep(BOOT_UPD_QUASI_SLEEP);
        }
        else
        {
          Black_box.Grab();
          Black_box.fflush("LOG");
          Black_box.fsave();
          Black_box.Release();
          LOG("%s: New FW received, waiting managers\n", this->name);
          LOG("%s: Current FW version %s from %s %s\n", this->name, APLICATION_VERSION_STRING, __DATE__, __TIME__);
          this->boot_cmd.cmd = UPGRADE_FW_CMD;
          memcpy(this->boot_cmd.fname, "FIRMWARE", sizeof("FIRMWARE"));
          Black_box.Grab();
          Black_box.Save_base("boot_cmd");
          Black_box.Release();
          // Задержка на передачу серверу информации о наличии новой прошивки
          vTaskDelay(10000);
          // Отключение GPS и GSM менеджеров
          output_message.code = COMMAND_SLEEP;
          this->Send_message(this->gps_queue);
          this->Send_message(this->server_queue);
          // Ожидание, пока менеджеры уснут

          uint32_t end_wait_time=xTaskGetTickCount()+3*60000;
          while(System.gnss_state.is_on || System.server_state.is_on)
          {
            // Защита, если кто-то не засыпает
            if(timeAfter(xTaskGetTickCount(), end_wait_time)) break;

            if(this->Wait_message(50))
            {
              if(this->input_message.code == MESSAGE_SERVER_RECV_DATA)
              {
                // Это сообщение требует обязательного ответа, иначе сервер/поток конфигуратора будет в постоянном ожидании. Отвечаем ошибкой
                this->output_message.code = COMMAND_SERVER_DATA_ANSWER_ERR;
                this->output_message.answer_size = 0;
                this->Send_message(this->server_answer_queue);
              }
            }
          }
          LOG("%s: Restart...\n", this->name);
          Black_box.Grab();
          // Рестарт процессора
          NVIC_SystemReset();
        }
      }
      // Пришел файл с настройками
      else if(strncmp(filename, "SETTINGS", strlen("SETTINGS")) == 0)
      {
        LOG("%s: New SETTINGS received\n", this->name);
        if(Parse_settings("SETTINGS", 1))
        {
          Create_base_file();
          RTC_WriteBackupRegister(RTC_BKP_DR8, 0x00);
          need_update_backup_settings_file();
          Black_box.Grab();
          Black_box.fopen("SETTINGS");
          Black_box.fflush("SETTINGS");
          Black_box.fsave();
          Black_box.fclose("SETTINGS");
          Black_box.Release();
          this->output_message.code = COMMAND_UPDATE_SETTINGS;
          this->Send_message(this->server_queue, 2000);
          this->Send_message(this->signal_queue, 2000);
          this->Send_message(this->gps_queue, 2000);
        }
      }
      // Пришел файл с прошивкой can или с прошивкой gnss
      else if(strncmp(filename, "CANPRO", strlen("CANPRO")) == 0)
      {
        static const char* const fname="CANPRO";

        uint8_t file_type=0xFF;

        Black_box.Grab();

        uint8_t is_opened=1;

        if(Black_box.fopened(fname, &is_opened) == BB_OPERATION_OK && !is_opened)
        {
          if(Black_box.fopen(fname) == BB_OPERATION_OK)
          {
            if(Black_box.fread(fname, 0, LOG_BUF, 8))
            {
              Black_box.fclose(fname);

              if(memcmp(LOG_BUF, "GNSS_FW", sizeof("GNSS_FW"))==0)      {file_type=1;}
              else if(memcmp(LOG_BUF, "NRF_BLE", sizeof("NRF_BLE"))==0) {file_type=2;}
              else                                                      {file_type=0;}
            }
          }
        }

        Black_box.Release();

        if(file_type==0)
        {
          LOG("%s: New CANPRO file received\n", this->name);
          this->output_message.code = COMMAND_UPDATE_CAN_FIMWARE;
          this->Send_message(this->signal_queue, 2000);
        }
        else if(file_type==1)
        {
          LOG("%s: New GNSS FW file received\n", this->name);
          this->output_message.code = COMMAND_UPDATE_GNSS_FIMWARE;
          this->Send_message(this->gps_queue, 2000);
        }
        else if(file_type==2)
        {
#if defined(EXTERNAL_BLE_BOARD_PRESENT)
          LOG("%s: New NRF BLE FW file received\n", this->name);
          external_ble_fw_update_task(fname);
#else
          LOG("%s: New NRF BLE FW file received, but EXTERNAL_BLE_BOARD_PRESENT is not defined\n", this->name);
#endif //defined(EXTERNAL_BLE_BOARD_PRESENT)
        }
        else
        {
          LOG("%s: received file fs error\n", this->name);
        }
      }
      break;
    }
    case MESSAGE_SERVER_SAVE_BASE:
    {
      Create_base_file();
      if(this->input_message.subcode) //если нужно сохранить backup
      {
        RTC_WriteBackupRegister(RTC_BKP_DR8, 0x00);
        need_update_backup_settings_file();
      }
      break;
    }
    case MESSAGE_SERVER_TAKE_PHOTO:
    {
      LOG("Command from server take a photo\n");
      this->output_message.code = COMMAND_SIGNAL_PHOTO_GET;
      this->Send_message(this->signal_queue, 2000);
      break;
    }
    case MESSAGE_SERVER_REBOOT_GNSS:
    {
      LOG("Command from server reboot gnss\n");
      this->output_message.code = COMMAND_GNSS_HW_COLD_START;
      this->Send_message(this->gps_queue, 2000);
      break;
    }
    case MESSAGE_SERVER_CAN_CONTROL:
    {
      LOG("%s: Command from server can control, subcode=%hhu\n", this->name, this->input_message.subcode);
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = this->input_message.subcode;
      this->output_message.ext_args = this->input_message.ext_args;

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      this->output_message.command_uid = this->input_message.command_uid;
      this->output_message.source_id = this->input_message.source_id;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)

      this->Send_message(this->signal_queue, 2000);
      break;
    }
    default: break;
  }
}

// Парсинг файла с настройками
bool CSyscom_manager::Parse_settings(const char* filename, const uint8_t w_sing)
{
  Black_box.Grab();
  Black_box.fopen(filename);
  uint8_t *ptr;
  TSetting_header header;
  uint32_t filesize = 0;
  if(!Black_box.fsize(filename, &filesize))
  {
    Black_box.fclose(filename);
    Black_box.Release();
    LOG("%s: Error filesize %s\n", this->name, filename);
    return false;
  }

  uint32_t roffset = 0;

  //если файл с подписью
  if(w_sing)
  {
    tea_context tea_ctx;

    if(filesize<24)
    {
      Black_box.fclose(filename);
      Black_box.Release();
      LOG("%s: Error filesize %s\n", this->name, filename);
      return false;
    }

    tea_certify_init(&tea_ctx, __get_vega_signat_key());

    roffset = sizeof(tea_ctx.cert);

    while(roffset < filesize)
    {
      uint16_t r_len=sizeof(logout_buff);
      if(roffset+r_len>filesize) r_len=filesize-roffset;

      Black_box.fread(filename, roffset, logout_buff, r_len);
      tea_certify_update(&tea_ctx, logout_buff, r_len);

      roffset+=r_len;
    }
    tea_certify_finish(&tea_ctx);

    memset(logout_buff, 0, sizeof(tea_ctx.cert));
    Black_box.fread(filename, 0, logout_buff, sizeof(tea_ctx.cert));

    if(memcmp(tea_ctx.cert, logout_buff, sizeof(tea_ctx.cert))!=0)
    {
      Black_box.fclose(filename);
      Black_box.Release();
      LOG("%s: Wrong file %s cert\n", this->name, filename);
      return false;
    }

    roffset = 24;
  }

  uint16_t counters[4]={0};

  while(roffset < filesize)
  {
    Black_box.fread(filename, roffset, (uint8_t*)&header, sizeof(header));
    if(header.size > sizeof(logout_buff))
    {
      Black_box.Release();
      LOG("%s: Error setting %d too big to parse\n", this->name, header.id);
      Black_box.Grab();
    }
    else
    {
      uint16_t setting_size = System.Find_setting(header.id, header.subid, &ptr);

      if(!System.Is_setting_id(header.id))
      {
        //это вообще не настройка, Петя гандон
        roffset += (sizeof(header) + header.size);

        continue;
      }

      if(header.id==VEGA_SETTINGS_INFO && memcmp("SETTINGS", filename, sizeof("SETTINGS"))==0)
      {
        //этой настройки не должно быть в файле полученном из вне
        roffset += (sizeof(header) + header.size);

        continue;
      }

#if defined(LOCK_SETTINGS_PRESENT)
      if((header.id == VEGA_SETTING_LOCK_FW_SETTINGS || (!Is_fw_and_setting_change_allowed() && Is_setting_in_locked_list(header.id))) &&
         memcmp("SETTINGS", filename, sizeof("SETTINGS"))==0)
      {
        //этой настройки не должно быть в файле полученном из вне
        roffset += (sizeof(header) + header.size);

        continue;
      }
#endif //defined(LOCK_SETTINGS_PRESENT)

      counters[0]++;

      if(setting_size && setting_size>header.size)
      {
        memset(ptr, 0, setting_size);
        Black_box.Release();
        __PRINTF("%s: Warning, setting %hu recv size: %hu, while real_size is: %hu\n", this->name, header.id, header.size, setting_size);
        Black_box.Grab();

        counters[1]++;
      }
      else if(!setting_size || setting_size<header.size)
      {
        Black_box.Release();
        if(setting_size == 0) {__PRINTF("%s: Setting id %hu subid %hu not found\n", this->name, header.id, header.subid);}
        else                  {__PRINTF("%s: Error setting %hu recv size: %hu, while real_size is: %hu\n", this->name, header.id, header.size, setting_size);}
        Black_box.Grab();
        roffset += (sizeof(header) + header.size);

        if(setting_size == 0) counters[2]++;
        else                  counters[3]++;

        continue;
      }

      Black_box.fread(filename, roffset + sizeof(header), ptr, header.size);

      //__BIN_BUFF_PRINTF(ptr, header.size, "id: %u subid: %u, size: %u setting: ", header.id, header.subid, header.size);

      //если настройка строковая, то добавляем нуль (защита от передачи не нультерминированных настроек)
      //если настройка в виде структуры, то приходится делать приведение
      if(header.id==VEGA_SETTING_APN)
      {
        TGSM_APN* cfg=(TGSM_APN*)ptr;
        cfg->APN[sizeof(cfg->APN)-1]='\0';
        cfg->user[sizeof(cfg->user)-1]='\0';
        cfg->password[sizeof(cfg->password)-1]='\0';
      }
      else if(header.id==VEGA_SETTING_SERVER_ADDR)
      {
        TGSM_server* cfg=(TGSM_server*)ptr;
        cfg->address[sizeof(cfg->address)-1]='\0';
      }
      //если одиночные строковые настройки
      else if(header.id==VEGA_SETTING_SERVER_PIN || header.id==VEGA_SETTING_PIN_CODE)
      {
        ptr[setting_size-1]='\0';
      }
      else
      {
        Shadow_settings_manage(&header, ptr);
      }
    }
    roffset += (sizeof(header) + header.size);
  }
  Black_box.fclose(filename);
  Black_box.Release();

#if defined(CUSTOM_CAN_SENSORS_PRESENT)
  System.DisableTranslateSettingForEmtyCanSensors();
#endif //defined(CUSTOM_CAN_SENSORS_PRESENT)

  LOG("%s: total settings: %u, warn size: %hu, not found: %hu, bad size: %hu\n", this->name,
      counters[0], counters[1], counters[2], counters[3]);

  return true;
}

bool CSyscom_manager::Is_track_point_present(uint8_t server_idx)
{
  if(server_idx>=MAX_SERVERS_COUNT) return false;

  if(System.connection_settings.server[server_idx].server_protocol && (System.connection_settings.server[server_idx].server_protocol != NDTP) && (System.connection_settings.server[server_idx].server_protocol != TACHO_ARM)
#if (!defined(TRACK_POINT_GEN_FOR_VEGA_PROTOCOL_PRESENT))
     && (System.connection_settings.server[server_idx].server_protocol != VEGA)
#endif //(!defined(TRACK_POINT_GEN_FOR_VEGA_PROTOCOL_PRESENT))
     )
  {
    return true;
  }
  else
  {
    return false;
  }
}

// Обработка сообщений от GPS
void CSyscom_manager::Handle_gps_messages(void)
{
    switch (this->input_message.code)
  {
    // Определено местоположение
    case MESSAGE_GPS_POSITION_FIXED:
    {
      LOG("%s: Position fixed\n", this->name);
      // Проверка сценария для датчика местоположения
      break;
    }
    case MESSAGE_GPS_POSITION_LOST:
    {
      LOG("%s: Position lost\n", this->name);
      // Проверка сценария для датчика местоположения
      break;
    }
    case MESSAGE_GPS_STOP_FIXED:
    {
      LOG("%s: Stop fixed\n", this->name);
      break;
    }
    case MESSAGE_GPS_MOVE_FIXED:
    {
      LOG("%s: Move fixed, trip counter = %d\n", this->name, System.gnss_state.trip_counter);
      break;
    }
    case MESSAGE_GPS_TRACK_DATA:
    {
      // Упаковка сообщения с координатами
      BB_message track_point;
      memset(&track_point, 0x00, sizeof(track_point));

      // Упаковка сообщения в ЧЯ
      System.Grab();
      track_point.len = Pack_vega_message();
      track_point.payload = this->message_buff;
      System.Release();

      // Установка серверов - получателей сообщения
      if(this->Is_track_point_present(0)) track_point.receiver_1 = 1;
      if(this->Is_track_point_present(1)) track_point.receiver_2 = 1;
      if(this->Is_track_point_present(2)) track_point.receiver_3 = 1;
      if(this->Is_track_point_present(3)) track_point.receiver_4 = 1;

      // Запись сообщения в черный ящик, если
      if(track_point.receiver_1 || track_point.receiver_2 || track_point.receiver_3 || track_point.receiver_4)
      {
        Black_box.Grab();
        Black_box.add_message(&track_point);
        Black_box.Release();
      }

      // Для протокола NDTP сообщения упаковываются отдельно
      memset(&track_point, 0x00, sizeof(track_point));
      if(System.connection_settings.server[0].server_protocol == NDTP) track_point.receiver_1 = 1;
      if(System.connection_settings.server[1].server_protocol == NDTP) track_point.receiver_2 = 1;
      if(System.connection_settings.server[2].server_protocol == NDTP) track_point.receiver_3 = 1;
      if(System.connection_settings.server[3].server_protocol == NDTP) track_point.receiver_4 = 1;
      if(track_point.receiver_1 || track_point.receiver_2 || track_point.receiver_3 || track_point.receiver_4)
      {
        System.Grab();
        track_point.len = Pack_vega_message(0xFFFE);
        track_point.payload = this->message_buff;
        System.Release();

        Black_box.Grab();
        Black_box.add_message(&track_point);
        Black_box.Release();
      }
      break;
    }
    case MESSAGE_GPS_CHECK:
    {
      LOG("%s: GPS hardware error fixed\n", this->name);
      break;
    }
    case MESSAGE_GPS_RESTORE:
    {
      LOG("%s: GPS hardware ok\n", this->name);
      break;
    }
    case MESSAGE_GPS_CANT_FIX:
    {
      LOG("%s: Can`t fix position\n", this->name);
      // Добавление события
      break;
    }
    case MESSAGE_GPS_JAMMED:
    {
      LOG("%s: GPS JAMMED\n", this->name);
      break;
    }
    case MESSAGE_GPS_SAVE_BASE:
    {
      // Запуск отложенного сохранения настроек
      save_settings_timer.Start(SEC*5);
      need_save_settings = 1;
      break;
    }
    default: break;
  }
}

// Трансляция сообщений в ЧЯ
uint16_t CSyscom_manager::Pack_vega_message(uint16_t index)
{
  static struct diag_bb_mess_struct
  {
    uint16_t few_mem_by_track:1;
    uint16_t max_bb_mess_size_by_track:15;
    uint16_t few_mem_by_timer:1;
    uint16_t max_bb_mess_size_by_timer:15;
  }diag_bb_mess={0};

  TVega_bb_message message;
  memset((uint8_t*)&message, 0x00, sizeof(message));
  memset(message_buff, 0x00, sizeof(message_buff));
  message.time = GetUnixTime();
  // Упаковка заголовка с координатами только если они валидны
  if(System.gnss_state.coord_status == VALID)
  {
#if !defined(GPS_MANAGER_MODEL_2_PRESENT)
    // Коорекция внутренних часов
    uint32_t unixtime = conv_time_to_unixtime(&System.gnss_state.utc);
    message.time = unixtime;
    SetUnixTime(unixtime);
#endif //!defined(GPS_MANAGER_MODEL_2_PRESENT)
    message.lat = System.gnss_state.lat;
    message.lon = System.gnss_state.lon;
    message.alt = (int16_t)System.gnss_state.alt;
    message.speed = (uint16_t)System.gnss_state.speed;
    message.direction = (uint16_t)System.gnss_state.direction;
    message.sat_count = System.gnss_state.sat_inuse;
  }
  message.parameters_len = 0;

  //Запись параметров со смещением на заголовок
  uint8_t* buff_ptr = this->message_buff + sizeof(TVega_bb_message);
  // Упаковка сообщений вместе с треком или по таймеру
  if(index == 0xFFFF || index == 0xFFDF)
  {
    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      if((index == 0xFFFF && System.connection_settings.regular_translation[i].translate_with_track != 0) || (index == 0xFFDF && this->sensor_send_timer[i] == 0 && System.connection_settings.regular_translation[i].translate_period))
      {
        // Рестарт таймера, если передача по таймеру
        if(index == 0xFFDF) this->sensor_send_timer[i] = System.connection_settings.regular_translation[i].translate_period;

        void* sensor_data_ptr = System.sensor_table[i].ptr;
        uint8_t size = System.sensor_table[i].size;
        TSensor_data_type data_type = System.sensor_table[i].data_type;
        uint16_t sensor_id = System.sensor_table[i].id;
        if(data_type==STRING && 0==strnlen((const char*)sensor_data_ptr, size)) continue;

        // Если показания очередного датчика не лезут в передающий буфер, значит не повезло
        if((buff_ptr + sizeof(sensor_id) + sizeof(size) + size) >= (this->message_buff + sizeof(this->message_buff)))
        {
          if(index == 0xFFFF)      diag_bb_mess.few_mem_by_track=1;
          else if(index == 0xFFDF) diag_bb_mess.few_mem_by_timer=1;

          break;
        }

        // Упаковка ID датчика
        memcpy(buff_ptr, &sensor_id, sizeof(sensor_id));
        buff_ptr += sizeof(sensor_id);
        // Упаковка длины данных датчика
        memcpy(buff_ptr, &size, sizeof(size));
        buff_ptr += sizeof(size);
        // Упаковка данных датчика
        memcpy(buff_ptr, sensor_data_ptr, size);
        buff_ptr += size;
      }
    }
  }
  // Упаковка сообщений NDTP протокола с фиксированным набором параметров
  else if(index == 0xFFFE)
  {
    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      if(
          System.sensor_table[i].id == VEGA_IGNITION            || \
          System.sensor_table[i].id == VEGA_ALARM_BUTTON        || \
          System.sensor_table[i].id == VEGA_ACC_VOLTAGE         || \
          System.sensor_table[i].id == VEGA_POWER_VOLTAGE       || \
          System.sensor_table[i].id == VEGA_GPS_ODOMETER        || \
          System.sensor_table[i].id == VEGA_PDOP                || \
          System.sensor_table[i].id == VEGA_GSM_SIGNAL_LEVEL    || \
          System.sensor_table[i].id == VEGA_GSENSOR_MOVE_SENSOR || \
          System.sensor_table[i].id == VEGA_INT_TEMP            || \
          System.sensor_table[i].id == CAN_ENGNE_UPTIME         || \
          System.sensor_table[i].id == CAN_ENGINE_UPTIME_P      || \
          System.sensor_table[i].id == CAN_ODOMETER_KM          || \
          System.sensor_table[i].id == CAN_ODOMETER_P           || \
          System.sensor_table[i].id == CAN_CONSUMED_FUEL_L      || \
          System.sensor_table[i].id == CAN_CONSUMED_FUEL_P      || \
          System.sensor_table[i].id == CAN_FUEL_LEVEL_L         || \
          System.sensor_table[i].id == CAN_FUEL_LEVEL_P         || \
          System.sensor_table[i].id == CAN_ENGINE_RPM           || \
          System.sensor_table[i].id == CAN_ENGINE_TEMP          || \
          System.sensor_table[i].id == CAN_SPEED                || \
          System.sensor_table[i].id == CAN_AXLE_1               || \
          System.sensor_table[i].id == CAN_AXLE_2               || \
          System.sensor_table[i].id == CAN_AXLE_3               || \
          System.sensor_table[i].id == CAN_AXLE_4               || \
          System.sensor_table[i].id == CAN_AXLE_5               || \
          System.sensor_table[i].id == CAN_SECURITY_NAVIS_BITFIELD    || \
          System.sensor_table[i].id == CAN_CONTROL_NAVIS_BITFIELD
          /*
          System.sensor_table[i].id == CAN_IN_SLEEP             || \
          System.sensor_table[i].id == CAN_DRIVER_DOOR          || \
          System.sensor_table[i].id == CAN_PASS_DOOR            || \
          System.sensor_table[i].id == CAN_TRUNK                || \
          System.sensor_table[i].id == CAN_HOOD                 || \
          System.sensor_table[i].id == CAN_HAND_BREAK           || \
          System.sensor_table[i].id == CAN_PEDAL_BREAK          || \
          System.sensor_table[i].id == CAN_ENGINE_IS_ON         || \
          */
      )
      {
        void* sensor_data_ptr = System.sensor_table[i].ptr;
        uint8_t size = System.sensor_table[i].size;
        TSensor_data_type data_type = System.sensor_table[i].data_type;
        uint16_t sensor_id = System.sensor_table[i].id;
        if(data_type==STRING && 0==strnlen((const char*)sensor_data_ptr, size)) continue;

        // Если показания очередного датчика не лезут в передающий буфер, значит не повезло
        if((buff_ptr + sizeof(sensor_id) + sizeof(size) + size) >= (this->message_buff + sizeof(this->message_buff))) break;

        // Упаковка ID датчика
        memcpy(buff_ptr, &sensor_id, sizeof(sensor_id));
        buff_ptr += sizeof(sensor_id);
        // Упаковка длины данных датчика
        memcpy(buff_ptr, &size, sizeof(size));
        buff_ptr += sizeof(size);
        // Упаковка данных датчика
        memcpy(buff_ptr, sensor_data_ptr, size);
        buff_ptr += size;
      }
    }
  }
  // Упаковка сообщений, содержащих показания одного произвольного датчика
  else
  {
    void* sensor_data_ptr = System.sensor_table[index].ptr;
    uint8_t size = System.sensor_table[index].size;
    TSensor_data_type data_type = System.sensor_table[index].data_type;
    uint16_t sensor_id = System.sensor_table[index].id;
    if( !(data_type==STRING && 0 == strnlen((const char*)sensor_data_ptr, size)))
    {
      // Упаковка ID датчика
      memcpy(buff_ptr, &sensor_id, sizeof(sensor_id));
      buff_ptr += sizeof(sensor_id);
      // Упаковка длины данных датчика
      memcpy(buff_ptr, &size, sizeof(size));
      buff_ptr += sizeof(size);
      // Упаковка данных датчика
      memcpy(buff_ptr, sensor_data_ptr, size);
      buff_ptr += size;
    }
  }
  // Упаковка заголовка сообщения перед параметрами
  message.parameters_len = ((buff_ptr - this->message_buff) - sizeof(TVega_bb_message));
  memcpy(message_buff, (uint8_t*)&message, sizeof(message));

  if(index == 0xFFFF)
  {
    if(diag_bb_mess.max_bb_mess_size_by_track<(sizeof(message) + message.parameters_len)) diag_bb_mess.max_bb_mess_size_by_track=(sizeof(message) + message.parameters_len);
  }
  else if(index == 0xFFDF)
  {
    if(diag_bb_mess.max_bb_mess_size_by_timer<(sizeof(message) + message.parameters_len)) diag_bb_mess.max_bb_mess_size_by_timer=(sizeof(message) + message.parameters_len);
  }

  return (sizeof(message) + message.parameters_len);
}

uint16_t Pack_vega_message_w_custom_data(uint8_t* message_buff, uint16_t message_buff_size,
                                         const uint16_t sensor_id, const void* data_ptr, const uint8_t data_size)
{
  if(data_size) return 0;

  TVega_bb_message* message=(TVega_bb_message*)message_buff;

  //memset(message_buff, 0, message_buff_size);

  message->time = GetUnixTime();
  // Упаковка заголовка с координатами только если они валидны
  if(System.gnss_state.coord_status == VALID)
  {
#if !defined(GPS_MANAGER_MODEL_2_PRESENT)
    // Коорекция внутренних часов
    uint32_t unixtime = conv_time_to_unixtime(&System.gnss_state.utc);
    message->time = unixtime;
    SetUnixTime(unixtime);
#endif //!defined(GPS_MANAGER_MODEL_2_PRESENT)
    message->lat = System.gnss_state.lat;
    message->lon = System.gnss_state.lon;
    message->alt = (int16_t)System.gnss_state.alt;
    message->speed = (uint16_t)System.gnss_state.speed;
    message->direction = (uint16_t)System.gnss_state.direction;
    message->sat_count = System.gnss_state.sat_inuse;
  }
  message->parameters_len = 0;

  //Запись параметров со смещением на заголовок
  uint8_t* buff_ptr = message_buff + sizeof(TVega_bb_message);

  // Упаковка сообщения
  // Упаковка ID датчика
  memcpy(buff_ptr, &sensor_id, sizeof(sensor_id));
  buff_ptr += sizeof(sensor_id);
  // Упаковка длины данных датчика
  memcpy(buff_ptr, &data_size, sizeof(data_size));
  buff_ptr += sizeof(data_size);
  // Упаковка данных датчика
  if(data_ptr!=NULL) memcpy(buff_ptr, data_ptr, data_size);
  buff_ptr += data_size;
  // Упаковка заголовка сообщения перед параметрами
  message->parameters_len = ((buff_ptr - message_buff) - sizeof(TVega_bb_message));
  //memcpy(message_buff, (uint8_t*)&message, sizeof(message));

  return (sizeof(*message) + message->parameters_len);
}

// Регулярная трансляция показаний датчиков
void CSyscom_manager::Translate_sensor_data(void)
{
  // Трансляция производится кратно одной секунде
  if(sensor_translation_timer.Is_empty())
  {
    sensor_translation_timer.Start(SEC*1);

    bool translate_by_timer=false;
    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      if(this->sensor_send_timer[i]) this->sensor_send_timer[i]--;
      if(!translate_by_timer && this->sensor_send_timer[i] == 0 && System.connection_settings.regular_translation[i].translate_period) translate_by_timer=true;
    }

    // Если есть хотябы один датчик, который нужно передать по времени
    if(translate_by_timer)
    {
      // Упаковка сообщения с координатами
      BB_message track_point;
      memset(&track_point, 0x00, sizeof(track_point));
      System.Grab();
      track_point.len = Pack_vega_message(0xFFDF);
      System.Release();

      // Установка серверов - получателей сообщения
      if(this->Is_track_point_present(0)) track_point.receiver_1 = 1;
      if(this->Is_track_point_present(1)) track_point.receiver_2 = 1;
      if(this->Is_track_point_present(2)) track_point.receiver_3 = 1;
      if(this->Is_track_point_present(3)) track_point.receiver_4 = 1;

      track_point.payload = this->message_buff;
      // Запись сообщения в черный ящик
      if(track_point.receiver_1 || track_point.receiver_2 || track_point.receiver_3 || track_point.receiver_4)
      {
        Black_box.Grab();
        Black_box.add_message(&track_point);
        Black_box.Release();
      }
    }

    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      // Трансляция по изменению показаний
      float current_value = 0;
      float bkp_value = 0;
      float delta = 0;

      System.Grab();
      if(System.sensor_table[i].data_type == UINT8 || System.sensor_table[i].data_type == BOOL) { current_value = *(uint8_t*)System.sensor_table[i].ptr; bkp_value = *(uint8_t*)System.sensor_table[i].ptr_bkp; delta = abs(current_value - bkp_value); }
      else if(System.sensor_table[i].data_type == UINT16) { current_value = *(uint16_t*)System.sensor_table[i].ptr; bkp_value = *(uint16_t*)System.sensor_table[i].ptr_bkp; delta = abs(current_value - bkp_value); }
      else if(System.sensor_table[i].data_type == UINT32) { current_value = *(uint32_t*)System.sensor_table[i].ptr; bkp_value = *(uint32_t*)System.sensor_table[i].ptr_bkp; delta = abs(current_value - bkp_value); }
      else if(System.sensor_table[i].data_type == FLOAT)  { current_value = *(float*)System.sensor_table[i].ptr; bkp_value = *(float*)System.sensor_table[i].ptr_bkp;        delta = abs(current_value - bkp_value); }
      else if(System.sensor_table[i].data_type == STRING) { if(strncmp((const char*)System.sensor_table[i].ptr, (const char*)System.sensor_table[i].ptr_bkp, System.sensor_table[i].size)==0) delta=0; else delta=1;}
      else if(System.sensor_table[i].data_type == UINT64) { if(memcmp(System.sensor_table[i].ptr, System.sensor_table[i].ptr_bkp, System.sensor_table[i].size)==0)                            delta=0; else delta=1;}

      if (System.sensor_table[i].id == CAN_STAT_TRANSMISSION) {
        System.connection_settings.regular_translation[i].translate_with_change = 1;
      }

      if(System.connection_settings.regular_translation[i].translate_with_change && (delta >= System.connection_settings.regular_translation[i].translate_with_change))
      {
        memcpy(System.sensor_table[i].ptr_bkp, System.sensor_table[i].ptr, System.sensor_table[i].size);
        // Упаковка сообщения с координатами
        BB_message track_point;
        memset(&track_point, 0x00, sizeof(track_point));
        //System.Grab();
        track_point.len = Pack_vega_message(i);
        System.Release();

        // Установка серверов - получателей сообщения
        if(this->Is_track_point_present(0)) track_point.receiver_1 = 1;
        if(this->Is_track_point_present(1)) track_point.receiver_2 = 1;
        if(this->Is_track_point_present(2)) track_point.receiver_3 = 1;
        if(this->Is_track_point_present(3)) track_point.receiver_4 = 1;

        track_point.payload = this->message_buff;
        // Запись сообщения в черный ящик
        Black_box.Grab();
        Black_box.add_message(&track_point);
        Black_box.Release();
      }
      else
      {
        System.Release();
      }
    }
  }
}

// Обработка сценариев
void CSyscom_manager::Handle_scenaries(void)
{
  // Проверка истечения вреени работы таймерных выходов и возврат в исходное состояние
  uint64_t current_time = Soft_timers.seconds_from_startup*1000 + Soft_timers.fast_ticks;
  for(uint8_t i = 0; i < MAX_DIGITAL_OUTPUTS; i++)
  {
    if(scenario_timer_outputs.output_switch_time[i] && scenario_timer_outputs.output_switch_time[i] <= current_time)
    {
      System.signal_state.digital_output[i] = scenario_timer_outputs.return_output_to_state[i];
      scenario_timer_outputs.output_switch_time[i] = 0;
    }
  }

  #if defined(BR_PRESENT)
  for(uint8_t i = 0; i < MAX_EXT_DIGITAL_OUTPUTS; i++)
  {
    if(scenario_timer_outputs.ext_output_switch_time[i] && scenario_timer_outputs.ext_output_switch_time[i] <= current_time)
    {
      System.signal_state.ext_digital_output[i] = scenario_timer_outputs.return_ext_output_to_state[i];
      scenario_timer_outputs.ext_output_switch_time[i] = 0;
    }
  }
  #endif //BR_PRESENT
  bool action;
  // Последовательный анализ сценариев
  for(uint16_t i = 0; i < MAX_SCENARIES; i++)
  {
    // Проверка наличия сценария, если он пуст, просто продолжаем
    if(System.scenario_settings.scenario[i].sensor_source_id == 0) continue;

    // Получение данных от первого датчика
    float first_sensor_data = 0;
    float second_sensor_data = 0;
    char const* sensor_name;
    TSensor_data_type data_type;
    void* sensor_data_address;
    action = false;

    // Если датчик не найден в системе, не пытаться выполнять сценарий
    if(System.Find_sensor(System.scenario_settings.scenario[i].sensor_source_id, &sensor_name, &sensor_data_address, &data_type, 0) == 0) continue;
    // Сценарии работают только с числовыми датчиками. Строковые датчики и датчики-массивы не анализируются для экономии оперативной памяти
    if(data_type == UINT8 || data_type == BOOL) {System.Grab(); first_sensor_data = (float)*((uint8_t*)sensor_data_address); System.Release();}
    else if(data_type == UINT16)                {System.Grab(); first_sensor_data = (float)*((uint16_t*)sensor_data_address); System.Release();}
    else if(data_type == UINT32)                {System.Grab(); first_sensor_data = (float)*((uint32_t*)sensor_data_address); System.Release();}
    else if(data_type == FLOAT)                 {System.Grab(); first_sensor_data = *((float*)sensor_data_address); System.Release();}
    else                                        {continue;}

    // Получение данных от второго датчика, если он запрограммирован
    if(System.scenario_settings.scenario[i].and_sensor_source_id)
    {
      // Если датчик не найден в системе, не пытаться выполнять сценарий
      if(System.Find_sensor(System.scenario_settings.scenario[i].and_sensor_source_id, &sensor_name, &sensor_data_address, &data_type, 0) == 0) continue;
      // Сценарии работают только с числовыми датчиками. Строковые датчики и датчики-массивы не анализируются для экономии оперативной памяти
      if(data_type == UINT8 || data_type == BOOL)       {System.Grab(); second_sensor_data = (float)*((uint8_t*)sensor_data_address); System.Release();}
      else if(data_type == UINT16)                      {System.Grab(); second_sensor_data = (float)*((uint16_t*)sensor_data_address); System.Release();}
      else if(data_type == UINT32)                      {System.Grab(); second_sensor_data = (float)*((uint32_t*)sensor_data_address); System.Release();}
      else if(data_type == FLOAT)                       {System.Grab(); second_sensor_data = *((float*)sensor_data_address); System.Release();}
      else                                              {continue;}
    }

    // Проверка совпадения первого условия в сценарии
    if(this->scenario_state[i].first_sensor_last_data != first_sensor_data)
    {
      if(System.scenario_settings.scenario[i].sensor_event == VALUE_LAY_DOWN)
      {
        // Если обнаружен переход через указанную границу вниз, то произошло совпадение условия в сценарии
        if(this->scenario_state[i].first_sensor_last_data >= System.scenario_settings.scenario[i].sensor_value &&\
           first_sensor_data < System.scenario_settings.scenario[i].sensor_value)
        {
          this->scenario_state[i].first_condition_occurs = true;
          action = true;
        }
        // Если показания находятся выше границы, то сбрасываем факт совпадения условия
        if(first_sensor_data >= System.scenario_settings.scenario[i].sensor_value) this->scenario_state[i].first_condition_occurs = false;
      }
      else if(System.scenario_settings.scenario[i].sensor_event == VALUE_RISE_UP)
      {
        // Если обнаружен переход через указанную границу вверх, то произошло совпадение условия в сценарии
        if(this->scenario_state[i].first_sensor_last_data <= System.scenario_settings.scenario[i].sensor_value &&\
           first_sensor_data > System.scenario_settings.scenario[i].sensor_value)
        {
           this->scenario_state[i].first_condition_occurs = true;
           action = true;
        }
        // Если показания находятся ниже границы, то сбрасываем факт совпадения условия
        if(first_sensor_data <= System.scenario_settings.scenario[i].sensor_value) this->scenario_state[i].first_condition_occurs = false;
      }
      else if(System.scenario_settings.scenario[i].sensor_event == VALUE_CHANGED)
      {
        // Если обнаружено изменение показаний, то произошло совпадение условия в сценарии
        this->scenario_state[i].first_condition_occurs = true;
        action = true;
      }
      else if(System.scenario_settings.scenario[i].sensor_event == VALUE_EQUAL)
      {
        // Если обнаружено совпадение с указанной границей, то произошло совпадение условия в сценарии
        if(this->scenario_state[i].first_sensor_last_data != System.scenario_settings.scenario[i].sensor_value &&\
           first_sensor_data == System.scenario_settings.scenario[i].sensor_value)
        {
           this->scenario_state[i].first_condition_occurs = true;
           action = true;
        }
        // Если показания перестали быть равны заданным, то сбрасываем факт совпадения условия
        if(first_sensor_data != System.scenario_settings.scenario[i].sensor_value) this->scenario_state[i].first_condition_occurs = false;
      }
      // Сохранение последних показаний первого датчика
      this->scenario_state[i].first_sensor_last_data = first_sensor_data;
    }


    // Проверка совпадения второго условия в сценарии (при его наличии)
    if(System.scenario_settings.scenario[i].use_and_condition && (this->scenario_state[i].second_sensor_last_data != second_sensor_data))
    {
      if(System.scenario_settings.scenario[i].and_sensor_event == VALUE_LAY_DOWN)
      {
        // Если обнаружен переход через указанную границу вниз, то произошло совпадение условия в сценарии
        if(this->scenario_state[i].second_sensor_last_data >= System.scenario_settings.scenario[i].and_sensor_value &&\
           second_sensor_data < System.scenario_settings.scenario[i].and_sensor_value)
        {
          this->scenario_state[i].second_condition_occurs = true;
          action = true;
        }
        // Если показания находятся выше границы, то сбрасываем факт совпадения условия
        if(second_sensor_data >= System.scenario_settings.scenario[i].and_sensor_value) this->scenario_state[i].second_condition_occurs = false;
      }
      else if(System.scenario_settings.scenario[i].and_sensor_event == VALUE_RISE_UP)
      {
        // Если обнаружен переход через указанную границу вверх, то произошло совпадение условия в сценарии
        if(this->scenario_state[i].second_sensor_last_data <= System.scenario_settings.scenario[i].and_sensor_value &&\
           second_sensor_data > System.scenario_settings.scenario[i].and_sensor_value)
        {
           this->scenario_state[i].second_condition_occurs = true;
           action = true;
        }
        // Если показания находятся ниже границы, то сбрасываем факт совпадения условия
        if(second_sensor_data <= System.scenario_settings.scenario[i].and_sensor_value) this->scenario_state[i].second_condition_occurs = false;
      }
      else if(System.scenario_settings.scenario[i].and_sensor_event == VALUE_CHANGED)
      {
        // Если обнаружено изменение показаний, то произошло совпадение условия в сценарии
        this->scenario_state[i].second_condition_occurs = true;
        action = true;
      }
      else if(System.scenario_settings.scenario[i].and_sensor_event == VALUE_EQUAL)
      {
        // Если обнаружено совпадение с указанной границей, то произошло совпадение условия в сценарии
        if(this->scenario_state[i].second_sensor_last_data != System.scenario_settings.scenario[i].and_sensor_value &&\
           second_sensor_data == System.scenario_settings.scenario[i].and_sensor_value)
        {
           this->scenario_state[i].second_condition_occurs = true;
           action = true;
        }
        // Если показания перестали быть равны заданным, то сбрасываем факт совпадения условия
        if(second_sensor_data != System.scenario_settings.scenario[i].and_sensor_value) this->scenario_state[i].second_condition_occurs = false;
      }
      // Сохранение последних показаний первого датчика
      this->scenario_state[i].second_sensor_last_data = second_sensor_data;
    }

    // Проверка необходимости выполнения сценария
    if(System.scenario_settings.scenario[i].use_and_condition == 0) this->scenario_state[i].second_condition_occurs = true; // Если галочка "И" не установлена, то второе условие считать совпавшим
    if(this->scenario_state[i].first_condition_occurs && this->scenario_state[i].second_condition_occurs && action == true)
    {
      // Сброс признаков совпадения условий
      //this->scenario_state[i].first_condition_occurs = false;
      //this->scenario_state[i].second_condition_occurs = false;
      // Включить цифровой выход
      if(System.scenario_settings.scenario[i].action.dout_on_num && (System.scenario_settings.scenario[i].action.dout_on_num <= MAX_DIGITAL_OUTPUTS))
      {
        System.signal_state.digital_output[System.scenario_settings.scenario[i].action.dout_on_num-1] = 1;
        if(System.scenario_settings.scenario[i].action.dout_on_time)
        {
          this->scenario_timer_outputs.output_switch_time[System.scenario_settings.scenario[i].action.dout_on_num-1] = (current_time + (System.scenario_settings.scenario[i].action.dout_on_time*1000));
          this->scenario_timer_outputs.return_output_to_state[System.scenario_settings.scenario[i].action.dout_on_num-1] = 0;
        }
      }
      // Выключить цифровой выход
      if(System.scenario_settings.scenario[i].action.dout_off_num && (System.scenario_settings.scenario[i].action.dout_off_num <= MAX_DIGITAL_OUTPUTS))
      {
        System.signal_state.digital_output[System.scenario_settings.scenario[i].action.dout_off_num-1] = 0;
        if(System.scenario_settings.scenario[i].action.dout_off_time)
        {
          this->scenario_timer_outputs.output_switch_time[System.scenario_settings.scenario[i].action.dout_off_num-1] = (current_time + (System.scenario_settings.scenario[i].action.dout_off_time*1000));
          this->scenario_timer_outputs.return_output_to_state[System.scenario_settings.scenario[i].action.dout_off_num-1] = 1;
        }
      }
#if defined(BR_PRESENT)
      // Включить внешний цифровой выход
      if(System.scenario_settings.scenario[i].action.ext_dout_on_num && (System.scenario_settings.scenario[i].action.ext_dout_on_num <= MAX_EXT_DIGITAL_OUTPUTS))
      {
        System.signal_state.ext_digital_output[System.scenario_settings.scenario[i].action.ext_dout_on_num-1] = 1;
        if(System.scenario_settings.scenario[i].action.ext_dout_on_time)
        {
          this->scenario_timer_outputs.ext_output_switch_time[System.scenario_settings.scenario[i].action.ext_dout_on_num-1] = (current_time + (System.scenario_settings.scenario[i].action.ext_dout_on_time*1000));
          this->scenario_timer_outputs.return_ext_output_to_state[System.scenario_settings.scenario[i].action.ext_dout_on_num-1] = 0;
        }
      }
      // Выключить внешний цифровой выход
      if(System.scenario_settings.scenario[i].action.ext_dout_off_num && (System.scenario_settings.scenario[i].action.ext_dout_off_num <= MAX_EXT_DIGITAL_OUTPUTS))
      {
        System.signal_state.ext_digital_output[System.scenario_settings.scenario[i].action.ext_dout_off_num-1] = 0;
        if(System.scenario_settings.scenario[i].action.ext_dout_off_time)
        {
          this->scenario_timer_outputs.ext_output_switch_time[System.scenario_settings.scenario[i].action.ext_dout_off_num-1] = (current_time + (System.scenario_settings.scenario[i].action.ext_dout_off_time*1000));
          this->scenario_timer_outputs.return_ext_output_to_state[System.scenario_settings.scenario[i].action.ext_dout_off_num-1] = 1;
        }
      }
#endif //BR_PRESENT
      // Позвонить на авторизованный номер
      if(System.scenario_settings.scenario[i].action.call_to_number)
      {
        bool call_allowed = false;
        uint32_t current_time = GetUnixTime();
        if( (current_time - scenario_last_call) > 10)
        {
          call_allowed = true;
          scenario_last_call = current_time;
        }
        else
        {
          LOG("%s: Try to make call blocked by protection timer\n", this->name);
          call_allowed = false;
        }
        if(call_allowed && System.scenario_settings.scenario[i].action.call_to_number && System.scenario_settings.scenario[i].action.call_to_number <= MAX_AUTORIZED_PHONES)
        {
          LOG("%s: Scenario %d make call to authorized number %d\n", this->name, i, System.scenario_settings.scenario[i].action.call_to_number);
          this->output_message.code = COMMAND_SERVER_CALL_TO_AUTH_NUM;
          this->output_message.subcode = System.scenario_settings.scenario[i].action.call_to_number-1;
          this->Send_message(this->server_queue, 1000);
        }
      }
      // Сделать фото
      if(System.scenario_settings.scenario[i].action.take_photo)
      {
        bool foto_allowed = false;
        uint32_t current_time = GetUnixTime();
        if( (current_time - scenario_last_photo) > 2)
        {
          foto_allowed = true;
          scenario_last_photo = current_time;
        }
        else
        {
          LOG("%s: Try to make photo blocked by protection timer\n", this->name);
          foto_allowed = false;
        }
        if(foto_allowed)
        {
          LOG("%s: Scenario %d take a photo\n", this->name, i);
          this->output_message.code = COMMAND_SIGNAL_PHOTO_GET;
          this->Send_message(this->signal_queue, 2000);
        }
      }
      // Отправить СМС на номер
      if(System.scenario_settings.scenario[i].action.send_sms_to_num)
      {
        bool sms_allowed = false;
        uint32_t current_time = GetUnixTime();
        if( (current_time - scenario_last_sms) > 2)
        {
          sms_allowed = true;
          scenario_last_sms = current_time;
        }
        else
        {
          LOG("%s: Try to send sms blocked by protection timer\n", this->name);
          sms_allowed = false;
        }
        if(sms_allowed && System.scenario_settings.scenario[i].action.send_sms_to_num)
        {
          uint8_t sms_text_size = strnlen((const char*)System.scenario_settings.scenario[i].action.sms_text, sizeof(System.scenario_settings.scenario[i].action.sms_text));
          // Отпраялем смс, только если строка не пустая и нультерминированная
          if(sms_text_size > 0 && sms_text_size < sizeof(System.scenario_settings.scenario[i].action.sms_text) && System.scenario_settings.scenario[i].action.send_sms_to_num <= MAX_AUTORIZED_PHONES)
          {
            LOG("%s: Scenario %d send sms to authorized number %d success\n", this->name, i, System.scenario_settings.scenario[i].action.send_sms_to_num);
            this->output_message.code = COMMAND_SERVER_SEND_SMS;
            this->output_message.subcode = System.scenario_settings.scenario[i].action.send_sms_to_num-1;
            this->output_message.pPayload = (void*)System.scenario_settings.scenario[i].action.sms_text;
            this->output_message.payload_size = sizeof(System.scenario_settings.scenario[i].action.sms_text);
            this->Send_message(this->server_queue, 1000);
          }
          else
          {
            LOG("%s: Scenario %d send sms to authorized number %d ERROR\n", this->name, i, System.scenario_settings.scenario[i].action.send_sms_to_num);
          }
        }
      }

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      this->output_message.command_uid = 0x00;
      this->output_message.source_id = 0xff;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)

      // Моргнуть поворотниками
      if(System.scenario_settings.scenario[i].action.can_blinker_flash)
      {
          LOG("%s: Scenario %d %s success\n", this->name, i, "can blinker flash");
          this->output_message.code = COMMAND_CAN_CONTROL;
          this->output_message.subcode = CAN_BLINKER_FLASHING_SUBCODE;
          this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_open_doors)
      {
         LOG("%s: Scenario %d %s success\n", this->name, i, "can open doors");
         this->output_message.code = COMMAND_CAN_CONTROL;
         this->output_message.subcode = CAN_ALL_DOOR_OPENING_SUBCODE;
         this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_close_doors)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can close doors");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_ALL_DOOR_CLOSING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_open_driver_door)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can open driver door");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_DRIVER_DOOR_OPENING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_horn)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can horn");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_HORN_SIGNALING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_trunk)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can hood");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_TRUNK_DOOR_OPENING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_stop_engine)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can stop engine");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_ENGINE_STOPING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_start_engine)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can start engine");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_ENGINE_STARTING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_start_webasto)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can start webasto");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WEBASTO_STARTING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_stop_webasto)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can stop webasto");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WEBASTO_STOPPING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_door_emulation)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can driver door emulation");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_EMULATION_DRIVER_DOOR_OPENING_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      if(System.scenario_settings.scenario[i].action.can_horn_and_brink)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can horn and brink");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_HORN_W_BLINKER_SIGNALIG_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_closing_3s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows closing 3s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_CLOSING_3S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_closing_7s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows closing 7s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_CLOSING_7S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_closing_11s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows closing 11s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_CLOSING_11S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_closing_15s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows closing 15s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_CLOSING_15S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_closing_19s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows closing 19s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_CLOSING_19S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_closing_23s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows closing 23s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_CLOSING_23S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_closing_29s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows closing 29s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_CLOSING_29S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_opening_3s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows opening 3s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_OPENING_3S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_opening_7s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows opening 7s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_OPENING_7S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_opening_11s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows opening 11s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_OPENING_11S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_opening_15s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows opening 15s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_OPENING_15S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_opening_19s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows opening 19s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_OPENING_19S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_opening_23s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows opening 23s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_OPENING_23S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
      else if(System.scenario_settings.scenario[i].action.can_windows_opening_29s)
      {
        LOG("%s: Scenario %d %s success\n", this->name, i, "can windows opening 29s");
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = CAN_WINDOWS_OPENING_29S_SUBCODE;
        this->Send_message(this->signal_queue, 2000);
      }
    }
  }
}

bool CSyscom_manager::do_start_engine()
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = CAN_ENGINE_STARTING_SUBCODE;
  return this->Send_message(this->signal_queue, 2000);
}

bool CSyscom_manager::do_stop_engine()
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = CAN_ENGINE_STOPING_SUBCODE;
  return this->Send_message(this->signal_queue, 2000);
}

bool CSyscom_manager::do_open_doors()
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = CAN_ALL_DOOR_OPENING_SUBCODE;
  return this->Send_message(this->signal_queue, 2000);
}

bool CSyscom_manager::do_close_doors()
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = CAN_ALL_DOOR_CLOSING_SUBCODE;
  return this->Send_message(this->signal_queue, 2000);
}

bool CSyscom_manager::do_yadrive_warming(uint8_t time_mins)
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = YADRIVE_WARMING_SUBCODE;
  memcpy(&this->output_message.ext_args, &time_mins, sizeof(time_mins));
  return this->Send_message(this->signal_queue, 2000);
}

bool CSyscom_manager::do_off_auto_open_doors()
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = CAN_OFF_AUTO_OPEN_DOORS;
  return this->Send_message(this->signal_queue, 2000);
}

bool CSyscom_manager::do_fix_fuel_view()
{
  this->output_message.code = COMMAND_CAN_CONTROL;
  this->output_message.subcode = CAN_FIX_FUEL_VIEW;
  return this->Send_message(this->signal_queue, 10000);
}

// Обработка сообщений протокола Вега
uint16_t CSyscom_manager::Handle_vega_messages(uint8_t* message, uint8_t* answer_buff, uint16_t answer_buff_size)
{
  // Входящее сообщение
  const TVega_message* input_mess = (const TVega_message*)message;

  // Входящая команда
  union
  {
    const TVega_command* input_comm;
    const TVega_command_w_nonce* input_comm_w_nonce;
  }input_comm;

  input_comm.input_comm = (const TVega_command*)(message + sizeof(TVega_message));

  // Ответ на входящую команду
  TVega_command_ans command_ans;

  // Аргументы ответа
  uint8_t* answer_args = 0;
  uint16_t get_ans_len = 0;

  union
  {
    TVega_get_param_ans_args     get_param_ans_args;
    TVega_get_ext_param_ans_args get_ext_param_ans_args;
  }get_ans_args;

  if(input_mess->type != COMMAND && input_mess->type != COMMAND_W_NONCE)
  {
    return 0;
  }

  EVega_command_code input_comm_code;
  uint8_t input_comm_arg_len;

  if(input_mess->type == COMMAND)
  {
    command_ans.uid = input_comm.input_comm->uid;
    input_comm_code = input_comm.input_comm->code;
    input_comm_arg_len = input_comm.input_comm->arg_len;
  }
  else //if(input_mess->type == COMMAND_W_SINGN)
  {
    command_ans.uid = input_comm.input_comm_w_nonce->uid;
    input_comm_code = input_comm.input_comm_w_nonce->code;
    input_comm_arg_len = input_comm.input_comm_w_nonce->arg_len;
  }

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
  this->output_message.command_uid = command_ans.uid;
  memset(&this->output_message.ext_args, 0, sizeof(this->output_message.ext_args));

  memset(&this->output_message.command_nonce, 0, sizeof(this->output_message.command_nonce));
  if(input_mess->type == COMMAND_W_NONCE)
  {
    memcpy(&this->output_message.command_nonce, input_comm.input_comm_w_nonce->nonce, sizeof(this->output_message.command_nonce));
  }
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)

  const char* cmd_verbose = NULL;

  switch(input_comm_code)
  {
    // Установить параметр
    case SET_PARAM:
    {
      TVega_set_param_args* args = (TVega_set_param_args*)(message + sizeof(TVega_message) + sizeof(TVega_command));
      uint8_t* received_setting_ptr = (message + sizeof(TVega_message) + sizeof(TVega_command) + sizeof(TVega_set_param_args));
      uint8_t* local_setting_ptr = 0;
      // Поиск настроек
      uint16_t setting_len = 0;
      char const* sensor_name;

      if(!System.Is_setting_id(args->id))
      {
        TSensor_data_type data_type;
        setting_len = System.Find_sensor(args->id, &sensor_name, (void**)&local_setting_ptr, &data_type, 0);
      }
      else
      {
        setting_len = System.Find_setting(args->id, args->subid, &local_setting_ptr);

        if(args->id==VEGA_SETTINGS_INFO)
        {
          //эта настройка не может быть установлена из вне
          setting_len = 0;
        }

        if(setting_len == args->len && args->id==VEGA_SETTING_CAN_CUSTOM_SENSORS)
        {
          const TSetting_header sheader = {.id=args->id, .subid=args->subid, .size=args->len};

          System.Grab();
          Shadow_settings_manage(&sheader, received_setting_ptr);
          System.Release();
        }
      }
      // Проверка на соответствие полученного размера тому что в настройках
      if(setting_len == args->len)
      {
        //логируем, если это управление выходами
        if((args->id>=VEGA_DIGITAL_OUTPUT_1 && args->id<=VEGA_DIGITAL_OUTPUT_1+MAX_DIGITAL_OUTPUTS)
#if defined(BR_PRESENT)
          || (args->id>=VEGA_EXT_DIGITAL_OUTPUT_1 && args->id<=VEGA_EXT_DIGITAL_OUTPUT_1+MAX_EXT_DIGITAL_OUTPUTS)
#endif //BR_PRESENT
#if defined(INTERNAL_NRF_PRESENT)
            || (args->id==VEGA_NRF_DESIRED_RELAY_STATE)
#endif //INTERNAL_NRF_PRESENT
              )
        {
          //LOG("%s: Setting %s %hhu->%hhu\n", this->name, sensor_name,  *local_setting_ptr, *received_setting_ptr);
        }

        System.Grab();
        memcpy(local_setting_ptr, received_setting_ptr, setting_len);
        System.Release();

        // Команда выполнена
        command_ans.result = DONE;

        if(System.Is_setting_id(args->id))
        {
          // Запуск отложенного сохранения настроек
          save_settings_timer.Start(SEC*5);
          need_save_settings = 1;
        }
      }
      else
      {
        // Ошибка
        command_ans.result = COMM_ERROR;
      }
      // Аргументы в ответе на команду установить параметр всегда нулевые
      command_ans.arg_len = 0;
      break;
    }
    // Считать параметр
    case GET_PARAM:
      {
      // Структура для заполнения аргументов результата для команды считать параметр
      memset(&get_ans_args.get_param_ans_args, 0x00, sizeof(get_ans_args.get_param_ans_args));

      TVega_get_param_args* args = (TVega_get_param_args*)(message + sizeof(TVega_message) + sizeof(TVega_command));
      get_ans_args.get_param_ans_args.id = args->id;
      // Для чтения показаний датчиков
      if(!System.Is_setting_id(args->id))
      {
        get_ans_args.get_param_ans_args.len = System.Find_sensor(args->id, NULL, (void**)&answer_args, NULL, NULL);
      }
      // Для чтения настроек
      else
      {
        get_ans_args.get_param_ans_args.len = System.Find_setting(args->id, args->subid, (uint8_t**)&answer_args);
      }
      // Если параметр найден успешно, будем его отправлять
      //args->id++; args->id--;
      if(get_ans_args.get_param_ans_args.len)
      {
        get_ans_len=sizeof(get_ans_args.get_param_ans_args);
        command_ans.arg_len =  get_ans_args.get_param_ans_args.len + sizeof(get_ans_args.get_param_ans_args);
        command_ans.result = DONE;
      }
      // Иначе вываливаем ошибку
      else
      {
        command_ans.result = UNKNOWN_PARAMETER;
        command_ans.arg_len = 0;
      }
      break;
    }
  case GET_EXT_PARAM:
    {
      // Структура для заполнения аргументов результата для команды считать параметр
      memset(&get_ans_args.get_ext_param_ans_args, 0, sizeof(get_ans_args.get_ext_param_ans_args));

      TVega_get_param_args* args = (TVega_get_param_args*)(message + sizeof(TVega_message) + sizeof(TVega_command));
      get_ans_args.get_ext_param_ans_args.id = args->id;
      get_ans_args.get_ext_param_ans_args.subid = args->subid;
      // Для чтения показаний датчиков
      if(!System.Is_setting_id(args->id))
      {
        TSensor_data_type data_type;

        get_ans_args.get_ext_param_ans_args.len = System.Find_sensor(args->id, NULL, (void**)&answer_args, &data_type, NULL);

        if(get_ans_args.get_ext_param_ans_args.len && data_type == UNITED)
        {
          get_ans_args.get_ext_param_ans_args.len = System.FindCustomCanSensorWithCorrectTypeAndSize(args->id, &data_type);
        }

        get_ans_args.get_ext_param_ans_args.data_type=data_type;
      }
      // Для чтения настроек
      else
      {
        get_ans_args.get_ext_param_ans_args.len = System.Find_setting(args->id, args->subid, (uint8_t**)&answer_args);
        get_ans_args.get_ext_param_ans_args.data_type=ARRAY; //настройки всегда как массив?
      }
      // Если параметр найден успешно, будем его отправлять
      if(get_ans_args.get_ext_param_ans_args.len)
      {
        get_ans_len=sizeof(get_ans_args.get_ext_param_ans_args);
        command_ans.arg_len =  get_ans_args.get_ext_param_ans_args.len + sizeof(get_ans_args.get_ext_param_ans_args);
        command_ans.result = DONE;
      }
      // Иначе вываливаем ошибку
      else
      {
        command_ans.result = UNKNOWN_PARAMETER;
        command_ans.arg_len = 0;
      }
      break;
    }
    // Рестарт
    case RESTART:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->need_restart = 1;
      cmd_verbose = "restart";
      break;
    }
    // Сброс к заводским настройкам
    case TOFACTORY:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->need_tofactory = 1;
      cmd_verbose = "tofactory";
      break;
    }
    case CLEAN_BB:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->need_bb_clean = 1;
      cmd_verbose = "clean black box";
      break;
    }
    case SET_OUTPUT:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      TVega_set_output_args* args = (TVega_set_output_args*)(message + sizeof(TVega_message) + sizeof(TVega_command));
      System.signal_state.digital_output[args->output_num-1] = args->state;
      LOG("Command set output not supported\n");
      break;
    }

#if defined(CAMERA_PRESENT)
    case TAKE_PHOTO:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_SIGNAL_PHOTO_GET;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "take a photo";
      break;
    }
#endif //CAMERA_PRESENT

    case GNSS_COOLD_START:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_GNSS_HW_COLD_START;
      if(!this->Send_message(this->gps_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "gnss coold start";
      break;
    }

#if (defined(CANPRO_PRESENT) ||  defined(CANLOG_V4_PRESENT) || defined(VEGA_CAN_PRESENT))
    case BLINKER_FLASH:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      //command_ans.arg_len = 0;

      using args_t = TVega_set_blinker_flash_args;
      uint32_t shift = sizeof(TVega_message) + sizeof(TVega_command);

      args_t* blink_args = reinterpret_cast<args_t*>((message + shift));

      blink_time_timeout = default_time_timeout;
      blink_sampling_timeout = default_sampling_timeout;

      if (input_comm_arg_len > 0 && blink_args->time > 0) {
        blink_time_timeout = blink_args->time * SEC;
      }

      if (input_comm_arg_len > (sizeof(args_t) / 2) && blink_args->sampling > 0) {
        blink_sampling_timeout = blink_args->sampling * MS;
      }

      __PRINTF("Blinker time: %d\n", blink_time_timeout);
      __PRINTF("Blinker sampling: %d\n", blink_sampling_timeout);

      if (!Start_blink()) {
        command_ans.result = BUSY;
        Clear_blink();
      }

      cmd_verbose = "blinker flash";
      break;
    }
    case CLOSE_DOORS:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      if (!do_close_doors()) {
        command_ans.result = BUSY;
      }
      cmd_verbose = "close doors";
      break;
    }
    case OPEN_DOORS:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      if (!do_open_doors()) {
        command_ans.result = BUSY;
      }
      cmd_verbose = "open doors";
      break;
    }
    case OPEN_DR_DOOR:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_DRIVER_DOOR_OPENING_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "open driver door";
      break;
    }
    case HORN:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_HORN_SIGNALING_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "horn";
      break;
    }
    case TRUNK:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_TRUNK_DOOR_OPENING_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "trunk";
      break;
    }
    case STOP_ENGINE:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      if (!do_stop_engine()) {
        command_ans.result = BUSY;
      }
      cmd_verbose = "stop engine";
      break;
    }
    case START_ENGINE:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      if (!do_start_engine()) {
        command_ans.result = BUSY;
      }
      cmd_verbose = "start engine";
      break;
    }
    case START_WEBASTO:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WEBASTO_STARTING_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "start webasto";
      break;
    }
    case STOP_WEBASTO:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WEBASTO_STOPPING_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "stop webasto";
      break;
    }
    case EMULATE_DRIVER_DOOR:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_EMULATION_DRIVER_DOOR_OPENING_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "emulate driver door";
      break;
    }
    case HORN_AND_BLINK:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_HORN_W_BLINKER_SIGNALIG_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "horn and blink";
      break;
    }
    case WINDOWS_CLOSING_3S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_CLOSING_3S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows closing 3s";
      break;
    }
    case WINDOWS_CLOSING_7S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_CLOSING_7S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows closing 7s";
      break;
    }
    case WINDOWS_CLOSING_11S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_CLOSING_11S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows closing 11s";
      break;
    }
    case WINDOWS_CLOSING_15S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_CLOSING_15S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows closing 15s";
      break;
    }
    case WINDOWS_CLOSING_19S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_CLOSING_19S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows closing 19s";
      break;
    }
    case WINDOWS_CLOSING_23S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_CLOSING_23S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows closing 23s";
      break;
    }
    case WINDOWS_CLOSING_29S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_CLOSING_29S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows closing 29s";
      break;
    }
    case WINDOWS_OPENING_3S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_OPENING_3S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows opening 3s";
      break;
    }
    case WINDOWS_OPENING_7S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_OPENING_7S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows opening 7s";
      break;
    }
    case WINDOWS_OPENING_11S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_OPENING_11S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows opening 11s";
      break;
    }
    case WINDOWS_OPENING_15S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_OPENING_15S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows opening 15s";
      break;
    }
    case WINDOWS_OPENING_19S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_OPENING_19S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows opening 19s";
      break;
    }
    case WINDOWS_OPENING_23S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_OPENING_23S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows opening 23s";
      break;
    }
    case WINDOWS_OPENING_29S:
    {
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.result = IN_PROCESS;
#else
      command_ans.result = DONE;
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = CAN_WINDOWS_OPENING_29S_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "windows opening 29s";
      break;
    }

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
    //команды Я.Драйв
    case YADRIVE_WARMING:
    {
      const uint8_t default_warming_time=15;
      uint8_t* warming_args=NULL;

      if(input_comm_arg_len==0)
      {
        warming_args = (uint8_t*)&default_warming_time;
        cmd_verbose = "yadrive warming (without args)";
      }
      else if(input_comm_arg_len==sizeof(uint8_t) && input_comm_arg_len<=sizeof(this->output_message.ext_args))
      {
        warming_args = (uint8_t*)(message+sizeof(TVega_message)+sizeof(TVega_command));
        cmd_verbose = "yadrive warming (with args)";
      }

      if(warming_args!=NULL)
      {
        command_ans.result = IN_PROCESS;
        command_ans.arg_len = 0;
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = YADRIVE_WARMING_SUBCODE;
        memcpy(&this->output_message.ext_args, warming_args, sizeof(*warming_args));
        if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      }
      else
      {
        command_ans.result = COMM_ERROR;
        command_ans.arg_len = 0;
        cmd_verbose = "yadrive warming wrong arg";
      }
      break;
    }
    case YADRIVE_STOP_WARMING:
    {
      command_ans.result = IN_PROCESS;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = YADRIVE_STOP_WARMING_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "yadrive stop warming";
      break;
    }
    case YADRIVE_START_OF_LEASE:
    {
      command_ans.result = IN_PROCESS;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = YADRIVE_START_OF_LEASE_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "yadrive start of lease";
      break;
    }
    case YADRIVE_END_OF_LEASE:
    {
      command_ans.result = IN_PROCESS;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = YADRIVE_END_OF_LEASE_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "yadrive end of lease";
      break;
    }
    case YADRIVE_UNLOCK_HOOD:
    {
      command_ans.result = IN_PROCESS;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = YADRIVE_HOOD_UNLOCK_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "yadrive hood unlock";
      break;
    }
    case YADRIVE_LOCK_HOOD:
    {
      command_ans.result = IN_PROCESS;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = YADRIVE_HOOD_LOCK_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "yadrive hood lock";
      break;
    }
    case YADRIVE_FORCED_END_OF_LEASE:
    {
      command_ans.result = IN_PROCESS;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = YADRIVE_FORCED_END_OF_LEASE_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "yadrive forced end of lease";
      break;
    }
    case YADRIVE_PANIC:
    {
      if(input_comm_arg_len==sizeof(TVega_panic_args) && input_comm_arg_len<=sizeof(this->output_message.ext_args))
      {
        TVega_panic_args* panic_args = (TVega_panic_args*)(message + sizeof(TVega_message) + sizeof(TVega_command));

        command_ans.result = IN_PROCESS;
        command_ans.arg_len = 0;
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = YADRIVE_PANIC_SUBCODE;
        memcpy(&this->output_message.ext_args, panic_args, input_comm_arg_len);
        if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
        cmd_verbose = "yadrive panic";
      }
      else
      {
        command_ans.result = COMM_ERROR;
        command_ans.arg_len = 0;
        cmd_verbose = "yadrive panic wrong arg";
      }
      break;
    }
    case YADRIVE_AUDIO_COMMANDS:
    {
      command_ans.arg_len = 0;
#if defined(JQ6500_AUDUO_PRESENT)
      bool is_busy;

      if(yandex_audio_cmd_callback(&message[sizeof(TVega_message) + sizeof(TVega_command)], input_comm_arg_len, &is_busy, this->input_message.source_id, this->output_message.command_uid, this->output_message.command_nonce, &cmd_verbose))
      {
        if(is_busy) command_ans.result = BUSY;
        else        command_ans.result = IN_PROCESS;

        break;
      }
#endif //defined(JQ6500_AUDUO_PRESENT)

      command_ans.result = COMM_ERROR;
      break;
    }
    case YADRIVE_DTC_CLEAR:
    {
      command_ans.result = IN_PROCESS;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_CAN_CONTROL;
      this->output_message.subcode = YADRIVE_DTC_CLEAR_SUBCODE;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "yadrive dtc clear";
      break;
    }
    case YADRIVE_EMERGENCY_STOP:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->need_emergency_stop = 1;
      cmd_verbose = "yadrive emergency stop";
      break;
    }
    case AUXILIARY_CAR_COMMANDS:
    {
      if(input_comm_arg_len>=sizeof(uint8_t) && input_comm_arg_len<=sizeof(this->output_message.ext_args))
      {
        uint8_t* arg = (uint8_t*)(message + sizeof(TVega_message) + sizeof(TVega_command));

        command_ans.result = IN_PROCESS;
        command_ans.arg_len = 0;
        this->output_message.code = COMMAND_CAN_CONTROL;
        this->output_message.subcode = AUXILIARY_CAR_COMMANDS_SUBCODE;
        memcpy(&this->output_message.ext_args, arg, input_comm_arg_len);
        if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
        cmd_verbose = "\"auxiliary car cmd\"";
      }
      else
      {
        command_ans.result = COMM_ERROR;
        command_ans.arg_len = 0;
        cmd_verbose = "\"auxiliary car cmd\"";
      }
      break;
    }
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
#endif //CANPRO_PRESENT || CANLOG_V4_PRESENT || VEGA_CAN_PRESENT

#if defined(PROD_TESTING_PRESENT)
    case START_PROD_TESTS:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_SIGNAL_START_PROD_TESTS;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "start prod tests";
      break;
    }
#endif //PROD_TESTING_PRESENT

#if defined(INTERNAL_NRF_PRESENT)
  case INTERNAL_NRF_ADD_RELAY:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_SIGNAL_NRF_ADD_RELAY;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "add nrf relay";
      break;
    }
  case INTERNAL_NRF_DEL_RELAY:
    {
      command_ans.result = DONE;
      command_ans.arg_len = 0;
      this->output_message.code = COMMAND_SIGNAL_NRF_DEL_RELAY;
      if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
      cmd_verbose = "delete nrf relay";
      break;
    }
  case INTERNAL_NRF_ADD_MARK:
    {
      const uint8_t* mark_id = (const uint8_t*)(message + sizeof(TVega_message) + sizeof(TVega_command));
      if(*mark_id<MAX_INTERNAL_NRF_MARKS_COUNT)
      {
        command_ans.result = DONE;
        command_ans.arg_len = 0;
        this->output_message.code = COMMAND_SIGNAL_NRF_ADD_MARK;
        this->output_message.subcode = *mark_id;
        if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
        cmd_verbose = "add nrf mark";
      }
      else
      {
        command_ans.result = COMM_ERROR;
        command_ans.arg_len = 0;
        cmd_verbose = "add nrf mark with wrong id";
      }
      break;
    }
  case INTERNAL_NRF_DEL_MARK:
    {
      const uint8_t* mark_id = (const uint8_t*)(message + sizeof(TVega_message) + sizeof(TVega_command));
      if(*mark_id<MAX_INTERNAL_NRF_MARKS_COUNT)
      {
        command_ans.result = DONE;
        command_ans.arg_len = 0;
        this->output_message.code = COMMAND_SIGNAL_NRF_DEL_MARK;
        this->output_message.subcode = *mark_id;
        if(!this->Send_message(this->signal_queue, 2000)) command_ans.result = BUSY;
        cmd_verbose = "delete nrf mark";
      }
      else
      {
        command_ans.result = COMM_ERROR;
        command_ans.arg_len = 0;
        cmd_verbose = "delete nrf mark with wrong id";
      }
      break;
    }
#endif //INTERNAL_NRF_PRESENT

#if defined(CAN_VEGA_FWD_PRESENT)
  case CAN_VEGA_FWD_CONFIG:
    {
      command_ans.arg_len = 0;

      extern bool can_fwd_cmd_callback(const uint8_t* const args, const uint16_t args_len, const uint8_t server_id, const char** verbose);
      if(can_fwd_cmd_callback(&message[sizeof(TVega_message) + sizeof(TVega_command)], input_comm_arg_len, this->input_message.source_id, &cmd_verbose))
      {
        command_ans.result = DONE;
        break;
      }

      command_ans.result = COMM_ERROR;
      break;
    }
#endif //defined(CAN_VEGA_FWD_PRESENT)

#if defined(FASTDATA_PRESENT)
  case FASTDATA_FWD_CONFIG:
    {
      command_ans.arg_len = 0;

      extern bool fastdata_fwd_cmd_callback(const uint8_t* const args, const uint16_t args_len, const uint8_t server_id, const char** verbose);
      const uint8_t* args = &message[sizeof(TVega_message) + sizeof(TVega_command)];
      if(fastdata_fwd_cmd_callback(args, input_comm_arg_len, this->input_message.source_id, &cmd_verbose))
      {
        command_ans.result = DONE;
        break;
      }

      command_ans.result = COMM_ERROR;
      break;
    }
#endif //defined(FASTDATA_PRESENT)

#if defined(YANDEX_OBD_PRESENT)
  case CAN_VEGA_OBD_COMMAND:
    {
      command_ans.arg_len = 0;
      bool is_busy;

      extern bool obd_cmd_callback(const uint8_t* const args, const uint16_t args_len, bool* const is_busy, const char** verbose);
      if(obd_cmd_callback(&message[sizeof(TVega_message) + sizeof(TVega_command)], input_comm_arg_len, &is_busy, &cmd_verbose))
      {
        if(is_busy) command_ans.result = BUSY;
        else        command_ans.result = DONE;

        break;
      }

      command_ans.result = COMM_ERROR;
      break;
    }
#endif //defined(YANDEX_OBD_PRESENT)

    case GSM_MODEM_COMMANDS:
    {
      command_ans.arg_len = 0;

      const uint8_t* arg = (const uint8_t*)(message + sizeof(TVega_message) + sizeof(TVega_command));

      if(input_comm_arg_len==sizeof(uint8_t) && arg[0]==0)
      {
#if MAX_SIM_COUNT > 1
        if(SERVER_manager.Is_busy_for_sim_change())
        {
          command_ans.result = BUSY;
          break;
        }

        SERVER_manager.Force_sim_change();

        command_ans.result = DONE;

        break;
#endif //MAX_SIM_COUNT > 1
      }

      command_ans.result = COMM_ERROR;
      break;
    }

    default:
    {
      command_ans.result = UNKNOWN_PARAMETER;
      command_ans.arg_len = 0;
      break;
    }
  }

  if(cmd_verbose!=NULL)
  {
    if(command_ans.result == BUSY)
    {
      LOG("%s: command %s is lost (dst manager is busy)\n", this->name, cmd_verbose);
    }
    else
    {
      LOG("%s: command %s recv\n", this->name, cmd_verbose);
    }
  }

  if(input_mess->type == COMMAND_W_NONCE)
  {
    //проверить использования стека задачи!
    TVega_command_ans_w_sign command_ans_w_sign;

    memcpy(command_ans_w_sign.nonce, input_comm.input_comm_w_nonce->nonce, sizeof(command_ans_w_sign.nonce));
    command_ans_w_sign.uid=input_comm.input_comm_w_nonce->uid;

    command_ans_w_sign.arg_len=command_ans.arg_len;
    command_ans_w_sign.result=command_ans.result;

    return Pack_vega_command_answer(this->input_message.source_id, COMMAND_ANS_W_SINGN,
                                    &command_ans_w_sign, answer_args, answer_buff, answer_buff_size,
                                    &get_ans_args, get_ans_len);
  }

  return Pack_vega_command_answer(this->input_message.source_id, COMMAND_ANS,
                                  &command_ans, answer_args, answer_buff, answer_buff_size,
                                  &get_ans_args, get_ans_len);
}

// Упаковка ответа на команду Вега
uint16_t CSyscom_manager::Pack_vega_command_answer(const uint8_t server_idx,
                                                   const EVega_message_type message_ans_type, const void* command_answer, uint8_t* answer_args,  uint8_t* buff, uint16_t max_size,
                                                   const void* get_ans, uint16_t get_ans_len)
                                                   //uint16_t get_ans_id, uint8_t get_ans_len)
{
  TVega_message message;
  uint32_t CRC32;
  uint8_t* buff_ptr = buff;

  uint8_t command_answer_arg_len;
  uint8_t command_answer_header_len;

  // Поля сообщения
  message.type = message_ans_type;

  if(message.type==COMMAND_ANS)
  {
    const TVega_command_ans* const ans = (const TVega_command_ans* const)command_answer;
    command_answer_arg_len=ans->arg_len;
    command_answer_header_len=sizeof(*ans);
  }
  else //if(message.type==COMMAND_ANS_W_SINGN)
  {
    const TVega_command_ans_w_sign* const ans = (const TVega_command_ans_w_sign* const)command_answer;
    command_answer_arg_len=ans->arg_len;
    command_answer_header_len=sizeof(*ans);
  }

  message.len = (command_answer_header_len + command_answer_arg_len);
  // Расчет размера всего сообщения
  uint16_t size = (sizeof(message) + message.len + sizeof(CRC32));
  // Проверка размера
  if(size > max_size) return 0;
  // Упаковка ответа
  memcpy(buff_ptr, &message, sizeof(message));
  buff_ptr += sizeof(message);
  memcpy(buff_ptr, command_answer, command_answer_header_len);
  buff_ptr += command_answer_header_len;

  // Если есть аргументы ответа на команду считать параметр, то закопировать их в ответ
  if(get_ans_len)
  {
    memcpy(buff_ptr, get_ans, get_ans_len);
    buff_ptr += get_ans_len;
    command_answer_arg_len = command_answer_arg_len - get_ans_len;
  }

  // Если есть аргументы, то закопировать их в ответ
  if(command_answer_arg_len)
  {
    if(command_answer_arg_len>1 && get_ans_len) {System.Grab();}
    memcpy(buff_ptr, answer_args, command_answer_arg_len);
    if(command_answer_arg_len>1 && get_ans_len) {System.Release();}
    buff_ptr += command_answer_arg_len;
  }

  if(message.type==COMMAND_ANS_W_SINGN)
  {
    TVega_command_ans_w_sign* ans = (TVega_command_ans_w_sign*)&buff[sizeof(sizeof(TVega_message))];

    mbedtls_poly1305_mac(get_cmd_ans_mac_key(), &ans->nonce[0], message.len-sizeof(ans->mac), ans->mac);

    if(server_idx==get_server_id_to_add_notify_to_blackbox())
    {
      BB_message track_point;
      memset(&track_point, 0x00, sizeof(track_point));

      System.Grab();
      memset(this->message_buff, 0, sizeof(this->message_buff));
      track_point.len = Pack_vega_message_w_custom_data(this->message_buff, sizeof(this->message_buff), BLE_EXT_BOARD_COMMAND_ANS_W_SINGN, ans, sizeof(*ans)+ans->arg_len);
      track_point.payload = this->message_buff;
      System.Release();

      if(System.connection_settings.server[0].server_protocol == VEGA) track_point.receiver_1 = 1;
      if(System.connection_settings.server[1].server_protocol == VEGA) track_point.receiver_2 = 1;
      if(System.connection_settings.server[2].server_protocol == VEGA) track_point.receiver_3 = 1;
      if(System.connection_settings.server[3].server_protocol == VEGA) track_point.receiver_4 = 1;

      if(track_point.receiver_1 || track_point.receiver_2 || track_point.receiver_3 || track_point.receiver_4)
      {
        Black_box.Grab();
        Black_box.add_message(&track_point);
        Black_box.Release();
      }
    }
  }

  //
  CRC32 = crc32_fast_full(buff, (message.len + sizeof(message)));
  memcpy(buff_ptr, &CRC32, sizeof(CRC32));
  // Возвращаем размер всего упакованного сообщения
  return size;
}

// Загрузка настроек
void CSyscom_manager::Load_settings(void)
{
  Black_box.Grab();
  // Инициализация ЧЯ
  Black_box.Init();
  // Структура для взаимодействия с загрузчиком
  memset(&boot_cmd, 0x00, sizeof(boot_cmd));
  Black_box.Register_in_base("boot_cmd", sizeof(boot_cmd), 0, &boot_cmd);

  Black_box.Register_in_base("gps_odo", sizeof(System.gnss_state.odometer), 0, &System.gnss_state.odometer);
  Black_box.Register_in_base("gps_trip", sizeof(System.gnss_state.trip_counter), 0, &System.gnss_state.trip_counter);
  Black_box.Register_in_base("traffic", sizeof(System.server_state.traffic_counter), 0, &System.server_state.traffic_counter);
  Black_box.Register_in_base("pulse_inputs", sizeof(System.signal_state.pulse_input), 0, &System.signal_state.pulse_input);
  //состояние выходов не сохраняем!
  //Black_box.Register_in_base("outputs", sizeof(System.signal_state.digital_output), 30, &System.signal_state.digital_output);
  #if defined(BR_PRESENT)
  Black_box.Register_in_base("ex_outputs", sizeof(System.signal_state.ext_digital_output), 20, &System.signal_state.ext_digital_output);
  Black_box.Register_in_base("ex_pulse_inputs", sizeof(System.signal_state.ext_pulse_input), 20, &System.signal_state.ext_pulse_input);
  #endif //BR_PRESENT
  Black_box.Register_in_base("motohours", sizeof(System.signal_state.motohours), 20, &System.signal_state.motohours);

  // Версия базы конфигурации (для использования в послерелизных прошивках)
  Black_box.Register_in_base("settings_ver", sizeof(System.settings_version), 0, &System.settings_version);
  Black_box.Release();
  System.Grab();
  // Детектирование первого старта
  if(System.settings_version != CURRENT_SETTINGS_VERSION)
  {
    Black_box.Grab();
    // Переинициализация черного ящика с форматированием памяти
    NOR_ftl.Format_memory();
    Black_box.Init();
    // Установка boot cmd
    Black_box.Register_in_base("boot_cmd", sizeof(boot_cmd), 0, &boot_cmd);
    memset(&boot_cmd, 0x00, sizeof(boot_cmd));
    Black_box.Save_base("boot_cmd");
    Black_box.Register_in_base("gps_odo", sizeof(System.gnss_state.odometer), 0, &System.gnss_state.odometer);
    Black_box.Register_in_base("gps_trip", sizeof(System.gnss_state.trip_counter), 0, &System.gnss_state.trip_counter);
    Black_box.Register_in_base("traffic", sizeof(System.server_state.traffic_counter), 0, &System.server_state.traffic_counter);
    Black_box.Register_in_base("pulse_inputs", sizeof(System.signal_state.pulse_input), 0, &System.signal_state.pulse_input);
    //состояние выходов не сохраняем!
    //Black_box.Register_in_base("outputs", sizeof(System.signal_state.digital_output), 30, &System.signal_state.digital_output);
    #if defined(BR_PRESENT)
    Black_box.Register_in_base("ex_outputs", sizeof(System.signal_state.ext_digital_output), 20, &System.signal_state.ext_digital_output);
    Black_box.Register_in_base("ex_pulse_inputs", sizeof(System.signal_state.ext_pulse_input), 20, &System.signal_state.ext_pulse_input);
    #endif //BR_PRESENT
    Black_box.Register_in_base("motohours", sizeof(System.signal_state.motohours), 20, &System.signal_state.motohours);

    //Установка признака первого старта
    System.first_start = 1;

    System.gnss_state.odometer = 0;
    System.gnss_state.trip_counter = 0;
    memset(System.server_state.traffic_counter, 0x00, sizeof(System.server_state.traffic_counter));
    memset(System.signal_state.pulse_input, 0x00, sizeof(System.signal_state.pulse_input));

    // Установка начального состояния выходов в 0 (транзисторы ОК закрыты)
    memset(System.signal_state.digital_output, 0x00, sizeof(System.signal_state.digital_output));

    // Установка настроек по умолчанию
    System.blackbox_settings.vega_protocol_bblox_enable = 0;
    System.blackbox_settings.add_lbs_param = 0;

    //Установка настроек по умолчанию
    // Настройки трека
    System.track_settings.filter_track_hdop = 3;
    System.track_settings.use_inertial_navigation = 0;
    System.track_settings.filter_track_on_stop = 1;
    System.track_settings.reset_odometer_on_stop = 0;
    System.track_settings.track_param.time_step = 10;
    System.track_settings.time_step_with_ign_off = 180;
    System.track_settings.track_param.distance_step = 10;
    System.track_settings.track_param.course_step = 7;
    System.track_settings.always_locate = 0;
    System.track_settings.odometer_by_ign = 0;
    GPS_manager.SetDefaultNavSystemSettings(&System.track_settings.nav_systems, NULL);
    GPS_manager.SetDefaultExtFiltersSettings(&System.track_settings.extended_filters);

    System.connection_settings.main_yandex_server_id = SERVER_TCP_2_CONN_ID;
    memset(System.connection_settings.balance_request_settings, 0x00, sizeof(System.connection_settings.balance_request_settings));

    // Настройки соедниения
    memcpy(System.connection_settings.server[0].address, "213.180.193.73:24450", sizeof("213.180.193.73:24450"));
    System.connection_settings.server[0].server_protocol = SERVER_OFF;
    System.connection_settings.server[0].connection_period = 0;
    System.connection_settings.ndtp_address[0] = 0;
    System.connection_settings.security[0].pin_cfg = PIN_DISABLED;
    System.connection_settings.security[0].pin[0] = '\0';

    memcpy(System.connection_settings.server[1].address, "mt.carsharing.yandex.net:24690", sizeof("mt.carsharing.yandex.net:24690"));
    System.connection_settings.server[1].server_protocol = VEGA;
    System.connection_settings.server[1].connection_period = 0;
    System.connection_settings.ndtp_address[1] = 0;
    System.connection_settings.security[1].pin_cfg=PIN_DISABLED;
    System.connection_settings.security[1].pin[0] = '\0';

    memcpy(System.connection_settings.server[2].address, "213.180.204.74:17578", sizeof("213.180.204.74:17578"));
    System.connection_settings.server[2].server_protocol = VEGA;
    System.connection_settings.server[2].connection_period = 0;
    System.connection_settings.ndtp_address[2] = 0;
    System.connection_settings.security[2].pin_cfg = PIN_DISABLED;
    System.connection_settings.security[2].pin[0] = '\0';

    memcpy(System.connection_settings.server[3].address, "89.189.183.233:5604", sizeof("89.189.183.233:5604"));
    System.connection_settings.server[3].server_protocol = SERVER_OFF;
    System.connection_settings.server[3].connection_period = 0;
    System.connection_settings.ndtp_address[3] = 0;
    System.connection_settings.security[3].pin_cfg = PIN_DISABLED;
    System.connection_settings.security[3].pin[0] = '\0';

    memcpy(System.connection_settings.APN[0].APN, "yandex.drive.msk", sizeof("yandex.drive.msk"));
    memcpy(System.connection_settings.APN[0].user, "", sizeof(""));
    memcpy(System.connection_settings.APN[0].password, "", sizeof(""));
#if MAX_SIM_COUNT > 1
    memcpy(System.connection_settings.APN[1].APN, "m2m.beeline.ru", sizeof("m2m.beeline.ru"));
    memcpy(System.connection_settings.APN[1].user, "beeline", sizeof("beeline"));
    memcpy(System.connection_settings.APN[1].password, "beeline", sizeof("beeline"));
#endif //MAX_SIM_COUNT > 1

    memset(System.connection_settings.regular_translation, 0x00, sizeof(System.connection_settings.regular_translation));
    // Настройки регулярной передачи по умолчанию
    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      if(i > System.Total_sensor_count) break;

      //Системные датчики
      else if(System.sensor_table[i].id == VEGA_MCU_FIRMWARE_VERSION) {System.connection_settings.regular_translation[i].translate_period = 3600; System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_GSM_FIRMWARE_VERSION) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_GPS_FIRMWARE_VERSION) { System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_SIM_ICCID) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_UPTIME) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_OPERATION_MODE) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_BB_MESSAGE_COUNT_1) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_TCP_STATE1) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_TCP_STATE4) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_SIM2_ICCID) {System.connection_settings.regular_translation[i].translate_with_change = 1;}

      //Навигационные датчики
      else if(System.sensor_table[i].id == VEGA_HDOP) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_PDOP) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_VDOP) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_GPS_INVIEW) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_GLONASS_INVIEW) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_GPS_ODOMETER) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_GPS_MOVE_SENSOR) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_TRIP_COUNTER) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_GPS_JAMMED) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_GPS_SPOOF_SENSOR) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_GPS_IS_ACTIVE) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_GEOZONE_1_ID+49) {System.connection_settings.regular_translation[i].translate_with_change = 1;} //зачем им 50я геозона?

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
      //Радиореле и радиометки
      else if(System.sensor_table[i].id == VEGA_NRF_RELAY_ID) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 3600;}
      else if(System.sensor_table[i].id == VEGA_NRF_ACTUAL_RELAY_STATE) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_NRF_VISIBLE_MARKS_BF) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_NRF_BATTLOW_MARKS_BF) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
#endif //(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)

      //Входы/выходы
      else if(System.sensor_table[i].id == VEGA_DIGITAL_INPUT_1+2) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_DIGITAL_OUTPUT_1+3) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_DIGITAL_OUTPUT_1+4) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_DIGITAL_OUTPUT_1+9) {System.connection_settings.regular_translation[i].translate_with_change = 1;}

      //else if(System.sensor_table[i].id == VEGA_CAN_ECO_MODE){System.connection_settings.regular_translation[i].translate_with_change = 1;}
      //else if(System.sensor_table[i].id == VEGA_CAN_WASHER_LIQUID){System.connection_settings.regular_translation[i].translate_period = 3600; System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_LAUNCH_SENSOR){System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == VEGA_DRIFT_SENSOR) {System.connection_settings.regular_translation[i].translate_with_change = 1;}

      else if(System.sensor_table[i].id == VEGA_TAMPER_1) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}
      else if(System.sensor_table[i].id == VEGA_TAMPER_1+1) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}

      else if(System.sensor_table[i].id == VEGA_IGNITION) {System.connection_settings.regular_translation[i].translate_with_change = 1;}

      else if(System.sensor_table[i].id == VEGA_GSENSOR_AXIS_X) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_GSENSOR_AXIS_Y) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_GSENSOR_AXIS_Z) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_GSENSOR_MOVE_SENSOR) {System.connection_settings.regular_translation[i].translate_with_change = 1;}

      else if(System.sensor_table[i].id == VEGA_INT_TEMP) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}
      else if(System.sensor_table[i].id == VEGA_POWER_VOLTAGE) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}
      else if(System.sensor_table[i].id == VEGA_ACC_VOLTAGE) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}

      //Датчики состовой сети
      else if(System.sensor_table[i].id == VEGA_MCC) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}
      else if(System.sensor_table[i].id == VEGA_MNC) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}
      else if(System.sensor_table[i].id == VEGA_LAC) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}
      else if(System.sensor_table[i].id == VEGA_CELLID) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}
      else if(System.sensor_table[i].id == VEGA_GSM_SIGNAL_LEVEL) {System.connection_settings.regular_translation[i].translate_with_track = 1; System.connection_settings.regular_translation[i].translate_period = 303;}

      else if(System.sensor_table[i].id == VEGA_TX_DATA_SERV1) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_TX_DATA_SERV4) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_RX_DATA_SERV1) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == VEGA_RX_DATA_SERV4) {System.connection_settings.regular_translation[i].translate_with_track = 1;}

      //Датчики CAN - шины сети
      else if(System.sensor_table[i].id == CAN_ODOMETER_KM) {System.connection_settings.regular_translation[i].translate_period = 60;}
      else if(System.sensor_table[i].id == CAN_FUEL_LEVEL_P) {System.connection_settings.regular_translation[i].translate_period = 60;}
      else if(System.sensor_table[i].id == CAN_FUEL_LEVEL_L) {System.connection_settings.regular_translation[i].translate_with_track = 1;}//это запас хода
      else if(System.sensor_table[i].id == CAN_ENGINE_RPM) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == CAN_ENGINE_TEMP) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == CAN_SPEED) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
      else if(System.sensor_table[i].id == CAN_MARKER_LIGHTS) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_DIPPED_BEAM) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_HIGH_BEAM) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_DRIVER_SAFE_BELT) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_IGNITION) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_PARKING) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_HAND_BREAK) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_PEDAL_BREAK) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_ENGINE_IS_ON) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_REVERSE_GEAR) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      else if(System.sensor_table[i].id == CAN_DRIVER_DOOR) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}
      else if(System.sensor_table[i].id == CAN_PASS_DOOR) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}
      else if(System.sensor_table[i].id == CAN_L_REAR_DOOR) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}
      else if(System.sensor_table[i].id == CAN_R_REAR_DOOR) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}
      else if(System.sensor_table[i].id == CAN_HOOD) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}
      else if(System.sensor_table[i].id == CAN_TRUNK) {System.connection_settings.regular_translation[i].translate_with_change = 1; System.connection_settings.regular_translation[i].translate_period = 65535;}
      else if(System.sensor_table[i].id == CAN_IN_SLEEP) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
    }

#if defined(ATOM_PRESENT)
#warning
    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      if(System.sensor_table[i].id >= ATOM_COOMUNICATION_STATE && System.sensor_table[i].id <= ATOM_VIDEO_ID) {System.connection_settings.regular_translation[i].translate_period = 5; /*System.connection_settings.regular_translation[i].translate_with_change = 1;*/}
    }
#endif //ATOM_PRESENT

#if defined(EXT_GSENSOR_PRESENT)
#warning
    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      if(System.sensor_table[i].id == VEGA_EXT_GSENS_COMM_STATE) {System.connection_settings.regular_translation[i].translate_period = 5*60; System.connection_settings.regular_translation[i].translate_with_change = 1;}
      if(System.sensor_table[i].id == VEGA_EXT_GSENS_FLAGS) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
      if(System.sensor_table[i].id == VEGA_EXT_GSENS_MANEUVER_STR) {System.connection_settings.regular_translation[i].translate_with_change = 1;}
    }
#endif //EXT_GSENSOR_PRESENT

    // Натсройки энергосбережения
    System.power_settings.sleep_by_ignition = 0;
    System.power_settings.sleep_by_stop = 0;
    System.power_settings.sleep_by_wakeup = 0;
    System.power_settings.slep_and_case = 0;
    System.power_settings.wakeup_by_ignition = 0;
    System.power_settings.wakeup_by_move = 0;
    System.power_settings.wakeup_by_time = 0;
    System.wakeup_by_can_activity = 0;
    System.power_settings_2.wake_by_dig_in = 0;

#if defined(GNSS_PSAVE_PRESENT)
    memset(&System.gnss_psave_settings, 0, sizeof(System.gnss_psave_settings));
#endif //defined(GNSS_PSAVE_PRESENT)

#if defined(MODEM_PSAVE_PRESENT)
    memset(&System.gsm_psave_settings, 0, sizeof(System.gsm_psave_settings));
#endif //defined(MODEM_PSAVE_PRESENT)

    // Настройки безопасности
#if (MAX_AUTORIZED_KEYS > 0)
    memset(&System.security_settings.autorized_keys, 0x00, sizeof(System.security_settings.autorized_keys));
#endif
    memset(&System.security_settings.autorized_phones, 0x00, sizeof(System.security_settings.autorized_phones));
    System.security_settings.use_pin = 1;  // Пин по умолчанию включен
    memcpy(&System.security_settings.pin, "1234", sizeof("1234")); //Пин по умолчанию 1234

    // Настройки геозон
    memset(&System.geozone_settings, 0x00, sizeof(System.geozone_settings));

    // Настройки датчиков
    System.sensor_settings.alarm_button.din_num = 0;
    System.sensor_settings.autorization.reset_after_stop = 0;
    System.sensor_settings.autorization.reset_by_ignition = 0;

#if defined(MAX_CAMERAS) && (MAX_CAMERAS > 0)
    memset(&System.sensor_settings.camera, 0x00, sizeof(System.sensor_settings.camera));
    memset(&System.sensor_settings.camera_type, 0x00, sizeof(System.sensor_settings.camera_type));
#endif // MAX_CAMERAS
    memset(&System.sensor_settings.dut, 0x00, sizeof(System.sensor_settings.dut));
#if defined(MAX_DALLAS_TEMP_SENSORS) && (MAX_DALLAS_TEMP_SENSORS > 0)
    memset(&System.sensor_settings.ext_temp, 0x00, sizeof(System.sensor_settings.ext_temp));
#endif // MAX_DALLAS_TEMP_SENSORS

    memset(&System.sensor_settings.freq_in, 0x00, sizeof(System.sensor_settings.freq_in));
    memset(&System.sensor_settings.freq_out, 0x00, sizeof(System.sensor_settings.freq_out));
    memset(&System.sensor_settings.pulse_in, 0x00, sizeof(System.sensor_settings.pulse_in));
    memset(&System.sensor_settings.mfi, 0x00, sizeof(System.sensor_settings.mfi));

    System.sensor_settings.mfi[2].input_type=FREQ_IN;
    System.sensor_settings.mfi[2].polarity=ACTIVE_0;

#if defined(BR_PRESENT)
    memset(System.signal_state.ext_digital_output, 0x00, sizeof(System.signal_state.ext_digital_output));// Установка начального состояния расширенных выходов в 0 (транзисторы ОК закрыты)
    memset(&System.sensor_settings.br_settings, 0x00, sizeof(System.sensor_settings.br_settings));
    memset(&System.sensor_settings.ext_mfi, 0x00, sizeof(System.sensor_settings.ext_mfi));
#endif //BR_PRESENT

#if defined(FRIDGE_PRESENT)
    System.sensor_settings.fridge.interface=FRIDGE_OFF;
    System.sensor_settings.fridge.protocol=PARTNER_PROTOCOL;
#endif //FRIDGE_PRESENT

#if defined(LORA_SENSORS_PRESENT)
    memset(&System.sensor_settings.lora_sens.sensors, 0, sizeof(System.sensor_settings.lora_sens.sensors));
    if(DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_33 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_34)
    {
      System.sensor_settings.lora_sens.interface=LORA_SENS_RS232;
    }
    else
    {
      System.sensor_settings.lora_sens.interface=LORA_SENS_OFF;
    }
#endif //LORA_SENSORS_PRESENT

#if defined(NRF_BEACONS_SCANNER_PRESENT)
    memset(System.sensor_settings.nrf_beacons_scanner.autorized_beacons, 0, sizeof(System.sensor_settings.nrf_beacons_scanner.autorized_beacons));
    System.sensor_settings.nrf_beacons_scanner.interface=NRF_BEACONS_SCANNER_OFF;
#endif //NRF_BEACONS_SCANNER_PRESENT

#if defined(TENZO_M_WEIGHER_PRESENT)
    System.sensor_settings.tenzo_m_weigher.interface=TENZO_M_WEIGHER_OFF;
    System.sensor_settings.tenzo_m_weigher.addr=1;
#endif //TENZO_M_WEIGHER_PRESENT

#if defined(IQFREEZE_PRESENT)
    System.sensor_settings.iqfreeze.interface=IQFREEZE_OFF;
#warning
    for(uint16_t i = 0; i < System.Total_sensor_count; i++)
    {
      if(System.sensor_table[i].id >= IQFREEZE_COOMUNICATION_STATE && System.sensor_table[i].id <= IQFREEZE_TEMP_PROBE) {System.connection_settings.regular_translation[i].translate_period = 3600; System.connection_settings.regular_translation[i].translate_with_change = 1;}
      if(System.sensor_table[i].id == IQFREEZE_COOMUNICATION_STATE || System.sensor_table[i].id == IQFREEZE_NO_CONNECT) {System.connection_settings.regular_translation[i].translate_with_track = 1;}
    }
#endif //IQFREEZE_PRESENT

#if defined(VOICE_CALL_PRESENT)
    System.voicecall_settings.spk_volume=80;
    System.voicecall_settings.mic_gain=0;
    System.voicecall_settings.allow_all_incoming=1;
#endif //VOICE_CALL_PRESENT

    System.sensor_settings.gps_movesensor.stopfix_time = 300;
    System.sensor_settings.gsensor_movesensor.stopfix_time = 300;
    System.sensor_settings.ignition.din_num  = 0;
    // Для L10, у которого нет зажиганя по умолчанию оно настраиватеся на первый МФВ
    if(DEVICE_TYPE == DEVICE_TYPE_CAN_WAY_L10)
    {
      System.sensor_settings.ignition.din_num  = 1;
      System.sensor_settings.mfi[0].input_type = DIG_IN;
      System.sensor_settings.mfi[0].polarity = ACTIVE_1;
    }

    System.sensor_settings.use_can_ignition = 0;

    System.sensor_settings.motohours.source = IGNITION;
    System.sensor_settings.keys_add_mode = 0;
    System.sensor_settings.temp_add_mode = 0;
    // Настройки сценариев
    memset(&System.scenario_settings, 0x00, sizeof(System.scenario_settings));
    // настройки CAN
    System.can_settings.frame_interval = 1;
    System.can_settings.program_no = 1188;

//total 2863000
#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3)
    //Создание системных файлов
    Black_box.fcreate("LOG",      500000,  FIFO_FILE);
    Black_box.fcreate("FIRMWARE", 662990,  SIMPLE_FILE);
    Black_box.fcreate("FOTO1",    5,  SIMPLE_FILE);
    Black_box.fcreate("FOTO2",    5,  SIMPLE_FILE);
    Black_box.fcreate("FOTO3",    200000,  SIMPLE_FILE);
    Black_box.fcreate("FOTO4",    200000,  SIMPLE_FILE);
    Black_box.fcreate("FOTO5",    200000,  SIMPLE_FILE);
    Black_box.fcreate("CANPRO",   1000000, SIMPLE_FILE);
    Black_box.fcreate("SETTINGS", 50000,   SIMPLE_FILE);
    Black_box.fcreate("BASE",     50000,   SIMPLE_FILE);
#elif (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_25K)
    //Создание системных файлов
    Black_box.fcreate("LOG",      500000,  FIFO_FILE);
    Black_box.fcreate("FIRMWARE", 263000,  SIMPLE_FILE);
    Black_box.fcreate("FOTO1",    200000,  SIMPLE_FILE);
    Black_box.fcreate("FOTO2",    200000,  SIMPLE_FILE);
    Black_box.fcreate("FOTO3",    200000,  SIMPLE_FILE);
    Black_box.fcreate("FOTO4",    200000,  SIMPLE_FILE);
    Black_box.fcreate("FOTO5",    200000,  SIMPLE_FILE);
    Black_box.fcreate("CANPRO",   1000000, SIMPLE_FILE);
    Black_box.fcreate("SETTINGS", 50000,   SIMPLE_FILE);
    Black_box.fcreate("BASE",     50000,   SIMPLE_FILE);
#else
#error
#endif //DEVICE_TYPE_VEGA_MT_32K_LTE || DEVICE_TYPE_VEGA_MT_25K || DEVICE_TYPE_VEGA_MT_32K_LTE_V2 || DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_32K_LTE_V3
    Black_box.Release();
    Create_base_file();
    LOG("%s: First application start, filesystem created, set default settings\n", this->name);
    if((RTC_ReadBackupRegister(RTC_BKP_DR7)==0x01 || RTC_ReadBackupRegister(RTC_BKP_DR7)==0x02))
    {
      //сброс настроек по команде, удалем backup (память модема) если он есть
      RTC_WriteBackupRegister(RTC_BKP_DR8, 0x02);
    }
    else
    {
      //произошел сброс настроек, будет попытка восстановить из backup (память модема)
      RTC_WriteBackupRegister(RTC_BKP_DR8, 0x01);
    }

    // Установка текущей версии настроек
    Black_box.Register_in_base("settings_ver", sizeof(System.settings_version), 0, &System.settings_version);
    System.settings_version = CURRENT_SETTINGS_VERSION;
    Black_box.Save_base("settings_ver");
  }
  // Проверка файла настроек на размер. Если файл его имеет, значит было сброшено питание во время обновления настроек и нужно их вычитать заново
  uint32_t settings_size = 0;
  Black_box.fsize("SETTINGS", &settings_size);
  if(settings_size && Parse_settings("SETTINGS", 1))
  {
    Create_base_file();

    RTC_WriteBackupRegister(RTC_BKP_DR8, 0x00);
    need_update_backup_settings_file();

    Black_box.Grab();
    Black_box.fopen("SETTINGS");
    Black_box.fflush("SETTINGS");
    Black_box.fsave();
    Black_box.fclose("SETTINGS");
    Black_box.Release();
  }
  // В противном случае просто загружаем настройки из файла BASE
  else
  {
    // Проверка, имеет ли файл базы размер
    Black_box.fsize("BASE", &settings_size);
    if(!settings_size)
    {
       // Переинициализация черного ящика с форматированием памяти
       NOR_ftl.Format_memory();
       NVIC_SystemReset();
    }
    Parse_settings("BASE");
  }

  Black_box.Grab();
  // Закрытие файлов
  Black_box.fclose("LOG");
  Black_box.fclose("FIRMWARE");
  Black_box.fclose("FOTO1");
  Black_box.fclose("FOTO2");
  Black_box.fclose("FOTO3");
  Black_box.fclose("FOTO4");
  Black_box.fclose("FOTO5");
  Black_box.fclose("CANPRO");
  Black_box.fclose("SETTINGS");
  Black_box.fclose("BASE");

  // Первое вычитывание сообщений из ЧЯ, чтобы указатели встали на место
  Black_box.read_message(this->message_buff, 1, 0);
  Black_box.read_message(this->message_buff, 2, 0);
  Black_box.read_message(this->message_buff, 3, 0);
  Black_box.read_message(this->message_buff, 4, 0);
  Black_box.Release();

  // Вывод информации о старте системы
  LOG("%s: System start, FW version %s from %s %s\n", this->name, APLICATION_VERSION_STRING, __DATE__, __TIME__);

  if(!System.Is_sensor_table_sorted(true))
  {
    LOG("%s: Warning, sensor table no sorted!\n", this->name);
  }

  if(this->boot_cmd.cmd == NORMAL_SYSTEM_START)
  {
    LOG("%s: Normal start\n", this->name);
  }
  else if(this->boot_cmd.cmd == RES_FLASHED_OK)
  {
    LOG("%s: Firmware upd succesful\n", this->name);
    was_update_firmware = true;
    this->boot_cmd.cmd = NORMAL_SYSTEM_START;
    Black_box.Grab();
    Black_box.Save_base("boot_cmd");
    Black_box.Release();
    for(uint8_t i = 0; i < MAX_SERVERS_COUNT; i++)
    {
      if(System.connection_settings.security[i].pin_cfg == PIN_UNUSED)
      {
        //для старых прошивок
        //System.connection_settings.security[i].pin_cfg = PIN_ENABLED;
        //memcpy(System.connection_settings.security[i].pin, System.security_settings.pin, sizeof(System.connection_settings.security[i].pin));
        System.connection_settings.security[i].pin_cfg = PIN_DISABLED;
      }
    }
    need_save_settings = 1;

  }
  else
  {
    LOG("%s: Firmware upd fail\n", this->name);
    was_update_firmware = true;
    this->boot_cmd.cmd = NORMAL_SYSTEM_START;
    Black_box.Grab();
    Black_box.Save_base("boot_cmd");
    Black_box.Release();
  }
  // Заполнение поля версия прошивки
  if(sizeof(APLICATION_VERSION_STRING) < sizeof(System.syscom_state.mcu_firmware_version))
  {
    memcpy(System.syscom_state.mcu_firmware_version, APLICATION_VERSION_STRING, sizeof(APLICATION_VERSION_STRING));
  }
  else
  {
    memcpy(System.syscom_state.mcu_firmware_version, "XXX", sizeof("XXX"));
  }

  //если случилось так, что опция использовать PIN включена или не не используется, и длина пароля 0, то отключаем пин и стави пароль 1234
  for(uint8_t i = 0; i < MAX_SERVERS_COUNT; i++)
  {
    if(System.connection_settings.security[i].pin_cfg == PIN_UNUSED || (System.connection_settings.security[i].pin_cfg == PIN_ENABLED && strnlen(System.connection_settings.security[i].pin, sizeof(System.connection_settings.security[i].pin))==0))
    {
      System.connection_settings.security[i].pin_cfg = PIN_DISABLED;
      memcpy(System.connection_settings.security[i].pin, "1234", sizeof("1234"));
      need_save_settings = 1;
    }
  }

  //для подключения по usb, если установленна опция использовать пин и pin пустой, то сбрасываем его в "1234"
  if(System.security_settings.use_pin && strnlen(System.security_settings.pin, sizeof(System.security_settings.pin))==0)
  {
    memcpy(System.security_settings.pin, "1234", sizeof("1234"));
    need_save_settings = 1;
  }

  System.Release();
}

// Сохранение показаний счетчиков (по сути всей базы конфигурации раз в 1 минуту)
void CSyscom_manager::Save_counters(void)
{
  if(save_counters_timer.Is_empty())
  {
    Black_box.Grab();
    Black_box.Save_base("all");
    Black_box.Release();
    save_counters_timer.Start(SEC*60);
  }
}

/* C code---------------------------------------------------------------------*/

// Функция запуска Syscom менеджера
void Start_syscom_manager(void* argument)
{
#if ( configUSE_APPLICATION_TASK_TAG == 1 )
  //
#endif /* configUSE_APPLICATION_TASK_TAG == 1 */

  Syscom_manager.Init();
  Syscom_manager.Main_cycle();
  Syscom_manager.Deinit();
  while(1);
}

// Запись в лог
void LOG(const char* string, ...)
{

#if defined(USE_UART_DEBUG_PORT)
#if !defined(__PRINTF_2)
  __DEBUG_UART_GRAB();
  va_list args;
  va_start(args, string);
  vprintf(string, args);
  va_end(args);
  __DEBUG_UART_RELEASE();
#endif //!defined(__PRINTF_2)
#endif //USE_UART_DEBUG_PORT

#if defined(USE_USB_DEBUG_PORT)
#error
#endif //USE_USB_DEBUG_PORT

  Black_box.Grab();
  // Запись временного штампа в начало каждого сообщения
  memset(LOG_BUF, 0x00, sizeof(LOG_BUF));
  uint32_t unixtime = GetUnixTime();
  time_struct_t time;
  conv_unixtime_to_time(&time, unixtime);
  sprintf((char*)LOG_BUF, "[%d/%d/%d][%d:%d:%d]", time.day, time.mon, time.year, time.hour, time.min, time.sec);
  uint8_t LOG_BUF_offset = strlen((char*)LOG_BUF);
  // Добавление самого сообщения после временного штампа
  va_list ap;
  va_start(ap, string);
  int retval = vsnprintf((char*)LOG_BUF + LOG_BUF_offset, sizeof(LOG_BUF) - LOG_BUF_offset, string, ap);
  va_end(ap);
  uint8_t size = strlen((char*)LOG_BUF);
  // Запись в файл лога
  uint8_t opened = 0;

  if(Black_box.fopened("LOG", &opened) == BB_OPERATION_FAILED)
  {
    Black_box.Release();
    return;
  }
  if(opened)
  {
    Black_box.Release();
    return;
  }
  Black_box.fopen("LOG");
  Black_box.fwrite("LOG", LOG_BUF, size);
  Black_box.fsave();
  Black_box.fclose("LOG");
  Black_box.Release();
}

// Запись в лог
void BIN_BUFF_LOG(const uint8_t* bin, uint8_t bin_len, const char* string, ...)
{
#if defined(USE_UART_DEBUG_PORT)
#if !defined(__PRINTF_2)
  __DEBUG_UART_GRAB();
  va_list args;
  va_start(args, string);
  vprintf(string, args);
  va_end(args);
  for(uint16_t __idx=0; __idx<bin_len; __idx++)
  {
    printf("%02hhX", *(bin+__idx));
  }
  printf("\n");
  __DEBUG_UART_RELEASE();
#endif //!defined(__PRINTF_2)
#endif //USE_UART_DEBUG_PORT

  Black_box.Grab();
  // Запись временного штампа в начало каждого сообщения
  memset(LOG_BUF, 0x00, sizeof(LOG_BUF));
  uint32_t unixtime = GetUnixTime();
  time_struct_t time;
  conv_unixtime_to_time(&time, unixtime);
  sprintf((char*)LOG_BUF, "[%d/%d/%d][%d:%d:%d]", time.day, time.mon, time.year, time.hour, time.min, time.sec);
  uint16_t LOG_BUF_offset = strlen((char*)LOG_BUF);

  // Добавление самого сообщения после временного штампа
  va_list ap;
  va_start(ap, string);
  int retval = vsnprintf((char*)LOG_BUF + LOG_BUF_offset, sizeof(LOG_BUF) - LOG_BUF_offset, string, ap);
  va_end(ap);
  uint16_t size = strlen((char*)LOG_BUF);

  for(uint16_t __idx=0; __idx<bin_len; __idx++)
  {
    if(2*__idx+size+4>=sizeof(LOG_BUF)) break;
    sprintf((char*)&LOG_BUF[2*__idx+size], "%02hhX", bin[__idx]);
  }

  size = strlen((char*)LOG_BUF);

  LOG_BUF[size++]='\n';

  // Запись в файл лога
  uint8_t opened = 0;

  if(Black_box.fopened("LOG", &opened) == BB_OPERATION_FAILED)
  {
    Black_box.Release();
    return;
  }
  if(opened)
  {
    Black_box.Release();
    return;
  }
  Black_box.fopen("LOG");
  Black_box.fwrite("LOG", LOG_BUF, size);
  Black_box.fsave();
  Black_box.fclose("LOG");
  Black_box.Release();
}

#ifdef __cplusplus
extern "C" {
#endif

const uint8_t* get_cmd_ans_mac_key(void)
{
  //7E B1 31 40 66 EE B4 25 8A 96 C6 70 7E FF E0 B2 4E CE D9 46 D6 99 CD 28 1E 07 36 7A 21 9B AD 65
  static const uint8_t key[32]=
  {
    0x7E, 0xB1, 0x31, 0x40, 0x66, 0xEE, 0xB4, 0x25, 0x8A, 0x96, 0xC6, 0x70, 0x7E, 0xFF, 0xE0, 0xB2,
    0x4E, 0xCE, 0xD9, 0x46, 0xD6, 0x99, 0xCD, 0x28, 0x1E, 0x07, 0x36, 0x7A, 0x21, 0x9B, 0xAD, 0x65
  };

  return key;
}

//ID сервера, для которого требуется формирования подписанных ответов на команды с записью в черный ящик
uint8_t get_server_id_to_add_notify_to_blackbox(void)
{
  return 5; //BLE плата
}

//Функция использует статический буфер!
void add_notify_to_blackbox(const uint8_t server_id, const uint32_t notify_uid, const uint8_t nonce[8], const notify_state_t notify_state, const char* notify_mess)
{
  if(server_id!=get_server_id_to_add_notify_to_blackbox() || memcmp(nonce, "\x00\x00\x00\x00\x00\x00\x00\x00", 8)==0)
  {
    return;
  }

  static uint8_t buff[sizeof(TVega_bb_message)+sizeof(TVega_command_ans_w_sign)+MAX_NOTIFY_LEN+1];
  memset(buff, 0, sizeof(buff));

  TVega_command_ans_w_sign* ans=(TVega_command_ans_w_sign*)&buff[sizeof(TVega_bb_message)];

  ans->uid=notify_uid;
  ans->result=Convert_notify_state_from_cmd_result(notify_state);
  memcpy(ans->nonce, nonce, sizeof(ans->nonce));
  ans->arg_len=0;

  int s_len;
  s_len=snprintf((char*)&buff[0]+sizeof(TVega_command_ans_w_sign), sizeof(buff)-sizeof(TVega_command_ans_w_sign), notify_mess);
  if(s_len>=0)
  {
    ans->arg_len=s_len+1;
  }

  mbedtls_poly1305_mac(get_cmd_ans_mac_key(), &ans->nonce[0], sizeof(*ans)+ans->arg_len-sizeof(ans->mac), ans->mac);

  BB_message track_point;
  memset(&track_point, 0x00, sizeof(track_point));

  System.Grab();
  track_point.len = Pack_vega_message_w_custom_data(buff, sizeof(buff)-sizeof(sizeof(TVega_bb_message)), BLE_EXT_BOARD_COMMAND_ANS_W_SINGN, NULL /*уже скопировано*/, sizeof(*ans)+ans->arg_len);
  track_point.payload = buff;
  System.Release();

  if(System.connection_settings.server[0].server_protocol == VEGA) track_point.receiver_1 = 1;
  if(System.connection_settings.server[1].server_protocol == VEGA) track_point.receiver_2 = 1;
  if(System.connection_settings.server[2].server_protocol == VEGA) track_point.receiver_3 = 1;
  if(System.connection_settings.server[3].server_protocol == VEGA) track_point.receiver_4 = 1;

  if(track_point.receiver_1 || track_point.receiver_2 || track_point.receiver_3 || track_point.receiver_4)
  {
    Black_box.Grab();
    Black_box.add_message(&track_point);
    Black_box.Release();
  }
}
#ifdef __cplusplus
}
#endif