/**
  ******************************************************************************
  * File Name          : syscom_manager.h
  * Description        : Осуществляет внутрисистемную коммуникацию между
  *                      менеджерами, определяет логику работы устройства,
  *                      осуществляет конфигурирование устройства
  *
  ******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef Syscom_MANAGER_H
#define Syscom_MANAGER_H

/* Includes ------------------------------------------------------------------*/
#include "FreeRTOS.h"
#include "task.h"
#include "manager.h"
#include "Soft_timers/Soft_timers.h"
#include "Black_box/Black_box.h"
#include "System/system.h"
#include "boot_config.h"
#include "serv_protocols_lib/server_notifies.h"

/* Defines- ------------------------------------------------------------------*/
#define SYSCOM_MANAGER_NAME             "syscom_man" // Имя менеджера
#define SYSCOM_MANAGER_INPUT_CUEUE_SIZE  10 // Число элементов во входящей очереди менеджера
#define SYSCOM_MANAGER_PRIORITY           2 // Приоритет задачи менеджера
#define SYSCOM_MANAGER_STACK_SIZE       256 // Размер стэка менеджера в словах

// Настройки менеджера, передаются в базу конфигурации
typedef struct
{
  uint8_t version;
  uint8_t autorized_keys;     //
  uint16_t autorized_phones;  //
  uint8_t reserved[20];       //
} TSyscom_conf;

// Структура для отслеживания сценария
typedef struct
{
  float first_sensor_last_data;
  float second_sensor_last_data;
  bool  first_condition_occurs;
  bool  second_condition_occurs;
} TScenario_state;

// Структура для отслеживания таймерных выходов сценариев
typedef struct
{
  uint64_t output_switch_time[MAX_DIGITAL_OUTPUTS];
#if defined(BR_PRESENT)
  uint64_t ext_output_switch_time[MAX_EXT_DIGITAL_OUTPUTS];
#endif //BR_PRESENT
  uint8_t  return_output_to_state[MAX_DIGITAL_OUTPUTS];
#if defined(BR_PRESENT)
  uint8_t  return_ext_output_to_state[MAX_EXT_DIGITAL_OUTPUTS];
#endif //BR_PRESENT
} TScenario_timer_outputs;

/* C++ code-------------------------------------------------------------------*/
class CSyscom_manager: public CManager
{
public:
  // Конструктор, определяется имя менеджера, указатель на исходящую очередь, размер входящей очереди, приоритет
  CSyscom_manager(const char* mngr_name, QueueHandle_t* output_queue, uint8_t input_queue_size, uint8_t priority, TaskFunction_t task_starter, uint16_t stack_size);
  // Инициализация менеджера, реализует наследник
  void Init();
  // Деинициализация менеджера, реализует наследник
  void Deinit(void);
  // Главный цикл менеджера
  void Main_cycle(void);

  // shared funcions
  bool do_start_engine();
  bool do_stop_engine();
  bool do_open_doors();
  bool do_close_doors();
  bool do_off_auto_open_doors();
  bool do_yadrive_warming(uint8_t time_mins);
  bool do_fix_fuel_view();

private:
  // Указатель на очереди входящих сообщений других менеджеров
  QueueHandle_t* gps_queue;
  QueueHandle_t* server_queue;
  QueueHandle_t* server_answer_queue;
  QueueHandle_t* signal_queue;
  CSoft_timer bb_timer;
  CSoft_timer sensor_translation_timer;
  CSoft_timer save_settings_timer;
  CSoft_timer delayed_sleep_timer;
  CSoft_timer save_counters_timer;

  const char* name;
  // Признак необходимости рестарта
  uint8_t need_restart;
  // Признак необходимости сброса к заводским настройкам
  uint8_t need_tofactory;
  // Признак необходимости стирания черного ящика
  uint8_t need_bb_clean;
  // Признак необходимости сохранить настройки
  uint8_t need_save_settings;
  // Буфер для упаковки сообщений в ЧЯ
  uint8_t message_buff[BB_MESSAGE_MAXSIZE];
  // Массив таймеров для регулярной отправки данных датчиков на сервер
  uint16_t sensor_send_timer[MAX_SENSOR_COUNT];
  // Структура для взаимодействия с бутлоадером
  boot_cmd_t boot_cmd;
  // Признак необходимости экстренной остановки
  uint8_t need_emergency_stop;

  // Буффер для вывода лога при старте
#if defined(STM32F413_423xx)
  uint8_t logout_buff[1024];
#elif defined(STM32L433xx)
  uint8_t logout_buff[256];
#elif defined(STM32F10X_HD)
  uint8_t logout_buff[256];
#endif //defined(STM32F413_423xx) || defined(STM32F10X_HD)
  uint32_t sleep_by_ignition_timer;
  uint32_t sleep_by_stop_timer;
  uint32_t sleep_by_wakeup_timer;


  // Состояния сценариев
  TScenario_state scenario_state[MAX_SCENARIES];
  uint32_t scenario_last_photo;
  uint32_t scenario_last_sms;
  uint32_t scenario_last_call;
  TScenario_timer_outputs scenario_timer_outputs;

  // Заполнение текущего состояния системы
  void Fill_system_state(void);
  // Создать файл с настройками
  void Create_base_file(void);
  // Обработка сохранения настроек
  void Handle_save_settings(void);
  // Отработка спящего режима
  void Handle_sleep_mode(void);
  // Переход в спящий режим
  void go_to_sleep(uint8_t sleep_type = NORMAL_SLEEP);
  // Получение и разбор сообщений извне
  void Receive_message(void);
  // Обработка входящих сообщений от серверной задачи
  void Handle_server_messages(void);

  // Парсинг файла с настройками
  bool Parse_settings(const char* filename, const uint8_t w_sing = 0);

  // Нужно ли формировать запись ЧЯ для данного сервера
  bool Is_track_point_present(uint8_t server_idx);
  // Обработка входящих сообщений от GPS
  void Handle_gps_messages(void);
  // Обработка сообщений от менеджера сигналов
  void Handle_signal_messages(void);
  // Упаковка сообщений в ЧЯ
  uint16_t Pack_vega_message(uint16_t index = 0xFFFF);
  // Регулярная трансляция показаний датчиков
  void Translate_sensor_data(void);
  // Обработка сценариев
  void Handle_scenaries(void);
  // Обработка сообщений протокола Вега
  uint16_t Handle_vega_messages(uint8_t* message, uint8_t* answer_buff, uint16_t answer_buff_size);
  // Опаковка ответа на команду
  uint16_t Pack_vega_command_answer(const uint8_t server_idx,
                                    const EVega_message_type message_ans_type, const void* command_answer, uint8_t* answer_args,  uint8_t* buff, uint16_t max_size,
                                    const void* get_ans, uint16_t get_ans_len);
  // Вывод лога в отладочный порт
  void Logout(void);
  // Загрузка настроек
  void Load_settings(void);
  // Сохранение показаний счетчиков
  void Save_counters(void);

  // Диагностика NOR FTL
  void NOR_FTL_diag(void);

// blinker variables
private:
  bool need_blink = false;
  uint32_t blink_time_timer;
  uint32_t blink_sampling_timer;

  uint32_t blink_time_timeout;
  uint32_t blink_sampling_timeout;

  static const uint32_t default_time_timeout = 2 * SEC;
  static const uint32_t default_sampling_timeout = 5 * MS100;

  bool Start_blink();
  void Check_blink();
  bool Do_blink();
  void Clear_blink();

private:
  bool was_update_firmware = false;
};

/* C code---------------------------------------------------------------------*/
// Функция запуска Syscom менеджера
extern void Start_syscom_manager(void* argument);
// Для обеспечения видимости другими менеджерами
extern CSyscom_manager Syscom_manager;
// Логгирование
void LOG(const char* string, ...);
void BIN_BUFF_LOG(const uint8_t* bin, uint8_t bin_len, const char* string, ...);

#ifdef __cplusplus
extern "C" {
#endif
extern const uint8_t* get_cmd_ans_mac_key(void);
extern uint8_t get_server_id_to_add_notify_to_blackbox(void);
#ifdef __cplusplus
}
#endif

#endif //Syscom_MANAGER_H