"""
    Various functions and classes, which are used to validate group (when updating card, limits, ...).
"""

from collections import defaultdict, Counter
import time
import re

import core.hosts

GB = 1024 * 1024 * 1024.

class EValidateStatus(object):
    STATUS_OK = 0
    STATUS_FAIL = 1


class TValidateInfo(object):
    __slots__ = ['field', 'status', 'reason']

    def __init__(self, field, status, reason):
        self.field = field
        self.status = status
        self.reason = reason


# ===============================================================================
# checkers for single fields
# ===============================================================================

def check_reqs_hosts_location_location(validator, group, card, field_path, field_value):
    """
        Check if all group hosts are located in specified locations.
    """
    del card

    assert isinstance(field_value, list)

    # FIXME: specify all location in group by default
    if not field_value:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    # FIXME: add better support for location in dynamic recluster
    if group.card.master is not None and group.card.master.card.name in ('ALL_DYNAMIC', 'ALL_PERSONAL', 'ALL_SOX'):
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    unknown_locations = set(field_value) - set(group.parent.db.hosts.get_all_locations())
    if len(unknown_locations) > 0:
        msg = "Unknown locations <%s> (should be one of <%s>)" % (
            ",".join(unknown_locations), ",".join(group.parent.db.hosts.get_all_locations()))
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, msg)

    failed_hosts = filter(lambda x: x.location not in field_value, group.getHosts())
    if len(failed_hosts) > 0:
        msg = "Hosts <%s> (total %d) in wrong locations (other than <%s>)" % (
            ",".join(map(lambda x: x.name, failed_hosts[:20]))
            , len(failed_hosts)
            , ",".join(field_value)
        )
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def check_reqs_hosts_location_dc(validator, group, card, field_path, field_value):
    """
        Check if all group hosts are located in specified dcs.
    """
    del card

    assert isinstance(field_value, list)

    # FIXME: specify all location in group by default
    if not field_value:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    # FIXME: add better support for location in dynamic recluster
    if group.card.master is not None and group.card.master.card.name in ('ALL_DYNAMIC', 'ALL_PERSONAL'):
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    failed_hosts = filter(lambda x: x.dc not in field_value, group.getHosts())
    if len(failed_hosts) > 0:
        msg = "Hosts <%s> in wrong dcs (other than <%s>)" % (
            ",".join(map(lambda x: x.name, failed_hosts[:20])), ",".join(field_value))
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

def validate_memory_guarantee(validator, checked_group, card, field_path, field_value):
    """
        Check for memory_guarantee of all host instances do not exceed host memory

        :param checked_group: group to validate memory limit for
        :return TValidateInfo:
    """

    # skip validation for autogenerated/unworking/unsorted/... groups
    if (checked_group.card.on_update_trigger is not None) or \
       (checked_group.card.name == "ALL_UNWORKING") or \
       (checked_group.card.tags.metaprj == "unsorted"):
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    cur_db = checked_group.parent.db

    # get list of affected groups
    affected_groups = set()
    for host in checked_group.getHosts():
        affected_groups |= set(cur_db.groups.get_host_groups(host))
    background_groups_memory = 0
    # FIXME: temporary commented, do not know how to fix it quickly
    # for group in affected_groups:
    #     if group.card.on_update_trigger is not None:
    #         background_groups_memory += group.card.reqs.instances.memory_guarantee.value
    affected_groups = filter(lambda x: x.card.on_update_trigger is None, affected_groups)

    # calculate memory needed on every host
    needed_memory = defaultdict(int)
    for group in affected_groups:
        mv = group.card.reqs.instances.memory_guarantee.value
        for instance in group.get_kinda_busy_instances():
            needed_memory[instance.host] += mv

    # check limits
    for host in checked_group.getHosts():
        have_memory = host.get_avail_memory()
        if have_memory < needed_memory[host] + background_groups_memory:
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, "Host %s has <%.2f Gb> memory, while needed <%.2f Gb>" % (
                host.name, have_memory / GB, needed_memory[host] / GB))

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_cpu_guarantee(validator, checked_group, card, field_path, field_value):
    """
        Check for cpu guarantee of all host instances do not exceed host cpu

        :param checked_group: group to validate memory limit for
        :return TValidateInfo:
    """

    # skip validation for autogenerated/unworking/unsorted/... groups
    if (checked_group.card.on_update_trigger is not None) or \
       (checked_group.card.name == "ALL_UNWORKING") or \
       (checked_group.card.tags.metaprj == "unsorted") or \
       (checked_group.card.properties.full_host_group) or \
       (checked_group.card.properties.fake_group):
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    # skip validation of some groups with known cpu overcommit
    # OR
    # with uneven instance count
    # OR
    # with cpu_policy=idle
    overcommit_master_groups = ['MAN_WEB_BASE', 'SAS_WEB_BASE', 'VLA_YT_RTC', 'GENCFG_4375_MAN',
                                'MAN_SAAS_CLOUD', 'SAS_SAAS_CLOUD', 'VLA_SAAS_CLOUD', 'SAS_MARKET_TEST_GENERAL']
    if checked_group.card.name in overcommit_master_groups:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)
    if checked_group.card.master is not None and checked_group.card.master.card.name in overcommit_master_groups:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    # skip background groups
    if checked_group.card.properties.background_group:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    cur_db = checked_group.parent.db

    for host in checked_group.get_kinda_busy_hosts():
        needed_cpu = sum(x.power for x in cur_db.groups.get_host_instances(host))
        have_cpu = host.power
        if needed_cpu > have_cpu:
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, 'Host {} has <{:.3f}> cpu, while needed <{:.3f}> by instances'.format(
                host.name, have_cpu, needed_cpu))

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_host_property(validator, checked_group, card, field_path, field_value):
    """
        Check if host property for our group same to same property for parent group
    """

    master_group = checked_group.card.master
    if master_group is None:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    master_value = master_group.card.get_card_value(field_path)
    if master_value != field_value:
        msg = "Field <%s> is host-only, but group <%s> and <%s> have different values <%s> and <%s>" % (
            ".".join(field_path), checked_group.card.name, master_group.card.name, field_value, master_value)
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

def validate_shifted_group_impl(checked_group, field_path):
    def fix_host(cur_db, host):
        """
            This function fix host for virtual machines to get template host, e. g. sas1-1382-24418.vm.search.yandex.net -> sas1-1382.search.yandex.net
        """

        m = re.match("^(.*)-\d+\.vm\.(.*)$", host.name)
        if m:
            return cur_db.hosts.get_host_by_name("%s.%s" % (m.group(1), m.group(2)))
        else:
            return host

    def get_host_to_shards_mapping(cur_db, group):
        """
            Find shards on all hosts and create mapping: <host name> -> <list of shards>
        """
        mapping = defaultdict(set)
        for intlookup in map(lambda x: group.parent.db.intlookups.get_intlookup(x), group.card.intlookups):
            for shard_id in xrange(intlookup.get_shards_count()):
                shard_with_tier = intlookup.get_tier_and_shard_id_for_shard(shard_id)
                for instance in intlookup.get_base_instances_for_shard(shard_id):
                    mapping[fix_host(cur_db, instance.host)].add(shard_with_tier)
        return mapping

    cur_db = checked_group.parent.db

    template_group = cur_db.groups.get_group(checked_group.card.properties.nidx_for_group)

    checked_group_mapping = get_host_to_shards_mapping(cur_db, checked_group)
    template_group_mapping = get_host_to_shards_mapping(cur_db, template_group)

    for host in (set(checked_group_mapping.keys()) | set(template_group_mapping.keys())):
        extra_shards = checked_group_mapping[host] - template_group_mapping[host]
        if len(extra_shards) > 0:
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, "Groups <%s> and <%s> have different shards on host <%s>: extra shards <%s>" % (
                checked_group.card.name, template_group.card.name, host.name, ' '.join(str(x) for x in extra_shards)))

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

def validate_shifted_group(validator, checked_group, card, field_path, field_value):
    """
        Some groups are created as 'shifted' group (these groups have intlookup as 'master' group intlookup with
        instances on different ports
    """

    # create list of groups to check
    groups_to_check = []

    if (checked_group.card.properties.nidx_for_group is not None) and (len(checked_group.card.intlookups) > 0):
        groups_to_check.append(checked_group)

    if len(checked_group.card.intlookups) > 0:
        if checked_group.card.master is not None:
            checked_group_master = checked_group.card.master
        else:
            checked_group_master = checked_group
        for slave_group in checked_group_master.card.slaves:
            if (slave_group.card.properties.nidx_for_group == checked_group.card.name) and (len(slave_group.card.intlookups) > 0):
                groups_to_check.append(slave_group)

    # check calculated groups
    for group in groups_to_check:
        result = validate_shifted_group_impl(group, field_path)
        if result.status == EValidateStatus.STATUS_FAIL:
            return result

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

def validate_except_cpu_models(validator, checked_group, card, field_path, field_value):
    """
        Some models can be forbidden to use in some groups. Here we check this constraint
    """

    if len(field_value) == 0:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    forbidden_models = field_value
    hosts_with_wrong_model = filter(lambda x: x.model in forbidden_models, checked_group.getHosts())
    if len(hosts_with_wrong_model) > 0:
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, "Group <%s> has hosts <%s> with forbidden models <%s>" % (
            checked_group.card.name, ",".join(map(lambda x: x.name, hosts_with_wrong_model)), ",".join(forbidden_models)))
    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_cpu_models(validator, checked_group, card, field_path, field_value):
    """
        For some groups we can use cpu model of only specified types. Here we check this constraint
    """

    if len(field_value) == 0:
        return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)

    allow_models = field_value
    hosts_with_wrong_model = filter(lambda x: x.model not in allow_models, checked_group.getHosts())
    if len(hosts_with_wrong_model) > 0:
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, "Group <%s> has hosts <%s> with models other than <%s>" % (
            checked_group.card.name, ",".join(map(lambda x: x.name, hosts_with_wrong_model)), ",".join(allow_models)))
    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_group_owners(validator, checked_group, card, field_path, field_value):
    """
        Check if owners are valid users. Moreover check if owners of common groups belongs to relatively narrow group
        of search developers/admins/...
    """
    wrong_users = filter(lambda x: x not in validator.avail_users, field_value)
    if len(wrong_users) > 0:
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, "Can not add users <%s> to group <%s> (they do not exist or can not be added for some other reason)" % (
            ",".join(wrong_users), checked_group.card.name))
    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_hbf_parent_macros(validator, checked_group, card, field_path, field_value):
    """Check if hbf parent macros satisfy regexp"""

    if field_value is not None and not checked_group.parent.db.hbfmacroses.has_hbf_macros(field_value):
        msg = 'Can not set macro <{}> as parent for <{}>:  macro not found in gencfg database'.format(field_value, checked_group.card.name)
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_metaprj(validator, checked_group, card, field_path, field_value):
    """Check if metaprj <unsorted> is applied only to unsorted groups (RX-111)"""

    if field_value == 'unsorted' and not (checked_group.card.name.startswith('ALL_UNSORTED')):
        fail_msg = 'Group {}: set metaprj to <unsorted> for non-unsorted group'.format(checked_group.card.name)
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_audit_cpu_service_groups(validator, checked_group, card, field_path, field_value):
    """Validate list of service groups for cpu audit (RX-160)

    Validate list of service groups:
        - check if our group in this list;
        - check that service_groups of our group is same to service_group of other groups"""

    if len(field_value):
        # check if our group in list
        if checked_group.card.name not in field_value:
            fail_msg = 'group missed in <{}> list <{}>'.format('.'.join(field_path), ','.join(field_value))
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)

        # check for other group list same as our group list
        for other_group_name in field_value:
            if not checked_group.parent.has_group(other_group_name):
                fail_msg = 'Group {} from <{}> does not exists'.format(other_group_name, '.'.join(field_path))
                return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)
        other_group = checked_group.parent.get_group(other_group_name)
        if set(field_value) != set(other_group.card.get_card_value(field_path)):
            fail_msg = 'Groups {} and {} have different serive groups in <{}>: <{}> and <{}> respectively'.format(checked_group.card.name, other_group.card.name,
                    '.'.join(field_path), ' '.join(field_value), ' '.join(other_group.card.get_card_value(field_path)))
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_ipip6_ext_tunnel(validator, checked_group, card, field_path, field_value):
    """Validate ipv4 tunnels info (GENCFG-1077)"""

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_ipip6_ext_tunnel_v2(validator, checked_group, card, field_path, field_value):
    """Validate ipv4 tunnels info (GENCFG-1077)"""

    if field_value:
        # check for mutual exlusive with old scheme
        if card.properties.ipip6_ext_tunnel:
            fail_msg = 'Group {} flags <properties.ipip6_ext_tunnel> and <properties.ipip6_ext_tunnel_v2> can not be set both at the same time'
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)

        # check that we have vlans for all hosts
        failed_hosts = [x for x in checked_group.getHosts() if 'vlan688' not in x.vlans]
        if failed_hosts:
            fail_msg = 'Group {} with flag <properties.ipip6_ext_tunnel_v2> has hosts without vlans: {}'.format(checked_group.card.name,
                    ','.join(x.name for x in failed_hosts))
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_hbf_slb_names(validator, checked_group, card, field_path, field_value):
    """Check if slb name is in our database alreardy (GENCFG-1781)"""
    for slb_name in field_value:
        if not checked_group.parent.db.slbs.has(slb_name):
            fail_msg = 'Group {} has slb name <{}> which is not present in gencfg'.format(checked_group.card.name, slb_name)
            return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


def validate_ipip6_ext_tunnel_pool_name(validator, checked_group, card, field_path, field_value):
    """Check for valid pool name (GENCFG-2240)"""
    avail_pool_names = sorted(checked_group.parent.db.ipv4tunnels.pool.keys())
    if field_value not in avail_pool_names:
        fail_msg = 'Group {} ipv4tunnel pool <{}> is not found in gencfg: <{}> available'.format(
                checked_group.card.name, field_value, ','.join(avail_pool_names)
        )
        return TValidateInfo(field_path, EValidateStatus.STATUS_FAIL, fail_msg)

    return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)


# ================================================================================
# main validator class
# ================================================================================
class TValidator(object):
    """
        Static class with bunch of falidation functions
    """

    SINGLE_FIELD_MAPPING = {
        ("reqs", "hosts", "location", "location"): check_reqs_hosts_location_location,
        ("reqs", "hosts", "location", "dc"): check_reqs_hosts_location_dc,
        ("reqs", "instances", "memory_guarantee"): validate_memory_guarantee,
        ("legacy", "funcs", "instancePower"): validate_cpu_guarantee,
        ("reqs", "hosts", "cpu_models"): validate_cpu_models,
        ("reqs", "hosts", "except_cpu_models"): validate_except_cpu_models,
        ("properties", "yasmagent_prestable_group"): validate_host_property,
        ("properties", "yasmagent_production_group"): validate_host_property,
        ("properties", "nonsearch"): validate_host_property,
        ("properties", "nidx_for_group"): validate_shifted_group,
        ("properties", "hbf_parent_macros"): validate_hbf_parent_macros,
        ("properties", "ipip6_ext_tunnel"): validate_ipip6_ext_tunnel,
        ("properties", "ipip6_ext_tunnel_v2"): validate_ipip6_ext_tunnel_v2,
        ("properties", "ipip6_ext_tunnel_pool_name"): validate_ipip6_ext_tunnel_pool_name,
        ("owners",):  validate_group_owners,
        ("managers",): validate_group_owners,
        ("developers",): validate_group_owners,
        ("watchers",): validate_group_owners,
        ("tags", "metaprj"): validate_metaprj,
        ('audit', 'cpu', 'service_groups'): validate_audit_cpu_service_groups,
        ('properties', 'mtn', 'tunnels', 'hbf_slb_name'): validate_hbf_slb_names,
    }

    def __init__(self, db):
        avail_users = [x.name for x in db.users.get_users()] + [x.name for x in db.staffgroups.get_all()]
        if db.version < '2.2.56':
            # =========================================== RX-447 START ===========================================
            for abcgroup in db.abcgroups.get_all():
                avail_users.append('abc:{}'.format(abcgroup.name))
                for role_id in abcgroup.roles:
                    avail_users.append('abc:{}:role={}'.format(abcgroup.name, role_id))
            # =========================================== RX-447 START ===========================================
        else:
            avail_users += list(db.abcgroups.available_abc_services())
        self.avail_users = set(avail_users)

    def validate_group_card_field(self, group, card, field_path, field_value):
        """
            Check if suggested <field_value> suits group requirements.

            :type group: core.igroups.IGroup
            :type card: core.card.Node
            :type field_path: list[str]
            :type field_value: T

            :param group: Group to validate card field for
            :param card: Group card (usually <group.card> if we validate specified group)
            :param field_path: path to field in group card to validate
            :param field_value: suggest value

            :return (core.igroup_validators.TValidateInfo): validate info
        """

        if tuple(field_path) not in TValidator.SINGLE_FIELD_MAPPING:
            return TValidateInfo(field_path, EValidateStatus.STATUS_OK, None)
        return TValidator.SINGLE_FIELD_MAPPING[tuple(field_path)](self, group, card, field_path, field_value)

    @staticmethod
    def validate(group):
        """
            Run validators, which are not connected to single field
        """

        # validate memory limits
        result = TValidator.validate_memory_limit_properties(group)
        if result.status == EValidateStatus.STATUS_FAIL:
            return result

        # validate volumes
        result = TValidator.validate_volumes(group)
        if result.status == EValidateStatus.STATUS_FAIL:
            return result

        return TValidateInfo(None, EValidateStatus.STATUS_OK, None)

    @staticmethod
    def validate_volumes(group):
        if hasattr(group.card.reqs, 'volumes'):
            for volume_info in group.card.reqs.volumes:
                # check that mount_point_workdir volume does not have guest_mp (GENCFG-1760)
                if volume_info.mount_point_workdir and (volume_info.guest_mp != ""):
                    fail_msg = 'Group {} volume with guest_mp <{}> is workdir volume (guest_mp must be <""> for workir volumes'.format(group.card.name, volume_info.guest_mp)
                    return TValidateInfo(['reqs', 'volumes'], EValidateStatus.STATUS_FAIL, fail_msg)
            # check that volumes have unique guest_mp
            guest_mps = [x.guest_mp for x in group.card.reqs.volumes]
            non_unique_guest_mps = [item for item, count in Counter(guest_mps).items() if count > 1]
            if non_unique_guest_mps:
                fail_msg = 'Group {} has volumes with same guest_mp <{}> (more than one volume with this guest_mp)'.format(group.card.name, ','.join(sorted(non_unique_guest_mps)))
                return TValidateInfo(['reqs', 'volumes'], EValidateStatus.STATUS_FAIL, fail_msg)

            # ======================================= GENCFG-2300 START ==========================================
            if len(guest_mps):
                if '/' not in guest_mps:
                    fail_msg = 'Missing root volume </> in group {}'.format(group.card.name)
                    return TValidateInfo(['reqs', 'volumes'], EValidateStatus.STATUS_FAIL, fail_msg)
                if '' not in guest_mps:
                    fail_msg = 'Missing workdir volume <> in group {}'.format(group.card.name)
                    return TValidateInfo(['reqs', 'volumes'], EValidateStatus.STATUS_FAIL, fail_msg)
            # ======================================= GENCFG-2300 FINISH =========================================

        return TValidateInfo(['reqs', 'volumes'], EValidateStatus.STATUS_OK, None)

    @staticmethod
    def validate_memory_limit_properties(group):
        """
            Check if memory limit fields (like reqs.instances.memory_guarantee, properies.fake_group,
            properties.full_host_group).
        """

        # check group properties for contradictions
        if (group.card.properties.fake_group) and (group.card.master is not None):
            msg = "Slave group <%s> can not be fake group" % group.card.name
            return TValidateInfo(None, EValidateStatus.STATUS_FAIL, msg)
        if (group.card.properties.full_host_group) and (group.card.master is not None):
            msg = "Slave group <%s> can not be full host group" % group.card.name
            return TValidateInfo(None, EValidateStatus.STATUS_FAIL, msg)
        if (group.card.properties.fake_group) and (group.card.properties.full_host_group):
            msg = "Group <%s> can not be fake and full_host group at the same time" % group.card.name
            return TValidateInfo(None, EValidateStatus.STATUS_FAIL, msg)
        if (not group.card.properties.fake_group) and \
           (not group.card.properties.full_host_group) and \
           (group.card.reqs.instances.memory_guarantee.value == 0):
            msg = ("Group <%s> should be either fake group, full host group, or group with memory limits (specify "
                   "properties.fake_group=True or <properties.full_host_group=True or reqs.instances.memory_guarantee=12Gb)") % group.card.name
            return TValidateInfo(None, EValidateStatus.STATUS_FAIL, msg)

        return TValidateInfo(None, EValidateStatus.STATUS_OK, None)
