"""
    Auxiliarily function to deal with portovm instances. This functions include creating guest groups/hosts from real host as argument.
    This module should not include nothing from core, because it is imported from core code

    FIXME: move to core
"""

import copy
import math

from core.hosts import IS_VM_GUEST_CYTHON, IS_HWADDR_GENERATED_CYTHON, IS_IPV6_GENERATED_CYTHON


class Dummy(object):
    __slots__ = ['params']

    def __init__(self, params):
        self.params = params


def gen_guest_group_name(host_group_name):
    return host_group_name + "_GUEST"


def gen_guest_group(vm_group):
    guest_group_name = gen_guest_group_name(vm_group.card.name)

    guest_group = vm_group.parent.db.groups.add_group(guest_group_name,
                                                      description="Autogenerated group of virtual hosts related with %s" % vm_group.card.name,
                                                      owners=vm_group.card.owners,
                                                      tags=dict(prj=vm_group.card.tags.prj, itype='none',
                                                                ctype=vm_group.card.tags.ctype))

    if vm_group.card.legacy.funcs.instancePower.startswith('exactly'):
        guest_group.card.legacy.funcs.instancePower = vm_group.card.legacy.funcs.instancePower
    else:
        vm_group_powers = {x.power for x in vm_group.get_instances()}
        if len(vm_group_powers) == 1:
            guest_group.card.legacy.funcs.instancePower = 'exactly{}'.format(int(vm_group_powers.pop()))
        else:
            guest_group.card.legacy.funcs.instancePower = vm_group.card.legacy.funcs.instancePower

    guest_group.card.properties.nonsearch = vm_group.card.properties.nonsearch
    guest_group.card.properties.export_to_cauth = vm_group.card.properties.export_to_cauth
    guest_group.card.properties.yasmagent_prestable_group = False
    guest_group.card.properties.yasmagent_production_group = False
    guest_group.card.properties.full_host_group = False
    guest_group.card.properties.fake_host_group = False
    guest_group.card.properties.created_from_portovm_group = vm_group.card.name
    guest_group.card.reqs.instances.memory_guarantee = vm_group.card.reqs.instances.memory_guarantee
    guest_group.card.reqs.instances.memory_overcommit = vm_group.card.reqs.instances.memory_overcommit

    # add guest group card properties
    vm_group.card.guest.hbf_project_id = guest_group.card.properties.hbf_project_id

    return guest_group


def gen_vm_host_name(db, src_instance):
    src_host = src_instance.host

    return "%s-%s.vm.search.yandex.net" % (src_host.get_short_name(), src_instance.port)


def gen_vm_host(db, src_instance):
    src_host = src_instance.host
    host_group = db.groups.get_group(src_instance.type)

    reqs = host_group.card.reqs
    new_host_name = gen_vm_host_name(db, src_instance)

    if not db.hosts.has_host(new_host_name):
        new_host = copy.copy(src_host)
        new_host.name = new_host_name
        new_host.domain = new_host_name[len(new_host.name.split('.')[0]):]

        new_host.power = reqs.instances.power
        if new_host.power == 0:
            new_host.power = src_instance.host.power

        new_host.disk = reqs.instances.disk.gigabytes()
        new_host.ssd = reqs.instances.ssd.gigabytes()
        new_host.memory = int(math.ceil(reqs.instances.memory_guarantee.gigabytes()))
        # TODO: use mapping
        new_host.ipv4addr = "unknown"
        new_host.ipv6addr = "unknown"
        new_host.vlan = 1347
        new_host.ipmi = 1
        new_host.os = "unknown"
        new_host.n_disks = 1
        new_host.invnum = ""
        new_host.vmfor = src_instance.host.name

        # priemka host groups ipv6 addrs should be generated according to https://st.yandex-team.ru/GENCFG-307
        if db.groups.get_group(src_instance.type).card.tags.ctype == 'priemka':
            new_host.flags = IS_VM_GUEST_CYTHON | IS_HWADDR_GENERATED_CYTHON | IS_IPV6_GENERATED_CYTHON
        else:
            new_host.flags = IS_VM_GUEST_CYTHON | IS_HWADDR_GENERATED_CYTHON

        db.hosts.add_host(new_host)

        import utils.pregen.update_hosts

        new_host.ipv6addr = utils.pregen.update_hosts.detect_ipv6addr(Dummy({'custom_hostname': new_host.name}))
        new_host.hwaddr = utils.pregen.update_hosts.detect_hwaddr(Dummy({'custom_hostname': new_host.name}))

    return db.hosts.get_host_by_name(new_host_name)


def guest_instance(instance, db=None, raise_non_portovm=True):
    """Return guest instance for normal instance"""

    if db is None:
        from core.db import CURDB
        db = CURDB

    if guest_instance.cached_db != db:
        guest_instance.cached_db = db
        guest_instance.cached_data.clear()

    if instance not in guest_instance.cached_data:
        group = db.groups.get_group(instance.type)

        from core.instances import Instance

        # create fake host
        fake_host = copy.deepcopy(instance.host)
        fake_host.name = gen_vm_host_name(db, instance)
        fake_host.domain = '.{}'.format(fake_host.name.partition('.')[2])
        fake_host.disk = group.card.reqs.instances.disk.gigabytes()
        fake_host.ssd = group.card.reqs.instances.ssd.gigabytes()
        fake_host.memory = group.card.reqs.instances.memory_guarantee.gigabytes()
        if group.card.tags.ctype == 'priemka':
            fake_host.flags = IS_VM_GUEST_CYTHON | IS_HWADDR_GENERATED_CYTHON | IS_IPV6_GENERATED_CYTHON
        else:
            fake_host.flags = IS_VM_GUEST_CYTHON | IS_HWADDR_GENERATED_CYTHON

        # create fake instance
        if hasattr(group.card, 'guest') and hasattr(group.card.guest, 'port'):
            fake_instance_port = group.card.guest.port
        else:
            fake_instance_port = 8041
        fake_instance = Instance(fake_host, instance.power, fake_instance_port, '{}_GUEST'.format(instance.type), 0)

        guest_instance.cached_data[instance] = fake_instance

    return guest_instance.cached_data[instance]
guest_instance.cached_db = None
guest_instance.cached_data = dict()


def may_be_guest_instance(instance, db=None):
    """Check if instance belongs to portovm groups and replace it with 'guest' instance (RX-141)"""

    if db is None:
        from core.db import CURDB
        db = CURDB
    group = db.groups.get_group(instance.type)

    if instance.type == 'FAKE':
        return instance
    if not group.has_portovm_guest_group():
        return instance

    from core.instances import FakeInstance
    from core.hosts import FakeHost

    # create fake host
    fake_host = FakeHost(gen_vm_host_name(db, instance), ref_host=instance.host)

    # create fake instance
    if hasattr(group.card, 'guest') and hasattr(group.card.guest, 'port'):
        fake_instance_port = group.card.guest.port
    else:
        fake_instance_port = 8041
    fake_instance = FakeInstance(fake_host, fake_instance_port, instance.host.power)
    fake_instance.type = '{}_GUEST'.format(instance.type)

    return fake_instance
