#!/skynet/python/bin/python

import os
import sys

sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..')))

import gencfg
from core.argparse.parser import ArgumentParserExt
import core.argparse.types as argparse_types
from core.db import DB, CURDB

from parser import BlinovParser
from transport import TMultiTransport

RESULT_TYPES = ["result", "counts", "ast"]


def get_parser():
    parser = ArgumentParserExt(description="New blinov script (support different types of transports)", usage="""
Usage:
    %(prog)s -f "itag=MSK_WEB_BASE AND itag=a_tier_PlatinumTier0" -r result -t slf
    %(prog)s -f "[I@ALL_SEARCH . I@a_geo_sas] [I@ALL_SEARCH . I@a_geo_man] [I@ALL_SEARCH . I@a_geo_msk]" -r counts -t cms
    %(prog)s -f "I@MSK_WEB_BASE AND [I@a_itype_base I@a_prj_web-main] MINUS [I@a_geo_msk AND I@a_tier_PlatinumTier0]" -r ast -t cms
    %(prog)s -f "itag=a_itype_base . itag=a_prj_web-main" -r result -t gg
    %(prog)s -f "itag=MSK_RESERVED@slf XOR itag=MSK_RESERVED@slf:http://cmsearch.yandex.ru/res/gencfg/releases/29185208/w-generated/searcherlookup.conf" -t slf
""")
    parser.add_argument("-f", "--formula", type=str, required=True,
                        help="Obligatory. Formula for blinovcalc or HEAD.conf calc.")
    parser.add_argument("-r", "--result-type", type=str, required=True,
                        choices=RESULT_TYPES,
                        help="Obligatory. Output type: (instances, ast, ast with counts, etc.) (one of %s)" % (
                        ",".join(RESULT_TYPES)))
    parser.add_argument("-t", "--transport", type=str, required=True,
                        help="Obligatory. Database with instances. Examples: <cms> - Cms HEAD, <sl> - autogenerated searcherlookup, <slf> - searcherlookup at default path (w-generated/searcherlookup.conf), <gg> - gencfg groups, <cms@CONF_NAME> - Cms CONF_NAME, <slf:file path> - path to searcherlookup, <gg:tag name> - gencfg group from specific tag, <gg:dir path> - gencfg groups from db, located on <dir path>")
    parser.add_argument("-c", "--config", type=argparse_types.pythonlambda, default={},
                        help="Optional. Config for selected transport (usually dict)")

    return parser


def normalize(options):
    options.transport = TMultiTransport(options.transport)


def main(options):
    blinov_parser = BlinovParser()

    ast = blinov_parser.parse(options.formula)

    if options.result_type == "ast":
        return ast
    elif options.result_type == "counts":
        return ast.counts(options.transport)
    elif options.result_type == "result":
        return list(ast.iterate(options.transport))
    else:
        raise Exception("Unknown result type %s" % options.result_type)


def print_result(options, result):
    if options.result_type != "result":
        print result
    else:
        print " ".join(map(lambda x: str(x), result))


if __name__ == '__main__':
    options = get_parser().parse_cmd()

    normalize(options)

    result = main(options)

    print_result(options, result)
