#!/skynet/python/bin/python
"""
    Theoretically, compuatational loads for every shard of tier are equal. But when we distribute instances between shards, could be situation, when some hosts
    get less requests than other. For example
        - we have two instances on host and decide, that both should receive same RPS. But it turned out, that we have not room in intlookup for one of instance, thus cumputational load of this host is halved.  This result some loss of computational power due to non-ideal distribution.

    This sctip evaluate such type of losses.
"""

import os
import sys

sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..')))

import math
from argparse import ArgumentParser

import gencfg
from core.db import CURDB
import core.argparse.types as argparse_types
from gaux.aux_weights import adjust_weights_with_snippet_load


class InstanceUsage(object):
    def __init__(self, instance, tier, shard_id, usage):
        self.instance = instance
        self.tier = tier
        self.shard_id = shard_id
        self.usage = usage

    def show(self, verbose=False):
        if verbose:
            return "%s:%s, tier %s, shard %s, power %s, usage %s" % (
            self.instance.host.name, self.instance.port, self.tier, self.shard_id, self.instance.power, self.usage)
        else:
            return "%s:%s, power %s, usage %s" % (
            self.instance.host.name, self.instance.port, self.instance.power, self.usage)


class HostUsage(object):
    def __init__(self, host):
        self.host = host
        self.instances = []

        # values calculated from instances
        self.total_instances_power = 0.0
        self.total_instances_usage = 0.0

    def append(self, instance):
        self.instances.append(instance)
        self.total_instances_power += instance.instance.power
        self.total_instances_usage += instance.usage

    def lo(self):
        return self.total_instances_usage / self.host.power

    def show(self, verbose=False):
        instances_str = " ".join(map(lambda x: "(%s)" % x.show(verbose), self.instances))
        return "Host %s, hpower %s, ipower %s, iusage %s, overload %.3f\n    %s" % (
            self.host.name, self.host.power, self.total_instances_power, self.total_instances_usage,
            self.total_instances_usage / self.host.power, instances_str)


def parse_cmd():
    parser = ArgumentParser(description="Show theoretical host usage distribution (calculated on sas configs data)")
    parser.add_argument("-c", "--sasconfig", dest="sasconfig", type=argparse_types.sasconfig, default=None,
                        required=True,
                        help="Obligatory. File with sas optimizer config")
    parser.add_argument("-r", "--report-type", type=str, required=True,
                        choices=["brief", "per_host", "most_underloaded", "most_overloaded"],
                        help="Oblibatory. Report type")
    parser.add_argument("-v", "--verbose", action="count", dest="verbose_level", default=0,
                        help="Optional. Verbose mode. Multiple -v options increase the verbosity. The maximum is 1.")
    parser.add_argument("-n", "--topn", type=int, default=1,
                        help="Optional. Option for report_types <most_underloaded> <most_overloaded> (number of hosts to show)")
    parser.add_argument("--ssd-load", type=float, default=0.0,
                        help="Optional. Simulate extra load generated by snippet requests. <1.0> means snippet load equals to search load")
    parser.add_argument("--adjust-ssd-weights", action="store_true", default=False,
                        help="Optional. Calculate case when instances weights adjusted with snippets load")
    parser.add_argument("--dispersion-limit", type=float, default=None,
                        help="Optional. Exit with status 1 if dispersion0 is more than specified one")

    if len(sys.argv) == 1:
        sys.argv.append('-h')
    options = parser.parse_args()

    return options


def main(options):
    # adjust sasconfig <power> to real power units
    total_config_power = 0.0
    total_intlookup_power = 0.0
    for elem in options.sasconfig:
        elem.intlookup = CURDB.intlookups.get_intlookup(elem.intlookup)

        total_config_power += elem.power * CURDB.tiers.tiers[elem.tier].get_shards_count()
        total_intlookup_power += sum(map(lambda x: x.power, elem.intlookup.get_used_base_instances()))
    for elem in options.sasconfig:
        elem.power *= total_intlookup_power / total_config_power

    instance_usage_by_host = dict()
    for elem in options.sasconfig:
        for i in range(elem.intlookup.get_shards_count()):
            basesearch_replicas = elem.intlookup.get_base_instances_for_shard(i)

            if options.adjust_ssd_weights:
                basesearch_adjusted_weights = adjust_weights_with_snippet_load(
                    map(lambda x: (x.power, x.host.ssd > 0), basesearch_replicas), options.ssd_load)
            else:
                basesearch_adjusted_weights = map(lambda x: x.power, basesearch_replicas)

            bassearch_total_search_weight = sum(basesearch_adjusted_weights)

            baseseach_ssd_N = len(filter(lambda x: x.host.ssd > 0, basesearch_replicas))
            if options.ssd_load > 0 and baseseach_ssd_N == 0:
                raise Exception("Found shard with not ssd replicas shilw ssd-load more than zero")

            elem_search_power = elem.power * (1 - options.ssd_load)
            elem_snippet_power = elem.power * options.ssd_load

            for instance, instance_adjusted_weight in zip(basesearch_replicas, basesearch_adjusted_weights):
                if instance.host not in instance_usage_by_host:
                    instance_usage_by_host[instance.host] = HostUsage(instance.host)

                # calculate instance usage separately by search and snippet queries
                instance_search_usage = instance_adjusted_weight / bassearch_total_search_weight * elem_search_power
                if baseseach_ssd_N > 0:
                    instance_snippet_usage = elem_snippet_power / baseseach_ssd_N * int((instance.host.ssd > 0))
                else:
                    assert (elem_snippet_power == 0)
                    instance_snippet_usage = 0.0
                instance_usage = instance_search_usage + instance_snippet_usage

                instance_usage = InstanceUsage(instance, elem.tier, i, instance_usage)
                instance_usage_by_host[instance.host].append(instance_usage)
    instance_usage_by_host = instance_usage_by_host.values()

    return instance_usage_by_host


def show_result(result, options):
    verbose = (options.verbose_level > 0)

    if options.report_type == "per_host":
        for host_usage in result:
            print host_usage.show(verbose=verbose)
    elif options.report_type == "brief":
        total_hosts = len(result)

        dispersion0 = sum(map(lambda x: math.fabs(1.0 - x.lo()), result)) / len(result)

        sx = sum(map(lambda x: x.lo(), result))
        sx2 = sum(map(lambda x: x.lo() * x.lo(), result))
        dispersion = (sx2 - sx * sx / len(result)) / len(result)

        min_overload = min(map(lambda x: x.lo(), result))
        max_overload = max(map(lambda x: x.lo(), result))

        print "Total hosts: %s" % total_hosts
        print "Dispersion0: %s" % dispersion0
        print "Dispersion: %s" % dispersion
        print "Minimal usage: %s" % min_overload
        print "Maximal usage: %s" % max_overload
    elif options.report_type in ["most_overloaded", "most_underloaded"]:
        reverse = options.report_type == "most_overloaded"
        result.sort(cmp=lambda x, y: cmp(x.lo(), y.lo()), reverse=reverse)

        for i in range(min(len(result), options.topn)):
            print result[i].show(verbose=verbose)
    else:
        raise Exception("Unknown report type %s" % options.report_type)


if __name__ == '__main__':
    options = parse_cmd()

    result = main(options)

    show_result(result, options)

    if options.dispersion_limit is not None:
        dispersion0 = sum(map(lambda x: math.fabs(1.0 - x.lo()), result)) / len(result)
        if dispersion0 > options.dispersion_limit:
            sys.exit(1)
