import os
import yt.wrapper as yt
import yt.yson as yson
import datetime

YT_OAUTH = os.environ['yt_token']


def make_hahn_client():
    client = yt.client.YtClient(proxy=None, token=None, config=None)
    client.config["proxy"]["url"] = "hahn"
    client.config["token"] = YT_OAUTH
    return client


def make_markov_client():
    client = yt.client.YtClient(proxy=None, token=None, config=None)
    client.config["proxy"]["url"] = "markov"
    client.config["token"] = YT_OAUTH
    return client


YT_CLIENT = make_hahn_client()
YT_MARKOV_CLIENT = make_markov_client()


class SimpleYTBasedCollector:
    """Default YT class. Should to describe fields in schema attribute"""

    def __init__(self, yt_table_name, recreate=False, *args, **kwargs):
        self.yt_table_name = yt_table_name
        self.yt_table = '//home/helpdesk/{}'.format(self.yt_table_name)
        self.schema = []
        self.recreate = recreate
        self.yt_client = kwargs.get('client', yt)
        self.is_table_dynamic = kwargs.get('is_table_dynamic', True)
        self.static_append = kwargs.get('static_append', True)

    def __create_table(self):
        print('Creating YT Table')
        self.yt_client.create("table", self.yt_table, attributes={
                              "schema": self.schema, "dynamic": self.is_table_dynamic}, force=True)

        if self.is_table_dynamic:
            self.yt_client.mount_table(self.yt_table)

    def write_data_to_yt(self, data):
        self.check_or_create_table()

        if self.is_table_dynamic:
            self.yt_client.mount_table(self.yt_table)
            data_len = int(len(data) / 100000) + 1

            for num in range(data_len):
                start = num * 100000
                end = (num + 1) * 100000
                self.yt_client.insert_rows(self.yt_table, data[start:end],
                                           update=True, format=yt.JsonFormat(
                    attributes={"encode_utf8": False}))
        else:
            table = yt.TablePath(name=self.yt_table, append=self.static_append)
            self.yt_client.write_table(table, data,
                                       format=yt.JsonFormat(attributes={
                                           "encode_utf8": False}))

    def check_or_create_table(self):
        if not self.yt_client.exists(self.yt_table) or self.recreate:
            print(
                'Cannot find table with path {} creating it...'.format(self.yt_table))
            self.__create_table()


class STStaticSyncer(SimpleYTBasedCollector):
    def __init__(self):
        self.date_now = datetime.datetime.now().strftime('%d-%m-%Y')
        super().__init__(yt_table_name='monitoring/st_issues_static',
                         recreate=True,
                         client=YT_CLIENT,
                         is_table_dynamic=False,
                         static_append=False
                         )
        self.schema = [
            {"name": "key", "required": True, "type": "string"},
            {"name": "resolved", "required": False, "type": "int64"},
            {"name": "created", "required": False, "type": "int64"},
            {"name": "type", "required": True, "type": "string"},
            {"name": "fixVersions", "required": False, "type": "int64"},
            {"name": "abcservice", "required": False, "type": "int64"},
            {"name": "sla_604_spent", "required": False, "type": "int64"},
            {"name": "sla_604_status", "required": False, "type": "string"},
            {"name": "sla_605_spent", "required": False, "type": "int64"},
            {"name": "sla_605_status", "required": False, "type": "string"},
            {"name": "resolution", "required": False, "type": "string"},
            {"name": "assignee", "required": False, "type": "string"},
            {"name": "marked", "required": False, "type": "string"},
            {"name": "line", "required": False, "type": "string"},
            {"name": "status", "required": False, "type": "int64"},
            {"name": "customFields", "required": False, "type": "string"},
            {"name": "components", "required": False, "type": "any"},
            {"name": "tags", "required": False, "type": "any"},
            {"name": "last_update", "required": False, "type": "int64"}
        ]

        self.dict_ver = self._fetch_data_st_fixvers()
        self.dict_components = self._fetch_data_st_components()

    def _get_schema(self, table_name):
        raw_schema = YT_MARKOV_CLIENT.get_attribute(table_name, "schema")
        data = yson.convert.yson_to_json(raw_schema)["$value"]
        for col_info in data:
            for key in col_info:
                value = col_info[key]
                if isinstance(col_info[key], str):
                    col_info[key] = {'true': True,
                                     'false': False}.get(value, value)
        return data

    def _fetch_data(self):
        print("Start fetching data ST from //home/startrek/tables/prod/yandex-team/queue/HDRFS/issues")

        exclude = ['type', 'resolution', 'abcservice', 'sla_604_spent', 'sla_605_spent',
                   'sla_604_status', 'sla_605_status', 'marked', 'line', 'last_update', 'priority',
                   'status', 'assignee', 'resolver', 'author', 'modifier']

        fields = []
        self.st_issues_schema = self._get_schema(
            "//home/startrek/tables/prod/yandex-team/queue/HDRFS/issues")
        for col_info in self.st_issues_schema:
            if col_info['name'] not in exclude:
                fields.append('st_issues.{name} as {name}'.format(name=col_info['name']))

        query = '{}, ' \
                'st_types.key as type, ' \
                'st_resolutions.key as resolution, ' \
                'st_priorities.shortId as priority, ' \
                'st_statuses.shortId as status, ' \
                'st_users.login as assignee, ' \
                'st_users1.login as resolver, ' \
                'st_users2.login as author, ' \
                'st_users3.login as modifier ' \
                'FROM [//home/startrek/tables/prod/yandex-team/queue/HDRFS/issues] as st_issues ' \
                'JOIN [//home/startrek/tables/prod/yandex-team/common/types] as st_types ' \
                'ON st_issues.type = st_types.id ' \
                'LEFT JOIN [//home/startrek/tables/prod/yandex-team/common/resolutions] as st_resolutions ' \
                'ON st_issues.resolution = st_resolutions.id ' \
                'LEFT JOIN [//home/startrek/tables/prod/yandex-team/common/statuses] as st_statuses ' \
                'ON st_issues.status = st_statuses.id ' \
                'LEFT JOIN [//home/startrek/tables/prod/yandex-team/common/users] as st_users ' \
                'ON st_issues.assignee = st_users.uid ' \
                'LEFT JOIN [//home/startrek/tables/prod/yandex-team/common/users] as st_users1 ' \
                'ON st_issues.resolver = st_users1.uid ' \
                'LEFT JOIN [//home/startrek/tables/prod/yandex-team/common/users] as st_users2 ' \
                'ON st_issues.author = st_users2.uid ' \
                'LEFT JOIN [//home/startrek/tables/prod/yandex-team/common/users] as st_users3 ' \
                'ON st_issues.modifier = st_users3.uid ' \
                'LEFT JOIN [//home/startrek/tables/prod/yandex-team/common/priorities] as st_priorities ' \
                'ON st_issues.priority = st_priorities.id'.format(', '.join(fields))

        yt_data = list(YT_MARKOV_CLIENT.select_rows(
            query, format=yt.JsonFormat(attributes={"encode_utf8": False})))
        return yt_data

    def _fetch_data_st_fixvers(self):
        table_name = '//home/startrek/tables/prod/yandex-team/queue/HDRFS/versions'
        print("Start fetching fixVersions from {}".format(table_name))
        yt_data = YT_MARKOV_CLIENT.select_rows(
            'id, shortId from [{}]"'.format(table_name),
            format=yt.JsonFormat(attributes={"encode_utf8": False}))
        data = {x["id"]: x["shortId"] for x in yt_data}
        return data

    def _fetch_data_st_components(self):
        table_name = '//home/startrek/tables/prod/yandex-team/queue/HDRFS/components'
        print("Start fetching Components from {}".format(table_name))
        yt_data = YT_MARKOV_CLIENT.select_rows(
            'id, name from [{}]"'.format(table_name),
            format=yt.JsonFormat(attributes={"encode_utf8": False}))
        data = {x["id"]: x["name"] for x in yt_data}
        return data

    def _processCustomFields(self, item):
        customfields = item["customFields"]

        abcserviceid = customfields.get('abcService')
        item["abcservice"] = abcserviceid[0]["id"] if abcserviceid else 0

        marked = customfields.get('marked')
        item["marked"] = marked[0]["id"] if marked else None

        item["line"] = customfields.get('line')

    def _processFixVersions(self, item):
        fixver = item["fixVersions"]
        del item["fixVersions"]
        item["fixVersions"] = self.dict_ver.get(fixver[0]) if fixver else 0

    def _processSla(self, item):
        sla_info = item["sla"]
        del item["sla"]
        if not sla_info:
            return
        for sla_id in sla_info:
            counter = sla_info[sla_id]
            sla_sett_id = counter["settings"]["id"]
            if sla_sett_id not in [604, 605]:
                continue
            sla_spent = counter.get("spent", 0)
            sla_status = counter["violationStatus"]
            item["sla_{}_spent".format(sla_sett_id)] = sla_spent
            item["sla_{}_status".format(sla_sett_id)] = sla_status

    def _processComponents(self, item):
        components = item['components']
        del item['components']
        components_name = []
        for component in components:
            if component in self.dict_components:
                components_name.append(self.dict_components[component])
        item['components'] = components_name

    def _translate_data(self, data, tables):
        print("Start translating data")
        result = []
        for table in tables:
            result.append({
                'table': table,
                'fields': [x['name'] for x in table.schema],
                'data': []
            })
        date_now_timestamp = int(datetime.datetime.now().timestamp())
        for item in data:
            self._processCustomFields(item)
            self._processSla(item)
            self._processFixVersions(item)
            self._processComponents(item)
            item['customFields'] = str(item['customFields'])
            item['date'] = self.date_now
            item['last_update'] = date_now_timestamp

            for table in result:
                table['data'].append({key: item[key] for key in item if key in table['fields']})

        print("Parsed {} records".format(len(data)))
        return result

    def main(self):
        full_snapshot = STFullSnapshotStatic()

        pre_data = self._fetch_data()
        data = self._translate_data(pre_data, [self, full_snapshot])
        del pre_data

        for table in data:
            # перезаписываем таблицу в случае если она либо не существует либо у нее recreate=True
            if not self.yt_client.exists(table['table'].yt_table) or table['table'].recreate:
                print("Start upload to YT {}".format(table['table'].yt_table))
                table['table'].write_data_to_yt(table['data'])
        print("The end of syncing")


class STFullSnapshotStatic(SimpleYTBasedCollector):
    """Таблица-слепок очереди HDRFS на конкретный день"""
    def __init__(self):
        self.date_now = datetime.datetime.now().strftime('%d-%m-%Y')
        super().__init__(yt_table_name='monitoring/st_issues/{}'.format(self.date_now),
                         recreate=False,
                         client=YT_CLIENT,
                         is_table_dynamic=False,
                         static_append=False
                         )
        self.schema = [
            {'name': 'version', 'required': True, 'type': 'uint64'},
            {'name': 'key', 'required': True, 'type': 'string'},
            {'name': 'aliases', 'required': False, 'type': 'any'},
            {'name': 'status', 'required': True, 'type': 'uint64'},
            {'name': 'resolution', 'required': False, 'type': 'string'},
            {'name': 'type', 'required': True, 'type': 'string'},
            {'name': 'summary', 'required': True, 'type': 'string'},
            {'name': 'description', 'required': False, 'type': 'string'},
            {'name': 'created', 'required': True, 'type': 'uint64'},
            {'name': 'updated', 'required': True, 'type': 'uint64'},
            {'name': 'start', 'required': False, 'type': 'uint64'},
            {'name': 'end', 'required': False, 'type': 'uint64'},
            {'name': 'dueDate', 'required': False, 'type': 'uint64'},
            {'name': 'resolved', 'required': False, 'type': 'uint64'},
            {'name': 'resolver', 'required': False, 'type': 'string'},
            {'name': 'author', 'required': True, 'type': 'string'},
            {'name': 'modifier', 'required': True, 'type': 'string'},
            {'name': 'assignee', 'required': False, 'type': 'string'},
            {'name': 'priority', 'required': True, 'type': 'uint64'},
            {'name': 'affectedVersions', 'required': False, 'type': 'any'},
            {"name": "abcservice", "required": False, "type": "int64"},
            {'name': 'fixVersions', 'required': False, 'type': 'any'},
            {'name': 'components', 'required': False, 'type': 'any'},
            {'name': 'tags', 'required': False, 'type': 'any'},
            {'name': 'sprint', 'required': False, 'type': 'any'},
            {'name': 'customFields', 'required': False, 'type': 'string'},
            {'name': 'followers', 'required': False, 'type': 'any'},
            {'name': 'access', 'required': False, 'type': 'any'},
            {'name': 'unique', 'required': False, 'type': 'string'},
            {'name': 'followingGroups', 'required': False, 'type': 'any'},
            {'name': 'followingMaillists', 'required': False, 'type': 'any'},
            {'name': 'parent', 'required': False, 'type': 'string'},
            {'name': 'epic', 'required': False, 'type': 'string'},
            {'name': 'originalEstimation', 'required': False, 'type': 'uint64'},
            {'name': 'estimation', 'required': False, 'type': 'uint64'},
            {'name': 'spent', 'required': False, 'type': 'uint64'},
            {'name': 'storyPoints', 'required': False, 'type': 'double'},
            {'name': 'ranks', 'required': False, 'type': 'any'},
            {'name': 'ranking', 'required': False, 'type': 'any'},
            {'name': 'goals', 'required': False, 'type': 'any'},
            {'name': 'votedBy', 'required': False, 'type': 'any'},
            {'name': 'favoritedBy', 'required': False, 'type': 'any'},
            {'name': 'emailFrom', 'required': False, 'type': 'string'},
            {'name': 'emailTo', 'required': False, 'type': 'any'},
            {'name': 'emailCc', 'required': False, 'type': 'any'},
            {'name': 'emailCreatedBy', 'required': False, 'type': 'string'},
            {'name': 'orgId', 'required': True, 'type': 'uint64'},
            {"name": "sla_604_spent", "required": False, "type": "int64"},
            {"name": "sla_604_status", "required": False, "type": "string"},
            {"name": "sla_605_spent", "required": False, "type": "int64"},
            {"name": "sla_605_status", "required": False, "type": "string"},
            {'name': 'date', 'required': True, 'type': 'string'}]


STStaticSyncer().main()
