package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/helpdesk/infra/baldr/internal/baldrerrors"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/models"
)

const (
	TypeDesktop  = "DESKTOPS"
	TypeNotebook = "NOTEBOOKS"
	TypeServers  = "SERVERS"
	TypeNodes    = "NODES"
)

type BotOS struct {
	InventoryNumber string `json:"instance_number"`
	Type            string `json:"item_segment3"`
	FQDN            string `json:"XXCSI_FQDN"`
	Status          string `json:"status_name"`
	Owner           string `json:"EMPLOYEE_OWNED"`
	ExtUser         string `json:"EXTERNAL_USER"`
	Profile         string `json:"HARDWARE_PROFILE"`
}

type botOSResp struct {
	Result  int     `json:"res"`
	Message string  `json:"msg"`
	OS      []BotOS `json:"os"`
}

type ClientConfig struct {
	schema     string
	host       string
	port       string
	apiPath    string
	oauthToken string
}

type Client struct {
	*ClientConfig
	httpClient *http.Client
}

type Request struct {
	apiMethod  string
	httpMethod string
	reqBody    string
	reqParam   string
	headers    map[string]string
}

func readResponseBody(r *http.Response) []byte {
	defer func(response *http.Response) {
		err := response.Body.Close()
		if err != nil {
			log.Printf("ERROR: close response body failed: %v", err)
		}
	}(r)

	body, _ := ioutil.ReadAll(r.Body)
	return body
}

func (c *Client) doRequest(req *Request) (*http.Response, error) {
	pathURL := fmt.Sprintf("%s://%s:%s/%s/%s%s", c.schema, c.host, c.port, c.apiPath, req.apiMethod, req.reqParam)
	body := []byte(req.reqBody)
	var r, err = http.NewRequest(req.httpMethod, pathURL, bytes.NewBuffer(body))
	if err != nil {
		return nil, fmt.Errorf("doRequest(): new request: %w", err)
	}

	for k, v := range req.headers {
		r.Header.Set(k, v)
	}

	resp, err := c.httpClient.Do(r)
	if err != nil {
		return nil, fmt.Errorf("doRequest(): request: %w", err)
	}

	return resp, nil
}

func (c *Client) findInvNumBySN(sn string) (BotOS, error) {
	var instance BotOS
	urlParam := fmt.Sprintf(
		"?format=json&output=instance_number%%7Citem_segment3%%7CXXCSI_FQDN%%7Cstatus_name%%7CEMPLOYEE_OWNED&sn=%s",
		url.QueryEscape(sn))
	req := Request{
		apiMethod:  "osinfo.php",
		httpMethod: http.MethodGet,
		reqBody:    "",
		reqParam:   urlParam,
		headers:    nil,
	}

	var resp *http.Response
	var err error

	for i := 0; i < 5; i++ {
		resp, err = c.doRequest(&req)
		if err != nil {
			time.Sleep(1 * time.Second)
			continue
		}
		break
	}

	if err != nil {
		return instance, fmt.Errorf("findInvNumBySN(%q): request: %w", sn, err)
	}

	body := readResponseBody(resp)

	var botResp botOSResp
	err = json.Unmarshal(body, &botResp)
	if err != nil {
		return instance, fmt.Errorf("findInvNumBySN(%q): unmarshal response: %w", sn, err)
	}

	if botResp.Result != 1 {
		return instance, fmt.Errorf("findInvNumBySN(%q): response: %s", sn, botResp.Message)
	}

	var validInstances []BotOS
	for _, inst := range botResp.OS {
		if inst.Type == TypeDesktop || inst.Type == TypeNotebook ||
			inst.Type == TypeServers || inst.Type == TypeNodes {
			validInstances = append(validInstances, inst)
		}
	}

	if len(validInstances) == 0 {
		return instance, fmt.Errorf("findInvNumBySN(%q): instance not found", sn)
	} else if len(validInstances) > 1 {
		var invNumbers string
		for _, inst := range validInstances {
			invNumbers += fmt.Sprintf(", %s", inst.InventoryNumber)
		}
		invNumbers = strings.TrimLeft(invNumbers, ", ")
		return instance, fmt.Errorf("findInvNumBySN(%q): to many instances (%s)", sn, invNumbers)
	}

	return validInstances[0], nil
}

func (c *Client) findInvNumByMAC(mac net.HardwareAddr) (BotOS, error) {
	var instance BotOS

	urlParam := fmt.Sprintf(
		"?format=json&output=instance_number%%7Citem_segment3%%7CXXCSI_FQDN%%7Cstatus_name%%7CEMPLOYEE_OWNED&mac=%s",
		url.QueryEscape(strings.ReplaceAll(mac.String(), ":", "")))
	req := Request{
		apiMethod:  "osinfo.php",
		httpMethod: http.MethodGet,
		reqBody:    "",
		reqParam:   urlParam,
		headers:    nil,
	}

	var resp *http.Response
	var err error

	for i := 0; i < 5; i++ {
		resp, err = c.doRequest(&req)
		if err != nil {
			time.Sleep(1 * time.Second)
			continue
		}
		break
	}

	if err != nil {
		return instance, fmt.Errorf("findInvNumByMAC(%q): request: %w", mac.String(), err)
	}

	body := readResponseBody(resp)

	var botResp botOSResp
	err = json.Unmarshal(body, &botResp)
	if err != nil {
		return instance, fmt.Errorf("findInvNumByMAC(%q): unmarshal response: %w", mac.String(), err)
	}

	if botResp.Result != 1 {
		return instance, fmt.Errorf("findInvNumByMAC(%q): response: %s", mac.String(), botResp.Message)
	}

	var validInstances []BotOS
	for _, inst := range botResp.OS {
		if inst.Type == TypeDesktop || inst.Type == TypeNotebook ||
			inst.Type == TypeServers || inst.Type == TypeNodes {
			validInstances = append(validInstances, inst)
		}
	}

	if len(validInstances) == 0 {
		return instance, fmt.Errorf("findInvNumByMAC(%q): instance not found", mac.String())
	} else if len(validInstances) > 1 {
		var invNumbers string
		for _, inst := range validInstances {
			invNumbers += fmt.Sprintf(", %s", inst.InventoryNumber)
		}
		invNumbers = strings.TrimLeft(invNumbers, ", ")
		return instance, fmt.Errorf("findInvNumByMAC(%q): to many instances (%s)", mac.String(), invNumbers)
	}

	return validInstances[0], nil
}

func (c *Client) findInvNum(sn string, macs []net.HardwareAddr) (BotOS, error) {
	instance, err := c.findInvNumBySN(sn)
	if err == nil {
		return instance, nil
	}
	log.Printf("FindInvNum(): %s", err.Error())

	instance, err = c.findInvNumBySN("S" + sn)
	if err == nil {
		return instance, nil
	}
	log.Printf("FindInvNum(): %s", err.Error())

	for _, mac := range macs {
		instance, err := c.findInvNumByMAC(mac)
		if err == nil {
			return instance, nil
		}
		log.Printf("FindInvNum(): %s", err.Error())
	}

	return instance, fmt.Errorf("FindInvNum(): %w", err)
}

func findInventoryNumberFromBot(config bot, dep *models.Deploy) error {
	botConfig := ClientConfig{
		schema:     "https",
		host:       "bot.yandex-team.ru",
		port:       "443",
		apiPath:    "api",
		oauthToken: config.token,
	}

	client := &Client{
		&botConfig,
		&http.Client{},
	}

	hw, err := client.findInvNum(dep.Options[models.OptionHWSerialNumber], dep.MACAddresses)
	if err != nil {
		return fmt.Errorf("findInventoryNumberFromBot(): %w", err)
	}

	dep.InventoryNumber = hw.InventoryNumber
	dep.Options[models.OptionHWInventoryNumber] = dep.InventoryNumber
	dep.Options[models.OptionFQDN] = hw.FQDN
	dep.Options[models.OptionHWStatus] = hw.Status
	dep.Options[models.OptionHWOwner] = hw.Owner
	dep.Options[models.OptionHWExtUser] = hw.ExtUser
	dep.Options[models.OptionHWProfile] = hw.Profile
	dep.Options[models.OptionHWType] = hw.Type

	return nil
}

func (env *Env) findInventoryNumber(dep *models.Deploy) error {
	err := findInventoryNumberFromBot(env.bot, dep)
	if err != nil {
		dep.InventoryNumber = models.UndefinedInventoryNumber
		dep.ErrorCode = baldrerrors.CodeInventoryNumberNotDefined
		dep.Message = baldrerrors.CodeInventoryNumberNotDefined.String()
		return fmt.Errorf("findInventoryNumber: %w", err)
	}

	return nil
}
