package main

import (
	"encoding/json"
	"encoding/xml"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"text/template"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/helpdesk/infra/baldr/internal/baldrerrors"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/models"
)

const (
	deployEnvProd             = "prod"
	deployEnvTest             = "test"
	preseedTypeUser           = "user"
	preseedTypeZombie         = "zombie"
	preseedTypeZombieGPU      = "zombie-gpu"
	linuxReleaseBionic        = "bionic"
	linuxReleaseBionicVersion = "18.04"
	linuxReleaseFocal         = "focal"
	linuxReleaseFocalVersion  = "20.04"
)

func newID() (id string, err error) {
	var u4 uuid.UUID
	u4, err = uuid.NewV4()
	if err != nil {
		err = fmt.Errorf("newID: %w", err)
	} else {
		id = u4.String()
	}

	return
}

func (env *Env) ipxeGenStartupScript(w http.ResponseWriter, r *http.Request) {
	tmplData := &models.Deploy{}
	tmplData.Options = make(map[string]string, 5)
	tmplData.Options[models.OptionProdAPIHost] = env.apiProdURL
	tmplData.Options[models.OptionTestAPIHost] = env.apiTestURL
	tmplData.Options[models.OptionEnvironment] = env.currentEnv

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}
	err := executeTemplate(tmplData, f, w, ipxeStartupTemplate)
	if err != nil {
		fmt.Printf("ipxeGenStartupScript(): %v\n", err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("ipxeGenStartupScript(): failed to write response: %v\n", err)
		}
	}
}

func (env *Env) ipxeGenBootScript(w http.ResponseWriter, r *http.Request) {
	err := env.handlingRequest(w, r)
	if err != nil {
		fmt.Printf("ipxeGenScript(): %v\n", err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("ipxeGenScript(): failed to write response: %v\n", err)
		}
	}
}

func (env *Env) handlingRequest(w http.ResponseWriter, r *http.Request) error {
	var err error
	dep := &models.Deploy{}
	dep.Options = make(map[string]string, 50)
	dep.CreationTime = time.Now()

	dep.ID, err = newID()
	if err != nil {
		return fmt.Errorf("handlingRequest(): %w", err)
	}
	dep.Status = models.DeployStatusOpen
	dep.Options[models.OptionID] = dep.ID

	err = env.gatheringData(r, dep)

	// Close task if error
	if err != nil {
		dep.Status = models.DeployStatusClosed
		dep.CloseTime = time.Now()
		fmt.Printf("handlingRequest(): %s: %v\n", dep.ID, err)
	}

	// Close old tasks for current inventory number
	if dep.Status != models.DeployStatusClosed {
		err := env.db.CloseWithNewDeploy(dep)
		if err != nil {
			fmt.Printf("handlingRequest(): %s: %v\n", dep.ID, err)
		}
	}

	// Close task for Animals and Wipe
	if dep.Status != models.DeployStatusClosed {
		if dep.Options[models.OptionDeployType] == DeployTypeAnimals ||
			dep.Options[models.OptionDeployType] == DeployTypeAnimalsTest ||
			dep.Options[models.OptionDeployType] == DeployTypeWipe {
			dep.Status = models.DeployStatusClosed
			dep.CloseTime = time.Now()
		}
	}

	// Add entry to DB
	err = env.db.AddNewEntry(dep)
	if err != nil {
		fmt.Printf("handlingRequest(): %s: %v\n", dep.ID, err)
		dep.ErrorCode = baldrerrors.CodeDBError
		dep.Message = dep.ErrorCode.String()
	}

	dep.Options[models.OptionProdAPIHost] = env.apiProdURL
	dep.Options[models.OptionTestAPIHost] = env.apiTestURL
	if dep.Status == models.DeployStatusOpen {
		dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	}
	dep.Options[models.OptionEnvironment] = env.currentEnv

	// Execute template
	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}
	err = executeTemplate(dep, f, w, ipxeTemplate)
	if err != nil {
		return fmt.Errorf("handlingRequest(): %w", err)
	}

	return nil
}

func (env *Env) gatheringData(r *http.Request, dep *models.Deploy) error {
	var err error
	// Processing iPXE request data
	err = iPXERequestData(r, dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Define deployment template
	templateName, ok := r.URL.Query()[models.OptionDeploymentTemplate]
	if ok {
		dep.Options[models.OptionDeploymentTemplate] = templateName[0]
	}

	// Define location and local parameters
	err = env.locationData(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Define deploy server
	err = env.findDeployServer(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Define inventory number, FQDN and hardware status by HW
	err = env.findInventoryNumber(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Get data from OFR
	err = env.findTask(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Check OS name
	err = env.checkOS(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Check domain and ms office settings
	env.checkLinuxDomain(dep)

	// Define system and user locale
	err = env.defineLocale(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Define OS build version
	err = env.defineOSBuild(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Define drivers profile
	err = env.defineDriversProfile(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Check boot mode
	err = env.checkBootMode(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Define OU
	err = env.defineOU(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	// Validate data for Windows deployment
	err = checkWindowsDeploymentData(dep)
	if err != nil {
		return fmt.Errorf("gatheringData(): %w", err)
	}

	return err
}

func (env *Env) checkOS(dep *models.Deploy) error {
	if dep.Options[models.OptionOS] == "" {
		dep.Message = "Undefined OS"
		dep.ErrorCode = baldrerrors.CodeUndefinedOS
		return fmt.Errorf("checkOS: %s", dep.ErrorCode)
	}

	// Get supported operating systems
	var supOS []string
	var err error
	supOS, err = env.db.SupportedOS()
	if err != nil {
		dep.ErrorCode = baldrerrors.CodeDBError
		dep.Message = dep.ErrorCode.String()
		return fmt.Errorf("checkOS: %w", err)
	}

	isSupportedOS := stringInSlice(dep.Options[models.OptionOS], supOS)
	if !isSupportedOS {
		dep.Message = fmt.Sprintf("Unsupported OS %q", dep.Options[models.OptionOS])
		dep.ErrorCode = baldrerrors.CodeUnsupportedOS
		return fmt.Errorf("checkOS: %s %q", dep.ErrorCode, dep.Options[models.OptionOS])
	}

	return nil
}

func (env *Env) checkLinuxDomain(dep *models.Deploy) {
	if dep.Options[models.OptionOS] == "Linux" && dep.Options[models.OptionWithDomain] == "true" {
		if dep.Message != "" {
			dep.Message += " "
		}
		dep.Message = "Linux cannot be deployed with a domain, the option \"with domain\" will be ignored."
		dep.Options[models.OptionWithDomain] = "false"
	}

	if dep.Options[models.OptionOS] == "Linux" && dep.Options[models.OptionWithOffice] == "true" {
		if dep.Message != "" {
			dep.Message += " "
		}
		dep.Message += "Linux cannot be deployed with MS Office, the \"with Office\" option will be ignored"
		dep.Options[models.OptionWithOffice] = "false"
	}
}

func (env *Env) defineLocale(dep *models.Deploy) error {
	if dep.Options[models.OptionLanguageProfile] != "" {
		// ToDo: поиск языкового профиля, если указан в OFR
	} else {
		dep.Options[models.OptionLanguageProfile] = dep.Options[models.OptionDefaultLanguageProfile]
	}

	return nil
}

func (env *Env) defineOSBuild(dep *models.Deploy) error {
	// Define Windows Build Version
	if strings.HasPrefix(dep.Options[models.OptionOS], "Windows") {
		err := env.windowsBuild(dep)
		if err != nil {
			return fmt.Errorf("defineOSBuild(): %w", err)
		}
	}

	return nil
}

func (env *Env) windowsBuild(dep *models.Deploy) error {
	var operationSystems []models.OperationSystem
	var err error

	if dep.Options[models.OptionOSVersion] != "" {
		err = parseWindowsVersionString(dep)
		if err != nil {
			dep.ErrorCode = baldrerrors.CodeParseOSVersionStringError
			dep.Message = fmt.Sprintf("Invalid build %q for %s", dep.Options[models.OptionOSVersion], dep.Options[models.OptionOS])
			return fmt.Errorf("windowsBuild(): %w", err)
		}
	}

	if dep.Options[models.OptionOSBuild] == "" {
		operationSystems, err = env.db.OSDefaultBuild(dep)
	} else {
		operationSystems, err = env.db.OSBuild(dep)
	}

	if err != nil {
		dep.ErrorCode = baldrerrors.CodeDBError
		dep.Message = dep.ErrorCode.String()
		return fmt.Errorf("windowsBuild(): %w", err)
	}

	if len(operationSystems) == 0 {
		dep.ErrorCode = baldrerrors.CodeUnsupportedOSBuild
		dep.Message = fmt.Sprintf("Invalid build %q for %s", dep.Options[models.OptionOSVersion], dep.Options[models.OptionOS])
		return fmt.Errorf("windowsBuild(): %s", dep.Message)
	}

	if len(operationSystems) > 1 {
		dep.ErrorCode = baldrerrors.CodeDBInvalidData
		dep.Message = fmt.Sprintf("Invalid build %q for %s", dep.Options[models.OptionOSVersion], dep.Options[models.OptionOS])
		return fmt.Errorf("windowsBuild(): %s", dep.Message)
	}

	dep.Options[models.OptionOSEdition] = operationSystems[0].Edition
	dep.Options[models.OptionOSBuild] = operationSystems[0].Build
	dep.Options[models.OptionOSUpdated] = operationSystems[0].Updated
	dep.Options[models.OptionOSProductKey] = operationSystems[0].ProductKey
	return nil
}

func parseWindowsVersionString(dep *models.Deploy) (err error) {
	// parse OS version string
	// format: "OSEdition OSBuild updated OSUpdated"
	// for example: "Enterprise 2004 updated May 2020"
	// where
	// OSEdition = "Enterprise"
	// OSBuild = "2004"
	// OSUpdated = "May 2020"
	err = fmt.Errorf("parseWindowsVersionString(): failed to parse OS version string %q", dep.Options[models.OptionOSVersion])

	versionString := strings.SplitN(dep.Options[models.OptionOSVersion], " ", 2)
	fmt.Printf("[DEBUG] versionString: %q\t%+v\n", dep.Options[models.OptionOSVersion], versionString)
	if len(versionString) == 2 && versionString[0] != "" {
		dep.Options[models.OptionOSEdition] = versionString[0]
	} else {
		fmt.Printf("[DEBUG] OptionOSEdition\n")
		return
	}

	versionString = strings.SplitN(versionString[1], " ", 2)
	if len(versionString) == 2 && versionString[0] != "" {
		dep.Options[models.OptionOSBuild] = versionString[0]
		fmt.Printf("[DEBUG] OptionOSBuild: %s\n", dep.Options[models.OptionOSBuild])
	} else {
		fmt.Printf("[DEBUG] OptionOSBuild\n")
		return
	}

	if !strings.HasPrefix(versionString[1], "updated ") {
		fmt.Printf("[DEBUG] HasPrefix\n")
		return
	}

	versionString = strings.SplitN(versionString[1], " ", 2)
	if len(versionString) == 2 && versionString[1] != "" {
		dep.Options[models.OptionOSUpdated] = versionString[1]
	} else {
		fmt.Printf("[DEBUG] OptionOSUpdated\n")
		return
	}

	err = nil
	return
}

func (env *Env) defineDriversProfile(dep *models.Deploy) error {
	if !strings.HasPrefix(dep.Options[models.OptionOS], "Windows") {
		return nil
	}

	var profiles []string
	var err error
	profiles, err = env.db.DriversProfile(dep)
	if err != nil {
		dep.ErrorCode = baldrerrors.CodeDBError
		dep.Message = dep.ErrorCode.String()
		return fmt.Errorf("defineDriversProfile(): %w", err)
	}

	if len(profiles) > 1 {
		dep.Message = fmt.Sprintf("More than one driver profile %q for %s %s with %s %s",
			strings.Join(profiles, ", "), dep.Options[models.OptionHWManufacturer],
			dep.Options[models.OptionHWModel], dep.Options[models.OptionOS], dep.Options[models.OptionOSBuild])
		dep.ErrorCode = baldrerrors.CodeDBInvalidData
		return fmt.Errorf("defineDriversProfile(): %s", dep.Message)
	}

	if len(profiles) == 0 {
		dep.Message = fmt.Sprintf("Drivers profile not found for %s %s",
			dep.Options[models.OptionHWManufacturer], dep.Options[models.OptionHWModel])
		dep.Options[models.OptionDriversProfile] = models.DefaultDriversProfile
		dep.ErrorCode = baldrerrors.CodeDriversProfileNotFound
	} else {
		dep.Options[models.OptionDriversProfile] = profiles[0]
	}

	return nil
}

func (env *Env) checkBootMode(dep *models.Deploy) error {
	if dep.Options[models.OptionIPXEUEFIMode] != models.IPXEModeUEFI {
		err := env.checkLegacyBoot(dep)
		if err != nil {
			return fmt.Errorf("checkBootMode(): %w", err)
		}
	}

	return nil
}

func (env *Env) checkLegacyBoot(dep *models.Deploy) error {
	var checks []bool
	var err error
	checks, err = env.db.SupportBootingFromLegacyBIOS(dep)
	if err != nil {
		dep.ErrorCode = baldrerrors.CodeDBError
		dep.Message = dep.ErrorCode.String()
		return fmt.Errorf("checkLegacyBoot(): %w", err)
	}

	if len(checks) > 1 {
		dep.Message = fmt.Sprintf("legacy BIOS boot mode support: more than one entry for %s %s",
			dep.Options[models.OptionHWManufacturer], dep.Options[models.OptionHWModel])
		dep.ErrorCode = baldrerrors.CodeDBInvalidData
		return fmt.Errorf("checkLegacyBoot(): more than one entry for %s %s",
			dep.Options[models.OptionHWManufacturer], dep.Options[models.OptionHWModel])
	}

	if len(checks) != 0 && !checks[0] {
		dep.Message = fmt.Sprintf("legacy BIOS boot mode is not supported for %s %s",
			dep.Options[models.OptionHWManufacturer], dep.Options[models.OptionHWModel])
		dep.ErrorCode = baldrerrors.CodeUnsupportedBootMode
		return fmt.Errorf("checkLegacyBoot(): %s", dep.Message)
	}

	return nil
}

func (env *Env) defineOU(dep *models.Deploy) error {
	if dep.Options[models.OptionADOU] != "" {
		return nil
	}

	if strings.HasPrefix(dep.Options[models.OptionOS], "Windows") && dep.Options[models.OptionWithDomain] == "true" {
		err := env.windowsOU(dep)
		if err != nil {
			return fmt.Errorf("defineOU(): %w", err)
		}
	}

	return nil
}

func (env *Env) windowsOU(dep *models.Deploy) error {
	var paths []string
	var err error
	paths, err = env.db.OU(dep)
	if err != nil {
		dep.ErrorCode = baldrerrors.CodeDBError
		dep.Message = dep.ErrorCode.String()
		return fmt.Errorf("windowsOU(): %w", err)
	}

	if len(paths) > 1 {
		dep.Message = fmt.Sprintf("More than one OU %q for %s and %s",
			strings.Join(paths, ", "), dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
		dep.ErrorCode = baldrerrors.CodeDBInvalidData
		return fmt.Errorf("windowsOU(): More than one OU %q for %s and %s",
			strings.Join(paths, ", "), dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
	}

	if len(paths) == 0 {
		if dep.Message == "" {
			dep.Message = fmt.Sprintf("Use default OU for %s %s",
				dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
		} else {
			dep.Message = fmt.Sprintf("%s\nUse default OU for %s %s",
				dep.Message, dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
		}

		paths, err = env.db.DefaultOU()
		if err != nil {
			dep.ErrorCode = baldrerrors.CodeDBError
			dep.Message = dep.ErrorCode.String()
			return fmt.Errorf("windowsOU(): %w", err)
		}

		if len(paths) > 1 {
			dep.Message = fmt.Sprintf("More than one OU %q for %s and %s",
				strings.Join(paths, ", "), dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
			dep.ErrorCode = baldrerrors.CodeDBInvalidData
			return fmt.Errorf("windowsOU(): more than one OU %q for %s and %s",
				strings.Join(paths, ", "), dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
		}

		if len(paths) == 0 {
			dep.Message = fmt.Sprintf("OU not found for %s and %s",
				dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
			dep.ErrorCode = baldrerrors.CodeDBInvalidData
			return fmt.Errorf("windowsOU(): OU not found for %s and %s",
				dep.Options[models.OptionOS], dep.Options[models.OptionCountry])
		}
	}

	dep.Options[models.OptionADOU] = paths[0]
	return nil
}

func checkWindowsDeploymentData(dep *models.Deploy) error {
	if !strings.HasPrefix(dep.Options[models.OptionOS], "Windows") {
		return nil
	}

	if dep.Options[models.OptionWithDomain] == "false" {
		if err := checkUserName(dep); err != nil {
			return fmt.Errorf("checkWindowsDeploymentData(): %w", err)
		}
	}

	if err := checkComputerName(dep); err != nil {
		return fmt.Errorf("checkWindowsDeploymentData(): %w", err)
	}

	return nil
}

func checkUserName(dep *models.Deploy) error {
	if len(dep.Options[models.OptionUserName]) < 2 || len(dep.Options[models.OptionUserName]) > 20 {
		dep.ErrorCode = baldrerrors.CodeInvalidUserName
		dep.Message = fmt.Sprintf("Invalid username %q. A username length can be 2 characters minimum "+
			"and 20 characters maximum", dep.Options[models.OptionUserName])
		return fmt.Errorf("checkUserName(): %s", dep.Message)
	}

	isNumeric := true
	for _, ch := range dep.Options[models.OptionUserName] {
		if (ch < 'a' || ch > 'z') && (ch < '0' || ch > '9') && ch != '_' && ch != '-' {
			dep.ErrorCode = baldrerrors.CodeInvalidUserName
			dep.Message = fmt.Sprintf("Invalid username %q. A username can only contain alphanumeric "+
				"characters (letters a-z, numbers 0-9) with the exception of underscores and dashes",
				dep.Options[models.OptionUserName])
			return fmt.Errorf("checkUserName(): %s", dep.Message)
		}

		if ch < '0' || ch > '9' {
			isNumeric = false
		}
	}

	if isNumeric {
		dep.ErrorCode = baldrerrors.CodeInvalidUserName
		dep.Message = fmt.Sprintf("Invalid username %q. A username cannot only contain numbers", dep.Options[models.OptionUserName])
		return fmt.Errorf("checkUserName(): %s", dep.Message)
	}

	if dep.Options[models.OptionWithDomain] == "false" && dep.Options[models.OptionComputerName] == dep.Options[models.OptionUserName] {
		dep.ErrorCode = baldrerrors.CodeInvalidUserName
		dep.Message = fmt.Sprintf("Invalid username %q. We can't create a local user account "+
			"which has the same name with computer account name", dep.Options[models.OptionUserName])
		return fmt.Errorf("checkUserName(): %s", dep.Message)
	}

	return nil
}

func checkComputerName(dep *models.Deploy) error {
	reservedNames := []string{
		"anonymous",
		"batch",
		"builtin",
		"dialup",
		"interactive",
		"internet",
		"local",
		"network",
		"null",
		"proxy",
		"restricted",
		"self",
		"server",
		"service",
		"system",
		"users",
		"world",
	}

	if len(dep.Options[models.OptionComputerName]) < 2 || len(dep.Options[models.OptionComputerName]) > 15 {
		dep.ErrorCode = baldrerrors.CodeInvalidComputerName
		dep.Message = fmt.Sprintf("Invalid computername %q. A computername length can be 2 characters minimum "+
			"and 15 characters maximum", dep.Options[models.OptionComputerName])
		return fmt.Errorf("checkComputerName(): %s", dep.Message)
	}

	for _, ch := range dep.Options[models.OptionComputerName] {
		if (ch < 'a' || ch > 'z') && (ch < '0' || ch > '9') && ch != '_' && ch != '-' {
			dep.ErrorCode = baldrerrors.CodeInvalidComputerName
			dep.Message = fmt.Sprintf("Invalid computername %q. A computername can only contain alphanumeric "+
				"characters (letters a-z, numbers 0-9) with the exception of underscores and dashes",
				dep.Options[models.OptionComputerName])
			return fmt.Errorf("checkComputerName(): %s", dep.Message)
		}
	}

	if reserved := stringInSlice(dep.Options[models.OptionComputerName], reservedNames); reserved {
		dep.ErrorCode = baldrerrors.CodeInvalidComputerName
		dep.Message = fmt.Sprintf("Invalid computername. %q is reserved word in Windows. "+
			"We cannot use reserved word as computername. Here's a list of all reserved words in Windows: %s.",
			dep.Options[models.OptionComputerName], strings.Join(reservedNames, ", "))
		return fmt.Errorf("checkComputerName(): %s", dep.Message)
	}

	return nil
}

func ping(w http.ResponseWriter, r *http.Request) {
	w.WriteHeader(http.StatusOK)
	_, err := w.Write([]byte("Ok"))
	if err != nil {
		fmt.Printf("ping(): %v\n", err)
	}
}

func (env *Env) printVersion(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")

	info := struct {
		Version   string `json:"version"`
		Commit    string `json:"commit"`
		BuildTime string `json:"buildTime"`
	}{
		Version:   version,
		Commit:    commit,
		BuildTime: buildTime,
	}

	err := json.NewEncoder(w).Encode(info)
	if err != nil {
		fmt.Printf("printVersion(): error during encode: %v", err)
		http.Error(w, "Something went wrong", http.StatusInternalServerError)
	}
}

func (env *Env) deployConfig(w http.ResponseWriter, r *http.Request) {
	ids, ok := r.URL.Query()["id"]
	if !ok || len(ids[0]) == 0 {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("deployConfig(): failed to write response: %v\n", err)
		}
		return
	}

	id := strings.ToLower(ids[0])
	if _, err := uuid.FromString(id); err != nil {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("deployConfig(): failed to write response: %v\n", err)
		}
		return
	}

	configs, err := env.db.DeploymentData(id)
	if err != nil {
		fmt.Printf("deployConfig(): %v\n", err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("deployConfig(): failed to write response: %v\n", err)
		}
		return
	}

	if len(configs) == 0 {
		w.WriteHeader(http.StatusNotFound)
		_, err = w.Write([]byte("Not found"))
		if err != nil {
			fmt.Printf("deployConfig(): failed to write response: %v\n", err)
		}
		return
	} else if len(configs) > 1 {
		fmt.Printf("deployConfig(): more than one open task by id %s\n", id)
		w.WriteHeader(http.StatusInternalServerError)
		_, err := w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("deployConfig(): failed to write response: %v\n", err)
		}
		return
	}

	if configs[0][models.OptionWithDomain] == "true" {
		configs[0][models.OptionADRobotName] = env.ad.robotName
		configs[0][models.OptionADRobotDomain] = env.ad.domain
		configs[0][models.OptionADRobotPassword] = env.ad.robotPassword
	}

	configs[0][models.OptionLocalAdminPassword] = env.ad.localAdminPassword
	if env.currentEnv == deployEnvProd {
		configs[0][models.OptionCurrentAPIHost] = env.apiProdURL
	} else if env.currentEnv == deployEnvTest {
		configs[0][models.OptionCurrentAPIHost] = env.apiTestURL
	}

	xmlData, err := xml.Marshal(configs[0])
	if err != nil {
		fmt.Printf("deployConfig(): marshaling data failed: %v\n", err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("deployConfig(): failed to write response: %v\n", err)
		}
		return
	}

	w.WriteHeader(http.StatusOK)
	_, err = w.Write(xmlData)
	if err != nil {
		fmt.Printf("deployConfig(): failed to write response: %v\n", err)
	}
}

func (env *Env) bootstrapIni(w http.ResponseWriter, r *http.Request) {
	ids, ok := r.URL.Query()["id"]
	if !ok || len(ids[0]) == 0 {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("bootstrapIni(): failed to write response: %v\n", err)
		}
		return
	}

	id := strings.ToLower(ids[0])
	if _, err := uuid.FromString(id); err != nil {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("bootstrapIni(): failed to write response: %v\n", err)
		}
		return
	}

	shares, err := env.db.DeploymentShare(id)
	if err != nil {
		fmt.Printf("bootstrapIni(): find deployment server by id (%s) failed: %v\n", id, err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("bootstrapIni(): failed to write response: %v\n", err)
		}
		return
	}

	if len(shares) == 0 {
		w.WriteHeader(http.StatusNotFound)
		_, err = w.Write([]byte("Not found"))
		if err != nil {
			fmt.Printf("bootstrapIni(): failed to write response: %v\n", err)
		}
		return
	} else if len(shares) > 1 {
		fmt.Printf("bootstrapIni(): more than one deployment share by id %s\n", id)
		w.WriteHeader(http.StatusInternalServerError)
		_, err := w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("bootstrapIni(): failed to write response: %v\n", err)
		}
		return
	}

	shares[0].Username = env.smb.username
	shares[0].Domain = env.smb.domain
	shares[0].Password = env.smb.password

	err = executeTemplate(&shares[0], template.FuncMap{}, w, bootstrapIniTemplate)
	if err != nil {
		fmt.Printf("bootstrapIni(): %v\n", err)
	}
}

func (env *Env) gatheringManualDeployData(r *http.Request, dep *models.Deploy) error {
	var err error
	// Processing iPXE request data
	err = iPXERequestData(r, dep)
	if err != nil {
		return fmt.Errorf("gatheringManualDeployData(): %w", err)
	}

	// Define location and local parameters
	err = env.locationData(dep)
	if err != nil {
		return fmt.Errorf("gatheringManualDeployData(): %w", err)
	}

	// Define deploy server
	err = env.findDeployServer(dep)
	if err != nil {
		return fmt.Errorf("gatheringManualDeployData(): %w", err)
	}

	return err
}

func (env *Env) taskClose(w http.ResponseWriter, r *http.Request) {
	if err := r.ParseForm(); err != nil {
		fmt.Printf("taskClose(): parse form: %v", err)
		w.WriteHeader(http.StatusBadRequest)
		_, err = w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("taskClose(): failed to write response: %v\n", err)
		}
		return
	}

	id := r.FormValue("id")
	if _, err := uuid.FromString(id); err != nil {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("taskClose(): failed to write response: %v\n", err)
		}
		return
	}

	statusCode := baldrerrors.CodeOk
	status := r.FormValue("status")

	if status != "" {
		code, err := strconv.Atoi(status)
		if err != nil {
			w.WriteHeader(http.StatusBadRequest)
			_, err := w.Write([]byte("Bad request"))
			if err != nil {
				fmt.Printf("taskClose(): failed to write response: %v\n", err)
			}
			return
		}

		statusCode = baldrerrors.ErrorCode(code)
	}

	config, err := env.db.DeploymentInfo(id)
	if err != nil {
		fmt.Printf("taskClose(): %v\n", err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("taskClose(): failed to write response: %v\n", err)
		}
		return
	}

	if config.ID != id || config.Status != models.DeployStatusOpen {
		w.WriteHeader(http.StatusNotFound)
		_, err = w.Write([]byte("Not found"))
		if err != nil {
			fmt.Printf("taskClose(): failed to write response: %v\n", err)
		}
		return
	}

	if statusCode != baldrerrors.CodeOk {
		config.ErrorCode = statusCode
	}

	if config.Message == "" {
		config.Message = config.ErrorCode.String()
	} else {
		config.Message += fmt.Sprintf(", %s", config.ErrorCode.String())
	}

	err = env.db.CloseTask(id, config.ErrorCode, config.Message)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		_, err := w.Write([]byte("Internal Server Error"))
		if err != nil {
			fmt.Printf("taskClose(): failed to write response: %v\n", err)
		}
		return
	}

	w.WriteHeader(http.StatusOK)
	_, err = w.Write([]byte("Ok"))
	if err != nil {
		fmt.Printf("taskClose(): failed to write response: %v\n", err)
	}
}

func (env *Env) preseedGenScript(w http.ResponseWriter, r *http.Request) {
	if err := r.ParseForm(); err != nil {
		fmt.Printf("preseedGenScript(): parse form: %v", err)
		w.WriteHeader(http.StatusBadRequest)
		_, err = w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("preseedGenScript(): failed to write response: %v\n", err)
		}
		return
	}

	id := r.FormValue("id")
	if _, err := uuid.FromString(id); err != nil {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("preseedGenScript(): failed to write response: %v\n", err)
		}
		return
	}

	preseedType := r.FormValue("type")
	if !(preseedType == preseedTypeUser || preseedType == preseedTypeZombie || preseedType == preseedTypeZombieGPU) {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("preseedGenScript(): failed to write response: %v\n", err)
		}
		return
	}

	linuxRelease := r.FormValue("os_release")
	if !(linuxRelease == linuxReleaseBionic || linuxRelease == linuxReleaseFocal) {
		w.WriteHeader(http.StatusBadRequest)
		_, err := w.Write([]byte("Bad request"))
		if err != nil {
			fmt.Printf("preseedGenScript(): failed to write response: %v\n", err)
		}
		return
	}

	config, err := env.db.DeploymentInfo(id)
	if err != nil {
		fmt.Printf("preseedGenScript(): %v\n", err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("preseedGenScript(): failed to write response: %v\n", err)
		}
		return
	}

	if config.ID != id || config.Status != models.DeployStatusOpen {
		w.WriteHeader(http.StatusNotFound)
		_, err = w.Write([]byte("Not found"))
		if err != nil {
			fmt.Printf("preseedGenScript(): failed to write response: %v\n", err)
		}
		return
	}

	config.Options[models.OptionProdAPIHost] = env.apiProdURL
	config.Options[models.OptionTestAPIHost] = env.apiTestURL
	config.Options[models.OptionEnvironment] = env.currentEnv
	var tmpl string
	if preseedType == preseedTypeUser {
		tmpl = userPreseedTemplate
	} else if preseedType == preseedTypeZombieGPU {
		tmpl = zombieGPUPreseedTemplate
	} else if preseedType == preseedTypeZombie {
		tmpl = zombiePreseedTemplate
	}

	if linuxRelease == linuxReleaseFocal {
		config.Options[models.OptionOSVersion] = linuxReleaseFocalVersion
		config.Options[models.OptionOSReleaseName] = linuxReleaseFocal
	} else if linuxRelease == linuxReleaseBionic {
		config.Options[models.OptionOSVersion] = linuxReleaseBionicVersion
		config.Options[models.OptionOSReleaseName] = linuxReleaseBionic
	}

	// Execute template
	err = executeTemplate(config, template.FuncMap{}, w, tmpl)
	if err != nil {
		fmt.Printf("preseedGenScript(): %v\n", err)
		w.WriteHeader(http.StatusInternalServerError)
		_, err = w.Write([]byte("Internal server error"))
		if err != nil {
			fmt.Printf("preseedGenScript(): failed to write response: %v\n", err)
		}
	}
}
