package main

import (
	"fmt"
	"net"
	"net/http"
	"strings"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/helpdesk/infra/baldr/internal/baldrerrors"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/models"
)

func iPXERequestData(r *http.Request, dep *models.Deploy) error {
	var err error
	if err = r.ParseForm(); err != nil {
		err = fmt.Errorf("iPXERequestData(): parse iPXE request: %w", err)
		dep.ErrorCode = baldrerrors.CodeBadRequest
		dep.Message = dep.ErrorCode.String()
		return err
	}

	dep.SerialNumber = strings.ToLower(strings.TrimSpace(r.FormValue("serialnumber")))
	if dep.SerialNumber == "" {
		dep.SerialNumber = models.UndefinedSerialNumber
	}
	dep.Options[models.OptionHWSerialNumber] = dep.SerialNumber

	for _, mac := range strings.Split(strings.ToLower(r.FormValue("mac")), ",") {
		hardwareAddress, err := net.ParseMAC(strings.ToLower(mac))
		if err != nil {
			continue
		}
		dep.MACAddresses = append(dep.MACAddresses, hardwareAddress)
	}

	hwUUID := strings.ToLower(r.FormValue("uuid"))
	if valid := isValidUUID(hwUUID); !valid {
		hwUUID = models.UndefinedUUID
	}
	dep.Options[models.OptionHWUUID] = hwUUID

	uefiMode := strings.ToLower(strings.TrimSpace(r.FormValue("platform")))
	if !(uefiMode == models.IPXEModeBIOS || uefiMode == models.IPXEModeUEFI) {
		uefiMode = models.UndefinedIPXEMode
	}
	dep.Options[models.OptionIPXEUEFIMode] = uefiMode

	manufacturer := strings.ToLower(strings.TrimSpace(r.FormValue("manufacturer")))
	if manufacturer == "" {
		manufacturer = models.UnknownManufacturer
	}
	dep.Options[models.OptionHWManufacturer] = manufacturer

	product := strings.ToLower(strings.TrimSpace(r.FormValue("product")))
	if product == "" {
		product = models.UnknownProduct
	}
	dep.Options[models.OptionHWModel] = product

	dep.NextServer = net.ParseIP(r.FormValue("nextserver"))
	if dep.NextServer == nil {
		dep.NextServer = net.ParseIP(models.UndefinedNextServer)
	}

	ipxeFileName := strings.ToLower(strings.TrimSpace(r.FormValue("filename")))
	if ipxeFileName == "" {
		ipxeFileName = models.UndefinedIPXEFileName
	}
	dep.Options[models.OptionIPXEFileName] = ipxeFileName

	hostname := strings.ToLower(strings.TrimSpace(r.FormValue("hostname")))
	if hostname == "" {
		hostname = models.UndefinedIPXEHostname
	}
	dep.Options[models.OptionIPXEHostname] = hostname

	asset := strings.ToLower(strings.TrimSpace(r.FormValue("asset")))
	if asset == "" {
		asset = models.UndefinedHWAsset
	}
	dep.Options[models.OptionHWAsset] = asset

	dhcpUserClass := strings.ToLower(strings.TrimSpace(r.FormValue("user-class")))
	if dhcpUserClass == "" {
		dhcpUserClass = models.UndefinedIPXEDHCPUserClass
	}
	dep.Options[models.OptionIPXEDHCPUserClass] = dhcpUserClass

	if dep.IPAddress = net.ParseIP(r.RemoteAddr); dep.IPAddress == nil {
		dep.IPAddress = net.ParseIP(models.UndefinedIPAddress)
	}

	if dep.IPAddress.String() == models.UndefinedIPAddress ||
		dep.Options[models.OptionIPXEUEFIMode] == models.UndefinedIPXEMode ||
		(len(dep.MACAddresses) == 1 && len(dep.MACAddresses) == 0) {
		dep.ErrorCode = baldrerrors.CodeBadRequest
		dep.Message = dep.ErrorCode.String()
		return fmt.Errorf("iPXERequestData(): bad request: remote IP %q, boot mode %q, MAC addresses: %q",
			r.RemoteAddr, r.FormValue("platform"), r.FormValue("mac"))
	}

	return nil
}

func isValidUUID(id string) bool {
	_, err := uuid.FromString(id)
	return err == nil
}
