package main

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log"
	"net/http"
	"strings"

	"github.com/go-openapi/runtime/middleware/header"

	"a.yandex-team.ru/library/go/yandex/yav"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

type malformedRequest struct {
	status int
	msg    string
}

func (mr *malformedRequest) Error() string {
	return mr.msg
}

type keyEscrowRequest struct {
	Key             string `json:"key"`
	Passphrase      string `json:"passphrase"`
	PartitionGUID   string `json:"partition_guid"`
	DeployID        string `json:"deploy_id"`
	SerialNumber    string `json:"serial_number"`
	InventoryNumber string `json:"inventory_number"`
	Hostname        string `json:"hostname"`
}

type keyEscrowResponse struct {
	StatusCode int    `json:"status_code"`
	Message    string `json:"message"`
}

func (env *Env) keyEscrow(w http.ResponseWriter, r *http.Request) {
	response := env.handlingKeyEscrow(w, r)
	js, err := json.Marshal(response)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(response.StatusCode)
	_, err = w.Write(js)
	if err != nil {
		log.Printf("keyEscrow(): write response %q: %s", js, err.Error())
	}
}

func (env *Env) handlingKeyEscrow(w http.ResponseWriter, r *http.Request) keyEscrowResponse {
	var keyRequest keyEscrowRequest
	var keyResponse keyEscrowResponse
	var err error

	err = handlingRequest(w, r, &keyRequest)
	if err != nil {
		log.Printf("handlingKeyEscrow(): hostname %q, inventory number %q, serial number %q, deploy ID %q, partition ID %q: %s",
			keyRequest.Hostname, keyRequest.InventoryNumber, keyRequest.SerialNumber, keyRequest.DeployID, keyRequest.PartitionGUID, err.Error())
		var mr *malformedRequest
		if errors.As(err, &mr) {
			keyResponse.StatusCode = mr.status
			keyResponse.Message = mr.msg
		} else {
			keyResponse.StatusCode = http.StatusInternalServerError
			keyResponse.Message = http.StatusText(http.StatusInternalServerError)
		}
		return keyResponse
	}

	keyResponse.Message, err = handlingYAV(env.yav, keyRequest)
	if err != nil {
		log.Printf("handlingKeyEscrow(): hostname %q, inventory number %q, serial number %q, deploy ID %q, partition ID %q: %s",
			keyRequest.Hostname, keyRequest.InventoryNumber, keyRequest.SerialNumber, keyRequest.DeployID, keyRequest.PartitionGUID, err.Error())
		keyResponse.StatusCode = http.StatusInternalServerError
		return keyResponse
	}

	keyResponse.StatusCode = http.StatusCreated
	return keyResponse
}

func handlingRequest(w http.ResponseWriter, r *http.Request, dest *keyEscrowRequest) error {
	err := decodeJSONBody(w, r, &dest)
	if err != nil {
		return fmt.Errorf("handlingRequest(): decode JSON: %w", err)
	}

	err = checkKeyEscrowRequest(dest)
	if err != nil {
		return fmt.Errorf("handlingRequest(): decode JSON: %w", err)
	}

	return nil
}

func decodeJSONBody(w http.ResponseWriter, r *http.Request, dst interface{}) error {
	if r.Header.Get("Content-Type") != "" {
		value, _ := header.ParseValueAndParams(r.Header, "Content-Type")
		if value != "application/json" {
			msg := "Content-Type header is not application/json"
			return &malformedRequest{status: http.StatusUnsupportedMediaType, msg: msg}
		}
	}

	r.Body = http.MaxBytesReader(w, r.Body, 1048576)

	dec := json.NewDecoder(r.Body)
	dec.DisallowUnknownFields()

	err := dec.Decode(&dst)
	if err != nil {
		var syntaxError *json.SyntaxError
		var unmarshalTypeError *json.UnmarshalTypeError

		switch {
		case errors.As(err, &syntaxError):
			msg := fmt.Sprintf("Request body contains badly-formed JSON (at position %d)", syntaxError.Offset)
			return &malformedRequest{status: http.StatusBadRequest, msg: msg}

		case errors.Is(err, io.ErrUnexpectedEOF):
			msg := "Request body contains badly-formed JSON"
			return &malformedRequest{status: http.StatusBadRequest, msg: msg}

		case errors.As(err, &unmarshalTypeError):
			msg := fmt.Sprintf("Request body contains an invalid value for the %q field (at position %d)",
				unmarshalTypeError.Field, unmarshalTypeError.Offset)
			return &malformedRequest{status: http.StatusBadRequest, msg: msg}

		case strings.HasPrefix(err.Error(), "json: unknown field "):
			fieldName := strings.TrimPrefix(err.Error(), "json: unknown field ")
			msg := fmt.Sprintf("Request body contains unknown field %s", fieldName)
			return &malformedRequest{status: http.StatusBadRequest, msg: msg}

		case errors.Is(err, io.EOF):
			msg := "Request body must not be empty"
			return &malformedRequest{status: http.StatusBadRequest, msg: msg}

		case err.Error() == "http: request body too large":
			msg := "Request body must not be larger than 1MB"
			return &malformedRequest{status: http.StatusRequestEntityTooLarge, msg: msg}

		default:
			return err
		}
	}

	err = dec.Decode(&struct{}{})
	if err != io.EOF {
		msg := "Request body must only contain a single JSON object"
		return &malformedRequest{status: http.StatusBadRequest, msg: msg}
	}

	return nil
}

func checkKeyEscrowRequest(r *keyEscrowRequest) error {
	if !isValidUUID(r.DeployID) {
		msg := fmt.Sprintf("invalid deploy ID %q", r.DeployID)
		return &malformedRequest{status: http.StatusBadRequest, msg: msg}
	}

	if r.InventoryNumber == "" {
		msg := "inventory number is required"
		return &malformedRequest{status: http.StatusBadRequest, msg: msg}
	}

	return nil
}

func handlingYAV(yc *httpyav.Client, keyReq keyEscrowRequest) (string, error) {
	const (
		secretTag       = "hd_luks_recovery_key"
		abcServiceID    = 24584
		abcServiceScope = "administration"
	)

	ctx := context.Background()
	secret, err := searchSecrets(yc, ctx, keyReq.PartitionGUID, []string{secretTag})
	if err != nil {
		return "fail to search secret", fmt.Errorf("handlingYAV(): %w", err)
	}

	err = createYAVSecret(yc, ctx, keyReq.PartitionGUID, []string{secretTag}, &secret)
	if err != nil {
		return "fail to create secret", fmt.Errorf("handlingYAV(): %w", err)
	}

	err = addYAVSecretRole(yc, ctx, abcServiceID, abcServiceScope, &secret)
	if err != nil {
		return "fail to add secret role", fmt.Errorf("handlingYAV(): %w", err)
	}

	var versionID string
	versionID, err = addYAVSecretVersion(yc, ctx, keyReq, secret.SecretUUID)
	if err != nil {
		return "fail to create secret version", fmt.Errorf("handlingYAV(): %w", err)
	}

	return fmt.Sprintf("%s %s", secret.SecretUUID, versionID), nil
}

func searchSecrets(yc *httpyav.Client, ctx context.Context, partitionGUID string, secretTags []string) (secret yav.Secret, err error) {
	yavGetSecretRequest := yav.GetSecretsRequest{
		Asc:               false,
		OrderBy:           "created_at",
		Page:              0,
		PageSize:          0,
		Query:             fmt.Sprintf("recovery_key_%s", partitionGUID),
		QueryType:         "exact",
		Role:              "",
		Tags:              secretTags,
		WithHiddenSecrets: false,
		WithTVMApps:       false,
		Without:           nil,
		Yours:             true,
	}

	var yavGetSecretResponse *yav.GetSecretsResponse
	yavGetSecretResponse, err = yc.GetSecrets(ctx, yavGetSecretRequest)
	if err != nil {
		err = fmt.Errorf("searchSecrets(): %w", err)
		return
	}

	if yavGetSecretResponse == nil {
		err = fmt.Errorf("searchSecrets(): empty response")
		return
	}

	if yavGetSecretResponse.Status != "ok" {
		err = fmt.Errorf("searchSecrets(): status %q, code%q", yavGetSecretResponse.Status, yavGetSecretResponse.Code)
	}

	if len(yavGetSecretResponse.Secrets) != 0 {
		secret = yavGetSecretResponse.Secrets[0]
	}

	return
}

func createYAVSecret(yc *httpyav.Client, ctx context.Context, partitionGUID string, secretTags []string, secret *yav.Secret) error {
	if secret.SecretUUID != "" {
		return nil
	}

	yavSecretRequest := yav.SecretRequest{
		Name:    fmt.Sprintf("recovery_key_%s", partitionGUID),
		Comment: "",
		State:   "",
		Tags:    secretTags,
	}

	yavSecretResponse, err := yc.CreateSecret(ctx, yavSecretRequest)
	if err != nil {
		return fmt.Errorf("createYAVSecret(): %w", err)
	}

	if yavSecretResponse == nil {
		return fmt.Errorf("createYAVSecret(): empty response")
	}

	if yavSecretResponse.Status != "ok" {
		return fmt.Errorf("createYAVSecret(): status %q, code%q", yavSecretResponse.Status, yavSecretResponse.Code)
	}

	secret.SecretUUID = yavSecretResponse.SecretUUID

	return nil
}

func addYAVSecretRole(yc *httpyav.Client, ctx context.Context, abcServiceID uint, abcServiceScope string, secret *yav.Secret) error {
	secretRoleRequest := yav.SecretRoleRequest{
		AbcID:     abcServiceID,
		AbcScope:  abcServiceScope,
		AbcRoleID: 0,
		Login:     "",
		Role:      "OWNER",
		StaffID:   0,
		UID:       0,
	}

	if existRoleInSecret(*secret, secretRoleRequest) {
		return nil
	}

	yavResponse, err := yc.AddSecretRole(ctx, secret.SecretUUID, secretRoleRequest)
	if err != nil {
		return fmt.Errorf("addYAVSecretRole(): %w", err)
	}

	if yavResponse == nil {
		return fmt.Errorf("addYAVSecretRole(): empty response")
	}

	if yavResponse.Status != "ok" {
		return fmt.Errorf("addYAVSecretRole(): status %q, code%q", yavResponse.Status, yavResponse.Code)
	}

	return nil
}

func existRoleInSecret(secret yav.Secret, roleRequest yav.SecretRoleRequest) bool {
	for _, role := range secret.SecretRoles {
		if role.AbcID == roleRequest.AbcID &&
			role.AbcScopeName == roleRequest.AbcScope &&
			role.RoleSlug == roleRequest.Role {
			return true
		}
	}

	return false
}

func addYAVSecretVersion(yc *httpyav.Client, ctx context.Context, keyReq keyEscrowRequest, secretID string) (string, error) {
	comment := fmt.Sprintf("hostname: %s; inventory number: %s; serial number: %s; deploy ID: %s",
		keyReq.Hostname, keyReq.InventoryNumber, keyReq.SerialNumber, keyReq.DeployID)
	keySecret := yav.Value{
		Key:      "key",
		Value:    keyReq.Key,
		Encoding: "",
	}
	passphraseSecret := yav.Value{
		Key:      "passphrase",
		Value:    keyReq.Passphrase,
		Encoding: "",
	}
	yavCreateVersionRequest := yav.CreateVersionRequest{
		Comment: comment,
		Values:  []yav.Value{keySecret, passphraseSecret},
	}

	yavCreateVersionResponse, err := yc.CreateVersion(ctx, secretID, yavCreateVersionRequest)
	if err != nil {
		return "", fmt.Errorf("addYAVSecretVersion(): %w", err)
	}

	if yavCreateVersionResponse == nil {
		return "", fmt.Errorf("addYAVSecretVersion(): empty response")
	}

	if yavCreateVersionResponse.Status != "ok" {
		return "", fmt.Errorf("addYAVSecretVersion(): status %q, code%q", yavCreateVersionResponse.Status, yavCreateVersionResponse.Code)
	}

	return yavCreateVersionResponse.VersionUUID, nil
}
