package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/helpdesk/infra/baldr/internal/baldrerrors"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/models"
)

const (
	OFRTagRegularComputer        = "newcomputerofr"
	OFRTagForNewUser             = "newuserofr"
	OFRTagForInterview           = "computerforinterviewofr"
	OFRTagBrowserPerformanceTest = "computerforperformancetestsofr"
	OFRTagWithoutOFR             = "withoutofr"
	OFRTagZombie                 = "newzombieofr"
)

const (
	DeployTypeAnimals                = "Animals"
	DeployTypeAnimalsTest            = "AnimalsTest"
	DeployTypeBrowserPerformanceTest = "BrowserPerfTest"
	DeployTypeDefault                = "WithoutOFR"
	DeployTypeForInterview           = "Interview"
	DeployTypeForNewUser             = "NewUser"
	DeployTypeRegularComputer        = "Computer"
	DeployTypeWipe                   = "Wipe"
	DeployTypeZombie                 = "Zombie"
)

type ofr struct {
	Result []ofrTask
	Error  bool
}

type ofrError struct {
	Result bool
	Error  string
}

type ofrTask struct {
	STTask        string   `json:"key"`
	STTaskStatus  string   `json:"status"`
	Tags          []string `json:"type"`
	ComputerName  string   `json:"computerName"`
	UserName      string   `json:"user"`
	InDomain      string   `json:"domain"`
	NeedOffice    string   `json:"office"`
	OS            string   `json:"os"`
	OSVersion     string   `json:"osVersion"`
	LocaleProfile string   `json:"localeProfile"`
}

type ofr2 struct {
	Result ofrTaskInfo
	Error  bool
}

type ofrAttr struct {
	ID    string `json:"attribute_id"`
	Name  string `json:"attribute_name"`
	Value string `json:"attribute_value"`
}

type ofrTaskInfo struct {
	Attributes []ofrAttr `json:"attrs"`
}

func ofrData(config bot, dep *models.Deploy) ([]ofrTask, error) {
	var ofrResp ofr
	var tasks []ofrTask

	client := &http.Client{}
	req, err := http.NewRequest("GET", config.baseURL+url.QueryEscape(dep.InventoryNumber), nil)
	if err != nil {
		dep.ErrorCode = baldrerrors.CodeHTTPRequestError
		dep.Message = "Request to OFR failed"
		return tasks, fmt.Errorf("ofrData(): create new request: %w", err)
	}

	req.Header.Add("Authorization", "OAuth "+config.token)
	req.Header.Add("Accept", "application/json")
	var resp *http.Response
	for i := 0; i < 5; i++ {
		time.Sleep(time.Duration(i) * time.Second)
		resp, err = client.Do(req)
		if err != nil {
			continue
		} else {
			break
		}
	}
	if err != nil {
		dep.ErrorCode = baldrerrors.CodeHTTPRequestError
		dep.Message = "Request to OFR failed"
		return tasks, fmt.Errorf("ofrData(): get response: %w", err)
	}
	defer func(response *http.Response, dep *models.Deploy) {
		err := response.Body.Close()
		if err != nil {
			fmt.Printf("ERROR: %s: close OFR response body failed: %v\n", dep.ID, err)
		}
	}(resp, dep)

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		dep.ErrorCode = baldrerrors.CodeHTTPResponseError
		dep.Message = "OFR error"
		return tasks, fmt.Errorf("ofrData(): read response body: %w", err)
	}

	if resp.StatusCode != 200 && resp.StatusCode != 404 {
		var ofrResp ofrError
		err = json.Unmarshal(body, &ofrResp)
		if err != nil {
			dep.ErrorCode = baldrerrors.CodeOFRUnmarshalError
			dep.Message = "OFR error"
			return tasks, fmt.Errorf("ofrData(): response status code %d: unmarshal data: %w", resp.StatusCode, err)
		}

		dep.ErrorCode = baldrerrors.CodeOFRError
		dep.Message = "OFR error: " + resp.Status
		return tasks, fmt.Errorf("ofrData(): %s", ofrResp.Error)
	}

	if resp.StatusCode == 200 {
		err = json.Unmarshal(body, &ofrResp)
		if err != nil {
			dep.ErrorCode = baldrerrors.CodeOFRUnmarshalError
			dep.Message = "OFR error"
			return tasks, fmt.Errorf("ofrData(): response status code %d: unmarshal data: %w", resp.StatusCode, err)
		}

		tasks = ofrResp.Result
	}

	return tasks, nil
}

func (env *Env) getZombieHostname(id string) (string, error) {
	const AttrComputerName = "computerName2"
	config := bot{
		baseURL: "https://bot.yandex-team.ru/api/v2/ofr2/task/id/",
		token:   env.bot.token,
	}

	client := &http.Client{}
	req, err := http.NewRequest("GET", config.baseURL+url.QueryEscape(id), nil)
	if err != nil {
		return "", fmt.Errorf("getZombieHostname(): create new request: %w", err)
	}

	req.Header.Add("Authorization", "OAuth "+config.token)
	req.Header.Add("Accept", "application/json")
	var resp *http.Response
	for i := 0; i < 5; i++ {
		time.Sleep(time.Duration(i) * time.Second)
		resp, err = client.Do(req)
		if err != nil {
			continue
		} else {
			break
		}
	}
	if err != nil {
		return "", fmt.Errorf("getZombieHostname(): get response: %w", err)
	}
	defer func(response *http.Response) {
		err := response.Body.Close()
		if err != nil {
			fmt.Printf("ERROR: close OFR response body failed: %v\n", err)
		}
	}(resp)

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return "", fmt.Errorf("getZombieHostname(): read response body: %w", err)
	}

	if resp.StatusCode != 200 {
		var ofrResp ofrError
		err = json.Unmarshal(body, &ofrResp)
		if err != nil {
			return "", fmt.Errorf("getZombieHostname(): response status code %d: unmarshal data: %w", resp.StatusCode, err)
		}
		return "", fmt.Errorf("getZombieHostname(): response code %d: %s", resp.StatusCode, ofrResp.Error)
	}

	var ofrResp ofr2
	err = json.Unmarshal(body, &ofrResp)
	if err != nil {
		return "", fmt.Errorf("getZombieHostname(): response status code %d: unmarshal data: %w", resp.StatusCode, err)
	}

	for _, attr := range ofrResp.Result.Attributes {
		if attr.Name == AttrComputerName {
			return attr.Value, nil
		}
	}

	return "", fmt.Errorf("getZombieHostname(): attribute %q not found", AttrComputerName)
}

func (env *Env) getTasks(dep *models.Deploy) error {
	var allTasks []ofrTask

	config := bot{
		baseURL: env.bot.baseURL + "windeploy/inventorynumber/",
		token:   env.bot.token,
	}
	tasks, err := ofrData(config, dep)
	if err != nil {
		return fmt.Errorf("getTasks(): find common tasks: %w", err)
	}
	allTasks = append(allTasks, tasks...)

	config = bot{
		baseURL: env.bot.baseURL + "zombie/inventorynumber/",
		token:   env.bot.token,
	}
	tasks, err = ofrData(config, dep)
	if err != nil {
		return fmt.Errorf("getTasks(): find zombie tasks: %w", err)
	}

	for index, task := range tasks {
		hostname, err := env.getZombieHostname(task.STTask)
		if err != nil {
			return fmt.Errorf("getTasks(): find zombie hostname for task %s: %w", task.STTask, err)
		}

		tasks[index].ComputerName = hostname
	}
	allTasks = append(allTasks, tasks...)

	if len(allTasks) > 1 {
		var taskList string
		for i, task := range allTasks {
			if i != 0 {
				taskList += ", "
			}
			taskList += task.STTask
		}

		dep.ErrorCode = baldrerrors.CodeMoreThanOneOFRFound
		dep.Message = "More than one OFR task found: " + taskList
		return fmt.Errorf("getTasks(): %s", dep.Message)
	}

	if len(allTasks) == 0 || (allTasks[0].InDomain == "yes" && strings.HasPrefix(allTasks[0].OS, "Windows")) {
		task := ofrTask{
			STTask:        "WithoutOFR",
			STTaskStatus:  "None",
			Tags:          []string{OFRTagWithoutOFR},
			ComputerName:  fmt.Sprintf("i%s", dep.InventoryNumber),
			UserName:      "",
			InDomain:      "yes",
			NeedOffice:    "yes",
			OS:            "Windows 10",
			OSVersion:     "",
			LocaleProfile: "",
		}
		allTasks = []ofrTask{task}
	}

	dep.Options[models.OptionSTTask] = allTasks[0].STTask
	dep.Options[models.OptionSTTaskStatus] = strings.ToLower(allTasks[0].STTaskStatus)
	dep.Options[models.OptionComputerName] = strings.ToLower(allTasks[0].ComputerName)
	dep.Options[models.OptionUserName] = strings.ToLower(allTasks[0].UserName)
	dep.Options[models.OptionOS] = allTasks[0].OS
	dep.Options[models.OptionOSVersion] = allTasks[0].OSVersion
	dep.Options[models.OptionLanguageProfile] = strings.ToLower(allTasks[0].LocaleProfile)

	if allTasks[0].NeedOffice == "yes" {
		dep.Options[models.OptionWithOffice] = "true"
	} else {
		dep.Options[models.OptionWithOffice] = "false"
	}

	if allTasks[0].InDomain == "yes" {
		dep.Options[models.OptionWithDomain] = "true"
	} else {
		dep.Options[models.OptionWithDomain] = "false"
	}

	var deployTags []string

	validTags := make(map[string]bool, 6)
	validTags[OFRTagRegularComputer] = true
	validTags[OFRTagForNewUser] = true
	validTags[OFRTagForInterview] = true
	validTags[OFRTagBrowserPerformanceTest] = true
	validTags[OFRTagWithoutOFR] = true
	validTags[OFRTagZombie] = true

	for _, tag := range allTasks[0].Tags {
		if _, ok := validTags[strings.ToLower(tag)]; ok {
			if exist := stringInSlice(tag, deployTags); !exist {
				deployTags = append(deployTags, tag)
			}
		}
	}

	if len(deployTags) == 0 {
		dep.ErrorCode = baldrerrors.CodeOFRTagNotFound
		dep.Message = "OFR tag not found"
		return fmt.Errorf("getTasks(): %s", dep.Message)
	} else if len(deployTags) > 1 {
		dep.ErrorCode = baldrerrors.CodeMoreThanOneOFRTagFound
		dep.Message = "More than one OFR tag found " + strings.Join(deployTags, ", ")
		return fmt.Errorf("getTasks(): %s", dep.Message)
	}

	switch deployTags[0] {
	case OFRTagRegularComputer:
		dep.Options[models.OptionDeployType] = DeployTypeRegularComputer
	case OFRTagForNewUser:
		dep.Options[models.OptionDeployType] = DeployTypeForNewUser
		dep.Options[models.OptionWithDomain] = "true"
	case OFRTagForInterview:
		dep.Options[models.OptionDeployType] = DeployTypeForInterview
		dep.Options[models.OptionComputerName] = "interview"
		dep.Options[models.OptionUserName] = "interviewee"
		dep.Options[models.OptionOS] = "Windows 10"
		dep.Options[models.OptionWithDomain] = "false"
		dep.Options[models.OptionWithOffice] = "false"
	case OFRTagBrowserPerformanceTest:
		dep.Options[models.OptionDeployType] = DeployTypeBrowserPerformanceTest
		dep.Options[models.OptionComputerName] = "browser-perftst"
		dep.Options[models.OptionUserName] = "zomb-browser"
		dep.Options[models.OptionWithDomain] = "false"
		dep.Options[models.OptionWithOffice] = "false"
	case OFRTagWithoutOFR:
		dep.Options[models.OptionDeployType] = DeployTypeDefault
	case OFRTagZombie:
		dep.Options[models.OptionDeployType] = DeployTypeZombie
		dep.Options[models.OptionWithDomain] = "false"
		dep.Options[models.OptionWithOffice] = "false"
	}

	if templateName, ok := dep.Options[models.OptionDeploymentTemplate]; ok {
		deploymentTemplate, err := env.db.DeploymentTemplate(templateName)
		if err != nil {
			dep.ErrorCode = baldrerrors.CodeDBError
			dep.Message = "Get template data failed: " + templateName
			return fmt.Errorf("getTasks(): %w", err)
		}

		if dep.Options[models.OptionDeployType] == DeployTypeDefault {
			for key, value := range deploymentTemplate {
				dep.Options[key] = value
			}
		} else {
			dep.Message = fmt.Sprintf("The task %q found. The deployment will continue with the settings from the task",
				dep.Options[models.OptionSTTask])
		}

	}

	return nil
}

func (env *Env) findTask(dep *models.Deploy) error {
	err := env.getTasks(dep)
	if err != nil {
		return fmt.Errorf("findTask(): %w", err)
	}

	return nil
}
