package main

import (
	"bytes"
	"net"
	"strings"
	"testing"
	"text/template"
	"time"

	"a.yandex-team.ru/helpdesk/infra/baldr/internal/baldrerrors"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/models"
)

func TestBootstrapIni(t *testing.T) {
	var err error
	iPXE := `[Settings]
Priority=Default

[Default]
DeployRoot=\\baldr2.yandex-team.ru\ds$
UserID=smbuser
UserDomain=WORKGROUP
UserPassword=veryLongAndStrongPassword
KeyboardLocale=en-US
SkipBDDWelcome=YES`

	share := models.DeploymentShare{
		Hostname:  "baldr2.yandex-team.ru",
		SharePath: "ds$",
		Username:  "smbuser",
		Domain:    "WORKGROUP",
		Password:  "veryLongAndStrongPassword",
	}

	var resp bytes.Buffer
	err = executeTemplate(&share, template.FuncMap{}, &resp, bootstrapIniTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%s\n\nExpected:\n%s", resp.String(), iPXE)
	}
}

func TestStartupTestingIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set api_host test.baldr.yandex.net
set handler_url https://${api_host}/v1/ipxe

:netboot
menu iPXE boot menu
  item --gap -- ----------------------------------- Testing ----------------------------------
  item --gap -- ------------------------------ Standard workflow -----------------------------
  item auto-deploy   Automatic deploy system
  item --gap -- ----------------------------- Additional workflow ----------------------------
  item market-wh     Market Warehouse
  item market-std    Market Standard
  item win-11        Windows 11 Enterprise 21H2
  item win-pro       Windows 10 Professional
  item win21h1       Windows 10 Enterprise 21H1
  item win20h2       Windows 10 Enterprise 20H2
  item animals       Animals
  item animals-test  Animals (Test)
  item wipe          Wipe
  item linux         Linux Pre-deployment
  item --gap --	----------------------------- Change environment -----------------------------
  item change-env    Go to Production
  choose --default autodeploy --timeout 5000 target && goto ${target} || goto netboot

:auto-deploy
params
set idx:int32 0
  :loop isset ${net${idx}/mac} || goto loop_done
    isset ${macs} || set macs ${net${idx}/mac:hexhyp} && inc idx && goto loop
    set macs ${macs},${net${idx}/mac:hexhyp}
    inc idx && goto loop
  :loop_done
param mac ${macs}
param serialnumber ${serial}
param uuid ${uuid}
param platform ${platform}
param manufacturer ${manufacturer}
param product ${product}
param nextserver ${next-server}
param filename ${filename}
param hostname ${hostname}
param user_class ${user-class}
param asset ${asset}

chain --timeout 30000 ${handler_url}##params || goto auto-deploy

:market-wh
set handler_url https://${api_host}/v1/ipxe?template=market_warehouse
goto auto-deploy

:market-std
set handler_url https://${api_host}/v1/ipxe?template=market_standard
goto auto-deploy

:win-11
set handler_url https://${api_host}/v1/ipxe?template=windows_11_21h2
goto auto-deploy

:win-pro
set handler_url https://${api_host}/v1/ipxe?template=windows_pro
goto auto-deploy

:win21h1
set handler_url https://${api_host}/v1/ipxe?template=windows_21h1
goto auto-deploy

:win20h2
set handler_url https://${api_host}/v1/ipxe?template=windows_20h2
goto auto-deploy

:animals
set handler_url https://${api_host}/v1/ipxe?template=animals
goto auto-deploy

:animals-test
set handler_url https://${api_host}/v1/ipxe?template=animals_test
goto auto-deploy

:wipe
set handler_url https://${api_host}/v1/ipxe?template=wipe
goto auto-deploy

:linux
set handler_url https://${api_host}/v1/ipxe?template=linux_pre_deploy
goto auto-deploy

:change-env
set api_host api.baldr.yandex.net
chain https://${api_host}/v1/ipxe || goto change-env`

	dep := models.Deploy{}
	dep.Options = make(map[string]string, 5)

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvTest

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeStartupTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestStartupProdIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set api_host api.baldr.yandex.net
set handler_url https://${api_host}/v1/ipxe

:netboot
menu iPXE boot menu
  item --gap -- -------------------------------- Production ----------------------------------
  item --gap -- ------------------------------ Standard workflow -----------------------------
  item auto-deploy   Automatic deploy system
  item --gap -- ----------------------------- Additional workflow ----------------------------
  item market-wh     Market Warehouse
  item market-std    Market Standard
  item win-11        Windows 11 Enterprise 21H2
  item win-pro       Windows 10 Professional
  item win21h1       Windows 10 Enterprise 21H1
  item win20h2       Windows 10 Enterprise 20H2
  item animals       Animals
  item animals-test  Animals (Test)
  item wipe          Wipe
  item linux         Linux Pre-deployment
  item --gap --	----------------------------- Change environment -----------------------------
  item change-env    Go to Testing
  choose --default autodeploy --timeout 5000 target && goto ${target} || goto netboot

:auto-deploy
params
set idx:int32 0
  :loop isset ${net${idx}/mac} || goto loop_done
    isset ${macs} || set macs ${net${idx}/mac:hexhyp} && inc idx && goto loop
    set macs ${macs},${net${idx}/mac:hexhyp}
    inc idx && goto loop
  :loop_done
param mac ${macs}
param serialnumber ${serial}
param uuid ${uuid}
param platform ${platform}
param manufacturer ${manufacturer}
param product ${product}
param nextserver ${next-server}
param filename ${filename}
param hostname ${hostname}
param user_class ${user-class}
param asset ${asset}

chain --timeout 30000 ${handler_url}##params || goto auto-deploy

:market-wh
set handler_url https://${api_host}/v1/ipxe?template=market_warehouse
goto auto-deploy

:market-std
set handler_url https://${api_host}/v1/ipxe?template=market_standard
goto auto-deploy

:win-11
set handler_url https://${api_host}/v1/ipxe?template=windows_11_21h2
goto auto-deploy

:win-pro
set handler_url https://${api_host}/v1/ipxe?template=windows_pro
goto auto-deploy

:win21h1
set handler_url https://${api_host}/v1/ipxe?template=windows_21h1
goto auto-deploy

:win20h2
set handler_url https://${api_host}/v1/ipxe?template=windows_20h2
goto auto-deploy

:animals
set handler_url https://${api_host}/v1/ipxe?template=animals
goto auto-deploy

:animals-test
set handler_url https://${api_host}/v1/ipxe?template=animals_test
goto auto-deploy

:wipe
set handler_url https://${api_host}/v1/ipxe?template=wipe
goto auto-deploy

:linux
set handler_url https://${api_host}/v1/ipxe?template=linux_pre_deploy
goto auto-deploy

:change-env
set api_host test.baldr.yandex.net
chain https://${api_host}/v1/ipxe || goto change-env`

	dep := models.Deploy{}
	dep.Options = make(map[string]string, 5)

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvProd

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeStartupTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestAnimalsIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host test.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

echo -n Enter homer number: && read number
set animals_api_path proxy.yandex-team.ru/api/homer_preseed/
kernel https://${server}/linux/installer/18.04/amd64/linux
imgargs linux biosdevname=0 initrd=initrd.gz auto=true url=https://${animals_api_path}?homer=${number} priority=critical \
interface=auto debian-installer/locale=en_US.UTF-8 console-setup/layoutcode=en localechooser/translation/warn-light=true \
localechooser/translation/warn-severe=true console-setup/toggle=Alt+Shift
initrd https://${server}/linux/installer/18.04/amd64/initrd.gz
boot || shell
`
	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusClosed,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeAnimals
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.CloseTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvTest
	dep.Options[models.OptionDeploymentTemplate] = "animals"

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestErrorAnimalsIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host test.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

:menu
menu iPXE boot menu
item --gap --         ------------------------------------ Error -----------------------------------
item --gap --         Inventory number is not defined
item --gap --         ------------------------------------------------------------------------------
item --gap --         Inventory number: undefined
item --gap --         Serial number: To Be Filled By O.E.M.
item --gap --         MAC address 0: f8:75:a4:3b:91:d9
item --gap --         MAC address 1: 4c:1d:96:45:d8:7d
item --gap --         Manufacturer: To Be Filled By O.E.M.
item --gap --         Product: To Be Filled By O.E.M.
item --gap --         UEFI mode: efi
item --gap --         IP: 2a02:6b8:0:410:bd79:1dbc:38d7:1a4f
item --gap --         UUID: 90123456-7890-1234-5678-901234567890
item --gap --         ID: fae5063b-0e30-4c30-92ee-7d2f91a58659
item --gap --         ------------------------------------------------------------------------------
item continue         Try again
item --gap --         ------------------------------------------------------------------------------
item sh               iPXE shell
item reset            Reboot
item shutdown         Shutdown
choose target && goto ${target} || goto menu

:sh
shell

:reset
reboot

:shutdown
poweroff

:continue
set handler_url https://${api_host}/v1/ipxe?template=animals
params
set idx:int32 0
  :loop isset ${net${idx}/mac} || goto loop_done
    isset ${macs} || set macs ${net${idx}/mac:hexhyp} && inc idx && goto loop
    set macs ${macs},${net${idx}/mac:hexhyp}
    inc idx && goto loop
  :loop_done
param mac ${macs}
param serialnumber ${serial}
param uuid ${uuid}
param platform ${platform}
param manufacturer ${manufacturer}
param product ${product}
param nextserver ${next-server}
param filename ${filename}
param hostname ${hostname}
param user_class ${user-class}
param asset ${asset}
chain ${handler_url}##params || shell
`
	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "To Be Filled By O.E.M.",
		InventoryNumber:              models.UndefinedInventoryNumber,
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusClosed,
		ErrorCode:                    baldrerrors.CodeInventoryNumberNotDefined,
		Message:                      "Inventory number is not defined",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "To Be Filled By O.E.M."
	dep.Options[models.OptionHWModel] = "To Be Filled By O.E.M."
	dep.Options[models.OptionHWSerialNumber] = "To Be Filled By O.E.M."
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeAnimals
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.CloseTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvTest
	dep.Options[models.OptionDeploymentTemplate] = "animals"

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestAnimalsTestIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host test.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

echo -n Enter homer number: && read number
set animals_api_path proxy.yandex-team.ru/api/homer_preseed_test/
kernel https://${server}/linux/installer/18.04/amd64/linux
imgargs linux biosdevname=0 initrd=initrd.gz auto=true url=https://${animals_api_path}?homer=${number} priority=critical \
interface=auto debian-installer/locale=en_US.UTF-8 console-setup/layoutcode=en localechooser/translation/warn-light=true \
localechooser/translation/warn-severe=true console-setup/toggle=Alt+Shift
initrd https://${server}/linux/installer/18.04/amd64/initrd.gz
boot || shell
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusClosed,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeAnimalsTest
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.CloseTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvTest
	dep.Options[models.OptionDeploymentTemplate] = "animals_test"

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestWipeIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host test.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

kernel https://${server}/linux/wiper/vmlinuz
initrd https://${server}/linux/wiper/wiper
imgargs vmlinuz quiet initrd=wiper fb=false priority=critical
boot || shell
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusClosed,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeWipe
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.CloseTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvTest
	dep.Options[models.OptionDeploymentTemplate] = "wipe"

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestWindowsIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host api.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

kernel https://${server}/boot/wimboot/2.7.3/wimboot
initrd https://${server}/boot/bcd                                   BCD
initrd https://${server}/boot/boot.sdi                              boot.sdi
initrd https://${api_host}/v1/wimboot/config.xml?id=${ulid}         config.xml
initrd https://${api_host}/v1/wimboot/bootstrap.ini?id=${ulid}      bootstrap.ini
initrd https://${server}/boot/deploy/winpeshl.ini                   winpeshl.ini
initrd https://${server}/boot/deploy/deploy-util/v1/deploy-util.exe deploy-util.exe
initrd https://${server}/boot/deploy/wallpaper/prod/winpe.jpg       winpe.jpg
initrd https://${server}/boot/deploy/wim/10.0.22000.1-01/boot.wim   boot.wim

sync --timeout 15000
boot || shell
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeDefault
	dep.Options[models.OptionWithDomain] = "true"
	dep.Options[models.OptionWithOffice] = "true"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "WithoutOFR"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionDriversProfile] = "{Lenovo Thinkpad X1 Carbon 7th gen W10}"
	dep.Options[models.OptionADOU] = "OU=Yandex.Computers,OU=Windows10,OU=WinWorkstations,OU=AllWinWorkstations,DC=ld,DC=yandex,DC=ru"
	dep.Options[models.OptionOS] = "Windows 10"
	dep.Options[models.OptionOSEdition] = "Enterprise"
	dep.Options[models.OptionOSBuild] = "1909"
	dep.Options[models.OptionOSUpdated] = "May 2020"
	dep.Options[models.OptionOSProductKey] = "NPPR9-FWDCX-D2C8J-H872K-2YT43"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	dep.Options[models.OptionEnvironment] = deployEnvProd

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestWindowsZombieIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host api.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

kernel https://${server}/boot/wimboot/2.7.3/wimboot
initrd https://${server}/boot/bcd                                   BCD
initrd https://${server}/boot/boot.sdi                              boot.sdi
initrd https://${api_host}/v1/wimboot/config.xml?id=${ulid}         config.xml
initrd https://${api_host}/v1/wimboot/bootstrap.ini?id=${ulid}      bootstrap.ini
initrd https://${server}/boot/deploy/winpeshl.ini                   winpeshl.ini
initrd https://${server}/boot/deploy/deploy-util/v1/deploy-util.exe deploy-util.exe
initrd https://${server}/boot/deploy/wallpaper/prod/winpe.jpg       winpe.jpg
initrd https://${server}/boot/deploy/wim/10.0.22000.1-01/boot.wim   boot.wim

sync --timeout 15000
boot || shell
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionFQDN] = "zomb-prj-100.zombie.yandex-team.ru"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionDriversProfile] = "{Lenovo Thinkpad X1 Carbon 7th gen W10}"
	dep.Options[models.OptionOS] = "Windows 10"
	dep.Options[models.OptionOSEdition] = "Enterprise"
	dep.Options[models.OptionOSBuild] = "1909"
	dep.Options[models.OptionOSUpdated] = "May 2020"
	dep.Options[models.OptionOSProductKey] = "NPPR9-FWDCX-D2C8J-H872K-2YT43"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	dep.Options[models.OptionEnvironment] = deployEnvProd
	dep.Options[models.OptionDeployType] = DeployTypeZombie

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestLinuxUserIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host api.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

set base-url-focal https://${server}/linux/installer/20.04/amd64
set base-url-bionic https://${server}/linux/installer/18.04/amd64
:menu-linux-user
menu Linux
item --gap --       ------------------------------- Select version -------------------------------
item user-focal     Ubuntu 20.04 (Focal)
item --gap --       ------------------- Legacy versions ONLY for old hardware --------------------
item user-bionic    Ubuntu 18.04 (Bionic)
choose --default user-focal --timeout 5000 target && goto ${target} || goto menu-linux-user

:user-bionic
kernel ${base-url-bionic}/linux
initrd ${base-url-bionic}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=user&os_release=bionic netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell

:user-focal
kernel ${base-url-focal}/linux
initrd ${base-url-focal}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=user&os_release=focal netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeRegularComputer
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "WithoutOFR"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionUserName] = "knight"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	dep.Options[models.OptionEnvironment] = deployEnvProd

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestLinuxUserPreDeployIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host api.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

set base-url-focal https://${server}/linux/installer/20.04/amd64
set base-url-bionic https://${server}/linux/installer/18.04/amd64
goto user-focal
:menu-linux-user
menu Linux
item --gap --       ------------------------------- Select version -------------------------------
item user-focal     Ubuntu 20.04 (Focal)
item --gap --       ------------------- Legacy versions ONLY for old hardware --------------------
item user-bionic    Ubuntu 18.04 (Bionic)
choose --default user-focal --timeout 5000 target && goto ${target} || goto menu-linux-user

:user-bionic
kernel ${base-url-bionic}/linux
initrd ${base-url-bionic}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=user&os_release=bionic netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell

:user-focal
kernel ${base-url-focal}/linux
initrd ${base-url-focal}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=user&os_release=focal netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeRegularComputer
	dep.Options[models.OptionWithDomain] = ""
	dep.Options[models.OptionWithOffice] = ""
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = ""
	dep.Options[models.OptionLanguageProfile] = ""
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "WithoutOFR"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "changeme"
	dep.Options[models.OptionUserName] = "changeme"
	dep.Options[models.OptionOS] = "Linux"
	dep.Options[models.OptionOSBuild] = ""
	dep.Options[models.OptionOSEdition] = ""
	dep.Options[models.OptionOSUpdated] = ""
	dep.Options[models.OptionOSVersion] = linuxReleaseFocalVersion
	dep.Options[models.OptionOSReleaseName] = linuxReleaseFocal
	dep.Options[models.OptionOSProductKey] = ""
	dep.Options[models.OptionDeploymentTemplate] = "linux_pre_deploy"
	dep.Options[models.OptionADOU] = ""
	dep.Options[models.OptionDriversProfile] = ""

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	dep.Options[models.OptionEnvironment] = deployEnvProd

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestLinuxZombieIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host api.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

set base-url-focal https://${server}/linux/installer/20.04/amd64
set base-url-bionic https://${server}/linux/installer/18.04/amd64
:menu-zombie
menu Zombie
item --gap --          ------------------------------- Select zombie --------------------------------
item zombie-focal      Ubuntu 20.04 (Focal)
item zombie-focal-gpu  Ubuntu Server 20.04 (Focal)
item --gap --          ------------------- Legacy versions ONLY for old hardware --------------------
item zombie-bionic     Ubuntu 18.04 (Bionic)
item zombie-bionic-gpu Ubuntu Server 18.04 (Bionic)
choose --default zombie-focal --timeout 5000 target && goto ${target} || goto menu-zombie

:zombie-focal-gpu
kernel ${base-url-focal}/linux
initrd ${base-url-focal}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=zombie-gpu&os_release=focal netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell

:zombie-bionic-gpu
kernel ${base-url-bionic}/linux
initrd ${base-url-bionic}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=zombie-gpu&os_release=bionic netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell

:zombie-focal
kernel ${base-url-focal}/linux
initrd ${base-url-focal}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=zombie&os_release=focal netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell

:zombie-bionic
kernel ${base-url-bionic}/linux
initrd ${base-url-bionic}/initrd.gz
imgargs linux auto=true initrd=initrd.gz fb=false priority=critical preseed/locale=en_US.UTF-8 kbd-chooser/method=gb \
preseed/url=https://${api_host}/v1/preseed/?id=${ulid}&type=zombie&os_release=bionic netcfg/get_domain=unassigned-domain \
netcfg/choose_interface=auto
boot || shell
`
	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionFQDN] = "zomb-prj-100.zombie.yandex-team.ru"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "zomb-prj-100"
	dep.Options[models.OptionUserName] = "zomb-prj-100"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	dep.Options[models.OptionEnvironment] = deployEnvProd
	dep.Options[models.OptionDeployType] = DeployTypeZombie

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestWarningIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-spb.yandex-team.ru
set api_host api.baldr.yandex.net
set ulid fae5063b-0e30-4c30-92ee-7d2f91a58659

:menu
menu iPXE boot menu
item --gap --         ----------------------------------- Warning ----------------------------------
item --gap --         DriversProfileNotFound
item --gap --         ------------------------------------------------------------------------------
item --gap --         Inventory number: 100338961
item --gap --         Serial number: system serial number
item --gap --         MAC address 0: 10:c3:7b:45:95:9d
item --gap --         MAC address 1: 10:c3:7b:45:95:9c
item --gap --         Manufacturer: aquarius
item --gap --         Product: aquarius std s20 s61
item --gap --         UEFI mode: efi
item --gap --         IP: 2a02:6b8:0:2304:3a22:e2ff:fe17:4591
item --gap --         UUID: eb6d05f6-ffd7-4033-b555-64ad80dc8c88
item --gap --         ID: fae5063b-0e30-4c30-92ee-7d2f91a58659
item --gap --         ------------------------------------------------------------------------------
item continue         Continue
item --gap --         ------------------------------------------------------------------------------
item sh               iPXE shell
item reset            Reboot
item shutdown         Shutdown
choose target && goto ${target} || goto menu

:sh
shell

:reset
reboot

:shutdown
poweroff

:continue
kernel https://${server}/boot/wimboot/2.7.3/wimboot
initrd https://${server}/boot/bcd                                   BCD
initrd https://${server}/boot/boot.sdi                              boot.sdi
initrd https://${api_host}/v1/wimboot/config.xml?id=${ulid}         config.xml
initrd https://${api_host}/v1/wimboot/bootstrap.ini?id=${ulid}      bootstrap.ini
initrd https://${server}/boot/deploy/winpeshl.ini                   winpeshl.ini
initrd https://${server}/boot/deploy/deploy-util/v1/deploy-util.exe deploy-util.exe
initrd https://${server}/boot/deploy/wallpaper/prod/winpe.jpg       winpe.jpg
initrd https://${server}/boot/deploy/wim/10.0.22000.1-01/boot.wim   boot.wim

sync --timeout 15000
boot || shell
`
	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "system serial number",
		InventoryNumber:              "100338961",
		IPAddress:                    net.ParseIP("2a02:6b8:0:2304:3a22:e2ff:fe17:4591"),
		NextServer:                   net.ParseIP("77.88.12.85"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeDriversProfileNotFound,
		Message:                      "DriversProfileNotFound",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "eb6d05f6-ffd7-4033-b555-64ad80dc8c88"
	dep.Options[models.OptionHWManufacturer] = "aquarius"
	dep.Options[models.OptionHWModel] = "aquarius std s20 s61"
	dep.Options[models.OptionHWSerialNumber] = "system serial number"
	dep.Options[models.OptionHWInventoryNumber] = "100338961"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeDefault
	dep.Options[models.OptionWithDomain] = "true"
	dep.Options[models.OptionWithOffice] = "true"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Saint Petersburg"
	dep.Options[models.OptionOffice] = "Benua BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-spb.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "WithoutOFR"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i100338961"
	dep.Options[models.OptionDriversProfile] = "Nothing"
	dep.Options[models.OptionADOU] = "OU=Yandex.Computers,OU=Windows10,OU=WinWorkstations,OU=AllWinWorkstations,DC=ld,DC=yandex,DC=ru"
	dep.Options[models.OptionOS] = "Windows 10"
	dep.Options[models.OptionOSEdition] = "Enterprise"
	dep.Options[models.OptionOSBuild] = "1909"
	dep.Options[models.OptionOSUpdated] = "May 2020"
	dep.Options[models.OptionOSProductKey] = "NPPR9-FWDCX-D2C8J-H872K-2YT43"

	for _, macString := range []string{"10:c3:7b:45:95:9d", "10:c3:7b:45:95:9c"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	dep.Options[models.OptionEnvironment] = deployEnvProd

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestErrorIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host test.baldr.yandex.net
set ulid 4490057d-828e-485b-9f08-caa3a68b4baf

:menu
menu iPXE boot menu
item --gap --         ------------------------------------ Error -----------------------------------
item --gap --         legacy BIOS boot mode is not supported for lenovo 20qes1f100
item --gap --         ------------------------------------------------------------------------------
item --gap --         Inventory number: 104059015
item --gap --         Serial number: pf21eqtr
item --gap --         MAC address 0: f8:75:a4:3b:91:d9
item --gap --         MAC address 1: 4c:1d:96:45:d8:7d
item --gap --         Manufacturer: lenovo
item --gap --         Product: 20qes1f100
item --gap --         UEFI mode: pcbios
item --gap --         IP: 2a02:6b8:0:410:bd79:1dbc:38d7:1a4f
item --gap --         UUID: 90123456-7890-1234-5678-901234567890
item --gap --         ID: 4490057d-828e-485b-9f08-caa3a68b4baf
item --gap --         ------------------------------------------------------------------------------
item continue         Try again
item --gap --         ------------------------------------------------------------------------------
item sh               iPXE shell
item reset            Reboot
item shutdown         Shutdown
choose target && goto ${target} || goto menu

:sh
shell

:reset
reboot

:shutdown
poweroff

:continue
set handler_url https://${api_host}/v1/ipxe
params
set idx:int32 0
  :loop isset ${net${idx}/mac} || goto loop_done
    isset ${macs} || set macs ${net${idx}/mac:hexhyp} && inc idx && goto loop
    set macs ${macs},${net${idx}/mac:hexhyp}
    inc idx && goto loop
  :loop_done
param mac ${macs}
param serialnumber ${serial}
param uuid ${uuid}
param platform ${platform}
param manufacturer ${manufacturer}
param product ${product}
param nextserver ${next-server}
param filename ${filename}
param hostname ${hostname}
param user_class ${user-class}
param asset ${asset}
chain ${handler_url}##params || shell
`

	dep := models.Deploy{
		ID:                           "4490057d-828e-485b-9f08-caa3a68b4baf",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusClosed,
		ErrorCode:                    baldrerrors.CodeUnsupportedBootMode,
		Message:                      "legacy BIOS boot mode is not supported for lenovo 20qes1f100",
	}
	dep.Options[models.OptionID] = "4490057d-828e-485b-9f08-caa3a68b4baf"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionIPXEUEFIMode] = "pcbios"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeDefault
	dep.Options[models.OptionWithDomain] = "true"
	dep.Options[models.OptionWithOffice] = "true"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "WithoutOFR"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionDriversProfile] = "{Lenovo Thinkpad X1 Carbon 7th gen W10}"
	dep.Options[models.OptionADOU] = "OU=Yandex.Computers,OU=Windows10,OU=WinWorkstations,OU=AllWinWorkstations,DC=ld,DC=yandex,DC=ru"
	dep.Options[models.OptionOS] = "Windows 10"
	dep.Options[models.OptionOSEdition] = "Enterprise"
	dep.Options[models.OptionOSBuild] = "1909"
	dep.Options[models.OptionOSUpdated] = "May 2020"
	dep.Options[models.OptionOSProductKey] = "NPPR9-FWDCX-D2C8J-H872K-2YT43"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.CloseTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:19+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvTest

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestErrorTemplateIPXE(t *testing.T) {
	var err error
	iPXE := `#!ipxe

set server baldr-msk2.yandex-team.ru
set api_host test.baldr.yandex.net
set ulid 4490057d-828e-485b-9f08-caa3a68b4baf

:menu
menu iPXE boot menu
item --gap --         ------------------------------------ Error -----------------------------------
item --gap --         legacy BIOS boot mode is not supported for lenovo 20qes1f100
item --gap --         ------------------------------------------------------------------------------
item --gap --         Inventory number: 104059015
item --gap --         Serial number: pf21eqtr
item --gap --         MAC address 0: f8:75:a4:3b:91:d9
item --gap --         MAC address 1: 4c:1d:96:45:d8:7d
item --gap --         Manufacturer: lenovo
item --gap --         Product: 20qes1f100
item --gap --         UEFI mode: pcbios
item --gap --         IP: 2a02:6b8:0:410:bd79:1dbc:38d7:1a4f
item --gap --         UUID: 90123456-7890-1234-5678-901234567890
item --gap --         ID: 4490057d-828e-485b-9f08-caa3a68b4baf
item --gap --         ------------------------------------------------------------------------------
item continue         Try again
item --gap --         ------------------------------------------------------------------------------
item sh               iPXE shell
item reset            Reboot
item shutdown         Shutdown
choose target && goto ${target} || goto menu

:sh
shell

:reset
reboot

:shutdown
poweroff

:continue
set handler_url https://${api_host}/v1/ipxe?template=windows_20h2
params
set idx:int32 0
  :loop isset ${net${idx}/mac} || goto loop_done
    isset ${macs} || set macs ${net${idx}/mac:hexhyp} && inc idx && goto loop
    set macs ${macs},${net${idx}/mac:hexhyp}
    inc idx && goto loop
  :loop_done
param mac ${macs}
param serialnumber ${serial}
param uuid ${uuid}
param platform ${platform}
param manufacturer ${manufacturer}
param product ${product}
param nextserver ${next-server}
param filename ${filename}
param hostname ${hostname}
param user_class ${user-class}
param asset ${asset}
chain ${handler_url}##params || shell
`

	dep := models.Deploy{
		ID:                           "4490057d-828e-485b-9f08-caa3a68b4baf",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusClosed,
		ErrorCode:                    baldrerrors.CodeUnsupportedBootMode,
		Message:                      "legacy BIOS boot mode is not supported for lenovo 20qes1f100",
	}
	dep.Options[models.OptionID] = "4490057d-828e-485b-9f08-caa3a68b4baf"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionIPXEUEFIMode] = "pcbios"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeDefault
	dep.Options[models.OptionWithDomain] = "true"
	dep.Options[models.OptionWithOffice] = "true"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "WithoutOFR"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionDriversProfile] = "{Lenovo Thinkpad X1 Carbon 7th gen W10}"
	dep.Options[models.OptionADOU] = "OU=Yandex.Computers,OU=Windows10,OU=WinWorkstations,OU=AllWinWorkstations,DC=ld,DC=yandex,DC=ru"
	dep.Options[models.OptionOS] = "Windows 10"
	dep.Options[models.OptionOSEdition] = "Enterprise"
	dep.Options[models.OptionOSBuild] = "1909"
	dep.Options[models.OptionOSUpdated] = "May 2020"
	dep.Options[models.OptionOSProductKey] = "NPPR9-FWDCX-D2C8J-H872K-2YT43"
	dep.Options[models.OptionDeploymentTemplate] = "windows_20h2"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.CloseTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:19+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionEnvironment] = deployEnvTest

	f := template.FuncMap{"split": func(s string, n int) (result []string) {
		for i := 0; i < len(s); i += n {
			if i+n < len(s) {
				result = append(result, s[i:i+n])
			} else {
				result = append(result, s[i:])
			}
		}
		return result
	},
		"HasPrefix": strings.HasPrefix,
	}

	var resp bytes.Buffer
	err = executeTemplate(dep, f, &resp, ipxeTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != iPXE {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), iPXE)
	}
}

func TestPreseedFocalUser(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for focal)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string i104059015

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string focal
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select and options
d-i base-installer/kernel/image string linux-generic-hwe-20.04
d-i debian-installer/add-kernel-opts string mem_sleep_default=deep

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string knight
d-i passwd/username string knight
d-i passwd/user-password-crypted password $1$fENO7/iN$keuIeFNVdf7qIkzRNOd8/1
d-i passwd/auto-login boolean false
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false

d-i preseed/early_command string \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/packages/partman-crypto_101ubuntu4+luksopts1_amd64.udeb ; \
udpkg -i partman-crypto_101ubuntu4\+luksopts1_amd64.udeb
### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string crypto
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

d-i partman-crypto/passphrase password fJq6dAtU3maaT-X
d-i partman-crypto/passphrase-again password fJq6dAtU3maaT-X
d-i partman-crypto/luksformat_options string --type luks1
### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        user-efi-parts ::                                       \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                4096 4096 4096 ext4                             \
                        $primary{ }                             \
                        $bootable{ }                            \
                        method{ format } format{ }              \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ /boot }                     \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                16384 16384 16384 linux-swap                    \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select user-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, ubuntu-desktop

d-i pkgsel/include string acpi traceroute python3-pip network-manager-openvpn network-manager-openvpn-gnome wget libnss3-tools gnome-calculator gnome-characters gnome-logs gnome-system-monitor intel-microcode ubuntu-software ubuntu-restricted-addons libavcodec-extra unrar
### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  i104059015" /target/etc/hosts ; \
# Install proprietary video drivers, if needed.
in-target ubuntu-drivers autoinstall --no-oem ; \
# Requirements for password quality
in-target mkdir -p /etc/security/pwquality.conf.d/ ; \
echo "minlen = 10" > /target/etc/security/pwquality.conf.d/10minlen.conf ; \
echo "ucredit = -1" > /target/etc/security/pwquality.conf.d/10credit.conf ; \
echo "dcredit = -1" >> /target/etc/security/pwquality.conf.d/10credit.conf ; \
echo "ocredit = -1" >> /target/etc/security/pwquality.conf.d/10credit.conf ; \
# Add Yandex Browser repo and install it
in-target add-apt-repository -n 'deb [arch=amd64] http://repo.yandex.ru/yandex-browser/deb beta main' ; \
in-target apt-key adv --fetch-keys http://repo.yandex.ru/yandex-browser/YANDEX-BROWSER-KEY.GPG ; \
# Add salt repo
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/3001 focal main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/latest/SALTSTACK-GPG-KEY.pub  ; \
# Add common.dist.yandex.ru/common (all/amd64)
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/all/' ; \
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/amd64/' ; \
in-target apt-key adv --keyserver keyserver.ubuntu.com --recv-key 7FCD11186050CD1A ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/focal_apt_preferences -O /target/etc/apt/preferences ; \
in-target apt update ; \
in-target apt install -y yandex-browser-beta salt-minion osquery-yandex-fleet-workstations-config yandex-archive-keyring ; \
sed -i "/yandex-browser/d" /target/etc/apt/sources.list ; \
# Add Skotty
in-target apt install -y libccid pcscd ; \
in-target systemctl enable pcscd.socket ; \
in-target systemctl restart pcscd.socket ; \
in-target apt install -y yandex-skotty ; \
#enable osqueryd service
in-target /lib/systemd/systemd-sysv-install enable osqueryd ; \
in-target ln -s /lib/systemd/system/osqueryd.service /etc/systemd/system/multi-user.target.wants/osqueryd.service ; \
# Install and configure pam-cryptsetup(sync user password and luks passphrase)
wget -q https://baldr-msk2.yandex-team.ru/linux/files/packages/libpam-cryptsetup_0.1_amd64.deb https://baldr-msk2.yandex-team.ru/linux/files/packages/pam-cryptsetup-tools_0.1_amd64.deb -P /target/tmp/ ; \
in-target apt install -y /tmp/libpam-cryptsetup_0.1_amd64.deb /tmp/pam-cryptsetup-tools_0.1_amd64.deb ; \
echo "password [default=ignore] pam_cryptsetup.so crypt_name=$(dmsetup ls --target crypt | cut -f1)" >> /target/etc/pam.d/common-password ; \
# Install custom luks unlock password script for inital setup
sed -i '${s/$/,tries=7,keyscript=\/lib\/cryptsetup\/scripts\/unlock/}' /target/etc/crypttab ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/scripts/unlock -O /target/lib/cryptsetup/scripts/unlock ; \
chmod +x /target/lib/cryptsetup/scripts/unlock ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/hdluks/hdluksescrow -O /target/tmp/hdluksescrow ; \
chmod +x /target/tmp/hdluksescrow ; \
/target/tmp/hdluksescrow --id fae5063b-0e30-4c30-92ee-7d2f91a58659 -i "104059015" -S "pf21eqtr" -s https://api.baldr.yandex.net/v1/key_escrow -d /dev/$(dmsetup ls --target crypt | cut -d '_' -f1) -k "fJq6dAtU3maaT-X" ; \
in-target update-initramfs -u -k all ; \
# Fix nvidia s3 issues on xps
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/nvidia-no-msi.conf -O /target/etc/modprobe.d/nvidia-no-msi.conf ; \
# Use intel graphics on Dell XPS
if grep -q 'XPS 15 9570' /sys/devices/virtual/dmi/id/product_name ; then in-target prime-select on-demand ; fi ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion-user -O /target/etc/salt/minion ; \
mkdir -p /target/etc/salt/pki/minion/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt.baldr.yandex.net.master_sign.pub -O /target/etc/salt/pki/minion/master_sign.pub ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# Setup YandexInternalCA for Firefox and YaBro
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/firefox-policies.json -O /target/usr/lib/firefox/distribution/policies.json ; \
in-target mkdir -p --mode 700 /home/knight/.pki/nssdb/ ; \
in-target modutil -dbdir sql:/home/knight/.pki/nssdb/ -add "P11KIT-trust" -libfile /usr/lib/x86_64-linux-gnu/pkcs11/p11-kit-trust.so -force ; \
in-target chown knight:knight -R /home/knight/.pki/ ; \
# Install dconf profiles
wget -q https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/user -O /target/etc/dconf/profile/user ; \
mkdir -p /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/all/. -P /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/users/. -P /target/etc/dconf/db/local.d/ ; \
in-target dconf update ; \
# Disable netplan
rm /target/etc/netplan/01-netcfg.yaml ; \
# Copy password change instructions
mkdir /target/home/knight/Desktop ; \
wget -q -O /target/home/knight/Desktop/Changing\ Password\ Manual.pdf https://s3.mds.yandex.net/helpdesk/New%20user%20manual.pdf ; \
in-target chown knight:knight -R /home/knight/Desktop ; \
# Install firmware update service and scripts(uses lvfs)
wget -q https://baldr-msk2.yandex-team.ru/linux/files/fwupdater/fwupdater-focal.sh -O /target/opt/yandex/fwupdater.sh ; \
chmod +x /target/opt/yandex/fwupdater.sh ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/fwupdater/fwupdater.service -O /target/etc/systemd/system/fwupdater.service ; \
in-target ln -s /etc/systemd/system/fwupdater.service /etc/systemd/system/multi-user.target.wants/fwupdater.service ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
sed -i '/GRUB_CMDLINE_LINUX_DEFAULT/c\GRUB_CMDLINE_LINUX_DEFAULT="quiet splash"' /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeRegularComputer
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionUserName] = "knight"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSVersion] = linuxReleaseFocalVersion
	dep.Options[models.OptionOSReleaseName] = linuxReleaseFocal
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, userPreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedFocalUserPreDeploy(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for focal)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string changeme

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string focal
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select and options
d-i base-installer/kernel/image string linux-generic-hwe-20.04
d-i debian-installer/add-kernel-opts string mem_sleep_default=deep

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string changeme
d-i passwd/username string changeme
d-i passwd/user-password-crypted password $1$fENO7/iN$keuIeFNVdf7qIkzRNOd8/1
d-i passwd/auto-login boolean false
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false

d-i preseed/early_command string \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/packages/partman-crypto_101ubuntu4+luksopts1_amd64.udeb ; \
udpkg -i partman-crypto_101ubuntu4\+luksopts1_amd64.udeb
### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string crypto
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

d-i partman-crypto/passphrase password fJq6dAtU3maaT-X
d-i partman-crypto/passphrase-again password fJq6dAtU3maaT-X
d-i partman-crypto/luksformat_options string --type luks1
### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        user-efi-parts ::                                       \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                4096 4096 4096 ext4                             \
                        $primary{ }                             \
                        $bootable{ }                            \
                        method{ format } format{ }              \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ /boot }                     \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                16384 16384 16384 linux-swap                    \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select user-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, ubuntu-desktop

d-i pkgsel/include string acpi traceroute python3-pip network-manager-openvpn network-manager-openvpn-gnome wget libnss3-tools gnome-calculator gnome-characters gnome-logs gnome-system-monitor intel-microcode ubuntu-software ubuntu-restricted-addons libavcodec-extra unrar
### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  changeme" /target/etc/hosts ; \
# Install proprietary video drivers, if needed.
in-target ubuntu-drivers autoinstall --no-oem ; \
# Requirements for password quality
in-target mkdir -p /etc/security/pwquality.conf.d/ ; \
echo "minlen = 10" > /target/etc/security/pwquality.conf.d/10minlen.conf ; \
echo "ucredit = -1" > /target/etc/security/pwquality.conf.d/10credit.conf ; \
echo "dcredit = -1" >> /target/etc/security/pwquality.conf.d/10credit.conf ; \
echo "ocredit = -1" >> /target/etc/security/pwquality.conf.d/10credit.conf ; \
# Add Yandex Browser repo and install it
in-target add-apt-repository -n 'deb [arch=amd64] http://repo.yandex.ru/yandex-browser/deb beta main' ; \
in-target apt-key adv --fetch-keys http://repo.yandex.ru/yandex-browser/YANDEX-BROWSER-KEY.GPG ; \
# Add salt repo
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/3001 focal main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/latest/SALTSTACK-GPG-KEY.pub  ; \
# Add common.dist.yandex.ru/common (all/amd64)
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/all/' ; \
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/amd64/' ; \
in-target apt-key adv --keyserver keyserver.ubuntu.com --recv-key 7FCD11186050CD1A ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/focal_apt_preferences -O /target/etc/apt/preferences ; \
in-target apt update ; \
in-target apt install -y yandex-browser-beta salt-minion osquery-yandex-fleet-workstations-config yandex-archive-keyring ; \
sed -i "/yandex-browser/d" /target/etc/apt/sources.list ; \
# Add Skotty
in-target apt install -y libccid pcscd ; \
in-target systemctl enable pcscd.socket ; \
in-target systemctl restart pcscd.socket ; \
in-target apt install -y yandex-skotty ; \
#enable osqueryd service
in-target /lib/systemd/systemd-sysv-install enable osqueryd ; \
in-target ln -s /lib/systemd/system/osqueryd.service /etc/systemd/system/multi-user.target.wants/osqueryd.service ; \
# Install and configure pam-cryptsetup(sync user password and luks passphrase)
wget -q https://baldr-msk2.yandex-team.ru/linux/files/packages/libpam-cryptsetup_0.1_amd64.deb https://baldr-msk2.yandex-team.ru/linux/files/packages/pam-cryptsetup-tools_0.1_amd64.deb -P /target/tmp/ ; \
in-target apt install -y /tmp/libpam-cryptsetup_0.1_amd64.deb /tmp/pam-cryptsetup-tools_0.1_amd64.deb ; \
echo "password [default=ignore] pam_cryptsetup.so crypt_name=$(dmsetup ls --target crypt | cut -f1)" >> /target/etc/pam.d/common-password ; \
# Install custom luks unlock password script for inital setup
sed -i '${s/$/,tries=7,keyscript=\/lib\/cryptsetup\/scripts\/unlock/}' /target/etc/crypttab ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/scripts/unlock -O /target/lib/cryptsetup/scripts/unlock ; \
chmod +x /target/lib/cryptsetup/scripts/unlock ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/hdluks/hdluksescrow -O /target/tmp/hdluksescrow ; \
chmod +x /target/tmp/hdluksescrow ; \
/target/tmp/hdluksescrow --id fae5063b-0e30-4c30-92ee-7d2f91a58659 -i "104059015" -S "pf21eqtr" -s https://api.baldr.yandex.net/v1/key_escrow -d /dev/$(dmsetup ls --target crypt | cut -d '_' -f1) -k "fJq6dAtU3maaT-X" ; \
in-target update-initramfs -u -k all ; \
# Fix nvidia s3 issues on xps
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/nvidia-no-msi.conf -O /target/etc/modprobe.d/nvidia-no-msi.conf ; \
# Use intel graphics on Dell XPS
if grep -q 'XPS 15 9570' /sys/devices/virtual/dmi/id/product_name ; then in-target prime-select on-demand ; fi ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion-user -O /target/etc/salt/minion ; \
mkdir -p /target/etc/salt/pki/minion/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt.baldr.yandex.net.master_sign.pub -O /target/etc/salt/pki/minion/master_sign.pub ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# Setup YandexInternalCA for Firefox and YaBro
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/firefox-policies.json -O /target/usr/lib/firefox/distribution/policies.json ; \
in-target mkdir -p --mode 700 /etc/skel/.pki/nssdb/ ; \
in-target modutil -dbdir sql:/etc/skel/.pki/nssdb/ -add "P11KIT-trust" -libfile /usr/lib/x86_64-linux-gnu/pkcs11/p11-kit-trust.so -force ; \
# Install dconf profiles
wget -q https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/user -O /target/etc/dconf/profile/user ; \
mkdir -p /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/all/. -P /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/users/. -P /target/etc/dconf/db/local.d/ ; \
in-target dconf update ; \
# Disable netplan
rm /target/etc/netplan/01-netcfg.yaml ; \
# Copy password change instructions
mkdir /target/etc/skel/Desktop ; \
wget -q -O /target/etc/skel/Desktop/Changing\ Password\ Manual.pdf https://s3.mds.yandex.net/helpdesk/New%20user%20manual.pdf ; \
# Install firmware update service and scripts(uses lvfs)
wget -q https://baldr-msk2.yandex-team.ru/linux/files/fwupdater/fwupdater-focal.sh -O /target/opt/yandex/fwupdater.sh ; \
chmod +x /target/opt/yandex/fwupdater.sh ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/fwupdater/fwupdater.service -O /target/etc/systemd/system/fwupdater.service ; \
in-target ln -s /etc/systemd/system/fwupdater.service /etc/systemd/system/multi-user.target.wants/fwupdater.service ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# install post-install script
wget -q https://baldr-msk2.yandex-team.ru/linux/files/ya-postinstall-user -O /target/usr/local/bin/ya-postinstall ; \
chmod +x /target/usr/local/bin/ya-postinstall ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
sed -i '/GRUB_CMDLINE_LINUX_DEFAULT/c\GRUB_CMDLINE_LINUX_DEFAULT="quiet splash"' /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeRegularComputer
	dep.Options[models.OptionWithDomain] = ""
	dep.Options[models.OptionWithOffice] = ""
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = ""
	dep.Options[models.OptionLanguageProfile] = ""
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "WithoutOFR"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "changeme"
	dep.Options[models.OptionUserName] = "changeme"
	dep.Options[models.OptionOS] = "Linux"
	dep.Options[models.OptionOSBuild] = ""
	dep.Options[models.OptionOSEdition] = ""
	dep.Options[models.OptionOSUpdated] = ""
	dep.Options[models.OptionOSVersion] = linuxReleaseFocalVersion
	dep.Options[models.OptionOSReleaseName] = linuxReleaseFocal
	dep.Options[models.OptionOSProductKey] = ""
	dep.Options[models.OptionDeploymentTemplate] = "linux_pre_deploy"
	dep.Options[models.OptionADOU] = ""
	dep.Options[models.OptionDriversProfile] = ""

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionDeploymentTaskIsOpen] = "yes"
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, userPreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedBionicUser(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for bionic)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string i104059015

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string bionic
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select and options
d-i base-installer/kernel/image string linux-generic-hwe-18.04
d-i debian-installer/add-kernel-opts string mem_sleep_default=deep

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string knight
d-i passwd/username string knight
d-i passwd/user-password-crypted password $1$fENO7/iN$keuIeFNVdf7qIkzRNOd8/1
d-i passwd/auto-login boolean false
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false

### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string lvm
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        user-efi-parts ::                                       \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                16384 16384 16384 linux-swap                    \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select user-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, ubuntu-desktop

d-i pkgsel/include string acpi traceroute python-pip network-manager-openvpn network-manager-openvpn-gnome wget libnss3-tools gnome-calculator gnome-characters gnome-logs gnome-system-monitor intel-microcode ubuntu-restricted-addons libavcodec-extra unrar
### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  i104059015" /target/etc/hosts ; \
# Install HWE graphical stack
in-target apt-get install --install-recommends -y xserver-xorg-hwe-18.04 ; \
# Install proprietary video drivers, if needed.
in-target ubuntu-drivers autoinstall ; \
# Requirements for password quality
in-target mkdir -p /etc/security/pwquality.conf.d/ ; \
echo "minlen = 10" > /target/etc/security/pwquality.conf.d/10minlen.conf ; \
echo "ucredit = -1" > /target/etc/security/pwquality.conf.d/10credit.conf ; \
echo "dcredit = -1" >> /target/etc/security/pwquality.conf.d/10credit.conf ; \
echo "ocredit = -1" >> /target/etc/security/pwquality.conf.d/10credit.conf ; \
# Add Yandex Browser repo and install it
in-target add-apt-repository -n 'deb [arch=amd64] http://repo.yandex.ru/yandex-browser/deb beta main' ; \
in-target apt-key adv --fetch-keys http://repo.yandex.ru/yandex-browser/YANDEX-BROWSER-KEY.GPG ; \
# Add salt repo
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2 bionic main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2/SALTSTACK-GPG-KEY.pub  ; \
# Add common.dist.yandex.ru/common (all/amd64)
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/all/' ; \
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/amd64/' ; \
in-target apt-key adv --keyserver keyserver.ubuntu.com --recv-key 7FCD11186050CD1A ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/bionic_apt_preferences -O /target/etc/apt/preferences ; \
in-target apt update ; \
in-target apt install -y yandex-browser-beta salt-minion osquery-yandex-fleet-workstations-config yandex-archive-keyring ; \
sed -i "/yandex-browser/d" /target/etc/apt/sources.list ; \
# Add Skotty
in-target apt install -y libccid pcscd ; \
in-target systemctl enable pcscd.socket ; \
in-target systemctl restart pcscd.socket ; \
in-target apt install -y yandex-skotty ; \
#enable osqueryd service
in-target /lib/systemd/systemd-sysv-install enable osqueryd ; \
in-target ln -s /lib/systemd/system/osqueryd.service /etc/systemd/system/multi-user.target.wants/osqueryd.service ; \
# Fix nvidia s3 issues on xps
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/nvidia-no-msi.conf -O /target/etc/modprobe.d/nvidia-no-msi.conf ; \
# Use intel graphics on Dell XPS
if grep -q 'XPS 15 9570' /sys/devices/virtual/dmi/id/product_name ; then in-target prime-select intel ; fi ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion-user -O /target/etc/salt/minion ; \
mkdir -p /target/etc/salt/pki/minion/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt.baldr.yandex.net.master_sign.pub -O /target/etc/salt/pki/minion/master_sign.pub ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# Setup YandexInternalCA for Firefox and YaBro
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/firefox-policies.json -O /target/usr/lib/firefox/distribution/policies.json ; \
in-target mkdir -p --mode 700 /home/knight/.pki/nssdb/ ; \
in-target modutil -dbdir sql:/home/knight/.pki/nssdb/ -add "P11KIT-trust" -libfile /usr/lib/x86_64-linux-gnu/pkcs11/p11-kit-trust.so -force ; \
in-target chown knight:knight -R /home/knight/.pki/ ; \
# Install dconf profiles
wget -q https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/user -O /target/etc/dconf/profile/user ; \
mkdir -p /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/all/. -P /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/users/. -P /target/etc/dconf/db/local.d/ ; \
in-target dconf update ; \
# Disable netplan
rm /target/etc/netplan/01-netcfg.yaml ; \
# Copy password change instructions
mkdir /target/home/knight/Desktop ; \
wget -q -O /target/home/knight/Desktop/Changing\ Password\ Manual.pdf https://s3.mds.yandex.net/helpdesk/New%20user%20manual.pdf ; \
in-target chown knight:knight -R /home/knight/Desktop ; \
# Install firmware update service and scripts(uses lvfs)
wget -q https://baldr-msk2.yandex-team.ru/linux/files/fwupdater/fwupdater-bionic.sh -O /target/opt/yandex/fwupdater.sh ; \
chmod +x /target/opt/yandex/fwupdater.sh ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/fwupdater/fwupdater.service -O /target/etc/systemd/system/fwupdater.service ; \
in-target ln -s /etc/systemd/system/fwupdater.service /etc/systemd/system/multi-user.target.wants/fwupdater.service ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeRegularComputer
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionUserName] = "knight"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSReleaseName] = linuxReleaseBionic
	dep.Options[models.OptionOSVersion] = linuxReleaseBionicVersion
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, userPreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedFocalZombie(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for focal)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string zomb-prj-100.zombie.yandex-team.ru

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string focal
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select
d-i base-installer/kernel/image string linux-generic-hwe-20.04

#enable unattended-upgrades
d-i pkgsel/update-policy select unattended-upgrades

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string zomb-prj-100
d-i passwd/username string zomb-prj-100
d-i passwd/user-password-crypted password $1$1cjJured$Kgj03GwBZIyqDUfvvgL0Q.
d-i passwd/auto-login boolean true
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false

### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string lvm
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        zombie-efi-parts ::                                     \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                4096 4096 4096 linux-swap                       \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select zombie-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, ubuntu-desktop-minimal

d-i pkgsel/include string acpi traceroute network-manager-openvpn network-manager-openvpn-gnome wget openssh-server libnss3-tools intel-microcode ubuntu-software vino libavcodec-extra ubuntu-restricted-extras chromium-codecs-ffmpeg-extra

### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  zomb-prj-100.zombie.yandex-team.ru" /target/etc/hosts ; \
# Add 3-rd party repos
in-target add-apt-repository -n 'deb [arch=amd64] http://repo.yandex.ru/yandex-browser/deb beta main' ; \
in-target apt-key adv --fetch-keys http://repo.yandex.ru/yandex-browser/YANDEX-BROWSER-KEY.GPG ; \
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/3001 focal main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/latest/SALTSTACK-GPG-KEY.pub  ; \
in-target apt update ; \
# Install additional software(YaBro and salt-minion)
in-target apt install -y yandex-browser-beta salt-minion python3-m2crypto python3-croniter ; \
sed -i "/yandex-browser/d" /target/etc/apt/sources.list ; \
# Remove ubuntu welcome screen
in-target apt remove -y gnome-initial-setup ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Enable unattended-upgrades
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/60unattended-upgrades -O /target/etc/apt/apt.conf.d/60unattended-upgrades ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion -O /target/etc/salt/minion ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# eGalax touch fix
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/91-libinput-egalax.rules -O /target/etc/udev/rules.d/91-libinput-egalax.rules ; \
# Setup YandexInternalCA for Firefox and YaBro
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/firefox-policies-zmb.json -O /target/usr/lib/firefox/distribution/policies.json ; \
in-target mkdir -p --mode 700 /home/zomb-prj-100/.pki/nssdb/ ; \
in-target modutil -dbdir sql:/home/zomb-prj-100/.pki/nssdb/ -add "P11KIT-trust" -libfile /usr/lib/x86_64-linux-gnu/pkcs11/p11-kit-trust.so -force ; \
in-target chown zomb-prj-100:zomb-prj-100 -R /home/zomb-prj-100/.pki/ ; \
# Install tab rotate addons into Firefox and YaBro
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/tabrotator.xpi -O /target/usr/lib/firefox/distribution/extensions/tabrotator@davidfichtmueller.de.xpi ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/extension_0_4_1_0.crx -O /target//opt/yandex/browser-beta/Extensions/extension_0_4_1_0.crx ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json -O /target/opt/yandex/browser-beta/Extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json ; \
# Install dconf profiles
wget -q https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/user -O /target/etc/dconf/profile/user ; \
mkdir -p /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/all/. -P /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/zombie/. -P /target/etc/dconf/db/local.d/ ; \
in-target dconf update ; \
# Disable network connectivity checks and netplan
touch /target/etc/NetworkManager/conf.d/20-connectivity-ubuntu.conf ; \
rm /target/etc/netplan/01-netcfg.yaml ; \
# Disable error reporing
echo 'enabled=0' > /target/etc/default/apport ; \
rm /target/etc/systemd/system/multi-user.target.wants/whoopsie.service ; \
# Fix autologin issue
printf "[daemon]\nTimedLoginEnable = true\nTimedLoginDelay = 0" >> /target/etc/gdm3/custom.conf ; \
# Enable VNC server for all conntions
in-target ln -s /usr/share/applications/vino-server.desktop /etc/xdg/autostart/vino-server.desktop ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Disable new release checks
sed -i '/Prompt/c\Prompt=never' /target/etc/update-manager/release-upgrades ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
sed -i '/GRUB_CMDLINE_LINUX_DEFAULT/c\GRUB_CMDLINE_LINUX_DEFAULT="quiet splash"' /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionFQDN] = "zomb-prj-100.zombie.yandex-team.ru"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionUserName] = "zomb-prj-100"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSReleaseName] = linuxReleaseFocal
	dep.Options[models.OptionOSVersion] = linuxReleaseFocalVersion
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, zombiePreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedBionicZombie(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for bionic)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string zomb-prj-100.zombie.yandex-team.ru

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string bionic
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select
d-i base-installer/kernel/image string linux-generic-hwe-18.04

#enable unattended-upgrades
d-i pkgsel/update-policy select unattended-upgrades

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string zomb-prj-100
d-i passwd/username string zomb-prj-100
d-i passwd/user-password-crypted password $1$1cjJured$Kgj03GwBZIyqDUfvvgL0Q.
d-i passwd/auto-login boolean true
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false

### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string lvm
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        zombie-efi-parts ::                                     \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                4096 4096 4096 linux-swap                       \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select zombie-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, ubuntu-desktop

d-i pkgsel/include string chromium-browser acpi traceroute python-pip network-manager-openvpn network-manager-openvpn-gnome wget openssh-server libnss3-tools intel-microcode libavcodec-extra ubuntu-restricted-extras chromium-codecs-ffmpeg-extra

### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  zomb-prj-100.zombie.yandex-team.ru" /target/etc/hosts ; \
# Install HWE graphical stack
in-target apt-get install --install-recommends -y xserver-xorg-hwe-18.04 ; \
# Add 3-rd party repos
in-target add-apt-repository -n 'deb [arch=amd64] http://repo.yandex.ru/yandex-browser/deb beta main' ; \
in-target apt-key adv --fetch-keys http://repo.yandex.ru/yandex-browser/YANDEX-BROWSER-KEY.GPG ; \
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2 bionic main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2/SALTSTACK-GPG-KEY.pub  ; \
in-target apt update ; \
# Install additional software(YaBro and salt-minion)
in-target apt install -y yandex-browser-beta salt-minion python-m2crypto python-croniter ; \
sed -i "/yandex-browser/d" /target/etc/apt/sources.list ; \
# Remove ubuntu welcome screen
in-target apt remove -y gnome-initial-setup ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Enable unattended-upgrades
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/60unattended-upgrades -O /target/etc/apt/apt.conf.d/60unattended-upgrades ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion -O /target/etc/salt/minion ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# eGalax touch fix
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/91-libinput-egalax.rules -O /target/etc/udev/rules.d/91-libinput-egalax.rules ; \
# Setup YandexInternalCA for Firefox and YaBro
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/firefox-policies-zmb.json -O /target/usr/lib/firefox/distribution/policies.json ; \
in-target mkdir -p --mode 700 /home/zomb-prj-100/.pki/nssdb/ ; \
in-target modutil -dbdir sql:/home/zomb-prj-100/.pki/nssdb/ -add "P11KIT-trust" -libfile /usr/lib/x86_64-linux-gnu/pkcs11/p11-kit-trust.so -force ; \
in-target chown zomb-prj-100:zomb-prj-100 -R /home/zomb-prj-100/.pki/ ; \
# Install tab rotate addons into Firefox and YaBro
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/tabrotator.xpi -O /target/usr/lib/firefox/distribution/extensions/tabrotator@davidfichtmueller.de.xpi ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/extension_0_4_1_0.crx -O /target//opt/yandex/browser-beta/Extensions/extension_0_4_1_0.crx ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json -O /target/opt/yandex/browser-beta/Extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json ; \
# Install dconf profiles
wget -q https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/user -O /target/etc/dconf/profile/user ; \
mkdir -p /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/all/. -P /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/zombie/. -P /target/etc/dconf/db/local.d/ ; \
in-target dconf update ; \
# Disable network connectivity checks and netplan
touch /target/etc/NetworkManager/conf.d/20-connectivity-ubuntu.conf ; \
rm /target/etc/netplan/01-netcfg.yaml ; \
# Disable error reporing
echo 'enabled=0' > /target/etc/default/apport ; \
rm /target/etc/systemd/system/multi-user.target.wants/whoopsie.service ; \
# Fix autologin issue
printf "[daemon]\nTimedLoginEnable = true\nTimedLoginDelay = 0" >> /target/etc/gdm3/custom.conf ; \
# Enable VNC server for all conntions
in-target ln -s /usr/share/applications/vino-server.desktop /etc/xdg/autostart/vino-server.desktop ; \
# intel flicker fix
mkdir -p /target/etc/X11/xorg.conf.d ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/20-intel_flicker_fix.conf -O /target/etc/X11/xorg.conf.d/20-intel_flicker_fix.conf ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionFQDN] = "zomb-prj-100.zombie.yandex-team.ru"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionUserName] = "zomb-prj-100"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSReleaseName] = linuxReleaseBionic
	dep.Options[models.OptionOSVersion] = linuxReleaseBionicVersion
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, zombiePreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedFocalZombieWithoutFQDN(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for focal)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string zomb-100

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string focal
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select
d-i base-installer/kernel/image string linux-generic-hwe-20.04

#enable unattended-upgrades
d-i pkgsel/update-policy select unattended-upgrades

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string zomb-prj-100
d-i passwd/username string zomb-prj-100
d-i passwd/user-password-crypted password $1$1cjJured$Kgj03GwBZIyqDUfvvgL0Q.
d-i passwd/auto-login boolean true
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false

### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string lvm
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        zombie-efi-parts ::                                     \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                4096 4096 4096 linux-swap                       \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select zombie-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, ubuntu-desktop-minimal

d-i pkgsel/include string acpi traceroute network-manager-openvpn network-manager-openvpn-gnome wget openssh-server libnss3-tools intel-microcode ubuntu-software vino libavcodec-extra ubuntu-restricted-extras chromium-codecs-ffmpeg-extra

### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  zomb-100" /target/etc/hosts ; \
# Add 3-rd party repos
in-target add-apt-repository -n 'deb [arch=amd64] http://repo.yandex.ru/yandex-browser/deb beta main' ; \
in-target apt-key adv --fetch-keys http://repo.yandex.ru/yandex-browser/YANDEX-BROWSER-KEY.GPG ; \
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/3001 focal main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/latest/SALTSTACK-GPG-KEY.pub  ; \
in-target apt update ; \
# Install additional software(YaBro and salt-minion)
in-target apt install -y yandex-browser-beta salt-minion python3-m2crypto python3-croniter ; \
sed -i "/yandex-browser/d" /target/etc/apt/sources.list ; \
# Remove ubuntu welcome screen
in-target apt remove -y gnome-initial-setup ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Enable unattended-upgrades
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/60unattended-upgrades -O /target/etc/apt/apt.conf.d/60unattended-upgrades ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion -O /target/etc/salt/minion ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# eGalax touch fix
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/91-libinput-egalax.rules -O /target/etc/udev/rules.d/91-libinput-egalax.rules ; \
# Setup YandexInternalCA for Firefox and YaBro
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/firefox-policies-zmb.json -O /target/usr/lib/firefox/distribution/policies.json ; \
in-target mkdir -p --mode 700 /home/zomb-prj-100/.pki/nssdb/ ; \
in-target modutil -dbdir sql:/home/zomb-prj-100/.pki/nssdb/ -add "P11KIT-trust" -libfile /usr/lib/x86_64-linux-gnu/pkcs11/p11-kit-trust.so -force ; \
in-target chown zomb-prj-100:zomb-prj-100 -R /home/zomb-prj-100/.pki/ ; \
# Install tab rotate addons into Firefox and YaBro
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/tabrotator.xpi -O /target/usr/lib/firefox/distribution/extensions/tabrotator@davidfichtmueller.de.xpi ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/extension_0_4_1_0.crx -O /target//opt/yandex/browser-beta/Extensions/extension_0_4_1_0.crx ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json -O /target/opt/yandex/browser-beta/Extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json ; \
# Install dconf profiles
wget -q https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/user -O /target/etc/dconf/profile/user ; \
mkdir -p /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/all/. -P /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-focal/zombie/. -P /target/etc/dconf/db/local.d/ ; \
in-target dconf update ; \
# Disable network connectivity checks and netplan
touch /target/etc/NetworkManager/conf.d/20-connectivity-ubuntu.conf ; \
rm /target/etc/netplan/01-netcfg.yaml ; \
# Disable error reporing
echo 'enabled=0' > /target/etc/default/apport ; \
rm /target/etc/systemd/system/multi-user.target.wants/whoopsie.service ; \
# Fix autologin issue
printf "[daemon]\nTimedLoginEnable = true\nTimedLoginDelay = 0" >> /target/etc/gdm3/custom.conf ; \
# Enable VNC server for all conntions
in-target ln -s /usr/share/applications/vino-server.desktop /etc/xdg/autostart/vino-server.desktop ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Disable new release checks
sed -i '/Prompt/c\Prompt=never' /target/etc/update-manager/release-upgrades ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
sed -i '/GRUB_CMDLINE_LINUX_DEFAULT/c\GRUB_CMDLINE_LINUX_DEFAULT="quiet splash"' /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "zomb-100"
	dep.Options[models.OptionUserName] = "zomb-prj-100"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSReleaseName] = linuxReleaseFocal
	dep.Options[models.OptionOSVersion] = linuxReleaseFocalVersion
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, zombiePreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedBionicZombieWithoutFQDN(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for bionic)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string zomb-100

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string bionic
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select
d-i base-installer/kernel/image string linux-generic-hwe-18.04

#enable unattended-upgrades
d-i pkgsel/update-policy select unattended-upgrades

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string zomb-prj-100
d-i passwd/username string zomb-prj-100
d-i passwd/user-password-crypted password $1$1cjJured$Kgj03GwBZIyqDUfvvgL0Q.
d-i passwd/auto-login boolean true
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false

### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string lvm
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        zombie-efi-parts ::                                     \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                4096 4096 4096 linux-swap                       \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select zombie-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, ubuntu-desktop

d-i pkgsel/include string chromium-browser acpi traceroute python-pip network-manager-openvpn network-manager-openvpn-gnome wget openssh-server libnss3-tools intel-microcode libavcodec-extra ubuntu-restricted-extras chromium-codecs-ffmpeg-extra

### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  zomb-100" /target/etc/hosts ; \
# Install HWE graphical stack
in-target apt-get install --install-recommends -y xserver-xorg-hwe-18.04 ; \
# Add 3-rd party repos
in-target add-apt-repository -n 'deb [arch=amd64] http://repo.yandex.ru/yandex-browser/deb beta main' ; \
in-target apt-key adv --fetch-keys http://repo.yandex.ru/yandex-browser/YANDEX-BROWSER-KEY.GPG ; \
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2 bionic main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2/SALTSTACK-GPG-KEY.pub  ; \
in-target apt update ; \
# Install additional software(YaBro and salt-minion)
in-target apt install -y yandex-browser-beta salt-minion python-m2crypto python-croniter ; \
sed -i "/yandex-browser/d" /target/etc/apt/sources.list ; \
# Remove ubuntu welcome screen
in-target apt remove -y gnome-initial-setup ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Enable unattended-upgrades
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/60unattended-upgrades -O /target/etc/apt/apt.conf.d/60unattended-upgrades ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion -O /target/etc/salt/minion ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# eGalax touch fix
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/91-libinput-egalax.rules -O /target/etc/udev/rules.d/91-libinput-egalax.rules ; \
# Setup YandexInternalCA for Firefox and YaBro
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/firefox-policies-zmb.json -O /target/usr/lib/firefox/distribution/policies.json ; \
in-target mkdir -p --mode 700 /home/zomb-prj-100/.pki/nssdb/ ; \
in-target modutil -dbdir sql:/home/zomb-prj-100/.pki/nssdb/ -add "P11KIT-trust" -libfile /usr/lib/x86_64-linux-gnu/pkcs11/p11-kit-trust.so -force ; \
in-target chown zomb-prj-100:zomb-prj-100 -R /home/zomb-prj-100/.pki/ ; \
# Install tab rotate addons into Firefox and YaBro
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/tabrotator.xpi -O /target/usr/lib/firefox/distribution/extensions/tabrotator@davidfichtmueller.de.xpi ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/extension_0_4_1_0.crx -O /target//opt/yandex/browser-beta/Extensions/extension_0_4_1_0.crx ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/browser-extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json -O /target/opt/yandex/browser-beta/Extensions/pjgjpabbgnnoohijnillgbckikfkbjed.json ; \
# Install dconf profiles
wget -q https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/user -O /target/etc/dconf/profile/user ; \
mkdir -p /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/all/. -P /target/etc/dconf/db/local.d/ ; \
wget -q -r -np -nH --cut-dirs=4 -R "index.html*" https://baldr-msk2.yandex-team.ru/linux/files/dconf-bionic/zombie/. -P /target/etc/dconf/db/local.d/ ; \
in-target dconf update ; \
# Disable network connectivity checks and netplan
touch /target/etc/NetworkManager/conf.d/20-connectivity-ubuntu.conf ; \
rm /target/etc/netplan/01-netcfg.yaml ; \
# Disable error reporing
echo 'enabled=0' > /target/etc/default/apport ; \
rm /target/etc/systemd/system/multi-user.target.wants/whoopsie.service ; \
# Fix autologin issue
printf "[daemon]\nTimedLoginEnable = true\nTimedLoginDelay = 0" >> /target/etc/gdm3/custom.conf ; \
# Enable VNC server for all conntions
in-target ln -s /usr/share/applications/vino-server.desktop /etc/xdg/autostart/vino-server.desktop ; \
# intel flicker fix
mkdir -p /target/etc/X11/xorg.conf.d ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/20-intel_flicker_fix.conf -O /target/etc/X11/xorg.conf.d/20-intel_flicker_fix.conf ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "zomb-100"
	dep.Options[models.OptionUserName] = "zomb-prj-100"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSReleaseName] = linuxReleaseBionic
	dep.Options[models.OptionOSVersion] = linuxReleaseBionicVersion
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, zombiePreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedGPUZombieFocal(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for focal)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string zomb-prj-100.zombie.yandex-team.ru

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string focal
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select
d-i base-installer/kernel/image string linux-generic-hwe-20.04

#enable unattended-upgrades
d-i pkgsel/update-policy select unattended-upgrades

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string zomb-prj-100
d-i passwd/username string zomb-prj-100
d-i passwd/user-password-crypted password $1$1cjJured$Kgj03GwBZIyqDUfvvgL0Q.
d-i passwd/auto-login boolean true
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false


### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string lvm
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        gpu-efi-parts ::                                        \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                16384 16384 16384 linux-swap                    \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select gpu-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, Basic Ubuntu server

d-i pkgsel/include string acpi traceroute wget openssh-server network-manager ubuntu-drivers-common software-properties-common intel-microcode

### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  zomb-prj-100.zombie.yandex-team.ru" /target/etc/hosts ; \
# Add 3-rd party repos
in-target add-apt-repository -n 'deb [arch=amd64] http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/3001 focal main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/py3/ubuntu/20.04/amd64/latest/SALTSTACK-GPG-KEY.pub ; \
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/all/' ; \
in-target apt-key adv --keyserver keyserver.ubuntu.com --recv-key 7FCD11186050CD1A ; \
in-target add-apt-repository -n 'deb http://developer.download.nvidia.com/compute/cuda/repos/ubuntu2004/x86_64/ /' ; \
in-target apt-key adv --fetch-keys http://developer.download.nvidia.com/compute/cuda/repos/ubuntu2004/x86_64/7fa2af80.pub ; \
in-target apt update ; \
# Install additional software(cauth and salt-minion)
in-target apt install -y salt-minion yandex-cauth python3-m2crypto python3-croniter ; \
in-target add-apt-repository -r 'deb http://common.dist.yandex.ru/common stable/all/' ; \
# Install proprietary video drivers, if needed.
in-target ubuntu-drivers install --gpgpu ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Enable unattended-upgrades
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/60unattended-upgrades -O /target/etc/apt/apt.conf.d/60unattended-upgrades ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion -O /target/etc/salt/minion ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# Disable netplan and use NetworkManager
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/01-netcfg.yaml -O /target/etc/netplan/01-netcfg.yaml; \
# Install YandexInternalCA
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionFQDN] = "zomb-prj-100.zombie.yandex-team.ru"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionUserName] = "zomb-prj-100"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSReleaseName] = linuxReleaseFocal
	dep.Options[models.OptionOSVersion] = linuxReleaseFocalVersion
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, zombieGPUPreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}

func TestPreseedGPUZombieBionic(t *testing.T) {
	var err error
	preseed := `#### Contents of the preconfiguration file (for bionic)

# Locales
d-i debian-installer/language string en
d-i debian-installer/country string RU
d-i debian-installer/locale string en_US
d-i debian-installer/locale select en_US.UTF-8
d-i debian-installer/keymap string us
d-i debconf/language string en
d-i localechooser/languagelist select en

# Keyboard
d-i localechooser/shortlist/en select
d-i localechooser/shortlist select EN
d-i console-setup/ask_detect boolean false
d-i console-setup/layoutcode string en
d-i console-setup/variant select USA
d-i console-setup/toggle select Alt+Shift
d-i localechooser/preferred-locale select en_US.UTF-8
d-i console-keymaps-at/keymap select en
d-i keyboard-configuration/xkb-keymap select en

# Network
d-i netcfg/choose_interface select auto
d-i netcfg/get_hostname string unassigned-hostname
d-i netcfg/hostname string zomb-prj-100.zombie.yandex-team.ru

### Mirror and apt settings
d-i mirror/country string manual
d-i mirror/http/hostname string mirror.yandex.ru
d-i mirror/http/directory string /ubuntu
d-i mirror/suite string bionic
d-i apt-setup/restricted boolean true
d-i apt-setup/universe boolean true
d-i apt-setup/multiverse boolean true
d-i apt-setup/backports boolean true
d-i apt-setup/services-select multiselect security
d-i apt-setup/security_host string mirror.yandex.ru
d-i apt-setup/security_path string /ubuntu

#kernel select
d-i base-installer/kernel/image string linux-generic-hwe-18.04

#enable unattended-upgrades
d-i pkgsel/update-policy select unattended-upgrades

### Clock and time zone setup
d-i clock-setup/utc boolean true
d-i time/zone string Europe/Moscow
d-i clock-setup/ntp-server string ntp.yandex.net

### Account setup
d-i passwd/make-user boolean true
d-i passwd/user-fullname string zomb-prj-100
d-i passwd/username string zomb-prj-100
d-i passwd/user-password-crypted password $1$1cjJured$Kgj03GwBZIyqDUfvvgL0Q.
d-i passwd/auto-login boolean true
d-i user-setup/allow-password-weak boolean true
d-i user-setup/encrypt-home boolean false


### Partitioning
d-i partman/choose_partition select finish
d-i partman/confirm boolean true
d-i partman/confirm_nooverwrite boolean true
d-i partman-auto/method string lvm
d-i partman-lvm/device_remove_lvm boolean true
d-i partman-lvm/confirm boolean true
d-i partman-lvm/confirm_nooverwrite boolean true
d-i partman-auto-lvm/no_boot boolean true
d-i partman-md/device_remove_md boolean true
d-i partman-partitioning/confirm_write_new_label boolean true

### Force uefi installation(sic!)
d-i partman-efi/non_efi_system boolean true

### enforce usage of GPT
d-i partman-basicfilesystems/choose_label string gpt
d-i partman-basicfilesystems/default_label string gpt
d-i partman-partitioning/choose_label string gpt
d-i partman-partitioning/default_label string gpt
d-i partman/choose_label string gpt
d-i partman/default_label string gpt

d-i partman-auto/expert_recipe string                           \
        gpu-efi-parts ::                                        \
                550 550 550 free                                \
                        $iflabel{ gpt }                         \
                        $reusemethod{ }                         \
                        method{ efi }                           \
                        format{ }                               \
                        options/discard{ discard }              \
                .                                               \
                51200 51300 -1 ext4                             \
                        lv_name{ root }                         \
                        method{ lvm } format{ }                 \
                        use_filesystem{ } filesystem{ ext4 }    \
                        mountpoint{ / }                         \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .                                               \
                16384 16384 16384 linux-swap                    \
                        lv_name{ swap }                         \
                        method{ swap } format{ }                \
                        $lvmok{ }                               \
                        options/discard{ discard }              \
                .
d-i     partman-auto/choose_recipe select gpu-efi-parts
d-i     partman-partitioning/confirm_write_new_label boolean true
d-i     partman/choose_partition select finish
d-i     partman/confirm boolean true
d-i     partman/confirm_nooverwrite boolean true

### Base system installation
tasksel tasksel/first multiselect standard system utilities, Basic Ubuntu server

d-i pkgsel/include string python-m2crypto acpi traceroute wget openssh-server network-manager ubuntu-drivers-common software-properties-common intel-microcode

### Boot loader installation
d-i grub-installer/only_debian boolean true
d-i grub-installer/with_other_os boolean true

## Post install customization
d-i preseed/late_command string \
# fix hostname(this fixes dns issues)
sed -i "/127.0.1.1/c\127.0.1.1  zomb-prj-100.zombie.yandex-team.ru" /target/etc/hosts ; \
# Add 3-rd party repos
in-target add-apt-repository -n 'deb [ arch=amd64 ] http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2 bionic main' ; \
in-target apt-key adv --fetch-keys http://mirror.yandex.ru/mirrors/repo.saltstack.com/apt/ubuntu/18.04/amd64/2019.2/SALTSTACK-GPG-KEY.pub ; \
in-target add-apt-repository -n 'deb http://common.dist.yandex.ru/common stable/all/' ; \
in-target apt-key adv --keyserver keyserver.ubuntu.com --recv-key 7FCD11186050CD1A ; \
in-target add-apt-repository -n 'deb http://developer.download.nvidia.com/compute/cuda/repos/ubuntu1804/x86_64 /' ; \
in-target apt-key adv --fetch-keys https://developer.download.nvidia.com/compute/cuda/repos/ubuntu1804/x86_64/7fa2af80.pub ; \
in-target apt update ; \
# Install additional software(cauth and salt-minion)
in-target apt install -y salt-minion yandex-cauth python-m2crypto python-croniter ; \
in-target add-apt-repository -r 'deb http://common.dist.yandex.ru/common stable/all/' ; \
# Install proprietary video drivers, if needed.
in-target ubuntu-drivers autoinstall ; \
# Setup NTP
mkdir -p /target/etc/systemd/timesyncd.conf.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/ntp.conf -O /target/etc/systemd/timesyncd.conf.d/local.conf ; \
# Enable unattended-upgrades
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/60unattended-upgrades -O /target/etc/apt/apt.conf.d/60unattended-upgrades ; \
# Configure salt-minion
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/minion -O /target/etc/salt/minion ; \
# Autorestart salt-minion
mkdir -p /target/etc/systemd/system/salt-minion.service.d/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/salt-override.conf -O /target/etc/systemd/system/salt-minion.service.d/override.conf ; \
# Disable netplan and use NetworkManager
wget -q https://baldr-msk2.yandex-team.ru/linux/files/etc/01-netcfg.yaml -O /target/etc/netplan/01-netcfg.yaml; \
# Install YandexInternalCA
mkdir -p /target/usr/local/share/ca-certificates/yandex/ ; \
wget -q https://baldr-msk2.yandex-team.ru/linux/files/YandexInternalCA.crt -O /target/usr/local/share/ca-certificates/yandex/YandexInternalCA.crt ; \
in-target update-ca-certificates ; \
# Install get_cert
wget -q https://baldr-msk2.yandex-team.ru/linux/files/get-cert/python3-get-cert_0.3.2-1_all.deb -O /target/tmp/get-cert.deb ; \
in-target apt install -y /tmp/get-cert.deb ; \
# Fix grub timeout
echo "GRUB_RECORDFAIL_TIMEOUT=0" >> /target/etc/default/grub ; \
in-target update-grub ; \
wget -q --post-data='id=fae5063b-0e30-4c30-92ee-7d2f91a58659' https://api.baldr.yandex.net/v1/task/close ; \
exit 0
### Finishing up the installation
d-i finish-install/reboot_in_progress note
`

	dep := models.Deploy{
		ID:                           "fae5063b-0e30-4c30-92ee-7d2f91a58659",
		CreationTime:                 time.Time{},
		MACAddresses:                 nil,
		SerialNumber:                 "pf21eqtr",
		InventoryNumber:              "104059015",
		IPAddress:                    net.ParseIP("2a02:6b8:0:410:bd79:1dbc:38d7:1a4f"),
		NextServer:                   net.ParseIP("95.108.198.23"),
		Options:                      make(map[string]string),
		SupportBootingFromLegacyBIOS: false,
		CloseTime:                    time.Time{},
		Status:                       models.DeployStatusOpen,
		ErrorCode:                    baldrerrors.CodeOk,
		Message:                      "",
	}
	dep.Options[models.OptionID] = "fae5063b-0e30-4c30-92ee-7d2f91a58659"
	dep.Options[models.OptionHWUUID] = "90123456-7890-1234-5678-901234567890"
	dep.Options[models.OptionHWManufacturer] = "lenovo"
	dep.Options[models.OptionHWModel] = "20qes1f100"
	dep.Options[models.OptionHWSerialNumber] = "pf21eqtr"
	dep.Options[models.OptionHWInventoryNumber] = "104059015"
	dep.Options[models.OptionHWStatus] = "OPERATION"
	dep.Options[models.OptionFQDN] = "zomb-prj-100.zombie.yandex-team.ru"
	dep.Options[models.OptionIPXEUEFIMode] = "efi"
	dep.Options[models.OptionIPXEFileName] = "ipxe.efi"
	dep.Options[models.OptionDeployType] = DeployTypeZombie
	dep.Options[models.OptionWithDomain] = "false"
	dep.Options[models.OptionWithOffice] = "false"
	dep.Options[models.OptionCountry] = "Russia"
	dep.Options[models.OptionCity] = "Moscow"
	dep.Options[models.OptionOffice] = "Morozov BC"
	dep.Options[models.OptionDistributionPoint] = "baldr-msk2.yandex-team.ru"
	dep.Options[models.OptionMSTimeZone] = "Russian Standard Time"
	dep.Options[models.OptionUnixTimeZone] = "Europe/Moscow"
	dep.Options[models.OptionDefaultLanguageProfile] = "en-ru"
	dep.Options[models.OptionDeploymentShare] = "ds$"
	dep.Options[models.OptionMSSystemLocale] = "0409:00000409"
	dep.Options[models.OptionMSUserLocale] = "ru-ru"
	dep.Options[models.OptionMSKeyboardLocale] = "0409:00000409;0419:00000419"
	dep.Options[models.OptionMSUILanguage] = "en-us"
	dep.Options[models.OptionDILanguage] = "en"
	dep.Options[models.OptionDICountry] = "RU"
	dep.Options[models.OptionDILocale] = "en_US.UTF-8"
	dep.Options[models.OptionDISupportedLocales] = "ru_RU.UTF-8"
	dep.Options[models.OptionDIKeymap] = "ru"
	dep.Options[models.OptionDIToggle] = "Alt+Shift"
	dep.Options[models.OptionSTTask] = "HDRFS-7777777"
	dep.Options[models.OptionSTTaskStatus] = "none"
	dep.Options[models.OptionComputerName] = "i104059015"
	dep.Options[models.OptionUserName] = "zomb-prj-100"
	dep.Options[models.OptionOS] = "Linux"

	for _, macString := range []string{"f8:75:a4:3b:91:d9", "4c:1d:96:45:d8:7d"} {
		mac, err := net.ParseMAC(macString)
		if err != nil {
			t.Errorf("%v", err)
		}
		dep.MACAddresses = append(dep.MACAddresses, mac)
	}

	dep.CreationTime, err = time.Parse(time.RFC3339, "2020-06-30T05:42:18+00:00")
	if err != nil {
		t.Errorf("%v", err)
	}

	dep.Options[models.OptionProdAPIHost] = "api.baldr.yandex.net"
	dep.Options[models.OptionTestAPIHost] = "test.baldr.yandex.net"
	dep.Options[models.OptionOSReleaseName] = linuxReleaseBionic
	dep.Options[models.OptionOSVersion] = linuxReleaseBionicVersion
	dep.Options[models.OptionEnvironment] = deployEnvProd

	var resp bytes.Buffer
	err = executeTemplate(dep, template.FuncMap{}, &resp, zombieGPUPreseedTemplate)
	if err != nil {
		t.Errorf("%v", err)
	}

	if resp.String() != preseed {
		t.Errorf("Got:\n%q\n\nExpected:\n%q", resp.String(), preseed)
	}
}
