// Package config описывает конфигурацию для сервера.
package config

import (
	"fmt"
	"os"
	"strings"

	"github.com/spf13/pflag"
)

// Возможные окружения для работы сервера
const (
	ApplicationEnvProd = "Production"
	ApplicationEnvTest = "Testing"
)

// Дефолтные значения для конфигурации.
const (
	DefaultServerAddress             = ":80"
	DefaultApplicationEnv            = ApplicationEnvTest
	DefaultAPIProdURL                = ""
	DefaultAPITestURL                = ""
	DefaultADRobotName               = ""
	DefaultADRobotDomain             = ""
	DefaultADRobotPassword           = ""
	DefaultWindowsLocalAdminPassword = ""
	DefaultLinuxLocalAdminPassword   = ""
	DefaultSMBUserName               = ""
	DefaultSMBUserDomain             = ""
	DefaultSMBUserPassword           = ""
	DefaultBotBaseURL                = ""
	DefaultBotToken                  = ""
	DefaultYavToken                  = ""
)

// Имена переменных окружения, через которые можно задать параметры конфигурации.
const (
	EnvVarDatabaseDSNs              = "DATABASE_DSN"
	EnvVarServerAddress             = "SERVER_ADDRESS"
	EnvVarApplicationEnv            = "APPLICATION_ENVIRONMENT"
	EnvVarAPIProdURL                = "API_PRODUCTION_URL"
	EnvVarAPITestURL                = "API_TESTING_URL"
	EnvVarADRobotName               = "AD_ROBOT_NAME"
	EnvVarADRobotDomain             = "AD_ROBOT_DOMAIN"
	EnvVarADRobotPassword           = "AD_ROBOT_PASSWORD"
	EnvVarWindowsLocalAdminPassword = "WINDOWS_LOCAL_ADMINISTRATOR_PASSWORD"
	EnvVarLinuxLocalAdminPassword   = "LINUX_LOCAL_ADMINISTRATOR_PASSWORD"
	EnvVarSMBUserName               = "SMB_USER_NAME"
	EnvVarSMBUserDomain             = "SMB_USER_DOMAIN"
	EnvVarSMBUserPassword           = "SMB_USER_PASSWORD"
	EnvVarBotBaseURL                = "BOT_BASE_URL"
	EnvVarBotToken                  = "BOT_OAUTH_TOKEN"
	EnvVarYavToken                  = "YAV_OAUTH_TOKEN"
)

// Configuration — конфигурация сервера.
type Configuration struct {
	// DatabaseDSNs — список DSN для подключения к хостам кластера PostgreSQL.
	DatabaseDSNs []string

	// ServerAddress — адрес, на котором сервер принимает запросы.
	ServerAddress string

	// ApplicationEnv — окружение, в котором работает сервер ("prod" или "test").
	ApplicationEnv string

	// APIProdURL — URL для продакшн версии API.
	APIProdURL string

	// APITestURL — URL для тестовой версии API.
	APITestURL string

	// ADRobotName — имя робота для подключения к Active Directory.
	ADRobotName string

	// ADRobotDomain — имя домена для подключения к Active Directory.
	ADRobotDomain string

	// ADRobotPassword — пароль робота для подключения к Active Directory.
	ADRobotPassword string

	// WindowsLocalAdminPassword — пароль локального администратора для Windows.
	WindowsLocalAdminPassword string

	// LinuxLocalAdminPassword — пароль локального администратора для Linux.
	LinuxLocalAdminPassword string

	// SMBUserName — имя робота для подключения к Distribution Point по SMB.
	SMBUserName string

	// SMBUserDomain — имя домена для подключения к Distribution Point по SMB.
	SMBUserDomain string

	// SMBUserPassword — пароль робота для подключения к Distribution Point по SMB
	SMBUserPassword string

	// BotBaseURL — base URL для API Bot.
	BotBaseURL string

	// BotToken — OAuth-токен для подключения к API Bot.
	BotToken string

	// YavToken — OAuth-токен для подключения к API Yandex Vault.
	YavToken string
}

// NewConfiguration парсит переменные окружения и флаги, создает по ним новую конфигурацию.
func NewConfiguration() (conf Configuration, err error) {
	// Выставляем дефолтные значения.
	conf.setDefault()

	// Парсим переменные окружения.
	err = conf.parseEnv()
	if err != nil {
		err = fmt.Errorf("parse environment variables: %w", err)
		return
	}

	// Парсим флаги.
	err = conf.parseFlags()
	if err != nil {
		err = fmt.Errorf("parse flags: %w", err)
		return
	}

	// Проверяем конфигурацию.
	err = conf.Validate()
	if err != nil {
		err = fmt.Errorf("validate: %w", err)
		return
	}

	return
}

// setDefault выставляет значения по-умолчанию.
func (c *Configuration) setDefault() {
	c.DatabaseDSNs = nil
	c.ServerAddress = DefaultServerAddress
	c.ApplicationEnv = DefaultApplicationEnv
	c.APIProdURL = DefaultAPIProdURL
	c.APITestURL = DefaultAPITestURL
	c.ADRobotName = DefaultADRobotName
	c.ADRobotDomain = DefaultADRobotDomain
	c.ADRobotPassword = DefaultADRobotPassword
	c.WindowsLocalAdminPassword = DefaultWindowsLocalAdminPassword
	c.LinuxLocalAdminPassword = DefaultLinuxLocalAdminPassword
	c.SMBUserName = DefaultSMBUserName
	c.SMBUserDomain = DefaultSMBUserDomain
	c.SMBUserPassword = DefaultSMBUserPassword
	c.BotBaseURL = DefaultBotBaseURL
	c.BotToken = DefaultBotToken
	c.YavToken = DefaultYavToken
}

// parseEnv парсит переменные окружения.
func (c *Configuration) parseEnv() error {
	if value, ok := os.LookupEnv(EnvVarDatabaseDSNs); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarDatabaseDSNs, value)
		}
		c.DatabaseDSNs = strings.Split(value, ",")
	}

	if value, ok := os.LookupEnv(EnvVarServerAddress); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarServerAddress, value)
		}
		c.ServerAddress = value
	}

	if value, ok := os.LookupEnv(EnvVarApplicationEnv); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarApplicationEnv, value)
		}
		c.ApplicationEnv = value
	}

	if value, ok := os.LookupEnv(EnvVarAPIProdURL); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarAPIProdURL, value)
		}
		c.APIProdURL = value
	}

	if value, ok := os.LookupEnv(EnvVarAPITestURL); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarAPITestURL, value)
		}
		c.APITestURL = value
	}

	if value, ok := os.LookupEnv(EnvVarADRobotName); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarADRobotName, value)
		}
		c.ADRobotName = value
	}

	if value, ok := os.LookupEnv(EnvVarADRobotDomain); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarADRobotDomain, value)
		}
		c.ADRobotDomain = value
	}

	if value, ok := os.LookupEnv(EnvVarADRobotPassword); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarADRobotPassword, value)
		}
		c.ADRobotPassword = value
	}

	if value, ok := os.LookupEnv(EnvVarWindowsLocalAdminPassword); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarWindowsLocalAdminPassword, value)
		}
		c.WindowsLocalAdminPassword = value
	}

	if value, ok := os.LookupEnv(EnvVarLinuxLocalAdminPassword); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarLinuxLocalAdminPassword, value)
		}
		c.LinuxLocalAdminPassword = value
	}

	if value, ok := os.LookupEnv(EnvVarSMBUserName); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarSMBUserName, value)
		}
		c.SMBUserName = value
	}

	if value, ok := os.LookupEnv(EnvVarSMBUserDomain); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarSMBUserDomain, value)
		}
		c.SMBUserDomain = value
	}

	if value, ok := os.LookupEnv(EnvVarSMBUserPassword); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarSMBUserPassword, value)
		}
		c.SMBUserPassword = value
	}

	if value, ok := os.LookupEnv(EnvVarBotBaseURL); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarBotBaseURL, value)
		}
		c.BotBaseURL = value
	}

	if value, ok := os.LookupEnv(EnvVarBotToken); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarBotToken, value)
		}
		c.BotToken = value
	}

	if value, ok := os.LookupEnv(EnvVarYavToken); ok {
		if value == "" {
			return fmt.Errorf("invalid value for the environment variable %s = %q", EnvVarYavToken, value)
		}
		c.YavToken = value
	}

	return nil
}

// parseFlags парсит флаги.
func (c *Configuration) parseFlags() error {
	flagSet := pflag.NewFlagSet(os.Args[0], pflag.ContinueOnError)
	flagSet.StringSliceVarP(&c.DatabaseDSNs, "database-dsn", "d", c.DatabaseDSNs,
		"--database-dsn=postgresql://localhost/mydb?user=other&password=secret "+
			"or -d=postgresql://user:secret@localhost:5432,postgresql://user:secret@localhost:6432")
	flagSet.StringVarP(&c.ServerAddress, "server-address", "a", c.ServerAddress, "--server-address=localhost:8080 or -a=localhost:8080")
	flagSet.StringVarP(&c.ApplicationEnv, "environment", "e", c.ApplicationEnv, "--environment=test or -e=prod")
	flagSet.StringVar(&c.APIProdURL, "api-prod-url", c.APIProdURL, "--api-prod-url=https://api.baldr.yandex.net")
	flagSet.StringVar(&c.APITestURL, "api-test-url", c.APITestURL, "--api-test-url=https://test.baldr.yandex.net")
	flagSet.StringVar(&c.ADRobotName, "ad-robot-name", c.ADRobotName, "--ad-robot-name=robot-name")
	flagSet.StringVar(&c.ADRobotDomain, "ad-robot-domain", c.ADRobotDomain, "--ad-robot-domain=example.com")
	flagSet.StringVar(&c.ADRobotPassword, "ad-robot-password", c.ADRobotPassword, "--ad-robot-password=secret")
	flagSet.StringVar(&c.WindowsLocalAdminPassword, "windows-local-admin-password", c.WindowsLocalAdminPassword, "--windows-local-admin-password=windows-secret")
	flagSet.StringVar(&c.LinuxLocalAdminPassword, "linux-local-admin-password", c.LinuxLocalAdminPassword, "--linux-local-admin-password=linux-secret")
	flagSet.StringVar(&c.SMBUserName, "smb-user-name", c.SMBUserName, "--smb-user-name=smb-user")
	flagSet.StringVar(&c.SMBUserDomain, "smb-user-domain", c.SMBUserDomain, "--smb-user-domain=example.com")
	flagSet.StringVar(&c.SMBUserPassword, "smb-user-password", c.SMBUserPassword, "--smb-user-password=super-secret")
	flagSet.StringVar(&c.BotBaseURL, "bot-base-url", c.BotBaseURL, "--bot-base-url=https://bot.yandex-team.ru")
	flagSet.StringVar(&c.BotToken, "bot-token", c.BotToken, "--bot-token=TOKEN")
	flagSet.StringVar(&c.YavToken, "yav-token", c.YavToken, "--yav-token=TOKEN")

	err := flagSet.Parse(os.Args[1:])
	if err != nil {
		fmt.Fprintf(os.Stderr, "Usage of %s:\n", os.Args[0])
		flagSet.PrintDefaults()
		return err
	}

	return nil
}

// Validate проверяет корректность конфигурации.
func (c Configuration) Validate() error {
	// Параметры подключения к БД должны быть определены.
	if len(c.DatabaseDSNs) == 0 {
		return fmt.Errorf("database DSN is not difined")
	}

	// Ни один из DSN не должен быть пустой строкой
	for _, dsn := range c.DatabaseDSNs {
		if dsn == "" {
			return fmt.Errorf("database DSN cannot be empty (%+v)", c.DatabaseDSNs)
		}
	}

	// ServerAddress не должен быть пустым.
	if c.ServerAddress == "" {
		return fmt.Errorf("server address is not defined")
	}

	// ApplicationEnv должно принимать одно из известных значений.
	if c.ApplicationEnv != ApplicationEnvProd && c.ApplicationEnv != ApplicationEnvTest {
		return fmt.Errorf("unknown application environment %q", c.ApplicationEnv)
	}

	// APIProdURL не должен быть пустым.
	if c.APIProdURL == "" {
		return fmt.Errorf("production API URL is not defined")
	}

	// APITestURL не должен быть пустым.
	if c.APITestURL == "" {
		return fmt.Errorf("testing API URL is not defined")
	}

	// ADRobotName не должен быть пустым.
	if c.ADRobotName == "" {
		return fmt.Errorf("the robot name for AD is not defined")
	}

	// ADRobotDomain не должен быть пустым.
	if c.ADRobotDomain == "" {
		return fmt.Errorf("the robot domain for AD is not defined")
	}

	// WindowsLocalAdminPassword не должен быть пустым.
	if c.WindowsLocalAdminPassword == "" {
		return fmt.Errorf("windows local administrator password is not defined")
	}

	// LinuxLocalAdminPassword не должен быть пустым.
	if c.LinuxLocalAdminPassword == "" {
		return fmt.Errorf("linux local administrator password is not defined")
	}

	// SMBUserName не должен быть пустым.
	if c.SMBUserName == "" {
		return fmt.Errorf("the user name for SMB server is not defined")
	}

	// SMBUserDomain не должен быть пустым.
	if c.SMBUserDomain == "" {
		return fmt.Errorf("the user domain for SMB server is not defined")
	}

	// BotBaseURL не должен быть пустым.
	if c.BotBaseURL == "" {
		return fmt.Errorf("the base URL for Bot API is not defined")
	}

	// BotToken не должен быть пустым.
	if c.BotToken == "" {
		return fmt.Errorf("OAuth token for Bot API is not defined")
	}

	// YavToken не должен быть пустым.
	if c.YavToken == "" {
		return fmt.Errorf("OAuth token for Yandex Vault API is not defined")
	}

	return nil
}
