package config

import (
	"fmt"
	"os"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestNewConfiguration(t *testing.T) {
	tests := []struct {
		name    string
		env     map[string]string
		flags   []string
		want    Configuration
		wantErr assert.ErrorAssertionFunc
	}{
		{
			name:  "Without flags and environment variables",
			env:   make(map[string]string),
			flags: []string{os.Args[0]},
			want: Configuration{
				DatabaseDSNs:              nil,
				ServerAddress:             DefaultServerAddress,
				ApplicationEnv:            DefaultApplicationEnv,
				APIProdURL:                DefaultAPIProdURL,
				APITestURL:                DefaultAPITestURL,
				ADRobotName:               DefaultADRobotName,
				ADRobotDomain:             DefaultADRobotDomain,
				ADRobotPassword:           DefaultADRobotPassword,
				WindowsLocalAdminPassword: DefaultWindowsLocalAdminPassword,
				LinuxLocalAdminPassword:   DefaultLinuxLocalAdminPassword,
				SMBUserName:               DefaultSMBUserName,
				SMBUserDomain:             DefaultSMBUserDomain,
				SMBUserPassword:           DefaultSMBUserPassword,
				BotBaseURL:                DefaultBotBaseURL,
				BotToken:                  DefaultBotToken,
				YavToken:                  DefaultYavToken},
			wantErr: assert.Error,
		},
		{
			name:  "Bad env",
			env:   map[string]string{EnvVarDatabaseDSNs: ""},
			flags: []string{os.Args[0]},
			want: Configuration{
				DatabaseDSNs:              nil,
				ServerAddress:             DefaultServerAddress,
				ApplicationEnv:            DefaultApplicationEnv,
				APIProdURL:                DefaultAPIProdURL,
				APITestURL:                DefaultAPITestURL,
				ADRobotName:               DefaultADRobotName,
				ADRobotDomain:             DefaultADRobotDomain,
				ADRobotPassword:           DefaultADRobotPassword,
				WindowsLocalAdminPassword: DefaultWindowsLocalAdminPassword,
				LinuxLocalAdminPassword:   DefaultLinuxLocalAdminPassword,
				SMBUserName:               DefaultSMBUserName,
				SMBUserDomain:             DefaultSMBUserDomain,
				SMBUserPassword:           DefaultSMBUserPassword,
				BotBaseURL:                DefaultBotBaseURL,
				BotToken:                  DefaultBotToken,
				YavToken:                  DefaultYavToken},
			wantErr: assert.Error,
		},
		{
			name:  "Bad flag",
			env:   map[string]string{EnvVarDatabaseDSNs: "postgresql://localhost/mydb?user=other&password=secret"},
			flags: []string{os.Args[0], "-a"},
			want: Configuration{
				DatabaseDSNs:              []string{"postgresql://localhost/mydb?user=other&password=secret"},
				ServerAddress:             DefaultServerAddress,
				ApplicationEnv:            DefaultApplicationEnv,
				APIProdURL:                DefaultAPIProdURL,
				APITestURL:                DefaultAPITestURL,
				ADRobotName:               DefaultADRobotName,
				ADRobotDomain:             DefaultADRobotDomain,
				ADRobotPassword:           DefaultADRobotPassword,
				WindowsLocalAdminPassword: DefaultWindowsLocalAdminPassword,
				LinuxLocalAdminPassword:   DefaultLinuxLocalAdminPassword,
				SMBUserName:               DefaultSMBUserName,
				SMBUserDomain:             DefaultSMBUserDomain,
				SMBUserPassword:           DefaultSMBUserPassword,
				BotBaseURL:                DefaultBotBaseURL,
				BotToken:                  DefaultBotToken,
				YavToken:                  DefaultYavToken},
			wantErr: assert.Error,
		},
		{
			name:  "Unknown flag",
			env:   map[string]string{EnvVarDatabaseDSNs: "postgresql://localhost/mydb?user=other&password=secret"},
			flags: []string{os.Args[0], "--unknown=undefined"},
			want: Configuration{
				DatabaseDSNs:              []string{"postgresql://localhost/mydb?user=other&password=secret"},
				ServerAddress:             DefaultServerAddress,
				ApplicationEnv:            DefaultApplicationEnv,
				APIProdURL:                DefaultAPIProdURL,
				APITestURL:                DefaultAPITestURL,
				ADRobotName:               DefaultADRobotName,
				ADRobotDomain:             DefaultADRobotDomain,
				ADRobotPassword:           DefaultADRobotPassword,
				WindowsLocalAdminPassword: DefaultWindowsLocalAdminPassword,
				LinuxLocalAdminPassword:   DefaultLinuxLocalAdminPassword,
				SMBUserName:               DefaultSMBUserName,
				SMBUserDomain:             DefaultSMBUserDomain,
				SMBUserPassword:           DefaultSMBUserPassword,
				BotBaseURL:                DefaultBotBaseURL,
				BotToken:                  DefaultBotToken,
				YavToken:                  DefaultYavToken},
			wantErr: assert.Error,
		},
		{
			name: "Configuration by environment",
			env: map[string]string{
				EnvVarDatabaseDSNs:              "postgresql://localhost/db?user=other&password=secret",
				EnvVarServerAddress:             "10.200.30.4:12345",
				EnvVarApplicationEnv:            ApplicationEnvProd,
				EnvVarAPIProdURL:                "https://api.baldr.yandex.net",
				EnvVarAPITestURL:                "https://test.baldr.yandex.net",
				EnvVarADRobotName:               "ad_robot_user_name",
				EnvVarADRobotDomain:             "ld.example.com",
				EnvVarADRobotPassword:           "ms_password",
				EnvVarWindowsLocalAdminPassword: "windows_password",
				EnvVarLinuxLocalAdminPassword:   "linux_password",
				EnvVarSMBUserName:               "smb_user_name",
				EnvVarSMBUserDomain:             "smb.example.com",
				EnvVarSMBUserPassword:           "smb_password",
				EnvVarBotBaseURL:                "https://bot.example.com",
				EnvVarBotToken:                  "BOT_TOKEN",
				EnvVarYavToken:                  "YAV_TOKEN",
			},
			flags: []string{os.Args[0]},
			want: Configuration{
				DatabaseDSNs:              []string{"postgresql://localhost/db?user=other&password=secret"},
				ServerAddress:             "10.200.30.4:12345",
				ApplicationEnv:            ApplicationEnvProd,
				APIProdURL:                "https://api.baldr.yandex.net",
				APITestURL:                "https://test.baldr.yandex.net",
				ADRobotName:               "ad_robot_user_name",
				ADRobotDomain:             "ld.example.com",
				ADRobotPassword:           "ms_password",
				WindowsLocalAdminPassword: "windows_password",
				LinuxLocalAdminPassword:   "linux_password",
				SMBUserName:               "smb_user_name",
				SMBUserDomain:             "smb.example.com",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN"},
			wantErr: assert.NoError,
		},
		{
			name: "Configuration by flags",
			env: map[string]string{
				EnvVarDatabaseDSNs:              "postgresql://localhost/db?user=other&password=secret",
				EnvVarServerAddress:             "10.200.30.4:12345",
				EnvVarApplicationEnv:            ApplicationEnvProd,
				EnvVarAPIProdURL:                "https://api.baldr.yandex.net",
				EnvVarAPITestURL:                "https://test.baldr.yandex.net",
				EnvVarADRobotName:               "ad_robot_user_name",
				EnvVarADRobotDomain:             "ld.example.com",
				EnvVarADRobotPassword:           "ms_password",
				EnvVarWindowsLocalAdminPassword: "windows_password",
				EnvVarLinuxLocalAdminPassword:   "linux_password",
				EnvVarSMBUserName:               "smb_user_name",
				EnvVarSMBUserDomain:             "smb.example.com",
				EnvVarSMBUserPassword:           "smb_password",
				EnvVarBotBaseURL:                "https://bot.example.com",
				EnvVarBotToken:                  "BOT_TOKEN",
				EnvVarYavToken:                  "YAV_TOKEN",
			},
			flags: []string{
				os.Args[0],
				"-d=postgresql://user:secret@localhost:5432,postgresql://user:secret@localhost:6432",
				"-a=localhost:8989",
				fmt.Sprintf("-e=%s", ApplicationEnvTest),
				"--api-prod-url=https://prod.baldr.yandex.net",
				"--api-test-url=https://dev.baldr.yandex.net",
				"--ad-robot-name=robot-name-with-flag",
				"--ad-robot-domain=flag.example.com",
				"--ad-robot-password=flag_secret",
				"--windows-local-admin-password=windows-secret",
				"--linux-local-admin-password=linux-secret",
				"--smb-user-name=smb-user",
				"--smb-user-domain=example.com",
				"--smb-user-password=super-secret",
				"--bot-base-url=https://bot.yandex-team.ru",
				"--bot-token=BOT_OAUTH_TOKEN",
				"--yav-token=YAV_OAUTH_TOKEN",
			},
			want: Configuration{
				DatabaseDSNs: []string{
					"postgresql://user:secret@localhost:5432",
					"postgresql://user:secret@localhost:6432",
				},
				ServerAddress:             "localhost:8989",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://prod.baldr.yandex.net",
				APITestURL:                "https://dev.baldr.yandex.net",
				ADRobotName:               "robot-name-with-flag",
				ADRobotDomain:             "flag.example.com",
				ADRobotPassword:           "flag_secret",
				WindowsLocalAdminPassword: "windows-secret",
				LinuxLocalAdminPassword:   "linux-secret",
				SMBUserName:               "smb-user",
				SMBUserDomain:             "example.com",
				SMBUserPassword:           "super-secret",
				BotBaseURL:                "https://bot.yandex-team.ru",
				BotToken:                  "BOT_OAUTH_TOKEN",
				YavToken:                  "YAV_OAUTH_TOKEN"},
			wantErr: assert.NoError,
		},
		{
			name: "Configuration with flags priority",
			env:  make(map[string]string),
			flags: []string{
				os.Args[0],
				"-d=postgresql://user:secret@localhost:5432,postgresql://user:secret@localhost:6432",
				"-a=localhost:8989",
				fmt.Sprintf("-e=%s", ApplicationEnvProd),
				"--api-prod-url=https://prod.baldr.yandex.net",
				"--api-test-url=https://dev.baldr.yandex.net",
				"--ad-robot-name=robot-name-with-flag",
				"--ad-robot-domain=flag.example.com",
				"--ad-robot-password=flag_secret",
				"--windows-local-admin-password=windows-secret",
				"--linux-local-admin-password=linux-secret",
				"--smb-user-name=smb-user",
				"--smb-user-domain=example.com",
				"--smb-user-password=super-secret",
				"--bot-base-url=https://bot.yandex-team.ru",
				"--bot-token=BOT_OAUTH_TOKEN",
				"--yav-token=YAV_OAUTH_TOKEN",
			},
			want: Configuration{
				DatabaseDSNs: []string{
					"postgresql://user:secret@localhost:5432",
					"postgresql://user:secret@localhost:6432",
				},
				ServerAddress:             "localhost:8989",
				ApplicationEnv:            ApplicationEnvProd,
				APIProdURL:                "https://prod.baldr.yandex.net",
				APITestURL:                "https://dev.baldr.yandex.net",
				ADRobotName:               "robot-name-with-flag",
				ADRobotDomain:             "flag.example.com",
				ADRobotPassword:           "flag_secret",
				WindowsLocalAdminPassword: "windows-secret",
				LinuxLocalAdminPassword:   "linux-secret",
				SMBUserName:               "smb-user",
				SMBUserDomain:             "example.com",
				SMBUserPassword:           "super-secret",
				BotBaseURL:                "https://bot.yandex-team.ru",
				BotToken:                  "BOT_OAUTH_TOKEN",
				YavToken:                  "YAV_OAUTH_TOKEN"},
			wantErr: assert.NoError,
		},
	}

	// Сохраняем флаги запуска в отдельную переменную.
	osArgs := os.Args

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			defer func() {
				// Восстанавливаем значение флагов запуска из переменной.
				os.Args = osArgs

				// Сбрасываем ранее выставленные переменные окружения.
				for variable := range tt.env {
					err := os.Unsetenv(variable)
					assert.NoError(t, err)
				}
			}()

			// Выставляем переменные окружения.
			for variable, value := range tt.env {
				err := os.Setenv(variable, value)
				assert.NoError(t, err)
			}

			// Задаем новые флаги запуска.
			os.Args = tt.flags

			conf, err := NewConfiguration()
			tt.wantErr(t, err)
			assert.Equal(t, tt.want, conf)
		})
	}
}

func TestConfiguration_setDefault(t *testing.T) {
	var conf Configuration
	want := Configuration{
		DatabaseDSNs:              nil,
		ServerAddress:             DefaultServerAddress,
		ApplicationEnv:            DefaultApplicationEnv,
		APIProdURL:                DefaultAPIProdURL,
		APITestURL:                DefaultAPITestURL,
		ADRobotName:               DefaultADRobotName,
		ADRobotDomain:             DefaultADRobotDomain,
		ADRobotPassword:           DefaultADRobotPassword,
		WindowsLocalAdminPassword: DefaultWindowsLocalAdminPassword,
		LinuxLocalAdminPassword:   DefaultLinuxLocalAdminPassword,
		SMBUserName:               DefaultSMBUserName,
		SMBUserDomain:             DefaultSMBUserDomain,
		SMBUserPassword:           DefaultSMBUserPassword,
		BotBaseURL:                DefaultBotBaseURL,
		BotToken:                  DefaultBotToken,
		YavToken:                  DefaultYavToken,
	}

	// Устанавливаем значения по-умолчанию.
	conf.setDefault()

	// Проверяем полученную конфигурацию.
	assert.Equal(t, want, conf)
}

func TestConfiguration_parseEnv(t *testing.T) {
	tests := []struct {
		name    string
		env     map[string]string
		want    Configuration
		wantErr assert.ErrorAssertionFunc
	}{
		{
			name:    "Database DSN #1",
			env:     map[string]string{EnvVarDatabaseDSNs: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "Database DSN #2",
			env:  map[string]string{EnvVarDatabaseDSNs: "postgresql://localhost/mydb?user=other&password=secret"},
			want: Configuration{
				DatabaseDSNs: []string{"postgresql://localhost/mydb?user=other&password=secret"},
			},
			wantErr: assert.NoError,
		},
		{
			name: "Database DSN #3",
			env:  map[string]string{EnvVarDatabaseDSNs: "postgresql://user:secret@localhost:5432,postgresql://user:secret@localhost:6432"},
			want: Configuration{
				DatabaseDSNs: []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
			},
			wantErr: assert.NoError,
		},
		{
			name: "Database DSN #3",
			env:  map[string]string{EnvVarDatabaseDSNs: "postgresql://user:secret@localhost:5432,postgresql://user:secret@localhost:6432,"},
			want: Configuration{
				DatabaseDSNs: []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432", ""},
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ServerAddress #1",
			env:     map[string]string{EnvVarServerAddress: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "ServerAddress #2",
			env:  map[string]string{EnvVarServerAddress: "localhost:8080"},
			want: Configuration{
				ServerAddress: "localhost:8080",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ApplicationEnv #1",
			env:     map[string]string{EnvVarApplicationEnv: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "ApplicationEnv #2",
			env:  map[string]string{EnvVarApplicationEnv: ApplicationEnvTest},
			want: Configuration{
				ApplicationEnv: ApplicationEnvTest,
			},
			wantErr: assert.NoError,
		},
		{
			name: "ApplicationEnv #3",
			env:  map[string]string{EnvVarApplicationEnv: ApplicationEnvProd},
			want: Configuration{
				ApplicationEnv: ApplicationEnvProd,
			},
			wantErr: assert.NoError,
		},
		{
			name:    "APIProdURL #1",
			env:     map[string]string{EnvVarAPIProdURL: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "APIProdURL #2",
			env:  map[string]string{EnvVarAPIProdURL: "https://api.example.com"},
			want: Configuration{
				APIProdURL: "https://api.example.com",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "APITestURL #1",
			env:     map[string]string{EnvVarAPITestURL: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "APITestURL #2",
			env:  map[string]string{EnvVarAPITestURL: "https://test.example.com"},
			want: Configuration{
				APITestURL: "https://test.example.com",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ADRobotName #1",
			env:     map[string]string{EnvVarADRobotName: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "ADRobotName #2",
			env:  map[string]string{EnvVarADRobotName: "ad_robot"},
			want: Configuration{
				ADRobotName: "ad_robot",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ADRobotDomain #1",
			env:     map[string]string{EnvVarADRobotDomain: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "ADRobotDomain #2",
			env:  map[string]string{EnvVarADRobotDomain: "ad_domain"},
			want: Configuration{
				ADRobotDomain: "ad_domain",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ADRobotPassword #1",
			env:     map[string]string{EnvVarADRobotPassword: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "ADRobotPassword #2",
			env:  map[string]string{EnvVarADRobotPassword: "ad_secret"},
			want: Configuration{
				ADRobotPassword: "ad_secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "WindowsLocalAdminPassword #1",
			env:     map[string]string{EnvVarWindowsLocalAdminPassword: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "WindowsLocalAdminPassword #2",
			env:  map[string]string{EnvVarWindowsLocalAdminPassword: "windows_secret"},
			want: Configuration{
				WindowsLocalAdminPassword: "windows_secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "LinuxLocalAdminPassword #1",
			env:     map[string]string{EnvVarLinuxLocalAdminPassword: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "LinuxLocalAdminPassword #2",
			env:  map[string]string{EnvVarLinuxLocalAdminPassword: "linux_secret"},
			want: Configuration{
				LinuxLocalAdminPassword: "linux_secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "SMBUserName #1",
			env:     map[string]string{EnvVarSMBUserName: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "SMBUserName #2",
			env:  map[string]string{EnvVarSMBUserName: "smb_user"},
			want: Configuration{
				SMBUserName: "smb_user",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "SMBUserDomain #1",
			env:     map[string]string{EnvVarSMBUserDomain: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "SMBUserDomain #2",
			env:  map[string]string{EnvVarSMBUserDomain: "smb_domain"},
			want: Configuration{
				SMBUserDomain: "smb_domain",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "SMBUserPassword #1",
			env:     map[string]string{EnvVarSMBUserPassword: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "SMBUserPassword #2",
			env:  map[string]string{EnvVarSMBUserPassword: "smb_secret"},
			want: Configuration{
				SMBUserPassword: "smb_secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "BotBaseURL #1",
			env:     map[string]string{EnvVarBotBaseURL: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "BotBaseURL #2",
			env:  map[string]string{EnvVarBotBaseURL: "https://bot.example.com"},
			want: Configuration{
				BotBaseURL: "https://bot.example.com",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "BotToken #1",
			env:     map[string]string{EnvVarBotToken: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "BotToken #2",
			env:  map[string]string{EnvVarBotToken: "BOT_TOKEN"},
			want: Configuration{
				BotToken: "BOT_TOKEN",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "YavToken #1",
			env:     map[string]string{EnvVarYavToken: ""},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name: "YavToken #2",
			env:  map[string]string{EnvVarYavToken: "YAV_TOKEN"},
			want: Configuration{
				YavToken: "YAV_TOKEN",
			},
			wantErr: assert.NoError,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			defer func() {
				// Сбрасываем ранее выставленные переменные окружения.
				for variable := range tt.env {
					err := os.Unsetenv(variable)
					assert.NoError(t, err)
				}
			}()

			// Выставляем переменные окружения.
			for variable, value := range tt.env {
				err := os.Setenv(variable, value)
				assert.NoError(t, err)
			}

			var conf Configuration
			err := conf.parseEnv()
			tt.wantErr(t, err)
			assert.Equal(t, tt.want, conf)
		})
	}
}

func TestConfiguration_parseFlags(t *testing.T) {
	tests := []struct {
		name    string
		flags   []string
		want    Configuration
		wantErr assert.ErrorAssertionFunc
	}{
		{
			name:    "No Flags",
			flags:   []string{os.Args[0]},
			want:    Configuration{},
			wantErr: assert.NoError,
		},
		{
			name:  "DatabaseDSNs, long name",
			flags: []string{os.Args[0], "--database-dsn=postgresql://localhost/mydb?user=other&password=secret"},
			want: Configuration{
				DatabaseDSNs: []string{"postgresql://localhost/mydb?user=other&password=secret"},
			},
			wantErr: assert.NoError,
		},
		{
			name:  "DatabaseDSNs, short name",
			flags: []string{os.Args[0], "-d=postgresql://user:secret@localhost:5432,postgresql://user:secret@localhost:6432"},
			want: Configuration{
				DatabaseDSNs: []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
			},
			wantErr: assert.NoError,
		},
		{
			name:    "DatabaseDSNs, without value",
			flags:   []string{os.Args[0], "-d"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "ServerAddress, long name",
			flags: []string{os.Args[0], "--server-address=localhost:8080"},
			want: Configuration{
				ServerAddress: "localhost:8080",
			},
			wantErr: assert.NoError,
		},
		{
			name:  "ServerAddress, short name",
			flags: []string{os.Args[0], "-a=localhost:8080"},
			want: Configuration{
				ServerAddress: "localhost:8080",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ServerAddress, without value",
			flags:   []string{os.Args[0], "-a"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "ApplicationEnv, long name",
			flags: []string{os.Args[0], "--environment=test"},
			want: Configuration{
				ApplicationEnv: "test",
			},
			wantErr: assert.NoError,
		},
		{
			name:  "ApplicationEnv, short name",
			flags: []string{os.Args[0], "-e=prod"},
			want: Configuration{
				ApplicationEnv: "prod",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ApplicationEnv, without value",
			flags:   []string{os.Args[0], "-e"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "APIProdURL",
			flags: []string{os.Args[0], "--api-prod-url=https://api.baldr.yandex.net"},
			want: Configuration{
				APIProdURL: "https://api.baldr.yandex.net",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "APIProdURL, without value",
			flags:   []string{os.Args[0], "--api-prod-url"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "APITestURL",
			flags: []string{os.Args[0], "--api-test-url=https://test.baldr.yandex.net"},
			want: Configuration{
				APITestURL: "https://test.baldr.yandex.net",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "APITestURL, without value",
			flags:   []string{os.Args[0], "--api-test-url"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "ADRobotName",
			flags: []string{os.Args[0], "--ad-robot-name=robot-name"},
			want: Configuration{
				ADRobotName: "robot-name",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ADRobotName, without value",
			flags:   []string{os.Args[0], "--ad-robot-name"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "ADRobotDomain",
			flags: []string{os.Args[0], "--ad-robot-domain=example.com"},
			want: Configuration{
				ADRobotDomain: "example.com",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ADRobotDomain, without value",
			flags:   []string{os.Args[0], "--ad-robot-domain"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "ADRobotPassword",
			flags: []string{os.Args[0], "--ad-robot-password=secret"},
			want: Configuration{
				ADRobotPassword: "secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "ADRobotPassword, without value",
			flags:   []string{os.Args[0], "--ad-robot-password"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "WindowsLocalAdminPassword",
			flags: []string{os.Args[0], "--windows-local-admin-password=windows-secret"},
			want: Configuration{
				WindowsLocalAdminPassword: "windows-secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "WindowsLocalAdminPassword, without value",
			flags:   []string{os.Args[0], "--windows-local-admin-password"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "LinuxLocalAdminPassword",
			flags: []string{os.Args[0], "--linux-local-admin-password=linux-secret"},
			want: Configuration{
				LinuxLocalAdminPassword: "linux-secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "LinuxLocalAdminPassword, without value",
			flags:   []string{os.Args[0], "--linux-local-admin-password"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "SMBUserName",
			flags: []string{os.Args[0], "--smb-user-name=smb-user"},
			want: Configuration{
				SMBUserName: "smb-user",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "SMBUserName, without value",
			flags:   []string{os.Args[0], "-smb-user-name"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "SMBUserDomain",
			flags: []string{os.Args[0], "--smb-user-domain=example.com"},
			want: Configuration{
				SMBUserDomain: "example.com",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "SMBUserDomain, without value",
			flags:   []string{os.Args[0], "--smb-user-domain"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "SMBUserPassword",
			flags: []string{os.Args[0], "--smb-user-password=super-secret"},
			want: Configuration{
				SMBUserPassword: "super-secret",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "SMBUserPassword, without value",
			flags:   []string{os.Args[0], "--smb-user-password"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "BotBaseURL",
			flags: []string{os.Args[0], "--bot-base-url=https://bot.yandex-team.ru"},
			want: Configuration{
				BotBaseURL: "https://bot.yandex-team.ru",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "BotBaseURL, without value",
			flags:   []string{os.Args[0], "--bot-base-url"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "BotToken",
			flags: []string{os.Args[0], "--bot-token=BOT_TOKEN"},
			want: Configuration{
				BotToken: "BOT_TOKEN",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "BotToken, without value",
			flags:   []string{os.Args[0], "--bot-token"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
		{
			name:  "YavToken",
			flags: []string{os.Args[0], "--yav-token=YAV_TOKEN"},
			want: Configuration{
				YavToken: "YAV_TOKEN",
			},
			wantErr: assert.NoError,
		},
		{
			name:    "YavToken, without value",
			flags:   []string{os.Args[0], "--yav-token"},
			want:    Configuration{},
			wantErr: assert.Error,
		},
	}

	// Сохраняем флаги запуска в отдельную переменную.
	osArgs := os.Args

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			defer func() {
				// Восстанавливаем значение флагов запуска из переменной.
				os.Args = osArgs
			}()

			os.Args = tt.flags

			var conf Configuration
			err := conf.parseFlags()
			tt.wantErr(t, err)
			assert.Equal(t, tt.want, conf)
		})
	}
}

func TestConfiguration_Validate(t *testing.T) {
	tests := []struct {
		name    string
		conf    Configuration
		wantErr assert.ErrorAssertionFunc
	}{
		{
			name: "DatabaseDSNs is nil",
			conf: Configuration{
				DatabaseDSNs:              nil,
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "DatabaseDSNs is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "DatabaseDSNs with empty string",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", ""},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "ServerAddress is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "ApplicationEnv is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            "",
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "ApplicationEnv is unknown",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            "unknown",
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "ApplicationEnv is 'Testing'",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.NoError,
		},
		{
			name: "ApplicationEnv is 'Production'",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvProd,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.NoError,
		},
		{
			name: "APIProdURL is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "APITestURL is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "ADRobotName is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "ADRobotDomain is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "ADRobotPassword is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.NoError,
		},
		{
			name: "WindowsLocalAdminPassword is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "LinuxLocalAdminPassword is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "SMBUserName is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "SMBUserDomain is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "SMBUserPassword is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.NoError,
		},
		{
			name: "BotBaseURL is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "BotToken is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.Error,
		},
		{
			name: "YavToken is empty",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "",
			},
			wantErr: assert.Error,
		},
		{
			name: "Ok",
			conf: Configuration{
				DatabaseDSNs:              []string{"postgresql://user:secret@localhost:5432", "postgresql://user:secret@localhost:6432"},
				ServerAddress:             "localhost",
				ApplicationEnv:            ApplicationEnvTest,
				APIProdURL:                "https://api.example.com",
				APITestURL:                "https://test.example.com",
				ADRobotName:               "ad_robot",
				ADRobotDomain:             "ad_domain",
				ADRobotPassword:           "ad_password",
				WindowsLocalAdminPassword: "windows_local_password",
				LinuxLocalAdminPassword:   "linux_local_password",
				SMBUserName:               "smb_user",
				SMBUserDomain:             "smb_domain",
				SMBUserPassword:           "smb_password",
				BotBaseURL:                "https://bot.example.com",
				BotToken:                  "BOT_TOKEN",
				YavToken:                  "YAV_TOKEN",
			},
			wantErr: assert.NoError,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			err := tt.conf.Validate()
			tt.wantErr(t, err)
		})
	}
}
