// Package handlers содержит хэндлеры для обработки запросов к http api.
package handler

import (
	"context"
	"errors"
	"net/http"

	"go.uber.org/zap"

	"a.yandex-team.ru/helpdesk/infra/baldr/internal/baldrerrors"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/logging"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/service/deploy"
	"a.yandex-team.ru/helpdesk/infra/baldr/internal/service/keyescrow"
)

// Option функция конфигурирования обработчика.
type Option func(h *DeployHandler) error

// DeployHandler описывает структуру обработчика.
type DeployHandler struct {
	prodEnvName string
	testEnvName string
	currentEnv  string
	prodAddress string
	testAddress string
	tm          deploy.TaskManager
	kk          keyescrow.KeyKeeper
}

// Log возвращает логгер из контекста.
func (h DeployHandler) Log(ctx context.Context) *zap.SugaredLogger {
	_, logger := logging.GetCtxLogger(ctx)
	logger = logger.With(zap.String(logging.ServiceKey, "Metrics handler"))

	return logger
}

// handleError обрабатывает ошибки в зависимости от их типа.
func (h DeployHandler) handleError(ctx context.Context, w http.ResponseWriter, err error) {
	var status int
	if errors.Is(err, baldrerrors.ErrNotFound) {
		h.Log(ctx).
			Warn(err)
		status = http.StatusNotFound
	} else if errors.Is(err, baldrerrors.ErrBadRequest) {
		h.Log(ctx).
			Warn(err)
		status = http.StatusBadRequest
	} else {
		h.Log(ctx).
			Error(err)
		status = http.StatusInternalServerError
	}

	w.WriteHeader(status)
}
