package logging

import (
	"context"

	"github.com/gofrs/uuid"
	"go.uber.org/zap"
)

const (
	contextKeyLogger        = ContextKey("Logger")
	contextKeyCorrelationID = ContextKey("CorrelationID")
)

// GetCtxLogger возвращает логгер из контекста.
func GetCtxLogger(ctx context.Context) (context.Context, *zap.SugaredLogger) {
	if ctx == nil {
		ctx = context.Background()
	}

	if ctxValue := ctx.Value(contextKeyLogger); ctxValue != nil {
		if ctxLogger, ok := ctxValue.(*zap.SugaredLogger); ok {
			return ctx, ctxLogger
		}
	}

	correlationID, _ := uuid.NewV4()
	logger := NewLogger().With(zap.String(CorrelationIDKey, correlationID.String()))

	ctx = context.WithValue(ctx, contextKeyCorrelationID, correlationID.String())

	return SetCtxLogger(ctx, logger), logger
}

// SetCtxLogger сохраняет логгер в контекст.
func SetCtxLogger(ctx context.Context, logger *zap.SugaredLogger) context.Context {
	return context.WithValue(ctx, contextKeyLogger, logger)
}
