package model

import (
	"fmt"
	"sort"
)

// Template описывает структуру шаблона для развертывания ОС, где:
// Name — имя шаблона;
// DisplayName — отображаемое в загрузочном меню имя шаблона;
// Options — параметры шаблона в виде набора пар ключ/значение;
// Priority — приоритет шаблона влияет на его позицию в загрузочном меню, чем больше, тем выше.
type Template struct {
	Name        string
	DisplayName string
	Options     map[string]string
	Priority    int
}

// Validate — метод, позволяющий проверить корректность шаблона.
func (t Template) Validate() error {
	switch {
	case t.Name == "":
		return fmt.Errorf("template name cannot be empty")
	case !isTemplateNameValid(t.Name):
		return fmt.Errorf("template name (%q) must contain only letters, numbers and underscore", t.Name)
	case t.DisplayName == "":
		return fmt.Errorf("template display name cannot be empty")
	case t.Options == nil:
		return fmt.Errorf("template options cannot be nil")
	}
	return nil
}

// isTemplateNameValid — проверяет корректность имени шаблона.
// Так как имя шаблона может использоваться как метка в скриптах iPXE,
// может содержать только английские буквы, цифры и '_'.
func isTemplateNameValid(name string) bool {
	if name == "" {
		return false
	}

	for _, char := range name {
		switch {
		case char >= 'a' && char <= 'z':
			continue
		case char >= 'A' && char <= 'Z':
			continue
		case char >= '0' && char <= '9':
			continue
		case char == '_':
			continue
		default:
			return false
		}
	}

	return true
}

// SortTemplates сортирует слайс шаблонов по их приоритету и DisplayName.
func SortTemplates(templates []Template) {
	sort.Slice(templates, func(i int, j int) bool {
		if templates[i].Priority > templates[j].Priority {
			return true
		} else if templates[i].Priority == templates[j].Priority {
			return templates[i].DisplayName < templates[j].DisplayName
		} else {
			return false
		}
	})
}
