package model

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestSortTemplates(t *testing.T) {
	tests := []struct {
		name      string
		templates []Template
		want      []Template
	}{
		{
			name:      "Empty slice",
			templates: []Template{},
			want:      []Template{},
		},
		{
			name: "Sort #1",
			templates: []Template{
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    2,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    3,
				},
			},
			want: []Template{
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    2,
				},
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    1,
				},
			},
		},
		{
			name: "Sort #2",
			templates: []Template{
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    2,
				},
			},
			want: []Template{
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    2,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    1,
				},
			},
		},
		{
			name: "Sort #3",
			templates: []Template{
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    1,
				},
			},
			want: []Template{
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    1,
				},
			},
		},
		{
			name: "Sort #4",
			templates: []Template{
				{
					Name:        "animals",
					DisplayName: "Animals",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "animals_test",
					DisplayName: "Animals (Test)",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    1,
				},
			},
			want: []Template{
				{
					Name:        "animals",
					DisplayName: "Animals",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "animals_test",
					DisplayName: "Animals (Test)",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    1,
				},
			},
		},
		{
			name: "Sort #5",
			templates: []Template{
				{
					Name:        "windows_11_21h2",
					DisplayName: "Windows 11 Enterprise 21H2",
					Options:     nil,
					Priority:    2,
				},
				{
					Name:        "animals",
					DisplayName: "Animals",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "animals_test",
					DisplayName: "Animals (Test)",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    3,
				},
			},
			want: []Template{
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_11_21h2",
					DisplayName: "Windows 11 Enterprise 21H2",
					Options:     nil,
					Priority:    2,
				},
				{
					Name:        "animals",
					DisplayName: "Animals",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "animals_test",
					DisplayName: "Animals (Test)",
					Options:     nil,
					Priority:    1,
				},
			},
		},
		{
			name: "Sort #6",
			templates: []Template{
				{
					Name:        "windows_11_21h2",
					DisplayName: "Windows 11 Enterprise 21H2",
					Options:     nil,
					Priority:    2,
				},
				{
					Name:        "animals",
					DisplayName: "Animals",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "animals_test",
					DisplayName: "Animals (Test)",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    4,
				},
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    5,
				},
			},
			want: []Template{
				{
					Name:        "windows_21h2",
					DisplayName: "Windows 10 Enterprise 21H2",
					Options:     nil,
					Priority:    5,
				},
				{
					Name:        "windows_21h1",
					DisplayName: "Windows 10 Enterprise 21H1",
					Options:     nil,
					Priority:    4,
				},
				{
					Name:        "windows_20h2",
					DisplayName: "Windows 10 Enterprise 20H2",
					Options:     nil,
					Priority:    3,
				},
				{
					Name:        "windows_11_21h2",
					DisplayName: "Windows 11 Enterprise 21H2",
					Options:     nil,
					Priority:    2,
				},
				{
					Name:        "animals",
					DisplayName: "Animals",
					Options:     nil,
					Priority:    1,
				},
				{
					Name:        "animals_test",
					DisplayName: "Animals (Test)",
					Options:     nil,
					Priority:    1,
				},
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			SortTemplates(tt.templates)
			assert.Equal(t, tt.want, tt.templates)
		})
	}
}

func TestTemplate_Validate(t *testing.T) {
	tests := []struct {
		name     string
		template Template
		want     assert.ErrorAssertionFunc
	}{
		{
			name: "Empty name",
			template: Template{
				Name:        "",
				DisplayName: "Empty",
				Options:     make(map[string]string),
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Invalid name #1",
			template: Template{
				Name:        " ",
				DisplayName: "DisplayName",
				Options:     make(map[string]string),
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Invalid name #2",
			template: Template{
				Name:        "invalid name",
				DisplayName: "DisplayName",
				Options:     make(map[string]string),
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Invalid name #3",
			template: Template{
				Name:        "имя",
				DisplayName: "DisplayName",
				Options:     make(map[string]string),
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Invalid name #4",
			template: Template{
				Name:        "invalid-name",
				DisplayName: "DisplayName",
				Options:     make(map[string]string),
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Invalid name #5",
			template: Template{
				Name:        "Name#5",
				DisplayName: "DisplayName",
				Options:     make(map[string]string),
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Empty display name",
			template: Template{
				Name:        "name",
				DisplayName: "",
				Options:     make(map[string]string),
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Nil options",
			template: Template{
				Name:        "name",
				DisplayName: "display name",
				Options:     nil,
				Priority:    0,
			},
			want: assert.Error,
		},
		{
			name: "Ok",
			template: Template{
				Name:        "Windows_11_21h2",
				DisplayName: "Windows 11 Enterprise 21H2",
				Options:     make(map[string]string),
				Priority:    100500,
			},
			want: assert.NoError,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			tt.want(t, tt.template.Validate())
		})
	}
}

func TestIsTemplateNameValid(t *testing.T) {
	tests := []struct {
		name         string
		templateName string
		want         bool
	}{
		{
			name:         "Empty name",
			templateName: "",
			want:         false,
		},
		{
			name:         "Invalid #1",
			templateName: "name with spaces",
			want:         false,
		},
		{
			name:         "Invalid #2",
			templateName: "name_with_comma,",
			want:         false,
		},
		{
			name:         "Invalid #3",
			templateName: "name_with_colon:",
			want:         false,
		},
		{
			name:         "Invalid #4",
			templateName: "имя_на_русском",
			want:         false,
		},
		{
			name:         "Invalid #5",
			templateName: " ",
			want:         false,
		},
		{
			name:         "Valid #1",
			templateName: "name",
			want:         true,
		},
		{
			name:         "Valid #2",
			templateName: "simple_name",
			want:         true,
		},
		{
			name:         "Valid #3",
			templateName: "name_with_numbers_1234",
			want:         true,
		},
		{
			name:         "Valid #4",
			templateName: "Capitalize",
			want:         true,
		},
		{
			name:         "Valid #5",
			templateName: "Windows_11_21h2",
			want:         true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			assert.Equal(t, tt.want, isTemplateNameValid(tt.templateName))
		})
	}
}
