package models

import (
	"database/sql"
	"encoding/json"
	"encoding/xml"
	"fmt"
	"math/rand"
	"net"
	"strings"
	"time"

	"a.yandex-team.ru/helpdesk/infra/baldr/internal/baldrerrors"
)

const (
	DeployStatusClosed         = "closed"
	DeployStatusOpen           = "open"
	IPXEModeUEFI               = "efi"
	IPXEModeBIOS               = "pcbios"
	UndefinedIPXEMode          = "undefined"
	UndefinedUUID              = "00000000-0000-0000-0000-000000000000"
	UndefinedMACAddress        = "00-00-00-00-00-00"
	UndefinedIPAddress         = "::"
	UndefinedSerialNumber      = "undefined"
	UndefinedHWAsset           = "undefined"
	UndefinedInventoryNumber   = "undefined"
	UndefinedIPXEFileName      = "undefined"
	UndefinedIPXEHostname      = "undefined"
	UndefinedIPXEDHCPUserClass = "undefined"
	UndefinedNextServer        = "0.0.0.0"
	UnknownManufacturer        = "unknown"
	UnknownProduct             = "unknown"
	NoDeployServer             = "none"
	NoDeploymentShare          = "none"
	DefaultDriversProfile      = "Nothing"
)

const (
	OptionID                     = "id"
	OptionDeploymentTaskIsOpen   = "deployment-task-is-open"
	OptionHWInventoryNumber      = "inventory-number"
	OptionHWProfile              = "hw-profile"
	OptionFQDN                   = "fqdn"
	OptionHWStatus               = "hw-status"
	OptionHWType                 = "hw-type"
	OptionHWUUID                 = "hw-uuid"
	OptionHWManufacturer         = "hw-manufacturer"
	OptionHWModel                = "hw-model"
	OptionHWSerialNumber         = "hw-serialnumber"
	OptionHWAsset                = "hw-asset"
	OptionHWOwner                = "hw-owner"
	OptionHWExtUser              = "hw-ext-user"
	OptionIPXEUEFIMode           = "uefi-mode"
	OptionIPXEHostname           = "ipxe-hostname"
	OptionIPXEFileName           = "ipxe-filename"
	OptionIPXEDHCPUserClass      = "dhcp-user-class"
	OptionDeployType             = "deploy-type"
	OptionSTTask                 = "st-task"
	OptionSTTaskStatus           = "st-task-status"
	OptionComputerName           = "computername"
	OptionUserName               = "username"
	OptionOS                     = "os"
	OptionOSVersion              = "os-version"
	OptionOSEdition              = "os-edition"
	OptionOSReleaseName          = "os-release-name"
	OptionOSBuild                = "build"
	OptionOSUpdated              = "os-updated"
	OptionOSProductKey           = "product-key"
	OptionWithDomain             = "with-domain"
	OptionWithOffice             = "with-office"
	OptionCountry                = "country"
	OptionCity                   = "city"
	OptionOffice                 = "office"
	OptionDistributionPoint      = "distribution-point"
	OptionDeploymentShare        = "smb-deployment-share"
	OptionDefaultLanguageProfile = "default-language-profile"
	OptionLanguageProfile        = "language-profile"
	OptionUnixTimeZone           = "unix-timezone"
	OptionMSTimeZone             = "ms-timezone"
	OptionMSSystemLocale         = "system-locale"
	OptionMSUserLocale           = "user-locale"
	OptionMSKeyboardLocale       = "keyboard-locale"
	OptionMSUILanguage           = "ui-language"
	OptionDILanguage             = "di-language"
	OptionDICountry              = "di-country"
	OptionDILocale               = "di-locale"
	OptionDISupportedLocales     = "di-supported-locales"
	OptionDIKeymap               = "di-keymap"
	OptionDIToggle               = "di-toggle"
	OptionADOU                   = "ou"
	OptionDriversProfile         = "drivers-profile"
	OptionADRobotName            = "robot-name"
	OptionADRobotDomain          = "robot-domain"
	OptionADRobotPassword        = "robot-password"
	OptionLocalAdminPassword     = "local-admin-password"
	OptionCurrentAPIHost         = "current-api-host"
	OptionProdAPIHost            = "prod-api-host"
	OptionTestAPIHost            = "test-api-host"
	OptionEnvironment            = "environment"
	OptionDeploymentTemplate     = "template"
)

type Deploy struct {
	ID                           string
	CreationTime                 time.Time
	MACAddresses                 []net.HardwareAddr
	SerialNumber                 string
	InventoryNumber              string
	IPAddress                    net.IP
	NextServer                   net.IP
	Options                      map[string]string
	SupportBootingFromLegacyBIOS bool
	CloseTime                    time.Time
	Status                       string
	ErrorCode                    baldrerrors.ErrorCode
	Message                      string
}

type IPXEData struct {
	UUID         string
	UEFIMode     string
	MACAddresses []net.HardwareAddr
	IPAddress    net.IP
	SerialNumber string
	Manufacturer string
	Product      string
	NextServer   net.IP
	IPXEFileName string
}

type OFRData struct {
	STTask          string
	STTaskStatus    string
	OFRType         string
	ComputerName    string
	UserName        string
	OS              string
	OSVersionString string
	OSEdition       string
	OSBuild         string
	OSUpdated       string
	ProductKey      string
	Locale          string
	Domain          bool
	Office          bool
}

type HWData struct {
	InventoryNumber string
	FQDN            string
	HWStatus        string
}

type LocationData struct {
	Country            string
	City               string
	Office             string
	Server             string
	TimezoneMS         string
	TimezoneUnix       string
	DefaultLangProfile string
	DeploymentShare    string
	SystemLocale       string
	UserLocale         string
	KeyboardLocale     string
	UILanguage         string
	DILanguage         string
	DICountry          string
	DILocale           string
	DISupportedLocales string
	DIKeymap           string
	DIToggle           string
}

type DeployXML map[string]string

type DeploymentShare struct {
	Hostname  string
	SharePath string
	Username  string
	Domain    string
	Password  string
}

type OperationSystem struct {
	Name       string
	Edition    string
	Build      string
	Updated    string
	Version    string
	ProductKey string
}

func (x DeployXML) MarshalXML(e *xml.Encoder, start xml.StartElement) error {
	tokens := []xml.Token{start}

	for key, value := range x {
		t := xml.StartElement{Name: xml.Name{Space: "", Local: key}}
		tokens = append(tokens, t, xml.CharData(value), xml.EndElement{Name: t.Name})
	}

	tokens = append(tokens, xml.EndElement{Name: start.Name})

	for _, t := range tokens {
		err := e.EncodeToken(t)
		if err != nil {
			return err
		}
	}

	// flush to ensure tokens are written
	return e.Flush()
}

func (db *DB) AddNewEntry(dep *Deploy) error {
	node := db.Primary()
	if node == nil {
		return fmt.Errorf("AddNewEntry(): master node unavailable")
	}
	var macAddresses []string
	for _, mac := range dep.MACAddresses {
		macAddresses = append(macAddresses, mac.String())
	}
	if len(macAddresses) == 0 {
		macAddresses = append(macAddresses, UndefinedMACAddress)
	}

	deployOptions, err := json.Marshal(dep.Options)
	if err != nil {
		return fmt.Errorf("AddNewEntry(): json marshaling error: %w", err)
	}

	_, err = node.DB().Exec(
		"INSERT INTO ops.deploy ("+
			"id, creation_time, mac_addresses, ip_address, serial_number, next_server, inventory_number, deploy_options,"+
			"close_time, status, error_code, message) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12)",
		dep.ID, dep.CreationTime, fmt.Sprintf("{\"%s\"}", strings.Join(macAddresses, "\", \"")),
		dep.IPAddress.String(), dep.SerialNumber, dep.NextServer.String(), dep.InventoryNumber, string(deployOptions),
		dep.CloseTime, dep.Status, dep.ErrorCode, dep.Message)
	if err != nil {
		return fmt.Errorf("AddNewEntry(): exec error: %w", err)
	}

	return nil
}

func (db *DB) DefineLocation(dep *Deploy) (int, error) {
	var numberOfResults int
	node := db.StandbyPreferred()
	if node == nil {
		return numberOfResults, fmt.Errorf("DefineLocation(): db node unavailable")
	}

	rows, err := node.DB().Query("SELECT cn.country_name, ct.city_name, o.office_name, lp.profile_name, ct.timezone_ms, "+
		"ct.timezone_unix, lp.ms_system_locale, lp.ms_user_locale, lp.ms_keyboard_locale, lp.ms_ui_language, "+
		"lp.di_language, lp.di_country, lp.di_locale, lp.di_supported_locales, lp.di_keymap, lp.di_toggle "+
		"FROM deploy.networks AS n "+
		"INNER JOIN deploy.offices AS o ON n.office = o.id "+
		"INNER JOIN deploy.cities AS ct ON o.city = ct.id "+
		"INNER JOIN deploy.countries AS cn ON ct.country = cn.id "+
		"INNER JOIN deploy.locale_profiles AS lp ON cn.default_locale = lp.id "+
		"WHERE $1 << n.net", dep.IPAddress.String())

	if err != nil {
		return numberOfResults, fmt.Errorf("DefineLocation(): query error: %w", err)
	}

	defer func(rows *sql.Rows, dep *Deploy) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("DefineLocation(): %s close rows: %v", dep.ID, err)
		}
	}(rows, dep)

	location := LocationData{}
	for rows.Next() {
		err = rows.Scan(&location.Country, &location.City, &location.Office, &location.DefaultLangProfile,
			&location.TimezoneMS, &location.TimezoneUnix, &location.SystemLocale, &location.UserLocale,
			&location.KeyboardLocale, &location.UILanguage, &location.DILanguage, &location.DICountry,
			&location.DILocale, &location.DISupportedLocales, &location.DIKeymap, &location.DIToggle)
		if err != nil {
			return numberOfResults, fmt.Errorf("DefineLocation(): scan error: %w", err)
		}
		numberOfResults++
	}

	dep.Options[OptionCountry] = location.Country
	dep.Options[OptionCity] = location.City
	dep.Options[OptionOffice] = location.Office
	dep.Options[OptionDefaultLanguageProfile] = location.DefaultLangProfile
	dep.Options[OptionMSTimeZone] = location.TimezoneMS
	dep.Options[OptionUnixTimeZone] = location.TimezoneUnix
	dep.Options[OptionMSSystemLocale] = location.SystemLocale
	dep.Options[OptionMSUserLocale] = location.UserLocale
	dep.Options[OptionMSKeyboardLocale] = location.KeyboardLocale
	dep.Options[OptionMSUILanguage] = location.UILanguage
	dep.Options[OptionDILanguage] = location.DILanguage
	dep.Options[OptionDICountry] = location.DICountry
	dep.Options[OptionDILocale] = location.DILocale
	dep.Options[OptionDISupportedLocales] = location.DISupportedLocales
	dep.Options[OptionDIKeymap] = location.DIKeymap
	dep.Options[OptionDIToggle] = location.DIToggle

	return numberOfResults, nil
}

func (db *DB) CloseWithNewDeploy(dep *Deploy) error {
	node := db.Primary()
	if node == nil {
		return fmt.Errorf("CloseWithNewDeploy(): master node unavailable")
	}

	_, err := node.DB().Exec(
		"UPDATE ops.deploy SET status = $1, error_code = $2, message = $3, close_time = now() "+
			"WHERE status = $4 AND inventory_number = $5",
		DeployStatusClosed, baldrerrors.CodeClosedByNewDeploy, "Task closed with new deploy", DeployStatusOpen,
		dep.InventoryNumber)
	if err != nil {
		return fmt.Errorf("CloseWithNewDeploy(): exec error: %w", err)
	}

	return nil
}

func (db *DB) CloseByTimeout() error {
	node := db.Primary()
	if node == nil {
		return fmt.Errorf("CloseByTimeout(): master node unavailable")
	}

	_, err := node.DB().Exec(
		"UPDATE ops.deploy SET status = $1, error_code = $2, message = $3, close_time = now() "+
			"WHERE status = $4 AND creation_time + '6 hours' < now()",
		DeployStatusClosed, baldrerrors.CodeClosedByTimeout, baldrerrors.CodeClosedByTimeout.String(), DeployStatusOpen)
	if err != nil {
		return fmt.Errorf("CloseByTimeout(): exec error: %w", err)
	}

	return nil
}

func (db *DB) CloseTask(id string, errorCode baldrerrors.ErrorCode, message string) error {
	node := db.Primary()
	if node == nil {
		return fmt.Errorf("CloseTask(): master node unavailable")
	}

	_, err := node.DB().Exec(
		"UPDATE ops.deploy SET status = $1, error_code = $2, message = $3, close_time = now() WHERE id = $4 AND status = $5",
		DeployStatusClosed, errorCode, message, id, DeployStatusOpen)
	if err != nil {
		return fmt.Errorf("CloseTask(): exec error: %w", err)
	}

	return nil
}

func (db *DB) SelectServer(dep *Deploy) error {
	var servers [][2]string
	node := db.StandbyPreferred()
	if node == nil {
		return fmt.Errorf("SelectServer(): db node unavailable")
	}
	rows, err := node.DB().Query("SELECT s.fqdn, s.share_path FROM deploy.servers AS s "+
		"INNER JOIN deploy.offices_servers AS os ON s.id = os.server "+
		"INNER JOIN deploy.offices AS o ON o.id = os.office "+
		"WHERE s.is_enabled AND o.office_name = $1", dep.Options[OptionOffice])

	if err != nil {
		return fmt.Errorf("SelectServer(): query error: %w", err)
	}

	defer func(rows *sql.Rows, dep *Deploy) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("SelectServer(): %s close rows: %v", dep.ID, err)
		}
	}(rows, dep)

	for rows.Next() {
		var server [2]string
		err = rows.Scan(&server[0], &server[1])
		if err != nil {
			return fmt.Errorf("SelectServer(): scan error: %w", err)
		}
		servers = append(servers, server)
	}

	if len(servers) == 0 {
		dep.Options[OptionDistributionPoint] = NoDeployServer
		dep.Options[OptionDeploymentShare] = NoDeploymentShare
	} else if len(servers) == 1 {
		dep.Options[OptionDistributionPoint] = servers[0][0]
		dep.Options[OptionDeploymentShare] = servers[0][1]
	} else {
		rand.Seed(time.Now().UnixNano())
		index := rand.Intn(len(servers))
		dep.Options[OptionDistributionPoint] = servers[index][0]
		dep.Options[OptionDeploymentShare] = servers[index][1]
	}

	return nil
}

func (db *DB) SupportedOS() ([]string, error) {
	var supOs []string
	node := db.StandbyPreferred()
	if node == nil {
		return supOs, fmt.Errorf("SupportedOS(): db node unavailable")
	}
	rows, err := node.DB().Query("SELECT os_name FROM deploy.supported_os")
	if err != nil {
		return supOs, fmt.Errorf("SupportedOS(): query error: %w", err)
	}

	defer func(rows *sql.Rows) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("SupportedOS(): close rows: %v", err)
		}
	}(rows)

	for rows.Next() {
		var os string
		err = rows.Scan(&os)
		if err != nil {
			return supOs, fmt.Errorf("SupportedOS(): scan error: %w", err)
		}
		supOs = append(supOs, os)
	}

	return supOs, nil
}

func (db *DB) OSDefaultBuild(dep *Deploy) ([]OperationSystem, error) {
	var operationSystems []OperationSystem
	node := db.StandbyPreferred()
	if node == nil {
		return operationSystems, fmt.Errorf("OSDefaultBuild(): db node unavailable")
	}

	rows, err := node.DB().Query("SELECT os.os_name, v.os_edition, v.os_build , v.os_updated, v.os_version, v.product_key "+
		"FROM deploy.os_versions AS v INNER JOIN deploy.supported_os AS os ON v.os = os.id "+
		"WHERE v.is_default AND os.os_name = $1", dep.Options[OptionOS])
	if err != nil {
		return operationSystems, fmt.Errorf("OSDefaultBuild(): query error: %w", err)
	}

	defer func(rows *sql.Rows, dep *Deploy) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("OSDefaultBuild(): %s close rows: %v", dep.ID, err)
		}
	}(rows, dep)

	for rows.Next() {
		var os OperationSystem
		err = rows.Scan(&os.Name, &os.Edition, &os.Build, &os.Updated, &os.Version, &os.ProductKey)
		if err != nil {
			return operationSystems, fmt.Errorf("OSDefaultBuild(): scan error: %w", err)
		}
		operationSystems = append(operationSystems, os)
	}

	return operationSystems, nil
}

func (db *DB) OSBuild(dep *Deploy) ([]OperationSystem, error) {
	var operationSystems []OperationSystem
	node := db.StandbyPreferred()
	if node == nil {
		return operationSystems, fmt.Errorf("OSBuild(): db node unavailable")
	}

	rows, err := node.DB().Query("SELECT os.os_name, v.os_edition, v.os_build , v.os_updated, v.os_version, v.product_key "+
		"FROM deploy.os_versions AS v INNER JOIN deploy.supported_os AS os ON v.os = os.id "+
		"WHERE os.os_name = $1 AND v.os_edition = $2 AND v.os_build = $3 AND v.os_updated = $4",
		dep.Options[OptionOS], dep.Options[OptionOSEdition], dep.Options[OptionOSBuild], dep.Options[OptionOSUpdated])
	if err != nil {
		return operationSystems, fmt.Errorf("OSBuild(): query error: %w", err)
	}

	defer func(rows *sql.Rows, dep *Deploy) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("OSBuild(): %s close rows: %v", dep.ID, err)
		}
	}(rows, dep)

	for rows.Next() {
		var os OperationSystem
		err = rows.Scan(&os.Name, &os.Edition, &os.Build, &os.Updated, &os.Version, &os.ProductKey)
		if err != nil {
			return operationSystems, fmt.Errorf("OSBuild(): scan error: %w", err)
		}
		operationSystems = append(operationSystems, os)
	}

	return operationSystems, err
}

func (db *DB) DriversProfile(dep *Deploy) ([]string, error) {
	var profiles []string
	node := db.StandbyPreferred()
	if node == nil {
		return profiles, fmt.Errorf("DriversProfile(): db node unavailable")
	}
	rows, err := node.DB().Query("SELECT dp.profile_name FROM deploy.drivers_profiles AS dp "+
		"INNER JOIN deploy.supported_os AS sos ON dp.os = sos.id "+
		"INNER JOIN deploy.os_versions AS osv ON dp.os_version = osv.id "+
		"INNER JOIN deploy.supported_products AS sp ON dp.product_name = sp.id "+
		"WHERE LOWER(sp.manufacturer) = $1 AND LOWER(sp.product_name) = $2 "+
		"AND sos.os_name = $3 AND osv.os_version = $4",
		dep.Options[OptionHWManufacturer], dep.Options[OptionHWModel], dep.Options[OptionOS], dep.Options[OptionOSBuild])

	if err != nil {
		return profiles, fmt.Errorf("DriversProfile(): query error: %w", err)
	}

	defer func(rows *sql.Rows, dep *Deploy) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("DriversProfile(): %s close rows: %v", dep.ID, err)
		}
	}(rows, dep)

	for rows.Next() {
		var profile string
		err = rows.Scan(&profile)
		if err != nil {
			return profiles, fmt.Errorf("DriversProfile(): scan error: %w", err)
		}
		profiles = append(profiles, profile)
	}

	if len(profiles) == 0 {
		rows, err := node.DB().Query("SELECT dp.profile_name FROM deploy.drivers_profiles AS dp "+
			"INNER JOIN deploy.supported_os AS sos ON dp.os = sos.id "+
			"INNER JOIN deploy.supported_products AS sp ON dp.product_name = sp.id "+
			"WHERE LOWER(sp.manufacturer) = $1 AND LOWER(sp.product_name) = $2 AND sos.os_name = $3",
			dep.Options[OptionHWManufacturer], dep.Options[OptionHWModel], dep.Options[OptionOS])

		if err != nil {
			return profiles, fmt.Errorf("DriversProfile(): query error: %w", err)
		}

		defer func(rows *sql.Rows, dep *Deploy) {
			err := rows.Close()
			if err != nil {
				fmt.Printf("DriversProfile(): %s close rows: %v", dep.ID, err)
			}
		}(rows, dep)

		for rows.Next() {
			var profile string
			err = rows.Scan(&profile)
			if err != nil {
				return profiles, fmt.Errorf("DriversProfile(): scan error: %w", err)
			}
			profiles = append(profiles, profile)
		}
	}

	return profiles, nil
}

func (db *DB) SupportBootingFromLegacyBIOS(dep *Deploy) ([]bool, error) {
	var checks []bool
	node := db.StandbyPreferred()
	if node == nil {
		return checks, fmt.Errorf("SupportBootingFromLegacyBIOS(): db node unavailable")
	}
	rows, err := node.DB().Query("SELECT sp.support_booting_from_legacy_bios FROM deploy.supported_products AS sp "+
		"WHERE LOWER(sp.manufacturer) = $1 AND LOWER(sp.product_name) = $2", dep.Options[OptionHWManufacturer], dep.Options[OptionHWModel])

	if err != nil {
		return checks, fmt.Errorf("SupportBootingFromLegacyBIOS(): query error: %w", err)
	}

	defer func(rows *sql.Rows, dep *Deploy) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("SupportBootingFromLegacyBIOS(): %s close rows: %v", dep.ID, err)
		}
	}(rows, dep)

	for rows.Next() {
		var check bool
		err = rows.Scan(&check)
		if err != nil {
			return checks, fmt.Errorf("SupportBootingFromLegacyBIOS(): scan error: %w", err)
		}
		checks = append(checks, check)
	}

	return checks, nil
}

func (db *DB) OU(dep *Deploy) ([]string, error) {
	var paths []string
	node := db.StandbyPreferred()
	if node == nil {
		return paths, fmt.Errorf("OU(): db node unavailable")
	}

	rows, err := node.DB().Query("SELECT ou.ou_path FROM deploy.os_ou_countries AS ooc "+
		"INNER JOIN deploy.ou AS ou ON ooc.ou_path = ou.id "+
		"INNER JOIN deploy.supported_os AS os ON ooc.os_name = os.id "+
		"INNER JOIN deploy.countries AS c ON ooc.country_name = c.id "+
		"WHERE os.os_name = $1 AND c.country_name = $2", dep.Options[OptionOS], dep.Options[OptionCountry])
	if err != nil {
		return paths, fmt.Errorf("OU(): query error: %w", err)
	}

	defer func(rows *sql.Rows, dep *Deploy) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("OU(): %s close rows: %v", dep.ID, err)
		}
	}(rows, dep)

	for rows.Next() {
		var path string
		err = rows.Scan(&path)
		if err != nil {
			return paths, fmt.Errorf("OU(): scan error: %w", err)
		}
		paths = append(paths, path)
	}

	return paths, err
}

func (db *DB) DefaultOU() ([]string, error) {
	var paths []string
	node := db.StandbyPreferred()
	if node == nil {
		return paths, fmt.Errorf("DefaultOU(): db node unavailable")
	}

	rows, err := node.DB().Query("SELECT ou.ou_path FROM deploy.ou AS ou WHERE ou.is_default")
	if err != nil {
		return paths, fmt.Errorf("DefaultOU(): query error: %w", err)
	}

	defer func(rows *sql.Rows) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("DefaultOU(): close rows: %v", err)
		}
	}(rows)

	for rows.Next() {
		var path string
		err = rows.Scan(&path)
		if err != nil {
			return paths, fmt.Errorf("DefaultOU(): scan error: %w", err)
		}
		paths = append(paths, path)
	}

	return paths, nil
}

func (db *DB) DeploymentInfo(id string) (*Deploy, error) {
	config := &Deploy{}
	var macAddresses string
	var ipAddress string
	var nextServer string
	var deployOptions []byte
	node := db.StandbyPreferred()
	if node == nil {
		return config, fmt.Errorf("DeploymentInfo(): db node unavailable")
	}

	row := node.DB().QueryRow("SELECT id, creation_time, mac_addresses, ip_address, serial_number, next_server, "+
		"inventory_number, deploy_options, close_time, status, error_code, message FROM ops.deploy WHERE id = $1", id)

	err := row.Scan(&config.ID, &config.CreationTime, &macAddresses, &ipAddress, &config.SerialNumber, &nextServer,
		&config.InventoryNumber, &deployOptions, &config.CloseTime, &config.Status, &config.ErrorCode, &config.Message)
	if err != nil && err != sql.ErrNoRows {
		return config, fmt.Errorf("DeploymentInfo(): scan error: %w", err)
	}

	if err == sql.ErrNoRows {
		return config, nil
	}

	macAddresses = strings.Trim(macAddresses, "{}")
	for _, mac := range strings.Split(macAddresses, ",") {
		hwAddr, err := net.ParseMAC(mac)
		if err != nil {
			return config, fmt.Errorf("DeploymentInfo(): parse MAC address: %w", err)
		}
		config.MACAddresses = append(config.MACAddresses, hwAddr)
	}
	config.NextServer = net.ParseIP(nextServer)
	config.IPAddress = net.ParseIP(ipAddress)
	err = json.Unmarshal(deployOptions, &config.Options)
	if err != nil {
		return config, fmt.Errorf("DeploymentInfo(): parse Options error: %w", err)
	}

	return config, nil
}

func (db *DB) DeploymentData(id string) ([]DeployXML, error) {
	var configs []DeployXML
	node := db.StandbyPreferred()
	if node == nil {
		return configs, fmt.Errorf("DeploymentData(): db node unavailable")
	}
	rows, err := node.DB().Query("SELECT deploy_options #>> '{}' FROM ops.deploy WHERE id = $1 AND status = $2", id, DeployStatusOpen)

	if err != nil {
		return configs, fmt.Errorf("DeploymentData(): query error: %w", err)
	}

	defer func(rows *sql.Rows) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("DeploymentData(): close rows: %v", err)
		}
	}(rows)

	for rows.Next() {
		var buffer []byte
		config := make(map[string]string, 50)
		err = rows.Scan(&buffer)
		if err != nil {
			return configs, fmt.Errorf("DeploymentData(): scan error: %w", err)
		}
		fmt.Printf("%s\n", buffer)
		err = json.Unmarshal(buffer, &config)
		if err != nil {
			return configs, fmt.Errorf("DeploymentData(): parse options error: %w", err)
		}
		configs = append(configs, config)
	}

	return configs, nil
}

func (db *DB) DeploymentShare(id string) ([]DeploymentShare, error) {
	var shares []DeploymentShare
	node := db.StandbyPreferred()
	if node == nil {
		return shares, fmt.Errorf("DeploymentShare(): db node unavailable")
	}

	rows, err := node.DB().Query(
		fmt.Sprintf("SELECT d.deploy_options ->> '%s', d.deploy_options ->> '%s' FROM ops.deploy AS d ",
			OptionDistributionPoint, OptionDeploymentShare)+
			"WHERE d.id = $1 AND d.status = $2", id, DeployStatusOpen)
	if err != nil {
		return shares, fmt.Errorf("DeploymentShare(): query error: %w", err)
	}

	defer func(rows *sql.Rows) {
		err := rows.Close()
		if err != nil {
			fmt.Printf("DeploymentShare(): close rows: %v", err)
		}
	}(rows)

	for rows.Next() {
		var share DeploymentShare
		err = rows.Scan(&share.Hostname, &share.SharePath)
		if err != nil {
			return shares, fmt.Errorf("DeploymentShare(): scan error: %w", err)
		}
		shares = append(shares, share)
	}

	fmt.Printf("%+v\n", shares)

	return shares, nil
}

func (db *DB) DeploymentTemplate(name string) (map[string]string, error) {
	deploymentTemplate := make(map[string]string, 10)
	node := db.StandbyPreferred()
	if node == nil {
		return deploymentTemplate, fmt.Errorf("DeploymentTemplate(): db node unavailable")
	}

	var templateData []byte
	err := node.DB().QueryRow("SELECT t.template_data FROM deploy.templates AS t WHERE t.template_name = $1", name).Scan(&templateData)
	if err != nil {
		return deploymentTemplate, fmt.Errorf("DeploymentTemplate(): query error: %w", err)
	}

	err = json.Unmarshal(templateData, &deploymentTemplate)
	if err != nil {
		return deploymentTemplate, fmt.Errorf("DeploymentTemplate(): json unmarshal error: %w", err)
	}

	return deploymentTemplate, nil
}
