package main

import (
	"encoding/json"
	"fmt"
	"log"
	"net"
	"net/http"
	"net/url"
	"strings"
	"sync"
	"time"

	"github.com/go-resty/resty/v2"
)

type BotCacheRequestType uint

type BotCacheResponse struct {
	found   bool
	BotData BotOS
	err     error
}

type BotCacheRequest struct {
	Type         BotCacheRequestType
	SerialNumber string
	MACAddresses []net.HardwareAddr
	BotData      BotOS
	Out          chan BotCacheResponse
}

const (
	BotCacheGetMAC BotCacheRequestType = iota
	BotCacheGetSN
	BotCacheSetSN
	BotCacheSetMAC
)

type botReqType int

func (rt botReqType) String() string {
	switch rt {
	case botFindBySN:
		return "botFindBySN"
	case botFindByMAC:
		return "botFindByMAC"
	default:
		return "Unknown"
	}
}

const (
	botFindBySN botReqType = iota
	botFindBySNWithS
	botFindByMAC
)

type botConfig struct {
	workers int
	In      chan Request
	Cache   chan BotCacheRequest
}

type botReqData struct {
	HostID       string
	SerialNumber string
	MACAddresses []net.HardwareAddr
}

type BotOSType string

func (osType BotOSType) String() string {
	return string(osType)
}

const (
	TypeDesktop  BotOSType = "DESKTOPS"
	TypeNotebook BotOSType = "NOTEBOOKS"
	TypeServers  BotOSType = "SERVERS"
	TypeNodes    BotOSType = "NODES"
)

type BotOS struct {
	InventoryNumber string    `json:"instance_number"`
	Type            BotOSType `json:"item_segment3"`
	FQDN            string    `json:"XXCSI_FQDN"`
	Status          string    `json:"status_name"`
	Owner           string    `json:"EMPLOYEE_OWNED"`
}

type BotOSResp struct {
	Result  int     `json:"res"`
	Message string  `json:"msg"`
	OS      []BotOS `json:"os"`
}

func BotCache(ch chan BotCacheRequest) {
	MACCache := make(map[string]BotOS)
	SNCache := make(map[string]BotOS)
	var resp BotCacheResponse
	for {
		req := <-ch
		switch req.Type {
		case BotCacheGetSN:
			resp.BotData, resp.found = getBotDataFromSNCache(req.SerialNumber, SNCache)
			resp.err = nil
		case BotCacheGetMAC:
			data := getBotDataFromMACCache(req.MACAddresses, MACCache)
			if len(data) == 0 {
				resp.BotData = BotOS{}
				resp.found = false
				resp.err = nil
			} else if len(data) == 1 {
				resp.BotData = data[0]
				resp.found = true
				resp.err = nil
			} else {
				var inventoryNumbers []string
				for _, bot := range data {
					inventoryNumbers = append(inventoryNumbers, bot.InventoryNumber)
				}
				var macAddresses []string
				for _, mac := range req.MACAddresses {
					macAddresses = append(macAddresses, mac.String())
				}
				resp.BotData = BotOS{}
				resp.found = false
				resp.err = fmt.Errorf("BotCache(): more than one inventory number found %q for MAC addresses: %s",
					strings.Join(inventoryNumbers, ", "), strings.Join(macAddresses, ", "))
			}
		case BotCacheSetMAC:
			if len(req.MACAddresses) != 1 || req.BotData.InventoryNumber == "" {
				resp.found = false
				resp.BotData = BotOS{}
				resp.err = fmt.Errorf("BotCache(): bad request, MAC addresses %v, Bot data: %v", req.MACAddresses, req.BotData)
			} else {
				MACCache[req.MACAddresses[0].String()] = req.BotData
				resp.found = true
				resp.err = nil
			}
		case BotCacheSetSN:
			if req.SerialNumber == "" || req.BotData.InventoryNumber == "" {
				resp.found = false
				resp.BotData = BotOS{}
				resp.err = fmt.Errorf("BotCache(): bad request, Serial number %q, Bot data: %v", req.SerialNumber, req.BotData)
			} else {
				SNCache[req.SerialNumber] = req.BotData
				resp.found = true
				resp.err = nil
			}
		default:
			resp.found = false
			resp.BotData = BotOS{}
			resp.err = fmt.Errorf("BotCache(): unknown request type %d", req.Type)
		}
		req.Out <- resp
	}
}

func getBotDataFromSNCache(SerialNumber string, SNCache map[string]BotOS) (data BotOS, ok bool) {
	data, ok = SNCache[SerialNumber]

	return
}

func getBotDataFromMACCache(MACAddresses []net.HardwareAddr, MACCache map[string]BotOS) (data []BotOS) {
	var bot BotOS
	store := make(map[string]BotOS)
	ok := false
	for _, mac := range MACAddresses {
		bot, ok = MACCache[mac.String()]
		if ok {
			store[bot.InventoryNumber] = bot
		}
	}

	for _, value := range store {
		data = append(data, value)
	}

	return
}

func botWorker(config botConfig) {
	//log.Printf("[Debug] run botWorker()")
	var req Request
	var resp Response
	for {
		req = <-config.In
		switch req.Type {
		case BotFindInvNum:
			switch hwData := req.Data.(type) {
			case botReqData:
				botData, err := botFindInventoryNumber(hwData, config.Cache)
				if err != nil {
					resp.Data = nil
					resp.err = fmt.Errorf("botWorker(): %s:%w", hwData.HostID, err)
				} else {
					resp.Data = botData
					resp.err = nil
				}
			default:
				resp.Data = nil
				resp.err = fmt.Errorf("botWorker(): bad request BotFindInvNum, get data type %T, expected botReqData", hwData)
			}
			req.Out <- resp
			req.wg.Done()
		default:
			resp.Data = nil
			resp.err = fmt.Errorf("botWorker(): bad request type %d", req.Type)
			req.Out <- resp
			req.wg.Done()
		}
	}
}

func botFindInventoryNumber(hwData botReqData, cache chan BotCacheRequest) (botOS BotOS, err error) {
	//log.Printf("[Debug] run botFindInventoryNumber() for %s", hwData.HostID)
	//defer log.Printf("[Debug] exit botFindInventoryNumber() for %s", hwData.HostID)
	req := BotCacheRequest{
		Type:         BotCacheGetSN,
		SerialNumber: hwData.SerialNumber,
		MACAddresses: hwData.MACAddresses,
		BotData:      BotOS{},
		Out:          make(chan BotCacheResponse),
	}
	cache <- req
	cacheResp := <-req.Out
	if cacheResp.err != nil {
		err = fmt.Errorf("botFindInventoryNumber(): find in cache by Serial number: %w", cacheResp.err)
		return
	}

	if cacheResp.found {
		botOS = cacheResp.BotData
		return
	}

	var wgSN sync.WaitGroup
	outSN := make(chan Response)
	serialNumber := strings.TrimSpace(hwData.SerialNumber)
	if serialNumber != "" {
		wgSN.Add(1)
		go findInvNum(botFindBySN, serialNumber, outSN, &wgSN, cache)
		wgSN.Add(1)
		go findInvNum(botFindBySNWithS, fmt.Sprintf("S%s", serialNumber), outSN, &wgSN, cache)
	}
	go waitclose(outSN, &wgSN)

	for resp := range outSN {
		if resp.err != nil {
			log.Printf("[Error] botFindInventoryNumber(): %s\n", resp.err.Error())
			continue
		}

		switch OS := resp.Data.(type) {
		case BotOS:
			if botOS.InventoryNumber == "" {
				botOS = OS
			} else if botOS.InventoryNumber != OS.InventoryNumber {
				err = fmt.Errorf("botFindInventoryNumber(): more than one inventory number (%q, %q) for %v",
					botOS.InventoryNumber, OS.InventoryNumber, hwData)
			}
		case nil:
			continue
		default:
			err = fmt.Errorf("botFindInventoryNumber(): %T: %v", botOS, resp)
		}
	}

	if err != nil || botOS.InventoryNumber != "" {
		return
	}

	var wgMAC sync.WaitGroup
	outMAC := make(chan Response)
	for _, mac := range hwData.MACAddresses {
		wgMAC.Add(1)
		go findInvNum(botFindByMAC, mac.String(), outMAC, &wgMAC, cache)
	}
	go waitclose(outMAC, &wgMAC)

	for resp := range outMAC {
		if resp.err != nil {
			log.Printf("[Error] botFindInventoryNumber(): %s\n", resp.err.Error())
			continue
		}

		switch OS := resp.Data.(type) {
		case BotOS:
			if botOS.InventoryNumber == "" {
				botOS = OS
			} else if botOS.InventoryNumber != OS.InventoryNumber {
				err = fmt.Errorf("botFindInventoryNumber(): more than one inventory number (%q, %q) for %v",
					botOS.InventoryNumber, OS.InventoryNumber, hwData)
			}
		case nil:
			continue
		default:
			err = fmt.Errorf("botFindInventoryNumber(): %T: %v", botOS, resp)
		}
	}

	return
}

func findInvNum(reqType botReqType, arg string, out chan Response, wg *sync.WaitGroup, cache chan BotCacheRequest) {
	//log.Printf("[Debug] run findInvNum() for %s by %s", arg, reqType)
	//defer log.Printf("[Debug] exit findInvNum() for %s by %s", arg, reqType)
	defer wg.Done()

	client := resty.New()
	client.SetRetryCount(5)
	client.SetRetryWaitTime(1 * time.Second)
	client.SetRetryMaxWaitTime(2 * time.Second)
	client.AddRetryCondition(func(resp *resty.Response, err error) bool {
		if resp.StatusCode() >= http.StatusBadRequest || err != nil {
			return true
		}
		return false
	})

	req := client.R()
	req.Method = http.MethodGet
	req.URL = "https://bot.yandex-team.ru/api/osinfo.php"
	req.SetQueryParam("format", "json")
	req.SetQueryParam("output", "instance_number|item_segment3|XXCSI_FQDN|status_name|EMPLOYEE_OWNED")
	if reqType == botFindBySN || reqType == botFindBySNWithS {
		req.SetQueryParam("sn", url.QueryEscape(arg))
	} else if reqType == botFindByMAC {
		req.SetQueryParam("mac", url.QueryEscape(strings.ToUpper(strings.ReplaceAll(arg, ":", ""))))
	} else {
		out <- Response{
			Data: nil,
			err:  fmt.Errorf("findInvNum(): invalid request type %d", reqType),
		}
		return
	}

	resp, err := req.Send()
	if err != nil {
		out <- Response{
			Data: nil,
			err:  fmt.Errorf("findInvNum(): request %s, arg %q:%w", reqType, arg, err),
		}
		return
	}

	var botResp BotOSResp
	err = json.Unmarshal(resp.Body(), &botResp)
	if err != nil {
		out <- Response{
			Data: nil,
			err:  fmt.Errorf("findInvNum(): request %s, arg %q: unmarshal %q:%w", reqType, arg, resp.Body(), err),
		}
		return
	}

	if botResp.Result != 1 {
		out <- Response{
			Data: nil,
			err:  nil,
		}
		return
	}

	var validInstances []BotOS
	for _, inst := range botResp.OS {
		if inst.Type == TypeDesktop || inst.Type == TypeNotebook ||
			inst.Type == TypeServers || inst.Type == TypeNodes {
			validInstances = append(validInstances, inst)
		}
	}

	if len(validInstances) == 0 {
		out <- Response{
			Data: nil,
			err:  nil,
		}
		return
	} else if len(validInstances) > 1 {
		var invNumbers string
		for _, inst := range validInstances {
			invNumbers += fmt.Sprintf(", %s", inst.InventoryNumber)
		}
		invNumbers = strings.TrimLeft(invNumbers, ", ")
		out <- Response{
			Data: nil,
			err:  fmt.Errorf("findInvNum(): request %s, arg %q: to many instances: %s", reqType, arg, invNumbers),
		}
		return
	}

	cacheReq := BotCacheRequest{
		Type:         0,
		SerialNumber: "",
		MACAddresses: nil,
		BotData:      validInstances[0],
		Out:          make(chan BotCacheResponse),
	}
	if reqType == botFindByMAC {
		cacheReq.Type = BotCacheSetMAC
		mac, err := net.ParseMAC(arg)
		if err != nil {
			out <- Response{
				Data: nil,
				err:  fmt.Errorf("findInvNum(): store result into cache: parse MAC address %q: %w", arg, err),
			}
			return
		}
		cacheReq.MACAddresses = []net.HardwareAddr{mac}
	} else if reqType == botFindBySN {
		cacheReq.Type = BotCacheSetSN
		cacheReq.SerialNumber = arg
	} else if reqType == botFindBySNWithS {
		cacheReq.Type = BotCacheSetSN
		cacheReq.SerialNumber = arg[1:]
	}
	cache <- cacheReq
	cacheResp := <-cacheReq.Out
	if cacheResp.err != nil {
		out <- Response{
			Data: nil,
			err:  fmt.Errorf("findInvNum(): store result into cache: %w", err),
		}
	}

	out <- Response{
		Data: validInstances[0],
		err:  nil,
	}
}
