package main

import (
	"fmt"
	"log"
	"strings"
	"sync"

	"a.yandex-team.ru/helpdesk/infra/ksc/internal/queue"
	"a.yandex-team.ru/helpdesk/infra/ksc/pkg/ksc"
)

const broTag = "task=INFRAWIN-576"

func (e *env) MarkBrowserDevelopers(hosts []Host) (err error) {
	log.Printf("[Debug] run MarkBrowserDevelopers()")
	defer log.Printf("[Debug] exit MarkBrowserDevelopers()")
	developers, err := e.GetBrowserDevelopersFromStaff()
	if err != nil {
		err = fmt.Errorf("env::MarkBrowserDevelopers():%w", err)
		return
	}
	log.Printf("[Debug] developers count %d", len(developers))

	var markedHosts []string
	markedHosts, err = e.MarkBrowserComputers(hosts, developers)
	log.Printf("[Debug] marked hosts count %d", len(markedHosts))

	computersFromADGroup, err := e.GetBrowserComputersFromAD()
	if err != nil {
		err = fmt.Errorf("env::MarkBrowserDevelopers():%w", err)
		return
	}
	log.Printf("[Debug] computers in group count %d", len(computersFromADGroup))

	err = e.AddGroupMembers(markedHosts, computersFromADGroup)
	if err != nil {
		err = fmt.Errorf("env::MarkBrowserDevelopers():%w", err)
		return
	}

	err = e.RemoveGroupMembers(markedHosts, computersFromADGroup)
	if err != nil {
		err = fmt.Errorf("env::MarkBrowserDevelopers():%w", err)
		return
	}
	return
}

func (e *env) GetBrowserDevelopersFromStaff() (developers []string, err error) {
	var groupsID []int
	groupsID, err = e.GetBrowserDevelopersGroupsFromStaff()
	if err != nil {
		err = fmt.Errorf("env::GetBrowserDevelopersFromStaff(): %w", err)
		return
	}

	developers = []string{"secondfry", "ikobylin", "agh0", "tonynasta", "pauline-sh", "matthewtff", "dmgor", "golubtsov", "artalex", "ko4evnik"}
	for _, groupID := range groupsID {
		var groupMembers []StaffGroupMember
		groupMembers, err = e.GetStaffGroupMembers(groupID)
		if err != nil {
			err = fmt.Errorf("env::GetBrowserDevelopersFromStaff(): %w", err)
			return
		}
		for _, member := range groupMembers {
			developers = append(developers, member.Person.Login)
		}
	}

	return
}

func (e *env) GetBrowserDevelopersGroupsFromStaff() (groupsID []int, err error) {
	const (
		GroupIDNetworkAndBackendService          = 5822
		GroupIDGeneralComponentsService          = 2791
		GroupIDRenderingEngineDevelopmentService = 1472
		GroupIDGroupOfSearchTechnologies         = 1740
		GroupIDUINinjas                          = 1700
		GroupIDUserInterfaceDevelopmentGroup     = 3271
	)

	worklist := new(queue.Queue)
	seen := make(map[int]bool)

	worklist.Push(GroupIDNetworkAndBackendService)
	seen[GroupIDNetworkAndBackendService] = true
	worklist.Push(GroupIDGeneralComponentsService)
	seen[GroupIDGeneralComponentsService] = true
	worklist.Push(GroupIDRenderingEngineDevelopmentService)
	seen[GroupIDRenderingEngineDevelopmentService] = true
	worklist.Push(GroupIDGroupOfSearchTechnologies)
	seen[GroupIDGroupOfSearchTechnologies] = true
	worklist.Push(GroupIDUINinjas)
	seen[GroupIDUINinjas] = true
	worklist.Push(GroupIDUserInterfaceDevelopmentGroup)
	seen[GroupIDUserInterfaceDevelopmentGroup] = true

	for {
		if id, ok := worklist.Pop(); ok {
			var list []StaffGroup
			list, err = e.GetStaffSubGroups(id)
			if err != nil {
				err = fmt.Errorf("env::GetBrowserDevelopersGroupsFromStaff(): group id %d: %w", id, err)
				return
			}
			for _, group := range list {
				if seen[group.ID] {
					continue
				}

				worklist.Push(group.ID)
				seen[group.ID] = true
			}
			continue
		}
		break
	}

	for group := range seen {
		groupsID = append(groupsID, group)
	}

	return
}

func (e *env) GetStaffSubGroups(id int) (groups []StaffGroup, err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Type: StaffGetSubgroups,
		Data: id,
		wg:   &wg,
		Out:  out,
	}
	wg.Add(1)
	e.staff.In <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::GetStaffSubGroups(): group ID %d:%w", id, resp.err)
		return
	}

	switch data := resp.Data.(type) {
	case []StaffGroup:
		groups = data
	default:
		err = fmt.Errorf("env::GetStaffSubGroups(): group ID %q: bad response, expected type %T, got %T", id, groups, data)
	}

	return
}

func (e *env) GetStaffGroupMembers(id int) (members []StaffGroupMember, err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Type: StaffGetGroupMembers,
		Data: id,
		wg:   &wg,
		Out:  out,
	}
	wg.Add(1)
	e.staff.In <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::GetStaffGroupMembers(): group ID %d:%w", id, resp.err)
		return
	}

	switch data := resp.Data.(type) {
	case []StaffGroupMember:
		members = data
	default:
		err = fmt.Errorf("env::GetStaffGroupMembers(): group ID %q: bad response, expected type %T, got %T", id, members, data)
	}

	return
}

func (e *env) MarkBrowserComputers(hosts []Host, developers []string) (markedHosts []string, err error) {
	for index := range hosts {
		browserComputer := IsBrowserComputer(&(hosts[index]), developers)
		tagExist := TagExist(broTag, hosts[index].Tags)

		if !browserComputer && !tagExist {
			continue
		} else if browserComputer && tagExist {
			markedHosts = append(markedHosts, strings.ToLower(hosts[index].DN))
		} else if browserComputer && !tagExist {
			hosts[index].Tags, err = e.SetBrowserTag(hosts[index].ID, hosts[index].KSCServer, true)
			if err != nil {
				err = fmt.Errorf("env::MarkBrowserComputers(): %w", err)
				return
			}
			markedHosts = append(markedHosts, strings.ToLower(hosts[index].DN))
		} else if !browserComputer && tagExist {
			hosts[index].Tags, err = e.SetBrowserTag(hosts[index].ID, hosts[index].KSCServer, false)
			if err != nil {
				err = fmt.Errorf("env::MarkBrowserComputers(): %w", err)
				return
			}
		}
	}

	return
}

func (e *env) SetBrowserTag(hostID string, server string, set bool) (tags []ksc.HostTag, err error) {
	itemTags := ksc.ListItemsTags{
		ID:   hostID,
		Tags: nil,
	}
	itemTags.Tags = append(itemTags.Tags, ksc.ItemTag{
		Value: broTag,
		Set:   set,
	})
	err = e.setHostTags(itemTags, server)
	if err != nil {
		err = fmt.Errorf("env::SetBrowserTag: set tag %t: %w", set, err)
		return
	}

	tags, err = e.getHostTags(hostID, server)
	if err != nil {
		err = fmt.Errorf("env::SetBrowserTag: get tag %t: %w", set, err)
		return
	}

	return
}

func TagExist(tagName string, tags []ksc.HostTag) bool {
	for _, tag := range tags {
		if tagName == tag.Name {
			return true
		}
	}
	return false
}

func IsBrowserComputer(host *Host, developers []string) (flag bool) {
	if stringInSlice(host.OEBSOwner, developers) {
		flag = true
		return
	}

	for _, user := range host.Users {
		if user.Dismissed {
			continue
		}

		if stringInSlice(user.SAMAccountName, developers) {
			flag = true
		} else {
			flag = false
			return
		}
	}

	return
}

func stringInSlice(s string, list []string) bool {
	s = strings.ToLower(s)
	for _, item := range list {
		if strings.ToLower(item) == s {
			return true
		}
	}

	return false
}

func (e *env) GetBrowserComputersFromAD() (members []string, err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Type: LDAPGetBroGroupMembers,
		Data: nil,
		wg:   &wg,
		Out:  out,
	}
	wg.Add(1)
	e.ldap.In <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::GetBrowserComputersFromAD():%w", resp.err)
		return
	}

	switch data := resp.Data.(type) {
	case []string:
		members = data
	default:
		err = fmt.Errorf("env::GetBrowserComputersFromAD(): bad response, expected type %T, got %T", members, data)
	}

	return
}

func (e *env) AddGroupMembers(browserComputers []string, existMembers []string) (err error) {
	for _, broComputer := range browserComputers {
		if stringInSlice(strings.ToLower(broComputer), existMembers) {
			continue
		}

		log.Printf("Browser computer without group: %s", broComputer)
		err = e.ModifyGroupMembers(broComputer, true)
		if err != nil {
			err = fmt.Errorf("env::AddGroupMembers(): %w", err)
			return
		}
	}

	return
}

func (e *env) RemoveGroupMembers(browserComputers []string, existMembers []string) (err error) {
	for _, groupMember := range existMembers {
		if stringInSlice(groupMember, browserComputers) {
			continue
		}

		log.Printf("Non-browser computer in group: %s", groupMember)
	}

	return
}

func (e *env) ModifyGroupMembers(dn string, add bool) (err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Data: dn,
		wg:   &wg,
		Out:  out,
	}
	if add {
		req.Type = LDAPAddComputerToGroup
	} else {
		req.Type = LDAPRemoveComputerFromGroup
	}

	wg.Add(1)
	e.ldap.In <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::ModifyGroupMembers(): DN %q: Add %t:%w", dn, add, resp.err)
		return
	}

	return
}
