package main

import (
	"context"
	"crypto/tls"
	"fmt"
	"log"
	"net"
	"net/http"
	"strings"
	"sync"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/helpdesk/infra/ksc/pkg/ksc"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type envConfig struct {
	ksc   []kscConfig
	bot   botConfig
	staff staffConfig
	ldap  ldapConfig
	yt    yt.Config
}

type env struct {
	sync.RWMutex
	ksc struct {
		hosts     []kscConfig
		ChanStore chan kscChanRequest
	}
	bot   botConfig
	staff staffConfig
	ldap  ldapConfig
	yt    struct {
		client yt.Client
		config yt.Config
	}
}

func newEnv(config envConfig) (*env, error) {
	//log.Printf("[Debug] run newEnv()")
	e := env{
		RWMutex: sync.RWMutex{},
		ksc: struct {
			hosts     []kscConfig
			ChanStore chan kscChanRequest
		}{hosts: config.ksc, ChanStore: make(chan kscChanRequest)},
		bot:   config.bot,
		staff: config.staff,
		ldap:  config.ldap,
		yt: struct {
			client yt.Client
			config yt.Config
		}{client: nil, config: config.yt},
	}

	err := e.init()
	if err != nil {
		err = fmt.Errorf("newEnv():%w", err)
	}
	//log.Printf("[Debug] exit newEnv()")
	return &e, err
}

func waitclose(ch chan Response, wg *sync.WaitGroup) {
	//log.Printf("[Debug] run waitclose():")
	wg.Wait()
	close(ch)
	//log.Printf("[Debug] exit waitclose()")
}

func (e *env) init() (err error) {
	//log.Printf("[Debug] run env::init()")
	//defer log.Printf("[Debug] exit env::init()")
	err = e.initKSC()
	if err != nil {
		err = fmt.Errorf("env::init():%w", err)
		return
	}

	err = e.initBot()
	if err != nil {
		err = fmt.Errorf("env::init():%w", err)
		return
	}

	err = e.initStaff()
	if err != nil {
		err = fmt.Errorf("env::init():%w", err)
		return
	}

	err = e.initLDAP()
	if err != nil {
		err = fmt.Errorf("env::init():%w", err)
		return
	}

	err = e.initYT()
	if err != nil {
		err = fmt.Errorf("env::init():%w", err)
		return
	}

	return
}

func (e *env) initBot() error {
	e.Lock()
	defer e.Unlock()
	e.bot.Cache = make(chan BotCacheRequest)
	go BotCache(e.bot.Cache)
	e.bot.In = make(chan Request)
	for i := 0; i < e.bot.workers; i++ {
		go botWorker(e.bot)
	}

	return nil
}

func (e *env) initStaff() error {
	e.Lock()
	defer e.Unlock()
	e.staff.cache = make(chan StaffCacheRequest)
	go StaffCache(e.staff.cache)

	e.staff.In = make(chan Request)
	for i := 0; i < e.staff.workers; i++ {
		go staffWorker(e.staff)
	}

	return nil
}

func (e *env) initLDAP() error {
	e.Lock()
	defer e.Unlock()
	e.ldap.Cache = make(chan LDAPCacheRequest)
	go LDAPCache(e.ldap.Cache)
	e.ldap.In = make(chan Request)
	for i := 0; i < e.ldap.workers; i++ {
		go ldapWorker(e.ldap)
	}

	return nil
}

func (e *env) initYT() error {
	var err error
	e.yt.client, err = ythttp.NewClient(&e.yt.config)
	if err != nil {
		err = fmt.Errorf("initYT():%w", err)
	}

	return err
}

func (e *env) initKSC() error {
	//log.Printf("[Debug] run env::initKSC()")
	//defer log.Printf("[Debug] exit env::initKSC()")
	e.Lock()
	defer e.Unlock()

	go kscChanStore(e.ksc.ChanStore)
	for _, config := range e.ksc.hosts {
		ch, err := initKSC(config)
		if err != nil {
			return fmt.Errorf("env::initKSC():%w", err)
		}

		resp := e.setKscChan(config.serverName, ch)
		if !resp.ok {
			return fmt.Errorf("initKSC(): failed to store the channel for %s", config.serverName)
		}
	}

	return nil
}

func (e *env) getKscChan(server string) kscChanResponse {
	//log.Printf("[Debug] run env::getKscChan() for %s", server)
	//defer log.Printf("[Debug] exit env::getKscChan() for %s", server)
	out := make(chan kscChanResponse)
	req := kscChanRequest{
		Type:       kscChanGet,
		ServerName: server,
		ch:         nil,
		Out:        out,
	}

	e.ksc.ChanStore <- req

	return <-out
}

func (e *env) setKscChan(server string, ch chan Request) kscChanResponse {
	//log.Printf("[Debug] run env::setKscChan() for %s", server)
	//defer log.Printf("[Debug] exit env::setKscChan() for %s", server)
	out := make(chan kscChanResponse)
	req := kscChanRequest{
		Type:       kscChanSet,
		ServerName: server,
		ch:         ch,
		Out:        out,
	}

	e.ksc.ChanStore <- req
	return <-out
}

func initKSC(config kscConfig) (chan Request, error) {
	//log.Printf("[Debug] run initKSC() for %s", config.serverName)
	//defer log.Printf("[Debug] exit initKSC() for %s", config.serverName)
	ch := make(chan Request)
	for i := 0; i < config.workers; i++ {
		client := ksc.New(config.config)
		client.SetTLSClientConfig(&tls.Config{InsecureSkipVerify: true})
		client.SetRetryCount(5)
		client.SetRetryWaitTime(1 * time.Second)
		client.SetRetryMaxWaitTime(2 * time.Second)
		client.AddRetryCondition(func(resp *resty.Response, err error) bool {
			if resp.StatusCode() >= http.StatusBadRequest || err != nil {
				return true
			}
			return false
		})
		ctx := context.Background()
		err := client.Login(ctx, ksc.BasicAuthentication, "")
		if err != nil {
			return ch, fmt.Errorf("initKSC(): instance %q:%w", client.Server, err)
		}

		go kscWorker(ctx, client, config.serverName, ch)
	}

	return ch, nil
}

func (e *env) gatherData() (hosts []Host, err error) {
	log.Printf("[Debug] Start gatherData")
	defer log.Printf("[Debug] End gatherData")
	var hostsWg sync.WaitGroup
	hostsCh := make(chan Response)

	err = e.getHosts(hostsCh, &hostsWg)
	if err != nil {
		err = fmt.Errorf("env::gatherData():%w", err)
		return
	}
	go waitclose(hostsCh, &hostsWg)

	var hostWg sync.WaitGroup
	hostCh := make(chan Response)

	go func() {
		tokens := make(chan int, 60)
		for resp := range hostsCh {
			if resp.err != nil {
				log.Printf("[Error] %s\n", resp.err.Error())
				continue
			}

			switch host := resp.Data.(type) {
			case Host:
				tokens <- 0
				hostWg.Add(1)
				go e.getHostData(host, hostCh, &hostWg, tokens)
			default:
				fmt.Printf("[Unknown] %T: %v\n", host, resp)
			}
		}

		waitclose(hostCh, &hostWg)
	}()

	for resp := range hostCh {
		if resp.err != nil {
			log.Printf("[Error] %s\n", resp.err.Error())
			continue
		}

		switch host := resp.Data.(type) {
		case Host:
			hosts = append(hosts, host)
		default:
			fmt.Printf("[Unknown] %T: %v\n", host, resp)
		}
	}

	fmt.Printf("Length: %d\n", len(hosts))

	return
}

func (e *env) getHosts(ch chan Response, wg *sync.WaitGroup) error {
	//log.Printf("[Debug] run env::getHosts()")
	//defer log.Printf("[Debug] exit env::getHosts()")
	e.RLock()
	defer e.RUnlock()
	req := Request{
		Type: KSCGetHosts,
		Data: nil,
		wg:   wg,
		Out:  ch,
	}

	for _, server := range e.ksc.hosts {
		resp := e.getKscChan(server.serverName)
		in, ok := resp.ch, resp.ok
		if !ok {
			return fmt.Errorf("env::getHosts(): channel for %q not found", server.serverName)
		}

		wg.Add(1)
		in <- req
	}

	return nil
}

func (e *env) getHostData(host Host, ch chan Response, wg *sync.WaitGroup, tokens chan int) {
	defer wg.Done()
	var resp Response
	e.hostData(&host)
	resp.Data = host
	ch <- resp
	<-tokens
}

func (e *env) hostData(host *Host) {
	var wg sync.WaitGroup
	out := make(chan Response)
	wg.Add(3)
	go e.getHWInfo(host.ID, host.KSCServer, out, &wg)
	go e.getUsers(host.ID, host.KSCServer, out, &wg)
	go e.getTags(host.ID, host.KSCServer, out, &wg)
	if host.DNSDomain == "ld.yandex.ru" {
		wg.Add(1)
		go e.getDN(host.WinHostname, out, &wg)
	}
	go waitclose(out, &wg)
	for resp := range out {
		if resp.err != nil {
			log.Printf("env::hostData():%v", resp.err)
			continue
		}

		switch data := resp.Data.(type) {
		case Hardware:
			host.Hardware = data
		case []UserInfo:
			host.Users = data
		case []ksc.HostTag:
			host.Tags = data
		case string:
			host.DN = data
		default:
			log.Printf("[Unknown] %T: %v\n", data, resp)
		}
	}
}

func (e *env) getHWInfo(hostID string, server string, ch chan Response, wg *sync.WaitGroup) {
	//log.Printf("[Debug] run getHWInfo() for %s", hostID)
	//defer log.Printf("[Debug] exit getHWInfo() for %s", hostID)

	defer wg.Done()

	var err error
	var hw Hardware
	hw.Devices, err = e.getHWInventoryFromKSC(hostID, server)
	if err != nil {
		ch <- Response{err: fmt.Errorf("env::getHWInfo():%w", err), Data: hw}
		return
	}

	hwData := botRequest(hostID, hw.Devices)

	var botOS BotOS
	botOS, err = e.getHWInventoryFromBot(hostID, hwData)
	if err != nil {
		ch <- Response{err: fmt.Errorf("env::getHWInfo():%w", err), Data: hw}
		return
	}

	hw.OEBSInventoryNumber = botOS.InventoryNumber
	hw.OEBSType = botOS.Type
	hw.OEBSStatus = botOS.Status
	hw.OEBSOwner = botOS.Owner
	hw.OEBSFQDN = botOS.FQDN

	var ownerInfo StaffUserAttr
	ownerInfo, err = e.checkHWOwner(hw.OEBSOwner)
	if err != nil {
		ch <- Response{err: fmt.Errorf("env::getHWInfo():%w", err), Data: hw}
		return
	}

	hw.OEBSOwnerIsDismissed = ownerInfo.Dismissed
	hw.OEBSOwnerDepartmentID = ownerInfo.DepartmentID
	hw.OEBSOwnerDepartmentName = ownerInfo.DepartmentName

	ch <- Response{
		Data: hw,
		err:  nil,
	}
}

func (e *env) getHWInventoryFromKSC(hostID string, server string) (devices []ksc.HwInvPCView, err error) {
	var wg sync.WaitGroup
	chKSC := make(chan Response)
	req := Request{
		Type: KSCGetHostHW,
		Data: hostID,
		wg:   &wg,
		Out:  chKSC,
	}

	respKSCCh := e.getKscChan(server)
	if !respKSCCh.ok {
		err = fmt.Errorf("env::getHWInventoryFromKSC(): the channel to %s not found, host ID %q", server, hostID)
		return
	}

	wg.Add(1)
	respKSCCh.ch <- req
	//wg.Wait()
	resp := <-chKSC
	if resp.err != nil {
		err = fmt.Errorf("env::getHWInventoryFromKSC(): host ID %q:%w", hostID, resp.err)
		return
	}

	switch dev := resp.Data.(type) {
	case []ksc.HwInvPCView:
		devices = dev
	default:
		err = fmt.Errorf("env::getHWInventoryFromKSC(): host ID %q: bad response, expected type %T, got %T", hostID, devices, dev)
	}
	return
}

func botRequest(hostID string, devices []ksc.HwInvPCView) botReqData {
	var hwData botReqData
	hwData.HostID = hostID
	for _, dev := range devices {
		if dev.DeviceType == ksc.KLDevTypeMotherBoard {
			hwData.SerialNumber = dev.BiosSerialNumber
		} else if dev.DeviceType == ksc.KLDevTypeNetworkAdapter {
			if dev.DeviceXPar1 == "" {
				continue
			}

			if !strings.HasPrefix(dev.DeviceID, "PCI\\") {
				continue
			}

			mac, err := net.ParseMAC(dev.DeviceXPar1)
			if err != nil {
				log.Printf("botRequest(): host ID %q: parse MAC address %q failed", hostID, dev.DeviceXPar1)
				continue
			} else {
				hwData.MACAddresses = append(hwData.MACAddresses, mac)
			}
		}
	}

	return hwData
}

func (e *env) getHWInventoryFromBot(hostID string, hwData botReqData) (resp BotOS, err error) {
	var wg sync.WaitGroup
	chBot := make(chan Response)
	botReq := Request{
		Type: BotFindInvNum,
		Data: hwData,
		wg:   &wg,
		Out:  chBot,
	}

	wg.Add(1)
	e.bot.In <- botReq
	//wgBot.Wait()

	botData := <-chBot
	if botData.err != nil {
		err = fmt.Errorf("env::getHWInventoryFromKSC(): host ID %q:%w", hostID, botData.err)
		return
	}

	switch botOS := botData.Data.(type) {
	case BotOS:
		resp = botOS
	default:
		err = fmt.Errorf("env::getHWInventoryFromBot(): host ID %q: bad response, expected type %T, got %T", hostID, resp, botOS)
	}
	return
}

func (e *env) checkHWOwner(owner string) (userAttr StaffUserAttr, err error) {
	//log.Printf("[Debug] run checkHWOwner(): %q", owner)
	//defer log.Printf("[Debug] exit checkHWOwner(): %q", owner)
	if owner == "" {
		return
	}

	var wg sync.WaitGroup
	out := make(chan Response)

	req := Request{
		Type: StaffGetUserInfo,
		Data: owner,
		wg:   &wg,
		Out:  out,
	}
	wg.Add(1)
	e.staff.In <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::checkHWOwner():%w", err)
		return
	}

	switch attr := resp.Data.(type) {
	case StaffUserAttr:
		userAttr = attr
	default:
		err = fmt.Errorf("env::checkHWOwner(): owner %q: bad response, expected type %T, got %T", owner, userAttr, attr)
	}
	return
}

func (e *env) getUsers(hostID string, server string, ch chan Response, wg *sync.WaitGroup) {
	defer wg.Done()
	var users []UserInfo
	var err error

	users, err = e.getAllHostUsers(hostID, server)
	if err != nil {
		ch <- Response{err: fmt.Errorf("env::getHWInfo():%w", err), Data: users}
		return
	}

	users = e.getUsersData(users)

	ch <- Response{
		Data: users,
		err:  nil,
	}
}

func (e *env) getAllHostUsers(hostID string, server string) (users []UserInfo, err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Type: KSCGetHostUsers,
		Data: hostID,
		wg:   &wg,
		Out:  out,
	}

	respKSCCh := e.getKscChan(server)
	if !respKSCCh.ok {
		err = fmt.Errorf("env::getAllHostUsers(): the channel to %s not found, host ID %q", server, hostID)
		return
	}

	wg.Add(1)
	respKSCCh.ch <- req
	//wg.Wait()
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::getAllHostUsers(): host ID %q:%w", hostID, resp.err)
		return
	}

	switch KSCUsers := resp.Data.(type) {
	case []ksc.GlobalUsersListView:
		var user UserInfo
		for _, KSCUser := range KSCUsers {
			user.GlobalUsersListView = KSCUser
			users = append(users, user)
		}
	default:
		err = fmt.Errorf("env::getAllHostUsers(): host ID %q: bad response, expected type []GlobalUsersListView, got %T", hostID, KSCUsers)
	}
	return
}

func (e *env) getUsersData(users []UserInfo) (fullUsersData []UserInfo) {
	var wg sync.WaitGroup
	out := make(chan Response)
	go func() {
		for _, user := range users {
			wg.Add(1)
			req := Request{
				Type: StaffGetUserInfo,
				Data: user,
				wg:   &wg,
				Out:  out,
			}
			e.staff.In <- req
		}
		waitclose(out, &wg)
	}()

	for resp := range out {
		if resp.err != nil {
			log.Printf("env::getUsersData():%v", resp.err)
			continue
		}

		switch user := resp.Data.(type) {
		case UserInfo:
			fullUsersData = append(fullUsersData, user)
		default:
			log.Printf("env::getUsersData(): bad response, expected type UserInfo, got %T", user)
		}
	}

	return
}

func (e *env) getTags(hostID string, server string, ch chan Response, wg *sync.WaitGroup) {
	defer wg.Done()
	var tags []ksc.HostTag
	var err error

	tags, err = e.getHostTags(hostID, server)
	if err != nil {
		ch <- Response{err: fmt.Errorf("env::getTags():%w", err), Data: tags}
		return
	}

	ch <- Response{
		Data: tags,
		err:  nil,
	}
}

func (e *env) getHostTags(hostID string, server string) (tags []ksc.HostTag, err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Type: KSCGetHostTags,
		Data: hostID,
		wg:   &wg,
		Out:  out,
	}

	respKSCCh := e.getKscChan(server)
	if !respKSCCh.ok {
		err = fmt.Errorf("env::getHostTags(): the channel to %s not found, host ID %q", server, hostID)
		return
	}

	wg.Add(1)
	respKSCCh.ch <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::getHostTags(): host ID %q:%w", hostID, resp.err)
		return
	}

	switch data := resp.Data.(type) {
	case []ksc.HostTag:
		tags = data
	default:
		err = fmt.Errorf("env::getHostTags(): host ID %q: bad response, expected type %T, got %T", hostID, tags, data)
	}
	return
}

func (e *env) setHostTags(itemTags ksc.ListItemsTags, server string) (err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Type: KSCSetHostTags,
		Data: itemTags,
		wg:   &wg,
		Out:  out,
	}

	respKSCCh := e.getKscChan(server)
	if !respKSCCh.ok {
		err = fmt.Errorf("env::setHostTags(): the channel to %s not found, host ID %q", server, itemTags.ID)
		return
	}

	wg.Add(1)
	respKSCCh.ch <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::setHostTags(): host ID %q:%w", itemTags.ID, resp.err)
		return
	}

	return
}

func (e *env) getDN(hostID string, ch chan Response, wg *sync.WaitGroup) {
	defer wg.Done()
	var dn string
	var err error

	dn, err = e.getHostDN(hostID)
	if err != nil {
		err = fmt.Errorf("env::getDN():%w", err)
	}

	ch <- Response{
		Data: dn,
		err:  err,
	}
}

func (e *env) getHostDN(hostID string) (dn string, err error) {
	var wg sync.WaitGroup
	out := make(chan Response)
	req := Request{
		Type: LDAPFindComputerDN,
		Data: hostID,
		wg:   &wg,
		Out:  out,
	}

	wg.Add(1)
	e.ldap.In <- req
	resp := <-out
	if resp.err != nil {
		err = fmt.Errorf("env::getHostDN(): host ID %q:%w", hostID, resp.err)
		return
	}

	switch data := resp.Data.(type) {
	case string:
		dn = data
	default:
		err = fmt.Errorf("env::getHostDN(): host ID %q: bad response, expected type %T, got %T", hostID, dn, data)
	}
	return
}
