package main

import (
	"context"
	"fmt"
	"log"
	"time"

	"a.yandex-team.ru/helpdesk/infra/ksc/pkg/ksc"
)

type kscConfig struct {
	workers    int
	serverName string
	config     ksc.Config
}

type kscCahanRequestType uint

type kscChanResponse struct {
	ch chan Request
	ok bool
}

type kscChanRequest struct {
	Type       kscCahanRequestType
	ServerName string
	ch         chan Request
	Out        chan kscChanResponse
}

const (
	kscChanGet = iota
	kscChanSet
)

func kscChanStore(ch chan kscChanRequest) {
	//log.Printf("[Debug] run kscChanStore()")
	kscChanStore := make(map[string]chan Request, 10)
	var resp kscChanResponse
	for {
		req := <-ch
		switch req.Type {
		case kscChanGet:
			resp.ch, resp.ok = kscChanStore[req.ServerName]
		case kscChanSet:
			kscChanStore[req.ServerName] = req.ch
			resp.ch = nil
			resp.ok = true
		}
		req.Out <- resp
	}
}

func kscWorker(ctx context.Context, c *ksc.Client, server string, ch chan Request) {
	//log.Printf("[Debug] run kscWorker() for %s", server)
	var req Request
	var resp Response

	ping := make(chan int)
	go func() {
		for {
			time.Sleep(10 * time.Second)
			ping <- 0
		}
	}()

	for {
		select {
		case req = <-ch:
			switch req.Type {
			case KSCGetHosts:
				hosts, err := kscGetHosts(ctx, c)
				if err != nil {
					resp.err = fmt.Errorf("[%s]kscWorker():%w", server, err)
					req.Out <- resp
				} else {
					log.Printf("[Debug] %s: found %d hosts", server, len(hosts))
					var host Host
					for _, computer := range hosts {
						host.HostInfo = computer
						host.KSCServer = server
						resp.Data = host
						resp.err = nil
						req.Out <- resp
					}
				}
				req.wg.Done()
			case KSCGetHostHW:
				switch hostID := req.Data.(type) {
				case string:
					devices, err := kscGetHostHWInventory(ctx, c, hostID)
					if err != nil {
						resp.Data = nil
						resp.err = fmt.Errorf("kscWorker(): %s:%w", hostID, err)
					} else {
						resp.Data = devices
						resp.err = nil
					}
				default:
					resp.Data = nil
					resp.err = fmt.Errorf("kscWorker(): bad request KSCGetHostHW, get data type %T, expected string", hostID)
				}
				req.Out <- resp
				req.wg.Done()
			case KSCGetHostTags:
				switch hostID := req.Data.(type) {
				case string:
					tags, err := kscGetHostTags(ctx, c, hostID)
					if err != nil {
						resp.Data = nil
						resp.err = fmt.Errorf("kscWorker(): %s:%w", hostID, err)
					} else {
						resp.Data = tags
						resp.err = nil
					}
				default:
					resp.Data = nil
					resp.err = fmt.Errorf("kscWorker(): bad request KSCGetHostTags, get data type %T, expected string", hostID)
				}
				req.Out <- resp
				req.wg.Done()
			case KSCSetHostTags:
				switch itemTags := req.Data.(type) {
				case ksc.ListItemsTags:
					err := kscSetHostTags(ctx, c, itemTags)
					if err != nil {
						resp.Data = nil
						resp.err = fmt.Errorf("kscWorker(): %s:%w", itemTags.ID, err)
					} else {
						resp.Data = nil
						resp.err = nil
					}
				default:
					resp.Data = nil
					resp.err = fmt.Errorf("kscWorker(): bad request KSCSetHostTags, get data type %T, expected string", itemTags)
				}
				req.Out <- resp
				req.wg.Done()
			case KSCGetHostUsers:
				switch hostID := req.Data.(type) {
				case string:
					devices, err := kscGetAllHostUsers(ctx, c, hostID)
					if err != nil {
						resp.Data = nil
						resp.err = fmt.Errorf("kscWorker(): %s:%w", hostID, err)
					} else {
						resp.Data = devices
						resp.err = nil
					}
				default:
					resp.Data = nil
					resp.err = fmt.Errorf("kscWorker(): bad request KSCGetHostUsers, get data type %T, expected string", hostID)
				}
				req.Out <- resp
				req.wg.Done()
			case KSCRemoveHosts:
				switch hostIDs := req.Data.(type) {
				case []string:
					err := kscRemoveHosts(ctx, c, hostIDs)
					if err != nil {
						resp.Data = nil
						resp.err = fmt.Errorf("kscWorker():%w", err)
					} else {
						resp.Data = nil
						resp.err = nil
					}
				default:
					resp.Data = nil
					resp.err = fmt.Errorf("kscWorker(): bad request KSCRemoveHosts, get data type %T, expected []string", hostIDs)
				}
				req.Out <- resp
				req.wg.Done()
			case KSCRemoveUnassignedHosts:
				err := kscRemoveUnassignedHosts(ctx, c)
				if err != nil {
					resp.Data = nil
					resp.err = fmt.Errorf("[%s]kscWorker():%w", server, err)
				} else {
					resp.err = nil
					resp.Data = nil
				}
				req.Out <- resp
				req.wg.Done()
			default:
				resp.Data = nil
				resp.err = fmt.Errorf("kscWorker(): bad request type %d", req.Type)
				req.Out <- resp
				req.wg.Done()
			}
		case <-ping:
			kscPing(ctx, c)
		}
	}
}

func kscGetHosts(ctx context.Context, c *ksc.Client) (hosts []ksc.HostInfo, err error) {
	//log.Printf("[Debug] run kscGetHosts() for %s", c.Server)
	//defer log.Printf("[Debug] exit kscGetHosts() for %s", c.Server)
	var reqID string
	var quantity int
	reqID, quantity, err = c.HostGroup.FindHosts(ctx, fmt.Sprintf("(&(%s=%t))", ksc.KLHostFromUnassigned, false),
		[]ksc.HostAttribute{ksc.KLHostDisplayName, ksc.KLHostGroupID,
			ksc.KLHostCreated, ksc.KLHostLastVisible, ksc.KLHostLastInfoUpdate, ksc.KLHostStatus,
			ksc.KLHostLastUpdate, ksc.KLHostLastNetworkAgentConnected, ksc.KLHostKeepConnection, ksc.KLHostID,
			ksc.KLHostNetworkAgentID, ksc.KLHostWinHostname, ksc.KLHostWinDomain, ksc.KLHostDNSDomain,
			ksc.KLHostDNSName, ksc.KLHostFQDN, ksc.KLHostComputerType, ksc.KLHostPlatformType,
			ksc.KLHostOSName, ksc.KLHostOSVersionMajor, ksc.KLHostOSVersionMinor, ksc.KLHostLastFullScan,
			ksc.KLHostVirusCount, ksc.KLHostRTPState, ksc.KLHostRTPErrorCode, ksc.KLHostComment, ksc.KLHostIP,
			ksc.KLHostConnectionIP, ksc.KLHostFromUnassigned, ksc.KLHostUncuredCount, ksc.KLHostStatusID,
			ksc.KLHostStatusMask, ksc.KLHostStatusHSDP, ksc.KLHostHasUpdateAgent, ksc.KLHostNetworkAgentVersion,
			ksc.KLHostAntiVirusVersion, ksc.KLHostAntiVirusBasesTime, ksc.KLHostLastSystemStart,
			ksc.KLHostRebootRequired, ksc.KLHostRebootRequestReason, ksc.KLHostRebootRequestTime,
			ksc.KLHostOSSPVersionMajor, ksc.KLHostOSSPVersionMinor, ksc.KLHostCPUArch, ksc.KLHostOSBuildNumber,
			ksc.KLHostOSReleaseID, ksc.KLHostNetworkAgentVersionID, ksc.KLHostOwnerID, ksc.KLHostOwnerIsCustom,
			ksc.KLHostProductTag, ksc.KLHostHSDPReasonID, ksc.KLHostAntiSpamStatus, ksc.KLHostDLPStatus,
			ksc.KLHostCollabSrvsStatus, ksc.KLHostEmailAVStatus, ksc.KLHostEDRStatus, ksc.KLHostVMType,
			ksc.KLHostVMVDI, ksc.KLHostManagedOtherServer, ksc.KLHostHSDPReasonsAll, ksc.KLHostClusterID,
			ksc.KLHostIsCluster, ksc.KLHostIsClusterNode, ksc.KLHostClusterNodeWMISupported,
			ksc.KLHostClusterWMISupported, ksc.KLHostClusterNodeWMIFullActive, ksc.KLHostClusterWMIActiveID,
			ksc.KLHostClusterWMIProduct, ksc.KLHostClusterWMIVersion, ksc.KLHostEncryptionState,
			ksc.KLHostCloudHostID, ksc.KLHostCloudHostPlatform, ksc.KLHostCloudHostType,
			ksc.KLHostWUALastSearchValid, ksc.KLHostWUALastSearch, ksc.KLHostWUASwitched,
			ksc.KLHostSlaveServerDN, ksc.KLHostVirtualServerDN, ksc.KLHostVirtualServerID, ksc.KLHostGroupName},
		ksc.HostSearchExtraParams{}, 10)
	if err != nil {
		err = fmt.Errorf("kscGetHosts():%w", err)
		return
	}

	/*defer func() {
		err := c.ChunkAccessor.Release(ctx, reqID)
		if err != nil {
			log.Printf("kscGetHosts():ChunkAccessor.Release(): %s\n", err.Error())
		}
	}()*/

	for index := 0; index < quantity; index += 1000 {
		var hostsChunk []ksc.HostInfo
		_, err = c.ChunkAccessor.GetItemsChunk(ctx, reqID, index, 1000, &hostsChunk)
		if err != nil {
			err = fmt.Errorf("kscGetHosts():%w", err)
			return
		}
		hosts = append(hosts, hostsChunk...)
	}

	return
}

func kscGetHostHWInventory(ctx context.Context, c *ksc.Client, hostID string) (devices []ksc.HwInvPCView, err error) {
	//log.Printf("[Debug] run kscGetHostHWInventory() for %s", c.Server)
	//defer log.Printf("[Debug] exit kscGetHostHWInventory() for %s", c.Server)
	iter, err := c.SrvView.ResetIterator(ctx, ksc.HWInvPCSrvViewName,
		fmt.Sprintf(`(&(KLHST_WKS_HOSTNAME = %q))`, hostID),
		[]ksc.Attributer{ksc.KLDevHostID, ksc.KLDevID, ksc.KLDevType, ksc.KLDevName, ksc.KLDevDescription,
			ksc.KLDevManufacturer, ksc.KLDevDriverProvider, ksc.KLDevDriverVersion, ksc.KLDevDriverDescription,
			ksc.KLDevDriverDate, ksc.KLDevXPar1, ksc.KLDevXPar2, ksc.KLDevXPar3, ksc.KLDevBiosManufacturer,
			ksc.KLDevBiosVersion, ksc.KLDevBiosSerialNumber, ksc.KLDevBiosReleaseDate, ksc.KLDevChipset,
			ksc.KLDevCPUSpeed, ksc.KLDevCPUCores, ksc.KLDevCPUThreads, ksc.KLDevCPUPlatform, ksc.KLDevSerialNumber,
			ksc.KLDevRevision, ksc.KLDevStorageRotationRate, ksc.KLDevRAMType, ksc.KLDevRAMPartNumber,
			ksc.KLDevVideoCardMemorySize, ksc.KLDevSoundCardCodec}, 10, &ksc.HwInvPCViewParams{})
	if err != nil {
		err = fmt.Errorf("kscGetHostHWInventory():%w", err)
		return
	}
	defer func(iter string, hostID string) {
		err := c.SrvView.ReleaseIterator(ctx, iter)
		if err != nil {
			log.Printf("kscGetHostHWInventory(): host ID: %s:%v", hostID, err)
		}
	}(iter, hostID)

	count, err := c.SrvView.GetRecordCount(ctx, iter)
	if err != nil {
		err = fmt.Errorf("kscGetHostHWInventory():%w", err)
		return
	}

	err = c.SrvView.GetRecordRange(ctx, iter, 0, count, &devices)
	if err != nil {
		err = fmt.Errorf("kscGetHostHWInventory():%w", err)
		return
	}

	return
}

func kscGetAllHostUsers(ctx context.Context, c *ksc.Client, hostID string) (users []ksc.GlobalUsersListView, err error) {
	//log.Printf("[Debug] run kscGetAllHostUsers() for %s", c.Server)
	//defer log.Printf("[Debug] exit kscGetAllHostUsers() for %s", c.Server)
	iter, err := c.SrvView.ResetIterator(ctx, ksc.GlobalUsersListSrvViewName,
		fmt.Sprintf(`(&(%s=%t))`, ksc.KLUserIsLocal, false),
		[]ksc.Attributer{ksc.KLUserTrusteeID, ksc.KLUserTrusteeIDOrig, ksc.KLUserVServer, ksc.KLUserBinID,
			ksc.KLUserIsUser, ksc.KLUserIsLocal, ksc.KLUserIsBuiltin, ksc.KLUserIsInternalUser, ksc.KLUserSID,
			ksc.KLUserSIDHash, ksc.KLUserDistinguishedName, ksc.KLUserObjectGUID, ksc.KLUserDisplayName,
			ksc.KLUserPrincipalName, ksc.KLUserSamAccountName, ksc.KLUserMail, ksc.KLUserMailExtra,
			ksc.KLUserTelephoneNumber, ksc.KLUserTelephoneExtra, ksc.KLUserMobile, ksc.KLUserDepartment,
			ksc.KLUserCompany, ksc.KLUserManager, ksc.KLUserDomainSAM, ksc.KLUserDomain2000, ksc.KLUserDescription,
			ksc.KLUserGroupID, ksc.KLUserVServerName, ksc.KLUserUserID, ksc.KLUserInternalUserDescription,
			ksc.KLUserDisabled, ksc.KLUserVisited}, 10, &ksc.GlobalUsersListViewParams{HostID: hostID})
	if err != nil {
		err = fmt.Errorf("kscGetAllHostUsers():%w", err)
		return
	}
	defer func(iter string, hostID string) {
		err := c.SrvView.ReleaseIterator(ctx, iter)
		if err != nil {
			log.Printf("kscGetAllHostUsers(): host ID: %s:%v", hostID, err)
		}
	}(iter, hostID)

	count, err := c.SrvView.GetRecordCount(ctx, iter)
	if err != nil {
		err = fmt.Errorf("kscGetAllHostUsers():%w", err)
		return
	}

	err = c.SrvView.GetRecordRange(ctx, iter, 0, count, &users)
	if err != nil {
		err = fmt.Errorf("kscGetAllHostUsers():%w", err)
		return
	}

	return
}

func kscGetHostTags(ctx context.Context, c *ksc.Client, hostID string) (tags []ksc.HostTag, err error) {
	//log.Printf("[Debug] run kscGetHostTags() for %s", c.Server)
	//defer log.Printf("[Debug] exit kscGetHostTags() for %s", c.Server)
	tags, err = c.HostTagsAPI.GetHostTags(ctx, hostID)
	if err != nil {
		err = fmt.Errorf("kscGetHostTags():%w", err)
	}

	return
}

func kscSetHostTags(ctx context.Context, c *ksc.Client, itemTags ksc.ListItemsTags) (err error) {
	//log.Printf("[Debug] run kscGetHostTags() for %s", c.Server)
	//defer log.Printf("[Debug] exit kscGetHostTags() for %s", c.Server)
	err = c.ListTags.SetTags(ctx, []ksc.ListItemsTags{itemTags}, false)
	if err != nil {
		err = fmt.Errorf("kscSetHostTags():%w", err)
	}

	return
}

func kscRemoveHosts(ctx context.Context, c *ksc.Client, hostIDs []string) (err error) {
	err = c.HostGroup.RemoveHosts(ctx, hostIDs, true)
	if err != nil {
		err = fmt.Errorf("kscRemoveHosts():%w", err)
	}

	return
}

func kscRemoveUnassignedHosts(ctx context.Context, c *ksc.Client) (err error) {
	var reqID string
	var quantity int
	reqID, quantity, err = c.HostGroup.FindHosts(ctx,
		fmt.Sprintf(`(&(%s=%v)(%s<>%q))`, ksc.KLHostFromUnassigned, true, ksc.KLHostNetworkAgentID, ""),
		[]ksc.HostAttribute{ksc.KLHostID}, ksc.HostSearchExtraParams{}, 10)
	if err != nil {
		err = fmt.Errorf("kscRemoveUnassignedHosts():%w", err)
		return
	}

	var hosts []ksc.HostInfo
	for index := 0; index < quantity; index += 1000 {
		var hostsChunk []ksc.HostInfo
		_, err = c.ChunkAccessor.GetItemsChunk(ctx, reqID, index, 1000, &hostsChunk)
		if err != nil {
			err = fmt.Errorf("kscGetHosts():%w", err)
			return
		}
		hosts = append(hosts, hostsChunk...)
	}

	var hostIDs []string
	for _, host := range hosts {
		hostIDs = append(hostIDs, host.ID)
	}

	if len(hostIDs) == 0 {
		return
	}

	err = kscRemoveHosts(ctx, c, hostIDs)
	if err != nil {
		err = fmt.Errorf("kscRemoveUnassignedHosts():%w", err)
		return
	}

	return
}

func kscPing(ctx context.Context, c *ksc.Client) {
	var err error
	if c.XKscSession {
		err = c.Session.Ping(ctx)
	} else {
		_, err = c.HostGroup.GetStaticInfo(ctx, []ksc.StaticInfoValue{ksc.SVInstancePort})
	}

	if err != nil {
		log.Printf("kscPing(): %q, %v", c.Server, err)
	} else {
		return
	}

	err = c.Login(ctx, ksc.BasicAuthentication, "")
	if err != nil {
		log.Printf("kscPing(): %q, %v", c.Server, err)
	}
}
