package main

import (
	"log"
	"strings"
	"time"
)

type Mark string

func (m Mark) String() string {
	return string(m)
}

const (
	OutdatedHost               Mark = "Not connected in a long time"
	HostWithoutUsersAndInvNum  Mark = "Host Without Users and Undefined Inventory Number"
	UnusedHost                 Mark = "Device status other than 'OPERATION'"
	DismissedOwner             Mark = "The owner was dismissed"
	DuplicateByHostname        Mark = "Duplicate by hostname"
	DuplicateByInventoryNumber Mark = "Duplicate by inventory number"
)

func MarkupHosts(hosts []Host) []Host {
	log.Printf("Start MarkupHosts")
	defer log.Printf("Finish MarkupHosts")
	hosts = MarkOutdatedHosts(hosts)
	hosts = MarkHostsWithoutUsersAndInvNum(hosts)
	hosts = MarkUnusedDevices(hosts)
	hosts = MarkDevicesWithDismissedOwners(hosts)
	hosts = MarkDuplicatedHostsByHostname(hosts)
	hosts = MarkDuplicatedHostsByInventoryNumber(hosts)

	return hosts
}

func MarkOutdatedHosts(hosts []Host) []Host {
	redLine := time.Now().Add(-62 * 24 * time.Hour)
	for index := range hosts {
		if hosts[index].MarkToDelete {
			continue
		}

		if hosts[index].LastNetworkAgentConnected.Before(redLine) {
			hosts[index].MarkToDelete = true
			hosts[index].DeleteReason = OutdatedHost.String()
		}
	}

	return hosts
}

func MarkHostsWithoutUsersAndInvNum(hosts []Host) []Host {
	for index := range hosts {
		if hosts[index].MarkToDelete {
			continue
		}

		if strings.Contains(strings.ToLower(hosts[index].OSName), "server") {
			continue
		}

		if len(hosts[index].Users) == 0 && hosts[index].OEBSInventoryNumber == "" {
			hosts[index].MarkToDelete = true
			hosts[index].DeleteReason = HostWithoutUsersAndInvNum.String()
		}
	}

	return hosts
}

func MarkUnusedDevices(hosts []Host) []Host {
	for index := range hosts {
		if hosts[index].MarkToDelete {
			continue
		}

		if hosts[index].OEBSStatus == "" {
			continue
		}

		if strings.Contains(strings.ToLower(hosts[index].OSName), "server") {
			continue
		}

		if hosts[index].OEBSStatus != "OPERATION" {
			hosts[index].MarkToDelete = true
			hosts[index].DeleteReason = UnusedHost.String()
		}
	}

	return hosts
}

func MarkDevicesWithDismissedOwners(hosts []Host) []Host {
	for index := range hosts {
		if hosts[index].MarkToDelete {
			continue
		}

		if hosts[index].OEBSOwner == "" {
			continue
		}

		if strings.Contains(strings.ToLower(hosts[index].OSName), "server") {
			continue
		}

		if hosts[index].OEBSOwnerIsDismissed {
			hosts[index].MarkToDelete = true
			hosts[index].DeleteReason = DismissedOwner.String()
		}
	}

	return hosts
}

type LastConnectedIndex struct {
	Index         int
	LastConnected time.Time
}

func MarkDuplicatedHostsByHostname(hosts []Host) []Host {
	dict := make(map[string][]LastConnectedIndex)
	for index := range hosts {
		if hosts[index].MarkToDelete {
			continue
		}

		if strings.ToLower(hosts[index].DNSDomain) != "ld.yandex.ru" {
			continue
		}

		dict[hosts[index].WinHostname] = append(dict[hosts[index].WinHostname], LastConnectedIndex{
			Index:         index,
			LastConnected: hosts[index].LastNetworkAgentConnected,
		})
	}

	for _, indexes := range dict {
		if len(indexes) < 2 {
			continue
		}

		oldestIndexes := ListOfOlderIndexes(indexes)
		for _, index := range oldestIndexes {
			hosts[index].MarkToDelete = true
			hosts[index].DeleteReason = DuplicateByHostname.String()
		}
	}

	return hosts
}

func MarkDuplicatedHostsByInventoryNumber(hosts []Host) []Host {
	dict := make(map[string][]LastConnectedIndex)
	for index := range hosts {
		if hosts[index].MarkToDelete {
			continue
		}

		if hosts[index].OEBSInventoryNumber == "" {
			continue
		}

		if strings.ToLower(hosts[index].DNSDomain) != "ld.yandex.ru" {
			continue
		}

		if strings.Contains(strings.ToLower(hosts[index].OSName), "server") {
			continue
		}

		dict[hosts[index].OEBSInventoryNumber] = append(dict[hosts[index].OEBSInventoryNumber], LastConnectedIndex{
			Index:         index,
			LastConnected: hosts[index].LastNetworkAgentConnected,
		})
	}

	for _, indexes := range dict {
		if len(indexes) < 2 {
			continue
		}

		oldestIndexes := ListOfOlderIndexes(indexes)
		for _, index := range oldestIndexes {
			hosts[index].MarkToDelete = true
			hosts[index].DeleteReason = DuplicateByInventoryNumber.String()
		}
	}

	return hosts
}

func ListOfOlderIndexes(indexes []LastConnectedIndex) (list []int) {
	if len(indexes) < 2 {
		return
	}

	newestIndex := indexes[0]
	for _, index := range indexes[1:] {
		if index.LastConnected.After(newestIndex.LastConnected) {
			newestIndex = index
		}
	}

	for _, index := range indexes {
		if index.Index != newestIndex.Index {
			list = append(list, index.Index)
		}
	}

	return
}
