package main

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"github.com/go-resty/resty/v2"
)

type StaffCacheRequestType uint

type StaffCacheResponse struct {
	found    bool
	userAttr StaffUserAttr
}

type StaffCacheRequest struct {
	Type     StaffCacheRequestType
	Username string
	UserAttr StaffUserAttr
	Out      chan StaffCacheResponse
}

const (
	StaffCacheGet StaffCacheRequestType = iota
	StaffCacheSet
)

type staffConfig struct {
	host    string
	token   string
	workers int
	In      chan Request
	cache   chan StaffCacheRequest
}

type StaffUserAttr struct {
	Dismissed      bool
	DepartmentName string
	DepartmentID   int
}

type StaffUser struct {
	Login    string `json:"login"`
	Official struct {
		IsDismissed bool `json:"is_dismissed"`
	} `json:"official"`
	DepartmentGroup struct {
		Department Department `json:"department"`
	} `json:"department_group"`
}

type Department struct {
	ID   int `json:"id"`
	Name struct {
		Full struct {
			En string `json:"en"`
		} `json:"full"`
	} `json:"name"`
}

type StaffGroup struct {
	ID   int `json:"id"`
	Name struct {
		Full struct {
			En string `json:"en"`
		} `json:"full"`
	} `json:"name"`
}

type StaffGroupMember struct {
	ID       int    `json:"id"`
	JoinedAt string `json:"joined_at"`
	Group    struct {
		ID      int    `json:"id"`
		Name    string `json:"name"`
		Service struct {
			ID int `json:"id"`
		} `json:"service"`
		URL        string `json:"url"`
		Department struct {
			ID int `json:"id"`
		} `json:"department"`
		IsDeleted bool `json:"is_deleted"`
	} `json:"group"`
	Person struct {
		ID   int    `json:"id"`
		UID  string `json:"uid"`
		Name struct {
			First struct {
				Ru string `json:"ru"`
				En string `json:"en"`
			} `json:"first"`
			Last struct {
				Ru string `json:"ru"`
				En string `json:"en"`
			} `json:"last"`
		} `json:"name"`
		Login    string `json:"login"`
		Official struct {
			Affiliation  string `json:"affiliation"`
			IsDismissed  bool   `json:"is_dismissed"`
			IsRobot      bool   `json:"is_robot"`
			IsHomeworker bool   `json:"is_homeworker"`
		} `json:"official"`
	} `json:"person"`
}

func StaffCache(ch chan StaffCacheRequest) {
	cache := make(map[string]StaffUserAttr)
	var resp StaffCacheResponse
	for {
		req := <-ch
		switch req.Type {
		case StaffCacheGet:
			resp.userAttr, resp.found = cache[req.Username]
		case StaffCacheSet:
			cache[req.Username] = req.UserAttr
			resp.found = false
			resp.userAttr = req.UserAttr
		}
		req.Out <- resp
	}
}

func staffWorker(config staffConfig) {
	//log.Printf("[Debug] run staffWorker()")
	var req Request
	var resp Response
	for {
		req = <-config.In
		switch req.Type {
		case StaffGetGroupMembers:
			switch groupID := req.Data.(type) {
			case int:
				members, err := staffGetGroupMembers(groupID, config)
				if err != nil {
					resp.Data = nil
					resp.err = fmt.Errorf("staffWorker(): %d:%w", groupID, err)
				} else {
					resp.Data = members
					resp.err = nil
				}
			default:
				resp.Data = nil
				resp.err = fmt.Errorf("staffWorker(): bad request StaffGetGroupMembers, get data type %T", groupID)
			}
			req.Out <- resp
			req.wg.Done()
		case StaffGetSubgroups:
			switch groupID := req.Data.(type) {
			case int:
				groups, err := staffGetSubgroups(groupID, config)
				if err != nil {
					resp.Data = nil
					resp.err = fmt.Errorf("staffWorker(): %d:%w", groupID, err)
				} else {
					resp.Data = groups
					resp.err = nil
				}
			default:
				resp.Data = nil
				resp.err = fmt.Errorf("staffWorker(): bad request StaffGetSubgroups, get data type %T", groupID)
			}
			req.Out <- resp
			req.wg.Done()
		case StaffGetUserInfo:
			switch user := req.Data.(type) {
			case UserInfo:
				userInfo, err := staffFindUser(user.SAMAccountName, config)
				if err != nil {
					resp.Data = nil
					resp.err = fmt.Errorf("staffWorker(): %q:%w", user.SAMAccountName, err)
				} else {
					user.StaffUserAttr = userInfo
					resp.Data = user
					resp.err = nil
				}
			case string:
				userInfo, err := staffFindUser(user, config)
				if err != nil {
					resp.Data = nil
					resp.err = fmt.Errorf("staffWorker(): %q:%w", user, err)
				} else {
					resp.Data = userInfo
					resp.err = nil
				}
			default:
				resp.Data = nil
				resp.err = fmt.Errorf("staffWorker(): bad request StaffGetUserInfo, get data type %T", user)
			}
			req.Out <- resp
			req.wg.Done()
		default:
			resp.Data = nil
			resp.err = fmt.Errorf("staffWorker(): bad request type %d", req.Type)
			req.Out <- resp
			req.wg.Done()
		}
	}
}

func staffFindUser(username string, config staffConfig) (userAttr StaffUserAttr, err error) {
	//log.Printf("[Debug] run staffFindUser()")
	//defer log.Printf("[Debug] exit staffFindUser()")
	var foundInCache bool
	userAttr, foundInCache = findUserInStaffCache(username, config.cache)
	if foundInCache {
		return
	}

	client := resty.New()
	client.SetRetryCount(5)
	client.SetRetryWaitTime(1 * time.Second)
	client.SetRetryMaxWaitTime(2 * time.Second)
	client.AddRetryCondition(func(resp *resty.Response, err error) bool {
		if resp.StatusCode() >= http.StatusBadRequest || err != nil {
			return true
		}
		return false
	})

	req := client.R()
	req.Method = http.MethodGet
	req.URL = config.host + "persons"
	req.SetQueryParam("_pretty", "1")
	req.SetQueryParam("_one", "1")
	req.SetQueryParam("_fields", "login,department_group.department.id,department_group.department.name.full.en,official.is_dismissed")
	req.SetQueryParam("login", url.QueryEscape(username))
	client.SetHeader("Authorization", fmt.Sprintf("OAuth %s", config.token))

	var resp *resty.Response
	resp, err = req.Send()
	if err != nil {
		err = fmt.Errorf("staffFindUser(): username %q:%w", username, err)
		return
	}

	var user StaffUser
	err = json.Unmarshal(resp.Body(), &user)
	if err != nil {
		err = fmt.Errorf("staffFindUser(): username %q: unmarshal %q:%w", username, resp.Body(), err)
		return
	}

	userAttr.Dismissed = user.Official.IsDismissed
	userAttr.DepartmentID = user.DepartmentGroup.Department.ID
	userAttr.DepartmentName = user.DepartmentGroup.Department.Name.Full.En

	putUserIntoStaffCache(username, userAttr, config.cache)

	return
}

func findUserInStaffCache(username string, cache chan StaffCacheRequest) (StaffUserAttr, bool) {
	//log.Printf("[Debug] run findUserInStaffCache()")
	//defer log.Printf("[Debug] exit findUserInStaffCache()")
	out := make(chan StaffCacheResponse)
	req := StaffCacheRequest{
		Type:     StaffCacheGet,
		Username: username,
		UserAttr: StaffUserAttr{},
		Out:      out,
	}
	cache <- req
	resp := <-out

	return resp.userAttr, resp.found
}

func putUserIntoStaffCache(username string, userAttr StaffUserAttr, cache chan StaffCacheRequest) {
	//log.Printf("[Debug] run putUserIntoStaffCache()")
	//defer log.Printf("[Debug] exit putUserIntoStaffCache()")
	out := make(chan StaffCacheResponse)
	req := StaffCacheRequest{
		Type:     StaffCacheSet,
		Username: username,
		UserAttr: userAttr,
		Out:      out,
	}
	cache <- req
	<-out
}

func staffGetSubgroups(id int, config staffConfig) (groups []StaffGroup, err error) {
	startURL, err := url.Parse(config.host + "groups")
	if err != nil {
		return nil, fmt.Errorf("staffGetSubgroups(): parse Staff API host %q: %w", config.host, err)
	}

	parameters := url.Values{}
	parameters.Add("parent.department.id", strconv.Itoa(id))
	parameters.Add("_fields", "department.name.full.en,department.id")
	startURL.RawQuery = parameters.Encode()
	pageURL := startURL.String()
	for pageURL != "" {
		var pageGroups []StaffGroup
		pageGroups, pageURL, err = staffGetSubgroupPage(pageURL, config.token)
		if err != nil {
			err = fmt.Errorf("staffGetSubgroups(): get page %q: %w", pageURL, err)
			return
		}

		groups = append(groups, pageGroups...)
	}

	return
}

func staffGetSubgroupPage(pageURL string, token string) ([]StaffGroup, string, error) {
	var departments []StaffGroup
	nextURL := ""
	client := resty.New()
	client.SetRetryCount(5)
	client.SetRetryWaitTime(1 * time.Second)
	client.SetRetryMaxWaitTime(2 * time.Second)
	client.AddRetryCondition(func(resp *resty.Response, err error) bool {
		if resp.StatusCode() >= http.StatusBadRequest || err != nil {
			return true
		}
		return false
	})
	resp, err := client.R().
		SetHeader("Authorization", "OAuth "+token).
		Get(pageURL)

	if err != nil {
		return departments, nextURL, fmt.Errorf("staffGetSubgroupPage(): get response from Staff API: %w", err)
	}

	result := struct {
		Links struct {
			Last string `json:"last"`
			Next string `json:"next"`
		} `json:"links"`
		Page   int `json:"page"`
		Limit  int `json:"limit"`
		Result []struct {
			Department StaffGroup `json:"department"`
		} `json:"result"`
		Total int `json:"total"`
		Pages int `json:"pages"`
	}{}

	err = json.Unmarshal(resp.Body(), &result)
	if err != nil {
		return departments, nextURL, fmt.Errorf("staffGetSubgroupPage(): unmarshal response %q: %w", resp.Body(), err)
	}

	for _, group := range result.Result {
		departments = append(departments, group.Department)
	}

	return departments, result.Links.Next, nil
}

func staffGetGroupMembers(groupID int, config staffConfig) (groupMembers []StaffGroupMember, err error) {
	startURL, err := url.Parse(config.host + "groupmembership")
	if err != nil {
		err = fmt.Errorf("staffGetGroupMembers(): parse Staff API host %q: %w", config.host, err)
		return
	}

	parameters := url.Values{}
	parameters.Add("group.department.id", strconv.Itoa(groupID))
	parameters.Add("_fields", "login")
	startURL.RawQuery = parameters.Encode()
	pageURL := startURL.String()

	for pageURL != "" {
		var pageGroupMembers []StaffGroupMember
		pageGroupMembers, pageURL, err = staffGetGroupMembersPage(pageURL, config.token)
		if err != nil {
			err = fmt.Errorf("staffGetSubgroups(): get page %q: %w", pageURL, err)
			return
		}

		groupMembers = append(groupMembers, pageGroupMembers...)
	}

	return
}

func staffGetGroupMembersPage(pageURL string, token string) ([]StaffGroupMember, string, error) {
	var groupMembers []StaffGroupMember
	client := resty.New()
	client.SetRetryCount(5)
	client.SetRetryWaitTime(1 * time.Second)
	client.SetRetryMaxWaitTime(2 * time.Second)
	client.AddRetryCondition(func(resp *resty.Response, err error) bool {
		if resp.StatusCode() >= http.StatusBadRequest || err != nil {
			return true
		}
		return false
	})
	resp, err := client.R().
		SetHeader("Authorization", "OAuth "+token).
		Get(pageURL)

	if err != nil {
		return groupMembers, "", fmt.Errorf("staffGetGroupMembersPage(): get response from Staff API: %w", err)
	}

	result := struct {
		Links struct {
			Last string `json:"last"`
			Next string `json:"next"`
		} `json:"links"`
		Page   int `json:"page"`
		Limit  int `json:"limit"`
		Result []StaffGroupMember
		Total  int `json:"total"`
		Pages  int `json:"pages"`
	}{}

	err = json.Unmarshal(resp.Body(), &result)
	if err != nil {
		return groupMembers, "", fmt.Errorf("staffGetGroupMembersPage(): unmarshal response %q: %w", resp.Body(), err)
	}

	groupMembers = append(groupMembers, result.Result...)

	return groupMembers, result.Links.Next, nil
}
