package main

import (
	"context"
	"fmt"
	"log"
	"sort"
	"time"

	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

func (e *env) WriteHostsData(hosts []Host) error {
	log.Printf("Start WriteHostsData")
	defer log.Printf("Finish WriteHostsData")

	const LatestYTTable = "//home/helpdesk/cmdb/ksc/ksc-latest"

	t := time.Now()
	path := fmt.Sprintf("//home/helpdesk/cmdb/ksc/ksc13-%d-%02d-%02d_%02d_%02d", t.Year(), t.Month(), t.Day(), t.Hour(), t.Minute())
	tablePath := ypath.Path(path)
	err := e.WriteTableData(tablePath, hosts, false)
	if err != nil {
		return fmt.Errorf("env::WriteHostsData(): %w", err)
	}

	err = e.WriteTableData(LatestYTTable, hosts, true)
	if err != nil {
		return fmt.Errorf("env::WriteHostsData(): %w", err)
	}

	return nil
}

func (e *env) WriteTableData(tablePath ypath.Path, hosts []Host, tableExist bool) (err error) {
	for attempt := 0; attempt < 5; attempt++ {
		err = e.writeTableData(tablePath, hosts, tableExist)
		if err != nil {
			log.Printf("env::WriteTableData(): attempt %d: %s", attempt, err.Error())
			time.Sleep(time.Duration(1*attempt+1) * time.Second)
			continue
		}

		return
	}

	return fmt.Errorf("env::WriteTableData(): %w", err)
}

func (e *env) writeTableData(tablePath ypath.Path, hosts []Host, tableExist bool) (err error) {
	ctx := context.Background()
	var w yt.TableWriter
	if !tableExist {
		err = e.CreateTable(tablePath)
		if err != nil {
			return fmt.Errorf("env::writeTableData(): %w", err)
		}
	}

	w, err = yt.WriteTable(ctx, e.yt.client, tablePath, yt.WithExistingTable(), yt.WithBatchSize(10485760))
	if err != nil {
		return fmt.Errorf("env::writeTableData(): %w", err)
	}
	defer func() {
		err := w.Commit()
		if err != nil {
			log.Printf("[Error] env::writeTableData(): commit: %s", err.Error())
		}
	}()

	sort.Slice(hosts, func(i, j int) bool {
		return hosts[i].ID < hosts[j].ID
	})

	for _, host := range hosts {
		err = w.Write(host)
		if err != nil {
			return fmt.Errorf("env::writeTableData(): %w", err)
		}
	}

	return nil
}

func (e *env) CreateTable(tablePath ypath.Path) (err error) {
	tableSchema, err := schema.Infer(Host{})
	if err != nil {
		return fmt.Errorf("env::CreateTable(): YT infer schema: %w", err)
	}
	for index, col := range tableSchema.Columns {
		if col.Name == "ID" {
			tableSchema.Columns[index].SortOrder = schema.SortAscending
		}
	}

	ctx := context.Background()
	_, err = yt.CreateTable(ctx, e.yt.client, tablePath, yt.WithSchema(tableSchema))
	if err != nil {
		return fmt.Errorf("createTable(): YT create table: %w", err)
	}

	return nil
}
