package ksc

import (
	"context"
	"fmt"
	"net/http"
	"time"
)

type AsyncActionStateChecker service

const apiMethodAsyncActionStateCheckerGetState apiMethod = "/api/v1.0/AsyncActionStateChecker.CheckActionState"

type AsyncActionState struct {
	Finalized      bool
	Succeeded      bool
	StateCode      int
	StateData      AsyncActionStateData
	NextCheckDelay int
}

type AsyncActionStateData struct {
	Code        int
	Description string
	FileName    string
	LineNumber  int
	Module      string
	Message     string
	SubCode     int
}

type asyncActionState struct {
	Finalized      bool                 `json:"bFinalized"`
	Succeeded      bool                 `json:"bSuccededFinalized"`
	StateCode      int                  `json:"lStateCode"`
	StateData      asyncActionStateData `json:"pStateData"`
	NextCheckDelay int                  `json:"lNextCheckDelay"`
	Error          kscError             `json:"PxgError"`
}

func (as *asyncActionState) toAsyncActionState() AsyncActionState {
	aas := AsyncActionState{
		Finalized:      as.Finalized,
		Succeeded:      as.Succeeded,
		StateCode:      as.StateCode,
		StateData:      as.StateData.toAsyncActionStateData(),
		NextCheckDelay: as.NextCheckDelay,
	}

	return aas
}

type asyncActionStateData struct {
	Description string `json:"GNRL_EA_DESCRIPTION"`
	ErrorInfo   struct {
		Type  string `json:"type"`
		Value struct {
			Code       int    `json:"KLBLAG_ERROR_CODE"`
			FileName   string `json:"KLBLAG_ERROR_FNAME"`
			LineNumber int    `json:"KLBLAG_ERROR_LNUMBER"`
			Module     string `json:"KLBLAG_ERROR_MODULE"`
			Message    string `json:"KLBLAG_ERROR_MSG"`
			SubCode    int    `json:"KLBLAG_ERROR_SUBCODE"`
		} `json:"value"`
	} `json:"KLBLAG_ERROR_INFO"`
}

func (sd *asyncActionStateData) toAsyncActionStateData() AsyncActionStateData {
	aasd := AsyncActionStateData{
		Code:        sd.ErrorInfo.Value.Code,
		Description: sd.Description,
		FileName:    sd.ErrorInfo.Value.FileName,
		LineNumber:  sd.ErrorInfo.Value.LineNumber,
		Module:      sd.ErrorInfo.Value.Module,
		Message:     sd.ErrorInfo.Value.Message,
		SubCode:     sd.ErrorInfo.Value.SubCode,
	}

	return aasd
}

func (aasc *AsyncActionStateChecker) CheckActionStateCycle(ctx context.Context, actionGUID string) error {
	err := aasc.checkActionStateCycle(ctx, actionGUID)
	if err != nil {
		err = fmt.Errorf("AsyncActionStateChecker::CheckActionStateCycle(): %w", err)
	}

	return err
}

func (aasc *AsyncActionStateChecker) checkActionStateCycle(ctx context.Context, actionGUID string) error {
	for {
		state, err := aasc.checkActionState(ctx, actionGUID)
		if err != nil {
			return fmt.Errorf("AsyncActionStateChecker::checkActionStateCycle(): %w", err)
		}

		if !state.Finalized {
			time.Sleep(time.Duration(state.NextCheckDelay) * time.Millisecond)
			continue
		}

		if !state.Succeeded {
			return fmt.Errorf("AsyncActionStateChecker::checkActionStateCycle: %d %s: code %d %d: module %s: line %d: file: %s: %s",
				state.StateCode,
				state.StateData.Description,
				state.StateData.ErrorInfo.Value.Code,
				state.StateData.ErrorInfo.Value.SubCode,
				state.StateData.ErrorInfo.Value.Module,
				state.StateData.ErrorInfo.Value.LineNumber,
				state.StateData.ErrorInfo.Value.FileName,
				state.StateData.ErrorInfo.Value.Message)
		} else {
			return nil
		}
	}
}

func (aasc *AsyncActionStateChecker) CheckActionState(ctx context.Context, actionGUID string) (AsyncActionState, error) {
	state, err := aasc.checkActionState(ctx, actionGUID)
	if err != nil {
		return AsyncActionState{}, fmt.Errorf("AsyncActionStateChecker::CheckActionState(): %w", err)
	}

	return state.toAsyncActionState(), nil
}

func (aasc *AsyncActionStateChecker) checkActionState(ctx context.Context, actionGUID string) (asyncActionState, error) {
	req := aasc.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodAsyncActionStateCheckerGetState.String()

	reqData := struct {
		ActionGUID string `json:"wstrActionGuid"`
	}{
		ActionGUID: actionGUID,
	}

	req.Body = reqData

	resp := asyncActionState{}

	err := aasc.client.do(ctx, req, &resp)
	if err != nil {
		return resp, fmt.Errorf("AsyncActionStateChecker::checkActionState(): %w", err)
	}

	if resp.Error.Code != 0 {
		return resp, fmt.Errorf("AsyncActionStateChecker::checkActionState(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp, nil
}
