package ksc

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
)

type ChunkAccessor service

const (
	apiMethodChunkAccessorGetItemsChunk apiMethod = "/api/v1.0/ChunkAccessor.GetItemsChunk"
	apiMethodChunkAccessorRelease       apiMethod = "/api/v1.0/ChunkAccessor.chunkAccessorRelease"
)

func (ca *ChunkAccessor) GetItemsChunk(ctx context.Context, accessorID string, start int, count int, out interface{}) (quantity int, err error) {
	var raw []json.RawMessage
	raw, quantity, err = ca.getItemsChunk(ctx, accessorID, start, count)
	if err != nil {
		err = fmt.Errorf("ChunkAccessor::GetItemsChunk(): %w", err)
		return
	}

	switch outType := out.(type) {
	case *[]HostInfo:
		var host klHostParams
		for _, item := range raw {
			err = json.Unmarshal(item, &host)
			if err != nil {
				err = fmt.Errorf("ChunkAccessor::GetItemsChunk(): unmarshal data: %w", err)
				return
			}
			var hostInfo HostInfo
			hostInfo, err = host.toHostInfo()
			if err != nil {
				err = fmt.Errorf("ChunkAccessor::GetItemsChunk(): cast data: %w", err)
				return
			}
			*outType = append(*outType, hostInfo)
		}
	default:
		err = fmt.Errorf("ChunkAccessor::GetItemsChunk(): unsupported type %T", outType)
	}

	return
}

func (ca *ChunkAccessor) getItemsChunk(ctx context.Context, accessorID string, start int, count int) (chunk []json.RawMessage, quantity int, err error) {
	req := ca.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodChunkAccessorGetItemsChunk.String()

	reqData := struct {
		Accessor string `json:"strAccessor"`
		Start    int    `json:"nStart"`
		Count    int    `json:"nCount"`
	}{
		Accessor: accessorID,
		Start:    start,
		Count:    count,
	}

	resp := struct {
		Chunk struct {
			Array []struct {
				Type  string          `json:"type"`
				Value json.RawMessage `json:"value"`
			} `json:"KLCSP_ITERATOR_ARRAY"`
		} `json:"pChunk"`
		Quantity int      `json:"PxgRetVal"`
		Error    kscError `json:"PxgError"`
	}{}

	req.Body = reqData

	err = ca.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("ChunkAccessor::getItemsChunk(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("ChunkAccessor::getItemsChunk(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	quantity = resp.Quantity

	for _, item := range resp.Chunk.Array {
		chunk = append(chunk, item.Value)
	}

	return
}

func (ca *ChunkAccessor) Release(ctx context.Context, accessorID string) error {
	err := ca.release(ctx, accessorID)
	if err != nil {
		err = fmt.Errorf("ChunkAccessor::Release(): %w", err)
	}

	return err
}

func (ca *ChunkAccessor) release(ctx context.Context, accessorID string) error {
	req := ca.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodChunkAccessorRelease.String()

	reqData := struct {
		Accessor string `json:"strAccessor"`
	}{
		Accessor: accessorID,
	}

	req.Body = reqData

	var resp struct {
		Error kscError `json:"PxgError"`
	}

	err := ca.client.do(ctx, req, &resp)
	if err != nil {
		return fmt.Errorf("ChunkAccessor::release(): %w", err)
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("ChunkAccessor::release(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}
