package ksc

import (
	"context"
	"crypto/tls"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/go-resty/resty/v2"
)

const (
	BasicAuthentication    AuthType = 1
	TokenAuthentication    AuthType = 2
	WebTokenAuthentication AuthType = 3
	GatewayAuthentication  AuthType = 4
)

type AuthType int

func (at AuthType) String() string {
	switch at {
	case BasicAuthentication:
		return "BasicAuthentication"
	case TokenAuthentication:
		return "KSCTokenAuthentication"
	case WebTokenAuthentication:
		return "WebTokenAuthentication"
	case GatewayAuthentication:
		return "GatewayAuthentication"
	default:
		return fmt.Sprintf("UnknownType%d", int(at))
	}
}

const apiMethodLogin apiMethod = "/api/v1.0/login"

type apiMethod string

func (am apiMethod) String() string {
	return string(am)
}

type service struct {
	client *Client
}

type Config struct {
	Server       string
	UserName     string
	Password     string
	Domain       string
	InternalUser bool
	VServerName  string
	XKscSession  bool
	Debug        bool
}

type Client struct {
	AsyncActionStateChecker *AsyncActionStateChecker
	ChunkAccessor           *ChunkAccessor
	HostGroup               *HostGroup
	HostTagsAPI             *HostTagsAPI
	ListTags                *ListTags
	Policy                  *Policy
	PolicyProfile           *PolicyProfile
	ServerHierarchy         *ServerHierarchy
	Session                 *Session
	SrvView                 *SrvView
	SsContents              *SsContents
	UserName                string
	Password                string
	Domain                  string
	InternalUser            bool
	Server                  string
	VServerName             string
	XKscSessionToken        string
	XKscSession             bool
	client                  *resty.Client
	common                  service
	Debug                   bool
}

func New(cfg Config) *Client {
	httpClient := resty.New()

	c := &Client{
		client:       httpClient,
		Server:       cfg.Server,
		UserName:     base64str(cfg.UserName),
		Password:     base64str(cfg.Password),
		Domain:       base64str(cfg.Domain),
		InternalUser: cfg.InternalUser,
		VServerName:  base64str(cfg.VServerName),
		XKscSession:  cfg.XKscSession,
		Debug:        cfg.Debug,
	}

	c.common.client = c
	c.AsyncActionStateChecker = (*AsyncActionStateChecker)(&c.common)
	c.ChunkAccessor = (*ChunkAccessor)(&c.common)
	c.HostGroup = (*HostGroup)(&c.common)
	c.HostTagsAPI = (*HostTagsAPI)(&c.common)
	c.ListTags = (*ListTags)(&c.common)
	c.Policy = (*Policy)(&c.common)
	c.PolicyProfile = (*PolicyProfile)(&c.common)
	c.ServerHierarchy = (*ServerHierarchy)(&c.common)
	c.Session = (*Session)(&c.common)
	c.SrvView = (*SrvView)(&c.common)
	c.SsContents = (*SsContents)(&c.common)
	c.client.SetBaseURL(c.Server)
	c.client.SetHeader("User-Agent", "kscapi")
	c.client.SetHeader("Content-Type", "application/json")
	c.client.Header.Set("Accept-Encoding", "gzip")
	return c
}

func (c *Client) SetTLSClientConfig(config *tls.Config) *Client {
	c.client.SetTLSClientConfig(config)
	return c
}

func (c *Client) SetRootCertificate(pemFilePath string) *Client {
	c.client.SetRootCertificate(pemFilePath)
	return c
}

func (c *Client) SetRootCertificateFromString(pemContent string) *Client {
	c.client.SetRootCertificateFromString(pemContent)
	return c
}

func (c *Client) SetTransport(transport http.RoundTripper) *Client {
	c.client.SetTransport(transport)
	return c
}

func (c *Client) SetRetryCount(count int) *Client {
	c.client.RetryCount = count
	return c
}

func (c *Client) SetRetryWaitTime(waitTime time.Duration) *Client {
	c.client.RetryWaitTime = waitTime
	return c
}

func (c *Client) SetRetryMaxWaitTime(maxWaitTime time.Duration) *Client {
	c.client.RetryMaxWaitTime = maxWaitTime
	return c
}

func (c *Client) SetRetryAfter(callback resty.RetryAfterFunc) *Client {
	c.client.RetryAfter = callback
	return c
}

func (c *Client) AddRetryCondition(condition resty.RetryConditionFunc) *Client {
	c.client.RetryConditions = append(c.client.RetryConditions, condition)
	return c
}

func (c *Client) AddRetryAfterErrorCondition() *Client {
	c.client.AddRetryAfterErrorCondition()
	return c
}

func (c *Client) do(ctx context.Context, req *resty.Request, out interface{}) error {
	if ctx == nil {
		return fmt.Errorf("do(): context must be non-nil")
	}

	req.SetContext(ctx)
	if c.XKscSession && c.XKscSessionToken != "" {
		req.SetHeader("X-KSC-Session", c.XKscSessionToken)
	}

	req.SetHeader("User-Agent", "kscapi")
	req.SetHeader("Content-Type", "application/json")
	req.SetHeader("Accept-Encoding", "gzip")
	resp, err := req.Send()

	if err != nil {
		select {
		case <-ctx.Done():
			return fmt.Errorf("do(): ctx is done: %w", ctx.Err())
		default:
		}

		return fmt.Errorf("do(): send request %q: %w", req.URL, err)
	}

	if resp.StatusCode() != http.StatusOK {
		return fmt.Errorf("do(): %q response code %d: %s", req.URL, resp.StatusCode(), resp.Status())
	}

	//fmt.Printf("[Debug] body: %s\n", resp.Body())
	err = json.Unmarshal(resp.Body(), out)
	if err != nil {
		return fmt.Errorf("do(): unmarshal response %q: %w", resp.Body(), err)
	}

	return nil
}

func (c *Client) Login(ctx context.Context, authType AuthType, token string) (err error) {
	switch authType {
	case BasicAuthentication:
		err = c.basicAuth(ctx)
	case TokenAuthentication:
		err = c.kscTAuth(ctx, token)
	case WebTokenAuthentication:
		err = c.kscWTAuth(ctx, token)
	case GatewayAuthentication:
		err = c.kscGwAuth(ctx, token)
	default:
		err = c.basicAuth(ctx)
	}

	if err != nil {
		return fmt.Errorf("Login(): %w", err)
	}

	return nil
}

func (c *Client) basicAuth(ctx context.Context) (err error) {
	if c.XKscSession {
		err = c.xkscSession(ctx)
	} else {
		err = c.kscAuth(ctx)
	}

	if err != nil {
		return fmt.Errorf("basicAuth(): %w", err)
	}

	return nil
}

func (c *Client) kscAuth(ctx context.Context) error {
	req := c.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodLogin.String()

	var auth string
	if c.InternalUser {
		auth = fmt.Sprintf(`KSCBasic user="%s", pass="%s", internal="1"`,
			c.UserName, c.Password)
	} else {
		auth = fmt.Sprintf(`KSCBasic user="%s", pass="%s", domain="%s", internal="0"`,
			c.UserName, c.Password, c.Domain)
	}

	req.SetHeader("Authorization", auth)
	if c.VServerName != "" {
		req.SetHeader("X-KSC-VServer", c.VServerName)
	}

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}
	err := c.do(ctx, req, &resp)
	if err != nil {
		return fmt.Errorf("kscAuth(): %w", err)
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("kscAuth(): return code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}

func (c *Client) xkscSession(ctx context.Context) error {
	sessionID, err := c.Session.startSession(ctx)

	if err != nil {
		return fmt.Errorf("xkscSession(): %w", err)
	}

	if sessionID != "" {
		c.XKscSessionToken = sessionID
	}

	return nil
}

func (c *Client) kscTAuth(ctx context.Context, token string) error {
	req := c.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodLogin.String()

	req.SetHeader("Authorization", fmt.Sprintf("KSCT %s", token))

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}
	err := c.do(ctx, req, &resp)
	if err != nil {
		return fmt.Errorf("kscTAuth(): %w", err)
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("kscTAuth(): return code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}

func (c *Client) kscWTAuth(ctx context.Context, token string) error {
	req := c.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodLogin.String()

	req.SetHeader("Authorization", fmt.Sprintf("KSCWT %s", token))

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}
	err := c.do(ctx, req, &resp)
	if err != nil {
		return fmt.Errorf("kscWTAuth(): %w", err)
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("kscWTAuth(): return code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}

func (c *Client) kscGwAuth(ctx context.Context, token string) error {
	req := c.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodLogin.String()

	req.SetHeader("Authorization", fmt.Sprintf("KSCGW %s", token))

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}
	err := c.do(ctx, req, &resp)
	if err != nil {
		return fmt.Errorf("kscGwAuth(): %w", err)
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("kscGwAuth(): return code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}

func base64str(s string) string {
	return base64.StdEncoding.EncodeToString([]byte(s))
}
