package ksc

import (
	"fmt"
)

type GlobalUsersListViewAttribute string

func (attr GlobalUsersListViewAttribute) String() string {
	return string(attr)
}

func (attr GlobalUsersListViewAttribute) Attribute() string {
	return string(attr)
}

const (
	KLUserTrusteeID               GlobalUsersListViewAttribute = "ul_llTrusteeId"            // unique account ID within users list. Supported by server 10SP1 and higher
	KLUserTrusteeIDOrig           GlobalUsersListViewAttribute = "ul_llTrusteeIdOrig"        // for alias users: original account ID. Supported by server 10SP2 and higher
	KLUserVServer                 GlobalUsersListViewAttribute = "ul_nVServer"               // virtual server ID. Supported by server 10SP1 and higher
	KLUserBinID                   GlobalUsersListViewAttribute = "ul_binId"                  // account ID, pair ("KLUserVServer", "KLUserBinID") uniquely identifies user
	KLUserIsUser                  GlobalUsersListViewAttribute = "ul_bUser"                  // user/group indicator
	KLUserIsLocal                 GlobalUsersListViewAttribute = "ul_Local"                  // local account indicator
	KLUserIsBuiltin               GlobalUsersListViewAttribute = "ul_Builtin"                // builtin account indicator
	KLUserIsInternalUser          GlobalUsersListViewAttribute = "ul_InternalUser"           // internal account indicator
	KLUserSID                     GlobalUsersListViewAttribute = "ul_imgSid"                 // SID
	KLUserSIDHash                 GlobalUsersListViewAttribute = "ul_binSidHash"             // MD5 SID hash
	KLUserDistinguishedName       GlobalUsersListViewAttribute = "ul_wstrDistringuishedName" // distinguished name
	KLUserObjectGUID              GlobalUsersListViewAttribute = "ul_strObjectGUID"          // object GUID
	KLUserDisplayName             GlobalUsersListViewAttribute = "ul_wstrDisplayName"        // display name
	KLUserPrincipalName           GlobalUsersListViewAttribute = "ul_wstrPrincipalName"      // principal name
	KLUserSamAccountName          GlobalUsersListViewAttribute = "ul_wstrSamAccountName"     // NT4-style (SAM) name
	KLUserMail                    GlobalUsersListViewAttribute = "ul_wstrMail"               // email, filled in AD
	KLUserMailExtra               GlobalUsersListViewAttribute = "ul_wstrMailExtra"          // email, may be filled by administrator
	KLUserTelephoneNumber         GlobalUsersListViewAttribute = "ul_wstrTelephoneNumber"    // phone number, filled in AD
	KLUserTelephoneExtra          GlobalUsersListViewAttribute = "ul_wstrTelephoneExtra"     // phone number, may be filled by administrator
	KLUserMobile                  GlobalUsersListViewAttribute = "ul_wstrMobile"             // mobile phone number
	KLUserDepartment              GlobalUsersListViewAttribute = "ul_wstrDepartment"         // user's department
	KLUserCompany                 GlobalUsersListViewAttribute = "ul_wstrCompany"            // user's company name
	KLUserManager                 GlobalUsersListViewAttribute = "ul_wstrManager"            // user's manager display name
	KLUserDomainSAM               GlobalUsersListViewAttribute = "ul_wstrDomainSAM"          // NT4-style (SAM) domain name
	KLUserDomain2000              GlobalUsersListViewAttribute = "ul_wstrDomain2000"         // DNS-style domain name
	KLUserDescription             GlobalUsersListViewAttribute = "ul_wstrDescription"        // description
	KLUserGroupID                 GlobalUsersListViewAttribute = "ul_nGroupId"               // group ID (for groups only)
	KLUserVServerName             GlobalUsersListViewAttribute = "ul_wstrVServerName"        // virtual server display name
	KLUserUserID                  GlobalUsersListViewAttribute = "ul_nSplUserId"             // ID of internal user or group
	KLUserInternalUserDescription GlobalUsersListViewAttribute = "ul_nSplUserDesc"           // description of internal user or group
	KLUserDisabled                GlobalUsersListViewAttribute = "ul_bDisabled"              // disabled flag of internal user
	KLUserVisited                 GlobalUsersListViewAttribute = "ul_bVisited"               // host, specified by 'strHostId' parameter, was visited by user
	KLUserFtsField                GlobalUsersListViewAttribute = "ul_wstrFtsField"           // search-only field for full-text search May be included only as a filter.
	// Following fields are included in search:
	// - ul_wstrDisplayName
	// - ul_wstrPrincipalName
	// - ul_wstrSamAccountName
	// - ul_wstrMail
	// - ul_wstrMailExtra
	// - ul_wstrTelephoneNumber
	// - ul_wstrTelephoneExtra
	// - ul_wstrMobile
	// - ul_wstrDepartment
	// - ul_wstrCompany
	// - ul_wstrManager
	// - ul_wstrDomainSAM
	// - ul_wstrDomain2000
	// - ul_wstrDistringuishedName
)

type globalUsersListView struct {
	TrusteeID         kscLong   `json:"ul_llTrusteeId"`
	TrusteeIDOrig     kscLong   `json:"ul_llTrusteeIdOrig"`
	VirtualServer     int       `json:"ul_nVServer"`
	BinID             kscBinary `json:"ul_binId"`
	IsUser            bool      `json:"ul_bUser"`
	IsLocal           bool      `json:"ul_Local"`
	IsBuiltin         bool      `json:"ul_Builtin"`
	IsInternalUser    bool      `json:"ul_InternalUser"`
	SID               kscBinary `json:"ul_imgSid"`
	SIDHash           kscBinary `json:"ul_binSidHash"`
	DistinguishedName string    `json:"ul_wstrDistringuishedName"`
	ObjectGUID        string    `json:"ul_strObjectGUID"`
	DisplayName       string    `json:"ul_wstrDisplayName"`
	PrincipalName     string    `json:"ul_wstrPrincipalName"`
	SAMAccountName    string    `json:"ul_wstrSamAccountName"`
	Mail              string    `json:"ul_wstrMail"`
	MailExtra         string    `json:"ul_wstrMailExtra"`
	TelephoneNumber   string    `json:"ul_wstrTelephoneNumber"`
	TelephoneExtra    string    `json:"ul_wstrTelephoneExtra"`
	Mobile            string    `json:"ul_wstrMobile"`
	Department        string    `json:"ul_wstrDepartment"`
	Company           string    `json:"ul_wstrCompany"`
	Manager           string    `json:"ul_wstrManager"`
	DomainSAM         string    `json:"ul_wstrDomainSAM"`
	Domain2000        string    `json:"ul_wstrDomain2000"`
	Description       string    `json:"ul_wstrDescription"`
	GroupID           int       `json:"ul_nGroupId"`
	VirtualServerName string    `json:"ul_wstrVServerName"`
	UserID            int       `json:"ul_nSplUserId"`
	UserDescription   string    `json:"ul_nSplUserDesc"`
	IsDisabled        bool      `json:"ul_bDisabled"`
	Visited           bool      `json:"ul_bVisited"`
}

type GlobalUsersListView struct {
	TrusteeID         int64
	TrusteeIDOrig     int64
	VirtualServer     int
	BinID             string
	IsUser            bool
	IsLocal           bool
	IsBuiltin         bool
	IsInternalUser    bool
	SID               string
	SIDHash           string
	DistinguishedName string
	ObjectGUID        string
	DisplayName       string
	PrincipalName     string
	SAMAccountName    string
	Mail              string
	MailExtra         string
	TelephoneNumber   string
	TelephoneExtra    string
	Mobile            string
	Department        string
	Company           string
	Manager           string
	DomainSAM         string
	Domain2000        string
	Description       string
	GroupID           int
	VirtualServerName string
	UserID            int
	UserDescription   string
	IsDisabled        bool
	Visited           bool
}

func (view *globalUsersListView) toGlobalUsersListView() (GlobalUsersListView, error) {
	info := GlobalUsersListView{
		TrusteeID:         view.TrusteeID.Value,
		TrusteeIDOrig:     view.TrusteeIDOrig.Value,
		VirtualServer:     view.VirtualServer,
		IsUser:            view.IsUser,
		IsLocal:           view.IsLocal,
		IsBuiltin:         view.IsBuiltin,
		IsInternalUser:    view.IsInternalUser,
		DistinguishedName: view.DistinguishedName,
		ObjectGUID:        view.ObjectGUID,
		DisplayName:       view.DisplayName,
		PrincipalName:     view.PrincipalName,
		SAMAccountName:    view.SAMAccountName,
		Mail:              view.Mail,
		MailExtra:         view.MailExtra,
		TelephoneNumber:   view.TelephoneNumber,
		TelephoneExtra:    view.TelephoneExtra,
		Mobile:            view.Mobile,
		Department:        view.Department,
		Company:           view.Company,
		Manager:           view.Manager,
		DomainSAM:         view.DomainSAM,
		Domain2000:        view.Domain2000,
		Description:       view.Description,
		GroupID:           view.GroupID,
		VirtualServerName: view.VirtualServerName,
		UserID:            view.UserID,
		UserDescription:   view.UserDescription,
		IsDisabled:        view.IsDisabled,
		Visited:           view.Visited,
	}

	var err error
	info.BinID, err = view.BinID.toHexString()
	if err != nil {
		err = fmt.Errorf("globalUsersListView::toGlobalUsersListView(): cast %T to %T (decode BinID): %w", view, info, err)
		return info, err
	}

	info.SID, err = view.SID.toHexString()
	if err != nil {
		err = fmt.Errorf("globalUsersListView::toGlobalUsersListView(): cast %T to %T (decode SID): %v", view, info, err)
		return info, err
	}

	info.SIDHash, err = view.SIDHash.toHexString()
	if err != nil {
		err = fmt.Errorf("globalUsersListView::toGlobalUsersListView(): cast %T to %T (decode SIDHash): %v", view, info, err)
		return info, err
	}

	return info, nil
}

type GlobalUsersListViewParams struct {
	HostID      string
	Top         int
	UseDistinct bool
}

func (params *GlobalUsersListViewParams) viewParams() viewParams {
	return globalUsersListViewParams{
		HostID:      params.HostID,
		Top:         params.Top,
		UseDistinct: params.UseDistinct,
	}
}

type globalUsersListViewParams struct {
	HostID      string `json:"strHostId,omitempty"`
	Top         int    `json:"TOP_N,omitempty"`
	UseDistinct bool   `json:"USE_DISTINCT,omitempty"`
}
