package ksc

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"time"
)

type HostGroup service

const (
	apiMethodHostGroupGetHostInfo                   apiMethod = "/api/v1.0/HostGroup.GetHostInfo"
	apiMethodHostGroupGetDomains                    apiMethod = "/api/v1.0/HostGroup.GetDomains"
	apiMethodHostGroupGetStaticInfo                 apiMethod = "/api/v1.0/HostGroup.GetStaticInfo"
	apiMethodHostGroupFindHosts                     apiMethod = "/api/v1.0/HostGroup.FindHosts"
	apiMethodHostGroupFindHostsAsync                apiMethod = "/api/v1.0/HostGroup.FindHostsAsync"
	apiMethodHostGroupFindHostsAsyncGetAccessor     apiMethod = "/api/v1.0/HostGroup.FindHostsAsyncGetAccessor"
	apiMethodHostGroupMoveHostsToGroup              apiMethod = "/api/v1.0/HostGroup.MoveHostsToGroup"
	apiMethodHostGroupRemoveHost                    apiMethod = "/api/v1.0/HostGroup.RemoveHost"
	apiMethodHostGroupRemoveGroup                   apiMethod = "/api/v1.0/HostGroup.RemoveGroup"
	apiMethodHostGroupRemoveHosts                   apiMethod = "/api/v1.0/HostGroup.RemoveHosts"
	apiMethodHostGroupGroupIDGroups                 apiMethod = "/api/v1.0/HostGroup.GroupIdGroups"
	apiMethodHostGroupGroupIDUnassigned             apiMethod = "/api/v1.0/HostGroup.GroupIdUnassigned"
	apiMethodHostGroupGroupIDSuper                  apiMethod = "/api/v1.0/HostGroup.GroupIdSuper"
	apiMethodHostGroupGetSubgroups                  apiMethod = "/api/v1.0/HostGroup.GetSubgroups"
	apiMethodHostGroupGetGroupInfoEx                apiMethod = "/api/v1.0/HostGroup.GetGroupInfoEx"
	apiMethodHostGroupGetHostProducts               apiMethod = "/api/v1.0/HostGroup.GetHostProducts"
	apiMethodHostGroupGetComponentsForProductOnHost apiMethod = "/api/v1.0/HostGroup.GetComponentsForProductOnHost"
)

type HostAttribute string

func (attr HostAttribute) String() string {
	return string(attr)
}

// List of host attributes
const (
	KLHostDisplayName               HostAttribute = "KLHST_WKS_DN"                        // host display name
	KLHostGroupID                   HostAttribute = "KLHST_WKS_GROUPID"                   // ID of administration group where host is located, host may be moved by changing value of this attribute
	KLHostCreated                   HostAttribute = "KLHST_WKS_CREATED"                   // Time of host record creation, read-only
	KLHostLastVisible               HostAttribute = "KLHST_WKS_LAST_VISIBLE"              // Last host visibility time
	KLHostLastInfoUpdate            HostAttribute = "KLHST_WKS_LAST_INFOUDATE"            // Last update host information time, read-only
	KLHostStatus                    HostAttribute = "KLHST_WKS_STATUS"                    // host status, bit set, read-only
	KLHostLastUpdate                HostAttribute = "KLHST_WKS_LAST_UPDATE"               // Last anti-virus update UTC time, read-only
	KLHostLastNetworkAgentConnected HostAttribute = "KLHST_WKS_LAST_NAGENT_CONNECTED"     // Last Network Agent connection UTC time, read-only
	KLHostKeepConnection            HostAttribute = "KLHST_WKS_KEEP_CONNECTION"           // connection between host and administration server will be persistent if this attribute is set to true
	KLHostID                        HostAttribute = "KLHST_WKS_HOSTNAME"                  // a unique server-generated string, read-only
	KLHostNetworkAgentID            HostAttribute = "KLHST_INSTANCEID"                    // a unique string, empty string if no Network Agent installed, read-only
	KLHostWinHostname               HostAttribute = "KLHST_WKS_WINHOSTNAME"               // host windows (NetBIOS) name
	KLHostWinDomain                 HostAttribute = "KLHST_WKS_WINDOMAIN"                 // name of NT domain or workgroup
	KLHostDNSDomain                 HostAttribute = "KLHST_WKS_DNSDOMAIN"                 // DNS suffix
	KLHostDNSName                   HostAttribute = "KLHST_WKS_DNSNAME"                   // DNS name without DNS suffix
	KLHostFQDN                      HostAttribute = "KLHST_WKS_FQDN"                      // host FQDN name
	KLHostComputerType              HostAttribute = "KLHST_WKS_CTYPE"                     // computer type, bit set
	KLHostPlatformType              HostAttribute = "KLHST_WKS_PTYPE"                     // platform type
	KLHostOSName                    HostAttribute = "KLHST_WKS_OS_NAME"                   // operating system name as a string
	KLHostOSVersionMajor            HostAttribute = "KLHST_WKS_OS_VER_MAJOR"              // major part of operating system version
	KLHostOSVersionMinor            HostAttribute = "KLHST_WKS_OS_VER_MINOR"              // minor part of operating system version
	KLHostLastFullScan              HostAttribute = "KLHST_WKS_LAST_FULLSCAN"             // UTC time of last full scan for viruses
	KLHostVirusCount                HostAttribute = "KLHST_WKS_VIRUS_COUNT"               // number of viruses found, read-only
	KLHostRTPState                  HostAttribute = "KLHST_WKS_RTP_STATE"                 // state of real-time protection, read-only
	KLHostRTPErrorCode              HostAttribute = "KLHST_WKS_RTP_ERROR_CODE"            // real-time protection error code, read-only
	KLHostComment                   HostAttribute = "KLHST_WKS_COMMENT"                   // comments
	KLHostIP                        HostAttribute = "KLHST_WKS_IP_LONG"                   // host ip address (in little-endian byte order)
	KLHostConnectionIP              HostAttribute = "KLHST_WKS_CONNECT_IP_LONG"           // "connection ip" address (in little-endian byte order), read-only
	KLHostFromUnassigned            HostAttribute = "KLHST_WKS_FROM_UNASSIGNED"           // true if host is located in "Unassigned computers" or its subgroup, read-only
	KLHostUncuredCount              HostAttribute = "KLHST_WKS_UNCURED_COUNT"             // number of uncured items in the TIF list, read-only
	KLHostStatusID                  HostAttribute = "KLHST_WKS_STATUS_ID"                 // extended status id, if it is different from OK see KLHostStatusMask attribute for reasons, read-only
	KLHostStatusMask                HostAttribute = "KLHST_WKS_STATUS_MASK"               // extended status mask, bit set, read-only
	KLHostStatusHSDP                HostAttribute = "KLHST_WKS_STATUS_HSDP"               // one or more HSDP-reasons if KLHostStatusMask has bit 18 set, read-only
	KLHostHasUpdateAgent            HostAttribute = "HST_HAS_UPDATE_AGENT"                // host has update agent, read-only
	KLHostNetworkAgentVersion       HostAttribute = "KLHST_WKS_NAG_VERSION"               // network agent build number in format A.B.C[.D], read-only
	KLHostAntiVirusVersion          HostAttribute = "KLHST_WKS_RTP_AV_VERSION"            // real-time protection build number in format A.B.C[.D], read-only
	KLHostAntiVirusBasesTime        HostAttribute = "KLHST_WKS_RTP_AV_BASES_TIME"         // anti-virus bases time, read-only
	KLHostLastSystemStart           HostAttribute = "KLHST_WKS_LAST_SYSTEM_START"         // last system start time, read-only
	KLHostRebootRequired            HostAttribute = "KLHST_WKS_RBT_REQUIRED"              // if reboot is required, read-only
	KLHostRebootRequestReason       HostAttribute = "KLHST_WKS_RBT_REQUEST_REASON"        // reboot request reasons mask, bit set, read-only
	KLHostRebootRequestTime         HostAttribute = "KLHST_WKS_RBT_REQUEST_TIME"          // reboot request time, read-only
	KLHostAppInfo                   HostAttribute = "KLHST_APP_INFO"                      // Info about products installed on the host, read-only, may be used in GetHostInfo method only
	KLHostOSSPVersionMajor          HostAttribute = "KLHST_WKS_OSSP_VER_MAJOR"            // service pack version major part (since KSC 10 SP1), read-only
	KLHostOSSPVersionMinor          HostAttribute = "KLHST_WKS_OSSP_VER_MINOR"            // service pack version minor part (since KSC 10 SP1), read-only
	KLHostCPUArch                   HostAttribute = "KLHST_WKS_CPU_ARCH"                  // CPU architecture from the operating system point of view (since KSC 10 SP1), read-only
	KLHostOSBuildNumber             HostAttribute = "KLHST_WKS_OS_BUILD_NUMBER"           // operating system version build number, read-only
	KLHostOSReleaseID               HostAttribute = "KLHST_WKS_OS_RELEASE_ID"             // operating system version release ID (for Windows 10), read-only
	KLHostNetworkAgentVersionID     HostAttribute = "KLHST_WKS_NAG_VER_ID"                // Network Agent version ID, read-only
	KLHostOwnerID                   HostAttribute = "KLHST_WKS_OWNER_ID"                  // owner ID, since KSC 10 SP2 can be used in GetHostInfo method, since KSC11 can be used in search, read-only
	KLHostOwnerIsCustom             HostAttribute = "KLHST_WKS_OWNER_IS_CUSTOM"           // if owner was changed via UpdateHost with KLHostCustomOwnerID, read-only
	KLHostCustomOwnerID             HostAttribute = "KLHST_WKS_CUSTOM_OWNER_ID"           // custom owner ID, for UpdateHost method only (since KSC 10 SP2), all-zero 16-byte identifier means 'clear', write-only
	KLHostProductTag                HostAttribute = "KLHST_WKS_PRODUCT_TAG_NAME"          // tag value assigned to the host by some product, read-only
	KLHostIncludedProductTag        HostAttribute = "KLHST_INCLUDED_WKS_PRODUCT_TAG_NAME" // this attribute is used to search by tag that is not assigned to the host by any product, search-only attribute
	KLHostHSDPReason                HostAttribute = "KLHST_WKS_HSDP_REASON"               // searching on the product-defined host status reason id, since KSC 10 SP2 MR1, search-only attribute
	KLHostHSDPAllReasons            HostAttribute = "KLHST_WKS_HSDP_ALLREASONS"           // searching on the product-defined host status reason id, since KSC 11, search-only attribute
	KLHostHSDPReasonID              HostAttribute = "KLHST_WKS_HSDP_REASON_ID"            // product-defined host status reason id provided by any product at the host, read-only
	KLHostAntiSpamStatus            HostAttribute = "KLHST_WKS_ANTI_SPAM_STATUS"          // anti-spam status (see Product component status), read-only
	KLHostDLPStatus                 HostAttribute = "KLHST_WKS_DLP_STATUS"                // DLP status (see Product component status), read-only
	KLHostCollabSrvsStatus          HostAttribute = "KLHST_WKS_COLLAB_SRVS_STATUS"        // collaboration servers protection status (see Product component status), read-only
	KLHostEmailAVStatus             HostAttribute = "KLHST_WKS_EMAIL_AV_STATUS"           // email antivirus status (see Product component status), read-only
	KLHostEDRStatus                 HostAttribute = "KLHST_WKS_EDR_STATUS"                // EDR component status (see Product component status), read-only
	KLHostNetworkLocationName       HostAttribute = "KLHST_WKS_NLA_NETWORK"               // network location name, read-only
	KLHostVMType                    HostAttribute = "HST_VM_TYPE"                         // virtual machine type, read-only
	KLHostVMVDI                     HostAttribute = "HST_VM_VDI"                          // dynamic virtual machine as a part of VDI, read-only
	KLHostManagedOtherServer        HostAttribute = "KLHST_MANAGED_OTHER_SERVER"          // it is known that the host is managed by some other Administration Server
	KLHostHSDPReasonsAll            HostAttribute = "KLHST_HSDP_REASONS_ALL"              // product-defined host statuses provided by any product at the host
	KLHostClusterID                 HostAttribute = "HST_CLUSTER_ID"                      // the cluster id if host is node of cluster, read-only
	KLHostIsCluster                 HostAttribute = "HST_ISCLUSTER"                       // the host is a cluster, read-only
	KLHostIsClusterNode             HostAttribute = "HST_ISCLUSTER_NODE"                  // the host is a cluster node, read-only
	KLHostClusterNodeWMISupported   HostAttribute = "KLHST_CLSTNODE_WMI_SUPPORTED"        // if True then the cluster type is 'Web and Messaging Integration' (WMI), read-only
	KLHostClusterWMISupported       HostAttribute = "KLHST_CLUSTER_WMI_SUPPORTED"         // the host is a cluster, and the cluster type is 'Web and Messaging Integration' (WMI), read-only
	KLHostClusterNodeWMIFullActive  HostAttribute = "KLHST_CLSTNODE_WMI_FULL_ACTIVE"      // if the host is a WMI cluster node working in the 'full mode' (True) or in the 'statistics mode' (False), read-only
	KLHostClusterWMIActiveID        HostAttribute = "KLHST_CLUSTER_WMI_FULL_ACTIVE_ID"    // string host name of current active wmi cluster node, read-only
	KLHostClusterWMIProduct         HostAttribute = "KLHST_CLUSTER_WMI_PRODUCT"           // product name, read-only
	KLHostClusterWMIVersion         HostAttribute = "KLHST_CLUSTER_WMI_VERSION"           // product version, read-only
	KLHostEncryptionState           HostAttribute = "HST_ENCRYPTION_STATE"                // host encryption state, read-only
	KLHostCloudHostID               HostAttribute = "KLHST_CLOUD_HOST_BINID"              // public cloud id of the host, read-only
	KLHostCloudHostPlatform         HostAttribute = "KLHST_CLOUD_HOST_PLATFORM"           // type of public cloud, read-only
	KLHostCloudHostType             HostAttribute = "KLHST_CLOUD_HOST_TYPE"               // public cloud type of the host, read-only
	KLHostWUALastSearchValid        HostAttribute = "WUA_LAST_SEARCH_VALID"               // if last WUA search date (KLHostWUALastSearch) is valid, read-only
	KLHostWUALastSearch             HostAttribute = "WUA_LAST_SEARCH_DATE"                // last WUA successful search UTC time truncated to 00:00, read-only
	KLHostWUASwitched               HostAttribute = "KLHST_WUA_SWITCHED"                  // if WUA has been switched to the Administration Server, read-only
	KLHostSlaveServerDN             HostAttribute = "KLSRVH_SRV_DN"                       // display name of the slave server where the found host is located, read-only
	KLHostVirtualServerDN           HostAttribute = "KLVSRV_DN"                           // virtual server display name where the found host is located, read-only
	KLHostVirtualServerID           HostAttribute = "KLVSRV_ID"                           // virtual server identifier where the found host is located, read-only
	KLHostGroupName                 HostAttribute = "name"                                // group name where the host is located, read-only
	KLHostComponentID               HostAttribute = "KLHST_PRCST_COMPONENT_ID"            // unique binary id of product component, search-only attribute
	KLHostIncludedComponentID       HostAttribute = "KLHST_PRCST_INCLUDED_COMPONENT_ID"   // include hosts where the component with the specified id, search-only attribute
	KLHostComponentStatus           HostAttribute = "KLHST_PRCST_COMPONENT_STATUS"        // component status, see product component status, search-only attribute
	KLHostComponentVersion          HostAttribute = "KLHST_PRCST_COMPONENT_VERSION"       // 64-bit integer, every 16 bits of which are interpreted as one of the 4-component version like 1.0.0.0, search-only attribute
	KLHostProductName               HostAttribute = "KLHST_PRCST_PRODUCT_NAME"            // the name of the product in which the component is implemented, search-only attribute
	KLHostProductVersion            HostAttribute = "KLHST_PRCST_PRODUCT_VERSION"         // the version of the product in which the component is implemented, search-only attribute
)

// Host status, bit set (KLHostStatus)
const (
	KLHostStatusVisible = 1 << iota // bit 0 set if host is visible,
	_
	KLHostStatusNAInstalled  // bit 2 set if Network Agent is installed
	KLHostStatusNAAlive      // bit 3 set if Network Agent is "alive"
	KLHostStatusRTPInstalled // bit 4 set if real-time protection is installed
	KLHostStatusTemporary    // bit 5 set if the computer has been temporarily switched into current server as a result od NLA profile switching
)

// Computer types, bit set (KLHostComputerType)
const (
	KLHostCTypeWorkstation         = 1 << iota // Workstation
	KLHostCTypeServer                          // Server
	KLHostCTypeMicrosoftSQLServer              // Any server running with Microsoft SQL Server
	KLHostCTypePDC                             // Primary domain controller
	KLHostCTypeBDC                             // Backup domain controller
	KLHostCTypeTimesourceServer                // Server running the Timesource service
	KLHostCTypeAFPServer                       // Apple File Protocol server
	KLHostCTypeNovellServer                    // Novell server
	KLHostCTypeDomainMember                    // Domain member
	KLHostCTypePrintServer                     // Server sharing print queue
	KLHostCTypeDialInServer                    // Server running dial-in service
	KLHostCTypeUnix                            // Some Unix
	KLHostCTypeWindows                         // Windows NT/2000/XP/2003/Vista workstation or server
	KLHostCTypeBSD                             // BSD or Windows for Workgroups (depending on KLHostPlatformType)
	KLHostCTypeNetWare                         // Microsoft File and Print for NetWare
	KLHostCTypeWindowsNT                       // Windows NT/2000/2003 server that is not a domain controller
	KLHostCTypeBrowserServer                   // Server that can run the browser service
	KLHostCTypeBackupBrowserServer             // Server running a browser service as backup
	KLHostCTypeMasterBrowserServer             // Server running the master browser service
	KLHostCTypeDomainMasterBrowser             // Server running the domain master browser
	KLHostCTypeOSF                             // OSF
	KLHostCTypeVMS                             // VMS or Linux (depending on KLHostPlatformType)
	KLHostCTypeWindows9x                       // Windows 9x/Me
	KLHostCTypeRootDFS                         // Root of a DFS tree
	KLHostCTypeNTCluster                       // NT Cluster
	KLHostCTypeIBMDSS                          // IBM DSS (Directory and Security Services) or equivalent
)

// Platform type, one of following constants (KLHostPlatformType)
const (
	KLHostPTypeMSDOS         = iota // MSDOS, Windows 3.xx, Windows 9x/Me
	KLHostPTypeOS2                  // OS/2
	KLHostPTypeWindows              // Windows NT/2000/XP/2003/Vista
	KLHostPTypeOSF                  // OSF
	KLHostPTypeVMS                  // VMS
	KLHostPTypeNovell               // Novell
	KLHostPTypeLinux                // Linux
	KLHostPTypeBSD                  // BSD
	KLHostPTypeWindowsMobile        // Windows Mobile
	KLHostPTypeSymbianMobile        // Symbian Mobile
	KLHostPTypeMacOS                // macOS
)

// State of real-time protection, possible values are listed below (KLHostRealTimeProtectionState)
const (
	KLHostRTPStateUnknown               = iota // Unknown
	KLHostRTPStateStopped                      // Stopped
	KLHostRTPStateSuspended                    // Suspended
	KLHostRTPStateStarting                     // Starting
	KLHostRTPStateRunning                      // Running (if anti-virus application does not support categories of state Running)
	KLHostRTPStateRunningMaxProtect            // Running with maximum protection
	KLHostRTPStateRunningMaxSpeed              // Running for maximum speed
	KLHostRTPStateRunningRecommended           // Running with recommended settings
	KLHostRTPStateRunningCustomSettings        // Running with custom settings
	KLHostRTPStateFailure                      // Failure
)

// Host general statuses (KLHostStatusID)
const (
	KLHostStatusOk       = 0
	KLHostStatusCritical = 1
	KLHostStatusWarning  = 2
)

// Possible host status descriptions in the extended status mask (KLHostStatusMask, KlHostStatStatusMask)
const (
	KLHostStatusNAInactive               = 1 << iota // bit 0 - Host with Network Agent installed is online but network agent is inactive
	KLHostStatusRTPNotRunningTooLong                 // bit 1 - Anti-virus application is installed but real-time protection is not running for more than KLSRV_HSTSTAT_SP_RPT_INACTIVE seconds
	KLHostStatusRTPNotRunning                        // bit 2 - Anti-virus application is installed but not running
	KLHostStatusTooMuchViruses                       // bit 3 - Number of viruses detected is too much (more than KLSRV_HSTSTAT_SP_VIRUSES)
	KLHostStatusRTPBadStatus                         // bit 4 - Anti-virus application is installed but real-time protection status differs from one set by the security administrator (see KLSRV_HSTSTAT_SP_RPT_DIFFERS)
	KLHostStatusAVNotInstall                         // bit 5 - Anti-virus application is not installed
	KLHostStatusFullScanTooLongAgo                   // bit 6 - Full scan for viruses performed too long ago (more than KLSRV_HSTSTAT_OLD_FSCAN days)
	KLHostStatusAVBasesOutdated                      // bit 7 - Anti-virus bases were updated too long ago (more than KLSRV_HSTSTAT_OLD_AV_BASES days)
	KLHostStatusNAInactiveTooLong                    // bit 8 - Network agent is inactive too long (more than KLSRV_HSTSTAT_OLD_LAST_CONNECT days)
	KLHostStatusOldLicense                           // bit 9 - Old license
	KLHostStatusUncuredTooMuch                       // bit 10 - Number of uncured objects is too much (more than KLSRV_HSTSTAT_SP_UNCURED)
	KLHostStatusRebootRequired                       // bit 11 - Reboot is required (for more than KLSRV_HSTSTAT_SP_BOOT_EXPIRED seconds, and the reason is one of the KLSRV_HSTSTAT_SP_BOOT_REASONS)
	KLHostStatusIncompatibleAppInstalled             // bit 12 - One or more incompatible applications are installed on the host
	KLHostStatusOneOrMoreVulnerabilities             // bit 13 - Host has one or more vulnerabilities (according to the KLSRV_HSTSTAT_SP_VULNERABILITY and KLSRV_HSTSTAT_SP_VULN_FLAGS)
	KLHostStatusSearchOSUpdatesTooLong               // bit 14 - Last search for operating system updates has been performed too long ago on the host (more than KLSRV_HSTSTAT_SP_UASEARCH days)
	KLHostStatusProperEncryptionStatus               // bit 15 - The host does not have proper encryption status (different from one of the KLSRV_HSTSTAT_SP_ENCRYPTION)
	KLHostStatusMobileDevicePolicy                   // bit 16 - Mobile device settings does not meet requirements of the security policy
	KLHostStatusUnhandledIncident                    // bit 17 - There are unhandled incidents
	KLHostStatusSuggestedByApp                       // bit 18 - Host status was suggested by the managed product (HSDP)
	KLHostStatusOutOfDiskSpace                       // bit 19 - Host is out of disk space, either synchronization errors occur or disk space is less than KLSRV_HSTSTAT_SP_DISKSPACE
)

// Reboot request reasons mask, bit set (KLHostRebootRequestReason, KLHostStatRebootReason)
const (
	KLHostRebootReqReasonUnspecified        = 1 << iota // bit 0 - Reason is unspecified
	KLHostRebootReqReasonAppUnusable                    // bit 1 - Application is unusable until reboot
	KLHostRebootReqReasonCompleteUpdate                 // bit 2 - Application is usable but reboot is required to complete update process
	KLHostRebootReqReasonInitiateUpdate                 // bit 3 - Reboot is required to initiate update process
	KLHostRebootReqReasonCompleteCuring                 // bit 4 - Reboot is required to complete scanning/curing
	KLHostRebootReqReasonCompleteInstall                // bit 5 - Reboot is required to complete installation/uninstallation
	KLHostRebootReqReasonCompleteEncryption             // bit 6 - Reboot needed to complete encryption process
)

// CPU architecture (KLHostCPUArch) from the operating system point of view (since KSC 10 SP1), following values are possible
const (
	KLHostCPUArchUnknown = iota // 0 - CPU architecture is unknown
	KLHostCPUArchX86            // 1 - CPU architecture is x86
	KLHostCPUArchX64            // 2 - CPU architecture is x64
)

// Network Agent version ID (KLHostNetworkAgentVersionID), some possible values are listed below:
const (
	KLHostNetworkAgentVersionIDKSC10MR1    = 0x000A0100 // Kaspersky Security Center 10 MR1
	KLHostNetworkAgentVersionIDKSC10SP1    = 0x000A1000 // Kaspersky Security Center 10 SP1
	KLHostNetworkAgentVersionIDKSC10SP2    = 0x000A2000 // Kaspersky Security Center 10 SP2
	KLHostNetworkAgentVersionIDKSC10SP2MR1 = 0x000A4000 // Kaspersky Security Center 10 SP2 MR1
	KLHostNetworkAgentVersionIDKSC10SP3    = 0x000A5000 // Kaspersky Security Center 10 SP3
	KLHostNetworkAgentVersionIDKSC11       = 0x000B0000 // Kaspersky Security Center 11
)

type ProductComponentStatus int

// Product component status, possible values:
const (
	ProductComponentStatusUnknown      ProductComponentStatus = 0  // 0 Status 'Unknown'
	ProductComponentStatusStopped      ProductComponentStatus = 1  // 1 Status 'Stopped'
	ProductComponentStatusPaused       ProductComponentStatus = 2  // 2 Status 'Paused'
	ProductComponentStatusStarting     ProductComponentStatus = 3  // 3 Status 'Starting'
	ProductComponentStatusRunning      ProductComponentStatus = 4  // 4 Status 'Running'
	ProductComponentStatusFailed       ProductComponentStatus = 9  // 9 Status 'Failed'
	ProductComponentStatusNotInstalled ProductComponentStatus = 10 // 10 Status 'Not installed'
	ProductComponentStatusNoLicense    ProductComponentStatus = 11 // 11 Status 'No License'
)

func (pcs ProductComponentStatus) String() string {
	switch pcs {
	case ProductComponentStatusUnknown:
		return "Unknown"
	case ProductComponentStatusStopped:
		return "Stopped"
	case ProductComponentStatusPaused:
		return "Paused"
	case ProductComponentStatusStarting:
		return "Starting"
	case ProductComponentStatusRunning:
		return "Running"
	case ProductComponentStatusFailed:
		return "Failed"
	case ProductComponentStatusNotInstalled:
		return "Not installed"
	case ProductComponentStatusNoLicense:
		return "No License"
	default:
		return fmt.Sprintf("UnknownStatus%d", int(pcs))
	}
}

// Product component well-known component IDs
const (
	KLHostProductAntiSpam            = "0x59C431F07D5C4633AC7BFCB569A2B394" // Spam protection (Anti-Spam)
	KLHostProductDLP                 = "0x952E730B74394BB8A3DC0005FF9896DF" // Data Leakage Prevention (DLP)
	KLHostProductCollaborationServer = "0xEC63B07296A940C59BDF8AA641D9FC21" // Collaboration servers protection
	KLHostProductEmailAntiVirus      = "0xAD73C0B0F9C04982A6BED48998E7BB91" // Email Anti-virus
	KLHostProductEDR                 = "0x7275439F21EE4C799A1A6EF02CF53920" // Endpoint Sensor (EDR)
)

// Virtual machine type (KLHostVMType), possible values:
const (
	KLHostVMTypeUndefined  = iota // 0 – We don't know whether virtualization is used or not
	KLHostVMTypeVM                // 1 - Some virtual machine (we don't know which one)
	KLHostVMTypeRealHW            // 2 - Real hardware
	KLHostVMTypeVMWare            // 3 - VMWare
	KLHostVMTypeHyperV            // 4 - Microsoft Hyper-V
	KLHostVMTypeVirtualPC         // 5 - Microsoft VirtualPC (obsolete)
	KLHostVMTypeParallels         // 6 - Parallels
	KLHostVMTypeVirtualBox        // 7 - Sun VirtualBox (supported only for Windows guests yet)
	KLHostVMTypeXen               // 8 - Xen
	KLHostVMTypeKVM               // 9 - Red Hat KVM
)

// Host encryption state (KLHostEncryptionState), of following values:
const (
	KLHostEncryptionStateNone              = iota // ENCRYPTION_NONE (0) Host doesn't have any encryption rules
	KLHostEncryptionStateInProgress               // ENCRYPTION_IN_PROGRESS (1) Encryption process is running
	KLHostEncryptionStateCanceledByUser           // ENCRYPTION_CANCELED_BY_USER (2) Operation of encryption canceled by user
	KLHostEncryptionStateFailure                  // ENCRYPTION_FAILURE (3) Error was occurred in encryption process
	KLHostEncryptionStateDone                     // ENCRYPTION_DONE (4) All encryption rules has been processed for host or device
	KLHostEncryptionStateWaitReboot               // ENCRYPTION_IN_PROGRESS_WAIT_REBOOT (5) In progress of encryption and host is need to reboot
	KLHostEncryptionStateFilesWithoutRules        // ENCRYPTION_HAS_ENC_FIELS_WO_RULE (6) Host has encrypted files without any rules
)

// Type of public cloud (KLHostCloudHostPlatform), one of following values:
const (
	KLHostCloudHostPlatformAWS   = 1 + iota // 1 AWS
	KLHostCloudHostPlatformAzure            // 2 Azure
	KLHostCloudHostPlatformNone             // 3 the host is not located in the public cloud
)

// Public cloud type of the host (KLHostCloudHostType), one of following values:
const (
	KLHostCloudHostTypeWindows = 2 // 2 Windows
	KLHostCloudHostTypeLinux   = 6 // 6 Linux
)

// Info about products installed on the host. Has following format
// "KLHST_APP_INFO"
//    |
//    +--<product> (paramParams)
//        |
//        +--<version> (paramParams)
//            |
//            +--"KLHST_APP_STATE" (paramInt)
//            |
//            +--"KLHST_APP_STATE_TIME" (paramDateTime)
const (
	KLHostAppState     = "KLHST_APP_STATE"      // product state (zero means "Inactive", non-zero means "Active")
	KLHostAppStateTime = "KLHST_APP_STATE_TIME" // last time where the product state KLHostAppState has changed
)

type GroupAttribute string

func (attr GroupAttribute) String() string {
	return string(attr)
}

// List of group attributes
const (
	KLGroupID                      GroupAttribute = "id"                         // group id, read-only
	KLGroupName                    GroupAttribute = "name"                       // group may be renamed by changing the value of this attribute
	KLGroupParentID                GroupAttribute = "parentId"                   // ID of parent group, group may be moved by changing the value of this attribute
	KLGroupDevicesAutoRemovePeriod GroupAttribute = "autoRemovePeriod"           // maximum number of days of host inactivity before deleting inactive computer
	KLGroupDevicesNotifyPeriod     GroupAttribute = "notifyPeriod"               // maximum number of days of host inactivity before notification event will be published
	KLGroupCreationDate            GroupAttribute = "creationDate"               // UTC time of group creation, read-only
	KLGroupDevicesInherited        GroupAttribute = "KLGRP_HlfInherited"         // whether klGroupDevicesAutoRemovePeriod and klGroupDevicesNotifyPeriod attributes must be inherited from the parent group. Changing this attribute turns on and off the inheritance. The attribute can be changed only if KLGRP_HlfForced is False
	KLGroupDevicesForceChildren    GroupAttribute = "KLGRP_HlfForceChildren"     // whether klGroupDevicesInherited attribute is locked into the True state for all child groups at all levels. Changing this attribute changes klGroupDevicesInherited of child groups
	KLGroupDevicesForced           GroupAttribute = "KLGRP_HlfForced"            // whether the parent group has klGroupDevicesForceChildren set into True, read-only
	KLGroupLastUpdate              GroupAttribute = "lastUpdate"                 // UTC time of last modification of group attributes, read-only
	KLGroupHostsNumber             GroupAttribute = "hostsNum"                   // number of hosts in group and subgroups on all levels, see also klGroupChildHostsDirectly, read-only
	KLGroupChildGroupsNumber       GroupAttribute = "childGroupsNum"             // number of subgroups on all levels, see also klGroupChildGroupsDirectly, read-only
	KLGroupFullName                GroupAttribute = "grp_full_name"              // full name of group (includes names of all parents), read-only
	KLGroupLevel                   GroupAttribute = "level"                      // depth of the group in the groups tree
	KLGroupStatusCritical          GroupAttribute = "KLSRV_HSTSTAT_CRITICAL"     // settings for host status 'Critical' calculation
	KLGroupStatusWarning           GroupAttribute = "KLSRV_HSTSTAT_WARNING"      // Settings for host status 'Warning' calculation
	KLGroupGroupID                 GroupAttribute = "KLGRP_GRP_GROUPID_GP"       // ID of parent or a grandparent or a grand-...grandparent group, search-only attribute
	KLGroupAutoInstallPackageID    GroupAttribute = "c_grp_autoInstallPackageId" // array, each entry of type Integer is the ID of the installation package to install automatically on computers located in the group and all child subgroups
	KLGroupFromUnassigned          GroupAttribute = "grp_from_unassigned"        // this attribute has value 'True' for Groups that are located under the group 'Unassigned', read-only
	KLGroupEnableFullScan          GroupAttribute = "grp_enable_fscan"           // Enable full scan
	KLGroupSlaveServerDN           GroupAttribute = "KLSRVH_SRV_DN"              // display name of the slave server where the found group is located, read-only
	KLGroupVirtualServerID         GroupAttribute = "KLVSRV_ID"                  // virtual server ID where the found group is located, read-only
	KLGroupVirtualServerDN         GroupAttribute = "KLVSRV_DN"                  // virtual server display name where the found group is located, read-only
	KLGroupChildGroupsDirectly     GroupAttribute = "KLGRP_CHLDGRP_CNT"          // number of subgroups located directly in this group (not including those located in subgroups), read-only
	KLGroupChildHostsDirectly      GroupAttribute = "KLGRP_CHLDHST_CNT"          // number of hosts located directly in this group (not including those located in subgroups), read-only
	KLGroupChildHostsOK            GroupAttribute = "KLGRP_CHLDHST_CNT_OK"       // number of hosts without host status 'Critical' or 'Warning' located directly in this group (not including those located in subgroups), read-only
	KLGroupChildHostsCritical      GroupAttribute = "KLGRP_CHLDHST_CNT_CRT"      // number of hosts without host status 'Critical' or 'Warning' located directly in this group (not including those located in subgroups), read-only
	KLGroupChildHostsWarning       GroupAttribute = "KLGRP_CHLDHST_CNT_WRN"      // Number of hosts with host status 'Warning' located directly in this group (not including those located in subgroups), read-only
)

// Settings for host status calculation
const (
	KLHostStatInherited          = "KLSRV_HSTSTAT_INHERITED"        // status settings are inherited from parent
	KLHostStatInheritable        = "KLSRV_HSTSTAT_INHERITABLE"      // status settings may be inherited from parent, read-only
	KLHostStatLocked             = "KLSRV_HSTSTAT_LOCKED"           // status settings are locked by policy, read-only
	KLHostStatStatusMask         = "KLSRV_HSTSTAT_MASK"             // extended status mask, bit set
	KLHostStatRTPInactiveMaxTime = "KLSRV_HSTSTAT_SP_RPT_INACTIVE"  // maximum number of seconds RTP may stay in 'not running' state without switching out the host from the OK status
	KLHostStatVirusesMaxCount    = "KLSRV_HSTSTAT_SP_VIRUSES"       // maximum number of viruses
	KLHostStatRTPDiffers         = "KLSRV_HSTSTAT_SP_RPT_DIFFERS"   // Set of bits, each bit means allowable RTP state
	KLHostStatOldFullScan        = "KLSRV_HSTSTAT_OLD_FSCAN"        // maximum days ago full host scanning has been completed
	KLHostStatOldAVBases         = "KLSRV_HSTSTAT_OLD_AV_BASES"     // maximum days ago AV database has been released for the host
	KLHostStatOldLastConnect     = "KLSRV_HSTSTAT_OLD_LAST_CONNECT" // maximum days ago the host connected to the Administration Server
	KLHostStatOldLicense         = "KLSRV_HSTSTAT_OLD_LICENSE"      // maximum days before the license expires
	KLHostStatUncured            = "KLSRV_HSTSTAT_SP_UNCURED"       // maximum number of uncured objects
	KLHostStatRebootReason       = "KLSRV_HSTSTAT_SP_BOOT_REASONS"  // set of bits, each bit means reboot reason
	KLHostStatRebootExpired      = "KLSRV_HSTSTAT_SP_BOOT_EXPIRED"  // maximum number of seconds after system reboot has been requested
	KLHostStatVulnerability      = "KLSRV_HSTSTAT_SP_VULNERABILITY" // severity rating of the vulnerability
	KLHostStatVulnerabilityFlags = "KLSRV_HSTSTAT_SP_VULN_FLAGS"    // vulnerability extra properties
	KLHostStatUASearch           = "KLSRV_HSTSTAT_SP_UASEARCH"      // maximum days after the operating system updates search has been completed
	KLHostStatEncryption         = "KLSRV_HSTSTAT_SP_ENCRYPTION"    // a bit set, each bit is an allowable encryption state
	KLHostStatDiskSpace          = "KLSRV_HSTSTAT_SP_DISKSPACE"     // maximum number of free space in megabytes
)

// Set of bits, each bit means allowable RTP state (KlHostStatRTPDiffers), bits are numbered from zero:
const (
	KlHostStatRTPStopped           = 1<<iota + 16 // bit 16 - rtp stopped
	KlHostStatRTPSuspended                        // bit 17 - rtp suspended
	KlHostStatRTPStarting                         // bit 18 - rtp starting
	KlHostStatRTPNormalMode                       // bit 19 - rtp is running in normal mode
	KlHostStatRTPMaxProtectionMode                // bit 20 - rtp is running in 'max protection mode'
	KlHostStatRTPMaxSpeedMode                     // bit 21 - rtp is running in 'max speed mode'
	KlHostStatRTPRecommendedMode                  // bit 22 - rtp is running in 'recommended mode'
	KlHostStatRTPCustomMode                       // bit 23 - rtp is running in 'custom mode'
	KlHostStatRTPFailed                           // bit 24 - rtp failed to run
)

// Severity rating of the vulnerability (KLHostStatVulnerability), set of bits:
const (
	KLHostStatVulnerabilityStatusWarning  = 1 << iota // bit 0 - Warning
	KLHostStatVulnerabilityStatusHigh                 // bit 1 - High
	KLHostStatVulnerabilityStatusCritical             // bit 2 - Critical
)

// Vulnerability extra properties (KLHostStatVulnerabilityFlags), set of bits:
const (
	KLHostStatVulnerabilityPatchingImpossible = 1 << iota // bit 0 - patching of the the vulnerability is impossible
	KLHostStatVulnerabilityPatchAssigned                  // bit 1 - a patch to fix the vulnerability is assigned
)

// Encryption status (KLHostStatEncryption), a bit set, each bit is an allowable encryption state:
const (
	KLHostStatEncryptionNotSupported   = 1 << iota // bit 0 - Encryption is not supported
	KLHostStatEncryptionNoRules                    // bit 1 - No encryption rules
	KLHostStatEncryptionIsRunning                  // bit 2 - Encrypting process is running
	KLHostStatEncryptionCancelled                  // bit 3 - Encrypting process has been cancelled
	KLHostStatEncryptionProcessFailed              // bit 4 - Encrypting process failed
	KLHostStatEncryptionRebootRequired             // bit 5 - Reboot is required to complete encryption
)

// Flags for groups remove
const (
	DeleteEmptyGroup                           = 1
	DeleteGroupWithSubgroupsPoliciesTasks      = 2
	DeleteGroupWithSubgroupsPoliciesTasksHosts = 3
)

type klHostParams struct {
	ID                        string               `json:"KLHST_WKS_HOSTNAME"`
	KSCDisplayName            string               `json:"KLHST_WKS_DN"`
	GroupID                   int                  `json:"KLHST_WKS_GROUPID"`
	Created                   kscDateTime          `json:"KLHST_WKS_CREATED"`
	LastVisible               kscDateTime          `json:"KLHST_WKS_LAST_VISIBLE"`
	LastInfoUpdate            kscDateTime          `json:"KLHST_WKS_LAST_INFOUDATE"`
	Status                    int                  `json:"KLHST_WKS_STATUS"`
	LastUpdate                kscDateTime          `json:"KLHST_WKS_LAST_UPDATE"`
	LastNetworkAgentConnected kscDateTime          `json:"KLHST_WKS_LAST_NAGENT_CONNECTED"`
	KeepConnection            bool                 `json:"KLHST_WKS_KEEP_CONNECTION"`
	NetworkAgentID            string               `json:"KLHST_INSTANCEID"`
	WinHostname               string               `json:"KLHST_WKS_WINHOSTNAME"`
	WinDomain                 string               `json:"KLHST_WKS_WINDOMAIN"`
	DNSDomain                 string               `json:"KLHST_WKS_DNSDOMAIN"`
	DNSName                   string               `json:"KLHST_WKS_DNSNAME"`
	FQDN                      string               `json:"KLHST_WKS_FQDN"`
	ComputerType              int                  `json:"KLHST_WKS_CTYPE"`
	PlatformType              int                  `json:"KLHST_WKS_PTYPE"`
	OSName                    string               `json:"KLHST_WKS_OS_NAME"`
	OSVersionMajor            int                  `json:"KLHST_WKS_OS_VER_MAJOR"`
	OSVersionMinor            int                  `json:"KLHST_WKS_OS_VER_MINOR"`
	LastFullScan              kscDateTime          `json:"KLHST_WKS_LAST_FULLSCAN"`
	VirusCount                kscLong              `json:"KLHST_WKS_VIRUS_COUNT"`
	RTPState                  int                  `json:"KLHST_WKS_RTP_STATE"`
	RTPErrorCode              int                  `json:"KLHST_WKS_RTP_ERROR_CODE"`
	Comment                   string               `json:"KLHST_WKS_COMMENT"`
	IP                        kscLong              `json:"KLHST_WKS_IP_LONG"`
	ConnectionIP              kscLong              `json:"KLHST_WKS_CONNECT_IP_LONG"`
	FromUnassigned            bool                 `json:"KLHST_WKS_FROM_UNASSIGNED"`
	UncuredCount              kscLong              `json:"KLHST_WKS_UNCURED_COUNT"`
	StatusID                  int                  `json:"KLHST_WKS_STATUS_ID"`
	StatusMask                int                  `json:"KLHST_WKS_STATUS_MASK"`
	StatusHSDP                string               `json:"KLHST_WKS_STATUS_HSDP"`
	HasUpdateAgent            bool                 `json:"HST_HAS_UPDATE_AGENT"`
	NetworkAgentVersion       string               `json:"KLHST_WKS_NAG_VERSION"`
	AntiVirusVersion          string               `json:"KLHST_WKS_RTP_AV_VERSION"`
	AntiVirusBasesTime        kscDateTime          `json:"KLHST_WKS_RTP_AV_BASES_TIME"`
	LastSystemStart           kscDateTime          `json:"KLHST_WKS_LAST_SYSTEM_START"`
	RebootRequired            bool                 `json:"KLHST_WKS_RBT_REQUIRED"`
	RebootRequestReason       int                  `json:"KLHST_WKS_RBT_REQUEST_REASON"`
	RebootRequestTime         kscDateTime          `json:"KLHST_WKS_RBT_REQUEST_TIME"`
	AppsInfo                  klAppsInfo           `json:"KLHST_APP_INFO"`
	OSSPVersionMajor          int                  `json:"KLHST_WKS_OSSP_VER_MAJOR"`
	OSSPVersionMinor          int                  `json:"KLHST_WKS_OSSP_VER_MINOR"`
	CPUArch                   int                  `json:"KLHST_WKS_CPU_ARCH"`
	OSBuildNumber             int                  `json:"KLHST_WKS_OS_BUILD_NUMBER"`
	OSReleaseID               int                  `json:"KLHST_WKS_OS_RELEASE_ID"`
	NetworkAgentVersionID     int                  `json:"KLHST_WKS_NAG_VER_ID"`
	OwnerID                   kscBinary            `json:"KLHST_WKS_OWNER_ID"`
	OwnerIsCustom             bool                 `json:"KLHST_WKS_OWNER_IS_CUSTOM"`
	ProductTagName            string               `json:"KLHST_WKS_PRODUCT_TAG_NAME"`
	HSDPReasonID              kscBinary            `json:"KLHST_WKS_HSDP_REASON_ID"`
	AntiSpamStatus            int                  `json:"KLHST_WKS_ANTI_SPAM_STATUS"`
	DLPStatus                 int                  `json:"KLHST_WKS_DLP_STATUS"`
	CollabSrvsStatus          int                  `json:"KLHST_WKS_COLLAB_SRVS_STATUS"`
	EmailAVStatus             int                  `json:"KLHST_WKS_EMAIL_AV_STATUS"`
	EDRStatus                 int                  `json:"KLHST_WKS_EDR_STATUS"`
	VMType                    int                  `json:"HST_VM_TYPE"`
	VMVDI                     bool                 `json:"HST_VM_VDI"`
	ManagedOtherServer        bool                 `json:"KLHST_MANAGED_OTHER_SERVER"`
	HSDPReasonsAll            klHostHSDPReasonsAll `json:"KLHST_HSDP_REASONS_ALL"`
	ClusterID                 string               `json:"HST_CLUSTER_ID"`
	IsCluster                 bool                 `json:"HST_ISCLUSTER"`
	IsClusterNode             bool                 `json:"HST_ISCLUSTER_NODE"`
	ClusterNodeWMISupported   bool                 `json:"KLHST_CLSTNODE_WMI_SUPPORTED"`
	ClusterWMISupported       bool                 `json:"KLHST_CLUSTER_WMI_SUPPORTED"`
	ClusterNodeWMIFullActive  bool                 `json:"KLHST_CLSTNODE_WMI_FULL_ACTIVE"`
	ClusterNodeWMIActiveID    string               `json:"KLHST_CLUSTER_WMI_FULL_ACTIVE_ID"`
	ClusterWMIProductName     string               `json:"KLHST_CLUSTER_WMI_PRODUCT"`
	ClusterWMIProductVersion  string               `json:"KLHST_CLUSTER_WMI_VERSION"`
	EncryptionState           int                  `json:"HST_ENCRYPTION_STATE"`
	CloudHostID               kscBinary            `json:"KLHST_CLOUD_HOST_BINID"`
	CloudHostPlatform         int                  `json:"KLHST_CLOUD_HOST_PLATFORM"`
	CloudHostType             int                  `json:"KLHST_CLOUD_HOST_TYPE"`
	WUALastSearchValid        bool                 `json:"WUA_LAST_SEARCH_VALID"`
	WUALAstSearchDate         kscDateTime          `json:"WUA_LAST_SEARCH_DATE"`
	WUASwitched               bool                 `json:"KLHST_WUA_SWITCHED"`
	SlaveServerDN             string               `json:"KLSRVH_SRV_DN"`
	VirtualServerDN           string               `json:"KLVSRV_DN"`
	VirtualServerID           int                  `json:"KLVSRV_ID"`
	GroupName                 string               `json:"name"`
}

// List of KLHST_WKS_PRODUCT_NAME and KLHST_WKS_PRODUCT_VERSION values for some products
// Product                                                         "KLHST_WKS_PRODUCT_NAME" "KLHST_WKS_PRODUCT_VERSION"
// Kaspersky Administration Server                                 "1093"                   "1.0.0.0"
// Kaspersky Network Agent                                         "1103"                   "1.0.0.0"
// Kaspersky Anti-Virus 5.0 for Windows Workstation                "Workstation"            "5.0.0.0"
// Kaspersky Anti-Virus 5.0 for Windows File Server                "Fileserver"             "5.0.0.0"
// Kaspersky Anti-Virus 6.0 for Windows Workstation                "KAVWKS6"                "6.0.0.0"
// Kaspersky Anti-Virus 6.0 for Windows File Server                "KAVFS6"                 "6.0.0.0"
// Kaspersky Anti-Virus 6.0 for Windows Workstation MP1            "KAVWKS6"                "6.0.4.0"
// Kaspersky Anti-Virus 6.0 for Windows File Server MP1            "KAVFS6"                 "6.0.4.0"
// Kaspersky Anti-Virus 6.0 for Windows Servers Enterprise Edition "KAVFSEE"                "6.0.0.0"
// Kaspersky Anti-Virus 8.0 for Windows Servers Enterprise Edition "KAVFSEE"                "8.0.0.0"
// Kaspersky Anti-Virus 6.0 for Linux File Server                  "KLIN"                   "6.0.0.0"
// Kaspersky Enterprise Security 8.0                               "KES"                    "8.1.0.0"
// Kaspersky Enterprise Security 10.0                              "KES"                    "10.1.0.0"
// Kaspersky Enterprise Security 10.0 Maintenance Release 1        "KES"                    "10.2.1.0"
// Kaspersky Enterprise Security 10.0 Service Pack 1               "KES"                    "10.2.2.0"
// Kaspersky Security for Virtualization Light Agent 3.0           "KSVLA"                  "3.0.0.0"
// Kaspersky Endpoint Security 10 MR1                              "KES"                    "10.2.1.0"
// Kaspersky Endpoint Security 10 SP1                              "KES"                    "10.2.2.0"
// Kaspersky Endpoint Security 10 SP1 MR1                          "KES"                    "10.2.4.0"
// Kaspersky Endpoint Security 11                                  "KES"                    "11.0.0.0"
// Kaspersky Security 10.1 for Windows Server                      "WSEE"                   "10.1.0.0'

type klAppsInfo struct {
	Type  string                   `json:"type"`
	Value klAppsInfoProductVersion `json:"value"`
}

type klAppsInfoProductVersion map[string]struct {
	Type  string `json:"type"`
	Value map[string]struct {
		Type  string `json:"type"`
		Value struct {
			AppState     int         `json:"KLHST_APP_STATE"`
			AppStateTime kscDateTime `json:"KLHST_APP_STATE_TIME"`
		} `json:"value"`
	} `json:"value"`
}

type KLAppInfo struct {
	Name      string
	Version   string
	State     int
	StateTime time.Time
}

type klHostHSDPReasonsAll struct {
	Type  string `json:"type"`
	Value struct {
		Reasons []klHostHSDPReason `json:"KLHST_HSDP_REASONS_ALL"`
	} `json:"value"`
}

type klHostHSDPReason struct {
	Type  string `json:"type"`
	Value struct {
		HDSPInfo struct {
			Type  string `json:"type"`
			Value struct {
				StatusID   int            `json:"KLCONN_HSDP_STATUS"`
				StatusInfo []klHSDPStatus `json:"KLCONN_HSDP_STATUS_ARR"`
			} `json:"value"`
		} `json:"KLCONN_HSDP_INFO_1"`
		ProductName    string `json:"KLHST_WKS_PRODUCT_NAME"`
		ProductVersion string `json:"KLHST_WKS_PRODUCT_VERSION"`
	} `json:"value"`
}

type klHSDPStatus struct {
	Type  string `json:"type"`
	Value struct {
		Description string    `json:"KLCONN_HSDP_STATUS_REASON_DESC"`
		GUID        kscBinary `json:"KLCONN_HSDP_STATUS_REASON_GUID"`
	} `json:"value"`
}

type HSDPProductStatus struct {
	ProductName    string
	ProductVersion string
	StatusID       int
	Reasons        []HSDPStatus
}

type HSDPStatus struct {
	GUID        string
	Description string
}

type StaticInfoValue string

func (value StaticInfoValue) String() string {
	return string(value)
}

const (
	SVAdmServerCert                StaticInfoValue = "KLADMSRV_SERVER_CERT"                 // server certificate
	SVAdmServerKey                 StaticInfoValue = "KLADMSRV_SERVER_KEY"                  // server key (paramBinary)
	SVInstancePort                 StaticInfoValue = "InstancePort"                         // instance port (paramInt)
	SVAdmServerAddresses           StaticInfoValue = "KLADMSRV_SERVER_ADDRESSSES"           // array of server addresses that can be used by clients to connect to the administration server (paramArray|paramString)
	SVAdmServerSystemAccount       StaticInfoValue = "KLADMSRV_SERVER_UNDER_SYSTEM_ACCOUNT" // Name of account used by Administration Server (paramString)
	SVAdmServerOLACerts            StaticInfoValue = "KLADMSRV_OLA_ROOTCER_ACTUAL"          // Array of allowed OLA certificates, paramArray, each entry is a certificate as 'DER-encoded binary X.509' of type paramBinary
	SVAdmServerOLARootCertsRevoked StaticInfoValue = "KLADMSRV_OLA_ROOTCER_REVOKED"         // Array of disallowed (revoked) OLA certificates, paramArray, each entry is a certificate as 'DER-encoded binary X.509' of type paramBinary
)

type staticInfo struct {
	B2BCloudMode                       bool        `json:"KLADMSRV_B2B_CLOUD_MODE"`        // if the server installed in the 'B2B Cloud' mode
	SingleEventAverageSize             kscDouble   `json:"KLADMSRV_EV_EV_SIZE"`            // average size of a single event, Kb
	ForceSyncSupported                 bool        `json:"KLADMSRV_FORCE_SYNC_SUPPORTED"`  // force sync supported
	RootGroupID                        int         `json:"KLADMSRV_GRP_ROOT"`              // ID of group root
	SuperGroupID                       int         `json:"KLADMSRV_GRP_SUPER"`             // ID of group super
	UnassignedGroupID                  int         `json:"KLADMSRV_GRP_UNASSIGNED"`        // ID of group unassigned
	IsVirtual                          bool        `json:"KLADMSRV_IS_VIRTUAL"`            // true if server is virtual
	MaintenanceSupported               bool        `json:"KLADMSRV_MAINTENANCE_SUPPORTED"` // if maintenance task must is supported
	NACIsBeingUsed                     bool        `json:"KLADMSRV_NAC_IS_BEING_USED"`     // true if NAC is used
	NetworkAgentRunning                bool        `json:"KLADMSRV_NAGENT_RUNNING"`        // true if network agent is running
	NeedUNCPath                        bool        `json:"KLADMSRV_NEED_UNC_PATH"`         // if UNC path must be specified into backup task settings
	PublicCloudMode                    bool        `json:"KLADMSRV_PCLOUD_MODE"`           // if public cloud support mode is turned on
	ProductFullVersion                 string      `json:"KLADMSRV_PRODUCT_FULL_VERSION"`  // server product full version
	ProductName                        string      `json:"KLADMSRV_PRODUCT_NAME"`          // server product name
	ProductVersion                     string      `json:"KLADMSRV_PRODUCT_VERSION"`       // server product version
	ServerHostname                     string      `json:"KLADMSRV_SERVER_HOSTNAME"`       // server host name
	ServerVersionID                    int         `json:"KLADMSRV_SERVER_VERSION_ID"`     // server version ID
	SPLPasswordPolicyComplianceEnabled bool        `json:"KLADMSRV_SPL_PPC_ENABLED"`       // if password policy compliance for SPL users is enabled
	VirtualServerID                    int         `json:"KLADMSRV_VSID"`                  // VS ID
	VirtualServerUID                   string      `json:"KLADMSRV_VSUID"`                 // VS UID
	NwtworkSize                        int         `json:"KLSRV_NETSIZE"`                  // network size
	InstancePort                       int         `json:"InstancePort"`                   // instance port
	OLARootCertsActual                 []kscBinary `json:"KLADMSRV_OLA_ROOTCER_ACTUAL"`    // Array of allowed OLA certificates, each entry is a certificate as 'DER-encoded binary X.509'
	OLARootCertsRevoked                []kscBinary `json:"KLADMSRV_OLA_ROOTCER_REVOKED"`   // Array of disallowed (revoked) OLA certificates, each entry is a certificate as 'DER-encoded binary X.509'
	ServerAddresses                    []string    `json:"KLADMSRV_SERVER_ADDRESSSES"`     // array of server addresses that can be used by clients to connect to the administration server
	ServerCert                         kscBinary   `json:"KLADMSRV_SERVER_CERT"`           // server certificate
	ServerKey                          kscBinary   `json:"KLADMSRV_SERVER_KEY"`            // server key
}

type StaticInfo struct {
	B2BCloudMode                       bool
	SingleEventAverageSize             float64
	ForceSyncSupported                 bool
	RootGroupID                        int
	SuperGroupID                       int
	UnassignedGroupID                  int
	IsVirtual                          bool
	MaintenanceSupported               bool
	NACIsBeingUsed                     bool
	NetworkAgentRunning                bool
	NeedUNCPath                        bool
	PublicCloudMode                    bool
	ProductFullVersion                 string
	ProductName                        string
	ProductVersion                     string
	ServerHostname                     string
	ServerVersionID                    int
	SPLPasswordPolicyComplianceEnabled bool
	VirtualServerID                    int
	VirtualServerUID                   string
	NwtworkSize                        int
	InstancePort                       int
	OLARootCertsActual                 []string
	OLARootCertsRevoked                []string
	ServerAddresses                    []string
	ServerCert                         string
	ServerKey                          string
}

func (si *staticInfo) toStaticInfo() (StaticInfo, error) {
	info := StaticInfo{
		B2BCloudMode:                       si.B2BCloudMode,
		SingleEventAverageSize:             si.SingleEventAverageSize.Value,
		ForceSyncSupported:                 si.ForceSyncSupported,
		RootGroupID:                        si.RootGroupID,
		SuperGroupID:                       si.SuperGroupID,
		UnassignedGroupID:                  si.UnassignedGroupID,
		IsVirtual:                          si.IsVirtual,
		MaintenanceSupported:               si.MaintenanceSupported,
		NACIsBeingUsed:                     si.NACIsBeingUsed,
		NetworkAgentRunning:                si.NetworkAgentRunning,
		NeedUNCPath:                        si.NeedUNCPath,
		PublicCloudMode:                    si.PublicCloudMode,
		ProductFullVersion:                 si.ProductFullVersion,
		ProductName:                        si.ProductName,
		ProductVersion:                     si.ProductVersion,
		ServerHostname:                     si.ServerHostname,
		ServerVersionID:                    si.ServerVersionID,
		SPLPasswordPolicyComplianceEnabled: si.SPLPasswordPolicyComplianceEnabled,
		VirtualServerID:                    si.VirtualServerID,
		VirtualServerUID:                   si.VirtualServerUID,
		NwtworkSize:                        si.NwtworkSize,
		InstancePort:                       si.InstancePort,
		OLARootCertsActual:                 nil,
		OLARootCertsRevoked:                nil,
		ServerAddresses:                    si.ServerAddresses,
		ServerCert:                         "",
		ServerKey:                          "",
	}

	cert, err := si.ServerCert.toBytes()
	if err != nil {
		return info, fmt.Errorf("staticInfo toStaticInfo(): cast %T to %T (ID): %w", si.ServerCert, info.ServerCert, err)
	}
	info.ServerCert = string(cert)

	key, err := si.ServerKey.toBytes()
	if err != nil {
		return info, fmt.Errorf("staticInfo toStaticInfo(): cast %T to %T (ID): %w", si.ServerKey, info.ServerKey, err)
	}
	info.ServerKey = string(key)

	for _, crt := range si.OLARootCertsActual {
		cert, err := crt.toBytes()
		if err != nil {
			return info, fmt.Errorf("staticInfo toStaticInfo(): cast %T to %T (ID): %w", si.OLARootCertsActual, info.OLARootCertsActual, err)
		}
		info.OLARootCertsActual = append(info.OLARootCertsActual, string(cert))
	}

	for _, crt := range si.OLARootCertsRevoked {
		cert, err := crt.toBytes()
		if err != nil {
			return info, fmt.Errorf("staticInfo toStaticInfo(): cast %T to %T (ID): %w", si.OLARootCertsRevoked, info.OLARootCertsRevoked, err)
		}
		info.OLARootCertsRevoked = append(info.OLARootCertsRevoked, string(cert))
	}

	return info, nil
}

type productComponent struct {
	ID          kscBinary `json:"KLHST_PRCST_COMPONENT_ID"`
	Status      int       `json:"KLHST_PRCST_COMPONENT_STATUS"`
	Version     kscLong   `json:"KLHST_PRCST_COMPONENT_VERSION"`
	DisplayName string    `json:"KLHST_PRCST_COMPONENT_DN"`
}

type ComponentVersion struct {
	Major    uint16
	Minor    uint16
	Build    uint16
	Revision uint16
}

func (cv ComponentVersion) String() string {
	return fmt.Sprintf("%d.%d.%d.%d", cv.Major, cv.Minor, cv.Build, cv.Revision)
}

type ProductComponent struct {
	ID          KscHexID
	Status      ProductComponentStatus
	Version     ComponentVersion
	DisplayName string
}

func (pc *productComponent) toProductComponent() (prodComp ProductComponent, err error) {
	prodComp.Status = ProductComponentStatus(pc.Status)
	prodComp.Version = pc.Version.toComponentVersion()
	prodComp.DisplayName = pc.DisplayName
	prodComp.ID, err = pc.ID.toKscHexID()
	if err != nil {
		err = fmt.Errorf("productComponent toProductComponent(): cast %T to %T (ID): %w", pc, prodComp, err)
	}
	return
}

type klProductVersion map[string]struct {
	Type  string `json:"type"`
	Value map[string]struct {
		Type  string `json:"type"`
		Value struct {
			BaseDate               kscDateTime `json:"BaseDate"`
			BaseFolder             string      `json:"BaseFolder"`
			BaseInstallDate        kscDateTime `json:"BaseInstallDate"`
			BaseRecords            int         `json:"BaseRecords"`
			ConnDisplayName        string      `json:"ConnDisplayName"`
			ConnProdVersion        string      `json:"ConnProdVersion"`
			ConnectorComponentName string      `json:"ConnectorComponentName"`
			ConnectorFlags         int         `json:"ConnectorFlags"`
			ConnectorInstanceID    string      `json:"ConnectorInstanceID"`
			ConnectorVersion       string      `json:"ConnectorVersion"`
			CustomName             string      `json:"CustomName"`
			DataFolder             string      `json:"DataFolder"`
			DisplayName            string      `json:"DisplayName"`
			FileName               string      `json:"FileName"`
			FilePath               string      `json:"FilePath"`
			Folder                 string      `json:"Folder"`
			InstallTime            kscDateTime `json:"InstallTime"`
			InstallationID         string      `json:"InstallationID"`
			KeyFolder              string      `json:"KeyFolder"`
			LastUpdateTime         kscDateTime `json:"LastUpdateTime"`
			LocID                  string      `json:"LocID"`
			MngProduct             bool        `json:"MngProduct"`
			ModuleType             int         `json:"ModuleType"`
			MustLoadOutside        bool        `json:"KLCONNAPPINST_MUST_LOAD_OUTSIDE"`
			ProdVersion            string      `json:"ProdVersion"`
			Tasks                  []string    `json:"Tasks"`
			TasksComplemented      []string    `json:"TasksComplemented"`
		} `json:"value"`
	} `json:"value"`
}

func (klpv klProductVersion) toProductVersion() (list []ProductVersion, err error) {
	for product, versions := range klpv {
		for version, data := range versions.Value {
			pv := ProductVersion{
				BaseDate:               time.Time{},
				BaseFolder:             data.Value.BaseFolder,
				BaseInstallDate:        time.Time{},
				BaseRecords:            data.Value.BaseRecords,
				ComponentVersion:       data.Value.ProdVersion,
				ConnDisplayName:        data.Value.ConnDisplayName,
				ConnProdVersion:        data.Value.ConnProdVersion,
				ConnectorComponentName: data.Value.ConnectorComponentName,
				ConnectorFlags:         data.Value.ConnectorFlags,
				ConnectorInstanceID:    data.Value.ConnectorInstanceID,
				ConnectorVersion:       data.Value.ConnectorVersion,
				CustomName:             data.Value.CustomName,
				DataFolder:             data.Value.DataFolder,
				DisplayName:            data.Value.DisplayName,
				FileName:               data.Value.FileName,
				FilePath:               data.Value.FilePath,
				Folder:                 data.Value.Folder,
				InstallTime:            time.Time{},
				InstallationID:         data.Value.InstallationID,
				KeyFolder:              data.Value.KeyFolder,
				LastUpdateTime:         time.Time{},
				LocID:                  data.Value.LocID,
				MngProduct:             data.Value.MngProduct,
				ModuleType:             data.Value.ModuleType,
				MustLoadOutside:        data.Value.MustLoadOutside,
				ProductName:            product,
				Tasks:                  data.Value.Tasks,
				TasksComplemented:      data.Value.TasksComplemented,
				Version:                version,
			}

			pv.BaseDate, err = data.Value.BaseDate.toTime()
			if err != nil {
				err = fmt.Errorf("HostGroup::klProductVersion::toProductVersion(): cast %T to %T (BaseDate): %w", list, pv, err)
				return list, err
			}

			pv.BaseInstallDate, err = data.Value.BaseInstallDate.toTime()
			if err != nil {
				err = fmt.Errorf("HostGroup::klProductVersion::toProductVersion(): cast %T to %T (BaseInstallDate): %w", list, pv, err)
				return list, err
			}

			pv.InstallTime, err = data.Value.InstallTime.toTime()
			if err != nil {
				err = fmt.Errorf("HostGroup::klProductVersion::toProductVersion(): cast %T to %T (InstallTime): %w", list, pv, err)
				return list, err
			}

			pv.LastUpdateTime, err = data.Value.LastUpdateTime.toTime()
			if err != nil {
				err = fmt.Errorf("HostGroup::klProductVersion::toProductVersion(): cast %T to %T (LastUpdateTime): %w", list, pv, err)
				return list, err
			}

			list = append(list, pv)
		}
	}

	return list, nil
}

type ProductVersion struct {
	BaseDate               time.Time
	BaseFolder             string
	BaseInstallDate        time.Time
	BaseRecords            int
	ComponentVersion       string
	ConnDisplayName        string
	ConnProdVersion        string
	ConnectorComponentName string
	ConnectorFlags         int
	ConnectorInstanceID    string
	ConnectorVersion       string
	CustomName             string
	DataFolder             string
	DisplayName            string
	FileName               string
	FilePath               string
	Folder                 string
	InstallTime            time.Time
	InstallationID         string
	KeyFolder              string
	LastUpdateTime         time.Time
	LocID                  string
	MngProduct             bool
	ModuleType             int
	MustLoadOutside        bool
	ProductName            string
	Tasks                  []string
	TasksComplemented      []string
	Version                string
}

type HostInfo struct {
	ID                        string
	KSCDisplayName            string
	WinHostname               string
	WinDomain                 string
	DNSDomain                 string
	DNSName                   string
	FQDN                      string
	GroupID                   int
	GroupName                 string
	FromUnassigned            bool
	Created                   time.Time
	LastVisible               time.Time
	LastInfoUpdate            time.Time
	LastSystemStart           time.Time
	Status                    int
	StatusID                  int
	StatusMask                int
	StatusHSDP                string
	HSDPReasonID              string
	HSDPReasonsAll            []HSDPProductStatus
	ComputerType              int
	PlatformType              int
	OSName                    string
	OSVersionMajor            int
	OSVersionMinor            int
	OSSPVersionMajor          int
	OSSPVersionMinor          int
	OSBuildNumber             int
	OSReleaseID               int
	CPUArch                   int
	VMType                    int
	VMVDI                     bool
	ClusterID                 string
	IsCluster                 bool
	IsClusterNode             bool
	ClusterNodeWMISupported   bool
	ClusterWMISupported       bool
	ClusterNodeWMIFullActive  bool
	ClusterNodeWMIActiveID    string
	ClusterWMIProductName     string
	ClusterWMIProductVersion  string
	CloudHostID               string
	CloudHostPlatform         int
	CloudHostType             int
	IP                        net.IP
	ConnectionIP              net.IP
	NetworkAgentID            string
	NetworkAgentVersion       string
	NetworkAgentVersionID     int
	LastNetworkAgentConnected time.Time
	HasUpdateAgent            bool
	KeepConnection            bool
	AntiVirusVersion          string
	AntiVirusBasesTime        time.Time
	LastUpdate                time.Time
	LastFullScan              time.Time
	RTPState                  int
	RTPErrorCode              int
	VirusCount                int64
	UncuredCount              int64
	RebootRequired            bool
	RebootRequestReason       int
	RebootRequestTime         time.Time
	AppsInfo                  []KLAppInfo
	OwnerID                   string
	OwnerIsCustom             bool
	ProductTagName            string
	AntiSpamStatus            int
	DLPStatus                 int
	CollabSrvsStatus          int
	EmailAVStatus             int
	EDRStatus                 int
	EncryptionState           int
	ManagedOtherServer        bool
	WUALastSearchValid        bool
	WUALAstSearchDate         time.Time
	WUASwitched               bool
	SlaveServerDN             string
	VirtualServerDN           string
	VirtualServerID           int
	Comment                   string
}

type klGRPParams struct {
	ID                     int         `json:"id"`
	Name                   string      `json:"name"`
	ParentID               int         `json:"parentId"`
	DeviceAutoRemovePeriod int         `json:"autoRemovePeriod"`
	DeviceNotifyPeriod     int         `json:"notifyPeriod"`
	CreationDate           kscDateTime `json:"creationDate"`
	DeviceInherited        bool        `json:"KLGRP_HlfInherited"`
	DeviceForceChildren    bool        `json:"KLGRP_HlfForceChildren"`
	DeviceForced           bool        `json:"KLGRP_HlfForced"`
	LastUpdate             kscDateTime `json:"lastUpdate"`
	HostsNumber            int         `json:"hostsNum"`
	ChildGroupsNumber      int         `json:"childGroupsNum"`
	FullName               string      `json:"grp_full_name"`
	Level                  int         `json:"level"`
	HostsCriticalStats     struct {
		Type  string                `json:"type"`
		Value hostStatusCalculation `json:"value"`
	} `json:"KLSRV_HSTSTAT_CRITICAL"`
	HostsWarningStats struct {
		Type  string                `json:"type"`
		Value hostStatusCalculation `json:"value"`
	} `json:"KLSRV_HSTSTAT_WARNING"`
	AutoInstallPackageID          []int  `json:"c_grp_autoInstallPackageId"`
	FromUnassigned                bool   `json:"grp_from_unassigned"`
	EnableFullScan                bool   `json:"grp_enable_fscan"`
	SlaveServerDisplayName        string `json:"KLSRVH_SRV_DN"`
	VirtualServerID               int    `json:"KLVSRV_ID"`
	VirtualServerDisplayName      string `json:"KLVSRV_DN"`
	NumberOfSubgroupsDirectly     int    `json:"KLGRP_CHLDGRP_CNT"`
	NumberOfHostsDirectly         int    `json:"KLGRP_CHLDHST_CNT"`
	NumberOfHostsDirectlyOk       int    `json:"KLGRP_CHLDHST_CNT_OK"`
	NumberOfHostsDirectlyCritical int    `json:"KLGRP_CHLDHST_CNT_CRT"`
	NumberOfHostsDirectlyWarning  int    `json:"KLGRP_CHLDHST_CNT_WRN"`
}

type hostStatusCalculation struct {
	Inherited           bool `json:"KLSRV_HSTSTAT_INHERITED"`
	Inheritable         bool `json:"KLSRV_HSTSTAT_INHERITABLE"`
	Locked              bool `json:"KLSRV_HSTSTAT_LOCKED"`
	Mask                int  `json:"KLSRV_HSTSTAT_MASK"`
	RTPInactive         int  `json:"KLSRV_HSTSTAT_SP_RPT_INACTIVE"`
	MaxViruses          int  `json:"KLSRV_HSTSTAT_SP_VIRUSES"`
	RTPDiffers          int  `json:"KLSRV_HSTSTAT_SP_RPT_DIFFERS"`
	OldFullScan         int  `json:"KLSRV_HSTSTAT_OLD_FSCAN"`
	OldAVBases          int  `json:"KLSRV_HSTSTAT_OLD_AV_BASES"`
	OldLastConnect      int  `json:"KLSRV_HSTSTAT_OLD_LAST_CONNECT"`
	OldLicense          int  `json:"KLSRV_HSTSTAT_OLD_LICENSE"`
	Uncured             int  `json:"KLSRV_HSTSTAT_SP_UNCURED"`
	RebootReason        int  `json:"KLSRV_HSTSTAT_SP_BOOT_REASONS"`
	RebootExpired       int  `json:"KLSRV_HSTSTAT_SP_BOOT_EXPIRED"`
	VulnerabilityStatus int  `json:"KLSRV_HSTSTAT_SP_VULNERABILITY"`
	VulnerabilityFlags  int  `json:"KLSRV_HSTSTAT_SP_VULN_FLAGS"`
	UASearch            int  `json:"KLSRV_HSTSTAT_SP_UASEARCH"`
	Encryption          int  `json:"KLSRV_HSTSTAT_SP_ENCRYPTION"`
	DiskSpace           int  `json:"KLSRV_HSTSTAT_SP_DISKSPACE"`
}

type HostStatusCalculation struct {
	Inherited           bool
	Inheritable         bool
	Locked              bool
	Mask                int
	RTPInactive         int
	MaxViruses          int
	RTPDiffers          int
	OldFullScan         int
	OldAVBases          int
	OldLastConnect      int
	OldLicense          int
	Uncured             int
	RebootReason        int
	RebootExpired       int
	VulnerabilityStatus int
	VulnerabilityFlags  int
	UASearch            int
	Encryption          int
	DiskSpace           int
}

type GroupInfo struct {
	ID                            int
	Name                          string
	FullName                      string
	CreationDate                  time.Time
	ParentID                      int
	Level                         int
	FromUnassigned                bool
	LastUpdate                    time.Time
	HostsNumber                   int
	ChildGroupsNumber             int
	NumberOfSubgroupsDirectly     int
	NumberOfHostsDirectly         int
	NumberOfHostsDirectlyOk       int
	NumberOfHostsDirectlyCritical int
	NumberOfHostsDirectlyWarning  int
	DeviceAutoRemovePeriod        int
	DeviceNotifyPeriod            int
	DeviceInherited               bool
	DeviceForceChildren           bool
	DeviceForced                  bool
	HostsCriticalStats            HostStatusCalculation
	HostsWarningStats             HostStatusCalculation
	AutoInstallPackageID          []int
	EnableFullScan                bool
	SlaveServerDisplayName        string
	VirtualServerID               int
	VirtualServerDisplayName      string
}

type groupTree struct {
	Type  string `json:"type"`
	Value struct {
		ID           int         `json:"id"`
		Name         string      `json:"name"`
		PartOfADView bool        `json:"grp_part_of_ad_view_by_rule"`
		Subgroups    []groupTree `json:"groups"`
	} `json:"value"`
}

type GroupTree struct {
	ID           int
	Name         string
	PartOfADView bool
	Subgroups    []GroupTree
}

func (h *klHostParams) toHostInfo() (HostInfo, error) {
	var err error
	hostInfo := HostInfo{
		ID:                        h.ID,
		KSCDisplayName:            h.KSCDisplayName,
		WinHostname:               h.WinHostname,
		WinDomain:                 h.WinDomain,
		DNSDomain:                 h.DNSDomain,
		DNSName:                   h.DNSName,
		FQDN:                      h.FQDN,
		GroupID:                   h.GroupID,
		GroupName:                 h.GroupName,
		FromUnassigned:            h.FromUnassigned,
		Created:                   time.Time{},
		LastVisible:               time.Time{},
		LastInfoUpdate:            time.Time{},
		LastSystemStart:           time.Time{},
		Status:                    h.Status,
		StatusID:                  h.StatusID,
		StatusMask:                h.StatusMask,
		StatusHSDP:                h.StatusHSDP,
		HSDPReasonID:              "",
		HSDPReasonsAll:            nil,
		ComputerType:              h.ComputerType,
		PlatformType:              h.PlatformType,
		OSName:                    h.OSName,
		OSVersionMajor:            h.OSVersionMajor,
		OSVersionMinor:            h.OSVersionMinor,
		OSSPVersionMajor:          h.OSSPVersionMajor,
		OSSPVersionMinor:          h.OSSPVersionMinor,
		OSBuildNumber:             h.OSBuildNumber,
		OSReleaseID:               h.OSReleaseID,
		CPUArch:                   h.CPUArch,
		VMType:                    h.VMType,
		VMVDI:                     h.VMVDI,
		ClusterID:                 h.ClusterID,
		IsCluster:                 h.IsCluster,
		IsClusterNode:             h.IsClusterNode,
		ClusterNodeWMISupported:   h.ClusterNodeWMISupported,
		ClusterWMISupported:       h.ClusterWMISupported,
		ClusterNodeWMIFullActive:  h.ClusterNodeWMIFullActive,
		ClusterNodeWMIActiveID:    h.ClusterNodeWMIActiveID,
		ClusterWMIProductName:     h.ClusterWMIProductName,
		ClusterWMIProductVersion:  h.ClusterWMIProductVersion,
		CloudHostID:               h.CloudHostID.Value, // unknown format
		CloudHostPlatform:         h.CloudHostPlatform,
		CloudHostType:             h.CloudHostType,
		IP:                        nil,
		ConnectionIP:              nil,
		NetworkAgentID:            h.NetworkAgentID,
		NetworkAgentVersion:       h.NetworkAgentVersion,
		NetworkAgentVersionID:     h.NetworkAgentVersionID,
		LastNetworkAgentConnected: time.Time{},
		HasUpdateAgent:            h.HasUpdateAgent,
		KeepConnection:            h.KeepConnection,
		AntiVirusVersion:          h.AntiVirusVersion,
		AntiVirusBasesTime:        time.Time{},
		LastUpdate:                time.Time{},
		LastFullScan:              time.Time{},
		RTPState:                  h.RTPState,
		RTPErrorCode:              h.RTPErrorCode,
		VirusCount:                h.VirusCount.Value,
		UncuredCount:              h.UncuredCount.Value,
		RebootRequired:            h.RebootRequired,
		RebootRequestReason:       h.RebootRequestReason,
		RebootRequestTime:         time.Time{},
		AppsInfo:                  nil,
		OwnerID:                   "",
		OwnerIsCustom:             h.OwnerIsCustom,
		ProductTagName:            h.ProductTagName,
		AntiSpamStatus:            h.AntiSpamStatus,
		DLPStatus:                 h.DLPStatus,
		CollabSrvsStatus:          h.CollabSrvsStatus,
		EmailAVStatus:             h.EmailAVStatus,
		EDRStatus:                 h.EDRStatus,
		EncryptionState:           h.EncryptionState,
		ManagedOtherServer:        h.ManagedOtherServer,
		WUALastSearchValid:        h.WUALastSearchValid,
		WUALAstSearchDate:         time.Time{},
		WUASwitched:               h.WUASwitched,
		SlaveServerDN:             h.SlaveServerDN,
		VirtualServerDN:           h.VirtualServerDN,
		VirtualServerID:           h.VirtualServerID,
		Comment:                   h.Comment,
	}

	hostInfo.Created, err = h.Created.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (Created): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.LastVisible, err = h.LastVisible.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (LastVisible): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.LastInfoUpdate, err = h.LastInfoUpdate.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (LastInfoUpdate): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.LastSystemStart, err = h.LastSystemStart.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (LastSystemStart): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.LastNetworkAgentConnected, err = h.LastNetworkAgentConnected.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (LastNetworkAgentConnected): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.AntiVirusBasesTime, err = h.AntiVirusBasesTime.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (AntiVirusBasesTime): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.LastUpdate, err = h.LastUpdate.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (LastUpdate): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.LastFullScan, err = h.LastFullScan.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (LastFullScan): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.RebootRequestTime, err = h.RebootRequestTime.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (RebootRequestTime): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.WUALAstSearchDate, err = h.WUALAstSearchDate.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (WUALAstSearchDate): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.IP = h.IP.toIPv4()

	hostInfo.ConnectionIP = h.ConnectionIP.toIPv4()

	hostInfo.OwnerID, err = h.OwnerID.toHexString()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (decode OwnerID): %w", h, hostInfo, err)
		return hostInfo, err
	}

	hostInfo.HSDPReasonID, err = h.HSDPReasonID.toHexString()
	if err != nil {
		err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (decode reasonID): %w", h, hostInfo, err)
		return hostInfo, err
	}

	for _, product := range h.HSDPReasonsAll.Value.Reasons {
		productStatus := HSDPProductStatus{
			ProductName:    product.Value.ProductName,
			ProductVersion: product.Value.ProductVersion,
			StatusID:       product.Value.HDSPInfo.Value.StatusID,
			Reasons:        nil,
		}
		for _, status := range product.Value.HDSPInfo.Value.StatusInfo {
			reason := HSDPStatus{
				GUID:        "",
				Description: status.Value.Description,
			}
			reason.GUID, err = status.Value.GUID.toHexString()
			if err != nil {
				err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (decode HSDPReasonsAll.HDSPInfo.StatusInfo.GUID) : %w", h, hostInfo, err)
				return hostInfo, err
			}
			productStatus.Reasons = append(productStatus.Reasons, reason)
		}
		hostInfo.HSDPReasonsAll = append(hostInfo.HSDPReasonsAll, productStatus)
	}

	for app, appData := range h.AppsInfo.Value {
		for version, versionData := range appData.Value {
			var appInfo KLAppInfo
			appInfo.Name = app
			appInfo.Version = version
			appInfo.State = versionData.Value.AppState
			appInfo.StateTime, err = versionData.Value.AppStateTime.toTime()
			if err != nil {
				err = fmt.Errorf("HostGroup::klHostParams::toHostInfo(): cast %T to %T (AppInfo StatrTime): %w", h, hostInfo, err)
				return hostInfo, err
			}
			hostInfo.AppsInfo = append(hostInfo.AppsInfo, appInfo)
		}
	}

	return hostInfo, nil
}

func (hsc *hostStatusCalculation) toHostStatusCalculation() HostStatusCalculation {
	hostStatus := HostStatusCalculation{
		Inherited:           hsc.Inherited,
		Inheritable:         hsc.Inheritable,
		Locked:              hsc.Locked,
		Mask:                hsc.Mask,
		RTPInactive:         hsc.RTPInactive,
		MaxViruses:          hsc.MaxViruses,
		RTPDiffers:          hsc.RTPDiffers,
		OldFullScan:         hsc.OldFullScan,
		OldAVBases:          hsc.OldAVBases,
		OldLastConnect:      hsc.OldLastConnect,
		OldLicense:          hsc.OldLicense,
		Uncured:             hsc.Uncured,
		RebootReason:        hsc.RebootReason,
		RebootExpired:       hsc.RebootExpired,
		VulnerabilityStatus: hsc.VulnerabilityStatus,
		VulnerabilityFlags:  hsc.VulnerabilityFlags,
		UASearch:            hsc.UASearch,
		Encryption:          hsc.Encryption,
		DiskSpace:           hsc.DiskSpace,
	}

	return hostStatus
}

func (g *klGRPParams) toGroupInfo() (GroupInfo, error) {
	var err error
	info := GroupInfo{
		ID:                            g.ID,
		Name:                          g.Name,
		FullName:                      g.FullName,
		CreationDate:                  time.Time{},
		ParentID:                      g.ParentID,
		Level:                         g.Level,
		FromUnassigned:                g.FromUnassigned,
		LastUpdate:                    time.Time{},
		HostsNumber:                   g.HostsNumber,
		ChildGroupsNumber:             g.ChildGroupsNumber,
		NumberOfSubgroupsDirectly:     g.NumberOfSubgroupsDirectly,
		NumberOfHostsDirectly:         g.NumberOfHostsDirectly,
		NumberOfHostsDirectlyOk:       g.NumberOfHostsDirectlyOk,
		NumberOfHostsDirectlyCritical: g.NumberOfHostsDirectlyCritical,
		NumberOfHostsDirectlyWarning:  g.NumberOfHostsDirectlyWarning,
		DeviceAutoRemovePeriod:        g.DeviceAutoRemovePeriod,
		DeviceNotifyPeriod:            g.DeviceNotifyPeriod,
		DeviceInherited:               g.DeviceInherited,
		DeviceForceChildren:           g.DeviceForceChildren,
		DeviceForced:                  g.DeviceForced,
		HostsCriticalStats:            g.HostsCriticalStats.Value.toHostStatusCalculation(),
		HostsWarningStats:             g.HostsWarningStats.Value.toHostStatusCalculation(),
		AutoInstallPackageID:          g.AutoInstallPackageID,
		EnableFullScan:                g.EnableFullScan,
		SlaveServerDisplayName:        g.SlaveServerDisplayName,
		VirtualServerID:               g.VirtualServerID,
		VirtualServerDisplayName:      g.VirtualServerDisplayName,
	}

	info.CreationDate, err = g.CreationDate.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klGRPParams::toGroupInfo(): cast %T to %T (CreationDate): %w", g, info, err)
		return info, err
	}

	info.LastUpdate, err = g.LastUpdate.toTime()
	if err != nil {
		err = fmt.Errorf("HostGroup::klGRPParams::toGroupInfo(): cast %T to %T (LastUpdate): %w", g, info, err)
		return info, err
	}

	return info, err
}

func (gt *groupTree) toGroupTree() GroupTree {
	tree := GroupTree{
		ID:           gt.Value.ID,
		Name:         gt.Value.Name,
		PartOfADView: gt.Value.PartOfADView,
		Subgroups:    nil,
	}

	for _, subTree := range gt.Value.Subgroups {
		tree.Subgroups = append(tree.Subgroups, subTree.toGroupTree())
	}

	return tree
}

type HostSearchExtraParams struct {
	Depth             int
	FromCurrentVSOnly bool
}

func (params *HostSearchExtraParams) searchParams() hostSearchExtraParams {
	return hostSearchExtraParams{
		Depth:             params.Depth,
		FromCurrentVSOnly: params.FromCurrentVSOnly,
	}
}

type hostSearchExtraParams struct {
	Depth             int  `json:"KLSRVH_SLAVE_REC_DEPTH"`
	FromCurrentVSOnly bool `json:"KLGRP_FIND_FROM_CUR_VS_ONLY"`
}

type failedSlaveInfo struct {
	ID          int    `json:"KLSRVH_SRV_ID"`
	DisplayName string `json:"KLSRVH_SRV_DN"`
}

// *********************************************************************************************************************

func (hg *HostGroup) FindHosts(ctx context.Context, filter string, fields []HostAttribute, params HostSearchExtraParams, lifeTime int) (id string, quantity int, err error) {
	id, quantity, err = hg.findHosts(ctx, filter, fields, params.searchParams(), lifeTime)
	if err != nil {
		err = fmt.Errorf("HostGroup::FindHosts(): %w", err)
	}
	return
}

func (hg *HostGroup) findHosts(ctx context.Context, filter string, fields []HostAttribute, params hostSearchExtraParams, lifeTime int) (id string, quantity int, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupFindHosts.String()

	if len(fields) == 0 {
		err = fmt.Errorf("HostGroup::findHosts(): field list is empty")
		return
	}

	resID := struct {
		ID       string   `json:"strAccessor"`
		Quantity int      `json:"PxgRetVal"`
		Error    kscError `json:"PxgError"`
	}{}

	reqData := struct {
		Filter         string                `json:"wstrFilter"`
		FieldsToReturn []HostAttribute       `json:"vecFieldsToReturn"`
		ExtraParams    hostSearchExtraParams `json:"pParams"`
		MaxLifeTime    int                   `json:"lMaxLifeTime"`
	}{
		Filter:         filter,
		FieldsToReturn: fields,
		ExtraParams:    params,
		MaxLifeTime:    lifeTime,
	}

	req.Body = reqData

	err = hg.client.do(ctx, req, &resID)
	if err != nil {
		err = fmt.Errorf("HostGroup::findHosts(): %w", err)
		return
	}

	if resID.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::findHosts(): response code %d: %s", resID.Error.Code, resID.Error.Message)
		return
	}

	id = resID.ID
	quantity = resID.Quantity

	return
}

func (hg *HostGroup) FindHostsAsync(ctx context.Context, filter string, fields []HostAttribute, params HostSearchExtraParams, lifeTime int) (requestID string, err error) {
	requestID, err = hg.findHostsAsync(ctx, filter, fields, params.searchParams(), lifeTime)
	if err != nil {
		err = fmt.Errorf("HostGroup::FindHostsAsync(): %w", err)
	}
	return
}

func (hg *HostGroup) findHostsAsync(ctx context.Context, filter string, fields []HostAttribute, params hostSearchExtraParams, lifeTime int) (requestID string, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupFindHostsAsync.String()

	if len(fields) == 0 {
		err = fmt.Errorf("HostGroup::findHostsAsync(): field list is empty")
		return
	}

	resp := struct {
		ID    string   `json:"strRequestId"`
		Error kscError `json:"PxgError"`
	}{}

	reqData := struct {
		Filter         string                `json:"wstrFilter"`
		FieldsToReturn []HostAttribute       `json:"vecFieldsToReturn"`
		ExtraParams    hostSearchExtraParams `json:"pParams"`
		MaxLifeTime    int                   `json:"lMaxLifeTime"`
	}{
		Filter:         filter,
		FieldsToReturn: fields,
		ExtraParams:    params,
		MaxLifeTime:    lifeTime,
	}

	req.Body = reqData

	err = hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::findHostsAsync(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::findHostsAsync(): response code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	requestID = resp.ID

	return
}

func (hg *HostGroup) FindHostsAsyncGetAccessor(ctx context.Context, requestID string) (accessorID string, quantity int, err error) {
	accessorID, quantity, err = hg.findHostsAsyncGetAccessor(ctx, requestID)
	if err != nil {
		err = fmt.Errorf("HostGroup::FindHostsAsyncGetAccessor(): %w", err)
	}
	return
}

func (hg *HostGroup) findHostsAsyncGetAccessor(ctx context.Context, requestID string) (accessorID string, quantity int, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupFindHostsAsyncGetAccessor.String()

	resp := struct {
		ID           string `json:"strAccessor"`
		Quantity     int    `json:"PxgRetVal"`
		FailedSlaves struct {
			Servers []failedSlaveInfo `json:"KLGRP_FAILED_SLAVES_PARAMS"`
		} `json:"pFailedSlavesInfo"`
		Error kscError `json:"PxgError"`
	}{}

	reqData := struct {
		RequestID string `json:"strRequestId"`
	}{
		RequestID: requestID,
	}

	req.Body = reqData

	err = hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::findHostsAsyncGetAccessor(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::findHostsAsyncGetAccessor(): response code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	if len(resp.FailedSlaves.Servers) != 0 {
		message := "HostGroup::findHostsAsyncGetAccessor(): failed slave servers:"
		first := true
		for _, server := range resp.FailedSlaves.Servers {
			if first {
				first = false
				message = fmt.Sprintf("%s %s[%d]", message, server.DisplayName, server.ID)
			} else {
				message = fmt.Sprintf("%s, %s[%d]", message, server.DisplayName, server.ID)
			}
		}
		err = fmt.Errorf(message)
	}

	accessorID = resp.ID
	quantity = resp.Quantity

	return
}

func (hg *HostGroup) GetGroupInfoEx(ctx context.Context, id int, fields []GroupAttribute) (GroupInfo, error) {
	var info GroupInfo
	groupParams, err := hg.getGroupInfoEx(ctx, id, fields)
	if err != nil {
		err = fmt.Errorf("HostGroup::GetGroupInfoEx(): %w", err)
		return info, err
	}

	info, err = groupParams.toGroupInfo()
	if err != nil {
		err = fmt.Errorf("HostGroup::GetGroupInfoEx(): %w", err)
	}

	return info, err
}

func (hg *HostGroup) getGroupInfoEx(ctx context.Context, id int, fields []GroupAttribute) (klGRPParams, error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGetGroupInfoEx.String()

	groupInfo := struct {
		GroupInfo klGRPParams `json:"PxgRetVal"`
		Error     kscError    `json:"PxgError"`
	}{}

	reqData := struct {
		ID             int              `json:"nGroupId"`
		FieldsToReturn []GroupAttribute `json:"pArrAttributes"`
	}{
		ID:             id,
		FieldsToReturn: fields,
	}

	if len(fields) == 0 {
		err := fmt.Errorf("HostGroup::getGroupInfoEx(): field list is empty")
		return groupInfo.GroupInfo, err
	}

	req.Body = reqData

	err := hg.client.do(ctx, req, &groupInfo)
	if err != nil {
		err := fmt.Errorf("HostGroup::getGroupInfoEx(): %w", err)
		return groupInfo.GroupInfo, err
	}

	if groupInfo.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::getGroupInfoEx(): code %d: %s", groupInfo.Error.Code, groupInfo.Error.Message)
		return groupInfo.GroupInfo, err
	}

	return groupInfo.GroupInfo, nil
}

func (hg *HostGroup) GetComponentsForProductOnHost(ctx context.Context, hostID string, productName string, productVersion string) ([]ProductComponent, error) {
	prodComp, err := hg.getComponentsForProductOnHost(ctx, hostID, productName, productVersion)
	if err != nil {
		err = fmt.Errorf("HostGroup::GetComponentsForProductOnHost(): %w", err)
	}

	return prodComp, err
}

func (hg *HostGroup) getComponentsForProductOnHost(ctx context.Context, hostID string, productName string, productVersion string) ([]ProductComponent, error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGetComponentsForProductOnHost.String()

	var components []ProductComponent
	reqData := struct {
		HostID         string `json:"strHostName"`
		ProductName    string `json:"strProductName"`
		ProductVersion string `json:"strProductVersion"`
	}{
		HostID:         hostID,
		ProductName:    productName,
		ProductVersion: productVersion,
	}

	req.Body = reqData

	resp := struct {
		Items []struct {
			Type  string           `json:"type"`
			Value productComponent `json:"value"`
		} `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err := hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::getComponentsForProductOnHost(): %w", err)
		return components, err
	}

	if resp.Error.Code != 0 {
		return components, fmt.Errorf("HostGroup::getComponentsForProductOnHost(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	for _, item := range resp.Items {
		component, err := item.Value.toProductComponent()
		if err != nil {
			return components, fmt.Errorf("HostGroup::getComponentsForProductOnHost(): export components: %w", err)
		}
		components = append(components, component)
	}

	return components, nil
}

func (hg *HostGroup) GetHostInfo(ctx context.Context, hostID string, fields []HostAttribute) (hostInfo HostInfo, err error) {
	var klHostInfo klHostParams
	klHostInfo, err = hg.getHostInfo(ctx, hostID, fields)
	if err != nil {
		err = fmt.Errorf("HostGroup::GetHostInfo(): %w", err)
		return
	}

	hostInfo, err = klHostInfo.toHostInfo()
	if err != nil {
		err = fmt.Errorf("HostGroup::GetHostInfo(): %w", err)
	}

	return
}

func (hg *HostGroup) getHostInfo(ctx context.Context, hostID string, fields []HostAttribute) (hostInfo klHostParams, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGetHostInfo.String()

	if len(fields) == 0 {
		err = fmt.Errorf("HostGroup::getHostInfo(): field list is empty")
		return
	}

	resp := struct {
		HostInfo klHostParams `json:"PxgRetVal"`
		Error    kscError     `json:"PxgError"`
	}{}

	reqData := struct {
		HostName       string          `json:"strHostName"`
		FieldsToReturn []HostAttribute `json:"pFields2Return"`
	}{
		HostName:       hostID,
		FieldsToReturn: fields,
	}

	req.Body = reqData

	err = hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::getHostInfo(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::getHostInfo(): response code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	hostInfo = resp.HostInfo

	return
}

func (hg *HostGroup) GetHostProducts(ctx context.Context, hostID string) (pv []ProductVersion, err error) {
	products, err := hg.getHostProducts(ctx, hostID)

	if err != nil {
		err = fmt.Errorf("HostGroup::GetHostProducts(): %w", err)
		return
	}

	pv, err = products.toProductVersion()
	if err != nil {
		err = fmt.Errorf("HostGroup::GetHostProducts(): %w", err)
	}

	return
}

func (hg *HostGroup) getHostProducts(ctx context.Context, hostID string) (klProductVersion, error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGetHostProducts.String()

	reqData := struct {
		HostID string `json:"strHostName"`
	}{
		HostID: hostID,
	}

	req.Body = reqData

	resp := struct {
		RetValue klProductVersion `json:"PxgRetVal"`
		Error    kscError         `json:"PxgError"`
	}{}

	err := hg.client.do(ctx, req, &resp)
	if err != nil {
		err := fmt.Errorf("HostGroup::getHostProducts(): %w", err)
		return resp.RetValue, err
	}

	if resp.Error.Code != 0 {
		return resp.RetValue, fmt.Errorf("HostGroup::getHostProducts(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp.RetValue, nil
}

func (hg *HostGroup) getSubgroups(ctx context.Context, id int, depth int) ([]groupTree, error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGetSubgroups.String()

	resp := struct {
		RetValue []groupTree `json:"PxgRetVal"`
		Error    kscError    `json:"PxgError"`
	}{}

	reqData := struct {
		ID    int `json:"nGroupId"`
		Depth int `json:"nDepth"`
	}{
		id,
		depth,
	}

	req.Body = reqData

	err := hg.client.do(ctx, req, &resp)
	if err != nil {
		err := fmt.Errorf("HostGroup::getSubgroups(): %w", err)
		return resp.RetValue, err
	}

	if resp.Error.Code != 0 {
		return resp.RetValue, fmt.Errorf("HostGroup::getSubgroups(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp.RetValue, nil
}

func (hg *HostGroup) GetSubgroups(ctx context.Context, id int, depth int) ([]GroupTree, error) {
	var tree []GroupTree
	groups, err := hg.getSubgroups(ctx, id, depth)
	if err != nil {
		err = fmt.Errorf("HostGroup::GetSubgroups(): %w", err)
		return tree, err
	}

	for _, group := range groups {
		tree = append(tree, group.toGroupTree())
	}

	return tree, nil
}

func (hg *HostGroup) GroupIDGroups(ctx context.Context) (int, error) {
	id, err := hg.groupIDGroups(ctx)
	if err != nil {
		err = fmt.Errorf("HostGroup::GroupIDGroups(): %w", err)
	}

	return id, err
}

func (hg *HostGroup) groupIDGroups(ctx context.Context) (id int, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGroupIDGroups.String()

	resp := struct {
		ID    int      `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err = hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::groupIDGroups(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::groupIDGroups(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	id = resp.ID
	return
}

func (hg *HostGroup) GroupIDSuper(ctx context.Context) (int, error) {
	id, err := hg.groupIDSuper(ctx)
	if err != nil {
		err = fmt.Errorf("HostGroup::GroupIDSuper(): %w", err)
	}

	return id, err
}

func (hg *HostGroup) groupIDSuper(ctx context.Context) (id int, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGroupIDSuper.String()

	resp := struct {
		ID    int      `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err = hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::groupIDSuper(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::groupIDSuper(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	id = resp.ID
	return
}

func (hg *HostGroup) GroupIDUnassigned(ctx context.Context) (int, error) {
	id, err := hg.groupIDUnassigned(ctx)
	if err != nil {
		err = fmt.Errorf("HostGroup::GroupIDUnassigned(): %w", err)
	}

	return id, err
}

func (hg *HostGroup) groupIDUnassigned(ctx context.Context) (id int, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGroupIDUnassigned.String()

	resp := struct {
		ID    int      `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err = hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::groupIDUnassigned(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::groupIDUnassigned(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	id = resp.ID
	return
}

func (hg *HostGroup) MoveHostsToGroup(ctx context.Context, GroupID int, hosts []string) error {
	err := hg.moveHostsToGroup(ctx, GroupID, hosts)
	if err != nil {
		err = fmt.Errorf("HostGroup::MoveHostsToGroup(): %w", err)
	}

	return err
}

func (hg *HostGroup) moveHostsToGroup(ctx context.Context, groupID int, hosts []string) error {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupMoveHostsToGroup.String()

	reqData := struct {
		GroupID int      `json:"nGroup"`
		HostsID []string `json:"pHostNames"`
	}{
		GroupID: groupID,
		HostsID: hosts,
	}

	req.Body = reqData

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}

	err := hg.client.do(ctx, req, &resp)
	if err != nil {
		err := fmt.Errorf("HostGroup::moveHostsToGroup(): %w", err)
		return err
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("HostGroup::moveHostsToGroup(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}

func (hg *HostGroup) RemoveGroup(ctx context.Context, groupID int, flag int) (actionGUID string, err error) {
	actionGUID, err = hg.removeGroup(ctx, groupID, flag)
	if err != nil {
		err = fmt.Errorf("HostGroup::RemoveGroup(): %w", err)
	}

	return
}

func (hg *HostGroup) removeGroup(ctx context.Context, groupID int, flag int) (actionGUID string, err error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupRemoveGroup.String()

	reqData := struct {
		GroupID int `json:"nGroup"`
		Flag    int `json:"nFlags"`
	}{
		GroupID: groupID,
		Flag:    flag,
	}

	req.Body = reqData

	resp := struct {
		ActionGUID string   `json:"strActionGuid"`
		Error      kscError `json:"PxgError"`
	}{}

	err = hg.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("HostGroup::removeGroup(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("HostGroup::removeGroup(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	actionGUID = resp.ActionGUID

	return
}

func (hg *HostGroup) RemoveHost(ctx context.Context, hostID string) error {
	err := hg.removeHost(ctx, hostID)
	if err != nil {
		err = fmt.Errorf("HostGroup::RemoveHost(): %w", err)
	}

	return err
}

func (hg *HostGroup) removeHost(ctx context.Context, hostID string) error {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupRemoveHost.String()

	reqData := struct {
		HostID string `json:"strHostName"`
	}{
		HostID: hostID,
	}

	req.Body = reqData

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}

	err := hg.client.do(ctx, req, &resp)
	if err != nil {
		err := fmt.Errorf("HostGroup::removeHost(): %w", err)
		return err
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("HostGroup::removeHost(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}

func (hg *HostGroup) RemoveHosts(ctx context.Context, hostsID []string, force bool) error {
	err := hg.removeHosts(ctx, hostsID, force)
	if err != nil {
		err = fmt.Errorf("HostGroup::RemoveHosts(): %w", err)
	}

	return err
}

func (hg *HostGroup) removeHosts(ctx context.Context, hostsID []string, force bool) error {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupRemoveHosts.String()

	reqData := struct {
		HostsID      []string `json:"pHostNames"`
		ForceDestroy bool     `json:"bForceDestroy"`
	}{
		HostsID:      hostsID,
		ForceDestroy: force,
	}

	req.Body = reqData

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}

	err := hg.client.do(ctx, req, &resp)
	if err != nil {
		err := fmt.Errorf("HostGroup::removeHosts(): %w", err)
		return err
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("HostGroup::removeHosts(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}

func (hg *HostGroup) GetStaticInfo(ctx context.Context, values []StaticInfoValue) (si StaticInfo, err error) {
	var info staticInfo
	info, err = hg.getStaticInfo(ctx, values)
	if err != nil {
		err = fmt.Errorf("HostGroup::GetStaticInfo(): %w", err)
		return
	}

	si, err = info.toStaticInfo()
	if err != nil {
		err = fmt.Errorf("HostGroup::GetStaticInfo(): %w", err)
	}

	return
}

func (hg *HostGroup) getStaticInfo(ctx context.Context, values []StaticInfoValue) (staticInfo, error) {
	req := hg.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostGroupGetStaticInfo.String()

	reqData := struct {
		Values []StaticInfoValue `json:"pValues"`
	}{
		Values: values,
	}

	req.Body = reqData

	resp := struct {
		Info  staticInfo `json:"PxgRetVal"`
		Error kscError   `json:"PxgError"`
	}{}

	err := hg.client.do(ctx, req, &resp)
	if err != nil {
		err := fmt.Errorf("HostGroup::getStaticInfo(): %w", err)
		return resp.Info, err
	}

	if resp.Error.Code != 0 {
		return resp.Info, fmt.Errorf("HostGroup::getStaticInfo(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp.Info, nil
}
