package ksc

import (
	"fmt"
	"time"
)

type HwInvPCViewAttribute string

func (attr HwInvPCViewAttribute) String() string {
	return string(attr)
}

func (attr HwInvPCViewAttribute) Attribute() string {
	return string(attr)
}

// Configuration of managed device
const (
	KLDevHostID              HwInvPCViewAttribute = "KLHST_WKS_HOSTNAME"  // host ID
	KLDevID                  HwInvPCViewAttribute = "dev_id"              // device ID
	KLDevType                HwInvPCViewAttribute = "dev_type"            // device type
	KLDevName                HwInvPCViewAttribute = "dev_name"            // device type
	KLDevDescription         HwInvPCViewAttribute = "dev_descr"           // device description
	KLDevManufacturer        HwInvPCViewAttribute = "dev_manuf"           // device manufacturer
	KLDevDriverProvider      HwInvPCViewAttribute = "drv_prov"            // driver provider
	KLDevDriverVersion       HwInvPCViewAttribute = "drv_ver"             // driver version
	KLDevDriverDescription   HwInvPCViewAttribute = "DriverDescription"   // driver description
	KLDevDriverDate          HwInvPCViewAttribute = "DriverDate"          // driver release date
	KLDevXPar1               HwInvPCViewAttribute = "dev_xpar1"           // MAC of network adapter
	KLDevXPar2               HwInvPCViewAttribute = "dev_xpar2"           // storage size in bytes or speed of network adapter in bits per second or frequency of RAM MHz
	KLDevXPar3               HwInvPCViewAttribute = "dev_xpar3"           // storage free size in bytes or capacity of RAM in bytes
	KLDevBiosManufacturer    HwInvPCViewAttribute = "BiosManuf"           // mother board BIOS manufacturer
	KLDevBiosVersion         HwInvPCViewAttribute = "BiosVersion"         // mother board BIOS version
	KLDevBiosSerialNumber    HwInvPCViewAttribute = "BiosSerialNum"       // mother board BIOS serial number
	KLDevBiosReleaseDate     HwInvPCViewAttribute = "BiosReleaseDate"     // mother board BIOS release date
	KLDevChipset             HwInvPCViewAttribute = "Chipset"             // mother board chipset
	KLDevCPUSpeed            HwInvPCViewAttribute = "CpuSpeed"            // CPU speed MHz
	KLDevCPUCores            HwInvPCViewAttribute = "CpuCores"            // CPU cores count
	KLDevCPUThreads          HwInvPCViewAttribute = "CpuThreads"          // CPU threads count
	KLDevCPUPlatform         HwInvPCViewAttribute = "CpuPlatform"         // CPU platform
	KLDevSerialNumber        HwInvPCViewAttribute = "SerialNum"           // device serial number
	KLDevRevision            HwInvPCViewAttribute = "Revision"            // device revision
	KLDevStorageRotationRate HwInvPCViewAttribute = "StorageRotationRate" // storage rotation rate
	KLDevRAMType             HwInvPCViewAttribute = "RamType"             // RAM type
	KLDevRAMPartNumber       HwInvPCViewAttribute = "RamPartNum"          // RAM part number
	KLDevVideoCardMemorySize HwInvPCViewAttribute = "VideoCardMemorySize" // video card memory size MB
	KLDevSoundCardCodec      HwInvPCViewAttribute = "SoundCardCodec"      // sound card codec
)

type DeviceType int

// Device types (KLDevName)
const (
	KLDevTypeMotherBoard    DeviceType = 0 // Mother board
	KLDevTypeCPU            DeviceType = 1 // CPU
	KLDevTypeRAM            DeviceType = 2 // RAM
	KLDevTypeStorage        DeviceType = 3 // Storage
	KLDevTypeVideoCard      DeviceType = 4 // Video card
	KLDevTypeSoundCard      DeviceType = 5 // Sound card
	KLDevTypeNetworkAdapter DeviceType = 6 // Network adapter
	KLDevTypeMonitor        DeviceType = 7 // Monitor
	KLDevTypeOpticalDrive   DeviceType = 8 // Optical driver
)

func (dev DeviceType) String() string {
	switch dev {
	case KLDevTypeMotherBoard:
		return "MotherBoard"
	case KLDevTypeCPU:
		return "CPU"
	case KLDevTypeRAM:
		return "RAM"
	case KLDevTypeStorage:
		return "Storage"
	case KLDevTypeVideoCard:
		return "VideoCard"
	case KLDevTypeSoundCard:
		return "SoundCard"
	case KLDevTypeNetworkAdapter:
		return "NetworkAdapter"
	case KLDevTypeMonitor:
		return "Monitor"
	case KLDevTypeOpticalDrive:
		return "OpticalDrive"
	default:
		return "Unknown"
	}
}

type hwInvPCView struct {
	HostID              string      `json:"KLHST_WKS_HOSTNAME"`
	DeviceID            string      `json:"dev_id"`
	DeviceType          int         `json:"dev_type"`
	DeviceName          string      `json:"dev_name"`
	DeviceDescription   string      `json:"dev_descr"`
	DeviceManufacturer  string      `json:"dev_manuf"`
	DriverProvider      string      `json:"drv_prov"`
	DriverVersion       string      `json:"drv_ver"`
	DriverDescription   string      `json:"DriverDescription"`
	DriverReleaseDate   kscDateTime `json:"DriverDate"`
	DeviceXPar1         string      `json:"dev_xpar1"`
	DeviceXPar2         kscLong     `json:"dev_xpar2"`
	DeviceXPar3         kscLong     `json:"dev_xpar3"`
	BiosManufacturer    string      `json:"BiosManuf"`
	BiosVersion         string      `json:"BiosVersion"`
	BiosSerialNumber    string      `json:"BiosSerialNum"`
	BiosReleaseDate     kscDateTime `json:"BiosReleaseDate"`
	Chipset             string      `json:"Chipset"`
	CPUSpeed            int         `json:"CpuSpeed"`
	CPUCores            int         `json:"CpuCores"`
	CPUThreads          int         `json:"CpuThreads"`
	CPUPlatform         string      `json:"CpuPlatform"`
	DeviceSerialNumber  string      `json:"SerialNum"`
	DeviceRevision      string      `json:"Revision"`
	StorageRotationRate int         `json:"StorageRotationRate"`
	RAMType             string      `json:"RamType"`
	RAMPartNumber       string      `json:"RamPartNum"`
	VideoCardMemorySize int         `json:"VideoCardMemorySize"`
	SoundCardCodec      string      `json:"SoundCardCodec"`
}

type HwInvPCView struct {
	HostID              string
	DeviceID            string
	DeviceType          DeviceType
	DeviceName          string
	DeviceDescription   string
	DeviceManufacturer  string
	DriverProvider      string
	DriverVersion       string
	DriverDescription   string
	DriverReleaseDate   time.Time
	DeviceXPar1         string
	DeviceXPar2         int64
	DeviceXPar3         int64
	BiosManufacturer    string
	BiosVersion         string
	BiosSerialNumber    string
	BiosReleaseDate     time.Time
	Chipset             string
	CPUSpeed            int
	CPUCores            int
	CPUThreads          int
	CPUPlatform         string
	DeviceSerialNumber  string
	DeviceRevision      string
	StorageRotationRate int
	RAMType             string
	RAMPartNumber       string
	VideoCardMemorySize int
	SoundCardCodec      string
}

func (view *hwInvPCView) toHwInvPCView() (HwInvPCView, error) {
	var err error
	info := HwInvPCView{
		HostID:              view.HostID,
		DeviceID:            view.DeviceID,
		DeviceType:          DeviceType(view.DeviceType),
		DeviceName:          view.DeviceName,
		DeviceDescription:   view.DeviceDescription,
		DeviceManufacturer:  view.DeviceManufacturer,
		DriverProvider:      view.DriverProvider,
		DriverVersion:       view.DriverVersion,
		DriverDescription:   view.DriverDescription,
		DriverReleaseDate:   time.Time{},
		DeviceXPar1:         view.DeviceXPar1,
		DeviceXPar2:         view.DeviceXPar2.Value,
		DeviceXPar3:         view.DeviceXPar3.Value,
		BiosManufacturer:    view.BiosManufacturer,
		BiosVersion:         view.BiosVersion,
		BiosSerialNumber:    view.BiosSerialNumber,
		BiosReleaseDate:     time.Time{},
		Chipset:             view.Chipset,
		CPUSpeed:            view.CPUSpeed,
		CPUCores:            view.CPUCores,
		CPUThreads:          view.CPUThreads,
		CPUPlatform:         view.CPUPlatform,
		DeviceSerialNumber:  view.DeviceSerialNumber,
		DeviceRevision:      view.DeviceRevision,
		StorageRotationRate: view.StorageRotationRate,
		RAMType:             view.RAMType,
		RAMPartNumber:       view.RAMPartNumber,
		VideoCardMemorySize: view.VideoCardMemorySize,
		SoundCardCodec:      view.SoundCardCodec,
	}

	info.DriverReleaseDate, err = view.DriverReleaseDate.toTime()
	if err != nil {
		err = fmt.Errorf("hwInvPCView::toHwInvPCView(): cast %T to %T (DriverReleaseDate): %w", view, info, err)
		return info, err
	}

	info.BiosReleaseDate, err = view.BiosReleaseDate.toTime()
	if err != nil {
		err = fmt.Errorf("hwInvPCView::toHwInvPCView(): cast %T to %T (BiosReleaseDate): %w", view, info, err)
		return info, err
	}

	return info, err
}

type HwInvPCViewParams struct {
	Top         int
	UseDistinct bool
}

func (params *HwInvPCViewParams) viewParams() viewParams {
	return hwInvPCViewParams{
		Top:         params.Top,
		UseDistinct: params.UseDistinct,
	}
}

type hwInvPCViewParams struct {
	Top         int  `json:"TOP_N,omitempty"`
	UseDistinct bool `json:"USE_DISTINCT,omitempty"`
}
