package ksc

import (
	"encoding/base64"
	"fmt"
	"net"
)

// KES policy sections
const (
	KESPolicyCompatibilityInfo            = ".CompatibilityInfo"
	KESPolicyConnACL                      = ".KLCONN_ACL_SECTION"
	KESPolicyCctgReferences               = ".KLPOL_CCTG_REFERENCES"
	KESPolicyNetworkLists                 = ".NETWORKLISTS"
	KESPolicyGeneralSettings              = "1d3e35bd"
	KESPolicyMailThreatProtection         = "265f2c1d"
	KESPolicyAMSIProtectionProvider       = "339e619d"
	KESPolicyEndpointSensor               = "36271f4d"
	KESPolicyNetworkSettings              = "44e2b601"
	KESPolicyExploitPrevention            = "49c71a58"
	KESPolicyFileThreatProtection         = "4b741d57"
	KESPolicyNetworkThreatProtection      = "55b2467a"
	KESPolicyWebControl                   = "63e770b7"
	KESPolicyDataEncryptionCommonSettings = "7ac1dbaa"
	KESPolicyExclusions                   = "852d0287"
	KESPolicyXXXXXXXXXXXXXXXXXXXXX01      = "87624fec"
	KESPolicyDataEncryption               = "895edee6"
	//KESPolicy = "9317f10f"
	KESPolicyNetFilter = "KLEVP_NF_SECTION"
	//KESPolicy = "a9084f82"
	//KESPolicy = "aea3307"
	//KESPolicy = "b96f0d22"
	//KESPolicy = "caf9927"
	//KESPolicy = "cf9370d5"
	//KESPolicy = "da3d6211"
	//KESPolicy = "e9221213"
	//KESPolicy = "ff7a5642"
)

type policyBoolParameter struct {
	Type  string `json:"type"`
	Value struct {
		Mandatory bool `json:"KLPRSS_Mnd"`
		Value     bool `json:"KLPRSS_Val"`
	} `json:"value"`
}

type policyStrParameter struct {
	Type  string `json:"type"`
	Value struct {
		Mandatory bool   `json:"KLPRSS_Mnd"`
		Value     string `json:"KLPRSS_Val"`
	} `json:"value"`
}

type kesPolicyScanExclusion struct {
	Type  string `json:"type"`
	Value struct {
		ItemName     string   `json:"KLPRSS_ValConcatItemName"`
		Comment      string   `json:"comment"`
		Components   []string `json:"components"`
		Enabled      bool     `json:"enabled"`
		FileHash     string   `json:"fileHash"`
		FileHashType int      `json:"fileHashType"`
		Object       struct {
			Type  string `json:"type"`
			Value struct {
				Path      string `json:"objectPath"`
				Recursive bool   `json:"recursive"`
			} `json:"value"`
		} `json:"objectMask"`
		ObjectName string `json:"objectName"`
	} `json:"value"`
}

type kesPolicyScanExclusions struct {
	Type  string `json:"type"`
	Value struct {
		Mandatory   bool                     `json:"KLPRSS_Mnd"`
		Exclusions  []kesPolicyScanExclusion `json:"KLPRSS_Val"`
		ConcatArray bool                     `json:"KLPRSS_ValConcatArray"`
	} `json:"value"`
}

type kesPolicyTrustedApplication struct {
	Type  string `json:"type"`
	Value struct {
		ItemName                                        string `json:"KLPRSS_ValConcatItemName"`
		Lck                                             bool   `json:"KLPRSS_ValLck"`
		AppID                                           int    `json:"appId"`
		Enabled                                         bool   `json:"enabled"`
		DoNotBlockInteractionWithAMSIProtectionProvider bool   `json:"excludeAmsiScan"`
		DoNotMonitorApplicationActivity                 bool   `json:"excludeBehavior"`
		DoNotMonitorChildApplicationActivity            bool   `json:"excludeChildsBehavior"`
		DoNotInheritRestrictionsOfTheParentProcess      bool   `json:"excludeHipsNoInherit"`
		ExcludeNet                                      struct {
			Type  string `json:"type"`
			Value struct {
				Enabled              bool `json:"enabled"`
				EncryptedTrafficOnly bool `json:"encryptedTrafficOnly"`
				Hosts                []struct {
					Type  string `json:"type"`
					Value struct {
						Octets   kscBinary `json:"octets"`
						ZoneIPv6 string    `json:"zoneIPv6"`
					} `json:"value"`
				} `json:"hosts"`
				Ports []int `json:"ports"`
			} `json:"value"`
		} `json:"excludeNet"`
		DoNotScanOpenedFiles                             bool   `json:"excludeOpenFiles"`
		DoNotBlockInteractionWithTheApplicationInterface bool   `json:"excludeSelfProt"`
		FilePath                                         string `json:"filePath"`
	} `json:"value"`
}

type kesPolicyTrustedApplications struct {
	Type  string `json:"type"`
	Value struct {
		Mandatory   bool                          `json:"KLPRSS_Mnd"`
		TrustedApps []kesPolicyTrustedApplication `json:"KLPRSS_Val"`
		ConcatArray bool                          `json:"KLPRSS_ValConcatArray"`
	} `json:"value"`
}

type kesPolicyCertificateStorage struct {
	Type  string `json:"type"`
	Value struct {
		Mandatory bool `json:"KLPRSS_Mnd"`
		Value     struct {
			Name policyStrParameter  `json:"name"`
			Use  policyBoolParameter `json:"use"`
		} `json:"KLPRSS_Val"`
	} `json:"value"`
}

type kesPolicyScanExclusionsAndTrustedZone struct {
	Type  string `json:"type"`
	Value struct {
		Mandatory bool `json:"KLPRSS_Mnd"`
		Value     struct {
			Type  string `json:"type"`
			Value struct {
				CertificateStorage  kesPolicyCertificateStorage  `json:"certificateStorage"`
				ScanExclusions      kesPolicyScanExclusions      `json:"scanExclusions"`
				TrustedApplications kesPolicyTrustedApplications `json:"trustedApplications"`
			} `json:"value"`
		} `json:"KLPRSS_Val"`
	} `json:"value"`
}

type kesPolicyThreats struct {
	Type  string `json:"type"`
	Value struct {
		Mandatory bool `json:"KLPRSS_Mnd"`
		Value     struct {
			Type  string `json:"type"`
			Value struct {
				Adware           policyBoolParameter `json:"adware"`
				AutoDialers      policyBoolParameter `json:"autoDialers"`
				MaliciousTools   policyBoolParameter `json:"maliciousTools"`
				MultiPackedFiles policyBoolParameter `json:"multiPackedFiles"`
				Other            policyBoolParameter `json:"other"`
				PackedFiles      policyBoolParameter `json:"packedFiles"`
				Trojans          policyBoolParameter `json:"trojans"`
				VirusesAndWorms  policyBoolParameter `json:"virusesAndWorms"`
			} `json:"value"`
		} `json:"KLPRSS_Val"`
	} `json:"value"`
}

type kesPolicySectionExclusions struct {
	Mandatory bool `json:"KLPRSS_Mnd"`
	Value     struct {
		Type  string `json:"type"`
		Value struct {
			ScanExclusionsAndTrustedZone kesPolicyScanExclusionsAndTrustedZone `json:"scanExclusionsAndTrustedZone"`
			Threats                      kesPolicyThreats                      `json:"threats"`
		} `json:"value"`
	} `json:"KLPRSS_Val"`
}

type PolicyBoolParameter struct {
	Locked  bool
	Enabled bool
}

type PolicyStrParameter struct {
	Locked bool
	Value  string
}

type ScanExclusion struct {
	ItemName        string
	Comment         string
	Components      []string
	Enabled         bool
	FileHash        string
	FileHashType    int
	ObjectName      string
	ObjectPath      string
	ObjectRecursive bool
}

type ExcludeHost struct {
	IP       net.IP
	ZoneIPv6 string
}

type TrustedApplication struct {
	ItemName                                         string
	Lck                                              bool
	AppID                                            int
	Enabled                                          bool
	DoNotBlockInteractionWithAMSIProtectionProvider  bool
	DoNotMonitorApplicationActivity                  bool
	DoNotMonitorChildApplicationActivity             bool
	DoNotInheritRestrictionsOfTheParentProcess       bool
	DoNotScanOpenedFiles                             bool
	DoNotBlockInteractionWithTheApplicationInterface bool
	FilePath                                         string
	ExcludeNet                                       struct {
		Enabled              bool
		EncryptedTrafficOnly bool
		Hosts                []ExcludeHost
		Ports                []int
	}
}

type KesPolicySectionExclusions struct {
	Locked                   bool
	ExclusionsAndTrustedZone struct {
		Locked             bool
		CertificateStorage struct {
			Locked bool
			Name   PolicyStrParameter
			Use    PolicyBoolParameter
		}
		ScanExclusions struct {
			Locked      bool
			ConcatArray bool
			Items       []ScanExclusion
		}
		TrustedApplications struct {
			Locked      bool
			ConcatArray bool
			Items       []TrustedApplication
		}
	}
	Threats struct {
		Locked bool
		Params struct {
			Adware           PolicyBoolParameter
			AutoDialers      PolicyBoolParameter
			MaliciousTools   PolicyBoolParameter
			MultiPackedFiles PolicyBoolParameter
			Other            PolicyBoolParameter
			PackedFiles      PolicyBoolParameter
			Trojans          PolicyBoolParameter
			VirusesAndWorms  PolicyBoolParameter
		}
	}
}

func (item KesPolicySectionExclusions) Name() string {
	return KESPolicyExclusions
}

func (item KesPolicySectionExclusions) Product() string {
	return KLProductKES
}

func (t *kesPolicySectionExclusions) toKesPolicySectionExclusions() (KesPolicySectionExclusions, error) {
	v := KesPolicySectionExclusions{}

	v.Locked = t.Mandatory

	v.ExclusionsAndTrustedZone.Locked = t.Value.Value.ScanExclusionsAndTrustedZone.Value.Mandatory

	v.ExclusionsAndTrustedZone.CertificateStorage.Locked =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.CertificateStorage.Value.Mandatory
	v.ExclusionsAndTrustedZone.CertificateStorage.Name.Locked =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.CertificateStorage.Value.Value.Name.Value.Mandatory
	v.ExclusionsAndTrustedZone.CertificateStorage.Name.Value =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.CertificateStorage.Value.Value.Name.Value.Value
	v.ExclusionsAndTrustedZone.CertificateStorage.Use.Locked =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.CertificateStorage.Value.Value.Use.Value.Mandatory
	v.ExclusionsAndTrustedZone.CertificateStorage.Use.Enabled =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.CertificateStorage.Value.Value.Use.Value.Value

	v.ExclusionsAndTrustedZone.ScanExclusions.Locked =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.ScanExclusions.Value.Mandatory
	v.ExclusionsAndTrustedZone.ScanExclusions.ConcatArray =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.ScanExclusions.Value.ConcatArray
	for _, item := range t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.ScanExclusions.Value.Exclusions {
		var excl ScanExclusion
		excl.Enabled = item.Value.Enabled
		excl.Comment = item.Value.Comment
		excl.Components = append(excl.Components, item.Value.Components...)
		excl.FileHash = item.Value.FileHash
		excl.FileHashType = item.Value.FileHashType
		excl.ObjectName = item.Value.ObjectName
		excl.ObjectPath = item.Value.Object.Value.Path
		excl.ObjectRecursive = item.Value.Object.Value.Recursive

		v.ExclusionsAndTrustedZone.ScanExclusions.Items = append(v.ExclusionsAndTrustedZone.ScanExclusions.Items, excl)
	}

	v.ExclusionsAndTrustedZone.TrustedApplications.Locked =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.TrustedApplications.Value.Mandatory
	v.ExclusionsAndTrustedZone.TrustedApplications.ConcatArray =
		t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.TrustedApplications.Value.ConcatArray
	for _, item := range t.Value.Value.ScanExclusionsAndTrustedZone.Value.Value.Value.TrustedApplications.Value.TrustedApps {
		var app TrustedApplication
		app.ItemName = item.Value.ItemName
		app.Lck = item.Value.Lck
		app.Enabled = item.Value.Enabled
		app.AppID = item.Value.AppID
		app.DoNotBlockInteractionWithAMSIProtectionProvider = item.Value.DoNotBlockInteractionWithAMSIProtectionProvider
		app.DoNotBlockInteractionWithTheApplicationInterface = item.Value.DoNotBlockInteractionWithTheApplicationInterface
		app.DoNotInheritRestrictionsOfTheParentProcess = item.Value.DoNotInheritRestrictionsOfTheParentProcess
		app.DoNotMonitorApplicationActivity = item.Value.DoNotMonitorApplicationActivity
		app.DoNotMonitorChildApplicationActivity = item.Value.DoNotMonitorChildApplicationActivity
		app.DoNotScanOpenedFiles = item.Value.DoNotScanOpenedFiles
		app.FilePath = item.Value.FilePath
		app.ExcludeNet.Enabled = item.Value.ExcludeNet.Value.Enabled
		app.ExcludeNet.EncryptedTrafficOnly = item.Value.ExcludeNet.Value.EncryptedTrafficOnly
		app.ExcludeNet.Ports = append(app.ExcludeNet.Ports, item.Value.ExcludeNet.Value.Ports...)
		for _, host := range item.Value.ExcludeNet.Value.Hosts {
			var exclHost ExcludeHost
			exclHost.ZoneIPv6 = host.Value.ZoneIPv6
			decoded, err := base64.StdEncoding.DecodeString(host.Value.Octets.Value)
			if err != nil {
				return v, fmt.Errorf("kesPolicySectionExclusions.publicStruct: %v", err)
			}
			exclHost.IP = decoded
			app.ExcludeNet.Hosts = append(app.ExcludeNet.Hosts, exclHost)
		}

		v.ExclusionsAndTrustedZone.TrustedApplications.Items = append(v.ExclusionsAndTrustedZone.TrustedApplications.Items, app)
	}

	v.Threats.Locked = t.Value.Value.Threats.Value.Mandatory
	v.Threats.Params.Adware.Locked = t.Value.Value.Threats.Value.Value.Value.Adware.Value.Mandatory
	v.Threats.Params.Adware.Enabled = t.Value.Value.Threats.Value.Value.Value.Adware.Value.Value
	v.Threats.Params.AutoDialers.Locked = t.Value.Value.Threats.Value.Value.Value.AutoDialers.Value.Mandatory
	v.Threats.Params.AutoDialers.Enabled = t.Value.Value.Threats.Value.Value.Value.AutoDialers.Value.Value
	v.Threats.Params.MaliciousTools.Locked = t.Value.Value.Threats.Value.Value.Value.MaliciousTools.Value.Mandatory
	v.Threats.Params.MaliciousTools.Enabled = t.Value.Value.Threats.Value.Value.Value.MaliciousTools.Value.Value
	v.Threats.Params.MultiPackedFiles.Locked = t.Value.Value.Threats.Value.Value.Value.MultiPackedFiles.Value.Mandatory
	v.Threats.Params.MultiPackedFiles.Enabled = t.Value.Value.Threats.Value.Value.Value.MultiPackedFiles.Value.Value
	v.Threats.Params.Other.Locked = t.Value.Value.Threats.Value.Value.Value.Other.Value.Mandatory
	v.Threats.Params.Other.Enabled = t.Value.Value.Threats.Value.Value.Value.Other.Value.Value
	v.Threats.Params.PackedFiles.Locked = t.Value.Value.Threats.Value.Value.Value.PackedFiles.Value.Mandatory
	v.Threats.Params.PackedFiles.Enabled = t.Value.Value.Threats.Value.Value.Value.PackedFiles.Value.Value
	v.Threats.Params.Trojans.Locked = t.Value.Value.Threats.Value.Value.Value.Trojans.Value.Mandatory
	v.Threats.Params.Trojans.Enabled = t.Value.Value.Threats.Value.Value.Value.Trojans.Value.Value
	v.Threats.Params.VirusesAndWorms.Locked = t.Value.Value.Threats.Value.Value.Value.VirusesAndWorms.Value.Mandatory
	v.Threats.Params.VirusesAndWorms.Enabled = t.Value.Value.Threats.Value.Value.Value.VirusesAndWorms.Value.Value

	return v, nil
}
