package ksc

import (
	"fmt"
	"time"
)

// Use SrvView KLPOL_PROFILE_HST to run queries like 'which hosts have the specified policy profile active',
// 'which policy profiles are active at the specified host' and so on
// The srvview contains following columns

type KLPolProfileHSTViewAttribute string

func (attr KLPolProfileHSTViewAttribute) String() string {
	return string(attr)
}

func (attr KLPolProfileHSTViewAttribute) Attribute() string {
	return string(attr)
}

const (
	KLPolProfName              KLPolProfileHSTViewAttribute = "KLSSPOL_PRF_NAME"                // policy profile name
	KLPolProfPolicyID          KLPolProfileHSTViewAttribute = "KLPOL_ID"                        // policy ID
	KLPolProfPolicyDisplayName KLPolProfileHSTViewAttribute = "KLPOL_DN"                        // policy display name
	KLPolProfProductID         KLPolProfileHSTViewAttribute = "KLHST_WKS_PRODUCT_ID"            // product name and version divided by slash (e.g. 'KES/11.0.0.0'), up to 64 chars, search-only
	KLPolProfHostID            KLPolProfileHSTViewAttribute = "KLHST_WKS_HOSTNAME"              // host name – a unique server-generated string, read-only
	KLPolProfHostDN            KLPolProfileHSTViewAttribute = "KLHST_WKS_DN"                    // host display name
	KLPolProfLastNAgentCon     KLPolProfileHSTViewAttribute = "KLHST_WKS_LAST_NAGENT_CONNECTED" // last Network Agent connection UTC time
	KLPolProfHostStatus        KLPolProfileHSTViewAttribute = "KLHST_WKS_STATUS"                // host status, see KLHostStatus
	KLPolProfFullTextSearch    KLPolProfileHSTViewAttribute = "FULLTEXT_SEARCH"                 // Synthetic field for full-text search queries, search-only
	// Following fields are included in search:
	// KLPolProfPolicyDisplayName
	// KLPolProfName
	// KLPolProfHostDN
)

type klPolProfileHSTView struct {
	HostID                    string      `json:"KLHST_WKS_HOSTNAME"`
	HostDisplayName           string      `json:"KLHST_WKS_DN"`
	LastNetworkAgentConnected kscDateTime `json:"KLHST_WKS_LAST_NAGENT_CONNECTED"`
	HostStatus                int         `json:"KLHST_WKS_STATUS"`
	PolicyDisplayName         string      `json:"KLPOL_DN"`
	PolicyID                  int         `json:"KLPOL_ID"`
	PolicyProfileName         string      `json:"KLSSPOL_PRF_NAME"`
}

type KLPolProfileHSTView struct {
	HostID                    string
	HostDisplayName           string
	LastNetworkAgentConnected time.Time
	HostStatus                int
	PolicyDisplayName         string
	PolicyID                  int
	PolicyProfileName         string
}

func (view *klPolProfileHSTView) toKLPolProfileHSTView() (KLPolProfileHSTView, error) {
	var err error
	polProf := KLPolProfileHSTView{
		HostID:                    view.HostID,
		HostDisplayName:           view.HostDisplayName,
		LastNetworkAgentConnected: time.Time{},
		HostStatus:                view.HostStatus,
		PolicyDisplayName:         view.PolicyDisplayName,
		PolicyID:                  view.PolicyID,
		PolicyProfileName:         view.PolicyProfileName,
	}

	polProf.LastNetworkAgentConnected, err = view.LastNetworkAgentConnected.toTime()
	if err != nil {
		err = fmt.Errorf("klPolProfileHSTView::toKLPolProfileHSTView(): cast %q to %T (LastNetworkAgentConnected): %w", view, polProf, err)
	}

	return polProf, err
}

type KLPolProfileHSTParams struct {
	HostID      string
	Top         int
	UseDistinct bool
}

func (params *KLPolProfileHSTParams) viewParams() viewParams {
	return klPolProfileHSTParams{
		Top:         params.Top,
		UseDistinct: params.UseDistinct,
	}
}

type klPolProfileHSTParams struct {
	Top         int  `json:"TOP_N,omitempty"`
	UseDistinct bool `json:"USE_DISTINCT,omitempty"`
}
