package ksc

import (
	"encoding/base64"
	"encoding/binary"
	"fmt"
	"net"
	"time"
)

type kscDateTime struct {
	Type  string `json:"type"`
	Value string `json:"value"`
}

func (dt *kscDateTime) toTime() (t time.Time, err error) {
	if dt.Value == "" {
		return
	}

	t, err = time.Parse(time.RFC3339, dt.Value)
	if err != nil {
		err = fmt.Errorf("kscDateTime::toTime: cast %q to %T: %w", dt.Value, t, err)
	}
	return
}

type kscLong struct {
	Type  string `json:"type"`
	Value int64  `json:"value"`
}

func (l kscLong) toComponentVersion() ComponentVersion {
	var version ComponentVersion
	b := make([]byte, 8)
	binary.BigEndian.PutUint64(b, uint64(l.Value))
	version.Major = binary.BigEndian.Uint16(b[:2])
	version.Minor = binary.BigEndian.Uint16(b[2:4])
	version.Build = binary.BigEndian.Uint16(b[4:6])
	version.Revision = binary.BigEndian.Uint16(b[6:])
	return version
}

func (l kscLong) toIPv4() net.IP {
	ip := make(net.IP, 4)
	binary.BigEndian.PutUint32(ip, uint32(l.Value))
	return ip
}

type kscBinary struct {
	Type  string `json:"type"`
	Value string `json:"value"`
}

func (b kscBinary) toKscHexID() (id KscHexID, err error) {
	if b.Value == "" {
		return
	}

	value, err := base64.StdEncoding.DecodeString(b.Value)
	if err != nil {
		err = fmt.Errorf("kscBinary::toKscHexID(): cast %q to %T: %w", b.Value, id, err)
		return id, err
	}

	id.value = value

	return
}

func (b kscBinary) toHexString() (hex string, err error) {
	if b.Value == "" {
		return
	}

	bytes, err := base64.StdEncoding.DecodeString(b.Value)
	if err != nil {
		err = fmt.Errorf("kscBinary::toHexString(): cast %q to %T: %w", b.Value, hex, err)
		return
	}
	hex = fmt.Sprintf("0x%X", bytes)

	return
}

func (b kscBinary) toBytes() (bytes []byte, err error) {
	if b.Value == "" {
		return
	}

	bytes, err = base64.StdEncoding.DecodeString(b.Value)
	if err != nil {
		err = fmt.Errorf("kscBinary::toBytes(): cast %q to %T: %w", b.Value, bytes, err)
	}

	return
}

type kscParams struct {
	Type  string                 `json:"type"`
	Value map[string]interface{} `json:"value"`
}

type kscError struct {
	Code    int    `json:"code"`
	Module  string `json:"module"`
	File    string `json:"file"`
	Line    int    `json:"line"`
	Message string `json:"message"`
	LocData struct {
		FormatID  int      `json:"format-id"`
		Format    string   `json:"format"`
		LocModule string   `json:"locmodule"`
		Args      []string `json:"args"`
	} `json:"locdata"`
}

type KscHexID struct {
	value []byte
}

func (id KscHexID) String() string {
	return fmt.Sprintf("0x%X", id.value)
}

type kscDouble struct {
	Type  string  `json:"type"`
	Value float64 `json:"value"`
}
