package ksc

import (
	"context"
	"fmt"
	"net/http"
)

type ListTags service

const (
	apiMethodHostsTagsListTagsGetAllTags apiMethod = "/api/v1.0/HostsTags.ListTags.GetAllTags"
	apiMethodHostsTagsListTagsSetTags    apiMethod = "/api/v1.0/HostsTags.ListTags.SetTags"
	apiMethodHostsTagsListTagsGetTags    apiMethod = "/api/v1.0/HostsTags.ListTags.GetTags"
)

const (
	KLTagsIncludeVS = "KLTAGS_INCLUDE_VS" // (boolean) optional flag; true(default) - to include tags information from virtual servers, false - to exclude tags information from virtual servers
	KLTagsValue     = "KLTAGS_VALUE"      // value of a tag being set
	KLTagsSet       = "KLTAGS_SET"        // true - to set tag, false - to reset it
)

type klItemTag struct {
	Type  string `json:"type"`
	Value struct {
		Value string `json:"KLTAGS_VALUE"`
		Set   bool   `json:"KLTAGS_SET"` // true - to set tag, false - to reset it
	} `json:"value"`
}

type klListItemsTags struct {
	Type  string `json:"type"`
	Value struct {
		ID   string      `json:"KLTAGS_ITEM_ID"`
		Tags []klItemTag `json:"KLTAGS_TAGS"`
	} `json:"value"`
}

type ListItemsTags struct {
	ID   string
	Tags []ItemTag
}

func (tags *ListItemsTags) toKlListItemsTag() klListItemsTags {
	itemsTags := klListItemsTags{
		Type: "params",
	}
	itemsTags.Value.ID = tags.ID

	for _, tag := range tags.Tags {
		itemsTags.Value.Tags = append(itemsTags.Value.Tags, tag.toKlItemTag())
	}

	return itemsTags
}

type ItemTag struct {
	Value string
	Set   bool
}

func (tag *ItemTag) toKlItemTag() klItemTag {
	itemTag := klItemTag{
		Type: "params",
	}
	itemTag.Value.Value = tag.Value
	itemTag.Value.Set = tag.Set
	return itemTag
}

func (lt *ListTags) GetAllTags(ctx context.Context, includeVS bool) ([]string, error) {
	tags, err := lt.getAllTags(ctx, includeVS)
	if err != nil {
		err = fmt.Errorf("ListTags::GetAllTags(): %w", err)
		return tags, err
	}

	return tags, nil
}

func (lt *ListTags) getAllTags(ctx context.Context, includeVS bool) ([]string, error) {
	req := lt.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostsTagsListTagsGetAllTags.String()

	reqData := struct {
		Params struct {
			IncludeVS bool `json:"KLTAGS_INCLUDE_VS"`
		} `json:"pParams"`
	}{}
	reqData.Params.IncludeVS = includeVS

	req.Body = reqData

	resp := struct {
		Tags  []string `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err := lt.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("ListTags::getAllTags(): %w", err)
		return nil, err
	}

	if resp.Error.Code != 0 {
		return nil, fmt.Errorf("ListTags::getAllTags(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp.Tags, nil
}

func (lt *ListTags) GetTags(ctx context.Context, id string) (map[string][]string, error) {
	tags, err := lt.getTags(ctx, []string{id})
	if err != nil {
		err = fmt.Errorf("ListTags::GetTags(): %w", err)
		return tags, err
	}

	return tags, nil
}

func (lt *ListTags) getTags(ctx context.Context, id []string) (map[string][]string, error) {
	req := lt.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostsTagsListTagsGetTags.String()

	tags := make(map[string][]string)
	reqData := struct {
		ID []string `json:"pIdArray"`
	}{
		ID: id,
	}

	req.Body = reqData

	resp := struct {
		Data []struct {
			Type  string `json:"params"`
			Value struct {
				ID   string   `json:"KLTAGS_ITEM_ID"`
				Tags []string `json:"KLTAGS_TAGS"`
			} `json:"value"`
		} `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err := lt.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("ListTags::getTags(): %w", err)
		return tags, err
	}

	if resp.Error.Code != 0 {
		return tags, fmt.Errorf("ListTags::getTags(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	for _, host := range resp.Data {
		tags[host.Value.ID] = host.Value.Tags
	}

	return tags, nil
}

func (lt *ListTags) SetTags(ctx context.Context, itemsList []ListItemsTags, fullReplace bool) (err error) {
	var klItemsList []klListItemsTags
	for _, item := range itemsList {
		klItemsList = append(klItemsList, item.toKlListItemsTag())
	}
	err = lt.setTags(ctx, klItemsList, fullReplace)
	if err != nil {
		err = fmt.Errorf("ListTags::SetTags(): %w", err)
	}
	return
}

func (lt *ListTags) setTags(ctx context.Context, items []klListItemsTags, fullReplace bool) error {
	req := lt.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodHostsTagsListTagsSetTags.String()

	reqData := struct {
		ListItemsTags []klListItemsTags `json:"pListItemsTags"`
		Params        struct {
			FullReplace bool `json:"KLTAGS_FULL_REPLACE"`
		} `json:"pParams"`
	}{}
	reqData.ListItemsTags = items
	reqData.Params.FullReplace = fullReplace

	req.Body = reqData

	resp := struct {
		Data []struct {
			Type  string `json:"params"`
			Value struct {
				ID   string   `json:"KLTAGS_ITEM_ID"`
				Tags []string `json:"KLTAGS_TAGS"`
			} `json:"value"`
		} `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err := lt.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("ListTags::setTags(): %w", err)
		return err
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("ListTags::setTags(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}
