package ksc

import (
	"context"
	"fmt"
	"net/http"
	"time"
)

const KLPolicyDefaultRevision = 0

const (
	KLPolicyID                          = "KLPOL_ID"                     // policy ID
	KLPolicyDisplayName                 = "KLPOL_DN"                     // policy display name
	KLPolicyProduct                     = "KLPOL_PRODUCT"                // product name, list of KLHST_WKS_PRODUCT_NAME and KLHST_WKS_PRODUCT_VERSION values for some products
	KLPolicyVersion                     = "KLPOL_VERSION"                // version name, list of KLHST_WKS_PRODUCT_NAME and KLHST_WKS_PRODUCT_VERSION values for some products
	KLPolicyCreated                     = "KLPOL_CREATED"                // creation time (UTC), read-only
	KLPolicyModified                    = "KLPOL_MODIFIED"               // last modification time (UTC), read-only
	KLPolicyGroupID                     = "KLPOL_GROUP_ID"               // policy group id, read-only
	KLPolicyActive                      = "KLPOL_ACTIVE"                 // true if policy is active
	KLPolicyRoaming                     = "KLPOL_ROAMING"                // true if policy is roaming
	KLPolicyAcceptParent                = "KLPOL_ACCEPT_PARENT"          // true if policy must be modified by parent policy
	KLPolicyForceDistributionToChildren = "KLPOL_FORCE_DISTRIB2CHILDREN" // true if policy must modify child policies
	KLPolicyForced                      = "KLPOL_FORCED"                 // true if policy is modified by parent policy, read-only
	KLPolicyInherited                   = "KLPOL_INHERITED"              // true if the policy is inherited for the specified 'effective' administration group, read-only
	KLPolicyGroupSyncID                 = "KLPOL_GSYN_ID"                // ID of the group synchronization which corresponds to the policy, read-only
	KLPolicyGroupName                   = "KLPOL_GROUP_NAME"             // administration group name where the policy is located, read-only
	KLPolicyVirtualServerName           = "KPOL_VS_NAME"                 // virtual server name where the policy is located, read-only
	KLPolicyProfilesNumber              = "KLPOL_PROFILES_NUM"           // amount of policy profiles in the policy, read-only
	KLPolicyHideOnSlaves                = "KLPOL_HIDE_ON_SLAVES"         // hide policy inherited from the main server on virtual and slave servers, read-only
)

const (
	KLPolicyOutbreakMask = "KLPOL_OUTBREAK_MASK"
	// Policy virus outbreak mask, may contain OR-ed combination of following numbers:
	KLPolicyVOTFilesystem = 1 << iota // filesystem
	KLPolicyVOTEmail                  // email
	KLPolicyVOTPerimeter              // perimeter
)

type Policy service

const (
	apiMethodPolicyGetPoliciesForGroup          apiMethod = "/api/v1.0/Policy.GetPoliciesForGroup"
	apiMethodPolicyGetEffectivePoliciesForGroup apiMethod = "/api/v1.0/Policy.GetEffectivePoliciesForGroup"
	apiMethodPolicyGetPolicyContents            apiMethod = "/api/v1.0/Policy.GetPolicyContents"
	apiMethodPolicyGetPolicyData                apiMethod = "/api/v1.0/Policy.GetPolicyData"
	apiMethodPolicyDeletePolicy                 apiMethod = "/api/v1.0/Policy.DeletePolicy"
)

type policyData struct {
	ID                          int         `json:"KLPOL_ID"`
	AcceptParent                bool        `json:"KLPOL_ACCEPT_PARENT"`
	Active                      bool        `json:"KLPOL_ACTIVE"`
	Created                     kscDateTime `json:"KLPOL_CREATED"`
	DisplayName                 string      `json:"KLPOL_DN"`
	Forced                      bool        `json:"KLPOL_FORCED"`
	ForceDistributionToChildren bool        `json:"KLPOL_FORCE_DISTRIB2CHILDREN"`
	GroupID                     int         `json:"KLPOL_GROUP_ID"`
	GroupName                   string      `json:"KLPOL_GROUP_NAME"`
	GroupSyncID                 int         `json:"KLPOL_GSYN_ID"`
	HideOnSlaves                bool        `json:"KLPOL_HIDE_ON_SLAVES"`
	Inherited                   bool        `json:"KLPOL_INHERITED"`
	Modified                    kscDateTime `json:"KLPOL_MODIFIED"`
	Product                     string      `json:"KLPOL_PRODUCT"`
	ProfilesNumber              int         `json:"KLPOL_PROFILES_NUM"`
	Roaming                     bool        `json:"KLPOL_ROAMING"`
	Version                     string      `json:"KLPOL_VERSION"`
	VirtualServerName           string      `json:"KPOL_VS_NAME"`
}

type PolicyData struct {
	ID                          int
	AcceptParent                bool
	Active                      bool
	Created                     time.Time
	DisplayName                 string
	Forced                      bool
	ForceDistributionToChildren bool
	GroupID                     int
	GroupName                   string
	GroupSyncID                 int
	HideOnSlaves                bool
	Inherited                   bool
	Modified                    time.Time
	Product                     string
	ProfilesNumber              int
	Roaming                     bool
	Version                     string
	VirtualServerName           string
}

func (pd *policyData) toPolicyData() (PolicyData, error) {
	var err error
	data := PolicyData{
		ID:                          pd.ID,
		AcceptParent:                pd.AcceptParent,
		Active:                      pd.Active,
		Created:                     time.Time{},
		DisplayName:                 pd.DisplayName,
		Forced:                      pd.Forced,
		ForceDistributionToChildren: pd.ForceDistributionToChildren,
		GroupID:                     pd.GroupID,
		GroupName:                   pd.GroupName,
		GroupSyncID:                 pd.GroupSyncID,
		HideOnSlaves:                pd.HideOnSlaves,
		Inherited:                   pd.Inherited,
		Modified:                    time.Time{},
		Product:                     pd.Product,
		ProfilesNumber:              pd.ProfilesNumber,
		Roaming:                     pd.Roaming,
		Version:                     pd.Version,
		VirtualServerName:           pd.VirtualServerName,
	}

	if pd.Created.Value != "" {
		data.Created, err = time.Parse(time.RFC3339, pd.Created.Value)
		if err != nil {
			err = fmt.Errorf("hostInfo: cast %T to %T (Created): %v", pd, data, err)
			return data, err
		}
	}

	if pd.Modified.Value != "" {
		data.Modified, err = time.Parse(time.RFC3339, pd.Modified.Value)
		if err != nil {
			err = fmt.Errorf("hostInfo: cast %T to %T (Modified): %v", pd, data, err)
			return data, err
		}
	}

	return data, nil
}

func (p *Policy) DeletePolicy(ctx context.Context, policyID int) (err error) {
	err = p.deletePolicy(ctx, policyID)
	if err != nil {
		err = fmt.Errorf("Policy::DeletePolicy: %w", err)
	}
	return
}

func (p *Policy) deletePolicy(ctx context.Context, policyID int) (err error) {
	req := p.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyDeletePolicy.String()

	resp := struct {
		RetVal PolicyData `json:"PxgRetVal"`
		Error  kscError   `json:"PxgError"`
	}{}

	reqData := struct {
		ID int `json:"nPolicy"`
	}{
		ID: policyID,
	}

	req.Body = reqData

	err = p.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("Policy::deletePolicy(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("Policy::deletePolicy(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	return
}

func (p *Policy) GetPolicyContents(ctx context.Context, policyID int, revisionID int, contentTimeout int) (contentsID string, err error) {
	contentsID, err = p.getPolicyContents(ctx, policyID, revisionID, contentTimeout)
	if err != nil {
		err = fmt.Errorf("Policy::GetPolicyContents(): %w", err)
	}
	return
}

func (p *Policy) getPolicyContents(ctx context.Context, policyID int, revisionID int, contentTimeout int) (contentsID string, err error) {
	req := p.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyGetPolicyContents.String()

	resp := struct {
		RetVal string   `json:"PxgRetVal"`
		Error  kscError `json:"PxgError"`
	}{}

	reqData := struct {
		PolicyID       int `json:"nPolicy"`
		RevisionID     int `json:"nRevisionId"`
		ContentTimeout int `json:"nLifeTime"`
	}{
		PolicyID:       policyID,
		RevisionID:     revisionID,
		ContentTimeout: contentTimeout,
	}

	req.Body = reqData

	err = p.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("Policy::getPolicyContents(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("Policy::getPolicyContents(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	contentsID = resp.RetVal
	return
}

func (p *Policy) GetPolicyData(ctx context.Context, policyID int) (PolicyData, error) {
	var data PolicyData
	policy, err := p.getPolicyData(ctx, policyID)
	if err != nil {
		err = fmt.Errorf("Policy::GetPolicyData(): %w", err)
		return data, err
	}

	data, err = policy.toPolicyData()
	if err != nil {
		err = fmt.Errorf("Policy::GetPolicyData(): %w", err)
		return data, err
	}

	return data, nil
}

func (p *Policy) getPolicyData(ctx context.Context, policyID int) (polData policyData, err error) {
	req := p.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyGetPolicyData.String()

	resp := struct {
		RetVal policyData `json:"PxgRetVal"`
		Error  kscError   `json:"PxgError"`
	}{}

	reqData := struct {
		ID int `json:"nPolicy"`
	}{
		ID: policyID,
	}

	req.Body = reqData

	err = p.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("Policy::getPolicyData(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("Policy::getPolicyData(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	polData = resp.RetVal
	return
}

func (p *Policy) GetPoliciesForGroup(ctx context.Context, groupID int) ([]PolicyData, error) {
	var data []PolicyData
	policies, err := p.getPoliciesForGroup(ctx, groupID)
	if err != nil {
		err = fmt.Errorf("Policy::GetPoliciesForGroup(): %w", err)
		return data, err
	}

	for _, policy := range policies {
		d, err := policy.toPolicyData()
		if err != nil {
			err = fmt.Errorf("Policy::GetPoliciesForGroup(): %w", err)
			return data, err
		}
		data = append(data, d)
	}

	return data, nil
}

func (p *Policy) getPoliciesForGroup(ctx context.Context, groupID int) ([]policyData, error) {
	req := p.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyGetPoliciesForGroup.String()

	var policiesData []policyData
	resp := struct {
		RetVal []struct {
			Type  string     `json:"type"`
			Value policyData `json:"value"`
		} `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	reqData := struct {
		ID int `json:"nGroupId"`
	}{
		ID: groupID,
	}

	req.Body = reqData

	err := p.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("Policy::getPoliciesForGroup(): %w", err)
		return policiesData, err
	}

	if resp.Error.Code != 0 {
		return policiesData, fmt.Errorf("Policy::getPoliciesForGroup(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	for _, policy := range resp.RetVal {
		policiesData = append(policiesData, policy.Value)
	}

	return policiesData, nil
}

func (p *Policy) GetEffectivePoliciesForGroup(ctx context.Context, groupID int) ([]PolicyData, error) {
	var data []PolicyData
	policies, err := p.getEffectivePoliciesForGroup(ctx, groupID)
	if err != nil {
		err = fmt.Errorf("Policy::GetEffectivePoliciesForGroup(): %w", err)
		return data, err
	}

	for _, policy := range policies {
		d, err := policy.toPolicyData()
		if err != nil {
			err = fmt.Errorf("Policy::GetEffectivePoliciesForGroup(): %w", err)
			return data, err
		}
		data = append(data, d)
	}

	return data, nil
}

func (p *Policy) getEffectivePoliciesForGroup(ctx context.Context, groupID int) ([]policyData, error) {
	req := p.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyGetEffectivePoliciesForGroup.String()

	var policiesData []policyData
	resp := struct {
		RetVal []struct {
			Type  string     `json:"type"`
			Value policyData `json:"value"`
		} `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	reqData := struct {
		ID int `json:"nGroupId"`
	}{
		ID: groupID,
	}

	req.Body = reqData

	err := p.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("Policy::getEffectivePoliciesForGroup(): %w", err)
		return policiesData, err
	}

	if resp.Error.Code != 0 {
		return policiesData, fmt.Errorf("Policy::getEffectivePoliciesForGroup(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	for _, policy := range resp.RetVal {
		policiesData = append(policiesData, policy.Value)
	}

	return policiesData, nil
}
