package ksc

import (
	"context"
	"fmt"
	"net/http"
)

type PolicyProfile service

const (
	apiMethodPolicyProfilesEnumProfiles       apiMethod = "/api/v1.0/PolicyProfiles.EnumProfiles"
	apiMethodPolicyProfilesGetProfile         apiMethod = "/api/v1.0/PolicyProfiles.GetProfile"
	apiMethodPolicyProfilesGetProfileSettings apiMethod = "/api/v1.0/PolicyProfiles.GetProfileSettings"
)

type policyProfileInfo struct {
	Expression expression `json:"EXPRESSION"`
	Enabled    struct {
		Type  string `json:"type"`
		Value struct {
			Mandatory bool `json:"KLPRSS_Mnd"`
			Value     bool `json:"KLPRSS_Val"`
		} `json:"value"`
	} `json:"KLSSPOL_PRF_ENABLED"`
	Protected struct {
		Type  string `json:"type"`
		Value struct {
			Mandatory bool `json:"KLPRSS_Mnd"`
			Value     bool `json:"KLPRSS_Val"`
			Lock      bool `json:"KLPRSS_ValLck"`
		} `json:"value"`
	} `json:"KLSSPOL_PRF_PROTECTED"`
}

type PolicyProfileInfo struct {
	Name               string
	Enabled            bool
	EnabledMandatory   bool
	Protected          bool
	ProtectedMandatory bool
	ProtectedLock      bool
	Expression         Expression
}

func (ppi *policyProfileInfo) toPolicyProfileInfo() PolicyProfileInfo {
	info := PolicyProfileInfo{
		Name:               "",
		Enabled:            ppi.Enabled.Value.Value,
		EnabledMandatory:   ppi.Enabled.Value.Mandatory,
		Protected:          ppi.Protected.Value.Value,
		ProtectedMandatory: ppi.Protected.Value.Mandatory,
		ProtectedLock:      ppi.Protected.Value.Lock,
		Expression:         Expression{},
	}
	return info
}

func (pp *PolicyProfile) EnumProfiles(ctx context.Context, policyID int, revisionID int) ([]PolicyProfileInfo, error) {
	var profilesInfo []PolicyProfileInfo

	data, err := pp.enumProfiles(ctx, policyID, revisionID)
	if err != nil {
		return profilesInfo, fmt.Errorf("PolicyProfile::EnumProfiles(): %w", err)
	}

	for name, profile := range data {
		info := profile.toPolicyProfileInfo()
		info.Name = name
		profilesInfo = append(profilesInfo, info)
	}

	return profilesInfo, nil
}

func (pp *PolicyProfile) enumProfiles(ctx context.Context, policyID int, revisionID int) (data map[string]policyProfileInfo, err error) {
	req := pp.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyProfilesEnumProfiles.String()
	req.SetHeader("Transfer-Encoding", "chunked")

	resp := struct {
		RetVal map[string]struct {
			Type  string            `json:"type"`
			Value policyProfileInfo `json:"value"`
		} `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	reqData := struct {
		PolicyID   int `json:"nPolicy"`
		RevisionID int `json:"nRevision"`
	}{
		PolicyID:   policyID,
		RevisionID: revisionID,
	}

	req.Body = reqData

	err = pp.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("PolicyProfile::enumProfiles(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("PolicyProfile::enumProfiles(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	data = make(map[string]policyProfileInfo, len(resp.RetVal))
	for name, profile := range resp.RetVal {
		data[name] = profile.Value
	}

	return
}

func (pp *PolicyProfile) GetProfile(ctx context.Context, policyID int, revisionID int, profileName string) (PolicyProfileInfo, error) {
	var profileInfo PolicyProfileInfo

	data, err := pp.getProfile(ctx, policyID, revisionID, profileName)
	if err != nil {
		return profileInfo, fmt.Errorf("PolicyProfile::GetProfile(): %w", err)
	}

	profileInfo = data.toPolicyProfileInfo()
	profileInfo.Name = profileName

	return profileInfo, nil
}

func (pp *PolicyProfile) getProfile(ctx context.Context, policyID int, revisionID int, profileName string) (data policyProfileInfo, err error) {
	req := pp.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyProfilesGetProfile.String()
	req.SetHeader("Transfer-Encoding", "chunked")

	resp := struct {
		RetVal policyProfileInfo `json:"PxgRetVal"`
		Error  kscError          `json:"PxgError"`
	}{}

	reqData := struct {
		PolicyID    int    `json:"nPolicy"`
		RevisionID  int    `json:"nRevision"`
		ProfileName string `json:"szwName"`
	}{
		PolicyID:    policyID,
		RevisionID:  revisionID,
		ProfileName: profileName,
	}

	req.Body = reqData

	err = pp.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("PolicyProfile::getProfile(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("PolicyProfile::getProfile(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	data = resp.RetVal

	return
}

func (pp *PolicyProfile) GetProfileSettings(ctx context.Context, policyID int, revisionID int, profileName string, lifeTime int) (contentsID string, err error) {
	contentsID, err = pp.getProfileSettings(ctx, policyID, revisionID, profileName, lifeTime)
	if err != nil {
		err = fmt.Errorf("PolicyProfile::GetProfileSettings(): %w", err)
	}

	return
}

func (pp *PolicyProfile) getProfileSettings(ctx context.Context, policyID int, revisionID int, profileName string, lifeTime int) (contentsID string, err error) {
	req := pp.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodPolicyProfilesGetProfileSettings.String()
	req.SetHeader("Transfer-Encoding", "chunked")

	resp := struct {
		RetVal string   `json:"PxgRetVal"`
		Error  kscError `json:"PxgError"`
	}{}

	reqData := struct {
		PolicyID       int    `json:"nPolicy"`
		RevisionID     int    `json:"nRevision"`
		ProfileName    string `json:"szwName"`
		ContentTimeout int    `json:"nLifeTime"`
	}{
		PolicyID:       policyID,
		RevisionID:     revisionID,
		ProfileName:    profileName,
		ContentTimeout: lifeTime,
	}

	req.Body = reqData

	err = pp.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("PolicyProfile::getProfileSettings(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("PolicyProfile::getProfileSettings(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	contentsID = resp.RetVal
	return
}
