package ksc

import (
	"context"
	"fmt"
	"net/http"
	"time"
)

type ServerHierarchy service

const (
	apiMethodServerHierarchyGetChildServers apiMethod = "/api/v1.0/ServerHierarchy.GetChildServers"
)

type SlaveServerAttribute string

func (attr SlaveServerAttribute) String() string {
	return string(attr)
}

const (
	SlaveServerID                            SlaveServerAttribute = "KLSRVH_SRV_ID"              // Slave server ID
	SlaveServerInstanceID                    SlaveServerAttribute = "KLSRVH_SRV_INST_ID"         // Slave server instance ID
	SlaveServerAddress                       SlaveServerAttribute = "KLSRVH_SRV_ADDR"            // Slave server address and port
	SlaveServerDisplayName                   SlaveServerAttribute = "KLSRVH_SRV_DN"              // Display name
	SlaveServerGroupID                       SlaveServerAttribute = "KLSRVH_SRV_GROUPID"         // ID of administration group where the slave server is located
	SlaveServerCertificate                   SlaveServerAttribute = "KLSRVH_SRV_CERTIFICATE"     // Slave server certificate.
	SlaveServerCertificateHash               SlaveServerAttribute = "KLSRVH_SRV_PUBLIC_KEY_HASH" // Slave server certificate MD5-hash
	SlaveServerStatus                        SlaveServerAttribute = "KLSRVH_SRV_STATUS"          // Slave server status: 0 means "Inactive", 1 means "Active"
	SlaveServerVersion                       SlaveServerAttribute = "KLSRVH_SRV_VERSION"         // Slave server version
	SlaveServerIsPassive                     SlaveServerAttribute = "KLSRVH_SRV_PASSIVE"         // Flag set if the slave is passive (does not connect to server, but accepts master connections instead)
	SlaveServerAvailableLastTime             SlaveServerAttribute = "KLSRVH_SRV_LAST_CONNECTED"  // Time when server was available last time
	SlaveServerMasterServerConnectionAddress SlaveServerAttribute = "KLSRVH_SRV_MASTER_ADDR"     // Master server connection address, valid for non-passive slaves
	SlaveServerIdentity                      SlaveServerAttribute = "KLSRVH_SRV_HOST_GUID"       // Slave server host identity
)

const (
	SlaveServerStatusInactive SlaveServerStatusID = 0
	SlaveServerStatusActive   SlaveServerStatusID = 1
)

type SlaveServerStatusID int

func (status SlaveServerStatusID) String() string {
	switch status {
	case SlaveServerStatusInactive:
		return "Inactive"
	case SlaveServerStatusActive:
		return "Active"
	default:
		return "Unknown"
	}
}

type SlaveServerInfo struct {
	ID                            int
	InstanceID                    string
	Address                       string
	DisplayName                   string
	GroupID                       int
	Certificate                   []byte
	CertificateHash               string
	Status                        SlaveServerStatusID
	Version                       string
	IsPassive                     bool
	LastConnected                 time.Time
	MasterServerConnectionAddress string
	Identity                      string
}

type slaveServerInfo struct {
	ID                            int         `json:"KLSRVH_SRV_ID"`
	InstanceID                    string      `json:"KLSRVH_SRV_INST_ID"`
	Address                       string      `json:"KLSRVH_SRV_ADDR"`
	DisplayName                   string      `json:"KLSRVH_SRV_DN"`
	GroupID                       int         `json:"KLSRVH_SRV_GROUPID"`
	Certificate                   kscBinary   `json:"KLSRVH_SRV_CERTIFICATE"`
	CertificateHash               string      `json:"KLSRVH_SRV_PUBLIC_KEY_HASH"`
	Status                        int         `json:"KLSRVH_SRV_STATUS"`
	Version                       string      `json:"KLSRVH_SRV_VERSION"`
	IsPassive                     bool        `json:"KLSRVH_SRV_PASSIVE"`
	LastConnected                 kscDateTime `json:"KLSRVH_SRV_LAST_CONNECTED"`
	MasterServerConnectionAddress string      `json:"KLSRVH_SRV_MASTER_ADDR"`
	Identity                      string      `json:"KLSRVH_SRV_HOST_GUID"`
}

func (ssi *slaveServerInfo) toSlaveServerInfo() (SlaveServerInfo, error) {
	info := SlaveServerInfo{
		ID:                            ssi.ID,
		InstanceID:                    ssi.InstanceID,
		Address:                       ssi.Address,
		DisplayName:                   ssi.DisplayName,
		GroupID:                       ssi.GroupID,
		Certificate:                   []byte{},
		CertificateHash:               ssi.CertificateHash,
		Status:                        SlaveServerStatusID(ssi.Status),
		Version:                       ssi.Version,
		IsPassive:                     ssi.IsPassive,
		LastConnected:                 time.Time{},
		MasterServerConnectionAddress: ssi.MasterServerConnectionAddress,
		Identity:                      ssi.Identity,
	}

	var err error
	info.Certificate, err = ssi.Certificate.toBytes()
	if err != nil {
		err = fmt.Errorf("ServerHierarchy::slaveServerInfo::toSlaveServerInfo(): cast %T to %T (Certificate): %w", ssi, info, err)
		return info, err
	}

	info.LastConnected, err = ssi.LastConnected.toTime()
	if err != nil {
		err = fmt.Errorf("ServerHierarchy::slaveServerInfo::toSlaveServerInfo(): cast %T to %T (LastConnected): %w", ssi, info, err)
		return info, err
	}

	return info, err
}

func (sh *ServerHierarchy) GetChildServers(ctx context.Context, groupID int) (info []SlaveServerInfo, err error) {
	servers, err := sh.getChildServers(ctx, groupID)
	if err != nil {
		err = fmt.Errorf("ServerHierarchy::GetChildServers(): %w", err)
		return
	}

	var data SlaveServerInfo
	for _, server := range servers {
		data, err = server.toSlaveServerInfo()
		if err != nil {
			err = fmt.Errorf("ServerHierarchy::GetChildServers(): %w", err)
			return
		}
		info = append(info, data)
	}

	return
}

func (sh *ServerHierarchy) getChildServers(ctx context.Context, groupID int) (info []slaveServerInfo, err error) {
	req := sh.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodServerHierarchyGetChildServers.String()

	reqData := struct {
		GroupID int `json:"nGroupId"`
	}{
		GroupID: groupID,
	}

	req.Body = reqData

	resp := struct {
		Servers []struct {
			Type  string          `json:"type"`
			Value slaveServerInfo `json:"value"`
		} `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	err = sh.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("ServerHierarchy::getChildServers(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("ServerHierarchy::getChildServers(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	for _, server := range resp.Servers {
		info = append(info, server.Value)
	}

	return
}
