package ksc

import (
	"context"
	"fmt"
	"net/http"
)

type Session service

const (
	apiMethodSessionPing         apiMethod = "/api/v1.0/Session.Ping"
	apiMethodSessionStartSession apiMethod = "/api/v1.0/Session.StartSession"
)

func (s *Session) startSession(ctx context.Context) (sessionID string, err error) {
	req := s.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSessionStartSession.String()

	var auth string
	if s.client.InternalUser {
		auth = fmt.Sprintf(`KSCBasic user="%s", pass="%s", internal="1"`,
			s.client.UserName, s.client.Password)
	} else {
		auth = fmt.Sprintf(`KSCBasic user="%s", pass="%s", domain="%s", internal="0"`,
			s.client.UserName, s.client.Password, s.client.Domain)
	}

	req.SetHeader("Authorization", auth)
	if s.client.VServerName != "" {
		req.SetHeader("X-KSC-VServer", s.client.VServerName)
	}

	resp := struct {
		SessionID string   `json:"PxgRetVal"`
		Error     kscError `json:"PxgError"`
	}{}

	err = s.client.do(ctx, req, &resp)
	if err != nil {
		return "", fmt.Errorf("startSession(): %w", err)
	}

	if resp.Error.Code != 0 {
		return "", fmt.Errorf("startSession(): return code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	sessionID = resp.SessionID

	return sessionID, err
}

func (s *Session) Ping(ctx context.Context) (err error) {
	err = s.ping(ctx)
	if err != nil {
		err = fmt.Errorf("Ping(): %w", err)
	}

	return
}

func (s *Session) ping(ctx context.Context) (err error) {
	req := s.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSessionPing.String()

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}

	err = s.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("ping(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("ping(): return code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	return err
}
